% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tree-sequences.R
\name{ts_load}
\alias{ts_load}
\title{Load a tree sequence file produced by a given model}
\usage{
ts_load(
  file,
  model = NULL,
  recapitate = FALSE,
  simplify = FALSE,
  mutate = FALSE,
  recombination_rate = NULL,
  mutation_rate = NULL,
  Ne = NULL,
  random_seed = NULL,
  simplify_to = NULL,
  keep_input_roots = FALSE,
  migration_matrix = NULL
)
}
\arguments{
\item{file}{A path to the tree-sequence file (either originating from a
slendr model or a standard non-slendr tree sequence)}

\item{model}{Optional \code{slendr_model} object which produced the
tree-sequence \code{file}. Used for adding various annotation data and
metadata to the standard tskit tree-sequence object.}

\item{recapitate}{Should the tree sequence be recapitated?}

\item{simplify}{Should the tree sequence be simplified down to a set of sampled
individuals (those explicitly recorded)?}

\item{mutate}{Should the tree sequence be mutated?}

\item{recombination_rate, Ne}{Arguments passed to \code{ts_recapitate}}

\item{mutation_rate}{Mutation rate passed to \code{ts_mutate}}

\item{random_seed}{Random seed passed to pyslim's \code{recapitate} method}

\item{simplify_to}{A character vector of individual names. If NULL, all
remembered individuals will be retained. Only used when \code{simplify =
  TRUE}.}

\item{keep_input_roots}{Should the history ancestral to the MRCA of all
samples be retained in the tree sequence? Default is \code{FALSE}.}

\item{migration_matrix}{Migration matrix used for coalescence of ancient
lineages (passed to \code{ts_recapitate})}
}
\value{
Tree-sequence object of the class \code{slendr_ts}, which serves as
an interface point for the Python module tskit using slendr functions with
the \code{ts_} prefix.
}
\description{
This function loads a tree sequence file simulated from a given slendr model.
Optionally, the tree sequence can be recapitated and simplified.
}
\details{
The loading, recapitation and simplification is performed using the Python
module pyslim which serves as a link between tree sequences generated by SLiM
and the tskit module for manipulation of tree sequence data. All of these
steps have been modelled after the official pyslim tutorial and documentation
available at: \url{https://tskit.dev/pyslim/docs/latest/tutorial.html}.

The recapitation and simplification steps can also be performed individually
using the functions \code{\link{ts_recapitate}} and
\code{\link{ts_simplify}}.
}
\examples{
\dontshow{check_dependencies(python = TRUE) # make sure dependencies are present
}
# load an example model with an already simulated tree sequence
slendr_ts <- system.file("extdata/models/introgression.trees", package = "slendr")
model <- read_model(path = system.file("extdata/models/introgression", package = "slendr"))

# load tree sequence generated by a given model
ts <- ts_load(slendr_ts, model)

# even tree sequences generated by non-slendr models can be
msprime_ts <- system.file("extdata/models/msprime.trees", package = "slendr")
ts <- ts_load(msprime_ts)

# load tree sequence and immediately simplify it only to sampled individuals
# (note that the example tree sequence is already simplified so this operation
# does not do anything in this case)
ts <- ts_load(slendr_ts, model = model, simplify = TRUE)

# load tree sequence and simplify it to a subset of sampled individuals
ts_small <- ts_simplify(ts, simplify_to = c("CH_1", "NEA_1", "NEA_2",
                                            "AFR_1", "AFR_2", "EUR_1", "EUR_2"))

# load tree sequence, recapitate it and simplify it
ts <- ts_load(slendr_ts, model, recapitate = TRUE, simplify = TRUE,
              recombination_rate = 1e-8, Ne = 10000, random_seed = 42)

# load tree sequence, recapitate it, simplify it and overlay neutral mutations
ts <- ts_load(slendr_ts, model, recapitate = TRUE, simplify = TRUE, random_seed = 42,
              recombination_rate = 1e-8, Ne = 10000, mutation_rate = 1e-8)

ts
}
\seealso{
\code{\link{ts_nodes}} for extracting useful information about
individuals, nodes, coalescent times and geospatial locations of nodes on a
map
}
