% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/centers.R
\name{get_areas_near_coordinates}
\alias{get_areas_near_coordinates}
\alias{areas_in_radius}
\alias{closest_n_areas}
\alias{closest_population}
\title{Make a tibble of census areas closest to a user-specified center}
\usage{
areas_in_radius(
  geography = c("state", "county", "tract", "block group"),
  center = lon_lat_from_area(state = "DC"),
  radius = 5,
  units = "miles",
  measure_from = "center of population",
  year = 2020,
  distance_fun = geosphere::distVincentyEllipsoid,
  batch_size = 50L
)

closest_n_areas(
  geography = c("state", "county", "tract", "block group"),
  center = lon_lat_from_area(state = "DC"),
  n = 50,
  measure_from = "center of population",
  year = 2020,
  distance_fun = geosphere::distVincentyEllipsoid,
  units = NULL,
  batch_size = 50L
)

closest_population(
  geography = c("state", "county", "tract", "block group"),
  center = lon_lat_from_area(state = "DC"),
  population = 1e+06,
  measure_from = "center of population",
  year = 2020,
  distance_fun = geosphere::distVincentyEllipsoid,
  units = NULL,
  batch_size = 50L
)
}
\arguments{
\item{geography}{The type of census areas that the resulting table will
contain. One of \code{c("state", "county", "tract", "block group")}.}

\item{center}{The longitude/latitude coordinates of the center of the circle.
A double vector of length 2 whose elements are finite numbers. Passed to
the \code{y} argument of \code{\link[geosphere:distm]{geosphere::distm()}}.

The first element is the longitude coordinate (positive for west, negative
for east). The second element is the latitude coordinate (positive for
north, negative for south).

The convenience function \code{\link[=lon_lat_from_area]{lon_lat_from_area()}} can be used to obtain the
longitude/latitude coordinates of the center of population of a
user-specified census area.

Defaults to the center of population of the District of Columbia according
to the 2020 decennial census.}

\item{radius}{A single, non-negative number specifying the radius of the
circle. Defaults to 5.}

\item{units}{A single string specifying the units of the resulting \code{distance}
column. If \code{NULL}, the \code{units} package does not need to be installed, and
units will be meters. Otherwise, this will be passed to the \code{value}
argument of \code{\link[units:units]{units::set_units}}\code{(mode = "standard")}.

For \code{areas_in_radius()}, this also used for the units of \code{radius}.}

\item{measure_from}{Currently can only be \code{"center of population"}, the
default.}

\item{year}{Must be 2020, 2010, or 2000. Defaults to 2020.}

\item{distance_fun}{Passed to the \code{fun} argument of \code{\link[geosphere:distm]{geosphere::distm()}}.
Defaults to \code{\link[geosphere:distVincentyEllipsoid]{geosphere::distVincentyEllipsoid}}, which results in the most
accurate measurement but is also the slowest.}

\item{batch_size}{The number of distances calculated in each iterative call
to \code{\link[geosphere:distm]{geosphere::distm()}}. When the request is satisfied, these functions
stop calculating distances in order to prevent potentially hundreds of
thousands of unnecessary calculations. Defaults to 50.}

\item{n}{A single positive integer specifying how many of the areas closest
to \code{center} should be gathered. Defaults to 50.}

\item{population}{A single positive integer specifying the target total
population of the areas returned. See \strong{Details}.}
}
\value{
A \code{\link[tibble:tibble]{tibble}} with each of the columns found in the
corresponding \code{USpopcenters} table, with two columns appended:

\code{geoid} - all FIPS code columns combined with \code{\link[=paste0]{paste0()}}.

\code{distance} - the number of \code{units} the area's \code{LONGITUDE}/\code{LATITUDE} center
of population is away from the coordinates given in \code{center}.
}
\description{
Returns a \code{\link[tibble:tibble]{tibble}} containing the census areas whose
centers of population are closest to some user-specified center. To specify
the center, the user can manually enter longitude/latitude coordinates or use
the helper function \code{\link[=lon_lat_from_area]{lon_lat_from_area()}} to automatically grab the
longitude/latitude coordinates of the center of population of an area. The
cutoff point for how many areas will be return depends on the function used.
}
\details{
\code{areas_in_radius()} returns all areas whose centers of population are within
the user-specified \code{radius} around \code{center}.

\code{closest_n_areas()} returns the top \code{n} areas whose centers of population are
closest areas to \code{center}.

Conceptually, \code{closest_population()} sequentially gathers the next closest
area to \code{center} until the total population of the areas meets or exceeds
\code{population}.

Distances are determined with \code{\link[geosphere:distm]{geosphere::distm()}}.

Requires the packages \code{USpopcenters} and \code{geosphere} to be installed.
Requires the \code{units} to be installed unless \code{units = NULL}.

Centers of population are based on the decennial census data. Only states,
counties, tracts, and block groups are currently supported. See the
documentation of the \code{USpopcenters} package and
\url{https://www.census.gov/geographies/reference-files/time-series/geo/centers-population.html}
for more information.
}
\examples{
# All states whose centers of population are within 300 kilometers of the
# center of population of New York County, New York (i.e, Manhattan):
areas_in_radius(
  geography = "state",
  center = lon_lat_from_area(state = "NY", county = "New York"),
  radius = 300,
  units = "km"
)

# The four census tracts whose centers of population are closest to the
# Four Corners (distance column is in meters due to setting units = NULL):
closest_n_areas("tract", center = c(-109.0452, 36.9991), n = 4, units = NULL)

# The counties closest to center of population of Kauai County, Hawaii whose
# total population reaches 3 million people:
closest_population(
  geography = "county",
  center = lon_lat_from_area("15007"),
  population = 3e6,
  units = "barleycorns"
)
}
\seealso{
\code{\link[=lon_lat_from_area]{lon_lat_from_area()}}
}
