% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glmFilter.R
\name{glmFilter}
\alias{glmFilter}
\title{Unsupervised Spatial Filtering with Eigenvectors in Generalized
Linear Regression Models}
\usage{
glmFilter(
  y,
  x = NULL,
  W,
  objfn = "AIC",
  MX = NULL,
  model,
  optim.method = "BFGS",
  sig = 0.05,
  bonferroni = TRUE,
  positive = TRUE,
  ideal.setsize = FALSE,
  min.reduction = 0.05,
  boot.MI = 100,
  resid.type = "pearson",
  alpha = 0.25,
  tol = 0.1,
  na.rm = TRUE
)
}
\arguments{
\item{y}{response variable}

\item{x}{vector/ matrix of regressors (default = NULL)}

\item{W}{spatial connectivity matrix}

\item{objfn}{the objective function to be used for eigenvector
selection. Possible criteria are: the maximization of model fit
('AIC' or 'BIC'), minimization of residual autocorrelation ('MI'),
significance level of candidate eigenvectors ('p'), significance of residual spatial
autocorrelation ('pMI'), or all eigenvectors in the candidate set ('all')}

\item{MX}{covariates used to construct the projection matrix (default = NULL) - see
Details}

\item{model}{a character string indicating the type of model to be estimated.
Currently, 'probit', 'logit', and 'poisson' are valid inputs}

\item{optim.method}{a character specifying the optimization method used by
the \code{optim} function}

\item{sig}{significance level to be used for eigenvector selection
if \code{objfn = 'p'} or \code{objfn = 'pMI'}}

\item{bonferroni}{Bonferroni adjustment for the significance level
(TRUE/ FALSE) if \code{objfn = 'p'}. Set to FALSE if \code{objfn = 'pMI'} -
see Details}

\item{positive}{restrict search to eigenvectors associated with positive
levels of spatial autocorrelation (TRUE/ FALSE)}

\item{ideal.setsize}{if \code{positive = TRUE}, uses the formula proposed by
Chun et al. (2016) to determine the ideal size of the candidate set
(TRUE/ FALSE)}

\item{min.reduction}{if \code{objfn} is either 'AIC' or 'BIC'. A value in the
interval [0,1) that determines the minimum reduction in AIC/ BIC (relative to the
current AIC/ BIC) a candidate eigenvector need to achieve in order to be selected}

\item{boot.MI}{number of iterations used to estimate the variance of Moran's I
(default = 100). Alternatively, if \code{boot.MI = NULL}, analytical results will
be used}

\item{resid.type}{character string specifying the residual type to be used.
Options are 'raw', 'deviance', and 'pearson' (default)}

\item{alpha}{a value in (0,1] indicating the range of candidate eigenvectors
according to their associated level of spatial autocorrelation, see e.g.,
Griffith (2003)}

\item{tol}{if \code{objfn = 'MI'}, determines the amount of remaining residual
autocorrelation at which the eigenvector selection terminates}

\item{na.rm}{remove observations with missing values (TRUE/ FALSE)}
}
\value{
An object of class \code{spfilter} containing the following
information:
\describe{
\item{\code{estimates}}{summary statistics of the parameter estimates}
\item{\code{varcovar}}{estimated variance-covariance matrix}
\item{\code{EV}}{a matrix containing the summary statistics of selected eigenvectors}
\item{\code{selvecs}}{vector/ matrix of selected eigenvectors}
\item{\code{evMI}}{Moran coefficient of all eigenvectors}
\item{\code{moran}}{residual autocorrelation in the initial and the
filtered model}
\item{\code{fit}}{adjusted R-squared of the initial and the filtered model}
\item{\code{residuals}}{initial and filtered model residuals}
\item{\code{other}}{a list providing supplementary information:
\describe{
\item{\code{ncandidates}}{number of candidate eigenvectors considered}
\item{\code{nev}}{number of selected eigenvectors}
\item{\code{condnum}}{condition number to assess the degree of multicollinearity
among the eigenvectors induced by the link function, see e.g., Griffith/ Amrhein
(1997)}
\item{\code{sel_id}}{ID of selected eigenvectors}
\item{\code{sf}}{vector representing the spatial filter}
\item{\code{sfMI}}{Moran coefficient of the spatial filter}
\item{\code{model}}{type of the regression model}
\item{\code{dependence}}{filtered for positive or negative spatial dependence}
\item{\code{objfn}}{selection criterion specified in the objective function of
the stepwise regression procedure}
\item{\code{bonferroni}}{TRUE/ FALSE: Bonferroni-adjusted significance level
(if \code{objfn='p'})}
\item{\code{siglevel}}{if \code{objfn = 'p'} or \code{objfn = 'pMI'}: actual
(unadjusted/ adjusted) significance level}
\item{\code{resid.type}}{residual type ('raw', 'deviance', or 'pearson')}
\item{\code{pseudoR2}}{McFadden's pseudo R-squared (filtered vs. unfiltered model)}
}
}
}
}
\description{
This function implements the eigenvector-based semiparametric
spatial filtering approach in a generalized linear regression framework using
maximum likelihood estimation (MLE). Eigenvectors are selected by an unsupervised
stepwise regression technique. Supported selection criteria are the minimization of
residual autocorrelation, maximization of model fit, significance of residual
autocorrelation, and the statistical significance of eigenvectors. Alternatively,
all eigenvectors in the candidate set can be included as well.
}
\details{
If \strong{W} is not symmetric, it gets symmetrized by
1/2 * (\strong{W} + \strong{W}') before the decomposition.

If covariates are supplied to \code{MX}, the function uses these regressors
to construct the following projection matrix:

\strong{M} = \strong{I} - \strong{X} (\strong{X}'\strong{X})^-1\strong{X}'

Eigenvectors from \strong{MWM} using this specification of
\strong{M} are not only mutually uncorrelated but also orthogonal
to the regressors specified in \code{MX}. Alternatively, if \code{MX = NULL}, the
projection matrix becomes \strong{M} = \strong{I} - \strong{11}'/ *n*,
where \strong{1} is a vector of ones and *n* represents the number of
observations. Griffith and Tiefelsdorf (2007) show how the choice of the appropriate
\strong{M} depends on the underlying process that generates the spatial
dependence.

The Bonferroni correction is only possible if eigenvector selection is based on
the significance level of the eigenvectors (\code{objfn = 'p'}). It is set to
FALSE if eigenvectors are added to the model until the residuals exhibit no
significant level of spatial autocorrelation (\code{objfn = 'pMI'}).
}
\note{
If the condition number (\code{condnum}) suggests high levels of
multicollinearity, eigenvectors can be sequentially removed from \code{selvecs}
and the model can be re-estimated using the \code{glm} function in order to
identify and manually remove the problematic eigenvectors. Moreover, if other
models that are currently not implemented here need to be estimated
(e.g., quasi-binomial models), users can extract eigenvectors using the function
\code{getEVs} and perform a supervised eigenvector search using the \code{glm}
function.

In contrast to eigenvector-based spatial filtering in linear regression models,
Chun (2014) notes that only a limited number of studies address the problem
of measuring spatial autocorrelation in generalized linear model residuals.
Consequently, eigenvector selection may be based on an objective function that
maximizes model fit rather than minimizes residual spatial autocorrelation.
}
\examples{
data(fakedata)

# poisson model
y_pois <- fakedataset$count
poisson <- glmFilter(y = y_pois, x = NULL, W = W, objfn = "MI", positive = FALSE,
model = "poisson", boot.MI = 100)
print(poisson)
summary(poisson, EV = FALSE)

# probit model - summarize EVs
y_prob <- fakedataset$indicator
probit <- glmFilter(y = y_prob, x = NULL, W = W, objfn = "p", positive = FALSE,
model = "probit", boot.MI = 100)
print(probit)
summary(probit, EV = TRUE)

# logit model - AIC objective function
y_logit <- fakedataset$indicator
logit <- glmFilter(y = y_logit, x = NULL, W = W, objfn = "AIC", positive = FALSE,
model = "logit", min.reduction = .05)
print(logit)
summary(logit, EV = FALSE)

}
\references{
Chun, Yongwan (2014): Analyzing Space-Time Crime Incidents Using
Eigenvector Spatial Filtering: An Application to Vehicle Burglary.
Geographical Analysis 46 (2): pp. 165 - 184.

Tiefelsdorf, Michael and Daniel A. Griffith (2007):
Semiparametric filtering of spatial autocorrelation: the eigenvector
approach. Environment and Planning A: Economy and Space, 39 (5):
pp. 1193 - 1221.

Griffith, Daniel A. (2003): Spatial Autocorrelation and Spatial Filtering:
Gaining Understanding Through Theory and Scientific Visualization.
Berlin/ Heidelberg, Springer.

Griffith, Daniel A. and Carl G. Amrhein (1997): Multivariate Statistical
Analysis for Geographers. Englewood Cliffs, Prentice Hall.
}
\seealso{
\code{\link{lmFilter}}, \code{\link{getEVs}}, \code{\link{MI.resid}},
\code{\link[stats]{optim}}
}
