% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/unif_stat.R
\name{unif_stat}
\alias{unif_stat}
\title{Circular and (hyper)spherical uniformity statistics}
\usage{
unif_stat(data, type = "all", data_sorted = FALSE, Rayleigh_m = 1,
  cov_a = 2 * pi, Rothman_t = 1/3, Cressie_t = 1/3, Pycke_q = 0.5,
  Riesz_s = 1, CCF09_dirs = NULL, K_CCF09 = 25, CJ12_reg = 3)
}
\arguments{
\item{data}{sample to compute the test statistic. An \bold{array} of size
\code{c(n, p, M)} containing \code{M} samples of size \code{n} of directions
(in Cartesian coordinates) on \eqn{S^{p-1}}. Alternatively, a
\bold{matrix} of size \code{c(n, M)} with the angles on \eqn{[0, 2\pi)} of
the \code{M} circular samples of size \code{n} on \eqn{S^{1}}. Other objects
accepted are an array of size \code{c(n, 1, M)} or a vector of size
\code{n} with angular data. Must not contain \code{NA}'s.}

\item{type}{type of test to be applied. A character vector containing any of
the following types of tests, depending on the dimension \eqn{p}:
\itemize{
  \item Circular data: any of the names available at object
  \code{\link{avail_cir_tests}}.
  \item (Hyper)spherical data: any of the names available at object
  \code{\link{avail_sph_tests}}.
}
If \code{type = "all"} (default), then \code{type} is set as
\code{avail_cir_tests} or \code{avail_sph_tests}, depending on the value of
\eqn{p}.}

\item{data_sorted}{is the circular data sorted? If \code{TRUE}, certain
statistics are faster to compute. Defaults to \code{FALSE}.}

\item{Rayleigh_m}{integer \eqn{m} for the \eqn{m}-modal Rayleigh test.
Defaults to \code{m = 1} (the standard Rayleigh test).}

\item{cov_a}{\eqn{a_n = a / n} parameter used in the length of the arcs
of the coverage-based tests. Must be positive. Defaults to \code{2 * pi}.}

\item{Rothman_t}{\eqn{t} parameter for the Rothman test, a real in
\eqn{(0, 1)}. Defaults to \code{1 / 3}.}

\item{Cressie_t}{\eqn{t} parameter for the Cressie test, a real in
\eqn{(0, 1)}. Defaults to \code{1 / 3}.}

\item{Pycke_q}{\eqn{q} parameter for the Pycke "\eqn{q}-test", a real in
\eqn{(0, 1)}. Defaults to \code{1 / 2}.}

\item{Riesz_s}{\eqn{s} parameter for the \eqn{s}-Riesz test, a real in
\eqn{(0, 2)}. Defaults to \code{1}.}

\item{CCF09_dirs}{a matrix of size \code{c(n_proj, p)} containing
\code{n_proj} random directions (in Cartesian coordinates) on \eqn{S^{p-1}}
to perform the CCF09 test. If \code{NULL} (default), a sample of size
\code{n_proj = 50} directions is computed internally.}

\item{K_CCF09}{integer giving the truncation of the series present in the
asymptotic distribution of the Kolmogorov-Smirnov statistic. Defaults to
\code{5e2}.}

\item{CJ12_reg}{type of asymptotic regime for CJ12 test, either \code{1}
(sub-exponential regime), \code{2} (exponential), or \code{3}
(super-exponential; default).}
}
\value{
A data frame of size \code{c(M, length(type))}, with column names
given by \code{type}, that contains the values of the test statistics.
}
\description{
Implementation of several statistics for assessing uniformity
on the (hyper)sphere
\eqn{S^{p-1} := \{{\bf x} \in R^p : ||{\bf x}|| = 1\}}{
S^{p-1} := \{x \in R^p : ||x|| = 1\}}, \eqn{p\ge 2}, for a sample
\eqn{{\bf X}_1,\ldots,{\bf X}_n\in S^{p-1}}{X_1,\ldots,X_n\in S^{p-1}}.

\code{unif_stat} receives a (several) sample(s) of directions in
\emph{Cartesian coordinates}, except for the circular case (\eqn{p=2}) in
which the sample(s) can be \emph{angles}
\eqn{\Theta_1,\ldots,\Theta_n\in [0, 2\pi)}.

\code{unif_stat} allows to compute several statistics to several samples
within a single call, facilitating thus Monte Carlo experiments.
}
\details{
Descriptions and references for most of the statistics are available
in García-Portugués and Verdebout (2018).
}
\examples{
## Circular data

# Sample
n <- 10
M <- 2
Theta <- r_unif_cir(n = n, M = M)

# Matrix
unif_stat(data = Theta, type = "all")

# Array
unif_stat(data = array(Theta, dim = c(n, 1, M)), type = "all")

# Vector
unif_stat(data = Theta[, 1], type = "all")

## Spherical data

# Circular sample in Cartesian coordinates
n <- 10
M <- 2
X <- array(dim = c(n, 2, M))
for (i in 1:M) X[, , i] <- cbind(cos(Theta[, i]), sin(Theta[, i]))

# Array
unif_stat(data = X, type = "all")

# High-dimensional data
X <- r_unif_sph(n = n, p = 3, M = M)
unif_stat(data = X, type = "all")

## Specific arguments

# Rothman
unif_stat(data = Theta, type = "Rothman", Rothman_t = 0.5)

# CCF09
unif_stat(data = X, type = "CCF09", CCF09_dirs = X[, , 1])
unif_stat(data = X, type = "CCF09", CCF09_dirs = X[, , 1], K_CCF09 = 1)

# CJ12
unif_stat(data = X, type = "CJ12", CJ12_reg = 3)
unif_stat(data = X, type = "CJ12", CJ12_reg = 1)
}
\references{
García-Portugués, E. and Verdebout, T. (2018) An overview of uniformity
tests on the hypersphere. \emph{arXiv:1804.00286}.
\url{https://arxiv.org/abs/1804.00286}.
}
