
// [[Rcpp::depends("RcppArmadillo")]]
#include <RcppArmadillo.h>
using namespace Rcpp;


//' @title Gauss--Legendre quadrature
//'
//' @description Convenience for computing the nodes \eqn{x_k} and weights
//' \eqn{w_k} of the \emph{Gauss--Legendre} quadrature formula
//' in \eqn{(a, b)}:
//' \deqn{\int_a^b f(x) w(x)\,\mathrm{d}x\approx\sum_{k=1}^N w_k f(x_k).}{
//' \int_a^b f(x) dx\approx\sum_{k=1}^N w_k f(x_k)}.
//'
//' @param a,b scalars giving the interval \eqn{(a, b)}. Defaults to
//' \eqn{(-1, 1)}.
//' @param N number of points used in the Gauss--Legendre quadrature. The
//' following choices are supported: \code{5}, \code{10}, \code{20},
//' \code{40}, \code{80}, \code{160}, \code{320}, \code{640}, \code{1280},
//' \code{2560}, and \code{5120}. Defaults to \code{40}.
//' @return A matrix of size \code{c(N, 1)} with the nodes \eqn{x_k}
//' (\code{Gauss_Legen_nodes}) or the corresponding weights \eqn{w_k}
//' (\code{Gauss_Legen_weights}).
//' @details For \eqn{C^\infty} functions, Gauss--Legendre quadrature
//' can be very efficient. It is exact for polynomials up to degree
//' \eqn{2N - 1}.
//'
//' The nodes and weights up to \eqn{N = 80} were retrieved from
//' \href{https://dlmf.nist.gov/3.5#v}{NIST} and have \eqn{10^{-21}} precision.
//' For \eqn{N = 160} onwards, the nodes and weights were computed with the
//' \code{gauss.quad} function from the \href{https://CRAN.R-project.org/package=statmod}{
//' \code{statmod}} package (Smyth, 1998), and have \eqn{10^{-15}} precision.
//' @references
//' \emph{NIST Digital Library of Mathematical Functions}. Release
//' 1.0.20 of 2018-09-15. F. W. J. Olver, A. B. Olde Daalhuis, D. W. Lozier,
//' B. I. Schneider, R. F. Boisvert, C. W. Clark, B. R. Miller,
//' and B. V. Saunders, eds. \url{https://dlmf.nist.gov/}
//'
//' Smyth, G. K. (1998). Numerical integration. In: \emph{Encyclopedia of
//' Biostatistics}, P. Armitage and T. Colton (eds.), Wiley, London,
//' pp. 3088-3095.
//' @examples
//' ## Integration of a smooth function in (1, 10)
//'
//' # Weights and nodes for integrating
//' x_k <- Gauss_Legen_nodes(a = 1, b = 10, N = 40)
//' w_k <- Gauss_Legen_weights(a = 1, b = 10, N = 40)
//'
//' # Check quadrature
//' f <- function(x) sin(x) * x^2 - log(x + 1)
//' integrate(f, lower = 1, upper = 10, rel.tol = 1e-12)
//' sum(w_k * f(x_k))
//'
//' # Exact for polynomials up to degree 2 * N - 1
//' f <- function(x) (((x + 0.5) / 1e3)^5 - ((x - 0.5)/ 5)^4 +
//'   ((x - 0.25) / 10)^2 + 1)^20
//' sum(w_k * f(x_k))
//' integrate(f, lower = -1, upper = 1, rel.tol = 1e-12)
//'
//' ## Integration on (0, pi)
//'
//' # Weights and nodes for integrating
//' th_k <- Gauss_Legen_nodes(a = 0, b = pi, N = 40)
//' w_k <- Gauss_Legen_weights(a = 0, b = pi, N = 40)
//'
//' # Check quadrature
//' p <- 4
//' psi <- function(th) -sin(th / 2)
//' w <- function(th) sin(th)^(p - 2)
//' integrate(function(th) psi(th) * w(th), lower = 0, upper = pi,
//'           rel.tol = 1e-12)
//' sum(w_k * psi(th_k) * w(th_k))
//'
//' # Integral with Gegenbauer polynomial
//' k <- 3
//' C_k <- function(th) drop(Gegen_polyn(theta = th, k = k, p = p))
//' integrate(function(th) psi(th) * C_k(th) * w(th), lower = 0, upper = pi,
//'           rel.tol = 1e-12)
//' th_k <- drop(Gauss_Legen_nodes(a = 0, b = pi, N = 80))
//' w_k <- drop(Gauss_Legen_weights(a = 0, b = pi, N = 80))
//' sum(w_k * psi(th_k) * C_k(th_k) * w(th_k))
//' @name Gauss_Legen


//' @rdname Gauss_Legen
//' @export
// [[Rcpp::export]]
arma::vec Gauss_Legen_nodes(double a = -1, double b = 1, arma::uword N = 40) {

  // Nodes (+-x_k) for the Gauss--Legendre quadrature in (-1, 1). The nodes
  // up to N = 80 are retrieved from https://dlmf.nist.gov/3.5#v, and for
  // N = 160 onwards they were computed with:
  // statmod::gauss.quad(n = N, kind = "legendre")
  arma::vec x_k = arma::zeros(N);
  if (N == 5) {

    x_k = {0.000000000000000, 0.538469310105683, 0.906179845938664};

  } else if (N == 10) {

    x_k = {0.148874338981631211, 0.433395394129247191,
           0.679409568299024406, 0.865063366688984511,
           0.973906528517171720};

  } else if (N == 20) {

    x_k = {0.076526521133497333755, 0.227785851141645078080,
           0.373706088715419560673, 0.510867001950827098004,
           0.636053680726515025453, 0.746331906460150792614,
           0.839116971822218823395, 0.912234428251325905868,
           0.963971927277913791268, 0.993128599185094924786};

  } else if (N == 40) {

    x_k = {0.038772417506050821933, 0.116084070675255208483,
           0.192697580701371099716, 0.268152185007253681141,
           0.341994090825758473007, 0.413779204371605001525,
           0.483075801686178712909, 0.549467125095128202076,
           0.612553889667980237953, 0.671956684614179548379,
           0.727318255189927103281, 0.778305651426519387695,
           0.824612230833311663196, 0.865959503212259503821,
           0.902098806968874296728, 0.932812808278676533361,
           0.957916819213791655805, 0.977259949983774262663,
           0.990726238699457006453, 0.998237709710559200350};

  } else if (N == 80) {

    x_k = {0.019511383256793997654, 0.058504437152420668629,
           0.097408398441584599063, 0.136164022809143886559,
           0.174712291832646812559, 0.212994502857666132572,
           0.250952358392272120493, 0.288528054884511853109,
           0.325664370747701914619, 0.362304753499487315619,
           0.398393405881969227024, 0.433875370831756093062,
           0.468696615170544477036, 0.502804111888784987594,
           0.536145920897131932020, 0.568671268122709784725,
           0.600330622829751743155, 0.631075773046871966248,
           0.660859898986119801736, 0.689637644342027600771,
           0.717365185362099880254, 0.744000297583597272317,
           0.769502420135041373866, 0.793832717504605449949,
           0.816954138681463470371, 0.838831473580255275617,
           0.859431406663111096977, 0.878722567678213828704,
           0.896675579438770683194, 0.913263102571757654165,
           0.928459877172445795953, 0.942242761309872674752,
           0.954590766343634905493, 0.965485089043799251452,
           0.974909140585727793386, 0.982848572738629070418,
           0.989291302499755531027, 0.994227540965688277892,
           0.997649864398237688899, 0.999553822651630629880};

  } else if (N == 160) {

    x_k = {0.009786689277549, 0.029356318347778,
           0.048914700399903, 0.068454342209658,
           0.087967757732556, 0.107447470971946,
           0.126886018843225, 0.146275954033103,
           0.165609847852822, 0.184880293084242,
           0.204079906817702, 0.223201333280567,
           0.242237246655374, 0.261180353886511,
           0.280023397474338, 0.298759158255680,
           0.317380458169650, 0.335880163007704,
           0.354251185146911, 0.372486486265367,
           0.390579080038724, 0.408522034816796,
           0.426308476279219, 0.443931590069149,
           0.461384624403979, 0.478660892662094,
           0.495753775944651, 0.512656725611419,
           0.529363265789701, 0.545866995855374,
           0.562161592885103, 0.578240814078785,
           0.594098499151297, 0.609728572692628,
           0.625125046495498, 0.640282021849564,
           0.655193691801340, 0.669854343378962,
           0.684258359780950, 0.698400222528118,
           0.712274513577825, 0.725875917399735,
           0.739199223012314, 0.752239325979266,
           0.764991230365148, 0.777450050649422,
           0.789611013598204, 0.801469460092987,
           0.813020846915655, 0.824260748489090,
           0.835184858572704, 0.845788991912262,
           0.856069085843345, 0.866021201847856,
           0.875641527062964, 0.884926375741905,
           0.893872190666099, 0.902475544508025,
           0.910733141144340, 0.918641816918758,
           0.926198541854185, 0.933400420813678,
           0.940244694609783, 0.946728741061852,
           0.952850076000975, 0.958606354222183,
           0.963995370383663, 0.969015059852787,
           0.973663499498934, 0.977938908433306,
           0.981839648696526, 0.985364225895890,
           0.988511289796807, 0.991279634879646,
           0.993668200892097, 0.995676073488025,
           0.997302485277280, 0.998546818751517,
           0.999408620641306, 0.999887752272163};

  } else if (N == 320) {

    x_k = {0.004901054997656, 0.014702694091793,
           0.024502920527642, 0.034300792683824,
           0.044095369165163, 0.053885708893136,
           0.063670871196293, 0.073449915900643,
           0.083221903419978, 0.092985894846162,
           0.102740952039330, 0.112486137718037,
           0.122220515549308, 0.131943150238600,
           0.141653107619675, 0.151349454744346,
           0.161031259972125, 0.170697593059729,
           0.180347525250467, 0.189980129363470,
           0.199594479882777, 0.209189653046264,
           0.218764726934397, 0.228318781558812,
           0.237850898950710, 0.247360163249056,
           0.256845660788576, 0.266306480187545,
           0.275741712435352, 0.285150450979842,
           0.294531791814415, 0.303884833564890,
           0.313208677576104, 0.322502427998261,
           0.331765191873002, 0.340996079219210,
           0.350194203118511, 0.359358679800499,
           0.368488628727644, 0.377583172679897,
           0.386641437838977, 0.395662553872328,
           0.404645654016739, 0.413589875161627,
           0.422494357931967, 0.431358246770859,
           0.440180690021731, 0.448960840010172,
           0.457697853125375, 0.466390889901188,
           0.475039115096782, 0.483641697776893,
           0.492197811391662, 0.500706633856054,
           0.509167347628841, 0.517579139791156,
           0.525941202124599, 0.534252731188889,
           0.542512928399065, 0.550721000102209,
           0.558876157653709, 0.566977617493023,
           0.575024601218978, 0.583016335664548,
           0.590952052971147, 0.598830990662405,
           0.606652391717429, 0.614415504643538,
           0.622119583548463, 0.629763888212025,
           0.637347684157243, 0.644870242720914,
           0.652330841123620, 0.659728762539170,
           0.667063296163485, 0.674333737282879,
           0.681539387341780, 0.688679554009844,
           0.695753551248474, 0.702760699376740,
           0.709700325136679, 0.716571761757984,
           0.723374349022072, 0.730107433325514,
           0.736770367742838, 0.743362512088683,
           0.749883232979311, 0.756331903893466,
           0.762707905232565, 0.769010624380237,
           0.775239455761178, 0.781393800899343,
           0.787473068475438, 0.793476674383745,
           0.799404041788239, 0.805254601178010,
           0.811027790421989, 0.816723054822951,
           0.822339847170813, 0.827877627795214,
           0.833335864617368, 0.838714033201178,
           0.844011616803638, 0.849228106424472,
           0.854363000855042, 0.859415806726511,
           0.864386038557238, 0.869273218799429,
           0.874076877885018, 0.878796554270790,
           0.883431794482715, 0.887982153159530,
           0.892447193095528, 0.896826485282558,
           0.901119608951252, 0.905326151611456,
           0.909445709091850, 0.913477885578797,
           0.917422293654362, 0.921278554333542,
           0.925046297100680, 0.928725159945059,
           0.932314789395692, 0.935814840555278,
           0.939224977133349, 0.942544871478573,
           0.945774204610239, 0.948912666248912,
           0.951959954846234, 0.954915777613911,
           0.957779850551836, 0.960551898475386,
           0.963231655041857, 0.965818862776064,
           0.968313273095077, 0.970714646332118,
           0.973022751759588, 0.975237367611243,
           0.977358281103520, 0.979385288455988,
           0.981318194910950, 0.983156814752184,
           0.984900971322824, 0.986550497042385,
           0.988105233422948, 0.989565031084503,
           0.990929749769477, 0.992199258356498,
           0.993373434873440, 0.994452166509907,
           0.995435349629393, 0.996322889781677,
           0.997114701716636, 0.997810709402343,
           0.998410846055053, 0.998915054203942,
           0.999323285871991, 0.999635503240465,
           0.999851682193268, 0.999971849980221};

  } else if (N == 640) {

    x_k = {0.002452450075841, 0.007357291226332,
           0.012261955374668, 0.017166324524144,
           0.022070280685152, 0.026973705878019,
           0.031876482135848, 0.036778491507352,
           0.041679616059695, 0.046579737881326,
           0.051478739084822, 0.056376501809716,
           0.061272908225338, 0.066167840533648,
           0.071061180972069, 0.075952811816324,
           0.080842615383263, 0.085730474033698,
           0.090616270175231, 0.095499886265085,
           0.100381204812929, 0.105260108383709,
           0.110136479600470, 0.115010201147179,
           0.119881155771550, 0.124749226287863,
           0.129614295579784, 0.134476246603187,
           0.139334962388958, 0.144190326045822,
           0.149042220763148, 0.153890529813760,
           0.158735136556748, 0.163575924440271,
           0.168412777004364, 0.173245577883736,
           0.178074210810573, 0.182898559617334,
           0.187718508239544, 0.192533940718591,
           0.197344741204508, 0.202150793958770,
           0.206951983357070, 0.211748193892104,
           0.216539310176351, 0.221325216944847,
           0.226105799057959, 0.230880941504155,
           0.235650529402770, 0.240414448006774,
           0.245172582705523, 0.249924819027528,
           0.254671042643203, 0.259411139367612,
           0.264144995163226, 0.268872496142654,
           0.273593528571395, 0.278307978870567,
           0.283015733619641, 0.287716679559169,
           0.292410703593513, 0.297097692793557,
           0.301777534399435, 0.306450115823237,
           0.311115324651716, 0.315773048648998,
           0.320423175759279, 0.325065594109522,
           0.329700192012145, 0.334326857967716,
           0.338945480667624, 0.343555948996768,
           0.348158152036223, 0.352751979065911,
           0.357337319567267, 0.361914063225891,
           0.366482099934209, 0.371041319794120,
           0.375591613119638, 0.380132870439533,
           0.384664982499962, 0.389187840267102,
           0.393701334929766, 0.398205357902031,
           0.402699800825840, 0.407184555573614,
           0.411659514250855, 0.416124569198737,
           0.420579612996700, 0.425024538465031,
           0.429459238667447, 0.433883606913664,
           0.438297536761963, 0.442700922021756,
           0.447093656756134, 0.451475635284422,
           0.455846752184715, 0.460206902296422,
           0.464555980722787, 0.468893882833421,
           0.473220504266813, 0.477535740932846,
           0.481839489015296, 0.486131644974332,
           0.490412105549007, 0.494680767759745,
           0.498937528910813, 0.503182286592794,
           0.507414938685055, 0.511635383358194,
           0.515843519076502, 0.520039244600395,
           0.524222458988856, 0.528393061601863,
           0.532550952102806, 0.536696030460905,
           0.540828196953617, 0.544947352169032,
           0.549053397008264, 0.553146232687840,
           0.557225760742074, 0.561291883025435,
           0.565344501714906, 0.569383519312345,
           0.573408838646822, 0.577420362876965,
           0.581417995493280, 0.585401640320482,
           0.589371201519804, 0.593326583591305,
           0.597267691376163, 0.601194430058972,
           0.605106705170015, 0.609004422587542,
           0.612887488540034, 0.616755809608457,
           0.620609292728507, 0.624447845192860,
           0.628271374653386, 0.632079789123384,
           0.635872996979791, 0.639650906965384,
           0.643413428190976, 0.647160470137605,
           0.650891942658712, 0.654607755982305,
           0.658307820713125, 0.661992047834791,
           0.665660348711945, 0.669312635092386,
           0.672948819109184, 0.676568813282811,
           0.680172530523225, 0.683759884131982,
           0.687330787804312, 0.690885155631202,
           0.694422902101453, 0.697943942103750,
           0.701448190928701, 0.704935564270876,
           0.708405978230839, 0.711859349317159,
           0.715295594448430, 0.718714630955258,
           0.722116376582258, 0.725500749490029,
           0.728867668257123, 0.732217051882006,
           0.735548819785006, 0.738862891810250,
           0.742159188227595, 0.745437629734541,
           0.748698137458146, 0.751940632956921,
           0.755165038222712, 0.758371275682583,
           0.761559268200682, 0.764728939080092,
           0.767880212064680, 0.771013011340933,
           0.774127261539775, 0.777222887738388,
           0.780299815462013, 0.783357970685736,
           0.786397279836273, 0.789417669793744,
           0.792419067893424, 0.795401401927500,
           0.798364600146796, 0.801308591262516,
           0.804233304447940, 0.807138669340144,
           0.810024616041686, 0.812891075122284,
           0.815737977620496, 0.818565255045369,
           0.821372839378091, 0.824160663073633,
           0.826928659062361, 0.829676760751663,
           0.832404902027543, 0.835113017256214,
           0.837801041285678, 0.840468909447289,
           0.843116557557316, 0.845743921918480,
           0.848350939321493, 0.850937547046572,
           0.853503682864954, 0.856049285040389,
           0.858574292330624, 0.861078643988886,
           0.863562279765330, 0.866025139908497,
           0.868467165166750, 0.870888296789698,
           0.873288476529611, 0.875667646642820,
           0.878025749891109, 0.880362729543087,
           0.882678529375557, 0.884973093674867,
           0.887246367238251, 0.889498295375158,
           0.891728823908565, 0.893937899176283,
           0.896125468032244, 0.898291477847788,
           0.900435876512919, 0.902558612437565,
           0.904659634552818, 0.906738892312161,
           0.908796335692686, 0.910831915196297,
           0.912845581850899, 0.914837287211580,
           0.916806983361771, 0.918754622914404,
           0.920680159013050, 0.922583545333047,
           0.924464736082610, 0.926323686003942,
           0.928160350374311, 0.929974685007138,
           0.931766646253049, 0.933536191000934,
           0.935283276678978, 0.937007861255691,
           0.938709903240912, 0.940389361686813,
           0.942046196188882, 0.943680366886896,
           0.945291834465877, 0.946880560157042,
           0.948446505738733, 0.949989633537337,
           0.951509906428194, 0.953007287836487,
           0.954481741738125, 0.955933232660606,
           0.957361725683876, 0.958767186441164,
           0.960149581119812, 0.961508876462086,
           0.962845039765976, 0.964158038885988,
           0.965447842233910, 0.966714418779576,
           0.967957738051613, 0.969177770138172,
           0.970374485687648, 0.971547855909389,
           0.972697852574385, 0.973824448015948,
           0.974927615130378, 0.976007327377619,
           0.977063558781889, 0.978096283932313,
           0.979105477983529, 0.980091116656290,
           0.981053176238044, 0.981991633583509,
           0.982906466115226, 0.983797651824104,
           0.984665169269950, 0.985508997581984,
           0.986329116459341, 0.987125506171562,
           0.987898147559065, 0.988647022033610,
           0.989372111578744, 0.990073398750237,
           0.990750866676501, 0.991404499058994,
           0.992034280172618, 0.992640194866098,
           0.993222228562342, 0.993780367258801,
           0.994314597527804, 0.994824906516887,
           0.995311281949107, 0.995773712123340,
           0.996212185914580, 0.996626692774212,
           0.997017222730290, 0.997383766387806,
           0.997726314928959, 0.998044860113441,
           0.998339394278742, 0.998609910340533,
           0.998856401793160, 0.999078862710417,
           0.999277287746870, 0.999451672140466,
           0.999602011718332, 0.999728302911465,
           0.999830542798737, 0.999908729272172,
           0.999962861923536, 0.999992951474173};

  } else if (N == 1280) {

    x_k = {0.001226705047316, 0.003680107758140,
           0.006133488317577, 0.008586831958188,
           0.011040123912749, 0.013493349414354,
           0.015946493696494, 0.018399541993147,
           0.020852479538872, 0.023305291568893,
           0.025757963319191, 0.028210480026588,
           0.030662826928844, 0.033114989264736,
           0.035566952274156, 0.038018701198196,
           0.040470221279231, 0.042921497761020,
           0.045372515888784, 0.047823260909303,
           0.050273718070997, 0.052723872624020,
           0.055173709820349, 0.057623214913869,
           0.060072373160466, 0.062521169818112,
           0.064969590146957, 0.067417619409414,
           0.069865242870253, 0.072312445796684,
           0.074759213458450, 0.077205531127913,
           0.079651384080143, 0.082096757593010,
           0.084541636947267, 0.086986007426641,
           0.089429854317924, 0.091873162911061,
           0.094315918499233, 0.096758106378952,
           0.099199711850148, 0.101640720216255,
           0.104081116784301, 0.106520886864999,
           0.108960015772830, 0.111398488826135,
           0.113836291347205, 0.116273408662364,
           0.118709826102063, 0.121145529000964,
           0.123580502698030, 0.126014732536614,
           0.128448203864547, 0.130880902034223,
           0.133312812402693, 0.135743920331749,
           0.138174211188012, 0.140603670343021,
           0.143032283173324, 0.145460035060559,
           0.147886911391549, 0.150312897558388,
           0.152737978958525, 0.155162140994858,
           0.157585369075816, 0.160007648615452,
           0.162428965033528, 0.164849303755602,
           0.167268650213119, 0.169686989843494,
           0.172104308090206, 0.174520590402879,
           0.176935822237372, 0.179349989055870,
           0.181763076326966, 0.184175069525753,
           0.186585954133907, 0.188995715639779,
           0.191404339538480, 0.193811811331967,
           0.196218116529135, 0.198623240645896,
           0.201027169205275, 0.203429887737493,
           0.205831381780054, 0.208231636877830,
           0.210630638583154, 0.213028372455903,
           0.215424824063584, 0.217819978981422,
           0.220213822792447, 0.222606341087584,
           0.224997519465734, 0.227387343533861,
           0.229775798907086, 0.232162871208765,
           0.234548546070579, 0.236932809132622,
           0.239315646043486, 0.241697042460346,
           0.244076984049047, 0.246455456484194,
           0.248832445449233, 0.251207936636539,
           0.253581915747505, 0.255954368492621,
           0.258325280591569, 0.260694637773302,
           0.263062425776133, 0.265428630347823,
           0.267793237245657, 0.270156232236543,
           0.272517601097091, 0.274877329613696,
           0.277235403582628, 0.279591808810117,
           0.281946531112436, 0.284299556315990,
           0.286650870257397, 0.289000458783576,
           0.291348307751833, 0.293694403029943,
           0.296038730496238, 0.298381276039692,
           0.300722025560000, 0.303060964967674,
           0.305398080184116, 0.307733357141713,
           0.310066781783913, 0.312398340065316,
           0.314728017951753, 0.317055801420378,
           0.319381676459745, 0.321705629069895,
           0.324027645262441, 0.326347711060655,
           0.328665812499541, 0.330981935625936,
           0.333296066498579, 0.335608191188202,
           0.337918295777615, 0.340226366361783,
           0.342532389047918, 0.344836349955559,
           0.347138235216652, 0.349438030975638,
           0.351735723389538, 0.354031298628029,
           0.356324742873535, 0.358616042321307,
           0.360905183179504, 0.363192151669278,
           0.365476934024858, 0.367759516493634,
           0.370039885336233, 0.372318026826610,
           0.374593927252124, 0.376867572913628,
           0.379138950125540, 0.381408045215940,
           0.383674844526639, 0.385939334413269,
           0.388201501245363, 0.390461331406436,
           0.392718811294070, 0.394973927319990,
           0.397226665910153, 0.399477013504827,
           0.401724956558668, 0.403970481540807,
           0.406213574934934, 0.408454223239369,
           0.410692412967154, 0.412928130646126,
           0.415161362819007, 0.417392096043474,
           0.419620316892251, 0.421846011953179,
           0.424069167829308, 0.426289771138968,
           0.428507808515855, 0.430723266609109,
           0.432936132083396, 0.435146391618988,
           0.437354031911842, 0.439559039673681,
           0.441761401632074, 0.443961104530514,
           0.446158135128503, 0.448352480201624,
           0.450544126541629, 0.452733060956510,
           0.454919270270587, 0.457102741324578,
           0.459283460975688, 0.461461416097682,
           0.463636593580962, 0.465808980332653,
           0.467978563276678, 0.470145329353834,
           0.472309265521875, 0.474470358755591,
           0.476628596046879, 0.478783964404832,
           0.480936450855808, 0.483086042443513,
           0.485232726229080, 0.487376489291139,
           0.489517318725908, 0.491655201647257,
           0.493790125186795, 0.495922076493943,
           0.498051042736013, 0.500177011098284,
           0.502299968784082, 0.504419903014851,
           0.506536801030237, 0.508650650088162,
           0.510761437464899, 0.512869150455149,
           0.514973776372120, 0.517075302547602,
           0.519173716332042, 0.521269005094621,
           0.523361156223332, 0.525450157125052,
           0.527535995225621, 0.529618657969916,
           0.531698132821928, 0.533774407264837,
           0.535847468801085, 0.537917304952454,
           0.539983903260141, 0.542047251284834,
           0.544107336606780, 0.546164146825870,
           0.548217669561707, 0.550267892453680,
           0.552314803161044, 0.554358389362988,
           0.556398638758713, 0.558435539067506,
           0.560469078028810, 0.562499243402303,
           0.564526022967971, 0.566549404526175,
           0.568569375897734, 0.570585924923991,
           0.572599039466890, 0.574608707409047,
           0.576614916653827, 0.578617655125407,
           0.580616910768863, 0.582612671550231,
           0.584604925456582, 0.586593660496098,
           0.588578864698142, 0.590560526113330,
           0.592538632813599, 0.594513172892288,
           0.596484134464202, 0.598451505665686,
           0.600415274654693, 0.602375429610865,
           0.604331958735592, 0.606284850252092,
           0.608234092405476, 0.610179673462825,
           0.612121581713253, 0.614059805467982,
           0.615994333060416, 0.617925152846201,
           0.619852253203307, 0.621775622532086,
           0.623695249255353, 0.625611121818448,
           0.627523228689309, 0.629431558358540,
           0.631336099339480, 0.633236840168276,
           0.635133769403946, 0.637026875628451,
           0.638916147446765, 0.640801573486943,
           0.642683142400185, 0.644560842860910,
           0.646434663566824, 0.648304593238982,
           0.650170620621864, 0.652032734483436,
           0.653890923615219, 0.655745176832364,
           0.657595482973708, 0.659441830901846,
           0.661284209503202, 0.663122607688089,
           0.664957014390780, 0.666787418569574,
           0.668613809206862, 0.670436175309194,
           0.672254505907343, 0.674068790056373,
           0.675879016835706, 0.677685175349184,
           0.679487254725139, 0.681285244116455,
           0.683079132700634, 0.684868909679862,
           0.686654564281077, 0.688436085756023,
           0.690213463381330, 0.691986686458566,
           0.693755744314309, 0.695520626300206,
           0.697281321793040, 0.699037820194795,
           0.700790110932716, 0.702538183459377,
           0.704282027252742, 0.706021631816227,
           0.707756986678765, 0.709488081394872,
           0.711214905544703, 0.712937448734121,
           0.714655700594755, 0.716369650784068,
           0.718079288985409, 0.719784604908091,
           0.721485588287434, 0.723182228884846,
           0.724874516487868, 0.726562440910245,
           0.728245991991988, 0.729925159599425,
           0.731599933625276, 0.733270303988705,
           0.734936260635379, 0.736597793537538,
           0.738254892694046, 0.739907548130454,
           0.741555749899063, 0.743199488078981,
           0.744838752776181, 0.746473534123567,
           0.748103822281027, 0.749729607435492,
           0.751350879801005, 0.752967629618763,
           0.754579847157192, 0.756187522711998,
           0.757790646606222, 0.759389209190308,
           0.760983200842150, 0.762572611967161,
           0.764157432998322, 0.765737654396243,
           0.767313266649220, 0.768884260273294,
           0.770450625812306, 0.772012353837953,
           0.773569434949847, 0.775121859775573,
           0.776669618970739, 0.778212703219039,
           0.779751103232310, 0.781284809750575,
           0.782813813542119, 0.784338105403525,
           0.785857676159744, 0.787372516664142,
           0.788882617798558, 0.790387970473357,
           0.791888565627488, 0.793384394228535,
           0.794875447272773, 0.796361715785224,
           0.797843190819707, 0.799319863458896,
           0.800791724814369, 0.802258766026666,
           0.803720978265341, 0.805178352729014,
           0.806630880645424, 0.808078553271482,
           0.809521361893328, 0.810959297826374,
           0.812392352415364, 0.813820517034428,
           0.815243783087121, 0.816662142006492,
           0.818075585255125, 0.819484104325186,
           0.820887690738491, 0.822286336046540,
           0.823680031830576, 0.825068769701635,
           0.826452541300596, 0.827831338298229,
           0.829205152395250, 0.830573975322368,
           0.831937798840332, 0.833296614739986,
           0.834650414842316, 0.835999190998497,
           0.837342935089948, 0.838681639028374,
           0.840015294755819, 0.841343894244712,
           0.842667429497919, 0.843985892548787,
           0.845299275461195, 0.846607570329598,
           0.847910769279081, 0.849208864465398,
           0.850501848075027, 0.851789712325212,
           0.853072449464013, 0.854350051770349,
           0.855622511554050, 0.856889821155897,
           0.858151972947672, 0.859408959332203,
           0.860660772743411, 0.861907405646352,
           0.863148850537266, 0.864385099943621,
           0.865616146424157, 0.866841982568932,
           0.868062600999366, 0.869277994368287,
           0.870488155359970, 0.871693076690189,
           0.872892751106255, 0.874087171387061,
           0.875276330343126, 0.876460220816639,
           0.877638835681498, 0.878812167843362,
           0.879980210239682, 0.881142955839752,
           0.882300397644748, 0.883452528687772,
           0.884599342033889, 0.885740830780176,
           0.886876988055761, 0.888007807021858,
           0.889133280871818, 0.890253402831163,
           0.891368166157632, 0.892477564141215,
           0.893581590104202, 0.894680237401212,
           0.895773499419246, 0.896861369577717,
           0.897943841328491, 0.899020908155932,
           0.900092563576935, 0.901158801140967,
           0.902219614430109, 0.903274997059091,
           0.904324942675328, 0.905369444958967,
           0.906408497622915, 0.907442094412884,
           0.908470229107425, 0.909492895517968,
           0.910510087488855, 0.911521798897384,
           0.912528023653838, 0.913528755701528,
           0.914523989016824, 0.915513717609196,
           0.916497935521248, 0.917476636828756,
           0.918449815640697, 0.919417466099292,
           0.920379582380039, 0.921336158691746,
           0.922287189276568, 0.923232668410041,
           0.924172590401115, 0.925106949592192,
           0.926035740359156, 0.926958957111410,
           0.927876594291906, 0.928788646377184,
           0.929695107877398, 0.930595973336356,
           0.931491237331550, 0.932380894474185,
           0.933264939409218, 0.934143366815387,
           0.935016171405242, 0.935883347925178,
           0.936744891155467, 0.937600795910289,
           0.938451057037763, 0.939295669419981,
           0.940134627973031, 0.940967927647035,
           0.941795563426178, 0.942617530328738,
           0.943433823407110, 0.944244437747850,
           0.945049368471685, 0.945848610733562,
           0.946642159722662, 0.947430010662440,
           0.948212158810644, 0.948988599459350,
           0.949759327934993, 0.950524339598384,
           0.951283629844749, 0.952037194103752,
           0.952785027839521, 0.953527126550680,
           0.954263485770372, 0.954994101066286,
           0.955718968040688, 0.956438082330443,
           0.957151439607040, 0.957859035576625,
           0.958560865980021, 0.959256926592755,
           0.959947213225083, 0.960631721722016,
           0.961310447963347, 0.961983387863670,
           0.962650537372413, 0.963311892473852,
           0.963967449187147, 0.964617203566353,
           0.965261151700457, 0.965899289713392,
           0.966531613764062, 0.967158120046369,
           0.967778804789233, 0.968393664256614,
           0.969002694747538, 0.969605892596113,
           0.970203254171559, 0.970794775878223,
           0.971380454155606, 0.971960285478378,
           0.972534266356407, 0.973102393334775,
           0.973664662993800, 0.974221071949054,
           0.974771616851390, 0.975316294386955,
           0.975855101277215, 0.976388034278970,
           0.976915090184379, 0.977436265820977,
           0.977951558051689, 0.978460963774859,
           0.978964479924262, 0.979462103469118,
           0.979953831414125, 0.980439660799461,
           0.980919588700811, 0.981393612229385,
           0.981861728531926, 0.982323934790742,
           0.982780228223709, 0.983230606084296,
           0.983675065661578, 0.984113604280256,
           0.984546219300667, 0.984972908118806,
           0.985393668166337, 0.985808496910614,
           0.986217391854689, 0.986620350537334,
           0.987017370533050, 0.987408449452088,
           0.987793584940454, 0.988172774679934,
           0.988546016388101, 0.988913307818329,
           0.989274646759811, 0.989630031037568,
           0.989979458512461, 0.990322927081212,
           0.990660434676404, 0.990991979266506,
           0.991317558855878, 0.991637171484784,
           0.991950815229405, 0.992258488201848,
           0.992560188550166, 0.992855914458353,
           0.993145664146371, 0.993429435870155,
           0.993707227921618, 0.993979038628669,
           0.994244866355220, 0.994504709501195,
           0.994758566502542, 0.995006435831239,
           0.995248315995307, 0.995484205538817,
           0.995714103041898, 0.995938007120748,
           0.996155916427639, 0.996367829650929,
           0.996573745515066, 0.996773662780602,
           0.996967580244190, 0.997155496738603,
           0.997337411132732, 0.997513322331600,
           0.997683229276362, 0.997847130944316,
           0.998005026348910, 0.998156914539744,
           0.998302794602582, 0.998442665659350,
           0.998576526868150, 0.998704377423262,
           0.998826216555150, 0.998942043530469,
           0.999051857652071, 0.999155658259016,
           0.999253444726574, 0.999345216466246,
           0.999430972925767, 0.999510713589137,
           0.999584437976647, 0.999652145644932,
           0.999713836187054, 0.999769509232658,
           0.999819164448276, 0.999862801537939,
           0.999900420244608, 0.999932020353828,
           0.999957601704709, 0.999977164231268,
           0.999990708184231, 0.999998236490325};

  } else if (N == 2560) {

    x_k = {0.000613472446131, 0.001840416414875,
           0.003067357613067, 0.004294294193679,
           0.005521224309686, 0.006748146114078,
           0.007975057759850, 0.009201957400020,
           0.010428843187621, 0.011655713275706,
           0.012882565817351, 0.014109398965661,
           0.015336210873769, 0.016562999694841,
           0.017789763582075, 0.019016500688708,
           0.020243209168019, 0.021469887173328,
           0.022696532858001, 0.023923144375454,
           0.025149719879152, 0.026376257522618,
           0.027602755459428, 0.028829211843222,
           0.030055624827697, 0.031281992566620,
           0.032508313213825, 0.033734584923216,
           0.034960805848770, 0.036186974144544,
           0.037413087964670, 0.038639145463365,
           0.039865144794929, 0.041091084113749,
           0.042316961574306, 0.043542775331170,
           0.044768523539008, 0.045994204352587,
           0.047219815926776, 0.048445356416544,
           0.049670823976973, 0.050896216763249,
           0.052121532930676, 0.053346770634669,
           0.054571928030761, 0.055797003274611,
           0.057021994521997, 0.058246899928824,
           0.059471717651127, 0.060696445845074,
           0.061921082666966, 0.063145626273242,
           0.064370074820482, 0.065594426465411,
           0.066818679364893, 0.068042831675949,
           0.069266881555745, 0.070490827161605,
           0.071714666651010, 0.072938398181598,
           0.074162019911172, 0.075385529997699,
           0.076608926599315, 0.077832207874326,
           0.079055371981211, 0.080278417078628,
           0.081501341325411, 0.082724142880578,
           0.083946819903330, 0.085169370553059,
           0.086391792989343, 0.087614085371955,
           0.088836245860865, 0.090058272616237,
           0.091280163798443, 0.092501917568053,
           0.093723532085848, 0.094945005512816,
           0.096166336010160, 0.097387521739294,
           0.098608560861855, 0.099829451539696,
           0.101050191934899, 0.102270780209766,
           0.103491214526831, 0.104711493048862,
           0.105931613938856, 0.107151575360054,
           0.108371375475931, 0.109591012450210,
           0.110810484446854, 0.112029789630079,
           0.113248926164351, 0.114467892214388,
           0.115686685945168, 0.116905305521924,
           0.118123749110155, 0.119342014875624,
           0.120560100984361, 0.121778005602665,
           0.122995726897112, 0.124213263034551,
           0.125430612182111, 0.126647772507202,
           0.127864742177517, 0.129081519361038,
           0.130298102226039, 0.131514488941080,
           0.132730677675022, 0.133946666597020,
           0.135162453876534, 0.136378037683325,
           0.137593416187459, 0.138808587559314,
           0.140023549969579, 0.141238301589256,
           0.142452840589665, 0.143667165142449,
           0.144881273419568, 0.146095163593313,
           0.147308833836300, 0.148522282321479,
           0.149735507222131, 0.150948506711875,
           0.152161278964668, 0.153373822154813,
           0.154586134456951, 0.155798214046077,
           0.157010059097534, 0.158221667787018,
           0.159433038290579, 0.160644168784629,
           0.161855057445939, 0.163065702451645,
           0.164276101979249, 0.165486254206624,
           0.166696157312013, 0.167905809474036,
           0.169115208871690, 0.170324353684353,
           0.171533242091785, 0.172741872274134,
           0.173950242411935, 0.175158350686115,
           0.176366195277994, 0.177573774369293,
           0.178781086142128, 0.179988128779019,
           0.181194900462893, 0.182401399377083,
           0.183607623705332, 0.184813571631797,
           0.186019241341053, 0.187224631018092,
           0.188429738848327, 0.189634563017596,
           0.190839101712165, 0.192043353118729,
           0.193247315424414, 0.194450986816784,
           0.195654365483840, 0.196857449614020,
           0.198060237396212, 0.199262727019745,
           0.200464916674399, 0.201666804550406,
           0.202868388838450, 0.204069667729673,
           0.205270639415679, 0.206471302088532,
           0.207671653940761, 0.208871693165365,
           0.210071417955811, 0.211270826506042,
           0.212469917010474, 0.213668687664006,
           0.214867136662014, 0.216065262200361,
           0.217263062475396, 0.218460535683959,
           0.219657680023380, 0.220854493691486,
           0.222050974886601, 0.223247121807548,
           0.224442932653657, 0.225638405624760,
           0.226833538921202, 0.228028330743834,
           0.229222779294024, 0.230416882773658,
           0.231610639385139, 0.232804047331394,
           0.233997104815873, 0.235189810042556,
           0.236382161215950, 0.237574156541098,
           0.238765794223578, 0.239957072469505,
           0.241147989485536, 0.242338543478873,
           0.243528732657263, 0.244718555229003,
           0.245908009402940, 0.247097093388478,
           0.248285805395578, 0.249474143634759,
           0.250662106317103, 0.251849691654261,
           0.253036897858448, 0.254223723142451,
           0.255410165719632, 0.256596223803925,
           0.257781895609848, 0.258967179352496,
           0.260152073247550, 0.261336575511280,
           0.262520684360544, 0.263704398012789,
           0.264887714686062, 0.266070632599005,
           0.267253149970860, 0.268435265021472,
           0.269616975971296, 0.270798281041386,
           0.271979178453417, 0.273159666429672,
           0.274339743193052, 0.275519406967076,
           0.276698655975885, 0.277877488444247,
           0.279055902597553, 0.280233896661825,
           0.281411468863720, 0.282588617430528,
           0.283765340590175, 0.284941636571230,
           0.286117503602905, 0.287292939915056,
           0.288467943738190, 0.289642513303462,
           0.290816646842683, 0.291990342588319,
           0.293163598773496, 0.294336413632002,
           0.295508785398287, 0.296680712307471,
           0.297852192595343, 0.299023224498363,
           0.300193806253667, 0.301363936099068,
           0.302533612273061, 0.303702833014822,
           0.304871596564213, 0.306039901161784,
           0.307207745048778, 0.308375126467129,
           0.309542043659469, 0.310708494869127,
           0.311874478340136, 0.313039992317230,
           0.314205035045852, 0.315369604772154,
           0.316533699743001, 0.317697318205970,
           0.318860458409359, 0.320023118602180,
           0.321185297034176, 0.322346991955808,
           0.323508201618269, 0.324668924273480,
           0.325829158174098, 0.326988901573513,
           0.328148152725853, 0.329306909885990,
           0.330465171309539, 0.331622935252859,
           0.332780199973059, 0.333936963728001,
           0.335093224776300, 0.336248981377327,
           0.337404231791213, 0.338558974278852,
           0.339713207101901, 0.340866928522786,
           0.342020136804701, 0.343172830211613,
           0.344325007008266, 0.345476665460179,
           0.346627803833653, 0.347778420395770,
           0.348928513414402, 0.350078081158203,
           0.351227121896622, 0.352375633899900,
           0.353523615439075, 0.354671064785981,
           0.355817980213257, 0.356964359994341,
           0.358110202403482, 0.359255505715736,
           0.360400268206969, 0.361544488153864,
           0.362688163833918, 0.363831293525449,
           0.364973875507598, 0.366115908060327,
           0.367257389464430, 0.368398318001525,
           0.369538691954067, 0.370678509605343,
           0.371817769239480, 0.372956469141443,
           0.374094607597039, 0.375232182892922,
           0.376369193316595, 0.377505637156407,
           0.378641512701564, 0.379776818242125,
           0.380911552069009, 0.382045712473995,
           0.383179297749724, 0.384312306189703,
           0.385444736088309, 0.386576585740789,
           0.387707853443264, 0.388838537492730,
           0.389968636187062, 0.391098147825017,
           0.392227070706235, 0.393355403131243,
           0.394483143401454, 0.395610289819178,
           0.396736840687614, 0.397862794310858,
           0.398988148993909, 0.400112903042663,
           0.401237054763924, 0.402360602465400,
           0.403483544455709, 0.404605879044382,
           0.405727604541862, 0.406848719259510,
           0.407969221509608, 0.409089109605360,
           0.410208381860892, 0.411327036591258,
           0.412445072112443, 0.413562486741362,
           0.414679278795868, 0.415795446594747,
           0.416910988457729, 0.418025902705481,
           0.419140187659622, 0.420253841642711,
           0.421366862978261, 0.422479249990738,
           0.423591001005560, 0.424702114349104,
           0.425812588348708, 0.426922421332670,
           0.428031611630255, 0.429140157571694,
           0.430248057488190, 0.431355309711916,
           0.432461912576022, 0.433567864414635,
           0.434673163562861, 0.435777808356792,
           0.436881797133499, 0.437985128231046,
           0.439087799988486, 0.440189810745863,
           0.441291158844216, 0.442391842625583,
           0.443491860433000, 0.444591210610510,
           0.445689891503156, 0.446787901456992,
           0.447885238819078, 0.448981901937491,
           0.450077889161322, 0.451173198840677,
           0.452267829326684, 0.453361778971492,
           0.454455046128278, 0.455547629151242,
           0.456639526395619, 0.457730736217672,
           0.458821256974700, 0.459911087025042,
           0.461000224728071, 0.462088668444209,
           0.463176416534919, 0.464263467362710,
           0.465349819291145, 0.466435470684833,
           0.467520419909445, 0.468604665331703,
           0.469688205319390, 0.470771038241354,
           0.471853162467503, 0.472934576368814,
           0.474015278317334, 0.475095266686180,
           0.476174539849545, 0.477253096182698,
           0.478330934061986, 0.479408051864839,
           0.480484447969771, 0.481560120756381,
           0.482635068605359, 0.483709289898484,
           0.484782783018631, 0.485855546349769,
           0.486927578276967, 0.487998877186396,
           0.489069441465326, 0.490139269502140,
           0.491208359686324, 0.492276710408475,
           0.493344320060306, 0.494411187034643,
           0.495477309725432, 0.496542686527738,
           0.497607315837749, 0.498671196052779,
           0.499734325571269, 0.500796702792791,
           0.501858326118049, 0.502919193948880,
           0.503979304688262, 0.505038656740311,
           0.506097248510284, 0.507155078404583,
           0.508212144830759, 0.509268446197509,
           0.510323980914685, 0.511378747393290,
           0.512432744045486, 0.513485969284594,
           0.514538421525094, 0.515590099182629,
           0.516641000674014, 0.517691124417225,
           0.518740468831414, 0.519789032336904,
           0.520836813355194, 0.521883810308960,
           0.522930021622061, 0.523975445719536,
           0.525020081027609, 0.526063925973694,
           0.527106978986394, 0.528149238495500,
           0.529190702932004, 0.530231370728091,
           0.531271240317146, 0.532310310133756,
           0.533348578613712, 0.534386044194011,
           0.535422705312857, 0.536458560409670,
           0.537493607925076, 0.538527846300924,
           0.539561273980278, 0.540593889407420,
           0.541625691027859, 0.542656677288326,
           0.543686846636782, 0.544716197522416,
           0.545744728395649, 0.546772437708137,
           0.547799323912774, 0.548825385463690,
           0.549850620816259, 0.550875028427099,
           0.551898606754071, 0.552921354256289,
           0.553943269394114, 0.554964350629160,
           0.555984596424300, 0.557004005243662,
           0.558022575552632, 0.559040305817862,
           0.560057194507267, 0.561073240090030,
           0.562088441036601, 0.563102795818702,
           0.564116302909333, 0.565128960782763,
           0.566140767914545, 0.567151722781510,
           0.568161823861774, 0.569171069634735,
           0.570179458581082, 0.571186989182792,
           0.572193659923136, 0.573199469286676,
           0.574204415759274, 0.575208497828091,
           0.576211713981586, 0.577214062709523,
           0.578215542502975, 0.579216151854319,
           0.580215889257243, 0.581214753206752,
           0.582212742199157, 0.583209854732096,
           0.584206089304520, 0.585201444416705,
           0.586195918570247, 0.587189510268073,
           0.588182218014436, 0.589174040314918,
           0.590164975676440, 0.591155022607249,
           0.592144179616937, 0.593132445216434,
           0.594119817918008, 0.595106296235277,
           0.596091878683202, 0.597076563778093,
           0.598060350037612, 0.599043235980772,
           0.600025220127942, 0.601006301000851,
           0.601986477122585, 0.602965747017593,
           0.603944109211688, 0.604921562232050,
           0.605898104607227, 0.606873734867137,
           0.607848451543074, 0.608822253167706,
           0.609795138275076, 0.610767105400609,
           0.611738153081114, 0.612708279854778,
           0.613677484261182, 0.614645764841289,
           0.615613120137458, 0.616579548693436,
           0.617545049054367, 0.618509619766794,
           0.619473259378659, 0.620435966439304,
           0.621397739499476, 0.622358577111326,
           0.623318477828417, 0.624277440205720,
           0.625235462799620, 0.626192544167912,
           0.627148682869816, 0.628103877465964,
           0.629058126518412, 0.630011428590641,
           0.630963782247556, 0.631915186055488,
           0.632865638582199, 0.633815138396886,
           0.634763684070177, 0.635711274174137,
           0.636657907282269, 0.637603581969519,
           0.638548296812272, 0.639492050388363,
           0.640434841277069, 0.641376668059119,
           0.642317529316692, 0.643257423633423,
           0.644196349594399, 0.645134305786168,
           0.646071290796736, 0.647007303215571,
           0.647942341633607, 0.648876404643242,
           0.649809490838344, 0.650741598814250,
           0.651672727167772, 0.652602874497192,
           0.653532039402275, 0.654460220484261,
           0.655387416345870, 0.656313625591311,
           0.657238846826270, 0.658163078657928,
           0.659086319694951, 0.660008568547497,
           0.660929823827219, 0.661850084147264,
           0.662769348122280, 0.663687614368410,
           0.664604881503303, 0.665521148146112,
           0.666436412917492, 0.667350674439611,
           0.668263931336144, 0.669176182232282,
           0.670087425754727, 0.670997660531697,
           0.671906885192932, 0.672815098369690,
           0.673722298694753, 0.674628484802428,
           0.675533655328546, 0.676437808910471,
           0.677340944187094, 0.678243059798841,
           0.679144154387675, 0.680044226597091,
           0.680943275072126, 0.681841298459359,
           0.682738295406912, 0.683634264564450,
           0.684529204583186, 0.685423114115885,
           0.686315991816858, 0.687207836341976,
           0.688098646348660, 0.688988420495889,
           0.689877157444203, 0.690764855855705,
           0.691651514394057, 0.692537131724490,
           0.693421706513799, 0.694305237430354,
           0.695187723144090, 0.696069162326520,
           0.696949553650730, 0.697828895791386,
           0.698707187424730, 0.699584427228589,
           0.700460613882368, 0.701335746067066,
           0.702209822465260, 0.703082841761124,
           0.703954802640418, 0.704825703790498,
           0.705695543900316, 0.706564321660419,
           0.707432035762954, 0.708298684901670,
           0.709164267771919, 0.710028783070657,
           0.710892229496450, 0.711754605749469,
           0.712615910531498, 0.713476142545936,
           0.714335300497794, 0.715193383093700,
           0.716050389041904, 0.716906317052272,
           0.717761165836297, 0.718614934107095,
           0.719467620579406, 0.720319223969604,
           0.721169742995690, 0.722019176377297,
           0.722867522835694, 0.723714781093784,
           0.724560949876111, 0.725406027908858,
           0.726250013919849, 0.727092906638553,
           0.727934704796084, 0.728775407125205,
           0.729615012360327, 0.730453519237515,
           0.731290926494484, 0.732127232870607,
           0.732962437106915, 0.733796537946094,
           0.734629534132496, 0.735461424412133,
           0.736292207532682, 0.737121882243488,
           0.737950447295564, 0.738777901441590,
           0.739604243435928, 0.740429472034602,
           0.741253585995320, 0.742076584077467,
           0.742898465042106, 0.743719227651982,
           0.744538870671524, 0.745357392866848,
           0.746174793005753, 0.746991069857732,
           0.747806222193967, 0.748620248787332,
           0.749433148412396, 0.750244919845424,
           0.751055561864384, 0.751865073248937,
           0.752673452780451, 0.753480699241996,
           0.754286811418348, 0.755091788095991,
           0.755895628063119, 0.756698330109635,
           0.757499893027156, 0.758300315609015,
           0.759099596650261, 0.759897734947661,
           0.760694729299702, 0.761490578506594,
           0.762285281370270, 0.763078836694390,
           0.763871243284342, 0.764662499947239,
           0.765452605491930, 0.766241558728994,
           0.767029358470746, 0.767816003531238,
           0.768601492726259, 0.769385824873338,
           0.770168998791746, 0.770951013302499,
           0.771731867228355, 0.772511559393824,
           0.773290088625161, 0.774067453750372,
           0.774843653599217, 0.775618687003210,
           0.776392552795621, 0.777165249811476,
           0.777936776887562, 0.778707132862427,
           0.779476316576382, 0.780244326871502,
           0.781011162591630, 0.781776822582374,
           0.782541305691119, 0.783304610767012,
           0.784066736660980, 0.784827682225724,
           0.785587446315720, 0.786346027787226,
           0.787103425498275, 0.787859638308687,
           0.788614665080064, 0.789368504675794,
           0.790121155961049, 0.790872617802795,
           0.791622889069785, 0.792371968632564,
           0.793119855363475, 0.793866548136651,
           0.794612045828028, 0.795356347315336,
           0.796099451478110, 0.796841357197685,
           0.797582063357200, 0.798321568841602,
           0.799059872537642, 0.799796973333883,
           0.800532870120699, 0.801267561790274,
           0.802001047236610, 0.802733325355520,
           0.803464395044637, 0.804194255203416,
           0.804922904733128, 0.805650342536868,
           0.806376567519556, 0.807101578587940,
           0.807825374650588, 0.808547954617906,
           0.809269317402124, 0.809989461917309,
           0.810708387079358, 0.811426091806008,
           0.812142575016828, 0.812857835633229,
           0.813571872578464, 0.814284684777624,
           0.814996271157647, 0.815706630647315,
           0.816415762177258, 0.817123664679953,
           0.817830337089728, 0.818535778342763,
           0.819239987377091, 0.819942963132599,
           0.820644704551034, 0.821345210575996,
           0.822044480152950, 0.822742512229218,
           0.823439305753988, 0.824134859678311,
           0.824829172955104, 0.825522244539153,
           0.826214073387112, 0.826904658457506,
           0.827593998710732, 0.828282093109061,
           0.828968940616642, 0.829654540199498,
           0.830338890825531, 0.831021991464524,
           0.831703841088142, 0.832384438669933,
           0.833063783185328, 0.833741873611648,
           0.834418708928098, 0.835094288115776,
           0.835768610157668, 0.836441674038655,
           0.837113478745510, 0.837784023266901,
           0.838453306593397, 0.839121327717462,
           0.839788085633461, 0.840453579337660,
           0.841117807828230, 0.841780770105245,
           0.842442465170686, 0.843102892028440,
           0.843762049684308, 0.844419937145993,
           0.845076553423120, 0.845731897527220,
           0.846385968471743, 0.847038765272055,
           0.847690286945440, 0.848340532511098,
           0.848989500990159, 0.849637191405665,
           0.850283602782590, 0.850928734147828,
           0.851572584530203, 0.852215152960467,
           0.852856438471302, 0.853496440097319,
           0.854135156875065, 0.854772587843020,
           0.855408732041598, 0.856043588513152,
           0.856677156301972, 0.857309434454291,
           0.857940422018280, 0.858570118044056,
           0.859198521583676, 0.859825631691145,
           0.860451447422419, 0.861075967835394,
           0.861699191989925, 0.862321118947812,
           0.862941747772810, 0.863561077530629,
           0.864179107288934, 0.864795836117346,
           0.865411263087445, 0.866025387272771,
           0.866638207748828, 0.867249723593076,
           0.867859933884944, 0.868468837705827,
           0.869076434139083, 0.869682722270040,
           0.870287701185997, 0.870891369976222,
           0.871493727731955, 0.872094773546411,
           0.872694506514780, 0.873292925734225,
           0.873890030303892, 0.874485819324904,
           0.875080291900361, 0.875673447135350,
           0.876265284136938, 0.876855802014177,
           0.877444999878106, 0.878032876841748,
           0.878619432020120, 0.879204664530223,
           0.879788573491052, 0.880371158023594,
           0.880952417250830, 0.881532350297736,
           0.882110956291285, 0.882688234360446,
           0.883264183636189, 0.883838803251484,
           0.884412092341299, 0.884984050042612,
           0.885554675494400, 0.886123967837647,
           0.886691926215343, 0.887258549772488,
           0.887823837656090, 0.888387789015168,
           0.888950403000754, 0.889511678765892,
           0.890071615465639, 0.890630212257072,
           0.891187468299283, 0.891743382753381,
           0.892297954782495, 0.892851183551778,
           0.893403068228402, 0.893953607981562,
           0.894502801982480, 0.895050649404403,
           0.895597149422604, 0.896142301214385,
           0.896686103959079, 0.897228556838049,
           0.897769659034688, 0.898309409734426,
           0.898847808124724, 0.899384853395081,
           0.899920544737032, 0.900454881344152,
           0.900987862412052, 0.901519487138388,
           0.902049754722853, 0.902578664367187,
           0.903106215275172, 0.903632406652636,
           0.904157237707454, 0.904680707649548,
           0.905202815690891, 0.905723561045504,
           0.906242942929460, 0.906760960560885,
           0.907277613159957, 0.907792899948913,
           0.908306820152041, 0.908819372995690,
           0.909330557708267, 0.909840373520236,
           0.910348819664124, 0.910855895374519,
           0.911361599888073, 0.911865932443502,
           0.912368892281584, 0.912870478645170,
           0.913370690779173, 0.913869527930578,
           0.914366989348437, 0.914863074283876,
           0.915357781990090, 0.915851111722352,
           0.916343062738004, 0.916833634296467,
           0.917322825659238, 0.917810636089890,
           0.918297064854077, 0.918782111219533,
           0.919265774456070, 0.919748053835586,
           0.920228948632061, 0.920708458121557,
           0.921186581582225, 0.921663318294301,
           0.922138667540107, 0.922612628604056,
           0.923085200772652, 0.923556383334484,
           0.924026175580239, 0.924494576802693,
           0.924961586296721, 0.925427203359288,
           0.925891427289454, 0.926354257388383,
           0.926815692959332, 0.927275733307660,
           0.927734377740823, 0.928191625568382,
           0.928647476101998, 0.929101928655437,
           0.929554982544569, 0.930006637087370,
           0.930456891603920, 0.930905745416411,
           0.931353197849140, 0.931799248228515,
           0.932243895883054, 0.932687140143388,
           0.933128980342260, 0.933569415814526,
           0.934008445897157, 0.934446069929241,
           0.934882287251979, 0.935317097208695,
           0.935750499144827, 0.936182492407934,
           0.936613076347698, 0.937042250315920,
           0.937470013666523, 0.937896365755555,
           0.938321305941188, 0.938744833583721,
           0.939166948045577, 0.939587648691308,
           0.940006934887593, 0.940424806003243,
           0.940841261409196, 0.941256300478522,
           0.941669922586424, 0.942082127110239,
           0.942492913429437, 0.942902280925621,
           0.943310228982534, 0.943716756986050,
           0.944121864324189, 0.944525550387101,
           0.944927814567081, 0.945328656258563,
           0.945728074858122, 0.946126069764477,
           0.946522640378487, 0.946917786103158,
           0.947311506343640, 0.947703800507229,
           0.948094668003368, 0.948484108243647,
           0.948872120641806, 0.949258704613732,
           0.949643859577466, 0.950027584953196,
           0.950409880163266, 0.950790744632169,
           0.951170177786555, 0.951548179055228,
           0.951924747869148, 0.952299883661429,
           0.952673585867345, 0.953045853924326,
           0.953416687271962, 0.953786085352004,
           0.954154047608362, 0.954520573487108,
           0.954885662436475, 0.955249313906862,
           0.955611527350829, 0.955972302223103,
           0.956331637980575, 0.956689534082303,
           0.957045989989514, 0.957401005165600,
           0.957754579076124, 0.958106711188817,
           0.958457400973583, 0.958806647902495,
           0.959154451449798, 0.959500811091912,
           0.959845726307430, 0.960189196577117,
           0.960531221383914, 0.960871800212942,
           0.961210932551495, 0.961548617889044,
           0.961884855717240, 0.962219645529913,
           0.962552986823073, 0.962884879094908,
           0.963215321845792, 0.963544314578277,
           0.963871856797099, 0.964197948009179,
           0.964522587723621, 0.964845775451714,
           0.965167510706935, 0.965487793004944,
           0.965806621863590, 0.966123996802909,
           0.966439917345129, 0.966754383014664,
           0.967067393338120, 0.967378947844292,
           0.967689046064169, 0.967997687530931,
           0.968304871779950, 0.968610598348793,
           0.968914866777221, 0.969217676607192,
           0.969519027382855, 0.969818918650561,
           0.970117349958854, 0.970414320858478,
           0.970709830902375, 0.971003879645685,
           0.971296466645750, 0.971587591462111,
           0.971877253656510, 0.972165452792891,
           0.972452188437403, 0.972737460158393,
           0.973021267526417, 0.973303610114232,
           0.973584487496802, 0.973863899251294,
           0.974141844957086, 0.974418324195758,
           0.974693336551101, 0.974966881609113,
           0.975238958958001, 0.975509568188181,
           0.975778708892280, 0.976046380665136,
           0.976312583103795, 0.976577315807520,
           0.976840578377782, 0.977102370418270,
           0.977362691534881, 0.977621541335730,
           0.977878919431147, 0.978134825433677,
           0.978389258958079, 0.978642219621331,
           0.978893707042628, 0.979143720843381,
           0.979392260647222, 0.979639326080002,
           0.979884916769788, 0.980129032346870,
           0.980371672443758, 0.980612836695183,
           0.980852524738097, 0.981090736211678,
           0.981327470757320, 0.981562728018647,
           0.981796507641504, 0.982028809273959,
           0.982259632566308, 0.982488977171071,
           0.982716842742993, 0.982943228939047,
           0.983168135418433, 0.983391561842578,
           0.983613507875138, 0.983833973181994,
           0.984052957431264, 0.984270460293285,
           0.984486481440632, 0.984701020548109,
           0.984914077292749, 0.985125651353818,
           0.985335742412813, 0.985544350153465,
           0.985751474261737, 0.985957114425826,
           0.986161270336162, 0.986363941685410,
           0.986565128168470, 0.986764829482477,
           0.986963045326802, 0.987159775403052,
           0.987355019415071, 0.987548777068940,
           0.987741048072976, 0.987931832137737,
           0.988121128976018, 0.988308938302852,
           0.988495259835512, 0.988680093293510,
           0.988863438398600, 0.989045294874776,
           0.989225662448270, 0.989404540847560,
           0.989581929803363, 0.989757829048637,
           0.989932238318588, 0.990105157350659,
           0.990276585884538, 0.990446523662159,
           0.990614970427698, 0.990781925927577,
           0.990947389910462, 0.991111362127263,
           0.991273842331140, 0.991434830277494,
           0.991594325723976, 0.991752328430482,
           0.991908838159156, 0.992063854674390,
           0.992217377742821, 0.992369407133338,
           0.992519942617076, 0.992668983967420,
           0.992816530960005, 0.992962583372714,
           0.993107140985680, 0.993250203581288,
           0.993391770944171, 0.993531842861216,
           0.993670419121559, 0.993807499516588,
           0.993943083839942, 0.994077171887516,
           0.994209763457452, 0.994340858350148,
           0.994470456368255, 0.994598557316678,
           0.994725161002572, 0.994850267235351,
           0.994973875826680, 0.995095986590480,
           0.995216599342924, 0.995335713902445,
           0.995453330089728, 0.995569447727713,
           0.995684066641599, 0.995797186658838,
           0.995908807609141, 0.996018929324473,
           0.996127551639059, 0.996234674389380,
           0.996340297414172, 0.996444420554431,
           0.996547043653414, 0.996648166556628,
           0.996747789111848, 0.996845911169099,
           0.996942532580670, 0.997037653201109,
           0.997131272887220, 0.997223391498070,
           0.997314008894984, 0.997403124941546,
           0.997490739503603, 0.997576852449260,
           0.997661463648882, 0.997744572975098,
           0.997826180302793, 0.997906285509119,
           0.997984888473484, 0.998061989077560,
           0.998137587205282, 0.998211682742842,
           0.998284275578701, 0.998355365603575,
           0.998424952710446, 0.998493036794560,
           0.998559617753423, 0.998624695486804,
           0.998688269896736, 0.998750340887514,
           0.998810908365698, 0.998869972240110,
           0.998927532421835, 0.998983588824224,
           0.999038141362890, 0.999091189955710,
           0.999142734522825, 0.999192774986642,
           0.999241311271830, 0.999288343305324,
           0.999333871016323, 0.999377894336292,
           0.999420413198958, 0.999461427540316,
           0.999500937298624, 0.999538942414408,
           0.999575442830458, 0.999610438491828,
           0.999643929345840, 0.999675915342084,
           0.999706396432413, 0.999735372570949,
           0.999762843714083, 0.999788809820473,
           0.999813270851047, 0.999836226769006,
           0.999857677539826, 0.999877623131260,
           0.999896063513350, 0.999912998658436,
           0.999928428541177, 0.999942353138586,
           0.999954772430114, 0.999965686397793,
           0.999975095026600, 0.999982998305366,
           0.999989396229529, 0.999994288811461,
           0.999997676135955, 0.999999558950266};

  } else if (N == 5120) {

    x_k = {0.000306766193667, 0.000920298465526,
           0.001533830390965, 0.002147361739037,
           0.002760892278794, 0.003374421779291,
           0.003987950009581, 0.004601476738719,
           0.005215001735760, 0.005828524769760,
           0.006442045609775, 0.007055564024861,
           0.007669079784079, 0.008282592656486,
           0.008896102411143, 0.009509608817112,
           0.010123111643454, 0.010736610659235,
           0.011350105633519, 0.011963596335375,
           0.012577082533869, 0.013190563998073,
           0.013804040497058, 0.014417511799899,
           0.015030977675672, 0.015644437893453,
           0.016257892222324, 0.016871340431367,
           0.017484782289667, 0.018098217566310,
           0.018711646030385, 0.019325067450986,
           0.019938481597206, 0.020551888238143,
           0.021165287142898, 0.021778678080572,
           0.022392060820274, 0.023005435131111,
           0.023618800782197, 0.024232157542646,
           0.024845505181578, 0.025458843468115,
           0.026072172171385, 0.026685491060515,
           0.027298799904639, 0.027912098472895,
           0.028525386534423, 0.029138663858368,
           0.029751930213879, 0.030365185370109,
           0.030978429096216, 0.031591661161361,
           0.032204881334709, 0.032818089385431,
           0.033431285082704, 0.034044468195705,
           0.034657638493619, 0.035270795745637,
           0.035883939720951, 0.036497070188761,
           0.037110186918270, 0.037723289678690,
           0.038336378239233, 0.038949452369120,
           0.039562511837577, 0.040175556413834,
           0.040788585867129, 0.041401599966702,
           0.042014598481802, 0.042627581181684,
           0.043240547835607, 0.043853498212836,
           0.044466432082644, 0.045079349214309,
           0.045692249377116, 0.046305132340355,
           0.046917997873323, 0.047530845745325,
           0.048143675725673, 0.048756487583681,
           0.049369281088675, 0.049982056009986,
           0.050594812116951, 0.051207549178917,
           0.051820266965234, 0.052432965245264,
           0.053045643788372, 0.053658302363933,
           0.054270940741329, 0.054883558689950,
           0.055496155979191, 0.056108732378460,
           0.056721287657167, 0.057333821584734,
           0.057946333930589, 0.058558824464170,
           0.059171292954922, 0.059783739172297,
           0.060396162885758, 0.061008563864776,
           0.061620941878827, 0.062233296697401,
           0.062845628089994, 0.063457935826109,
           0.064070219675262, 0.064682479406975,
           0.065294714790781, 0.065906925596219,
           0.066519111592841, 0.067131272550207,
           0.067743408237885, 0.068355518425454,
           0.068967602882503, 0.069579661378630,
           0.070191693683440, 0.070803699566554,
           0.071415678797599, 0.072027631146211,
           0.072639556382039, 0.073251454274740,
           0.073863324593982, 0.074475167109444,
           0.075086981590816, 0.075698767807796,
           0.076310525530095, 0.076922254527435,
           0.077533954569545, 0.078145625426169,
           0.078757266867062, 0.079368878661987,
           0.079980460580721, 0.080592012393050,
           0.081203533868773, 0.081815024777700,
           0.082426484889652, 0.083037913974463,
           0.083649311801976, 0.084260678142049,
           0.084872012764549, 0.085483315439357,
           0.086094585936365, 0.086705824025477,
           0.087317029476610, 0.087928202059693,
           0.088539341544666, 0.089150447701484,
           0.089761520300113, 0.090372559110531,
           0.090983563902731, 0.091594534446716,
           0.092205470512503, 0.092816371870123,
           0.093427238289621, 0.094038069541050,
           0.094648865394482, 0.095259625619999,
           0.095870349987698, 0.096481038267688,
           0.097091690230094, 0.097702305645053,
           0.098312884282714, 0.098923425913244,
           0.099533930306820, 0.100144397233636,
           0.100754826463898, 0.101365217767828,
           0.101975570915660, 0.102585885677644,
           0.103196161824045, 0.103806399125140,
           0.104416597351225, 0.105026756272605,
           0.105636875659604, 0.106246955282560,
           0.106856994911825, 0.107466994317767,
           0.108076953270769, 0.108686871541229,
           0.109296748899559, 0.109906585116190,
           0.110516379961564, 0.111126133206143,
           0.111735844620401, 0.112345513974829,
           0.112955141039935, 0.113564725586242,
           0.114174267384288, 0.114783766204629,
           0.115393221817836, 0.116002633994496,
           0.116612002505212, 0.117221327120607,
           0.117830607611316, 0.118439843747993,
           0.119049035301308, 0.119658182041947,
           0.120267283740615, 0.120876340168032,
           0.121485351094937, 0.122094316292082,
           0.122703235530243, 0.123312108580206,
           0.123920935212780, 0.124529715198789,
           0.125138448309073, 0.125747134314494,
           0.126355772985927, 0.126964364094268,
           0.127572907410430, 0.128181402705344,
           0.128789849749959, 0.129398248315242,
           0.130006598172178, 0.130614899091771,
           0.131223150845044, 0.131831353203036,
           0.132439505936808, 0.133047608817436,
           0.133655661616019, 0.134263664103671,
           0.134871616051527, 0.135479517230740,
           0.136087367412483, 0.136695166367948,
           0.137302913868346, 0.137910609684907,
           0.138518253588882, 0.139125845351539,
           0.139733384744168, 0.140340871538078,
           0.140948305504597, 0.141555686415074,
           0.142163014040878, 0.142770288153396,
           0.143377508524038, 0.143984674924231,
           0.144591787125427, 0.145198844899094,
           0.145805848016722, 0.146412796249822,
           0.147019689369925, 0.147626527148584,
           0.148233309357370, 0.148840035767879,
           0.149446706151725, 0.150053320280544,
           0.150659877925994, 0.151266378859751,
           0.151872822853516, 0.152479209679011,
           0.153085539107978, 0.153691810912180,
           0.154298024863405, 0.154904180733459,
           0.155510278294173, 0.156116317317398,
           0.156722297575007, 0.157328218838897,
           0.157934080880985, 0.158539883473210,
           0.159145626387538, 0.159751309395951,
           0.160356932270458, 0.160962494783089,
           0.161567996705896, 0.162173437810956,
           0.162778817870367, 0.163384136656251,
           0.163989393940752, 0.164594589496040,
           0.165199723094302, 0.165804794507756,
           0.166409803508639, 0.167014749869212,
           0.167619633361759, 0.168224453758591,
           0.168829210832037, 0.169433904354455,
           0.170038534098226, 0.170643099835752,
           0.171247601339462, 0.171852038381809,
           0.172456410735269, 0.173060718172343,
           0.173664960465556, 0.174269137387459,
           0.174873248710625, 0.175477294207654,
           0.176081273651171, 0.176685186813824,
           0.177289033468285, 0.177892813387256,
           0.178496526343459, 0.179100172109644,
           0.179703750458585, 0.180307261163082,
           0.180910703995960, 0.181514078730070,
           0.182117385138289, 0.182720622993518,
           0.183323792068686, 0.183926892136747,
           0.184529922970680, 0.185132884343492,
           0.185735776028214, 0.186338597797905,
           0.186941349425650, 0.187544030684559,
           0.188146641347771, 0.188749181188449,
           0.189351649979782, 0.189954047494991,
           0.190556373507319, 0.191158627790036,
           0.191760810116442, 0.192362920259860,
           0.192964957993645, 0.193566923091176,
           0.194168815325859, 0.194770634471130,
           0.195372380300450, 0.195974052587310,
           0.196575651105225, 0.197177175627742,
           0.197778625928434, 0.198380001780901,
           0.198981302958772, 0.199582529235705,
           0.200183680385383, 0.200784756181522,
           0.201385756397862, 0.201986680808174,
           0.202587529186258, 0.203188301305939,
           0.203788996941075, 0.204389615865550,
           0.204990157853279, 0.205590622678204,
           0.206191010114297, 0.206791319935558,
           0.207391551916019, 0.207991705829738,
           0.208591781450805, 0.209191778553337,
           0.209791696911482, 0.210391536299419,
           0.210991296491354, 0.211590977261525,
           0.212190578384198, 0.212790099633671,
           0.213389540784270, 0.213988901610353,
           0.214588181886306, 0.215187381386548,
           0.215786499885527, 0.216385537157722,
           0.216984492977641, 0.217583367119825,
           0.218182159358844, 0.218780869469299,
           0.219379497225823, 0.219978042403080,
           0.220576504775764, 0.221174884118599,
           0.221773180206343, 0.222371392813785,
           0.222969521715744, 0.223567566687071,
           0.224165527502648, 0.224763403937392,
           0.225361195766246, 0.225958902764189,
           0.226556524706232, 0.227154061367416,
           0.227751512522815, 0.228348877947536,
           0.228946157416717, 0.229543350705529,
           0.230140457589176, 0.230737477842892,
           0.231334411241946, 0.231931257561641,
           0.232528016577309, 0.233124688064317,
           0.233721271798065, 0.234317767553986,
           0.234914175107547, 0.235510494234245,
           0.236106724709613, 0.236702866309216,
           0.237298918808656, 0.237894881983562,
           0.238490755609603, 0.239086539462478,
           0.239682233317922, 0.240277836951701,
           0.240873350139618, 0.241468772657508,
           0.242064104281241, 0.242659344786722,
           0.243254493949888, 0.243849551546712,
           0.244444517353202, 0.245039391145399,
           0.245634172699380, 0.246228861791256,
           0.246823458197172, 0.247417961693312,
           0.248012372055888, 0.248606689061152,
           0.249200912485391, 0.249795042104926,
           0.250389077696113, 0.250983019035344,
           0.251576865899046, 0.252170618063683,
           0.252764275305754, 0.253357837401791,
           0.253951304128367, 0.254544675262086,
           0.255137950579590, 0.255731129857559,
           0.256324212872706, 0.256917199401782,
           0.257510089221573, 0.258102882108903,
           0.258695577840631, 0.259288176193655,
           0.259880676944906, 0.260473079871354,
           0.261065384750006, 0.261657591357906,
           0.262249699472134, 0.262841708869806,
           0.263433619328079, 0.264025430624143,
           0.264617142535228, 0.265208754838601,
           0.265800267311566, 0.266391679731464,
           0.266982991875674, 0.267574203521615,
           0.268165314446740, 0.268756324428543,
           0.269347233244555, 0.269938040672344,
           0.270528746489517, 0.271119350473720,
           0.271709852402636, 0.272300252053989,
           0.272890549205538, 0.273480743635081,
           0.274070835120459, 0.274660823439545,
           0.275250708370257, 0.275840489690549,
           0.276430167178413, 0.277019740611883,
           0.277609209769030, 0.278198574427964,
           0.278787834366836, 0.279376989363836,
           0.279966039197192, 0.280554983645174,
           0.281143822486090, 0.281732555498287,
           0.282321182460153, 0.282909703150118,
           0.283498117346648, 0.284086424828251,
           0.284674625373476, 0.285262718760910,
           0.285850704769184, 0.286438583176964,
           0.287026353762964, 0.287614016305929,
           0.288201570584655, 0.288789016377969,
           0.289376353464746, 0.289963581623899,
           0.290550700634383, 0.291137710275193,
           0.291724610325366, 0.292311400563979,
           0.292898080770151, 0.293484650723045,
           0.294071110201860, 0.294657458985842,
           0.295243696854276, 0.295829823586489,
           0.296415838961849, 0.297001742759769,
           0.297587534759701, 0.298173214741139,
           0.298758782483621, 0.299344237766726,
           0.299929580370077, 0.300514810073337,
           0.301099926656213, 0.301684929898454,
           0.302269819579853, 0.302854595480243,
           0.303439257379503, 0.304023805057552,
           0.304608238294355, 0.305192556869917,
           0.305776760564288, 0.306360849157562,
           0.306944822429873, 0.307528680161403,
           0.308112422132373, 0.308696048123052,
           0.309279557913748, 0.309862951284817,
           0.310446228016655, 0.311029387889706,
           0.311612430684454, 0.312195356181429,
           0.312778164161207, 0.313360854404404,
           0.313943426691683, 0.314525880803751,
           0.315108216521361, 0.315690433625308,
           0.316272531896431, 0.316854511115618,
           0.317436371063798, 0.318018111521946,
           0.318599732271083, 0.319181233092273,
           0.319762613766627, 0.320343874075300,
           0.320925013799494, 0.321506032720453,
           0.322086930619471, 0.322667707277885,
           0.323248362477076, 0.323828895998475,
           0.324409307623554, 0.324989597133836,
           0.325569764310885, 0.326149808936314,
           0.326729730791782, 0.327309529658993,
           0.327889205319700, 0.328468757555696,
           0.329048186148828, 0.329627490880987,
           0.330206671534107, 0.330785727890175,
           0.331364659731218, 0.331943466839316,
           0.332522148996593, 0.333100705985218,
           0.333679137587412, 0.334257443585440,
           0.334835623761614, 0.335413677898295,
           0.335991605777890, 0.336569407182856,
           0.337147081895694, 0.337724629698956,
           0.338302050375239, 0.338879343707190,
           0.339456509477502, 0.340033547468918,
           0.340610457464228, 0.341187239246271,
           0.341763892597933, 0.342340417302149,
           0.342916813141902, 0.343493079900225,
           0.344069217360198, 0.344645225304951,
           0.345221103517659, 0.345796851781552,
           0.346372469879905, 0.346947957596042,
           0.347523314713336, 0.348098541015213,
           0.348673636285141, 0.349248600306645,
           0.349823432863294, 0.350398133738709,
           0.350972702716560, 0.351547139580566,
           0.352121444114497, 0.352695616102172,
           0.353269655327459, 0.353843561574279,
           0.354417334626600, 0.354990974268441,
           0.355564480283872, 0.356137852457013,
           0.356711090572032, 0.357284194413151,
           0.357857163764642, 0.358429998410826,
           0.359002698136075, 0.359575262724812,
           0.360147691961512, 0.360719985630700,
           0.361292143516952, 0.361864165404895,
           0.362436051079207, 0.363007800324618,
           0.363579412925909, 0.364150888667912,
           0.364722227335512, 0.365293428713644,
           0.365864492587295, 0.366435418741504,
           0.367006206961362, 0.367576857032013,
           0.368147368738649, 0.368717741866519,
           0.369287976200921, 0.369858071527208,
           0.370428027630781, 0.370997844297097,
           0.371567521311666, 0.372137058460047,
           0.372706455527854, 0.373275712300755,
           0.373844828564467, 0.374413804104763,
           0.374982638707470, 0.375551332158464,
           0.376119884243676, 0.376688294749091,
           0.377256563460748, 0.377824690164736,
           0.378392674647202, 0.378960516694342,
           0.379528216092409, 0.380095772627708,
           0.380663186086599, 0.381230456255494,
           0.381797582920861, 0.382364565869220,
           0.382931404887147, 0.383498099761271,
           0.384064650278275, 0.384631056224899,
           0.385197317387933, 0.385763433554224,
           0.386329404510676, 0.386895230044242,
           0.387460909941934, 0.388026443990818,
           0.388591831978014, 0.389157073690699,
           0.389722168916101, 0.390287117441508,
           0.390851919054259, 0.391416573541753,
           0.391981080691439, 0.392545440290824,
           0.393109652127473, 0.393673715989003,
           0.394237631663088, 0.394801398937457,
           0.395365017599895, 0.395928487438247,
           0.396491808240407, 0.397054979794330,
           0.397618001888026, 0.398180874309561,
           0.398743596847058, 0.399306169288695,
           0.399868591422707, 0.400430863037387,
           0.400992983921082, 0.401554953862200,
           0.402116772649201, 0.402678440070605,
           0.403239955914987, 0.403801319970981,
           0.404362532027277, 0.404923591872623,
           0.405484499295822, 0.406045254085738,
           0.406605856031290, 0.407166304921455,
           0.407726600545267, 0.408286742691820,
           0.408846731150262, 0.409406565709803,
           0.409966246159708, 0.410525772289302,
           0.411085143887966, 0.411644360745139,
           0.412203422650323, 0.412762329393071,
           0.413321080763001, 0.413879676549786,
           0.414438116543157, 0.414996400532906,
           0.415554528308883, 0.416112499660995,
           0.416670314379211, 0.417227972253556,
           0.417785473074116, 0.418342816631035,
           0.418900002714517, 0.419457031114825,
           0.420013901622281, 0.420570614027266,
           0.421127168120222, 0.421683563691651,
           0.422239800532111, 0.422795878432223,
           0.423351797182668, 0.423907556574185,
           0.424463156397575, 0.425018596443697,
           0.425573876503471, 0.426128996367879,
           0.426683955827960, 0.427238754674815,
           0.427793392699607, 0.428347869693557,
           0.428902185447947, 0.429456339754121,
           0.430010332403484, 0.430564163187501,
           0.431117831897696, 0.431671338325658,
           0.432224682263034, 0.432777863501533,
           0.433330881832925, 0.433883737049043,
           0.434436428941780, 0.434988957303090,
           0.435541321924989, 0.436093522599555,
           0.436645559118928, 0.437197431275310,
           0.437749138860962, 0.438300681668211,
           0.438852059489443, 0.439403272117108,
           0.439954319343717, 0.440505200961844,
           0.441055916764125, 0.441606466543258,
           0.442156850092004, 0.442707067203188,
           0.443257117669695, 0.443807001284474,
           0.444356717840536, 0.444906267130957,
           0.445455648948872, 0.446004863087485,
           0.446553909340058, 0.447102787499918,
           0.447651497360455, 0.448200038715123,
           0.448748411357439, 0.449296615080983,
           0.449844649679399, 0.450392514946396,
           0.450940210675744, 0.451487736661278,
           0.452035092696898, 0.452582278576568,
           0.453129294094314, 0.453676139044228,
           0.454222813220464, 0.454769316417244,
           0.455315648428851, 0.455861809049633,
           0.456407798074005, 0.456953615296443,
           0.457499260511490, 0.458044733513754,
           0.458590034097905, 0.459135162058681,
           0.459680117190885, 0.460224899289381,
           0.460769508149104, 0.461313943565050,
           0.461858205332281, 0.462402293245925,
           0.462946207101177, 0.463489946693295,
           0.464033511817602, 0.464576902269489,
           0.465120117844413, 0.465663158337894,
           0.466206023545521, 0.466748713262947,
           0.467291227285892, 0.467833565410141,
           0.468375727431547, 0.468917713146028,
           0.469459522349567, 0.470001154838217,
           0.470542610408096, 0.471083888855387,
           0.471624989976342, 0.472165913567276,
           0.472706659424577, 0.473247227344694,
           0.473787617124147, 0.474327828559520,
           0.474867861447466, 0.475407715584704,
           0.475947390768022, 0.476486886794275,
           0.477026203460384, 0.477565340563338,
           0.478104297900194, 0.478643075268077,
           0.479181672464179, 0.479720089285761,
           0.480258325530150, 0.480796380994743,
           0.481334255477004, 0.481871948774463,
           0.482409460684724, 0.482946791005454,
           0.483483939534388, 0.484020906069336,
           0.484557690408168, 0.485094292348828,
           0.485630711689328, 0.486166948227747,
           0.486703001762233, 0.487238872091006,
           0.487774559012350, 0.488310062324623,
           0.488845381826248, 0.489380517315720,
           0.489915468591602, 0.490450235452526,
           0.490984817697196, 0.491519215124380,
           0.492053427532922, 0.492587454721731,
           0.493121296489789, 0.493654952636145,
           0.494188422959919, 0.494721707260302,
           0.495254805336553, 0.495787716988003,
           0.496320442014052, 0.496852980214170,
           0.497385331387900, 0.497917495334851,
           0.498449471854705, 0.498981260747216,
           0.499512861812205, 0.500044274849567,
           0.500575499659267, 0.501106536041338,
           0.501637383795888, 0.502168042723093,
           0.502698512623203, 0.503228793296537,
           0.503758884543485, 0.504288786164508,
           0.504818497960142, 0.505348019730989,
           0.505877351277728, 0.506406492401105,
           0.506935442901942, 0.507464202581128,
           0.507992771239625, 0.508521148678473,
           0.509049334698775, 0.509577329101712,
           0.510105131688533, 0.510632742260565,
           0.511160160619202, 0.511687386565912,
           0.512214419902235, 0.512741260429785,
           0.513267907950248, 0.513794362265381,
           0.514320623177016, 0.514846690487057,
           0.515372563997479, 0.515898243510333,
           0.516423728827743, 0.516949019751902,
           0.517474116085080, 0.517999017629620,
           0.518523724187938, 0.519048235562521,
           0.519572551555932, 0.520096671970807,
           0.520620596609856, 0.521144325275861,
           0.521667857771681, 0.522191193900247,
           0.522714333464560, 0.523237276267704,
           0.523760022112827, 0.524282570803159,
           0.524804922142000, 0.525327075932727,
           0.525849031978788, 0.526370790083707,
           0.526892350051085, 0.527413711684593,
           0.527934874787980, 0.528455839165070,
           0.528976604619758, 0.529497170956019,
           0.530017537977898, 0.530537705489519,
           0.531057673295079, 0.531577441198851,
           0.532097009005182, 0.532616376518497,
           0.533135543543293, 0.533654509884144,
           0.534173275345702, 0.534691839732689,
           0.535210202849908, 0.535728364502236,
           0.536246324494624, 0.536764082632101,
           0.537281638719774, 0.537798992562819,
           0.538316143966495, 0.538833092736136,
           0.539349838677149, 0.539866381595021,
           0.540382721295312, 0.540898857583663,
           0.541414790265786, 0.541930519147475,
           0.542446044034598, 0.542961364733099,
           0.543476481049000, 0.543991392788401,
           0.544506099757477, 0.545020601762481,
           0.545534898609743, 0.546048990105672,
           0.546562876056751, 0.547076556269542,
           0.547590030550685, 0.548103298706896,
           0.548616360544972, 0.549129215871783,
           0.549641864494280, 0.550154306219489,
           0.550666540854519, 0.551178568206551,
           0.551690388082847, 0.552202000290747,
           0.552713404637670, 0.553224600931110,
           0.553735588978644, 0.554246368587922,
           0.554756939566677, 0.555267301722720,
           0.555777454863938, 0.556287398798297,
           0.556797133333845, 0.557306658278705,
           0.557815973441082, 0.558325078629258,
           0.558833973651595, 0.559342658316534,
           0.559851132432593, 0.560359395808373,
           0.560867448252553, 0.561375289573888,
           0.561882919581217, 0.562390338083459,
           0.562897544889608, 0.563404539808740,
           0.563911322650012, 0.564417893222660,
           0.564924251335999, 0.565430396799424,
           0.565936329422413, 0.566442049014520,
           0.566947555385382, 0.567452848344713,
           0.567957927702312, 0.568462793268057,
           0.568967444851902, 0.569471882263888,
           0.569976105314133, 0.570480113812835,
           0.570983907570275, 0.571487486396815,
           0.571990850102896, 0.572493998499040,
           0.572996931395852, 0.573499648604015,
           0.574002149934298, 0.574504435197548,
           0.575006504204692, 0.575508356766742,
           0.576009992694788, 0.576511411800004,
           0.577012613893646, 0.577513598787048,
           0.578014366291629, 0.578514916218892,
           0.579015248380415, 0.579515362587865,
           0.580015258652986, 0.580514936387607,
           0.581014395603637, 0.581513636113072,
           0.582012657727984, 0.582511460260529,
           0.583010043522951, 0.583508407327568,
           0.584006551486789, 0.584504475813098,
           0.585002180119066, 0.585499664217349,
           0.585996927920680, 0.586493971041879,
           0.586990793393848, 0.587487394789572,
           0.587983775042120, 0.588479933964642,
           0.588975871370374, 0.589471587072635,
           0.589967080884826, 0.590462352620432,
           0.590957402093021, 0.591452229116247,
           0.591946833503845, 0.592441215069636,
           0.592935373627524, 0.593429308991497,
           0.593923020975624, 0.594416509394066,
           0.594909774061058, 0.595402814790928,
           0.595895631398083, 0.596388223697015,
           0.596880591502305, 0.597372734628611,
           0.597864652890681, 0.598356346103347,
           0.598847814081525, 0.599339056640213,
           0.599830073594499, 0.600320864759553,
           0.600811429950630, 0.601301768983071,
           0.601791881672300, 0.602281767833830,
           0.602771427283256, 0.603260859836259,
           0.603750065308606, 0.604239043516150,
           0.604727794274829, 0.605216317400665,
           0.605704612709769, 0.606192680018335,
           0.606680519142642, 0.607168129899060,
           0.607655512104040, 0.608142665574120,
           0.608629590125925, 0.609116285576166,
           0.609602751741640, 0.610088988439232,
           0.610574995485910, 0.611060772698731,
           0.611546319894836, 0.612031636891457,
           0.612516723505910, 0.613001579555596,
           0.613486204858004, 0.613970599230712,
           0.614454762491384, 0.614938694457768,
           0.615422394947703, 0.615905863779113,
           0.616389100770010, 0.616872105738492,
           0.617354878502746, 0.617837418881045,
           0.618319726691752, 0.618801801753315,
           0.619283643884270, 0.619765252903239,
           0.620246628628938, 0.620727770880163,
           0.621208679475803, 0.621689354234834,
           0.622169794976317, 0.622650001519408,
           0.623129973683342, 0.623609711287449,
           0.624089214151146, 0.624568482093935,
           0.625047514935412, 0.625526312495257,
           0.626004874593239, 0.626483201049218,
           0.626961291683143, 0.627439146315048,
           0.627916764765058, 0.628394146853388,
           0.628871292400339, 0.629348201226306,
           0.629824873151767, 0.630301307997294,
           0.630777505583545, 0.631253465731271,
           0.631729188261307, 0.632204672994584,
           0.632679919752117, 0.633154928355012,
           0.633629698624469, 0.634104230381768,
           0.634578523448290, 0.635052577645500,
           0.635526392794951, 0.635999968718291,
           0.636473305237254, 0.636946402173667,
           0.637419259349444, 0.637891876586594,
           0.638364253707211, 0.638836390533483,
           0.639308286887687, 0.639779942592192,
           0.640251357469454, 0.640722531342024,
           0.641193464032541, 0.641664155363736,
           0.642134605158429, 0.642604813239536,
           0.643074779430056, 0.643544503553086,
           0.644013985431810, 0.644483224889506,
           0.644952221749543, 0.645420975835376,
           0.645889486970561, 0.646357754978736,
           0.646825779683636, 0.647293560909087,
           0.647761098479004, 0.648228392217398,
           0.648695441948367, 0.649162247496106,
           0.649628808684896, 0.650095125339115,
           0.650561197283231, 0.651027024341803,
           0.651492606339485, 0.651957943101021,
           0.652423034451249, 0.652887880215096,
           0.653352480217586, 0.653816834283833,
           0.654280942239043, 0.654744803908518,
           0.655208419117646, 0.655671787691916,
           0.656134909456903, 0.656597784238280,
           0.657060411861810, 0.657522792153349,
           0.657984924938848, 0.658446810044350,
           0.658908447295990, 0.659369836519999,
           0.659830977542700, 0.660291870190507,
           0.660752514289934, 0.661212909667581,
           0.661673056150146, 0.662132953564419,
           0.662592601737287, 0.663052000495725,
           0.663511149666809, 0.663970049077702,
           0.664428698555665, 0.664887097928053,
           0.665345247022314, 0.665803145665989,
           0.666260793686718, 0.666718190912231,
           0.667175337170352, 0.667632232289002,
           0.668088876096197, 0.668545268420045,
           0.669001409088749, 0.669457297930610,
           0.669912934774020, 0.670368319447466,
           0.670823451779534, 0.671278331598900,
           0.671732958734338, 0.672187333014715,
           0.672641454268999, 0.673095322326244,
           0.673548937015605, 0.674002298166332,
           0.674455405607772, 0.674908259169362,
           0.675360858680639, 0.675813203971235,
           0.676265294870878, 0.676717131209390,
           0.677168712816690, 0.677620039522794,
           0.678071111157810, 0.678521927551948,
           0.678972488535508, 0.679422793938890,
           0.679872843592590, 0.680322637327198,
           0.680772174973403, 0.681221456361988,
           0.681670481323834, 0.682119249689918,
           0.682567761291314, 0.683016015959192,
           0.683464013524821, 0.683911753819562,
           0.684359236674877, 0.684806461922325,
           0.685253429393557, 0.685700138920328,
           0.686146590334486, 0.686592783467975,
           0.687038718152840, 0.687484394221221,
           0.687929811505356, 0.688374969837579,
           0.688819869050324, 0.689264508976120,
           0.689708889447594, 0.690153010297474,
           0.690596871358582, 0.691040472463838,
           0.691483813446262, 0.691926894138970,
           0.692369714375178, 0.692812273988195,
           0.693254572811437, 0.693696610678409,
           0.694138387422719, 0.694579902878075,
           0.695021156878278, 0.695462149257232,
           0.695902879848936, 0.696343348487491,
           0.696783555007095, 0.697223499242043,
           0.697663181026731, 0.698102600195654,
           0.698541756583404, 0.698980650024674,
           0.699419280354253, 0.699857647407032,
           0.700295751018000, 0.700733591022246,
           0.701171167254955, 0.701608479551416,
           0.702045527747014, 0.702482311677235,
           0.702918831177663, 0.703355086083983,
           0.703791076231980, 0.704226801457536,
           0.704662261596635, 0.705097456485361,
           0.705532385959896, 0.705967049856524,
           0.706401448011628, 0.706835580261689,
           0.707269446443293, 0.707703046393121,
           0.708136379947957, 0.708569446944685,
           0.709002247220290, 0.709434780611853,
           0.709867046956564, 0.710299046091705,
           0.710730777854662, 0.711162242082923,
           0.711593438614075, 0.712024367285806,
           0.712455027935905, 0.712885420402261,
           0.713315544522867, 0.713745400135812,
           0.714174987079292, 0.714604305191598,
           0.715033354311127, 0.715462134276375,
           0.715890644925940, 0.716318886098521,
           0.716746857632919, 0.717174559368035,
           0.717601991142874, 0.718029152796540,
           0.718456044168241, 0.718882665097285,
           0.719309015423084, 0.719735094985149,
           0.720160903623093, 0.720586441176634,
           0.721011707485592, 0.721436702389883,
           0.721861425729533, 0.722285877344666,
           0.722710057075510, 0.723133964762391,
           0.723557600245744, 0.723980963366103,
           0.724404053964104, 0.724826871880486,
           0.725249416956093, 0.725671689031867,
           0.726093687948858, 0.726515413548214,
           0.726936865671190, 0.727358044159142,
           0.727778948853528, 0.728199579595912,
           0.728619936227958, 0.729040018591434,
           0.729459826528213, 0.729879359880270,
           0.730298618489683, 0.730717602198634,
           0.731136310849407, 0.731554744284393,
           0.731972902346083, 0.732390784877074,
           0.732808391720066, 0.733225722717862,
           0.733642777713368, 0.734059556549598,
           0.734476059069666, 0.734892285116792,
           0.735308234534298, 0.735723907165612,
           0.736139302854266, 0.736554421443896,
           0.736969262778242, 0.737383826701148,
           0.737798113056565, 0.738212121688544,
           0.738625852441245, 0.739039305158930,
           0.739452479685966, 0.739865375866825,
           0.740277993546084, 0.740690332568425,
           0.741102392778633, 0.741514174021603,
           0.741925676142327, 0.742336898985909,
           0.742747842397556, 0.743158506222579,
           0.743568890306395, 0.743978994494527,
           0.744388818632602, 0.744798362566354,
           0.745207626141621, 0.745616609204348,
           0.746025311600585, 0.746433733176486,
           0.746841873778311, 0.747249733252431,
           0.747657311445316, 0.748064608203545,
           0.748471623373802, 0.748878356802879,
           0.749284808337672, 0.749690977825182,
           0.750096865112521, 0.750502470046902,
           0.750907792475646, 0.751312832246182,
           0.751717589206043, 0.752122063202872,
           0.752526254084412, 0.752930161698521,
           0.753333785893156, 0.753737126516386,
           0.754140183416384, 0.754542956441430,
           0.754945445439911, 0.755347650260323,
           0.755749570751267, 0.756151206761450,
           0.756552558139689, 0.756953624734904,
           0.757354406396127, 0.757754902972495,
           0.758155114313250, 0.758555040267748,
           0.758954680685442, 0.759354035415904,
           0.759753104308805, 0.760151887213927,
           0.760550383981161, 0.760948594460503,
           0.761346518502057, 0.761744155956037,
           0.762141506672764, 0.762538570502664,
           0.762935347296275, 0.763331836904241,
           0.763728039177316, 0.764123953966360,
           0.764519581122340, 0.764914920496336,
           0.765309971939532, 0.765704735303223,
           0.766099210438810, 0.766493397197805,
           0.766887295431828, 0.767280904992605,
           0.767674225731974, 0.768067257501881,
           0.768460000154379, 0.768852453541631,
           0.769244617515911, 0.769636491929598,
           0.770028076635180, 0.770419371485260,
           0.770810376332544, 0.771201091029848,
           0.771591515430100, 0.771981649386337,
           0.772371492751700, 0.772761045379447,
           0.773150307122940, 0.773539277835653,
           0.773927957371168, 0.774316345583180,
           0.774704442325488, 0.775092247452004,
           0.775479760816753, 0.775866982273862,
           0.776253911677576, 0.776640548882245,
           0.777026893742329, 0.777412946112402,
           0.777798705847142, 0.778184172801342,
           0.778569346829906, 0.778954227787842,
           0.779338815530276, 0.779723109912439,
           0.780107110789674, 0.780490818017435,
           0.780874231451286, 0.781257350946901,
           0.781640176360067, 0.782022707546680,
           0.782404944362745, 0.782786886664381,
           0.783168534307817, 0.783549887149390,
           0.783930945045554, 0.784311707852868,
           0.784692175428005, 0.785072347627750,
           0.785452224308994, 0.785831805328748,
           0.786211090544126, 0.786590079812357,
           0.786968772990782, 0.787347169936853,
           0.787725270508132, 0.788103074562295,
           0.788480581957126, 0.788857792550524,
           0.789234706200500, 0.789611322765174,
           0.789987642102780, 0.790363664071661,
           0.790739388530277, 0.791114815337195,
           0.791489944351097, 0.791864775430776,
           0.792239308435137, 0.792613543223198,
           0.792987479654090, 0.793361117587052,
           0.793734456881441, 0.794107497396723,
           0.794480238992478, 0.794852681528396,
           0.795224824864284, 0.795596668860057,
           0.795968213375746, 0.796339458271493,
           0.796710403407554, 0.797081048644296,
           0.797451393842200, 0.797821438861859,
           0.798191183563984, 0.798560627809390,
           0.798929771459014, 0.799298614373898,
           0.799667156415206, 0.800035397444208,
           0.800403337322290, 0.800770975910952,
           0.801138313071807, 0.801505348666581,
           0.801872082557113, 0.802238514605357,
           0.802604644673380, 0.802970472623361,
           0.803335998317598, 0.803701221618495,
           0.804066142388576, 0.804430760490477,
           0.804795075786947, 0.805159088140848,
           0.805522797415163, 0.805886203472978,
           0.806249306177502, 0.806612105392055,
           0.806974600980071, 0.807336792805099,
           0.807698680730800, 0.808060264620955,
           0.808421544339453, 0.808782519750302,
           0.809143190717621, 0.809503557105649,
           0.809863618778732, 0.810223375601338,
           0.810582827438045, 0.810941974153548,
           0.811300815612655, 0.811659351680293,
           0.812017582221499, 0.812375507101428,
           0.812733126185348, 0.813090439338644,
           0.813447446426816, 0.813804147315479,
           0.814160541870361, 0.814516629957309,
           0.814872411442283, 0.815227886191359,
           0.815583054070730, 0.815937914946700,
           0.816292468685694, 0.816646715154248,
           0.817000654219018, 0.817354285746773,
           0.817707609604398, 0.818060625658894,
           0.818413333777379, 0.818765733827084,
           0.819117825675359, 0.819469609189669,
           0.819821084237595, 0.820172250686834,
           0.820523108405198, 0.820873657260618,
           0.821223897121140, 0.821573827854924,
           0.821923449330250, 0.822272761415514,
           0.822621763979224, 0.822970456890009,
           0.823318840016614, 0.823666913227900,
           0.824014676392844, 0.824362129380541,
           0.824709272060201, 0.825056104301153,
           0.825402625972841, 0.825748836944826,
           0.826094737086788, 0.826440326268522,
           0.826785604359940, 0.827130571231073,
           0.827475226752066, 0.827819570793186,
           0.828163603224812, 0.828507323917442,
           0.828850732741694, 0.829193829568301,
           0.829536614268112, 0.829879086712098,
           0.830221246771343, 0.830563094317051,
           0.830904629220543, 0.831245851353258,
           0.831586760586752, 0.831927356792699,
           0.832267639842892, 0.832607609609240,
           0.832947265963772, 0.833286608778634,
           0.833625637926088, 0.833964353278518,
           0.834302754708423, 0.834640842088422,
           0.834978615291250, 0.835316074189763,
           0.835653218656935, 0.835990048565854,
           0.836326563789735, 0.836662764201901,
           0.836998649675802, 0.837334220085002,
           0.837669475303185, 0.838004415204156,
           0.838339039661832, 0.838673348550256,
           0.839007341743585, 0.839341019116098,
           0.839674380542192, 0.840007425896380,
           0.840340155053299, 0.840672567887700,
           0.841004664274458, 0.841336444088563,
           0.841667907205126, 0.841999053499378,
           0.842329882846666, 0.842660395122461,
           0.842990590202350, 0.843320467962040,
           0.843650028277358, 0.843979271024250,
           0.844308196078782, 0.844636803317139,
           0.844965092615627, 0.845293063850669,
           0.845620716898812, 0.845948051636718,
           0.846275067941171, 0.846601765689076,
           0.846928144757457, 0.847254205023456,
           0.847579946364338, 0.847905368657488,
           0.848230471780407, 0.848555255610721,
           0.848879720026174, 0.849203864904630,
           0.849527690124074, 0.849851195562610,
           0.850174381098466, 0.850497246609985,
           0.850819791975636, 0.851142017074003,
           0.851463921783796, 0.851785505983842,
           0.852106769553088, 0.852427712370606,
           0.852748334315585, 0.853068635267335,
           0.853388615105288, 0.853708273708998,
           0.854027610958137, 0.854346626732500,
           0.854665320912002, 0.854983693376680,
           0.855301744006691, 0.855619472682315,
           0.855936879283950, 0.856253963692119,
           0.856570725787463, 0.856887165450747,
           0.857203282562856, 0.857519077004796,
           0.857834548657695, 0.858149697402802,
           0.858464523121490, 0.858779025695250,
           0.859093205005697, 0.859407060934567,
           0.859720593363717, 0.860033802175128,
           0.860346687250899, 0.860659248473256,
           0.860971485724541, 0.861283398887224,
           0.861594987843892, 0.861906252477258,
           0.862217192670152, 0.862527808305532,
           0.862838099266475, 0.863148065436180,
           0.863457706697968, 0.863767022935286,
           0.864076014031699, 0.864384679870895,
           0.864693020336687, 0.865001035313008,
           0.865308724683915, 0.865616088333585,
           0.865923126146322, 0.866229838006550,
           0.866536223798814, 0.866842283407786,
           0.867148016718256, 0.867453423615142,
           0.867758503983479, 0.868063257708432,
           0.868367684675281, 0.868671784769434,
           0.868975557876424, 0.869279003881900,
           0.869582122671640, 0.869884914131545,
           0.870187378147636, 0.870489514606058,
           0.870791323393082, 0.871092804395099,
           0.871393957498626, 0.871694782590303,
           0.871995279556890, 0.872295448285277,
           0.872595288662472, 0.872894800575608,
           0.873193983911942, 0.873492838558857,
           0.873791364403855, 0.874089561334565,
           0.874387429238740, 0.874684968004256,
           0.874982177519112, 0.875279057671432,
           0.875575608349464, 0.875871829441580,
           0.876167720836274, 0.876463282422170,
           0.876758514088007, 0.877053415722658,
           0.877347987215112, 0.877642228454489,
           0.877936139330028, 0.878229719731094,
           0.878522969547178, 0.878815888667894,
           0.879108476982982, 0.879400734382304,
           0.879692660755847, 0.879984255993725,
           0.880275519986175, 0.880566452623559,
           0.880857053796363, 0.881147323395198,
           0.881437261310801, 0.881726867434033,
           0.882016141655878, 0.882305083867450,
           0.882593693959982, 0.882881971824837,
           0.883169917353500, 0.883457530437579,
           0.883744810968815, 0.884031758839067,
           0.884318373940322, 0.884604656164691,
           0.884890605404412, 0.885176221551846,
           0.885461504499482, 0.885746454139933,
           0.886031070365938, 0.886315353070361,
           0.886599302146191, 0.886882917486544,
           0.887166198984661, 0.887449146533908,
           0.887731760027779, 0.888014039359889,
           0.888295984423986, 0.888577595113936,
           0.888858871323736, 0.889139812947507,
           0.889420419879497, 0.889700692014080,
           0.889980629245753, 0.890260231469144,
           0.890539498579004, 0.890818430470209,
           0.891097027037766, 0.891375288176802,
           0.891653213782575, 0.891930803750467,
           0.892208057975988, 0.892484976354772,
           0.892761558782583, 0.893037805155308,
           0.893313715368960, 0.893589289319683,
           0.893864526903745, 0.894139428017538,
           0.894413992557586, 0.894688220420535,
           0.894962111503160, 0.895235665702362,
           0.895508882915171, 0.895781763038740,
           0.896054305970351, 0.896326511607414,
           0.896598379847466, 0.896869910588166,
           0.897141103727306, 0.897411959162804,
           0.897682476792703, 0.897952656515174,
           0.898222498228515, 0.898492001831153,
           0.898761167221639, 0.899029994298656,
           0.899298482961008, 0.899566633107634,
           0.899834444637593, 0.900101917450077,
           0.900369051444402, 0.900635846520013,
           0.900902302576484, 0.901168419513514,
           0.901434197230930, 0.901699635628687,
           0.901964734606871, 0.902229494065691,
           0.902493913905485, 0.902757994026721,
           0.903021734329993, 0.903285134716023,
           0.903548195085662, 0.903810915339888,
           0.904073295379806, 0.904335335106652,
           0.904597034421788, 0.904858393226705,
           0.905119411423021, 0.905380088912483,
           0.905640425596968, 0.905900421378478,
           0.906160076159144, 0.906419389841229,
           0.906678362327118, 0.906936993519331,
           0.907195283320513, 0.907453231633437,
           0.907710838361006, 0.907968103406252,
           0.908225026672334, 0.908481608062541,
           0.908737847480288, 0.908993744829124,
           0.909249300012722, 0.909504512934886,
           0.909759383499546, 0.910013911610766,
           0.910268097172735, 0.910521940089771,
           0.910775440266324, 0.911028597606970,
           0.911281412016414, 0.911533883399492,
           0.911786011661168, 0.912037796706535,
           0.912289238440817, 0.912540336769364,
           0.912791091597659, 0.913041502831311,
           0.913291570376059, 0.913541294137774,
           0.913790674022454, 0.914039709936226,
           0.914288401785347, 0.914536749476206,
           0.914784752915318, 0.915032412009329,
           0.915279726665015, 0.915526696789282,
           0.915773322289163, 0.916019603071825,
           0.916265539044562, 0.916511130114797,
           0.916756376190085, 0.917001277178110,
           0.917245832986687, 0.917490043523757,
           0.917733908697396, 0.917977428415807,
           0.918220602587325, 0.918463431120412,
           0.918705913923663, 0.918948050905803,
           0.919189841975685, 0.919431287042292,
           0.919672386014744, 0.919913138802280,
           0.920153545314279, 0.920393605460246,
           0.920633319149818, 0.920872686292758,
           0.921111706798967, 0.921350380578470,
           0.921588707541426, 0.921826687598123,
           0.922064320658980, 0.922301606634547,
           0.922538545435505, 0.922775136972664,
           0.923011381156965, 0.923247277899482,
           0.923482827111418, 0.923718028704106,
           0.923952882589014, 0.924187388677734,
           0.924421546881995, 0.924655357113654,
           0.924888819284700, 0.925121933307252,
           0.925354699093562, 0.925587116556010,
           0.925819185607111, 0.926050906159508,
           0.926282278125976, 0.926513301419422,
           0.926743975952884, 0.926974301639531,
           0.927204278392663, 0.927433906125712,
           0.927663184752239, 0.927892114185943,
           0.928120694340646, 0.928348925130306,
           0.928576806469014, 0.928804338270988,
           0.929031520450581, 0.929258352922278,
           0.929484835600691, 0.929710968400570,
           0.929936751236792, 0.930162184024369,
           0.930387266678441, 0.930611999114284,
           0.930836381247302, 0.931060412993033,
           0.931284094267148, 0.931507424985446,
           0.931730405063862, 0.931953034418462,
           0.932175312965442, 0.932397240621132,
           0.932618817301995, 0.932840042924621,
           0.933060917405740, 0.933281440662206,
           0.933501612611013, 0.933721433169282,
           0.933940902254265, 0.934160019783354,
           0.934378785674064, 0.934597199844049,
           0.934815262211092, 0.935032972693111,
           0.935250331208154, 0.935467337674402,
           0.935683992010170, 0.935900294133903,
           0.936116243964183, 0.936331841419718,
           0.936547086419356, 0.936761978882072,
           0.936976518726975, 0.937190705873310,
           0.937404540240450, 0.937618021747904,
           0.937831150315312, 0.938043925862450,
           0.938256348309222, 0.938468417575669,
           0.938680133581962, 0.938891496248409,
           0.939102505495447, 0.939313161243646,
           0.939523463413713, 0.939733411926484,
           0.939943006702931, 0.940152247664156,
           0.940361134731399, 0.940569667826028,
           0.940777846869547, 0.940985671783593,
           0.941193142489937, 0.941400258910481,
           0.941607020967263, 0.941813428582452,
           0.942019481678353, 0.942225180177402,
           0.942430524002169, 0.942635513075361,
           0.942840147319812, 0.943044426658496,
           0.943248351014515, 0.943451920311111,
           0.943655134471654, 0.943857993419648,
           0.944060497078735, 0.944262645372688,
           0.944464438225412, 0.944665875560950,
           0.944866957303475, 0.945067683377296,
           0.945268053706855, 0.945468068216729,
           0.945667726831628, 0.945867029476395,
           0.946065976076009, 0.946264566555582,
           0.946462800840360, 0.946660678855723,
           0.946858200527186, 0.947055365780397,
           0.947252174541140, 0.947448626735330,
           0.947644722289019, 0.947840461128391,
           0.948035843179769, 0.948230868369602,
           0.948425536624482, 0.948619847871131,
           0.948813802036405, 0.949007399047295,
           0.949200638830929, 0.949393521314565,
           0.949586046425599, 0.949778214091560,
           0.949970024240112, 0.950161476799053,
           0.950352571696317, 0.950543308859970,
           0.950733688218217, 0.950923709699392,
           0.951113373231969, 0.951302678744553,
           0.951491626165886, 0.951680215424844,
           0.951868446450438, 0.952056319171813,
           0.952243833518250, 0.952430989419166,
           0.952617786804108, 0.952804225602765,
           0.952990305744954, 0.953176027160633,
           0.953361389779890, 0.953546393532953,
           0.953731038350181, 0.953915324162070,
           0.954099250899250, 0.954282818492489,
           0.954466026872685, 0.954648875970877,
           0.954831365718235, 0.955013496046067,
           0.955195266885815, 0.955376678169056,
           0.955557729827503, 0.955738421793003,
           0.955918753997541, 0.956098726373236,
           0.956278338852342, 0.956457591367249,
           0.956636483850482, 0.956815016234703,
           0.956993188452707, 0.957171000437428,
           0.957348452121931, 0.957525543439422,
           0.957702274323238, 0.957878644706855,
           0.958054654523882, 0.958230303708066,
           0.958405592193288, 0.958580519913567,
           0.958755086803055, 0.958929292796042,
           0.959103137826952, 0.959276621830346,
           0.959449744740922, 0.959622506493511,
           0.959794907023084, 0.959966946264744,
           0.960138624153733, 0.960309940625425,
           0.960480895615335, 0.960651489059111,
           0.960821720892538, 0.960991591051538,
           0.961161099472166, 0.961330246090617,
           0.961499030843220, 0.961667453666441,
           0.961835514496881, 0.962003213271279,
           0.962170549926509, 0.962337524399583,
           0.962504136627647, 0.962670386547985,
           0.962836274098016, 0.963001799215297,
           0.963166961837521, 0.963331761902516,
           0.963496199348250, 0.963660274112822,
           0.963823986134472, 0.963987335351576,
           0.964150321702645, 0.964312945126327,
           0.964475205561409, 0.964637102946810,
           0.964798637221590, 0.964959808324942,
           0.965120616196200, 0.965281060774831,
           0.965441142000440, 0.965600859812770,
           0.965760214151698, 0.965919204957242,
           0.966077832169552, 0.966236095728919,
           0.966393995575767, 0.966551531650661,
           0.966708703894300, 0.966865512247522,
           0.967021956651299, 0.967178037046744,
           0.967333753375103, 0.967489105577762,
           0.967644093596243, 0.967798717372205,
           0.967952976847444, 0.968106871963893,
           0.968260402663623, 0.968413568888842,
           0.968566370581894, 0.968718807685262,
           0.968870880141565, 0.969022587893559,
           0.969173930884139, 0.969324909056335,
           0.969475522353316, 0.969625770718388,
           0.969775654094994, 0.969925172426714,
           0.970074325657268, 0.970223113730509,
           0.970371536590432, 0.970519594181166,
           0.970667286446979, 0.970814613332276,
           0.970961574781602, 0.971108170739635,
           0.971254401151194, 0.971400265961236,
           0.971545765114851, 0.971690898557274,
           0.971835666233870, 0.971980068090147,
           0.972124104071750, 0.972267774124458,
           0.972411078194192, 0.972554016227010,
           0.972696588169105, 0.972838793966812,
           0.972980633566600, 0.973122106915077,
           0.973263213958990, 0.973403954645225,
           0.973544328920802, 0.973684336732881,
           0.973823978028761, 0.973963252755878,
           0.974102160861804, 0.974240702294254,
           0.974378877001075, 0.974516684930257,
           0.974654126029925, 0.974791200248343,
           0.974927907533915, 0.975064247835180,
           0.975200221100816, 0.975335827279640,
           0.975471066320608, 0.975605938172811,
           0.975740442785483, 0.975874580107991,
           0.976008350089845, 0.976141752680689,
           0.976274787830308, 0.976407455488625,
           0.976539755605701, 0.976671688131734,
           0.976803253017064, 0.976934450212165,
           0.977065279667653, 0.977195741334280,
           0.977325835162938, 0.977455561104656,
           0.977584919110602, 0.977713909132084,
           0.977842531120548, 0.977970785027576,
           0.978098670804891, 0.978226188404354,
           0.978353337777964, 0.978480118877862,
           0.978606531656321, 0.978732576065758,
           0.978858252058728, 0.978983559587923,
           0.979108498606174, 0.979233069066452,
           0.979357270921865, 0.979481104125662,
           0.979604568631230, 0.979727664392091,
           0.979850391361912, 0.979972749494495,
           0.980094738743781, 0.980216359063852,
           0.980337610408927, 0.980458492733362,
           0.980579005991658, 0.980699150138450,
           0.980818925128511, 0.980938330916757,
           0.981057367458241, 0.981176034708154,
           0.981294332621828, 0.981412261154733,
           0.981529820262477, 0.981647009900810,
           0.981763830025617, 0.981880280592927,
           0.981996361558903, 0.982112072879850,
           0.982227414512214, 0.982342386412575,
           0.982456988537657, 0.982571220844320,
           0.982685083289565, 0.982798575830531,
           0.982911698424498, 0.983024451028884,
           0.983136833601246, 0.983248846099280,
           0.983360488480824, 0.983471760703851,
           0.983582662726478, 0.983693194506957,
           0.983803356003683, 0.983913147175188,
           0.984022567980144, 0.984131618377363,
           0.984240298325797, 0.984348607784534,
           0.984456546712806, 0.984564115069982,
           0.984671312815570, 0.984778139909220,
           0.984884596310718, 0.984990681979994,
           0.985096396877112, 0.985201740962280,
           0.985306714195846, 0.985411316538292,
           0.985515547950247, 0.985619408392474,
           0.985722897825877, 0.985826016211503,
           0.985928763510534, 0.986031139684293,
           0.986133144694245, 0.986234778501992,
           0.986336041069278, 0.986436932357984,
           0.986537452330134, 0.986637600947888,
           0.986737378173550, 0.986836783969560,
           0.986935818298501, 0.987034481123092,
           0.987132772406196, 0.987230692110815,
           0.987328240200087, 0.987425416637295,
           0.987522221385859, 0.987618654409338,
           0.987714715671435, 0.987810405135989,
           0.987905722766981, 0.988000668528530,
           0.988095242384899, 0.988189444300486,
           0.988283274239832, 0.988376732167617,
           0.988469818048663, 0.988562531847928,
           0.988654873530514, 0.988746843061661,
           0.988838440406750, 0.988929665531302,
           0.989020518400977, 0.989110998981577,
           0.989201107239042, 0.989290843139453,
           0.989380206649034, 0.989469197734144,
           0.989557816361286, 0.989646062497102,
           0.989733936108373, 0.989821437162023,
           0.989908565625114, 0.989995321464849,
           0.990081704648571, 0.990167715143764,
           0.990253352918050, 0.990338617939196,
           0.990423510175104, 0.990508029593820,
           0.990592176163528, 0.990675949852554,
           0.990759350629363, 0.990842378462564,
           0.990925033320899, 0.991007315173258,
           0.991089223988668, 0.991170759736295,
           0.991251922385450, 0.991332711905578,
           0.991413128266271, 0.991493171437257,
           0.991572841388408, 0.991652138089731,
           0.991731061511380, 0.991809611623645,
           0.991887788396958, 0.991965591801893,
           0.992043021809161, 0.992120078389618,
           0.992196761514256, 0.992273071154210,
           0.992349007280758, 0.992424569865313,
           0.992499758879432, 0.992574574294814,
           0.992649016083295, 0.992723084216854,
           0.992796778667610, 0.992870099407824,
           0.992943046409895, 0.993015619646364,
           0.993087819089915, 0.993159644713368,
           0.993231096489687, 0.993302174391977,
           0.993372878393482, 0.993443208467587,
           0.993513164587818, 0.993582746727844,
           0.993651954861471, 0.993720788962648,
           0.993789249005464, 0.993857334964149,
           0.993925046813074, 0.993992384526752,
           0.994059348079835, 0.994125937447115,
           0.994192152603528, 0.994257993524149,
           0.994323460184193, 0.994388552559018,
           0.994453270624121, 0.994517614355141,
           0.994581583727858, 0.994645178718192,
           0.994708399302205, 0.994771245456098,
           0.994833717156216, 0.994895814379043,
           0.994957537101203, 0.995018885299463,
           0.995079858950731, 0.995140458032054,
           0.995200682520622, 0.995260532393764,
           0.995320007628953, 0.995379108203799,
           0.995437834096057, 0.995496185283620,
           0.995554161744525, 0.995611763456947,
           0.995668990399204, 0.995725842549754,
           0.995782319887198, 0.995838422390274,
           0.995894150037867, 0.995949502808997,
           0.996004480682830, 0.996059083638671,
           0.996113311655965, 0.996167164714300,
           0.996220642793405, 0.996273745873149,
           0.996326473933544, 0.996378826954740,
           0.996430804917032, 0.996482407800853,
           0.996533635586780, 0.996584488255528,
           0.996634965787956, 0.996685068165064,
           0.996734795367991, 0.996784147378017,
           0.996833124176570, 0.996881725745208,
           0.996929952065640, 0.996977803119712,
           0.997025278889412, 0.997072379356868,
           0.997119104504351, 0.997165454314272,
           0.997211428769185, 0.997257027851784,
           0.997302251544903, 0.997347099831521,
           0.997391572694755, 0.997435670117864,
           0.997479392084249, 0.997522738577454,
           0.997565709581160, 0.997608305079192,
           0.997650525055517, 0.997692369494242,
           0.997733838379616, 0.997774931696029,
           0.997815649428013, 0.997855991560241,
           0.997895958077527, 0.997935548964827,
           0.997974764207237, 0.998013603789998,
           0.998052067698486, 0.998090155918226,
           0.998127868434880, 0.998165205234251,
           0.998202166302285, 0.998238751625070,
           0.998274961188833, 0.998310794979945,
           0.998346252984917, 0.998381335190402,
           0.998416041583194, 0.998450372150229,
           0.998484326878584, 0.998517905755478,
           0.998551108768271, 0.998583935904464,
           0.998616387151701, 0.998648462497767,
           0.998680161930588, 0.998711485438230,
           0.998742433008904, 0.998773004630960,
           0.998803200292890, 0.998833019983328,
           0.998862463691050, 0.998891531404971,
           0.998920223114150, 0.998948538807786,
           0.998976478475223, 0.999004042105941,
           0.999031229689566, 0.999058041215864,
           0.999084476674742, 0.999110536056249,
           0.999136219350576, 0.999161526548056,
           0.999186457639161, 0.999211012614508,
           0.999235191464853, 0.999258994181095,
           0.999282420754274, 0.999305471175572,
           0.999328145436313, 0.999350443527960,
           0.999372365442121, 0.999393911170544,
           0.999415080705118, 0.999435874037875,
           0.999456291160987, 0.999476332066770,
           0.999495996747679, 0.999515285196313,
           0.999534197405410, 0.999552733367852,
           0.999570893076662, 0.999588676525002,
           0.999606083706181, 0.999623114613644,
           0.999639769240983, 0.999656047581927,
           0.999671949630349, 0.999687475380262,
           0.999702624825824, 0.999717397961331,
           0.999731794781223, 0.999745815280079,
           0.999759459452623, 0.999772727293720,
           0.999785618798374, 0.999798133961733,
           0.999810272779086, 0.999822035245865,
           0.999833421357641, 0.999844431110131,
           0.999855064499188, 0.999865321520810,
           0.999875202171138, 0.999884706446453,
           0.999893834343178, 0.999902585857877,
           0.999910960987257, 0.999918959728167,
           0.999926582077598, 0.999933828032682,
           0.999940697590694, 0.999947190749052,
           0.999953307505317, 0.999959047857190,
           0.999964411802522, 0.999969399339304,
           0.999974010465674, 0.999978245179922,
           0.999982103480493, 0.999985585365995,
           0.999988690835219, 0.999991419887180,
           0.999993772521201, 0.999995748737146,
           0.999997348536106, 0.999998571922979,
           0.999999418920350, 0.999999889716025};

  } else {

    stop("N must be 5, 10, 20, 40, 80, 160, 320, 640, 1280, 2560 or 5120.");

  }

  // Duplicate nodes
  x_k = arma::join_vert(-x_k, x_k);

  // Transform nodes to (a, b)
  x_k *= 0.5 * (b - a);
  x_k += 0.5 * (a + b);

  // Nodes
  return x_k;

}


//' @rdname Gauss_Legen
//' @export
// [[Rcpp::export]]
arma::vec Gauss_Legen_weights(double a = -1, double b = 1, arma::uword N = 40) {

  // Weights (w_k) for the Gauss--Legendre quadrature in (-1, 1). The weights
  // up to N = 80 are retrieved from https://dlmf.nist.gov/3.5#v, and for
  // N = 160 onwards they were computed with:
  // statmod::gauss.quad(n = N, kind = "legendre")
  arma::vec w_k = arma::zeros(N);
  if (N == 5) {

    w_k = {0.568888888888889 / 2.0, 0.478628670499366, 0.236926885056189};

  } else if (N == 10) {

    w_k = {0.295524224714752870, 0.269266719309996355,
           0.219086362515982044, 0.149451349150580593,
           0.066671344308688138};

  } else if (N == 20) {

    w_k = {0.152753387130725850698, 0.149172986472603746788,
           0.142096109318382051329, 0.131688638449176626898,
           0.118194531961518417312, 0.101930119817240435037,
           0.083276741576704748725, 0.062672048334109063570,
           0.040601429800386941331, 0.017614007139152118312};

  } else if (N == 40) {

    w_k = {0.077505947978424811264, 0.077039818164247965588,
           0.076110361900626242372, 0.074723169057968264200,
           0.072886582395804059061, 0.070611647391286779696,
           0.067912045815233903826, 0.064804013456601038075,
           0.061306242492928939167, 0.057439769099391551367,
           0.053227846983936824355, 0.048695807635072232061,
           0.043870908185673271992, 0.038782167974472017640,
           0.033460195282547847393, 0.027937006980023401099,
           0.022245849194166957262, 0.016421058381907888713,
           0.010498284531152813615, 0.004521277098533191258};

  } else if (N == 80) {

    w_k = {0.039017813656306654811, 0.038958395962769531199,
           0.038839651059051968932, 0.038661759774076463327,
           0.038424993006959423185, 0.038129711314477638344,
           0.037776364362001397490, 0.037365490238730490027,
           0.036897714638276008839, 0.036373749905835978044,
           0.035794393953416054603, 0.035160529044747593496,
           0.034473120451753928794, 0.033733214984611522817,
           0.032941939397645401383, 0.032100498673487773148,
           0.031210174188114701642, 0.030272321759557980661,
           0.029288369583267847693, 0.028259816057276862397,
           0.027188227500486380674, 0.026075235767565117903,
           0.024922535764115491105, 0.023731882865930101293,
           0.022505090246332461926, 0.021244026115782006389,
           0.019950610878141998929, 0.018626814208299031429,
           0.017274652056269306359, 0.015896183583725688045,
           0.014493508040509076117, 0.013068761592401339294,
           0.011624114120797826917, 0.010161766041103064521,
           0.008683945269260858426, 0.007192904768117312753,
           0.005690922451403198649, 0.004180313124694895237,
           0.002663533589512681669, 0.001144950003186941535};

  } else if (N == 160) {

    w_k = {0.019572753617011, 0.019565254886698,
           0.019550260298992, 0.019527775598634,
           0.019497809399980, 0.019460373183707,
           0.019415481292412, 0.019363150925113,
           0.019303402130668, 0.019236257800085,
           0.019161743657758, 0.019079888251613,
           0.018990722942162, 0.018894281890496,
           0.018790602045195, 0.018679723128172,
           0.018561687619453, 0.018436540740906,
           0.018304330438912, 0.018165107365996,
           0.018018924861425, 0.017865838930763,
           0.017705908224430, 0.017539194015215,
           0.017365760174811, 0.017185673149345,
           0.016999001933912, 0.016805818046154,
           0.016606195498850, 0.016400210771563,
           0.016187942781346, 0.015969472852494,
           0.015744884685399, 0.015514264324478,
           0.015277700125204, 0.015035282720265,
           0.014787104984827, 0.014533262000968,
           0.014273851021238, 0.014008971431395,
           0.013738724712352, 0.013463214401269,
           0.013182546051910, 0.012896827194184,
           0.012606167292960, 0.012310677706128,
           0.012010471641932, 0.011705664115603,
           0.011396371905288, 0.011082713507318,
           0.010764809090810, 0.010442780451625,
           0.010116750965708, 0.009786845541829,
           0.009453190573723, 0.009115913891680,
           0.008775144713568, 0.008431013595341,
           0.008083652381036, 0.007733194152267,
           0.007379773177275, 0.007023524859515,
           0.006664585685838, 0.006303093174278,
           0.005939185821487, 0.005573003049860,
           0.005204685154448, 0.004834373249702,
           0.004462209216334, 0.004088335648567,
           0.003712895802587, 0.003336033547821,
           0.002957893325040, 0.002578620121838,
           0.002198359496796, 0.001817257759665,
           0.001435462753483, 0.001053127672213,
           0.000670438320153, 0.000288058528521};

  } else if (N == 320) {

    w_k = {0.009802031511030, 0.009801089716216,
           0.009799206217077, 0.009796381194583,
           0.009792614920167, 0.009787907755697,
           0.009782260153447, 0.009775672656046,
           0.009768145896433, 0.009759680597788,
           0.009750277573473, 0.009739937726945,
           0.009728662051675, 0.009716451631041,
           0.009703307638248, 0.009689231336187,
           0.009674224077333, 0.009658287303607,
           0.009641422546239, 0.009623631425623,
           0.009604915651157, 0.009585277021085,
           0.009564717422313, 0.009543238830246,
           0.009520843308578, 0.009497533009106,
           0.009473310171521, 0.009448177123193,
           0.009422136278944, 0.009395190140822,
           0.009367341297853, 0.009338592425798,
           0.009308946286895, 0.009278405729594,
           0.009246973688278, 0.009214653182989,
           0.009181447319131, 0.009147359287180,
           0.009112392362363, 0.009076549904364,
           0.009039835356978, 0.009002252247800,
           0.008963804187876, 0.008924494871354,
           0.008884328075139, 0.008843307658519,
           0.008801437562800, 0.008758721810929,
           0.008715164507104, 0.008670769836380,
           0.008625542064267, 0.008579485536321,
           0.008532604677728, 0.008484903992875,
           0.008436388064918, 0.008387061555351,
           0.008336929203534, 0.008285995826272,
           0.008234266317322, 0.008181745646937,
           0.008128438861384, 0.008074351082470,
           0.008019487507030, 0.007963853406447,
           0.007907454126134, 0.007850295085021,
           0.007792381775043, 0.007733719760602,
           0.007674314678036, 0.007614172235081,
           0.007553298210322, 0.007491698452627,
           0.007429378880603, 0.007366345482014,
           0.007302604313205, 0.007238161498535,
           0.007173023229770, 0.007107195765494,
           0.007040685430523, 0.006973498615270,
           0.006905641775153, 0.006837121429969,
           0.006767944163255, 0.006698116621680,
           0.006627645514383, 0.006556537612338,
           0.006484799747706, 0.006412438813179,
           0.006339461761309, 0.006265875603848,
           0.006191687411070, 0.006116904311096,
           0.006041533489205, 0.005965582187147,
           0.005889057702443, 0.005811967387690,
           0.005734318649843, 0.005656118949520,
           0.005577375800277, 0.005498096767875,
           0.005418289469574, 0.005337961573384,
           0.005257120797340, 0.005175774908751,
           0.005093931723461, 0.005011599105094,
           0.004928784964298, 0.004845497257989,
           0.004761743988582, 0.004677533203228,
           0.004592872993032, 0.004507771492282,
           0.004422236877668, 0.004336277367490,
           0.004249901220880, 0.004163116736993,
           0.004075932254223, 0.003988356149395,
           0.003900396836962, 0.003812062768197,
           0.003723362430378, 0.003634304345981,
           0.003544897071849, 0.003455149198376,
           0.003365069348686, 0.003274666177800,
           0.003183948371798, 0.003092924647004,
           0.003001603749124, 0.002909994452424,
           0.002818105558882, 0.002725945897339,
           0.002633524322658, 0.002540849714868,
           0.002447930978313, 0.002354777040801,
           0.002261396852743, 0.002167799386301,
           0.002073993634522, 0.001979988610484,
           0.001885793346437, 0.001791416892936,
           0.001696868317992, 0.001602156706222,
           0.001507291157995, 0.001412280788622,
           0.001317134727547, 0.001221862117600,
           0.001126472114349, 0.001030973885628,
           0.000935376611416, 0.000839689484513,
           0.000743921712961, 0.000648082526891,
           0.000552181197610, 0.000456227095865,
           0.000360229901103, 0.000264200571980,
           0.000168158195617, 0.000072241702290};

  } else if (N == 640) {

    w_k = {0.004904890318127, 0.004904772315981,
           0.004904536314527, 0.004904182319442,
           0.004903710339247, 0.004903120385291,
           0.004902412471771, 0.004901586615716,
           0.004900642836996, 0.004899581158315,
           0.004898401605217, 0.004897104206078,
           0.004895688992111, 0.004894155997365,
           0.004892505258718, 0.004890736815886,
           0.004888850711413, 0.004886846990675,
           0.004884725701880, 0.004882486896058,
           0.004880130627074, 0.004877656951614,
           0.004875065929189, 0.004872357622135,
           0.004869532095607, 0.004866589417584,
           0.004863529658859, 0.004860352893046,
           0.004857059196570, 0.004853648648671,
           0.004850121331404, 0.004846477329622,
           0.004842716731000, 0.004838839626007,
           0.004834846107917, 0.004830736272811,
           0.004826510219560, 0.004822168049837,
           0.004817709868102, 0.004813135781615,
           0.004808445900419, 0.004803640337341,
           0.004798719207995, 0.004793682630774,
           0.004788530726848, 0.004783263620160,
           0.004777881437430, 0.004772384308139,
           0.004766772364540, 0.004761045741645,
           0.004755204577226, 0.004749249011807,
           0.004743179188672, 0.004736995253846,
           0.004730697356104, 0.004724285646958,
           0.004717760280665, 0.004711121414213,
           0.004704369207315, 0.004697503822423,
           0.004690525424699, 0.004683434182033,
           0.004676230265027, 0.004668913846991,
           0.004661485103944, 0.004653944214609,
           0.004646291360403, 0.004638526725440,
           0.004630650496524, 0.004622662863138,
           0.004614564017451, 0.004606354154305,
           0.004598033471215, 0.004589602168358,
           0.004581060448575, 0.004572408517364,
           0.004563646582877, 0.004554774855904,
           0.004545793549883, 0.004536702880888,
           0.004527503067621, 0.004518194331415,
           0.004508776896218, 0.004499250988592,
           0.004489616837719, 0.004479874675372,
           0.004470024735930, 0.004460067256364,
           0.004450002476233, 0.004439830637674,
           0.004429551985402, 0.004419166766701,
           0.004408675231423, 0.004398077631969,
           0.004387374223299, 0.004376565262917,
           0.004365651010863, 0.004354631729715,
           0.004343507684576, 0.004332279143067,
           0.004320946375324, 0.004309509653995,
           0.004297969254224, 0.004286325453649,
           0.004274578532399, 0.004262728773081,
           0.004250776460778, 0.004238721883038,
           0.004226565329874, 0.004214307093745,
           0.004201947469565, 0.004189486754679,
           0.004176925248867, 0.004164263254337,
           0.004151501075712, 0.004138639020023,
           0.004125677396708, 0.004112616517599,
           0.004099456696914, 0.004086198251254,
           0.004072841499587, 0.004059386763256,
           0.004045834365956, 0.004032184633729,
           0.004018437894959, 0.004004594480370,
           0.003990654723007, 0.003976618958231,
           0.003962487523719, 0.003948260759441,
           0.003933939007671, 0.003919522612957,
           0.003905011922134, 0.003890407284298,
           0.003875709050810, 0.003860917575279,
           0.003846033213564, 0.003831056323748,
           0.003815987266150, 0.003800826403302,
           0.003785574099940, 0.003770230723010,
           0.003754796641644, 0.003739272227153,
           0.003723657853025, 0.003707953894915,
           0.003692160730626, 0.003676278740112,
           0.003660308305462, 0.003644249810895,
           0.003628103642747, 0.003611870189463,
           0.003595549841590, 0.003579142991762,
           0.003562650034697, 0.003546071367180,
           0.003529407388068, 0.003512658498260,
           0.003495825100704, 0.003478907600375,
           0.003461906404278, 0.003444821921430,
           0.003427654562847, 0.003410404741545,
           0.003393072872522, 0.003375659372746,
           0.003358164661154, 0.003340589158636,
           0.003322933288022, 0.003305197474077,
           0.003287382143495, 0.003269487724876,
           0.003251514648723, 0.003233463347435,
           0.003215334255291, 0.003197127808444,
           0.003178844444902, 0.003160484604531,
           0.003142048729029, 0.003123537261932,
           0.003104950648586, 0.003086289336152,
           0.003067553773581, 0.003048744411619,
           0.003029861702781, 0.003010906101345,
           0.002991878063353, 0.002972778046574,
           0.002953606510524, 0.002934363916430,
           0.002915050727236, 0.002895667407573,
           0.002876214423768, 0.002856692243827,
           0.002837101337410, 0.002817442175839,
           0.002797715232074, 0.002777920980708,
           0.002758059897953, 0.002738132461626,
           0.002718139151143, 0.002698080447507,
           0.002677956833287, 0.002657768792621,
           0.002637516811193, 0.002617201376228,
           0.002596822976474, 0.002576382102198,
           0.002555879245166, 0.002535314898637,
           0.002514689557351, 0.002494003717509,
           0.002473257876778, 0.002452452534260,
           0.002431588190490, 0.002410665347428,
           0.002389684508430, 0.002368646178262,
           0.002347550863061, 0.002326399070339,
           0.002305191308968, 0.002283928089168,
           0.002262609922489, 0.002241237321801,
           0.002219810801293, 0.002198330876443,
           0.002176798064016, 0.002155212882051,
           0.002133575849845, 0.002111887487941,
           0.002090148318125, 0.002068358863389,
           0.002046519647957, 0.002024631197230,
           0.002002694037803, 0.001980708697444,
           0.001958675705075, 0.001936595590769,
           0.001914468885730, 0.001892296122281,
           0.001870077833862, 0.001847814554994,
           0.001825506821295, 0.001803155169443,
           0.001780760137176, 0.001758322263275,
           0.001735842087550, 0.001713320150833,
           0.001690756994957, 0.001668153162746,
           0.001645509198007, 0.001622825645509,
           0.001600103050974, 0.001577341961065,
           0.001554542923368, 0.001531706486383,
           0.001508833199511, 0.001485923613043,
           0.001462978278133, 0.001439997746805,
           0.001416982571926, 0.001393933307196,
           0.001370850507133, 0.001347734727068,
           0.001324586523119, 0.001301406452189,
           0.001278195071948, 0.001254952940811,
           0.001231680617941, 0.001208378663229,
           0.001185047637270, 0.001161688101365,
           0.001138300617501, 0.001114885748331,
           0.001091444057177, 0.001067976107998,
           0.001044482465387, 0.001020963694556,
           0.000997420361325, 0.000973853032096,
           0.000950262273856, 0.000926648654155,
           0.000903012741088, 0.000879355103293,
           0.000855676309928, 0.000831976930654,
           0.000808257535641, 0.000784518695529,
           0.000760760981429, 0.000736984964912,
           0.000713191217980, 0.000689380313068,
           0.000665552823027, 0.000641709321102,
           0.000617850380928, 0.000593976576510,
           0.000570088482213, 0.000546186672752,
           0.000522271723171, 0.000498344208833,
           0.000474404705413, 0.000450453788881,
           0.000426492035491, 0.000402520021777,
           0.000378538324541, 0.000354547520854,
           0.000330548188064, 0.000306540903810,
           0.000282526246064, 0.000258504793230,
           0.000234477124303, 0.000210443819264,
           0.000186405459880, 0.000162362631625,
           0.000138315928680, 0.000114265968714,
           0.000090213445500, 0.000066159372118,
           0.000042106796160, 0.000018088772198};

  } else if (N == 1280) {

    w_k = {0.002453408863998, 0.002453394096384,
           0.002453364561249, 0.002453320258768,
           0.002453261189204, 0.002453187352919,
           0.002453098750355, 0.002452995382044,
           0.002452877248610, 0.002452744350762,
           0.002452596689302, 0.002452434265118,
           0.002452257079186, 0.002452065132578,
           0.002451858426444, 0.002451636962028,
           0.002451400740666, 0.002451149763781,
           0.002450884032877, 0.002450603549561,
           0.002450308315517, 0.002449998332524,
           0.002449673602447, 0.002449334127240,
           0.002448979908947, 0.002448610949702,
           0.002448227251721, 0.002447828817320,
           0.002447415648892, 0.002446987748927,
           0.002446545119998, 0.002446087764772,
           0.002445615685999, 0.002445128886525,
           0.002444627369275, 0.002444111137270,
           0.002443580193618, 0.002443034541517,
           0.002442474184246, 0.002441899125179,
           0.002441309367782, 0.002440704915601,
           0.002440085772275, 0.002439451941532,
           0.002438803427185, 0.002438140233139,
           0.002437462363386, 0.002436769822007,
           0.002436062613167, 0.002435340741126,
           0.002434604210229, 0.002433853024909,
           0.002433087189687, 0.002432306709173,
           0.002431511588066, 0.002430701831148,
           0.002429877443298, 0.002429038429477,
           0.002428184794733, 0.002427316544207,
           0.002426433683122, 0.002425536216796,
           0.002424624150627, 0.002423697490107,
           0.002422756240816, 0.002421800408415,
           0.002420829998660, 0.002419845017393,
           0.002418845470542, 0.002417831364122,
           0.002416802704237, 0.002415759497082,
           0.002414701748933, 0.002413629466160,
           0.002412542655213, 0.002411441322637,
           0.002410325475062, 0.002409195119201,
           0.002408050261859, 0.002406890909929,
           0.002405717070389, 0.002404528750302,
           0.002403325956825, 0.002402108697193,
           0.002400876978737, 0.002399630808870,
           0.002398370195091, 0.002397095144990,
           0.002395805666241, 0.002394501766606,
           0.002393183453933, 0.002391850736157,
           0.002390503621301, 0.002389142117472,
           0.002387766232867, 0.002386375975768,
           0.002384971354540, 0.002383552377642,
           0.002382119053613, 0.002380671391078,
           0.002379209398757, 0.002377733085445,
           0.002376242460028, 0.002374737531482,
           0.002373218308863, 0.002371684801317,
           0.002370137018071, 0.002368574968447,
           0.002366998661842, 0.002365408107748,
           0.002363803315738, 0.002362184295469,
           0.002360551056691, 0.002358903609229,
           0.002357241963004, 0.002355566128018,
           0.002353876114354, 0.002352171932188,
           0.002350453591778, 0.002348721103465,
           0.002346974477676, 0.002345213724930,
           0.002343438855821, 0.002341649881033,
           0.002339846811335, 0.002338029657579,
           0.002336198430702, 0.002334353141730,
           0.002332493801767, 0.002330620422008,
           0.002328733013725, 0.002326831588282,
           0.002324916157124, 0.002322986731778,
           0.002321043323859, 0.002319085945066,
           0.002317114607180, 0.002315129322064,
           0.002313130101674, 0.002311116958038,
           0.002309089903279, 0.002307048949593,
           0.002304994109268, 0.002302925394670,
           0.002300842818254, 0.002298746392553,
           0.002296636130189, 0.002294512043863,
           0.002292374146357, 0.002290222450545,
           0.002288056969372, 0.002285877715879,
           0.002283684703180, 0.002281477944474,
           0.002279257453048, 0.002277023242264,
           0.002274775325573, 0.002272513716502,
           0.002270238428668, 0.002267949475765,
           0.002265646871572, 0.002263330629944,
           0.002261000764828, 0.002258657290247,
           0.002256300220305, 0.002253929569192,
           0.002251545351177, 0.002249147580610,
           0.002246736271925, 0.002244311439635,
           0.002241873098337, 0.002239421262705,
           0.002236955947503, 0.002234477167563,
           0.002231984937810, 0.002229479273246,
           0.002226960188947, 0.002224427700082,
           0.002221881821893, 0.002219322569706,
           0.002216749958922, 0.002214164005029,
           0.002211564723589, 0.002208952130253,
           0.002206326240741, 0.002203687070863,
           0.002201034636503, 0.002198368953627,
           0.002195690038280, 0.002192997906588,
           0.002190292574752, 0.002187574059062,
           0.002184842375876, 0.002182097541640,
           0.002179339572873, 0.002176568486179,
           0.002173784298235, 0.002170987025801,
           0.002168176685715, 0.002165353294891,
           0.002162516870326, 0.002159667429088,
           0.002156804988336, 0.002153929565294,
           0.002151041177273, 0.002148139841657,
           0.002145225575911, 0.002142298397576,
           0.002139358324272, 0.002136405373693,
           0.002133439563619, 0.002130460911895,
           0.002127469436457, 0.002124465155306,
           0.002121448086528, 0.002118418248283,
           0.002115375658808, 0.002112320336419,
           0.002109252299503, 0.002106171566530,
           0.002103078156043, 0.002099972086660,
           0.002096853377082, 0.002093722046075,
           0.002090578112490, 0.002087421595254,
           0.002084252513361, 0.002081070885892,
           0.002077876731994, 0.002074670070896,
           0.002071450921896, 0.002068219304374,
           0.002064975237782, 0.002061718741642,
           0.002058449835562, 0.002055168539217,
           0.002051874872354, 0.002048568854801,
           0.002045250506459, 0.002041919847294,
           0.002038576897366, 0.002035221676790,
           0.002031854205763, 0.002028474504553,
           0.002025082593506, 0.002021678493037,
           0.002018262223636, 0.002014833805867,
           0.002011393260366, 0.002007940607842,
           0.002004475869078, 0.002000999064928,
           0.001997510216322, 0.001994009344255,
           0.001990496469808, 0.001986971614118,
           0.001983434798404, 0.001979886043959,
           0.001976325372137, 0.001972752804376,
           0.001969168362175, 0.001965572067114,
           0.001961963940840, 0.001958344005067,
           0.001954712281588, 0.001951068792261,
           0.001947413559019, 0.001943746603862,
           0.001940067948863, 0.001936377616164,
           0.001932675627977, 0.001928962006587,
           0.001925236774345, 0.001921499953680,
           0.001917751567077, 0.001913991637101,
           0.001910220186385, 0.001906437237630,
           0.001902642813607, 0.001898836937154,
           0.001895019631178, 0.001891190918660,
           0.001887350822645, 0.001883499366243,
           0.001879636572642, 0.001875762465093,
           0.001871877066911, 0.001867980401486,
           0.001864072492273, 0.001860153362791,
           0.001856223036636, 0.001852281537461,
           0.001848328888991, 0.001844365115022,
           0.001840390239408, 0.001836404286076,
           0.001832407279021, 0.001828399242297,
           0.001824380200036, 0.001820350176422,
           0.001816309195718, 0.001812257282246,
           0.001808194460393, 0.001804120754618,
           0.001800036189438, 0.001795940789442,
           0.001791834579279, 0.001787717583667,
           0.001783589827384, 0.001779451335279,
           0.001775302132261, 0.001771142243307,
           0.001766971693453, 0.001762790507803,
           0.001758598711527, 0.001754396329854,
           0.001750183388082, 0.001745959911567,
           0.001741725925729, 0.001737481456060,
           0.001733226528102, 0.001728961167469,
           0.001724685399837, 0.001720399250937,
           0.001716102746575, 0.001711795912607,
           0.001707478774961, 0.001703151359620,
           0.001698813692635, 0.001694465800111,
           0.001690107708224, 0.001685739443199,
           0.001681361031339, 0.001676972498991,
           0.001672573872573, 0.001668165178561,
           0.001663746443494, 0.001659317693967,
           0.001654878956640, 0.001650430258226,
           0.001645971625508, 0.001641503085321,
           0.001637024664563, 0.001632536390190,
           0.001628038289217, 0.001623530388721,
           0.001619012715834, 0.001614485297752,
           0.001609948161724, 0.001605401335058,
           0.001600844845128, 0.001596278719358,
           0.001591702985228, 0.001587117670290,
           0.001582522802134, 0.001577918408423,
           0.001573304516871, 0.001568681155249,
           0.001564048351389, 0.001559406133172,
           0.001554754528545, 0.001550093565505,
           0.001545423272108, 0.001540743676465,
           0.001536054806745, 0.001531356691168,
           0.001526649358016, 0.001521932835623,
           0.001517207152376, 0.001512472336725,
           0.001507728417165, 0.001502975422256,
           0.001498213380602, 0.001493442320869,
           0.001488662271777, 0.001483873262095,
           0.001479075320650, 0.001474268476322,
           0.001469452758045, 0.001464628194807,
           0.001459794815645, 0.001454952649654,
           0.001450101725982, 0.001445242073825,
           0.001440373722435, 0.001435496701113,
           0.001430611039223, 0.001425716766162,
           0.001420813911399, 0.001415902504440,
           0.001410982574850, 0.001406054152243,
           0.001401117266284, 0.001396171946687,
           0.001391218223224, 0.001386256125709,
           0.001381285684007, 0.001376306928043,
           0.001371319887783, 0.001366324593242,
           0.001361321074491, 0.001356309361646,
           0.001351289484875, 0.001346261474391,
           0.001341225360459, 0.001336181173398,
           0.001331128943563, 0.001326068701368,
           0.001321000477270, 0.001315924301778,
           0.001310840205445, 0.001305748218874,
           0.001300648372716, 0.001295540697663,
           0.001290425224467, 0.001285301983913,
           0.001280171006841, 0.001275032324136,
           0.001269885966729, 0.001264731965595,
           0.001259570351760, 0.001254401156289,
           0.001249224410302, 0.001244040144954,
           0.001238848391452, 0.001233649181047,
           0.001228442545033, 0.001223228514752,
           0.001218007121587, 0.001212778396965,
           0.001207542372363, 0.001202299079294,
           0.001197048549320, 0.001191790814045,
           0.001186525905116, 0.001181253854226,
           0.001175974693104, 0.001170688453531,
           0.001165395167324, 0.001160094866345,
           0.001154787582499, 0.001149473347727,
           0.001144152194020, 0.001138824153408,
           0.001133489257960, 0.001128147539789,
           0.001122799031049, 0.001117443763932,
           0.001112081770671, 0.001106713083543,
           0.001101337734864, 0.001095955756989,
           0.001090567182313, 0.001085172043271,
           0.001079770372338, 0.001074362202026,
           0.001068947564890, 0.001063526493521,
           0.001058099020551, 0.001052665178647,
           0.001047225000519, 0.001041778518908,
           0.001036325766604, 0.001030866776426,
           0.001025401581229, 0.001019930213914,
           0.001014452707410, 0.001008969094693,
           0.001003479408764, 0.000997983682674,
           0.000992481949496, 0.000986974242350,
           0.000981460594385, 0.000975941038792,
           0.000970415608793, 0.000964884337647,
           0.000959347258647, 0.000953804405125,
           0.000948255810442, 0.000942701507995,
           0.000937141531220, 0.000931575913581,
           0.000926004688580, 0.000920427889750,
           0.000914845550664, 0.000909257704918,
           0.000903664386146, 0.000898065628019,
           0.000892461464235, 0.000886851928528,
           0.000881237054663, 0.000875616876433,
           0.000869991427674, 0.000864360742242,
           0.000858724854032, 0.000853083796966,
           0.000847437604998, 0.000841786312118,
           0.000836129952335, 0.000830468559703,
           0.000824802168297, 0.000819130812221,
           0.000813454525618, 0.000807773342647,
           0.000802087297514, 0.000796396424435,
           0.000790700757674, 0.000785000331505,
           0.000779295180248, 0.000773585338239,
           0.000767870839848, 0.000762151719472,
           0.000756428011536, 0.000750699750491,
           0.000744966970817, 0.000739229707024,
           0.000733487993641, 0.000727741865231,
           0.000721991356383, 0.000716236501706,
           0.000710477335846, 0.000704713893462,
           0.000698946209251, 0.000693174317925,
           0.000687398254230, 0.000681618052933,
           0.000675833748825, 0.000670045376726,
           0.000664252971471, 0.000658456567934,
           0.000652656200998, 0.000646851905580,
           0.000641043716618, 0.000635231669069,
           0.000629415797922, 0.000623596138181,
           0.000617772724876, 0.000611945593059,
           0.000606114777807, 0.000600280314215,
           0.000594442237401, 0.000588600582509,
           0.000582755384700, 0.000576906679157,
           0.000571054501082, 0.000565198885707,
           0.000559339868271, 0.000553477484046,
           0.000547611768316, 0.000541742756391,
           0.000535870483596, 0.000529994985274,
           0.000524116296799, 0.000518234453550,
           0.000512349490933, 0.000506461444369,
           0.000500570349304, 0.000494676241193,
           0.000488779155516, 0.000482879127770,
           0.000476976193466, 0.000471070388137,
           0.000465161747332, 0.000459250306613,
           0.000453336101566, 0.000447419167787,
           0.000441499540894, 0.000435577256516,
           0.000429652350303, 0.000423724857918,
           0.000417794815038, 0.000411862257359,
           0.000405927220588, 0.000399989740453,
           0.000394049852689, 0.000388107593053,
           0.000382162997311, 0.000376216101245,
           0.000370266940651, 0.000364315551338,
           0.000358361969129, 0.000352406229860,
           0.000346448369379, 0.000340488423551,
           0.000334526428245, 0.000328562419351,
           0.000322596432769, 0.000316628504405,
           0.000310658670186, 0.000304686966043,
           0.000298713427922, 0.000292738091779,
           0.000286760993580, 0.000280782169304,
           0.000274801654938, 0.000268819486481,
           0.000262835699941, 0.000256850331334,
           0.000250863416689, 0.000244874992043,
           0.000238885093440, 0.000232893756935,
           0.000226901018593, 0.000220906914485,
           0.000214911480691, 0.000208914753299,
           0.000202916768402, 0.000196917562109,
           0.000190917170529, 0.000184915629778,
           0.000178912975985, 0.000172909245276,
           0.000166904473795, 0.000160898697687,
           0.000154891953101, 0.000148884276196,
           0.000142875703134, 0.000136866270085,
           0.000130856013229, 0.000124844968741,
           0.000118833172811, 0.000112820661635,
           0.000106807471412, 0.000100793638355,
           0.000094779198678, 0.000088764188619,
           0.000082748644422, 0.000076732602365,
           0.000070716098760, 0.000064699169984,
           0.000058681852529, 0.000052664183093,
           0.000046646198793, 0.000040627937642,
           0.000034609439807, 0.000028590751313,
           0.000022571937201, 0.000016553142390,
           0.000010535020809, 0.000004525733985};

  } else if (N == 2560) {

    w_k = {0.001226944738342, 0.001226942891308,
           0.001226939197240, 0.001226933656147,
           0.001226926268034, 0.001226917032914,
           0.001226905950800, 0.001226893021712,
           0.001226878245665, 0.001226861622683,
           0.001226843152792, 0.001226822836019,
           0.001226800672393, 0.001226776661950,
           0.001226750804723, 0.001226723100754,
           0.001226693550085, 0.001226662152759,
           0.001226628908823, 0.001226593818327,
           0.001226556881324, 0.001226518097870,
           0.001226477468023, 0.001226434991846,
           0.001226390669398, 0.001226344500753,
           0.001226296485975, 0.001226246625137,
           0.001226194918316, 0.001226141365587,
           0.001226085967035, 0.001226028722738,
           0.001225969632787, 0.001225908697268,
           0.001225845916274, 0.001225781289898,
           0.001225714818240, 0.001225646501397,
           0.001225576339474, 0.001225504332576,
           0.001225430480812, 0.001225354784291,
           0.001225277243131, 0.001225197857443,
           0.001225116627351, 0.001225033552977,
           0.001224948634446, 0.001224861871882,
           0.001224773265420, 0.001224682815191,
           0.001224590521332, 0.001224496383984,
           0.001224400403285, 0.001224302579381,
           0.001224202912421, 0.001224101402552,
           0.001223998049928, 0.001223892854707,
           0.001223785817043, 0.001223676937100,
           0.001223566215042, 0.001223453651036,
           0.001223339245247, 0.001223222997852,
           0.001223104909026, 0.001222984978946,
           0.001222863207791, 0.001222739595745,
           0.001222614142993, 0.001222486849728,
           0.001222357716136, 0.001222226742416,
           0.001222093928763, 0.001221959275377,
           0.001221822782462, 0.001221684450222,
           0.001221544278866, 0.001221402268604,
           0.001221258419652, 0.001221112732223,
           0.001220965206540, 0.001220815842824,
           0.001220664641298, 0.001220511602192,
           0.001220356725735, 0.001220200012160,
           0.001220041461704, 0.001219881074605,
           0.001219718851104, 0.001219554791445,
           0.001219388895878, 0.001219221164651,
           0.001219051598014, 0.001218880196224,
           0.001218706959540, 0.001218531888223,
           0.001218354982535, 0.001218176242741,
           0.001217995669114, 0.001217813261924,
           0.001217629021445, 0.001217442947956,
           0.001217255041733, 0.001217065303064,
           0.001216873732229, 0.001216680329525,
           0.001216485095234, 0.001216288029657,
           0.001216089133086, 0.001215888405821,
           0.001215685848166, 0.001215481460426,
           0.001215275242907, 0.001215067195920,
           0.001214857319778, 0.001214645614796,
           0.001214432081295, 0.001214216719597,
           0.001213999530023, 0.001213780512903,
           0.001213559668563, 0.001213336997341,
           0.001213112499567, 0.001212886175581,
           0.001212658025724, 0.001212428050338,
           0.001212196249772, 0.001211962624372,
           0.001211727174491, 0.001211489900484,
           0.001211250802707, 0.001211009881521,
           0.001210767137288, 0.001210522570374,
           0.001210276181146, 0.001210027969975,
           0.001209777937238, 0.001209526083307,
           0.001209272408563, 0.001209016913388,
           0.001208759598168, 0.001208500463287,
           0.001208239509136, 0.001207976736110,
           0.001207712144602, 0.001207445735013,
           0.001207177507743, 0.001206907463194,
           0.001206635601776, 0.001206361923894,
           0.001206086429962, 0.001205809120397,
           0.001205529995614, 0.001205249056031,
           0.001204966302076, 0.001204681734173,
           0.001204395352747, 0.001204107158235,
           0.001203817151065, 0.001203525331676,
           0.001203231700508, 0.001202936258002,
           0.001202639004604, 0.001202339940760,
           0.001202039066922, 0.001201736383540,
           0.001201431891072, 0.001201125589976,
           0.001200817480713, 0.001200507563747,
           0.001200195839544, 0.001199882308574,
           0.001199566971308, 0.001199249828223,
           0.001198930879792, 0.001198610126501,
           0.001198287568826, 0.001197963207259,
           0.001197637042285, 0.001197309074397,
           0.001196979304084, 0.001196647731848,
           0.001196314358184, 0.001195979183598,
           0.001195642208591, 0.001195303433672,
           0.001194962859352, 0.001194620486141,
           0.001194276314554, 0.001193930345113,
           0.001193582578337, 0.001193233014748,
           0.001192881654873, 0.001192528499243,
           0.001192173548386, 0.001191816802840,
           0.001191458263139, 0.001191097929825,
           0.001190735803440, 0.001190371884528,
           0.001190006173636, 0.001189638671318,
           0.001189269378124, 0.001188898294612,
           0.001188525421340, 0.001188150758870,
           0.001187774307763, 0.001187396068589,
           0.001187016041916, 0.001186634228316,
           0.001186250628367, 0.001185865242641,
           0.001185478071721, 0.001185089116189,
           0.001184698376632, 0.001184305853639,
           0.001183911547796, 0.001183515459702,
           0.001183117589951, 0.001182717939141,
           0.001182316507875, 0.001181913296759,
           0.001181508306396, 0.001181101537398,
           0.001180692990378, 0.001180282665948,
           0.001179870564730, 0.001179456687340,
           0.001179041034406, 0.001178623606548,
           0.001178204404399, 0.001177783428589,
           0.001177360679749, 0.001176936158520,
           0.001176509865537, 0.001176081801444,
           0.001175651966884, 0.001175220362504,
           0.001174786988955, 0.001174351846890,
           0.001173914936962, 0.001173476259831,
           0.001173035816155, 0.001172593606597,
           0.001172149631825, 0.001171703892508,
           0.001171256389311, 0.001170807122918,
           0.001170356093993, 0.001169903303224,
           0.001169448751290, 0.001168992438874,
           0.001168534366664, 0.001168074535351,
           0.001167612945626, 0.001167149598182,
           0.001166684493719, 0.001166217632936,
           0.001165749016537, 0.001165278645225,
           0.001164806519710, 0.001164332640704,
           0.001163857008916, 0.001163379625071,
           0.001162900489876, 0.001162419604059,
           0.001161936968344, 0.001161452583455,
           0.001160966450120, 0.001160478569077,
           0.001159988941056, 0.001159497566794,
           0.001159004447033, 0.001158509582513,
           0.001158012973977, 0.001157514622179,
           0.001157014527865, 0.001156512691788,
           0.001156009114704, 0.001155503797370,
           0.001154996740550, 0.001154487945003,
           0.001153977411499, 0.001153465140803,
           0.001152951133688, 0.001152435390929,
           0.001151917913298, 0.001151398701578,
           0.001150877756549, 0.001150355078998,
           0.001149830669707, 0.001149304529470,
           0.001148776659076, 0.001148247059320,
           0.001147715731001, 0.001147182674918,
           0.001146647891873, 0.001146111382672,
           0.001145573148120, 0.001145033189031,
           0.001144491506215, 0.001143948100489,
           0.001143402972671, 0.001142856123582,
           0.001142307554041, 0.001141757264882,
           0.001141205256927, 0.001140651531006,
           0.001140096087958, 0.001139538928616,
           0.001138980053818, 0.001138419464408,
           0.001137857161228, 0.001137293145124,
           0.001136727416947, 0.001136159977546,
           0.001135590827779, 0.001135019968498,
           0.001134447400566, 0.001133873124844,
           0.001133297142197, 0.001132719453491,
           0.001132140059596, 0.001131558961386,
           0.001130976159730, 0.001130391655512,
           0.001129805449610, 0.001129217542906,
           0.001128627936283, 0.001128036630632,
           0.001127443626843, 0.001126848925805,
           0.001126252528416, 0.001125654435575,
           0.001125054648178, 0.001124453167132,
           0.001123849993341, 0.001123245127715,
           0.001122638571159, 0.001122030324593,
           0.001121420388928, 0.001120808765085,
           0.001120195453981, 0.001119580456544,
           0.001118963773697, 0.001118345406367,
           0.001117725355488, 0.001117103621993,
           0.001116480206816, 0.001115855110898,
           0.001115228335177, 0.001114599880599,
           0.001113969748108, 0.001113337938655,
           0.001112704453190, 0.001112069292667,
           0.001111432458041, 0.001110793950270,
           0.001110153770320, 0.001109511919148,
           0.001108868397726, 0.001108223207019,
           0.001107576348001, 0.001106927821644,
           0.001106277628922, 0.001105625770820,
           0.001104972248316, 0.001104317062391,
           0.001103660214035, 0.001103001704236,
           0.001102341533984, 0.001101679704273,
           0.001101016216106, 0.001100351070470,
           0.001099684268374, 0.001099015810821,
           0.001098345698814, 0.001097673933367,
           0.001097000515485, 0.001096325446188,
           0.001095648726490, 0.001094970357408,
           0.001094290339964, 0.001093608675184,
           0.001092925364088, 0.001092240407711,
           0.001091553807085, 0.001090865563238,
           0.001090175677209, 0.001089484150037,
           0.001088790982761, 0.001088096176427,
           0.001087399732078, 0.001086701650766,
           0.001086001933538, 0.001085300581452,
           0.001084597595559, 0.001083892976920,
           0.001083186726596, 0.001082478845647,
           0.001081769335146, 0.001081058196152,
           0.001080345429740, 0.001079631036983,
           0.001078915018957, 0.001078197376739,
           0.001077478111409, 0.001076757224049,
           0.001076034715747, 0.001075310587589,
           0.001074584840663, 0.001073857476064,
           0.001073128494890, 0.001072397898233,
           0.001071665687192, 0.001070931862875,
           0.001070196426383, 0.001069459378824,
           0.001068720721308, 0.001067980454947,
           0.001067238580853, 0.001066495100145,
           0.001065750013942, 0.001065003323367,
           0.001064255029541, 0.001063505133592,
           0.001062753636649, 0.001062000539847,
           0.001061245844310, 0.001060489551182,
           0.001059731661601, 0.001058972176705,
           0.001058211097640, 0.001057448425550,
           0.001056684161582, 0.001055918306891,
           0.001055150862626, 0.001054381829945,
           0.001053611210001, 0.001052839003959,
           0.001052065212979, 0.001051289838228,
           0.001050512880871, 0.001049734342077,
           0.001048954223022, 0.001048172524875,
           0.001047389248817, 0.001046604396025,
           0.001045817967682, 0.001045029964972,
           0.001044240389078, 0.001043449241192,
           0.001042656522503, 0.001041862234209,
           0.001041066377500, 0.001040268953573,
           0.001039469963635, 0.001038669408884,
           0.001037867290525, 0.001037063609771,
           0.001036258367826, 0.001035451565901,
           0.001034643205219, 0.001033833286988,
           0.001033021812431, 0.001032208782771,
           0.001031394199230, 0.001030578063032,
           0.001029760375411, 0.001028941137594,
           0.001028120350814, 0.001027298016311,
           0.001026474135317, 0.001025648709076,
           0.001024821738832, 0.001023993225825,
           0.001023163171305, 0.001022331576525,
           0.001021498442729, 0.001020663771176,
           0.001019827563124, 0.001018989819829,
           0.001018150542554, 0.001017309732559,
           0.001016467391116, 0.001015623519485,
           0.001014778118943, 0.001013931190761,
           0.001013082736210, 0.001012232756575,
           0.001011381253127, 0.001010528227156,
           0.001009673679939, 0.001008817612765,
           0.001007960026926, 0.001007100923709,
           0.001006240304408, 0.001005378170317,
           0.001004514522738, 0.001003649362971,
           0.001002782692312, 0.001001914512075,
           0.001001044823559, 0.001000173628077,
           0.000999300926940, 0.000998426721461,
           0.000997551012957, 0.000996673802747,
           0.000995795092147, 0.000994914882487,
           0.000994033175085, 0.000993149971276,
           0.000992265272379, 0.000991379079737,
           0.000990491394678, 0.000989602218540,
           0.000988711552659, 0.000987819398377,
           0.000986925757042, 0.000986030629992,
           0.000985134018578, 0.000984235924148,
           0.000983336348056, 0.000982435291657,
           0.000981532756306, 0.000980628743362,
           0.000979723254183, 0.000978816290134,
           0.000977907852584, 0.000976997942898,
           0.000976086562442, 0.000975173712595,
           0.000974259394725, 0.000973343610212,
           0.000972426360432, 0.000971507646770,
           0.000970587470604, 0.000969665833323,
           0.000968742736312, 0.000967818180963,
           0.000966892168665, 0.000965964700814,
           0.000965035778805, 0.000964105404041,
           0.000963173577913, 0.000962240301832,
           0.000961305577202, 0.000960369405426,
           0.000959431787916, 0.000958492726086,
           0.000957552221345, 0.000956610275111,
           0.000955666888800, 0.000954722063838,
           0.000953775801640, 0.000952828103636,
           0.000951878971248, 0.000950928405909,
           0.000949976409049, 0.000949022982097,
           0.000948068126494, 0.000947111843676,
           0.000946154135082, 0.000945195002151,
           0.000944234446328, 0.000943272469065,
           0.000942309071800, 0.000941344255995,
           0.000940378023093, 0.000939410374550,
           0.000938441311826, 0.000937470836380,
           0.000936498949668, 0.000935525653159,
           0.000934550948314, 0.000933574836602,
           0.000932597319491, 0.000931618398455,
           0.000930638074966, 0.000929656350504,
           0.000928673226534, 0.000927688704555,
           0.000926702786033, 0.000925715472461,
           0.000924726765322, 0.000923736666106,
           0.000922745176300, 0.000921752297400,
           0.000920758030901, 0.000919762378298,
           0.000918765341091, 0.000917766920777,
           0.000916767118866, 0.000915765936860,
           0.000914763376263, 0.000913759438587,
           0.000912754125346, 0.000911747438046,
           0.000910739378211, 0.000909729947354,
           0.000908719146992, 0.000907706978655,
           0.000906693443857, 0.000905678544129,
           0.000904662281000, 0.000903644655997,
           0.000902625670653, 0.000901605326503,
           0.000900583625079, 0.000899560567924,
           0.000898536156577, 0.000897510392578,
           0.000896483277473, 0.000895454812806,
           0.000894425000129, 0.000893393840988,
           0.000892361336939, 0.000891327489533,
           0.000890292300329, 0.000889255770885,
           0.000888217902762, 0.000887178697519,
           0.000886138156724, 0.000885096281941,
           0.000884053074742, 0.000883008536691,
           0.000881962669366, 0.000880915474341,
           0.000879866953191, 0.000878817107495,
           0.000877765938833, 0.000876713448786,
           0.000875659638940, 0.000874604510883,
           0.000873548066200, 0.000872490306486,
           0.000871431233328, 0.000870370848324,
           0.000869309153068, 0.000868246149160,
           0.000867181838201, 0.000866116221791,
           0.000865049301534, 0.000863981079040,
           0.000862911555910, 0.000861840733758,
           0.000860768614201, 0.000859695198844,
           0.000858620489309, 0.000857544487212,
           0.000856467194173, 0.000855388611813,
           0.000854308741757, 0.000853227585628,
           0.000852145145058, 0.000851061421674,
           0.000849976417105, 0.000848890132988,
           0.000847802570956, 0.000846713732648,
           0.000845623619705, 0.000844532233762,
           0.000843439576465, 0.000842345649463,
           0.000841250454395, 0.000840153992916,
           0.000839056266674, 0.000837957277321,
           0.000836857026512, 0.000835755515906,
           0.000834652747156, 0.000833548721927,
           0.000832443441880, 0.000831336908674,
           0.000830229123982, 0.000829120089465,
           0.000828009806799, 0.000826898277648,
           0.000825785503694, 0.000824671486606,
           0.000823556228062, 0.000822439729741,
           0.000821321993326, 0.000820203020498,
           0.000819082812940, 0.000817961372340,
           0.000816838700387, 0.000815714798769,
           0.000814589669180, 0.000813463313309,
           0.000812335732862, 0.000811206929524,
           0.000810076905004, 0.000808945660997,
           0.000807813199208, 0.000806679521344,
           0.000805544629109, 0.000804408524211,
           0.000803271208361, 0.000802132683274,
           0.000800992950657, 0.000799852012235,
           0.000798709869718, 0.000797566524827,
           0.000796421979288, 0.000795276234818,
           0.000794129293145, 0.000792981155995,
           0.000791831825094, 0.000790681302176,
           0.000789529588969, 0.000788376687213,
           0.000787222598635, 0.000786067324981,
           0.000784910867982, 0.000783753229387,
           0.000782594410932, 0.000781434414364,
           0.000780273241433, 0.000779110893881,
           0.000777947373461, 0.000776782681921,
           0.000775616821021, 0.000774449792510,
           0.000773281598147, 0.000772112239694,
           0.000770941718905, 0.000769770037544,
           0.000768597197378, 0.000767423200170,
           0.000766248047688, 0.000765071741701,
           0.000763894283979, 0.000762715676295,
           0.000761535920425, 0.000760355018142,
           0.000759172971225, 0.000757989781456,
           0.000756805450611, 0.000755619980479,
           0.000754433372838, 0.000753245629480,
           0.000752056752191, 0.000750866742759,
           0.000749675602978, 0.000748483334638,
           0.000747289939539, 0.000746095419471,
           0.000744899776237, 0.000743703011635,
           0.000742505127468, 0.000741306125539,
           0.000740106007648, 0.000738904775612,
           0.000737702431230, 0.000736498976318,
           0.000735294412679, 0.000734088742138,
           0.000732881966502, 0.000731674087591,
           0.000730465107221, 0.000729255027214,
           0.000728043849393, 0.000726831575576,
           0.000725618207594, 0.000724403747267,
           0.000723188196430, 0.000721971556909,
           0.000720753830538, 0.000719535019148,
           0.000718315124573, 0.000717094148652,
           0.000715872093221, 0.000714648960123,
           0.000713424751194, 0.000712199468283,
           0.000710973113227, 0.000709745687881,
           0.000708517194083, 0.000707287633692,
           0.000706057008553, 0.000704825320520,
           0.000703592571448, 0.000702358763193,
           0.000701123897608, 0.000699887976559,
           0.000698651001901, 0.000697412975500,
           0.000696173899216, 0.000694933774916,
           0.000693692604467, 0.000692450389740,
           0.000691207132598, 0.000689962834921,
           0.000688717498576, 0.000687471125442,
           0.000686223717391, 0.000684975276303,
           0.000683725804060, 0.000682475302538,
           0.000681223773622, 0.000679971219196,
           0.000678717641147, 0.000677463041361,
           0.000676207421725, 0.000674950784129,
           0.000673693130467, 0.000672434462634,
           0.000671174782520, 0.000669914092025,
           0.000668652393045, 0.000667389687477,
           0.000666125977230, 0.000664861264197,
           0.000663595550288, 0.000662328837405,
           0.000661061127458, 0.000659792422353,
           0.000658522723998, 0.000657252034311,
           0.000655980355197, 0.000654707688578,
           0.000653434036364, 0.000652159400474,
           0.000650883782828, 0.000649607185347,
           0.000648329609949, 0.000647051058560,
           0.000645771533105, 0.000644491035509,
           0.000643209567701, 0.000641927131608,
           0.000640643729163, 0.000639359362297,
           0.000638074032942, 0.000636787743034,
           0.000635500494512, 0.000634212289312,
           0.000632923129370, 0.000631633016630,
           0.000630341953035, 0.000629049940525,
           0.000627756981049, 0.000626463076550,
           0.000625168228978, 0.000623872440283,
           0.000622575712412, 0.000621278047321,
           0.000619979446961, 0.000618679913287,
           0.000617379448258, 0.000616078053830,
           0.000614775731959, 0.000613472484613,
           0.000612168313744, 0.000610863221323,
           0.000609557209313, 0.000608250279679,
           0.000606942434389, 0.000605633675411,
           0.000604324004716, 0.000603013424273,
           0.000601701936061, 0.000600389542049,
           0.000599076244214, 0.000597762044535,
           0.000596446944987, 0.000595130947552,
           0.000593814054211, 0.000592496266945,
           0.000591177587739, 0.000589858018579,
           0.000588537561450, 0.000587216218340,
           0.000585893991238, 0.000584570882135,
           0.000583246893022, 0.000581922025892,
           0.000580596282744, 0.000579269665566,
           0.000577942176361, 0.000576613817123,
           0.000575284589858, 0.000573954496560,
           0.000572623539236, 0.000571291719887,
           0.000569959040519, 0.000568625503140,
           0.000567291109755, 0.000565955862371,
           0.000564619763002, 0.000563282813658,
           0.000561945016354, 0.000560606373097,
           0.000559266885910, 0.000557926556805,
           0.000556585387801, 0.000555243380918,
           0.000553900538172, 0.000552556861589,
           0.000551212353190, 0.000549867015000,
           0.000548520849044, 0.000547173857348,
           0.000545826041938, 0.000544477404848,
           0.000543127948102, 0.000541777673736,
           0.000540426583781, 0.000539074680269,
           0.000537721965243, 0.000536368440729,
           0.000535014108770, 0.000533658971404,
           0.000532303030673, 0.000530946288614,
           0.000529588747271, 0.000528230408693,
           0.000526871274916, 0.000525511347992,
           0.000524150629965, 0.000522789122885,
           0.000521426828802, 0.000520063749768,
           0.000518699887831, 0.000517335245047,
           0.000515969823473, 0.000514603625155,
           0.000513236652160, 0.000511868906540,
           0.000510500390357, 0.000509131105668,
           0.000507761054538, 0.000506390239028,
           0.000505018661200, 0.000503646323120,
           0.000502273226855, 0.000500899374471,
           0.000499524768033, 0.000498149409617,
           0.000496773301289, 0.000495396445122,
           0.000494018843187, 0.000492640497560,
           0.000491261410314, 0.000489881583527,
           0.000488501019276, 0.000487119719637,
           0.000485737686692, 0.000484354922521,
           0.000482971429204, 0.000481587208825,
           0.000480202263467, 0.000478816595218,
           0.000477430206157, 0.000476043098380,
           0.000474655273969, 0.000473266735014,
           0.000471877483606, 0.000470487521838,
           0.000469096851801, 0.000467705475589,
           0.000466313395293, 0.000464920613014,
           0.000463527130847, 0.000462132950888,
           0.000460738075238, 0.000459342505993,
           0.000457946245259, 0.000456549295136,
           0.000455151657726, 0.000453753335132,
           0.000452354329461, 0.000450954642820,
           0.000449554277314, 0.000448153235050,
           0.000446751518141, 0.000445349128695,
           0.000443946068823, 0.000442542340636,
           0.000441137946251, 0.000439732887776,
           0.000438327167334, 0.000436920787033,
           0.000435513748998, 0.000434106055339,
           0.000432697708184, 0.000431288709646,
           0.000429879061847, 0.000428468766915,
           0.000427057826966, 0.000425646244128,
           0.000424234020523, 0.000422821158280,
           0.000421407659524, 0.000419993526384,
           0.000418578760988, 0.000417163365467,
           0.000415747341949, 0.000414330692570,
           0.000412913419457, 0.000411495524751,
           0.000410077010579, 0.000408657879079,
           0.000407238132390, 0.000405817772647,
           0.000404396801988, 0.000402975222552,
           0.000401553036480, 0.000400130245913,
           0.000398706852991, 0.000397282859860,
           0.000395858268661, 0.000394433081539,
           0.000393007300642, 0.000391580928112,
           0.000390153966099, 0.000388726416750,
           0.000387298282216, 0.000385869564641,
           0.000384440266187, 0.000383010388997,
           0.000381579935224, 0.000380148907024,
           0.000378717306551, 0.000377285135957,
           0.000375852397400, 0.000374419093039,
           0.000372985225032, 0.000371550795530,
           0.000370115806702, 0.000368680260702,
           0.000367244159692, 0.000365807505837,
           0.000364370301297, 0.000362932548236,
           0.000361494248819, 0.000360055405211,
           0.000358616019576, 0.000357176094085,
           0.000355735630902, 0.000354294632196,
           0.000352853100139, 0.000351411036899,
           0.000349968444644, 0.000348525325552,
           0.000347081681792, 0.000345637515537,
           0.000344192828960, 0.000342747624239,
           0.000341301903547, 0.000339855669059,
           0.000338408922958, 0.000336961667415,
           0.000335513904615, 0.000334065636732,
           0.000332616865950, 0.000331167594449,
           0.000329717824410, 0.000328267558017,
           0.000326816797450, 0.000325365544895,
           0.000323913802538, 0.000322461572562,
           0.000321008857156, 0.000319555658504,
           0.000318101978793, 0.000316647820218,
           0.000315193184959, 0.000313738075211,
           0.000312282493164, 0.000310826441011,
           0.000309369920939, 0.000307912935145,
           0.000306455485820, 0.000304997575160,
           0.000303539205357, 0.000302080378609,
           0.000300621097111, 0.000299161363057,
           0.000297701178653, 0.000296240546086,
           0.000294779467563, 0.000293317945280,
           0.000291855981439, 0.000290393578239,
           0.000288930737883, 0.000287467462571,
           0.000286003754510, 0.000284539615897,
           0.000283075048944, 0.000281610055848,
           0.000280144638819, 0.000278678800063,
           0.000277212541784, 0.000275745866193,
           0.000274278775493, 0.000272811271897,
           0.000271343357613, 0.000269875034851,
           0.000268406305819, 0.000266937172732,
           0.000265467637798, 0.000263997703229,
           0.000262527371243, 0.000261056644048,
           0.000259585523861, 0.000258114012894,
           0.000256642113364, 0.000255169827488,
           0.000253697157480, 0.000252224105556,
           0.000250750673939, 0.000249276864841,
           0.000247802680483, 0.000246328123085,
           0.000244853194868, 0.000243377898048,
           0.000241902234849, 0.000240426207493,
           0.000238949818199, 0.000237473069194,
           0.000235995962696, 0.000234518500932,
           0.000233040686126, 0.000231562520501,
           0.000230084006283, 0.000228605145699,
           0.000227125940974, 0.000225646394335,
           0.000224166508009, 0.000222686284223,
           0.000221205725209, 0.000219724833192,
           0.000218243610402, 0.000216762059070,
           0.000215280181425, 0.000213797979700,
           0.000212315456124, 0.000210832612930,
           0.000209349452349, 0.000207865976616,
           0.000206382187961, 0.000204898088621,
           0.000203413680828, 0.000201928966818,
           0.000200443948825, 0.000198958629085,
           0.000197473009833, 0.000195987093308,
           0.000194500881744, 0.000193014377380,
           0.000191527582452, 0.000190040499202,
           0.000188553129865, 0.000187065476681,
           0.000185577541891, 0.000184089327731,
           0.000182600836448, 0.000181112070275,
           0.000179623031459, 0.000178133722239,
           0.000176644144859, 0.000175154301559,
           0.000173664194582, 0.000172173826173,
           0.000170683198574, 0.000169192314031,
           0.000167701174785, 0.000166209783084,
           0.000164718141171, 0.000163226251292,
           0.000161734115694, 0.000160241736623,
           0.000158749116324, 0.000157256257046,
           0.000155763161034, 0.000154269830538,
           0.000152776267804, 0.000151282475085,
           0.000149788454622, 0.000148294208671,
           0.000146799739479, 0.000145305049293,
           0.000143810140370, 0.000142315014953,
           0.000140819675295, 0.000139324123650,
           0.000137828362266, 0.000136332393397,
           0.000134836219294, 0.000133339842210,
           0.000131843264396, 0.000130346488105,
           0.000128849515593, 0.000127352349111,
           0.000125854990914, 0.000124357443256,
           0.000122859708390, 0.000121361788571,
           0.000119863686058, 0.000118365403099,
           0.000116866941955, 0.000115368304882,
           0.000113869494131, 0.000112370511965,
           0.000110871360635, 0.000109372042402,
           0.000107872559516, 0.000106372914245,
           0.000104873108838, 0.000103373145558,
           0.000101873026659, 0.000100372754401,
           0.000098872331042, 0.000097371758842,
           0.000095871040059, 0.000094370176952,
           0.000092869171781, 0.000091368026806,
           0.000089866744286, 0.000088365326480,
           0.000086863775651, 0.000085362094057,
           0.000083860283959, 0.000082358347619,
           0.000080856287298, 0.000079354105255,
           0.000077851803754, 0.000076349385056,
           0.000074846851421, 0.000073344205112,
           0.000071841448391, 0.000070338583521,
           0.000068835612765, 0.000067332538382,
           0.000065829362639, 0.000064326087796,
           0.000062822716117, 0.000061319249866,
           0.000059815691306, 0.000058312042699,
           0.000056808306310, 0.000055304484403,
           0.000053800579241, 0.000052296593089,
           0.000050792528210, 0.000049288386869,
           0.000047784171330, 0.000046279883858,
           0.000044775526718, 0.000043271102173,
           0.000041766612490, 0.000040262059935,
           0.000038757446771, 0.000037252775263,
           0.000035748047678, 0.000034243266282,
           0.000032738433341, 0.000031233551123,
           0.000029728621889, 0.000028223647916,
           0.000026718631462, 0.000025213574805,
           0.000023708480210, 0.000022203349953,
           0.000020698186311, 0.000019192991565,
           0.000017687768007, 0.000016182517943,
           0.000014677243707, 0.000013171947685,
           0.000011666632368, 0.000010161300472,
           0.000008655955245, 0.000007150601403,
           0.000005645248421, 0.000004139925747,
           0.000002634790132, 0.000001131875961};

  } else if (N == 5120) {

    w_k = {0.000613532368087, 0.000613532137140,
           0.000613531675246, 0.000613530982406,
           0.000613530058618, 0.000613528903885,
           0.000613527518205, 0.000613525901580,
           0.000613524054011, 0.000613521975498,
           0.000613519666041, 0.000613517125641,
           0.000613514354303, 0.000613511352022,
           0.000613508118803, 0.000613504654645,
           0.000613500959551, 0.000613497033521,
           0.000613492876557, 0.000613488488663,
           0.000613483869838, 0.000613479020082,
           0.000613473939401, 0.000613468627794,
           0.000613463085264, 0.000613457311813,
           0.000613451307443, 0.000613445072157,
           0.000613438605956, 0.000613431908843,
           0.000613424980823, 0.000613417821894,
           0.000613410432062, 0.000613402811329,
           0.000613394959698, 0.000613386877170,
           0.000613378563751, 0.000613370019442,
           0.000613361244248, 0.000613352238171,
           0.000613343001214, 0.000613333533381,
           0.000613323834678, 0.000613313905105,
           0.000613303744666, 0.000613293353370,
           0.000613282731212, 0.000613271878203,
           0.000613260794345, 0.000613249479641,
           0.000613237934098, 0.000613226157719,
           0.000613214150507, 0.000613201912468,
           0.000613189443607, 0.000613176743927,
           0.000613163813434, 0.000613150652133,
           0.000613137260027, 0.000613123637125,
           0.000613109783429, 0.000613095698945,
           0.000613081383679, 0.000613066837635,
           0.000613052060818, 0.000613037053236,
           0.000613021814893, 0.000613006345795,
           0.000612990645948, 0.000612974715358,
           0.000612958554031, 0.000612942161973,
           0.000612925539189, 0.000612908685687,
           0.000612891601474, 0.000612874286554,
           0.000612856740934, 0.000612838964622,
           0.000612820957624, 0.000612802719947,
           0.000612784251596, 0.000612765552581,
           0.000612746622908, 0.000612727462581,
           0.000612708071614, 0.000612688450007,
           0.000612668597771, 0.000612648514915,
           0.000612628201443, 0.000612607657363,
           0.000612586882687, 0.000612565877419,
           0.000612544641566, 0.000612523175140,
           0.000612501478145, 0.000612479550591,
           0.000612457392487, 0.000612435003841,
           0.000612412384660, 0.000612389534954,
           0.000612366454731, 0.000612343144000,
           0.000612319602769, 0.000612295831047,
           0.000612271828845, 0.000612247596170,
           0.000612223133030, 0.000612198439438,
           0.000612173515401, 0.000612148360927,
           0.000612122976027, 0.000612097360710,
           0.000612071514987, 0.000612045438867,
           0.000612019132359, 0.000611992595474,
           0.000611965828222, 0.000611938830611,
           0.000611911602654, 0.000611884144360,
           0.000611856455739, 0.000611828536801,
           0.000611800387558, 0.000611772008021,
           0.000611743398197, 0.000611714558101,
           0.000611685487743, 0.000611656187133,
           0.000611626656281, 0.000611596895200,
           0.000611566903899, 0.000611536682393,
           0.000611506230689, 0.000611475548801,
           0.000611444636742, 0.000611413494519,
           0.000611382122148, 0.000611350519641,
           0.000611318687006, 0.000611286624258,
           0.000611254331408, 0.000611221808469,
           0.000611189055451, 0.000611156072370,
           0.000611122859236, 0.000611089416060,
           0.000611055742859, 0.000611021839643,
           0.000610987706424, 0.000610953343215,
           0.000610918750031, 0.000610883926884,
           0.000610848873786, 0.000610813590751,
           0.000610778077793, 0.000610742334924,
           0.000610706362158, 0.000610670159510,
           0.000610633726991, 0.000610597064617,
           0.000610560172400, 0.000610523050355,
           0.000610485698496, 0.000610448116837,
           0.000610410305391, 0.000610372264174,
           0.000610333993199, 0.000610295492481,
           0.000610256762034, 0.000610217801873,
           0.000610178612014, 0.000610139192468,
           0.000610099543254, 0.000610059664384,
           0.000610019555875, 0.000609979217740,
           0.000609938649997, 0.000609897852659,
           0.000609856825742, 0.000609815569260,
           0.000609774083231, 0.000609732367672,
           0.000609690422593, 0.000609648248015,
           0.000609605843951, 0.000609563210418,
           0.000609520347434, 0.000609477255011,
           0.000609433933168, 0.000609390381920,
           0.000609346601284, 0.000609302591278,
           0.000609258351916, 0.000609213883215,
           0.000609169185194, 0.000609124257869,
           0.000609079101255, 0.000609033715370,
           0.000608988100231, 0.000608942255856,
           0.000608896182262, 0.000608849879466,
           0.000608803347486, 0.000608756586337,
           0.000608709596040, 0.000608662376613,
           0.000608614928070, 0.000608567250432,
           0.000608519343715, 0.000608471207937,
           0.000608422843119, 0.000608374249277,
           0.000608325426429, 0.000608276374594,
           0.000608227093792, 0.000608177584037,
           0.000608127845352, 0.000608077877755,
           0.000608027681264, 0.000607977255896,
           0.000607926601675, 0.000607875718614,
           0.000607824606737, 0.000607773266061,
           0.000607721696606, 0.000607669898391,
           0.000607617871436, 0.000607565615760,
           0.000607513131383, 0.000607460418325,
           0.000607407476605, 0.000607354306244,
           0.000607300907261, 0.000607247279677,
           0.000607193423512, 0.000607139338785,
           0.000607085025518, 0.000607030483730,
           0.000606975713444, 0.000606920714678,
           0.000606865487455, 0.000606810031792,
           0.000606754347713, 0.000606698435240,
           0.000606642294391, 0.000606585925189,
           0.000606529327654, 0.000606472501810,
           0.000606415447675, 0.000606358165271,
           0.000606300654622, 0.000606242915747,
           0.000606184948669, 0.000606126753411,
           0.000606068329992, 0.000606009678436,
           0.000605950798764, 0.000605891691001,
           0.000605832355165, 0.000605772791282,
           0.000605712999371, 0.000605652979457,
           0.000605592731562, 0.000605532255709,
           0.000605471551920, 0.000605410620218,
           0.000605349460626, 0.000605288073166,
           0.000605226457865, 0.000605164614741,
           0.000605102543821, 0.000605040245126,
           0.000604977718682, 0.000604914964511,
           0.000604851982636, 0.000604788773082,
           0.000604725335871, 0.000604661671029,
           0.000604597778579, 0.000604533658545,
           0.000604469310951, 0.000604404735822,
           0.000604339933182, 0.000604274903055,
           0.000604209645464, 0.000604144160436,
           0.000604078447995, 0.000604012508167,
           0.000603946340975, 0.000603879946443,
           0.000603813324597, 0.000603746475464,
           0.000603679399067, 0.000603612095432,
           0.000603544564582, 0.000603476806548,
           0.000603408821349, 0.000603340609017,
           0.000603272169572, 0.000603203503043,
           0.000603134609453, 0.000603065488832,
           0.000602996141203, 0.000602926566593,
           0.000602856765028, 0.000602786736536,
           0.000602716481139, 0.000602645998868,
           0.000602575289748, 0.000602504353805,
           0.000602433191065, 0.000602361801558,
           0.000602290185308, 0.000602218342343,
           0.000602146272689, 0.000602073976374,
           0.000602001453425, 0.000601928703871,
           0.000601855727737, 0.000601782525050,
           0.000601709095839, 0.000601635440133,
           0.000601561557958, 0.000601487449340,
           0.000601413114309, 0.000601338552894,
           0.000601263765124, 0.000601188751023,
           0.000601113510621, 0.000601038043948,
           0.000600962351030, 0.000600886431895,
           0.000600810286576, 0.000600733915097,
           0.000600657317489, 0.000600580493781,
           0.000600503444000, 0.000600426168178,
           0.000600348666341, 0.000600270938520,
           0.000600192984744, 0.000600114805043,
           0.000600036399444, 0.000599957767978,
           0.000599878910676, 0.000599799827567,
           0.000599720518677, 0.000599640984041,
           0.000599561223687, 0.000599481237646,
           0.000599401025945, 0.000599320588618,
           0.000599239925692, 0.000599159037200,
           0.000599077923171, 0.000598996583636,
           0.000598915018625, 0.000598833228170,
           0.000598751212302, 0.000598668971048,
           0.000598586504444, 0.000598503812518,
           0.000598420895303, 0.000598337752828,
           0.000598254385126, 0.000598170792228,
           0.000598086974164, 0.000598002930969,
           0.000597918662672, 0.000597834169304,
           0.000597749450899, 0.000597664507489,
           0.000597579339104, 0.000597493945777,
           0.000597408327538, 0.000597322484425,
           0.000597236416465, 0.000597150123693,
           0.000597063606138, 0.000596976863838,
           0.000596889896821, 0.000596802705122,
           0.000596715288773, 0.000596627647809,
           0.000596539782259, 0.000596451692160,
           0.000596363377541, 0.000596274838441,
           0.000596186074889, 0.000596097086919,
           0.000596007874565, 0.000595918437859,
           0.000595828776839, 0.000595738891534,
           0.000595648781981, 0.000595558448211,
           0.000595467890262, 0.000595377108164,
           0.000595286101954, 0.000595194871665,
           0.000595103417331, 0.000595011738986,
           0.000594919836667, 0.000594827710407,
           0.000594735360240, 0.000594642786203,
           0.000594549988328, 0.000594456966653,
           0.000594363721210, 0.000594270252036,
           0.000594176559165, 0.000594082642632,
           0.000593988502473, 0.000593894138726,
           0.000593799551422, 0.000593704740600,
           0.000593609706296, 0.000593514448541,
           0.000593418967378, 0.000593323262835,
           0.000593227334955, 0.000593131183769,
           0.000593034809316, 0.000592938211632,
           0.000592841390753, 0.000592744346716,
           0.000592647079557, 0.000592549589311,
           0.000592451876019, 0.000592353939713,
           0.000592255780433, 0.000592157398214,
           0.000592058793094, 0.000591959965110,
           0.000591860914299, 0.000591761640700,
           0.000591662144348, 0.000591562425282,
           0.000591462483539, 0.000591362319155,
           0.000591261932169, 0.000591161322621,
           0.000591060490545, 0.000590959435982,
           0.000590858158968, 0.000590756659544,
           0.000590654937743, 0.000590552993610,
           0.000590450827178, 0.000590348438487,
           0.000590245827578, 0.000590142994483,
           0.000590039939249, 0.000589936661910,
           0.000589833162507, 0.000589729441076,
           0.000589625497658, 0.000589521332293,
           0.000589416945019, 0.000589312335875,
           0.000589207504901, 0.000589102452137,
           0.000588997177622, 0.000588891681394,
           0.000588785963495, 0.000588680023964,
           0.000588573862842, 0.000588467480165,
           0.000588360875979, 0.000588254050321,
           0.000588147003228, 0.000588039734747,
           0.000587932244912, 0.000587824533767,
           0.000587716601355, 0.000587608447709,
           0.000587500072877, 0.000587391476898,
           0.000587282659809, 0.000587173621657,
           0.000587064362477, 0.000586954882314,
           0.000586845181209, 0.000586735259203,
           0.000586625116336, 0.000586514752650,
           0.000586404168188, 0.000586293362990,
           0.000586182337098, 0.000586071090556,
           0.000585959623401, 0.000585847935679,
           0.000585736027432, 0.000585623898700,
           0.000585511549528, 0.000585398979954,
           0.000585286190024, 0.000585173179780,
           0.000585059949263, 0.000584946498518,
           0.000584832827585, 0.000584718936509,
           0.000584604825331, 0.000584490494096,
           0.000584375942845, 0.000584261171622,
           0.000584146180470, 0.000584030969434,
           0.000583915538554, 0.000583799887876,
           0.000583684017443, 0.000583567927300,
           0.000583451617487, 0.000583335088050,
           0.000583218339035, 0.000583101370482,
           0.000582984182436, 0.000582866774943,
           0.000582749148047, 0.000582631301791,
           0.000582513236218, 0.000582394951377,
           0.000582276447308, 0.000582157724058,
           0.000582038781670, 0.000581919620192,
           0.000581800239665, 0.000581680640136,
           0.000581560821648, 0.000581440784250,
           0.000581320527983, 0.000581200052896,
           0.000581079359031, 0.000580958446437,
           0.000580837315156, 0.000580715965234,
           0.000580594396720, 0.000580472609656,
           0.000580350604089, 0.000580228380067,
           0.000580105937634, 0.000579983276836,
           0.000579860397719, 0.000579737300329,
           0.000579613984714, 0.000579490450921,
           0.000579366698993, 0.000579242728980,
           0.000579118540926, 0.000578994134880,
           0.000578869510888, 0.000578744668996,
           0.000578619609251, 0.000578494331703,
           0.000578368836396, 0.000578243123378,
           0.000578117192697, 0.000577991044400,
           0.000577864678535, 0.000577738095147,
           0.000577611294287, 0.000577484276002,
           0.000577357040339, 0.000577229587345,
           0.000577101917069, 0.000576974029560,
           0.000576845924865, 0.000576717603034,
           0.000576589064113, 0.000576460308152,
           0.000576331335194, 0.000576202145297,
           0.000576072738503, 0.000575943114863,
           0.000575813274426, 0.000575683217240,
           0.000575552943353, 0.000575422452816,
           0.000575291745677, 0.000575160821987,
           0.000575029681792, 0.000574898325144,
           0.000574766752092, 0.000574634962684,
           0.000574502956971, 0.000574370735003,
           0.000574238296829, 0.000574105642500,
           0.000573972772063, 0.000573839685572,
           0.000573706383072, 0.000573572864620,
           0.000573439130261, 0.000573305180046,
           0.000573171014029, 0.000573036632256,
           0.000572902034779, 0.000572767221650,
           0.000572632192919, 0.000572496948636,
           0.000572361488852, 0.000572225813620,
           0.000572089922988, 0.000571953817010,
           0.000571817495736, 0.000571680959216,
           0.000571544207505, 0.000571407240649,
           0.000571270058706, 0.000571132661722,
           0.000570995049752, 0.000570857222848,
           0.000570719181060, 0.000570580924440,
           0.000570442453042, 0.000570303766916,
           0.000570164866114, 0.000570025750693,
           0.000569886420698, 0.000569746876186,
           0.000569607117209, 0.000569467143821,
           0.000569326956072, 0.000569186554016,
           0.000569045937705, 0.000568905107192,
           0.000568764062532, 0.000568622803777,
           0.000568481330978, 0.000568339644192,
           0.000568197743469, 0.000568055628866,
           0.000567913300432, 0.000567770758223,
           0.000567628002294, 0.000567485032696,
           0.000567341849486, 0.000567198452715,
           0.000567054842438, 0.000566911018709,
           0.000566766981583, 0.000566622731111,
           0.000566478267353, 0.000566333590358,
           0.000566188700185, 0.000566043596882,
           0.000565898280511, 0.000565752751124,
           0.000565607008773, 0.000565461053514,
           0.000565314885406, 0.000565168504501,
           0.000565021910853, 0.000564875104517,
           0.000564728085551, 0.000564580854009,
           0.000564433409947, 0.000564285753419,
           0.000564137884482, 0.000563989803190,
           0.000563841509601, 0.000563693003768,
           0.000563544285751, 0.000563395355601,
           0.000563246213379, 0.000563096859137,
           0.000562947292932, 0.000562797514825,
           0.000562647524865, 0.000562497323113,
           0.000562346909626, 0.000562196284459,
           0.000562045447668, 0.000561894399310,
           0.000561743139444, 0.000561591668126,
           0.000561439985411, 0.000561288091360,
           0.000561135986024, 0.000560983669467,
           0.000560831141742, 0.000560678402911,
           0.000560525453025, 0.000560372292144,
           0.000560218920329, 0.000560065337635,
           0.000559911544119, 0.000559757539840,
           0.000559603324857, 0.000559448899227,
           0.000559294263007, 0.000559139416255,
           0.000558984359036, 0.000558829091398,
           0.000558673613406, 0.000558517925118,
           0.000558362026591, 0.000558205917884,
           0.000558049599054, 0.000557893070164,
           0.000557736331272, 0.000557579382433,
           0.000557422223711, 0.000557264855162,
           0.000557107276846, 0.000556949488823,
           0.000556791491152, 0.000556633283892,
           0.000556474867103, 0.000556316240844,
           0.000556157405177, 0.000555998360162,
           0.000555839105852, 0.000555679642315,
           0.000555519969608, 0.000555360087789,
           0.000555199996923, 0.000555039697066,
           0.000554879188281, 0.000554718470625,
           0.000554557544162, 0.000554396408951,
           0.000554235065053, 0.000554073512531,
           0.000553911751440, 0.000553749781846,
           0.000553587603810, 0.000553425217390,
           0.000553262622647, 0.000553099819645,
           0.000552936808445, 0.000552773589106,
           0.000552610161691, 0.000552446526263,
           0.000552282682883, 0.000552118631608,
           0.000551954372505, 0.000551789905635,
           0.000551625231058, 0.000551460348837,
           0.000551295259035, 0.000551129961714,
           0.000550964456934, 0.000550798744759,
           0.000550632825252, 0.000550466698475,
           0.000550300364489, 0.000550133823358,
           0.000549967075145, 0.000549800119914,
           0.000549632957724, 0.000549465588640,
           0.000549298012727, 0.000549130230045,
           0.000548962240657, 0.000548794044628,
           0.000548625642021, 0.000548457032901,
           0.000548288217328, 0.000548119195368,
           0.000547949967083, 0.000547780532539,
           0.000547610891797, 0.000547441044922,
           0.000547270991980, 0.000547100733031,
           0.000546930268143, 0.000546759597377,
           0.000546588720800, 0.000546417638474,
           0.000546246350464, 0.000546074856835,
           0.000545903157652, 0.000545731252979,
           0.000545559142882, 0.000545386827423,
           0.000545214306669, 0.000545041580684,
           0.000544868649533, 0.000544695513282,
           0.000544522171996, 0.000544348625741,
           0.000544174874579, 0.000544000918578,
           0.000543826757802, 0.000543652392319,
           0.000543477822193, 0.000543303047493,
           0.000543128068276, 0.000542952884618,
           0.000542777496578, 0.000542601904227,
           0.000542426107627, 0.000542250106846,
           0.000542073901952, 0.000541897493007,
           0.000541720880082, 0.000541544063239,
           0.000541367042548, 0.000541189818076,
           0.000541012389885, 0.000540834758048,
           0.000540656922628, 0.000540478883692,
           0.000540300641309, 0.000540122195544,
           0.000539943546468, 0.000539764694141,
           0.000539585638638, 0.000539406380021,
           0.000539226918362, 0.000539047253724,
           0.000538867386176, 0.000538687315789,
           0.000538507042626, 0.000538326566758,
           0.000538145888253, 0.000537965007176,
           0.000537783923599, 0.000537602637589,
           0.000537421149210, 0.000537239458536,
           0.000537057565635, 0.000536875470572,
           0.000536693173416, 0.000536510674240,
           0.000536327973107, 0.000536145070089,
           0.000535961965254, 0.000535778658672,
           0.000535595150410, 0.000535411440539,
           0.000535227529126, 0.000535043416244,
           0.000534859101959, 0.000534674586341,
           0.000534489869459, 0.000534304951384,
           0.000534119832185, 0.000533934511931,
           0.000533748990694, 0.000533563268539,
           0.000533377345542, 0.000533191221767,
           0.000533004897290, 0.000532818372178,
           0.000532631646499, 0.000532444720328,
           0.000532257593732, 0.000532070266783,
           0.000531882739551, 0.000531695012106,
           0.000531507084522, 0.000531318956864,
           0.000531130629206, 0.000530942101620,
           0.000530753374176, 0.000530564446944,
           0.000530375319994, 0.000530185993401,
           0.000529996467236, 0.000529806741567,
           0.000529616816468, 0.000529426692008,
           0.000529236368262, 0.000529045845297,
           0.000528855123190, 0.000528664202008,
           0.000528473081827, 0.000528281762718,
           0.000528090244750, 0.000527898527999,
           0.000527706612533, 0.000527514498429,
           0.000527322185755, 0.000527129674586,
           0.000526936964997, 0.000526744057053,
           0.000526550950832, 0.000526357646406,
           0.000526164143847, 0.000525970443230,
           0.000525776544623, 0.000525582448107,
           0.000525388153744, 0.000525193661618,
           0.000524998971798, 0.000524804084354,
           0.000524608999363, 0.000524413716898,
           0.000524218237031, 0.000524022559838,
           0.000523826685392, 0.000523630613763,
           0.000523434345032, 0.000523237879265,
           0.000523041216543, 0.000522844356935,
           0.000522647300518, 0.000522450047362,
           0.000522252597548, 0.000522054951146,
           0.000521857108229, 0.000521659068877,
           0.000521460833158, 0.000521262401151,
           0.000521063772929, 0.000520864948566,
           0.000520665928140, 0.000520466711722,
           0.000520267299392, 0.000520067691217,
           0.000519867887280, 0.000519667887655,
           0.000519467692412, 0.000519267301632,
           0.000519066715388, 0.000518865933757,
           0.000518664956809, 0.000518463784631,
           0.000518262417288, 0.000518060854858,
           0.000517859097423, 0.000517657145050,
           0.000517454997823, 0.000517252655813,
           0.000517050119098, 0.000516847387751,
           0.000516644461855, 0.000516441341482,
           0.000516238026708, 0.000516034517612,
           0.000515830814268, 0.000515626916754,
           0.000515422825147, 0.000515218539524,
           0.000515014059962, 0.000514809386537,
           0.000514604519325, 0.000514399458406,
           0.000514194203855, 0.000513988755749,
           0.000513783114168, 0.000513577279189,
           0.000513371250884, 0.000513165029338,
           0.000512958614624, 0.000512752006823,
           0.000512545206008, 0.000512338212262,
           0.000512131025659, 0.000511923646281,
           0.000511716074203, 0.000511508309501,
           0.000511300352259, 0.000511092202552,
           0.000510883860458, 0.000510675326056,
           0.000510466599422, 0.000510257680642,
           0.000510048569786, 0.000509839266939,
           0.000509629772175, 0.000509420085579,
           0.000509210207223, 0.000509000137188,
           0.000508789875557, 0.000508579422404,
           0.000508368777814, 0.000508157941859,
           0.000507946914625, 0.000507735696187,
           0.000507524286625, 0.000507312686022,
           0.000507100894455, 0.000506888912003,
           0.000506676738746, 0.000506464374768,
           0.000506251820141, 0.000506039074956,
           0.000505826139282, 0.000505613013206,
           0.000505399696804, 0.000505186190159,
           0.000504972493353, 0.000504758606461,
           0.000504544529569, 0.000504330262755,
           0.000504115806100, 0.000503901159684,
           0.000503686323591, 0.000503471297896,
           0.000503256082685, 0.000503040678037,
           0.000502825084034, 0.000502609300755,
           0.000502393328285, 0.000502177166701,
           0.000501960816089, 0.000501744276526,
           0.000501527548096, 0.000501310630880,
           0.000501093524959, 0.000500876230416,
           0.000500658747333, 0.000500441075789,
           0.000500223215869, 0.000500005167654,
           0.000499786931228, 0.000499568506667,
           0.000499349894060, 0.000499131093486,
           0.000498912105030, 0.000498692928770,
           0.000498473564794, 0.000498254013178,
           0.000498034274010, 0.000497814347370,
           0.000497594233344, 0.000497373932009,
           0.000497153443453, 0.000496932767761,
           0.000496711905008, 0.000496490855285,
           0.000496269618670, 0.000496048195250,
           0.000495826585103, 0.000495604788319,
           0.000495382804976, 0.000495160635163,
           0.000494938278959, 0.000494715736448,
           0.000494493007719, 0.000494270092849,
           0.000494046991922, 0.000493823705032,
           0.000493600232252, 0.000493376573669,
           0.000493152729371, 0.000492928699438,
           0.000492704483954, 0.000492480083007,
           0.000492255496682, 0.000492030725057,
           0.000491805768222, 0.000491580626261,
           0.000491355299259, 0.000491129787299,
           0.000490904090467, 0.000490678208848,
           0.000490452142528, 0.000490225891590,
           0.000489999456121, 0.000489772836204,
           0.000489546031927, 0.000489319043374,
           0.000489091870630, 0.000488864513782,
           0.000488636972913, 0.000488409248111,
           0.000488181339461, 0.000487953247050,
           0.000487724970961, 0.000487496511282,
           0.000487267868099, 0.000487039041498,
           0.000486810031565, 0.000486580838385,
           0.000486351462045, 0.000486121902633,
           0.000485892160232, 0.000485662234934,
           0.000485432126817, 0.000485201835975,
           0.000484971362495, 0.000484740706459,
           0.000484509867955, 0.000484278847072,
           0.000484047643895, 0.000483816258514,
           0.000483584691009, 0.000483352941478,
           0.000483121009999, 0.000482888896662,
           0.000482656601558, 0.000482424124770,
           0.000482191466386, 0.000481958626495,
           0.000481725605185, 0.000481492402542,
           0.000481259018654, 0.000481025453612,
           0.000480791707498, 0.000480557780406,
           0.000480323672421, 0.000480089383630,
           0.000479854914125, 0.000479620263991,
           0.000479385433318, 0.000479150422193,
           0.000478915230708, 0.000478679858944,
           0.000478444306997, 0.000478208574955,
           0.000477972662899, 0.000477736570929,
           0.000477500299127, 0.000477263847582,
           0.000477027216384, 0.000476790405626,
           0.000476553415392, 0.000476316245772,
           0.000476078896857, 0.000475841368735,
           0.000475603661494, 0.000475365775228,
           0.000475127710023, 0.000474889465968,
           0.000474651043157, 0.000474412441674,
           0.000474173661615, 0.000473934703062,
           0.000473695566114, 0.000473456250855,
           0.000473216757376, 0.000472977085768,
           0.000472737236121, 0.000472497208526,
           0.000472257003074, 0.000472016619850,
           0.000471776058952, 0.000471535320466,
           0.000471294404483, 0.000471053311096,
           0.000470812040394, 0.000470570592467,
           0.000470328967407, 0.000470087165305,
           0.000469845186252, 0.000469603030337,
           0.000469360697656, 0.000469118188298,
           0.000468875502352, 0.000468632639910,
           0.000468389601064, 0.000468146385909,
           0.000467902994530, 0.000467659427025,
           0.000467415683479, 0.000467171763991,
           0.000466927668647, 0.000466683397542,
           0.000466438950767, 0.000466194328416,
           0.000465949530575, 0.000465704557344,
           0.000465459408811, 0.000465214085068,
           0.000464968586208, 0.000464722912324,
           0.000464477063508, 0.000464231039853,
           0.000463984841449, 0.000463738468397,
           0.000463491920776, 0.000463245198693,
           0.000462998302230, 0.000462751231485,
           0.000462503986553, 0.000462256567520,
           0.000462008974484, 0.000461761207541,
           0.000461513266780, 0.000461265152294,
           0.000461016864177, 0.000460768402525,
           0.000460519767427, 0.000460270958982,
           0.000460021977280, 0.000459772822417,
           0.000459523494482, 0.000459273993575,
           0.000459024319787, 0.000458774473211,
           0.000458524453942, 0.000458274262075,
           0.000458023897704, 0.000457773360922,
           0.000457522651825, 0.000457271770504,
           0.000457020717055, 0.000456769491580,
           0.000456518094162, 0.000456266524901,
           0.000456014783892, 0.000455762871228,
           0.000455510787005, 0.000455258531318,
           0.000455006104263, 0.000454753505933,
           0.000454500736423, 0.000454247795828,
           0.000453994684244, 0.000453741401770,
           0.000453487948494, 0.000453234324517,
           0.000452980529933, 0.000452726564837,
           0.000452472429324, 0.000452218123491,
           0.000451963647431, 0.000451709001248,
           0.000451454185025, 0.000451199198867,
           0.000450944042868, 0.000450688717124,
           0.000450433221729, 0.000450177556785,
           0.000449921722380, 0.000449665718617,
           0.000449409545588, 0.000449153203393,
           0.000448896692127, 0.000448640011887,
           0.000448383162768, 0.000448126144868,
           0.000447868958282, 0.000447611603111,
           0.000447354079447, 0.000447096387389,
           0.000446838527035, 0.000446580498483,
           0.000446322301824, 0.000446063937164,
           0.000445805404591, 0.000445546704212,
           0.000445287836117, 0.000445028800402,
           0.000444769597174, 0.000444510226524,
           0.000444250688548, 0.000443990983348,
           0.000443731111019, 0.000443471071663,
           0.000443210865370, 0.000442950492248,
           0.000442689952386, 0.000442429245887,
           0.000442168372847, 0.000441907333365,
           0.000441646127540, 0.000441384755472,
           0.000441123217254, 0.000440861512987,
           0.000440599642772, 0.000440337606706,
           0.000440075404885, 0.000439813037412,
           0.000439550504383, 0.000439287805895,
           0.000439024942052, 0.000438761912950,
           0.000438498718689, 0.000438235359369,
           0.000437971835083, 0.000437708145938,
           0.000437444292029, 0.000437180273459,
           0.000436916090320, 0.000436651742718,
           0.000436387230751, 0.000436122554520,
           0.000435857714120, 0.000435592709654,
           0.000435327541223, 0.000435062208925,
           0.000434796712861, 0.000434531053127,
           0.000434265229826, 0.000433999243063,
           0.000433733092926, 0.000433466779527,
           0.000433200302959, 0.000432933663327,
           0.000432666860729, 0.000432399895265,
           0.000432132767036, 0.000431865476144,
           0.000431598022686, 0.000431330406766,
           0.000431062628485, 0.000430794687943,
           0.000430526585237, 0.000430258320475,
           0.000429989893750, 0.000429721305172,
           0.000429452554835, 0.000429183642843,
           0.000428914569297, 0.000428645334298,
           0.000428375937948, 0.000428106380347,
           0.000427836661598, 0.000427566781803,
           0.000427296741061, 0.000427026539474,
           0.000426756177146, 0.000426485654178,
           0.000426214970670, 0.000425944126726,
           0.000425673122448, 0.000425401957938,
           0.000425130633295, 0.000424859148626,
           0.000424587504028, 0.000424315699609,
           0.000424043735465, 0.000423771611703,
           0.000423499328424, 0.000423226885732,
           0.000422954283727, 0.000422681522513,
           0.000422408602190, 0.000422135522867,
           0.000421862284640, 0.000421588887615,
           0.000421315331895, 0.000421041617584,
           0.000420767744782, 0.000420493713595,
           0.000420219524124, 0.000419945176475,
           0.000419670670746, 0.000419396007046,
           0.000419121185475, 0.000418846206140,
           0.000418571069140, 0.000418295774582,
           0.000418020322565, 0.000417744713201,
           0.000417468946586, 0.000417193022824,
           0.000416916942024, 0.000416640704288,
           0.000416364309718, 0.000416087758420,
           0.000415811050496, 0.000415534186053,
           0.000415257165195, 0.000414979988022,
           0.000414702654643, 0.000414425165162,
           0.000414147519680, 0.000413869718306,
           0.000413591761140, 0.000413313648290,
           0.000413035379860, 0.000412756955954,
           0.000412478376677, 0.000412199642135,
           0.000411920752433, 0.000411641707671,
           0.000411362507960, 0.000411083153405,
           0.000410803644106, 0.000410523980174,
           0.000410244161711, 0.000409964188825,
           0.000409684061615, 0.000409403780196,
           0.000409123344663, 0.000408842755131,
           0.000408562011699, 0.000408281114478,
           0.000408000063571, 0.000407718859083,
           0.000407437501118, 0.000407155989788,
           0.000406874325191, 0.000406592507445,
           0.000406310536643, 0.000406028412897,
           0.000405746136314, 0.000405463707002,
           0.000405181125060, 0.000404898390601,
           0.000404615503730, 0.000404332464552,
           0.000404049273173, 0.000403765929703,
           0.000403482434246, 0.000403198786909,
           0.000402914987799, 0.000402631037023,
           0.000402346934691, 0.000402062680901,
           0.000401778275770, 0.000401493719399,
           0.000401209011897, 0.000400924153373,
           0.000400639143930, 0.000400353983680,
           0.000400068672724, 0.000399783211177,
           0.000399497599142, 0.000399211836726,
           0.000398925924039, 0.000398639861187,
           0.000398353648280, 0.000398067285421,
           0.000397780772720, 0.000397494110291,
           0.000397207298233, 0.000396920336657,
           0.000396633225672, 0.000396345965388,
           0.000396058555907, 0.000395770997343,
           0.000395483289800, 0.000395195433390,
           0.000394907428219, 0.000394619274396,
           0.000394330972029, 0.000394042521231,
           0.000393753922103, 0.000393465174758,
           0.000393176279304, 0.000392887235852,
           0.000392598044505, 0.000392308705378,
           0.000392019218578, 0.000391729584210,
           0.000391439802390, 0.000391149873220,
           0.000390859796814, 0.000390569573283,
           0.000390279202728, 0.000389988685267,
           0.000389698021003, 0.000389407210050,
           0.000389116252514, 0.000388825148507,
           0.000388533898139, 0.000388242501517,
           0.000387950958751, 0.000387659269955,
           0.000387367435231, 0.000387075454696,
           0.000386783328458, 0.000386491056624,
           0.000386198639306, 0.000385906076619,
           0.000385613368664, 0.000385320515556,
           0.000385027517404, 0.000384734374323,
           0.000384441086417, 0.000384147653798,
           0.000383854076578, 0.000383560354869,
           0.000383266488778, 0.000382972478417,
           0.000382678323896, 0.000382384025327,
           0.000382089582822, 0.000381794996489,
           0.000381500266437, 0.000381205392786,
           0.000380910375637, 0.000380615215103,
           0.000380319911300, 0.000380024464336,
           0.000379728874323, 0.000379433141369,
           0.000379137265590, 0.000378841247096,
           0.000378545085998, 0.000378248782404,
           0.000377952336431, 0.000377655748190,
           0.000377359017790, 0.000377062145344,
           0.000376765130961, 0.000376467974757,
           0.000376170676842, 0.000375873237330,
           0.000375575656328, 0.000375277933955,
           0.000374980070316, 0.000374682065527,
           0.000374383919698, 0.000374085632946,
           0.000373787205377, 0.000373488637106,
           0.000373189928247, 0.000372891078913,
           0.000372592089211, 0.000372292959258,
           0.000371993689168, 0.000371694279051,
           0.000371394729018, 0.000371095039183,
           0.000370795209664, 0.000370495240567,
           0.000370195132008, 0.000369894884097,
           0.000369594496950, 0.000369293970684,
           0.000368993305403, 0.000368692501227,
           0.000368391558265, 0.000368090476637,
           0.000367789256446, 0.000367487897813,
           0.000367186400850, 0.000366884765669,
           0.000366582992385, 0.000366281081114,
           0.000365979031961, 0.000365676845052,
           0.000365374520488, 0.000365072058392,
           0.000364769458875, 0.000364466722051,
           0.000364163848032, 0.000363860836936,
           0.000363557688872, 0.000363254403959,
           0.000362950982310, 0.000362647424034,
           0.000362343729254, 0.000362039898079,
           0.000361735930623, 0.000361431827001,
           0.000361127587329, 0.000360823211720,
           0.000360518700291, 0.000360214053154,
           0.000359909270424, 0.000359604352217,
           0.000359299298646, 0.000358994109829,
           0.000358688785876, 0.000358383326905,
           0.000358077733033, 0.000357772004370,
           0.000357466141035, 0.000357160143141,
           0.000356854010805, 0.000356547744140,
           0.000356241343264, 0.000355934808292,
           0.000355628139336, 0.000355321336516,
           0.000355014399942, 0.000354707329736,
           0.000354400126009, 0.000354092788875,
           0.000353785318459, 0.000353477714866,
           0.000353169978218, 0.000352862108629,
           0.000352554106212, 0.000352245971089,
           0.000351937703372, 0.000351629303179,
           0.000351320770623, 0.000351012105823,
           0.000350703308896, 0.000350394379954,
           0.000350085319115, 0.000349776126499,
           0.000349466802220, 0.000349157346392,
           0.000348847759134, 0.000348538040560,
           0.000348228190793, 0.000347918209943,
           0.000347608098128, 0.000347297855466,
           0.000346987482076, 0.000346676978070,
           0.000346366343566, 0.000346055578685,
           0.000345744683539, 0.000345433658248,
           0.000345122502927, 0.000344811217696,
           0.000344499802669, 0.000344188257966,
           0.000343876583702, 0.000343564779997,
           0.000343252846963, 0.000342940784725,
           0.000342628593394, 0.000342316273093,
           0.000342003823933, 0.000341691246036,
           0.000341378539520, 0.000341065704502,
           0.000340752741100, 0.000340439649428,
           0.000340126429610, 0.000339813081760,
           0.000339499605996, 0.000339186002438,
           0.000338872271203, 0.000338558412408,
           0.000338244426172, 0.000337930312613,
           0.000337616071854, 0.000337301704004,
           0.000336987209187, 0.000336672587521,
           0.000336357839126, 0.000336042964116,
           0.000335727962614, 0.000335412834735,
           0.000335097580601, 0.000334782200325,
           0.000334466694033, 0.000334151061839,
           0.000333835303865, 0.000333519420228,
           0.000333203411044, 0.000332887276438,
           0.000332571016523, 0.000332254631427,
           0.000331938121255, 0.000331621486139,
           0.000331304726191, 0.000330987841536,
           0.000330670832287, 0.000330353698566,
           0.000330036440494, 0.000329719058188,
           0.000329401551772, 0.000329083921358,
           0.000328766167071, 0.000328448289028,
           0.000328130287351, 0.000327812162159,
           0.000327493913570, 0.000327175541706,
           0.000326857046687, 0.000326538428627,
           0.000326219687655, 0.000325900823889,
           0.000325581837440, 0.000325262728441,
           0.000324943497003, 0.000324624143249,
           0.000324304667300, 0.000323985069276,
           0.000323665349297, 0.000323345507483,
           0.000323025543952, 0.000322705458832,
           0.000322385252234, 0.000322064924286,
           0.000321744475103, 0.000321423904810,
           0.000321103213528, 0.000320782401374,
           0.000320461468469, 0.000320140414936,
           0.000319819240897, 0.000319497946468,
           0.000319176531774, 0.000318854996937,
           0.000318533342074, 0.000318211567309,
           0.000317889672759, 0.000317567658552,
           0.000317245524803, 0.000316923271640,
           0.000316600899176, 0.000316278407538,
           0.000315955796844, 0.000315633067223,
           0.000315310218785, 0.000314987251660,
           0.000314664165964, 0.000314340961825,
           0.000314017639358, 0.000313694198690,
           0.000313370639941, 0.000313046963231,
           0.000312723168684, 0.000312399256420,
           0.000312075226564, 0.000311751079234,
           0.000311426814554, 0.000311102432648,
           0.000310777933633, 0.000310453317639,
           0.000310128584779, 0.000309803735184,
           0.000309478768969, 0.000309153686260,
           0.000308828487180, 0.000308503171848,
           0.000308177740391, 0.000307852192927,
           0.000307526529581, 0.000307200750477,
           0.000306874855735, 0.000306548845477,
           0.000306222719831, 0.000305896478911,
           0.000305570122848, 0.000305243651762,
           0.000304917065774, 0.000304590365009,
           0.000304263549589, 0.000303936619640,
           0.000303609575280, 0.000303282416637,
           0.000302955143829, 0.000302627756982,
           0.000302300256223, 0.000301972641668,
           0.000301644913446, 0.000301317071677,
           0.000300989116486, 0.000300661047995,
           0.000300332866328, 0.000300004571612,
           0.000299676163964, 0.000299347643518,
           0.000299019010383, 0.000298690264695,
           0.000298361406571, 0.000298032436142,
           0.000297703353521, 0.000297374158843,
           0.000297044852223, 0.000296715433790,
           0.000296385903669, 0.000296056261978,
           0.000295726508848, 0.000295396644397,
           0.000295066668757, 0.000294736582043,
           0.000294406384387, 0.000294076075906,
           0.000293745656731, 0.000293415126984,
           0.000293084486787, 0.000292753736269,
           0.000292422875548, 0.000292091904758,
           0.000291760824015, 0.000291429633448,
           0.000291098333180, 0.000290766923335,
           0.000290435404041, 0.000290103775419,
           0.000289772037595, 0.000289440190697,
           0.000289108234848, 0.000288776170168,
           0.000288443996789, 0.000288111714834,
           0.000287779324426, 0.000287446825692,
           0.000287114218757, 0.000286781503745,
           0.000286448680782, 0.000286115749994,
           0.000285782711507, 0.000285449565444,
           0.000285116311932, 0.000284782951094,
           0.000284449483059, 0.000284115907950,
           0.000283782225896, 0.000283448437019,
           0.000283114541445, 0.000282780539300,
           0.000282446430710, 0.000282112215803,
           0.000281777894700, 0.000281443467533,
           0.000281108934422, 0.000280774295496,
           0.000280439550880, 0.000280104700701,
           0.000279769745085, 0.000279434684156,
           0.000279099518042, 0.000278764246870,
           0.000278428870762, 0.000278093389851,
           0.000277757804257, 0.000277422114112,
           0.000277086319538, 0.000276750420660,
           0.000276414417609, 0.000276078310509,
           0.000275742099487, 0.000275405784670,
           0.000275069366186, 0.000274732844156,
           0.000274396218712, 0.000274059489980,
           0.000273722658087, 0.000273385723158,
           0.000273048685319, 0.000272711544700,
           0.000272374301426, 0.000272036955623,
           0.000271699507421, 0.000271361956946,
           0.000271024304323, 0.000270686549679,
           0.000270348693145, 0.000270010734846,
           0.000269672674907, 0.000269334513458,
           0.000268996250629, 0.000268657886539,
           0.000268319421324, 0.000267980855103,
           0.000267642188013, 0.000267303420173,
           0.000266964551717, 0.000266625582769,
           0.000266286513456, 0.000265947343905,
           0.000265608074250, 0.000265268704614,
           0.000264929235121, 0.000264589665908,
           0.000264249997094, 0.000263910228810,
           0.000263570361187, 0.000263230394348,
           0.000262890328424, 0.000262550163543,
           0.000262209899832, 0.000261869537420,
           0.000261529076434, 0.000261188517002,
           0.000260847859254, 0.000260507103316,
           0.000260166249320, 0.000259825297388,
           0.000259484247654, 0.000259143100244,
           0.000258801855287, 0.000258460512913,
           0.000258119073247, 0.000257777536418,
           0.000257435902558, 0.000257094171792,
           0.000256752344253, 0.000256410420063,
           0.000256068399357, 0.000255726282260,
           0.000255384068902, 0.000255041759414,
           0.000254699353919, 0.000254356852552,
           0.000254014255439, 0.000253671562709,
           0.000253328774492, 0.000252985890916,
           0.000252642912110, 0.000252299838206,
           0.000251956669328, 0.000251613405611,
           0.000251270047179, 0.000250926594164,
           0.000250583046694, 0.000250239404898,
           0.000249895668911, 0.000249551838852,
           0.000249207914860, 0.000248863897060,
           0.000248519785581, 0.000248175580556,
           0.000247831282109, 0.000247486890375,
           0.000247142405482, 0.000246797827558,
           0.000246453156734, 0.000246108393139,
           0.000245763536904, 0.000245418588158,
           0.000245073547031, 0.000244728413654,
           0.000244383188155, 0.000244037870666,
           0.000243692461314, 0.000243346960230,
           0.000243001367548, 0.000242655683394,
           0.000242309907898, 0.000241964041189,
           0.000241618083403, 0.000241272034667,
           0.000240925895108, 0.000240579664862,
           0.000240233344054, 0.000239886932818,
           0.000239540431285, 0.000239193839581,
           0.000238847157840, 0.000238500386192,
           0.000238153524767, 0.000237806573698,
           0.000237459533110, 0.000237112403138,
           0.000236765183911, 0.000236417875562,
           0.000236070478221, 0.000235722992016,
           0.000235375417079, 0.000235027753543,
           0.000234680001536, 0.000234332161193,
           0.000233984232639, 0.000233636216010,
           0.000233288111435, 0.000232939919044,
           0.000232591638973, 0.000232243271345,
           0.000231894816297, 0.000231546273960,
           0.000231197644461, 0.000230848927938,
           0.000230500124515, 0.000230151234328,
           0.000229802257508, 0.000229453194184,
           0.000229104044489, 0.000228754808554,
           0.000228405486511, 0.000228056078493,
           0.000227706584625, 0.000227357005049,
           0.000227007339888, 0.000226657589275,
           0.000226307753344, 0.000225957832226,
           0.000225607826054, 0.000225257734957,
           0.000224907559069, 0.000224557298518,
           0.000224206953442, 0.000223856523968,
           0.000223506010228, 0.000223155412358,
           0.000222804730487, 0.000222453964745,
           0.000222103115270, 0.000221752182189,
           0.000221401165635, 0.000221050065742,
           0.000220698882639, 0.000220347616461,
           0.000219996267339, 0.000219644835406,
           0.000219293320794, 0.000218941723633,
           0.000218590044062, 0.000218238282206,
           0.000217886438200, 0.000217534512178,
           0.000217182504270, 0.000216830414612,
           0.000216478243332, 0.000216125990565,
           0.000215773656444, 0.000215421241102,
           0.000215068744667, 0.000214716167282,
           0.000214363509068, 0.000214010770164,
           0.000213657950699, 0.000213305050813,
           0.000212952070635, 0.000212599010291,
           0.000212245869925, 0.000211892649664,
           0.000211539349640, 0.000211185969993,
           0.000210832510845, 0.000210478972338,
           0.000210125354602, 0.000209771657770,
           0.000209417881974, 0.000209064027351,
           0.000208710094030, 0.000208356082146,
           0.000208001991831, 0.000207647823221,
           0.000207293576448, 0.000206939251645,
           0.000206584848946, 0.000206230368483,
           0.000205875810389, 0.000205521174804,
           0.000205166461851, 0.000204811671670,
           0.000204456804395, 0.000204101860157,
           0.000203746839088, 0.000203391741330,
           0.000203036567006, 0.000202681316258,
           0.000202325989213, 0.000201970586009,
           0.000201615106781, 0.000201259551659,
           0.000200903920777, 0.000200548214273,
           0.000200192432277, 0.000199836574922,
           0.000199480642347, 0.000199124634683,
           0.000198768552065, 0.000198412394623,
           0.000198056162496, 0.000197699855815,
           0.000197343474719, 0.000196987019335,
           0.000196630489801, 0.000196273886253,
           0.000195917208823, 0.000195560457644,
           0.000195203632851, 0.000194846734581,
           0.000194489762967, 0.000194132718140,
           0.000193775600237, 0.000193418409397,
           0.000193061145748, 0.000192703809424,
           0.000192346400566, 0.000191988919302,
           0.000191631365768, 0.000191273740105,
           0.000190916042435, 0.000190558272905,
           0.000190200431643, 0.000189842518786,
           0.000189484534468, 0.000189126478824,
           0.000188768351988, 0.000188410154097,
           0.000188051885281, 0.000187693545683,
           0.000187335135429, 0.000186976654659,
           0.000186618103507, 0.000186259482109,
           0.000185900790596, 0.000185542029108,
           0.000185183197778, 0.000184824296742,
           0.000184465326130, 0.000184106286085,
           0.000183747176737, 0.000183387998222,
           0.000183028750678, 0.000182669434235,
           0.000182310049032, 0.000181950595205,
           0.000181591072886, 0.000181231482213,
           0.000180871823322, 0.000180512096345,
           0.000180152301418, 0.000179792438679,
           0.000179432508264, 0.000179072510304,
           0.000178712444938, 0.000178352312303,
           0.000177992112529, 0.000177631845756,
           0.000177271512119, 0.000176911111751,
           0.000176550644794, 0.000176190111374,
           0.000175829511636, 0.000175468845711,
           0.000175108113736, 0.000174747315845,
           0.000174386452177, 0.000174025522866,
           0.000173664528048, 0.000173303467858,
           0.000172942342434, 0.000172581151908,
           0.000172219896421, 0.000171858576106,
           0.000171497191099, 0.000171135741539,
           0.000170774227557, 0.000170412649292,
           0.000170051006883, 0.000169689300459,
           0.000169327530163, 0.000168965696127,
           0.000168603798489, 0.000168241837385,
           0.000167879812951, 0.000167517725324,
           0.000167155574639, 0.000166793361032,
           0.000166431084641, 0.000166068745602,
           0.000165706344052, 0.000165343880125,
           0.000164981353960, 0.000164618765692,
           0.000164256115456, 0.000163893403394,
           0.000163530629638, 0.000163167794323,
           0.000162804897589, 0.000162441939575,
           0.000162078920409, 0.000161715840235,
           0.000161352699188, 0.000160989497407,
           0.000160626235021, 0.000160262912175,
           0.000159899529001, 0.000159536085637,
           0.000159172582221, 0.000158809018890,
           0.000158445395779, 0.000158081713025,
           0.000157717970767, 0.000157354169139,
           0.000156990308280, 0.000156626388325,
           0.000156262409416, 0.000155898371683,
           0.000155534275267, 0.000155170120305,
           0.000154805906933, 0.000154441635288,
           0.000154077305511, 0.000153712917734,
           0.000153348472095, 0.000152983968733,
           0.000152619407786, 0.000152254789387,
           0.000151890113677, 0.000151525380795,
           0.000151160590871, 0.000150795744050,
           0.000150430840464, 0.000150065880255,
           0.000149700863554, 0.000149335790506,
           0.000148970661245, 0.000148605475904,
           0.000148240234628, 0.000147874937550,
           0.000147509584810, 0.000147144176541,
           0.000146778712887, 0.000146413193982,
           0.000146047619964, 0.000145681990968,
           0.000145316307136, 0.000144950568603,
           0.000144584775508, 0.000144218927988,
           0.000143853026182, 0.000143487070224,
           0.000143121060256, 0.000142754996411,
           0.000142388878835, 0.000142022707657,
           0.000141656483019, 0.000141290205062,
           0.000140923873915, 0.000140557489724,
           0.000140191052622, 0.000139824562752,
           0.000139458020247, 0.000139091425247,
           0.000138724777893, 0.000138358078316,
           0.000137991326659, 0.000137624523058,
           0.000137257667656, 0.000136890760584,
           0.000136523801982, 0.000136156791993,
           0.000135789730748, 0.000135422618391,
           0.000135055455060, 0.000134688240889,
           0.000134320976017, 0.000133953660584,
           0.000133586294729, 0.000133218878590,
           0.000132851412304, 0.000132483896007,
           0.000132116329842, 0.000131748713948,
           0.000131381048458, 0.000131013333514,
           0.000130645569254, 0.000130277755816,
           0.000129909893338, 0.000129541981960,
           0.000129174021819, 0.000128806013054,
           0.000128437955804, 0.000128069850206,
           0.000127701696400, 0.000127333494527,
           0.000126965244718, 0.000126596947120,
           0.000126228601868, 0.000125860209100,
           0.000125491768955, 0.000125123281571,
           0.000124754747091, 0.000124386165651,
           0.000124017537386, 0.000123648862439,
           0.000123280140948, 0.000122911373052,
           0.000122542558889, 0.000122173698599,
           0.000121804792319, 0.000121435840193,
           0.000121066842351, 0.000120697798938,
           0.000120328710093, 0.000119959575952,
           0.000119590396656, 0.000119221172344,
           0.000118851903154, 0.000118482589226,
           0.000118113230700, 0.000117743827712,
           0.000117374380402, 0.000117004888911,
           0.000116635353377, 0.000116265773937,
           0.000115896150733, 0.000115526483905,
           0.000115156773585, 0.000114787019924,
           0.000114417223050, 0.000114047383110,
           0.000113677500237, 0.000113307574575,
           0.000112937606262, 0.000112567595436,
           0.000112197542237, 0.000111827446804,
           0.000111457309280, 0.000111087129796,
           0.000110716908498, 0.000110346645526,
           0.000109976341015, 0.000109605995106,
           0.000109235607941, 0.000108865179656,
           0.000108494710392, 0.000108124200287,
           0.000107753649485, 0.000107383058119,
           0.000107012426333, 0.000106641754265,
           0.000106271042053, 0.000105900289840,
           0.000105529497765, 0.000105158665965,
           0.000104787794580, 0.000104416883751,
           0.000104045933618, 0.000103674944320,
           0.000103303915996, 0.000102932848787,
           0.000102561742829, 0.000102190598268,
           0.000101819415239, 0.000101448193881,
           0.000101076934338, 0.000100705636747,
           0.000100334301245, 0.000099962927982,
           0.000099591517084, 0.000099220068700,
           0.000098848582967, 0.000098477060027,
           0.000098105500016, 0.000097733903076,
           0.000097362269349, 0.000096990598970,
           0.000096618892083, 0.000096247148826,
           0.000095875369340, 0.000095503553765,
           0.000095131702239, 0.000094759814904,
           0.000094387891900, 0.000094015933364,
           0.000093643939440, 0.000093271910266,
           0.000092899845982, 0.000092527746731,
           0.000092155612648, 0.000091783443877,
           0.000091411240555, 0.000091039002824,
           0.000090666730826, 0.000090294424697,
           0.000089922084580, 0.000089549710614,
           0.000089177302939, 0.000088804861697,
           0.000088432387026, 0.000088059879069,
           0.000087687337962, 0.000087314763848,
           0.000086942156867, 0.000086569517159,
           0.000086196844865, 0.000085824140123,
           0.000085451403077, 0.000085078633863,
           0.000084705832627, 0.000084332999504,
           0.000083960134634, 0.000083587238163,
           0.000083214310226, 0.000082841350965,
           0.000082468360521, 0.000082095339035,
           0.000081722286646, 0.000081349203494,
           0.000080976089722, 0.000080602945469,
           0.000080229770873, 0.000079856566078,
           0.000079483331223, 0.000079110066450,
           0.000078736771896, 0.000078363447706,
           0.000077990094017, 0.000077616710969,
           0.000077243298711, 0.000076869857370,
           0.000076496387096, 0.000076122888026,
           0.000075749360303, 0.000075375804066,
           0.000075002219457, 0.000074628606613,
           0.000074254965681, 0.000073881296794,
           0.000073507600096, 0.000073133875731,
           0.000072760123835, 0.000072386344552,
           0.000072012538020, 0.000071638704380,
           0.000071264843776, 0.000070890956344,
           0.000070517042229, 0.000070143101570,
           0.000069769134505, 0.000069395141180,
           0.000069021121732, 0.000068647076304,
           0.000068273005033, 0.000067898908067,
           0.000067524785538, 0.000067150637593,
           0.000066776464371, 0.000066402266014,
           0.000066028042659, 0.000065653794450,
           0.000065279521531, 0.000064905224036,
           0.000064530902111, 0.000064156555895,
           0.000063782185527, 0.000063407791151,
           0.000063033372908, 0.000062658930937,
           0.000062284465380, 0.000061909976377,
           0.000061535464070, 0.000061160928601,
           0.000060786370108, 0.000060411788735,
           0.000060037184621, 0.000059662557908,
           0.000059287908738, 0.000058913237249,
           0.000058538543583, 0.000058163827883,
           0.000057789090289, 0.000057414330942,
           0.000057039549982, 0.000056664747551,
           0.000056289923792, 0.000055915078842,
           0.000055540212847, 0.000055165325941,
           0.000054790418274, 0.000054415489981,
           0.000054040541206, 0.000053665572089,
           0.000053290582767, 0.000052915573390,
           0.000052540544091, 0.000052165495017,
           0.000051790426304, 0.000051415338100,
           0.000051040230539, 0.000050665103766,
           0.000050289957924, 0.000049914793148,
           0.000049539609585, 0.000049164407375,
           0.000048789186655, 0.000048413947575,
           0.000048038690265, 0.000047663414877,
           0.000047288121544, 0.000046912810412,
           0.000046537481622, 0.000046162135313,
           0.000045786771629, 0.000045411390709,
           0.000045035992694, 0.000044660577727,
           0.000044285145949, 0.000043909697502,
           0.000043534232526, 0.000043158751162,
           0.000042783253553, 0.000042407739839,
           0.000042032210161, 0.000041656664663,
           0.000041281103483, 0.000040905526766,
           0.000040529934649, 0.000040154327276,
           0.000039778704788, 0.000039403067327,
           0.000039027415033, 0.000038651748049,
           0.000038276066517, 0.000037900370575,
           0.000037524660367, 0.000037148936033,
           0.000036773197716, 0.000036397445558,
           0.000036021679697, 0.000035645900277,
           0.000035270107441, 0.000034894301328,
           0.000034518482078, 0.000034142649837,
           0.000033766804743, 0.000033390946938,
           0.000033015076565, 0.000032639193764,
           0.000032263298675, 0.000031887391443,
           0.000031511472208, 0.000031135541111,
           0.000030759598294, 0.000030383643899,
           0.000030007678067, 0.000029631700938,
           0.000029255712657, 0.000028879713362,
           0.000028503703196, 0.000028127682301,
           0.000027751650818, 0.000027375608890,
           0.000026999556655, 0.000026623494258,
           0.000026247421839, 0.000025871339539,
           0.000025495247502, 0.000025119145869,
           0.000024743034777, 0.000024366914375,
           0.000023990784799, 0.000023614646192,
           0.000023238498697, 0.000022862342454,
           0.000022486177604, 0.000022110004291,
           0.000021733822654, 0.000021357632838,
           0.000020981434982, 0.000020605229227,
           0.000020229015716, 0.000019852794592,
           0.000019476565993, 0.000019100330065,
           0.000018724086945, 0.000018347836777,
           0.000017971579702, 0.000017595315864,
           0.000017219045401, 0.000016842768458,
           0.000016466485174, 0.000016090195691,
           0.000015713900153, 0.000015337598700,
           0.000014961291472, 0.000014584978614,
           0.000014208660265, 0.000013832336567,
           0.000013456007666, 0.000013079673695,
           0.000012703334802, 0.000012326991130,
           0.000011950642817, 0.000011574290004,
           0.000011197932835, 0.000010821571451,
           0.000010445205995, 0.000010068836608,
           0.000009692463429, 0.000009316086605,
           0.000008939706273, 0.000008563322577,
           0.000008186935659, 0.000007810545660,
           0.000007434152723, 0.000007057756992,
           0.000006681358604, 0.000006304957706,
           0.000005928554440, 0.000005552148949,
           0.000005175741379, 0.000004799331873,
           0.000004422920581, 0.000004046507654,
           0.000003670093254, 0.000003293677548,
           0.000002917260736, 0.000002540843071,
           0.000002164424940, 0.000001788007101,
           0.000001411591494, 0.000001035185059,
           0.000000658826568, 0.000000283024289};

  } else {

    stop("N must be 5, 10, 20, 40, 80, 160, 320, 640, 1280, 2560 or 5120.");

  }

  // Add Jacobian to weights
  w_k *= 0.5 * (b - a);

  // Duplicate weights
  return arma::repmat(w_k, 2, 1);

}
