% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/label_switching.R
\name{check_labels}
\alias{check_labels}
\title{Check for label switching}
\usage{
check_labels(fit, burnin = floor(fit$L/10), lagnum = floor(0.05 * (fit$L -
  burnin) + 1), showmessages = TRUE)
}
\arguments{
\item{fit}{Object of class \code{damcmc_res} or \code{bdmcmc_res}.}

\item{burnin}{Number of initial realizations to discard. By default, it is 1/10 of the total number of iterations.}

\item{lagnum}{Number of past realizations to use for the detection algorithm. Default is .05 of the total realizations after burnin realizations are removed.}

\item{showmessages}{Logical variable requesting to show informative messages (TRUE) or suppress them (FALSE). Default is TRUE.}
}
\value{
Logical value indicating if label switching was detected.
}
\description{
Checks if there is label switching present in the
posterior realizations using the chains for
mu. The algorithm is heuristic and works as follows;
for each chain of mu for a given
component, we look for sharp changes
in the generated values that lead to dramatically
different variability from the
variability observed in the past
history of the chain. The lag history is 5\% of the
total number of realizations, excluding the burnin realizations.

For examples see

\url{http://www.stat.missouri.edu/~amicheas/sppmix/sppmix_all_examples.html
#check_labels}
}
\details{
To avoid the label switching problem one can plot
the average of the intensities of the posterior realizations, i.e.,
the average of the surfaces instead of the surface of the posterior averages. However,
by doing so we lose the ability to perform mixture deconvolution,
namely, work with the posterior means of the ps, mus and sigmas of the
mixture intensity. In general, the average of posterior surfaces
for each realization of the ps, mus and sigmas, and the surface based on
the posterior means of the ps, mus and sigmas, need not be the same.

For a DAMCMC fit, avoiding label switching can be accomplished using
function \code{\link{plot_avgsurf}}, which plots the average posterior surface.
The surface of posterior means is plotted using
the function \code{\link{plot.damcmc_res}} on the returned value of \code{\link{GetPMEst}}.

When working with a BDMCMC fit, it is recommended that you use \code{\link{GetBDCompfit}} to retrieve
the realizations corresponding to a specific number of components and then fix the labels.
Of course the best estimate in this case, is the Bayesian model average intensity
obtained via \code{\link{GetBMA}}, but it can be very slow to compute. The BMA
is an average on surfaces based on each of the posterior realizations, and as such
does not suffer from the label switching problem.
}
\examples{

\dontrun{
# generate data
mix1 = normmix(ps=c(.4, .2,.4), mus=list(c(0.3, 0.3), c(.5,.5),c(0.7, 0.7)),
 sigmas = list(.02*diag(2),.05*diag(2), .02*diag(2)),lambda = 100,
 win = spatstat::square(1))
plot(mix1,main="True Poisson intensity surface (mixture of normal components)")
pp1 = rsppmix(mix1)
# Run Data augmentation MCMC and get posterior realizations
postfit = est_mix_damcmc(pp1,m=5)
#plot the chains
plot_chains(postfit)
#check labels
check_labels(postfit)
#plot the average posterior surface
plot(GetPMEst(postfit))
#plot the surface of posterior means, can be slow for large LL
avgsurf=plot_avgsurf(postfit, LL = 50, burnin = 1000)
# Fix label switching, start with approx=TRUE
post_fixed = FixLS_da(postfit, plot_result = TRUE)
plot_chains(post_fixed)
plot_chains(post_fixed,separate = FALSE)
#this one works better in theory
post_fixed = FixLS_da(postfit,approx=FALSE, plot_result = TRUE)
plot_chains(post_fixed)
plot_chains(post_fixed,separate = FALSE)}

}
\author{
Jiaxun Chen, Sakis Micheas
}
\references{
Jasra, A., Holmes, C.C. and Stephens, D. A. (2005). Markov Chain Monte Carlo Methods and the Label Switching Problem in Bayesian Mixtures. Statistical Science, 20, 50-67.
}
\seealso{
\code{\link{normmix}},
\code{\link{rsppmix}},
\code{\link{est_mix_damcmc}},
\code{\link{plot_chains}},
\code{\link{FixLS_da}}
}

