% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/markedIPPP_mix.R
\name{rMIPPP_cond_loc}
\alias{rMIPPP_cond_loc}
\title{Generate a Marked Poisson point process (conditional on location)}
\usage{
rMIPPP_cond_loc(surf, locPP, gammas, r, hyper = 0.01, truncate = FALSE,
  win = owin(c(-3, 3), c(-3, 3)), bigwin, discrete_mark = TRUE,
  open_new_window = FALSE, grayscale = FALSE, show_plots = TRUE,
  LL = 128, L = 50000, mark_distr_choice = 0, GRFmu = 0, df = 10,
  nu = 0.5, theta = 1, sig = 1)
}
\arguments{
\item{surf}{An object of type \code{intensity_surface} representing the
IPPP surface for the ground process. Omit this argument to create a surface randomly.}

\item{locPP}{The ground IPPP (locations of the events). If missing then these
are generated using a call to \code{\link{rsppmix}}. Note that if
\code{surf} is not supplied, then it will be generated which may lead to
completely inappropriate locations of the events, if the supplied \code{locPP}
was created with a completely different surface. It is safer to supply both the surface
and ground locations at the same time or none of the two, so that both will be generated.}

\item{gammas}{For discrete marks (\code{discrete_mark=TRUE}), this is
a vector of length equal to the number of marks.
These parameters should typically be non-negative and
they represent weights affecting the probability
fields of each mark. For values close to 0, we
get higher probabilities of observing this mark.
Large positive values lead to small probabilities of observing
the corresponding mark. Negative values are allowed, but they can lead
to a mark not being present in the generated pattern.
If the vector \code{gammas} is not supplied,
then we randomly generate the number of marks from \code{1:10}
and the values of the vector \code{gammas}
from a gamma distribution.}

\item{r}{Radius used to define the
neighborhood system. Any two locations
within this distance are considered
neighbors. If missing, we randomly select
the radius using the generated (ground) point
pattern over the window parameter \code{win}.}

\item{hyper}{Hyperparameter for the distribution of gamma.}

\item{truncate}{Logical variable indicating whether or not we
normalize the densities of the mixture components
to have all their mass within the window defined
in the window \code{win}. This affects the mixture model for the
intensity surface of the ground process.}

\item{win}{Object of type \code{\link[spatstat]{owin}} defining the window of observation.}

\item{bigwin}{Object of type \code{\link[spatstat]{owin}}. If supplied, this will be the
window of observation, even if the pattern is generated over \code{win}. Useful if we
do not truncate (\code{truncate=FALSE}) and we want better presentation of the generated MIPPP.}

\item{discrete_mark}{Logical flag indicating whether the mark is discrete or not.
Default is TRUE. For continuous marks set this to FALSE.}

\item{open_new_window}{Open a new window for a plot.}

\item{grayscale}{Logical to request plots in grayscale.}

\item{show_plots}{Logical variable requesting to produce exploratory plots of the
Marked IPPP intensity surface and generated point pattern.}

\item{LL}{Length of the side of the square grid.
The larger this value is, the better the picture resolution.}

\item{L}{Number of iterations. Required when sampling from
the mark model conditional on locations.}

\item{mark_distr_choice}{A number indicating which
mark distribution to use. Currently we have
only one choice in the discrete mark case, which is essentialy a Markov random field (MRF)
over the window. See details for more on the mark model currently used. For continuous marks,
we have two choices, Gaussian random field (GRF) for
\code{mark_distr_choice=0} or Chi-Square random field for
\code{mark_distr_choice=1}.}

\item{GRFmu}{This is the mean of the
Gaussian random field. Only stationarity
is currently supported (i.e., \code{GRFmu} does
not depend on location). Used only if
\code{discrete_mark=FALSE}.}

\item{df}{Degrees of freedom (an integer) for the
chi-square random field when \code{mark_distr_choice=1}. Default is \code{df=10}. Used only if
\code{discrete_mark=FALSE}.}

\item{nu, theta, sig}{Additional arguments passed to the
\code{\link{MaternCov}} function in order to
create the spatial covariance field.
Default values are \code{nu=.5},
\code{theta=1}, and \code{sig=1}.
See \code{\link{MaternCov}} for details. Used only if
\code{discrete_mark=FALSE}.}
}
\value{
A list containing the following components:
\item{surf}{The generated or supplied intensity surface object \code{surf} for the ground process.}
\item{gammas}{The generated or supplied parameters \code{gammas}. Returned only if \code{discrete_mark=TRUE}.}
\item{genMPP}{The generated point pattern as an object of class \code{\link[spatstat]{ppp}} and \code{sppmix}. The member \code{$marks} contains the marks at each of the generated locations. If the ground PP \code{locPP} was supplied, this is also the ground process for the MIPPP and only the marks are generated (at those locations).}
\item{r}{The generated or supplied parameter \code{r}. Returned only if \code{discrete_mark=TRUE}.}
\item{prob_fields}{In the continuous mark case this is the realization of the random field (as an image \code{\link[spatstat]{im}} object). For discrete marks, this is a list of size equal to the number of marks containing the probability fields for each mark value.}
\item{prob_field_params}{A list of the parameters used to create the continuous valued mark fields.  Returned only if \code{discrete_mark=FALSE}.}
}
\description{
This function generates realizations (point patterns) from
a given Marked IPPP or a generated one. See details for the choice of models
for the mark distribution. The location (ground) process is
a standard IPPP (unmarked) with mixture intensity surface, and is responsible
for the number of events in the point pattern.

For examples see

\url{http://www.stat.missouri.edu/~amicheas/sppmix/sppmix_all_examples.html
#rMIPPP_cond_loc}
}
\details{
We assume that the joint distribution of a
marked point pattern \code{N=[s,m(s)]} with \code{n}
events is of the form:

\code{p(N)=lambda^n*exp(-lambda)/(n!)*f(all s|theta1)*g(all m|theta2(s),all s)}

where \code{s} denotes a location and \code{m=m(s)}
a mark value at that location, lambda a parameter
with the interpretation as the average number of points
over the window of observation, and \code{f}, \code{g} are proper densities.

In order to simulate from this Marked IPPP
we first simulate the number of events
and their locations from an IPPP with
mixture intensity surface \code{lambda*f(s|theta1)} (e.g.,
using \code{\link{rsppmix}}), and then generate
the mark at that location \code{s}.

In the discrete mark case, the mark is modeled using
a mixture distribution of Dirac measures on
the marks with the probability \code{q(m,s)} of observing a
specific mark value \code{m} depending on the current location
\code{s} and the marks of its neighbors. Since
we have a window of observation, any point in there
can potentially be marked, which leads to \code{q(m,s)} being
a field. In particular, the probability \code{q(m,s)} is analogous to

\code{exp(-gammas_(j)*(sum over all neighbors of s of their marks minus m squared))}

and when we fit the MIPPP model, our goal
is to estimate the parameters \code{gammas}.

Note that if all \code{gammas} are zero then
we fall back to a discrete uniform mark distribution.

The neighborhood system is controlled by
\code{r} and is crucial in this case. Small values
tend to produce probability fields with concentrated
masses about observed events of the process,
whereas, large neighborhoods allow us to borrow
strength across locations and result in much smoother
probability fields.

In the continuous case the mark is generated from
a (typically stationary) Gaussian process or chi-squared random process,
e.g., using function \code{\link{rGRF}}.

See Micheas (2014) for more details on
Marked IPPP models via conditioning arguments.
}
\examples{

\dontrun{
# Create a marked point pattern; use randomization and discrete marks (default values)
newMPP=rMIPPP_cond_loc()
plot(newMPP$surf,main="True IPPP intensity surface for the locations")
newMPP$gammas
newMPP$genMPP
newMPP$r
print(table(newMPP$genMPP$marks))
#we can reproduce the random field plots anytime using the following call
plot_MPP_fields(newMPP$genMPP,newMPP$gammas,newMPP$r)
#Now generate continuous marks according to a Gaussian process
newMPP=rMIPPP_cond_loc(discrete_mark = FALSE)
plot(newMPP$surf,main="True IPPP intensity surface for the locations")
#now the marks are taken from a chi-square field
newMPP=rMIPPP_cond_loc(mark_distr_choice=1, discrete_mark = FALSE)
plot(newMPP$surf,main="True IPPP intensity surface for the locations")}

}
\author{
Sakis Micheas
}
\references{
Hierarchical Bayesian Modeling of Marked Non-Homogeneous Poisson Processes with finite mixtures and inclusion of covariate information. Micheas, A.C. (2014). Journal of Applied Statistics, 41, 12, 2596-2615, DOI: 10.1080/02664763.2014.922167.
}
\seealso{
\code{\link{plot_MPP_fields}}
}

