#' @title sppmix: Poisson point process modeling using normal mixture models for the intensity surface
#' @aliases sppmix-package
#' @description
#' This help page contains a summary of the features of
#' the R package \code{sppmix}.
#'
#' The main page for the package is
#' at
#'
#' \url{http://faculty.missouri.edu/~micheasa/sppmix/index.html}
#'
#' including the package vignettes and R code with examples for each
#' function of the package.
#'
#' @details
#' The \code{sppmix} package implements classes and methods
#' for modeling spatial point process data using
#' Poisson point processes, where the intensity
#' surface is assumed to be a multiple of a finite
#' additive mixture with normal components.
#'
#' Comprehensive accounts on modeling spatial point processes can be found in the books by
#' Illian et al. (2008), Gelfand et al. (2010), Diggle (2014), and
#' Baddeley, Rubak and Turner (2015).
#'
#' The idea of using mixtures of normals for the intensity
#' function is not entirely new (e.g., Thomas processes or Poisson cluster processes). However, the Bayesian framework (DAMCMC) for general mixtures
#' for a fixed number of components were recently presented by Chakraborty and Gelfand (2010),
#' and for a fixed or random number of components in the context of
#' marked point processes, by Micheas (2014) (both DAMCMC and BDMCMC). In Zhou et al. (2015), the authors
#' entertained space-time models based on Poisson point processes with mixture
#' intensity surfaces.
#'
#' The addition of marks and the introduction of time to a point pattern,
#' leads to general Marked Space-Time Poisson point processes, and
#' such models will be investigated in
#' future versions of the \code{sppmix} package.
#'
#' @section Getting Started:
#' We recommend that you run the basic demos
#' in order to get a first taste of the capabilities
#' of \code{sppmix}. Simply run
#'
#' \code{Demo_sppmix(#)}
#'
#' in order to view the available demos and
#' tutorials (vignettes), by choosing an appropriate
#' number #. A call \code{Demo_sppmix()} will bring up
#' a browser with all the available vignetters of the package.
#' The following demos are available:
#'
#' 1) Defining, generating, working with and basic plotting of sppmix objects.
#'
#' 2) Plotting in the sppmix package.
#'
#' 3-6) How to perform model fitting for a Poisson point process with intensity surface assumed to be a mixture of normals. There
#' are 4 cases discussed: fixed or random number of components and edge effects are considered or not considered.
#'
#' 7) Model checking in the sppmix package.
#'
#' 8) Perform model fitting for a (discrete) marked
#' Poisson point process with ground
#' (locations process) intensity surface
#' assumed to be a mixture of normals.
#'
#' @section Bayesian computational methods:
#' Data Augmentation MCMC (DAMCMC by Diebolt and Robert, 1994,
#' function \code{\link{est_mix_damcmc}})
#' and Birth-Death MCMC (BDMCMC by Stephens, 2000, function \code{\link{est_mix_bdmcmc}})
#' are the two main MCMC methods we have implemented
#' for estimating the parameters of the Poisson intensity surface, in
#' a Bayesian framework. The intensity surface consists
#' of a parameter \code{lambda}, interpreted as the
#' average number of points over the window of observation
#' and the mixture component parameters, including
#' the component probabilities \code{ps}, the component means \code{mus}, and
#' the component covariances \code{sigmas}. See the details
#' section of the function \code{\link{rsppmix}}. The
#' latter function is used for sampling a point pattern from the Poisson point process.
#'
#' @section Objects in \code{sppmix}:
#' We introduce an object of class \code{\link{normmix}} for
#' handling 2d mixtures of bivariate normal components. This
#' helps us build the Poisson point process
#' intensity surface as an object of class \code{intensity_surface}.
#'
#' The DAMCMC and BDMCMC functions, return objects \code{damcmc_res} and
#' \code{bdmcmc_res}, respectively, representing the fitted model
#' parameters, as well as, a plethora of information. The functions
#' \code{plot} and \code{summary} can then be applied to these objects
#' to obtain additional information. See
#' \code{\link{plot.damcmc_res}},
#' \code{\link{plot.bdmcmc_res}},
#' \code{\link{summary.damcmc_res}}, and
#' \code{\link{summary.bdmcmc_res}}, for more details.
#'
#' @section External dependencies:
#' We link and require several R packages, including
#' \code{\link[spatstat]{spatstat}}, \code{\link[rgl]{rgl}},
#' \code{ggplot2},
#' \code{\link[Rcpp]{Rcpp}}, \code{\link[RcppArmadillo]{RcppArmadillo}}, \code{\link[fields]{fields}},
#' and \code{mvtnorm}.
#'
#' In particular, we utilize the \code{\link[spatstat]{ppp}} and \code{\link[spatstat]{owin}} classes
#' from the \code{\link[spatstat]{spatstat}} package in order to
#' describe a point pattern.
#'
#' The MCMC algorithms are implemented in C++ using
#' \code{\link[Rcpp]{Rcpp}} and \code{\link[RcppArmadillo]{RcppArmadillo}}, and were
#' optimized after extensive testing, meaning that this approach
#' is significantly faster than some other implementations of 2d mixture models.
#'
#' Plotting is accomplished using the \code{\link[rgl]{rgl}}
#' package in order to create 3d plots of the
#' intensity surfaces. In addition, the \code{\link[fields]{fields}}
#' and \code{ggplot2} packages were used for 2d plots.
#'
#' @section Use the help pages and please report errors!:
#' There are many examples in the help pages of
#' each and every function in the \code{sppmix}
#' package. You will find all the answers you need there,
#' in the demos and in the vignettes.
#'
#' But just in case we missed something, or if
#' you find a typo, or a mistake or have any other suggestions
#' feel free to contact the package maintainer Sakis Micheas.
#'
#' @section License Information:
#' All code in this package is copyright Sakis Micheas and Jiaxun Chen
#' and is released under the MIT license (\url{https://cran.r-project.org/web/licenses/MIT}).
#'
#' Furthermore, a lot of our examples utilize the \code{\link{PlotUSAStates}} function
#' which requires the Cartographic Boundary Shapefiles (boundary data) provided by the
#' USA Census Bureau at \url{https://www.census.gov/geo/maps-data/data/tiger-cart-boundary.html}.
#' To our knowledge there is no special license required to use this data.
#'
#' If you use the \code{sppmix} package in your research, we would appreciate a citation.
#'
#' @section Citation:
#' In order to cite the \code{sppmix} package in publications
#' please use:
#'
#' Micheas, A., and Chen, J. (2017): sppmix-Modeling spatial
#' Poisson and related point processes using normal mixture
#' models. R package. Package website: \url{http://faculty.missouri.edu/~micheasa/sppmix/index.html}.
#'
#' @references
#'
#' Diebolt, J., and Robert, C. P. (1994). Estimation of
#' Finite Mixture Distributions through Bayesian Sampling.
#' Journal of the Royal Statistical Society B, 56, 2, 363-375.
#'
#' Stephens, M. (2000). Bayesian analysis of mixture models with an unknown
#' number of components-an alternative to reversible jump methods.
#' The Annals of Statistics, 28, 1, 40-74.
#'
#' McLachlan, G., and Peel, D. (2000). Finite Mixture Models. Wiley-Interscience.
#'
#' Jasra, A., Holmes, C.C. and Stephens, D. A. (2005). Markov Chain Monte Carlo Methods and the Label Switching Problem in Bayesian Mixtures. Statistical Science, 20, 50-67.
#'
#' Illian, J., Penttinen, A., Stoyan, H. and Stoyan, D. (2008) Statistical Analysis and Modelling of Spatial Point Patterns. Wiley.
#'
#' Chakraborty, A., and Gelfand, A.E. (2010). Measurement error in spatial point patterns. Bayesian Analysis, 5, 97-122.
#'
#' Gelfand, A.E., Diggle, P.J., Fuentes, M. and Guttorp, P., editors (2010) Handbook of Spatial Statistics. CRC Press.
#'
#' Diggle, P.J. (2013) Statistical Analysis of Spatial and Spatio-Temporal Point Patterns, Third edition. Chapman and Hall/CRC.
#'
#' Micheas, A. C. (2014). Hierarchical Bayesian Modeling of Marked Non-Homogeneous Poisson Processes with finite mixtures and inclusion of covariate information. Journal of Applied Statistics, 41, 12, 2596-2615.
#'
#' Zhou, Z., Matteson, D.S., Woodard, D.B., Henderson, S.G., and Micheas, A.C. (2015). A Spatio-Temporal Point Process Model for Ambulance Demand. Journal of the American Statistical Association, 110, 509, 6-15.
#'
#' Baddeley, A., Rubak, E. and Turner, R. (2015) Spatial Point Patterns: Methodology and Applications with R. Chapman and Hall/CRC Press.
#'
#' @author
#'
#' Jiaxun Chen (Author and Creator)
#' \email{chenjiaxun9@hotmail.com}.
#'
#' Athanasios (Sakis) Christou Micheas (Author and Maintainer)
#' \email{micheasa@missouri.edu}
#'
#' Significant contributions on the package skeleton creation, plotting functions and other code by
#' Yuchen Wang \email{ycwang0712@gmail.com}
#'
#'
#' @docType package
#' @name sppmix
#' @useDynLib sppmix
#' @importFrom Rcpp sourceCpp
NULL

.onAttach <- function(libname, pkgname) {
  packageStartupMessage("For help topics type ?sppmix.\n",
                        "To see the available tutorials run\n",
                        "Demo_sppmix()")
}


.onUnload <- function(libpath) {
  library.dynam.unload("sppmix", libpath)
}
