cont.cdftest <- function(sites=NULL, subpop=NULL, design=NULL, data.cont=NULL,
   popsize=NULL, popcorrect=FALSE, pcfsize=NULL, N.cluster=NULL,
   stage1size=NULL, sizeweight=FALSE, vartype="Local", testname="Wald_F",
   nclass=3, spsurvey.obj=NULL) {

################################################################################
# Function: cont.cdftest
# Purpose: Cumulative Distribution Function Inference for Probability Survey Data
# Programmer: Tom Kincaid
# Date: November 15, 2007
# Last Revised: June 13, 2008
# Description:
#   This function organizes input and output for conducting inference regarding
#   cumulative distribution functions (CDFs) generated by a probability survey.
#   Input can be either an object of class spsurvey.analysis (see the
#   documentation for function spsurvey.analysis) or through use of the other
#   arguments to this function.  For every response variable and every
#   population Type, differences between CDFs are tested for every pair of
#   subpopulations within a Type.  The inferential procedures divide the CDFs
#   into a discrete set of intervals (classes) and then utilize procedures that
#   have been developed for analysis of categorical data from probability
#   surveys.  Choices for inference are the Wald, Rao-Scott first order
#   corrected (mean eigenvalue corrected), and Rao-Scott second order corrected
#   (Satterthwaite corrected) test statistics. Both standard versions of the
#   three statistics, which are distributed as Chi-squared random variables, and
#   alternate version of the statistics, which are distributed as F random
#   variables, are available.  The default test statistic is the F distribution
#   version of the Wald statistic.
# Arguments:
#   sites = a data frame consisting of two variables: the first variable is site
#     IDs, and the second variable is a logical vector indicating which sites to
#     use in the analysis.  If spsurvey.obj is not provided, then this argument
#     is required.  The default is NULL.
#   subpop = a data frame describing sets of populations and subpopulations for
#     which estimates will be calculated.  The first variable is site IDs.  Each
#     subsequent variable identifies a Type of population, where the variable
#     name is used to identify Type.  A Type variable identifies each site with
#     one of the subpopulations of that Type.  If spsurvey.obj is not provided,
#     then this argument is required.  The default is NULL.
#   design = a data frame consisting of design variables.  If spsurvey.obj is not
#     provided, then this argument is required.  The default is NULL.  Variables
#     should be named as follows:
#       siteID = site IDs
#       wgt = final adjusted weights, which are either the weights for a
#         single-stage sample or the stage two weights for a two-stage sample
#       xcoord = x-coordinates for location, which are either the x-coordinates
#         for a single-stage sample or the stage two x-coordinates for a two-
#         stage sample
#       ycoord = y-coordinates for location, which are either the y-coordinates
#         for a single-stage sample or the stage two y-coordinates for a two-
#         stage sample
#       stratum = the stratum codes
#       cluster = the stage one sampling unit (primary sampling unit or cluster)
#         codes
#       wgt1 = final adjusted stage one weights
#       xcoord1 = the stage one x-coordinates for location
#       ycoord1 = the stage one y-coordinates for location
#       support = support values - the value one (1) for a site from a finite
#         resource or the measure of the sampling unit associated with a site
#         from an extensive resource, which is required for calculation of
#         finite and continuous population correction factors.
#       swgt = size-weights, which is the stage two size-weight for a two-stage
#         sample.
#       swgt1 = stage one size-weights.
#   data.cont = a data frame of continuous response variables.  The first 
#     variable is site IDs.  Subsequent variables are response variables.
#     Missing data (NA) is allowed.  If spsurvey.obj is not provided, then
#     this argument is required.  The default is NULL.
#   popsize = known size of the resource, which is used to perform ratio
#     adjustment to estimators expressed using measurement units for the
#     resource.  For a finite resource, this argument is either the total number
#     of sampling units or the known sum of size-weights.  For an extensive
#     resource, this argument is the measure of the resource, i.e., either known
#     total length for a linear resource or known total area for an areal
#     resource.  The argument must be in the form of a list containing an
#     element for each population Type in the subpop data frame, where NULL is a
#     valid choice for a population Type.  The list must be named using the
#     column names for the population Types in subpop. If a population Type
#     doesn't contain subpopulations, then each element of the list is either a
#     single value for an unstratified sample or a vector containing a value for
#     each stratum for a stratified sample, where elements of the vector are
#     named using the stratum codes.  If a population Type contains
#     subpopulations, then each element of the list is a list containing an
#     element for each subpopulation, where the list is named using the
#     subpopulation names.  The element for each subpopulation will be either a
#     single value for an unstratified sample or a named vector of values for a
#     stratified sample.  The default is NULL.
#     Example popsize for a stratified sample:
#       popsize = list("Pop 1"=c("Stratum 1"=750,
#                                "Stratum 2"=500,
#                                "Stratum 3"=250),
#                      "Pop 2"=list("SubPop 1"=c("Stratum 1"=350,
#                                                "Stratum 2"=250,
#                                                "Stratum 3"=150),
#                                   "SubPop 2"=c("Stratum 1"=250,
#                                                "Stratum 2"=150,
#                                                "Stratum 3"=100),
#                                   "SubPop 3"=c("Stratum 1"=150,
#                                                "Stratum 2"=150,
#                                                "Stratum 3"=75)),
#                      "Pop 3"=NULL)
#     Example popsize for an unstratified sample:
#       popsize = list("Pop 1"=1500,
#                      "Pop 2"=list("SubPop 1"=750,
#                                   "SubPop 2"=500,
#                                   "SubPop 3"=375),
#                      "Pop 3"=NULL)
#   popcorrect = a logical value that indicates whether finite or continuous 
#     population correction factors should be employed during variance
#     estimation, where TRUE = use the correction factors and FALSE = do not use
#     the correction factors.  The default is FALSE.
#   pcfsize = size of the resource, which is required for calculation of finite
#     and continuous population correction factors for a single-stage sample.
#     For a stratified sample this argument must be a vector containing a value
#     for each stratum and must have the names attribute set to identify the
#     stratum codes.  The default is NULL.
#   N.cluster = the number of stage one sampling units in the resource, which is
#     required for calculation of finite and continuous population correction
#     factors for a two-stage sample.  For a stratified sample this argument
#     must be a vector containing a value for each stratum and must have the
#     names attribute set to identify the stratum codes.  The default is NULL.
#   stage1size = size of the stage one sampling units of a two-stage sample,
#     which is required for calculation of finite and continuous population
#     correction factors for a two-stage sample and must have the names
#     attribute set to identify the stage one sampling unit codes.  For a
#     stratified sample, the names attribute must be set to identify both
#     stratum codes and stage one sampling unit codes using a convention where
#     the two codes are separated by the & symbol, e.g., "Stratum 1&Cluster 1".
#     The default is NULL.
#   sizeweight = a logical value that indicates whether size-weights should 
#     be used in the analysis, where TRUE = use the size-weights and FALSE = do
#     not use the size-weights.  The default is FALSE.
#   vartype = the choice of variance estimator, where "Local" = local mean
#     estimator and "SRS" = SRS estimator.  The default is "Local".
#   testname = name of the test statistic to be reported in the output data
#     frame.  Choices for the name are: "Wald", "Wald_F", "Mean_Eigenvalue",
#     "Mean_Eigenvalue_F", "Satterthwaite", and "Satterthwaite_F".  The default
#     is "Wald_F".
#   nclass = number of classes for the CDF.  The default is 3.
#   spsurvey.obj = A list of class spsurvey.analysis that was produced by the
#     function spsurvey.analysis.  Depending on input to that function, some
#     elements of the list may be NULL.  The default is NULL.
# Results:
#   A data frame of CDF test results for all pairs of subpopulations within each
#   population Type for every response variable.  The data frame includes the
#   test statistic specified by argument testname plus its degrees of freedom
#   and p-value.
# Other Functions Required:
#   dframe.check - check site IDs, the sites data frame, the subpop data frame,
#     and the data.rr data frame to assure valid contents and, as necessary,
#   vecprint - takes an input vector and outputs a character string with line
#     breaks inserted
#   uniqueID - creates unique site IDs by appending a unique number to each
#     occurrence of a site ID
#   input.check - check input values for errors, consistency, and compatibility
#     with spsurvey.analysis analytical functions
#   cdf.test - test for differences between cumulative distribution functions
#     (CDFs)
# Examples:
#   mysiteID <- paste("Site", 1:100, sep="")
#   n <- 200
#   mysiteID <- paste("Site", 1:n, sep="")
#   mysites <- data.frame(siteID=mysiteID, Active=rep(TRUE, n))
#   mysubpop <- data.frame(siteID=mysiteID,  Resource_Class=sample(c("Agr",
#     "Forest", "Urban"), n, replace=TRUE))
#   mydesign <- data.frame(siteID=mysiteID, wgt=runif(n, 10, 100),
#     xcoord=runif(n), ycoord=runif(n), stratum=rep(c("Stratum1",
#     "Stratum2"), n/2))
#   mypopsize <- list(Resource_Class=list(Agr=c(Stratum1=2500, Stratum2=1500),
#     Forest=c(Stratum1=1000, Stratum2=500), Urban=c(Stratum1=600,
#     Stratum2=450)))
#   ContVar <- numeric(n)
#   tst <- mysubpop$Resource_Class == "Agr"
#   ContVar[tst] <- rnorm(sum(tst), 10, 1)
#   tst <- mysubpop$Resource_Class == "Forest"
#   ContVar[tst] <- rnorm(sum(tst), 10.1, 1)
#   tst <- mysubpop$Resource_Class == "Urban"
#   ContVar[tst] <- rnorm(sum(tst), 10.5, 1)
#   mydata.cont <- data.frame(siteID=mysiteID, ContVar=ContVar)
#   cont.cdftest(sites=mysites, subpop=mysubpop, design=mydesign,
#     data.cont=mydata.cont, popsize=mypopsize, testname="Mean_Eigenvalue")
################################################################################

# Create a data frame for warning messages

   warn.ind <- FALSE
   warn.df <- NULL
   fname <- "cont.cdftest"

# Check for a valid test statistic name

   temp <- match(testname, c("Wald", "Wald_F", "Mean_Eigenvalue",
      "Mean_Eigenvalue_F", "Satterthwaite", "Satterthwaite_F"), nomatch=0)
   if(temp == 0) {
      stop(paste("\nThe value provided for argument testname, \"", testname, "\", is not a valid test statistic name.\n", sep=""))
   }

# Begin the section when an object of class "spsurvey.analysis" was input to the 
# function

   if(inherits(spsurvey.obj, "spsurvey.analysis")) {

# Assign variables from the input list

      sites <- spsurvey.obj$sites
      subpop <- spsurvey.obj$subpop
      design <- spsurvey.obj$design
      data.cont <- spsurvey.obj$data.cont
      stratum.ind <- spsurvey.obj$stratum.ind
      cluster.ind <- spsurvey.obj$cluster.ind
      popsize <- spsurvey.obj$popsize
      pcfactor.ind <- spsurvey.obj$pcfactor.ind
      pcfsize <- spsurvey.obj$pcfsize
      N.cluster <- spsurvey.obj$N.cluster
      stage1size <- spsurvey.obj$stage1size
      swgt.ind <- spsurvey.obj$swgt.ind
      vartype <- spsurvey.obj$vartype

# Assign some required values from the subpop data frame

      ntypes <- dim(subpop)[2]
      typenames <- names(subpop)

# Begin the section when an object of class "spsurvey.analysis" was not input to
# the function

   } else {

# Check that the required data frames have been provided

      if(is.null(sites))
         stop("\nThe sites data frame must be provided.")
      if(!is.data.frame(sites))
         stop("\nThe sites argument must be a data frame.")
      if(is.null(subpop))
         stop("\nThe subpop data frame must be provided.")
      if(!is.data.frame(subpop))
         stop("\nThe subpop argument must be a data frame.")
      if(is.null(design))
         stop("\nThe design data frame must be provided.")
      if(!is.data.frame(design))
         stop("\nThe design argument must be a data frame.")
      if(is.null(data.cont))
         stop("\nThe data.cont data frame must be provided.")
      if(!is.data.frame(data.cont))
         stop("\nThe data.cont argument must be a data frame.")

# Check the design data frame for required names

      design.names <- names(design)
      temp <- match(design.names, c("siteID", "wgt", "xcoord", "ycoord",
         "stratum", "cluster", "wgt1", "xcoord1", "ycoord1", "support", "swgt",
         "swgt1"), nomatch=0)
      if(any(temp == 0)) {
         temp.str <- vecprint(design.names[temp == 0])
         stop(paste("\nThe following names used in the design data frame do not match the required names:\n", temp.str))
      }

# Check the sites data frame, the design data frame, the subpop data frame, and
# the data.cont data frame to assure valid contents

      temp <- dframe.check(sites, design, subpop, NULL, data.cont, design.names)
      design <- temp$design
      subpop <- temp$subpop
      data.cont <- temp$data.cont

# Assign variables from the design data frame

      siteID <- design$siteID
      wgt <- design$wgt
      xcoord <- design$xcoord
      ycoord <- design$ycoord
      stratum <- design$stratum
      cluster <- design$cluster
      wgt1 <- design$wgt1
      xcoord1 <- design$xcoord1
      ycoord1 <- design$ycoord1
      support <- design$support
      swgt <- design$swgt
      swgt1 <- design$swgt1

# Check site IDs for repeat values and, as necessary, create unique site IDs and
# output a warning message

      temp <- sapply(split(siteID, siteID), length)
      if(any(temp > 1)) {
         warn.ind <- TRUE
         temp.str <- vecprint(names(temp)[temp > 1])
         warn <- paste("The following site ID values occur more than once among the values that were \ninput to the function:\n", temp.str, sep="")
         act <- "Unique site ID values were created.\n"
         warn.df <- rbind(warn.df, data.frame(func=I(fname),
            subpoptype=NA, subpop=NA, indicator=NA, stratum=NA, warning=I(warn),
            action=I(act)))
         siteID <- uniqueID(siteID)
         subpop[,1] <- siteID
         data.cont[,1] <- siteID
      }

# Assign some required values from the subpop data frame

      ntypes <- dim(subpop)[2]
      typenames <- names(subpop)

# Assign a logical value to the indicator variable for a stratified sample

      stratum.ind <- length(unique(stratum)) > 1

# If the sample is stratified, convert stratum to a factor, determine stratum 
# levels, and calculate number of strata

      if(stratum.ind) {
         stratum <- factor(stratum)
         stratum.levels <- levels(stratum)
         nstrata <- length(stratum.levels)
      } else {
         stratum.levels <- NULL
         nstrata <- NULL
      }

# Assign a logical value to the indicator variable for a two stage sample

      cluster.ind <- length(unique(cluster)) > 1

# If the sample has two stages, convert cluster to a factor, determine cluster 
# levels, and calculate number of clusters

      if(cluster.ind) {
         if(stratum.ind) {
            cluster.in <- cluster
            cluster <- tapply(cluster, stratum, factor)
            cluster.levels <- sapply(cluster, levels, simplify=FALSE)
            ncluster <- sapply(cluster.levels, length)
         } else {
            cluster <- factor(cluster)
            cluster.levels <- levels(cluster)
            ncluster <- length(cluster.levels)
         }
      }

# Ensure that popsize is a list

      if(!is.null(popsize) && !is.list(popsize))
         stop("\nThe popsize argument must be a list")

# If the population correction factor is to be used, ensure that support values
# are provided

      if(popcorrect && is.null(support))
         stop("\nThe logical value that indicates whether finite or continuous population \ncorrection factors should be employed during variance estimation was set to \nTRUE, but support values were not provided in the design data frame.")

# Assign the value of popcorrect to the indicator variable for use of the
# population correction factor

      pcfactor.ind <- popcorrect

# If the sample uses size-weights, ensure that size weights are provided

      if(sizeweight) {
         if(is.null(swgt))
            stop("\nThe logical value that indicates whether size-weights should be employed in the analysis was set to \nTRUE, but size-weights were not provided in the design data frame.")
         if(cluster.ind && is.null(swgt1))
            stop("\nThe sample has two stages and the logical value that indicates whether size- \nweights should be employed in the analysis was set to TRUE, but stage one \nsize-weights were not provided in the design data frame.")
      }

# Assign the value of sizeweight to the indicator variable for use of size
# weights

      swgt.ind <- sizeweight

# Determine the number of response values

      nresp <- dim(design)[1]

# Check for compatibility of input values

      temp <- input.check(nresp, wgt, NULL, NULL, xcoord, ycoord, stratum.ind,
         stratum, stratum.levels, nstrata, cluster.ind, cluster, cluster.levels,
         ncluster, wgt1, xcoord1, ycoord1, popsize, pcfactor.ind, pcfsize,
         N.cluster, stage1size, support, swgt.ind, swgt, swgt1, vartype,
         conf=95, subpop=subpop)
      popsize <- temp$popsize
      pcfsize <- temp$pcfsize
      N.cluster <- temp$N.cluster
      stage1size <- temp$stage1size

# If the sample was stratified and had two stages, then reset cluster to its 
# input value

      if(stratum.ind && cluster.ind)
         cluster <- cluster.in

# If the sample has two stages, determine whether there are a sufficient number
# of sites in each stage one sampling unit to allow variance calculation

      if(cluster.ind) {
         temp.ind <- FALSE
         for(itype in 2:ntypes) {
            temp <- apply(table(cluster, subpop[,itype]) == 1, 2, sum)
            if(any(temp > 0)) {
               temp.ind <- TRUE
               temp.str <- vecprint(names(temp)[temp > 0])
               warn <- paste("A variance estimate cannot be calculated since the following subpopulation(s) of \npopulation ", typenames[itype], " include one or more stage one sampling units \nwith a single site:\n", temp.str, sep="")
               act <- "Program execution was terminated.\n"
               warn.df <- rbind(warn.df, data.frame(func=I(fname),
                  subpoptype=NA, subpop=NA, indicator=NA, stratum=NA,
                  warning=I(warn), action=I(act)))
            }
         }
         if(temp.ind) {
            stop("\nThe program was terminated due to the presence of one or more stage one sampling \nunits with a single site.  Enter the following command to view the warning \nmessages that were generated: warnprnt() \n")
         }
      }

# As necessary, assign missing values to the design variables

      if(is.null(xcoord))
         xcoord <- rep(NA, nresp)
      if(is.null(ycoord))
         ycoord <- rep(NA, nresp)
      if(is.null(stratum))
         stratum <- rep(NA, nresp)
      if(is.null(cluster))
         cluster <- rep(NA, nresp)
      if(is.null(wgt1))
         wgt1 <- rep(NA, nresp)
      if(is.null(xcoord1))
         xcoord1 <- rep(NA, nresp)
      if(is.null(ycoord1))
         ycoord1 <- rep(NA, nresp)
      if(is.null(support))
         support <- rep(NA, nresp)
      if(is.null(swgt))
         swgt <- rep(NA, nresp)
      if(is.null(swgt1))
         swgt1 <- rep(NA, nresp)

# Recreate the design data frame

      design <- data.frame(siteID=siteID, wgt=wgt, xcoord=xcoord, ycoord=ycoord,
         stratum=stratum, cluster=cluster, wgt1=wgt1, xcoord1=xcoord1,
         ycoord1=ycoord1, support=support, swgt=swgt, swgt1=swgt1)

# End the section when an object of class "spsurvey.analysis" was not input to
# the function

   }

# Loop through all response variables

   nvar <- dim(data.cont)[2]
   varnames <- names(data.cont)
   nrow <- 0 
   for(ivar in 2:nvar) {

# Determine the set of upper bounds for the response variable

      temp <- data.cont[!is.na(data.cont[,ivar]), ivar]
      bounds <- sort(temp)[floor(seq(length(temp)/nclass, length(temp),
                           length=nclass))]

# Loop through all types of populations

      for(itype in 2:ntypes) {

# Find unique subpopulations of this type of population

         subpopnames <- levels(factor(subpop[,itype]))

# Check whether the number of suspopulations is greater than one

         if(length(subpopnames) == 1) {
            warn.ind <- TRUE
            warn <- paste("Population type", typenames[itype], "contains a single subpopulation. \nNo CDF tests could be performed\n")
            act <- "None.\n"
            warn.df <- rbind(warn.df, data.frame(func=I(fname),
               subpoptype=I(typenames[itype]), subpop=NA,
               indicator=I(varnames[ivar]), stratum=NA, warning=I(warn),
               action=I(act)))
            next
         }

# Loop through all combinations of subpopulations of this type

# Begin the loop for subpopulation 1

         for(isubpop1 in 1:(length(subpopnames)-1)) {

# Select sites in the subpopulation

            subpop1.ind <- subpop[,itype] == subpopnames[isubpop1]

# Determine whether subpopulation 1 is empty

            if(all(is.na(data.cont[subpop1.ind, ivar]))) {
               warn.ind <- TRUE
               warn <- paste("Subpopulation", subpopnames[isubpop1], "of population type", typenames[itype], "for indicator", varnames[ivar], "\ncontains no data.\n")
               act <- "None.\n"
               warn.df <- rbind(warn.df, data.frame(func=I(fname),
                  subpoptype=I(typenames[itype]),
                  subpop=I(subpopnames[isubpop1]), indicator=I(varnames[ivar]),
                  stratum=NA,  warning=I(warn), action=I(act)))
               next
            }

# Determine whether subpopulation 1 contains a single value

            if(sum(!is.na(data.cont[subpop1.ind, ivar])) == 1) {
               warn.ind <- TRUE
               warn <- paste("Subpopulation", subpopnames[isubpop1], "of population type", typenames[itype], "for indicator", varnames[ivar], "\ncontains a single value.  No analysis was performed.\n")
               act <- "None.\n"
               warn.df <- rbind(warn.df, data.frame(func=I(fname),
                  subpoptype=I(typenames[itype]),
                  subpop=I(subpopnames[isubpop1]), indicator=I(varnames[ivar]),
                  stratum=NA,  warning=I(warn), action=I(act)))
               next
            }

# For a stratified sample, remove values from pcfsize, N.cluster, and stage1size
# for strata that do not occur in the subpopulation

            if(stratum.ind) {
               temp.pcfsize_1 <- pcfsize[!is.na(match(names(pcfsize),
                  unique(design[subpop1.ind, 5])))]
               temp.N.cluster_1 <- N.cluster[!is.na(match(names(N.cluster),
                  unique(design[subpop1.ind, 5])))]
               temp.stage1size_1 <- stage1size[!is.na(match(names(stage1size),
                  unique(design[subpop1.ind, 5])))]
            } else {
               temp.pcfsize_1 <- pcfsize
               temp.N.cluster_1 <- N.cluster
               temp.stage1size_1 <- stage1size
           }

# Select values from popsize

            if(is.list(popsize[[itype-1]]))
               temp.popsize_1 <- popsize[[itype-1]][[isubpop1]]
            else
               temp.popsize_1 <- popsize[[itype-1]]

# Begin the loop for subpopulation 2

            for(isubpop2 in (isubpop1+1):length(subpopnames)) {

# Select sites in the subpopulation

               subpop2.ind <- subpop[,itype] == subpopnames[isubpop2]

# Determine whether subpopulation 2 is empty

               if(all(is.na(data.cont[subpop1.ind, ivar]))) {
                  warn.ind <- TRUE
                  warn <- paste("Subpopulation", subpopnames[isubpop2], "of population type", typenames[itype], "for indicator", varnames[ivar], "\ncontains no data.\n")
                  act <- "None.\n"
                  warn.df <- rbind(warn.df, data.frame(func=I(fname),
                     subpoptype=I(typenames[itype]),
                     subpop=I(subpopnames[isubpop2]),
                     indicator=I(varnames[ivar]), stratum=NA,  warning=I(warn),
                     action=I(act)))
                  next
               }

# Determine whether subpopulation 2 contains a single value

               if(sum(!is.na(data.cont[subpop1.ind, ivar])) == 1) {
                  warn.ind <- TRUE
                  warn <- paste("Subpopulation", subpopnames[isubpop2], "of population type", typenames[itype], "for indicator", varnames[ivar], "\ncontains a single value.  No analysis was performed.\n")
                  act <- "None.\n"
                  warn.df <- rbind(warn.df, data.frame(func=I(fname),
                     subpoptype=I(typenames[itype]),
                     subpop=I(subpopnames[isubpop2]),
                     indicator=I(varnames[ivar]), stratum=NA,  warning=I(warn),
                     action=I(act)))
                  next
               }

# For a stratified sample, remove values from pcfsize, N.cluster, and stage1size
# for strata that do not occur in the subpopulation

            if(stratum.ind) {
               temp.pcfsize_2 <- pcfsize[!is.na(match(names(pcfsize),
                  unique(design[subpop2.ind, 5])))]
               temp.N.cluster_2 <- N.cluster[!is.na(match(names(N.cluster),
                  unique(design[subpop2.ind, 5])))]
               temp.stage1size_2 <- stage1size[!is.na(match(names(stage1size),
                  unique(design[subpop2.ind, 5])))]
            } else {
               temp.pcfsize_2 <- pcfsize
               temp.N.cluster_2 <- N.cluster
               temp.stage1size_2 <- stage1size
           }

# Select values from popsize

            if(is.list(popsize[[itype-1]]))
               temp.popsize_2 <- popsize[[itype-1]][[isubpop2]]
            else
               temp.popsize_2 <- popsize[[itype-1]]

# Perform the test for the response variable

               warn.vec <- c(typenames[itype], paste(subpopnames[isubpop1], ":", 
                  subpopnames[isubpop2], sep=""), varnames[ivar])

               temp <- cdf.test(bounds=bounds, z_1=data.cont[subpop1.ind,ivar],
                  wgt_1=design[subpop1.ind,2], x_1=design[subpop1.ind,3],
                  y_1=design[subpop1.ind,4], z_2=data.cont[subpop2.ind,ivar],
                  wgt_2=design[subpop2.ind,2], x_2=design[subpop2.ind,3],
                  y_2=design[subpop2.ind,4], stratum_1=design[subpop1.ind,5],
                  stratum_2=design[subpop2.ind,5], cluster_1=design[subpop1.ind,6],
                  cluster_2=design[subpop2.ind,6], wgt1_1=design[subpop1.ind,7],
                  x1_1=design[subpop1.ind,8], y1_1=design[subpop1.ind,9],
                  wgt1_2=design[subpop2.ind,7], x1_2=design[subpop2.ind,8],
                  y1_2=design[subpop2.ind,9], popsize_1=temp.popsize_1,
                  popsize_2=temp.popsize_2, popcorrect_1=pcfactor.ind,
                  pcfsize_1=temp.pcfsize_1, N.cluster_1=temp.N.cluster_1,
                  stage1size_1=temp.stage1size_1, support_1=design[subpop1.ind,10],
                  popcorrect_2=pcfactor.ind, pcfsize_2=temp.pcfsize_2,
                  N.cluster_2=temp.N.cluster_2, stage1size_2=temp.stage1size_2,
                  support_2=design[subpop2.ind,10], sizeweight_1=swgt.ind,
                  swgt_1=design[subpop1.ind,11], swgt1_1=design[subpop1.ind,12],
                  sizeweight_2=swgt.ind, swgt_2=design[subpop2.ind,11],
                  swgt1_2=design[subpop2.ind,12], vartype_1=vartype,
                  vartype_2=vartype, check.ind=FALSE, warn.ind=warn.ind,
                  warn.df=warn.df, warn.vec=warn.vec)
               temp.test <- temp$Results[testname,]
               warn.ind <- temp$warn.ind
               warn.df <- temp$warn.df

# Assign estimates for the response variable to data frames

               if(nrow == 0) {
                  contsum <- data.frame(Type=typenames[itype], 
                     Subpopulation_1=subpopnames[isubpop1], 
                     Subpopulation_2=subpopnames[isubpop2], 
                     Indicator=varnames[ivar], temp.test)
                  nrow <- 1
               } else {
                  contsum <- rbind(
                     contsum, data.frame(Type=typenames[itype], 
                     Subpopulation_1=subpopnames[isubpop1], 
                     Subpopulation_2=subpopnames[isubpop2], 
                     Indicator=varnames[ivar], temp.test, 
                     row.names=nrow+1))
                  nrow <- nrow + 1
               }

# End of the loops for subpopulations

            }
         }

# End of the loop for type of population

      }

# End of the loop for response variables

   }

# Modify dimension names

   rownames(contsum) <- NULL
   names(contsum)[5] <- testname
   if(testname %in% c("Wald", "Mean_Eigenvalue", "Satterthwaite")) {
      contsum <- contsum[,-7]
      names(contsum)[6] <- "Degrees_of_Freedom"
   }

# As necessary, output a message indicating that warning messages were generated
# during execution of the program

   if(warn.ind) {
      warn.df <<- warn.df
      if(nrow(warn.df) == 1)
         cat("During execution of the program, a warning message was generated.  The warning \nmessage is stored in a data frame named 'warn.df'.  Enter the following command \nto view the warning message: warnprnt()\n")
      else
         cat(paste("During execution of the program,", nrow(warn.df), "warning messages were generated.  The warning \nmessages are stored in a data frame named 'warn.df'.  Enter the following \ncommand to view the warning messages: warnprnt() \nTo view a subset of the warning messages (say, messages number 1, 3, and 5), \nenter the following command: warnprnt(m=c(1,3,5))\n"))
   }

# Return the data frame

   contsum
}
