################################################################################
# Function: irslin
# Programmer: Tom Kincaid
# Date: November 17, 2005
# Last Revised: August 18, 2016
#'
#' Select an Independent Random Sample (IRS) of a Linear Resource
#'
#' This function selects an IRS of a linear resource.
#'
#' @param shapefilename Name of the input shapefile.  If shapefilename equals
#'   NULL, then the shapefile or shapefiles in the working directory are used.
#'   The default is NULL.
#'
#' @param linframe Data frame containing id, mdcaty, len, and mdm.
#'
#' @param samplesize Number of points to select in the sample.  The default is
#'   100.
#'
#' @param SiteBegin First number to start siteID numbering.  The default is 1.
#'
#' @return Data frame of sample points containing: siteID, id, x, y, mdcaty,
#'   and weight.
#'
#' @section Other Functions Required:
#'   \describe{
#'     \item{\code{linSampleIRS}}{C function to select a sample from a linear
#'       resource}
#'   }
#'
#' @author Tom Kincaid \email{Kincaid.Tom@epa.gov}
#'
#' @keywords survey
#'
#' @export
################################################################################

irslin <- function (shapefilename = NULL, linframe, samplesize = 100,
   SiteBegin = 1) {

# Ensure that the processor is little-endian

   if(.Platform$endian == "big")
      stop("\nA little-endian processor is required for the irslin function.")

# Pick sample points

   len.cumsum <- cumsum(linframe$len*linframe$mdm)
   samp.pos <- runif(samplesize, 0, len.cumsum[nrow(linframe)])
   ordr <- rank(samp.pos)
   samp.pos <- sort(samp.pos)
   temp <- .Call("linSampleIRS", shapefilename, len.cumsum, samp.pos,
      linframe$id, linframe$len, linframe$mdm)
   temp$id <- temp$id[ordr]
   temp$x <- temp$x[ordr]
   temp$y <- temp$y[ordr]
   mdcaty <- linframe$mdcaty[match(temp$id, linframe$id)]
   mdm <- linframe$mdm[match(temp$id, linframe$id)]

# Assign Site ID

   siteID <- SiteBegin - 1 + 1:length(temp$id)

# Create the output data frame

   rho <- data.frame(siteID=siteID, id=temp$id, xcoord=temp$x, ycoord=temp$y,
      mdcaty=mdcaty, wgt=1/mdm)
   row.names(rho) <- 1:nrow(rho)

# Return the sample

   rho
}
