% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/survey_statistics.r
\name{survey_mean}
\alias{survey_mean}
\title{Calculate the mean and its variation using survey methods}
\usage{
survey_mean(
  x,
  na.rm = FALSE,
  vartype = c("se", "ci", "var", "cv"),
  level = 0.95,
  proportion = FALSE,
  prop_method = c("logit", "likelihood", "asin", "beta", "mean"),
  deff = FALSE,
  df = NULL,
  .svy = current_svy(),
  ...
)
}
\arguments{
\item{x}{A variable or expression, or empty}

\item{na.rm}{A logical value to indicate whether missing values should be dropped}

\item{vartype}{Report variability as one or more of: standard error ("se", default),
confidence interval ("ci"), variance ("var") or coefficient of variation
("cv").}

\item{level}{(For vartype = "ci" only) A single number or vector of numbers indicating
the confidence level}

\item{proportion}{Use methods to calculate the proportion that may have more accurate
confidence intervals near 0 and 1. Based on
\code{\link[survey]{svyciprop}}.}

\item{prop_method}{Type of proportion method to use if proportion is \code{TRUE}. See
\code{\link[survey]{svyciprop}} for details.}

\item{deff}{A logical value to indicate whether the design effect should be returned.}

\item{df}{(For vartype = "ci" only) A numeric value indicating the degrees of freedom
for t-distribution. The default (NULL) uses \code{\link[survey]{degf}},
but Inf is the usual survey package's default (except in
\code{\link[survey]{svyciprop}}.}

\item{.svy}{A \code{tbl_svy} object. When called from inside a summarize function
the default automatically sets the survey to the current survey.}

\item{...}{Ignored}
}
\description{
Calculate means and proportions from complex survey data. A wrapper
around \code{\link[survey]{svymean}}, or if \code{proportion = TRUE},
\code{\link[survey]{svyciprop}}. \code{survey_mean} should always be
called from \code{\link{summarise}}.
}
\examples{
library(survey)
data(api)

dstrata <- apistrat \%>\%
  as_survey_design(strata = stype, weights = pw)

dstrata \%>\%
  summarise(api99 = survey_mean(api99),
            api_diff = survey_mean(api00 - api99, vartype = c("ci", "cv")))

dstrata \%>\%
  group_by(awards) \%>\%
  summarise(api00 = survey_mean(api00))

# Leave x empty to calculate the proportion in each group
dstrata \%>\%
  group_by(awards) \%>\%
  summarise(pct = survey_mean())

# Setting proportion = TRUE uses a different method for calculating confidence intervals
dstrata \%>\%
  summarise(high_api = survey_mean(api00 > 875, proportion = TRUE, vartype = "ci"))

# level takes a vector for multiple levels of confidence intervals
dstrata \%>\%
  summarise(api99 = survey_mean(api99, vartype = "ci", level = c(0.95, 0.65)))

# Note that the default degrees of freedom in srvyr is different from
# survey, so your confidence intervals might not be exact matches. To
# Replicate survey's behavior, use df = Inf
dstrata \%>\%
  summarise(srvyr_default = survey_mean(api99, vartype = "ci"),
            survey_defualt = survey_mean(api99, vartype = "ci", df = Inf))

comparison <- survey::svymean(~api99, dstrata)
confint(comparison) # survey's default
confint(comparison, df = survey::degf(dstrata)) # srvyr's default

}
