% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/STVARconstruction.R
\name{reorder_B_columns}
\alias{reorder_B_columns}
\title{Reorder columns of impact matrix B (and lambda parameters if any) of a structural STVAR model
  that is identified by heteroskedasticity or non-Gaussianity.}
\usage{
reorder_B_columns(stvar, perm, calc_std_errors = FALSE)
}
\arguments{
\item{stvar}{a class 'stvar' object defining a structural STVAR model that is identified by heteroskedasticity
or non-Gaussianity, typically created with \code{fitSSTVAR}.}

\item{perm}{an integer vector of length \eqn{d} specifying the new order of the columns of the impact matrix.
For model identified by...
\describe{
  \item{heteroskedasticity}{also lambda parameters of each regime will be reordered accordingly.}
  \item{non-Gaussianity}{the columns of the impact matrices of all the regimes and the component specific distribution
    parameters (degrees of freedom parameters) are reordered accordingly.}
}}

\item{calc_std_errors}{should approximate standard errors be calculated?}
}
\value{
Returns an S3 object of class \code{'stvar'} defining a smooth transition VAR model. The returned list
 contains the following components (some of which may be \code{NULL} depending on the use case):
   \item{data}{The input time series data.}
   \item{model}{A list describing the model structure.}
   \item{params}{The parameters of the model.}
   \item{std_errors}{Approximate standard errors of the parameters, if calculated.}
   \item{transition_weights}{The transition weights of the model.}
   \item{regime_cmeans}{Conditional means of the regimes, if data is provided.}
   \item{total_cmeans}{Total conditional means of the model, if data is provided.}
   \item{total_ccovs}{Total conditional covariances of the model, if data is provided.}
   \item{uncond_moments}{A list of unconditional moments including regime autocovariances, variances, and means.}
   \item{residuals_raw}{Raw residuals, if data is provided.}
   \item{residuals_std}{Standardized residuals, if data is provided.}
   \item{structural_shocks}{Recovered structural shocks, if applicable.}
   \item{loglik}{Log-likelihood of the model, if data is provided.}
   \item{IC}{The values of the information criteria (AIC, HQIC, BIC) for the model, if data is provided.}
   \item{all_estimates}{The parameter estimates from all estimation rounds, if applicable.}
   \item{all_logliks}{The log-likelihood of the estimates from all estimation rounds, if applicable.}
   \item{which_converged}{Indicators of which estimation rounds converged, if applicable.}
   \item{which_round}{Indicators of which round of optimization each estimate belongs to, if applicable.}
}
\description{
\code{reorder_B_columns} reorder columns of impact matrix B (and lambda parameters if any) of
  a structural STVAR model that is identified by heteroskedasticity or non-Gaussianity.
}
\details{
The order of the columns of the impact matrix can be changed without changing the implied reduced
  form model (as long as, for models identified by heteroskedasticity, the order of lambda parameters is also changed accordingly;
  and for model identified by non-Gaussianity, ordering of the columns of all the impact matrices and the component specific
  distribution  parameters is also changed accordingly). Note that constraints imposed on the impact matrix via \code{B_constraints}
  will also be modified accordingly.

  Also all signs in any column of impact matrix can be swapped (without changing the implied reduced form model)
  with the function \code{swap_B_signs}. This obviously also swaps the sign constraints (if any) in the corresponding columns of
  the impact matrix.
}
\examples{
# Create a structural two-variate Student's t STVAR p=2, M=2 model with logistic transition
# weights and the first lag of the second variable  as the switching variable, and shocks
# identified by heteroskedasticity:
theta_222logt <- c(0.356914, 0.107436, 0.356386, 0.086330, 0.139960, 0.035172, -0.164575,
  0.386816, 0.451675, 0.013086, 0.227882, 0.336084, 0.239257, 0.024173, -0.021209, 0.707502,
  0.063322, 0.027287, 0.009182, 0.197066, -0.03, 0.24, -0.76, -0.02, 3.36, 0.86, 0.1, 0.2, 7)
mod222logt <- STVAR(p=2, M=2, d=2, params=theta_222logt, weight_function="logistic",
  weightfun_pars=c(2, 1), cond_dist="Student", identification="heteroskedasticity")

# Print the parameter values, W and lambdas are printed in the bottom:
mod222logt

# Reverse the ordering of the columns of W (or equally the impact matrix):
mod222logt_rev <- reorder_B_columns(mod222logt, perm=c(2, 1))
mod222logt_rev # The columns of the impact matrix are in a reversed order

# Swap the ordering of the columns of the impact matrix back to the original:
mod222logt_rev2 <- reorder_B_columns(mod222logt_rev, perm=c(2, 1))
mod222logt_rev2 # The columns of the impact matrix are back in the original ordering

# Below code does not do anything, as perm=1:2, so the ordering does not change:
mod222logt3 <- reorder_B_columns(mod222logt, perm=c(1, 2))
mod222logt3 # The ordering of the columns did not change from the original
}
\references{
\itemize{
   \item Lütkepohl H., Netšunajev A. 2018. Structural vector autoregressions with smooth transition in variances.
     \emph{Journal of Economic Dynamics & Control}, \strong{84}, 43-57.
 }
}
\seealso{
\code{\link{GIRF}}, \code{\link{fitSSTVAR}}, \code{\link{swap_B_signs}}
}
