% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gee_methods.R
\name{standardize_gee}
\alias{standardize_gee}
\title{Regression standardization in conditional generalized estimating equations}
\usage{
standardize_gee(
  formula,
  link = "identity",
  data,
  values,
  clusterid,
  case_control = FALSE,
  ci_level = 0.95,
  ci_type = "plain",
  contrasts = NULL,
  family = "gaussian",
  reference = NULL,
  transforms = NULL
)
}
\arguments{
\item{formula}{A formula to be used with \code{"gee"} in the \pkg{drgee} package.}

\item{link}{The link function to be used with \code{"gee"}.}

\item{data}{The data.}

\item{values}{A named list or data.frame specifying the variables and values
at which marginal means of the outcome will be estimated.}

\item{clusterid}{An optional string containing the name of a cluster identification variable
when data are clustered.}

\item{case_control}{Whether the data comes from a case-control study.}

\item{ci_level}{Coverage probability of confidence intervals.}

\item{ci_type}{A string, indicating the type of confidence intervals.
Either "plain", which gives untransformed intervals, or "log", which gives
log-transformed intervals.}

\item{contrasts}{A vector of contrasts in the following format:
If set to \code{"difference"} or \code{"ratio"}, then \eqn{\psi(x)-\psi(x_0)}
or \eqn{\psi(x) / \psi(x_0)} are constructed, where \eqn{x_0} is a reference
level specified by the \code{reference} argument. Has to be \code{NULL}
if no references are specified.}

\item{family}{The family argument which is used to fit the glm model for the outcome.}

\item{reference}{A vector of reference levels in the following format:
If \code{contrasts} is not \code{NULL}, the desired reference level(s). This
must be a vector or list the same length as \code{contrasts}, and if not named,
it is assumed that the order is as specified in contrasts.}

\item{transforms}{A vector of transforms in the following format:
If set to \code{"log"}, \code{"logit"}, or \code{"odds"}, the standardized
mean \eqn{\theta(x)} is transformed into \eqn{\psi(x)=\log\{\theta(x)\}},
\eqn{\psi(x)=\log[\theta(x)/\{1-\theta(x)\}]}, or
\eqn{\psi(x)=\theta(x)/\{1-\theta(x)\}}, respectively.
If the vector is \code{NULL}, then \eqn{\psi(x)=\theta(x)}.}
}
\value{
An object of class \code{std_glm}.
This is basically a list with components estimates and covariance estimates in \code{res}.
Results for transformations, contrasts, references are stored in \code{res_contrasts}.
Obtain numeric results in a data frame with the \link{tidy} function.
}
\description{
\code{standardize_gee} performs regression standardization in linear and log-linear
fixed effects models, at specified values of the exposure, over the sample
covariate distribution. Let \eqn{Y}, \eqn{X}, and \eqn{Z} be the outcome,
the exposure, and a vector of covariates, respectively. It is assumed that
data are clustered with a cluster indicator \eqn{i}. \code{standardize_gee} uses
fitted fixed effects model, with cluster-specific intercept \eqn{a_i} (see
\code{details}), to estimate the standardized mean
\eqn{\theta(x)=E\{E(Y|i,X=x,Z)\}}, where \eqn{x} is a specific value of
\eqn{X}, and the outer expectation is over the marginal distribution of
\eqn{(a_i,Z)}.
}
\details{
\code{standardize_gee} assumes that a fixed effects model
\deqn{\eta\{E(Y|i,X,Z)\}=a_i+h(X,Z;\beta)} has been fitted. The link
function \eqn{\eta} is assumed to be the identity link or the log link. The
conditional generalized estimating equation (CGEE) estimate of \eqn{\beta}
is used to obtain estimates of the cluster-specific means:
\deqn{\hat{a}_i=\sum_{j=1}^{n_i}r_{ij}/n_i,} where
\deqn{r_{ij}=Y_{ij}-h(X_{ij},Z_{ij};\hat{\beta})} if \eqn{\eta} is the
identity link, and \deqn{r_{ij}=Y_{ij}\exp\{-h(X_{ij},Z_{ij};\hat{\beta})\}}
if \eqn{\eta} is the log link, and \eqn{(X_{ij},Z_{ij})} is the value of
\eqn{(X,Z)} for subject \eqn{j} in cluster \eqn{i}, \eqn{j=1,...,n_i},
\eqn{i=1,...,n}. The CGEE estimate of \eqn{\beta} and the estimate of
\eqn{a_i} are used to estimate the mean \eqn{E(Y|i,X=x,Z)}:
\deqn{\hat{E}(Y|i,X=x,Z)=\eta^{-1}\{\hat{a}_i+h(X=x,Z;\hat{\beta})\}.} For
each \eqn{x} in the \code{x} argument, these estimates are averaged across
all subjects (i.e. all observed values of \eqn{Z} and all estimated values
of \eqn{a_i}) to produce estimates \deqn{\hat{\theta}(x)=\sum_{i=1}^n
\sum_{j=1}^{n_i} \hat{E}(Y|i,X=x,Z_i)/N,} where \eqn{N=\sum_{i=1}^n n_i}.
The variance for \eqn{\hat{\theta}(x)} is obtained by the sandwich formula.
}
\note{
The variance calculation performed by \code{standardize_gee} does not condition
on the observed covariates \eqn{\bar{Z}=(Z_{11},...,Z_{nn_i})}. To see how
this matters, note that
\deqn{var\{\hat{\theta}(x)\}=E[var\{\hat{\theta}(x)|\bar{Z}\}]+var[E\{\hat{\theta}(x)|\bar{Z}\}].}
The usual parameter \eqn{\beta} in a generalized linear model does not
depend on \eqn{\bar{Z}}. Thus, \eqn{E(\hat{\beta}|\bar{Z})} is independent
of \eqn{\bar{Z}} as well (since \eqn{E(\hat{\beta}|\bar{Z})=\beta}), so that
the term \eqn{var[E\{\hat{\beta}|\bar{Z}\}]} in the corresponding variance
decomposition for \eqn{var(\hat{\beta})} becomes equal to 0. However,
\eqn{\theta(x)} depends on \eqn{\bar{Z}} through the average over the sample
distribution for \eqn{Z}, and thus the term
\eqn{var[E\{\hat{\theta}(x)|\bar{Z}\}]} is not 0, unless one conditions on
\eqn{\bar{Z}}.
}
\examples{

require(drgee)

set.seed(4)
n <- 300
ni <- 2
id <- rep(1:n, each = ni)
ai <- rep(rnorm(n), each = ni)
Z <- rnorm(n * ni)
X <- rnorm(n * ni, mean = ai + Z)
Y <- rnorm(n * ni, mean = ai + X + Z + 0.1 * X^2)
dd <- data.frame(id, Z, X, Y)
fit.std <- standardize_gee(
  formula = Y ~ X + Z + I(X^2),
  link = "identity",
  data = dd,
  values = list(X = seq(-3, 3, 0.5)),
  clusterid = "id"
)
print(fit.std)
plot(fit.std)

}
\references{
Goetgeluk S. and Vansteelandt S. (2008). Conditional generalized
estimating equations for the analysis of clustered and longitudinal data.
\emph{Biometrics} \bold{64}(3), 772-780.

Martin R.S. (2017). Estimation of average marginal effects in multiplicative
unobserved effects panel models. \emph{Economics Letters} \bold{160}, 16-19.

Sjölander A. (2019). Estimation of marginal causal effects in the presence
of confounding by cluster. \emph{Biostatistics} doi:
10.1093/biostatistics/kxz054
}
\author{
Arvid Sjölander.
}
