% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glm_methods.R
\name{standardize_glm}
\alias{standardize_glm}
\title{Get regression standardized estimates from a glm}
\usage{
standardize_glm(
  formula,
  data,
  values,
  clusterid,
  matched_density_cases,
  matched_density_controls,
  matching_variable,
  p_population,
  case_control = FALSE,
  ci_level = 0.95,
  ci_type = "plain",
  contrasts = NULL,
  family = "gaussian",
  reference = NULL,
  transforms = NULL
)
}
\arguments{
\item{formula}{The formula which is used to fit the model for the outcome.}

\item{data}{The data.}

\item{values}{A named list or data.frame specifying the variables and values
at which marginal means of the outcome will be estimated.}

\item{clusterid}{An optional string containing the name of a cluster identification variable
when data are clustered.}

\item{matched_density_cases}{A function of the matching variable.
The probability (or density) of the matched variable among the cases.}

\item{matched_density_controls}{A function of the matching variable.
The probability (or density) of the matched variable among the controls.}

\item{matching_variable}{The matching variable extracted from the data set.}

\item{p_population}{Specifies the incidence in the population when \code{case_control=TRUE}.}

\item{case_control}{Whether the data comes from a case-control study.}

\item{ci_level}{Coverage probability of confidence intervals.}

\item{ci_type}{A string, indicating the type of confidence intervals.
Either "plain", which gives untransformed intervals, or "log", which gives
log-transformed intervals.}

\item{contrasts}{A vector of contrasts in the following format:
If set to \code{"difference"} or \code{"ratio"}, then \eqn{\psi(x)-\psi(x_0)}
or \eqn{\psi(x) / \psi(x_0)} are constructed, where \eqn{x_0} is a reference
level specified by the \code{reference} argument. Has to be \code{NULL}
if no references are specified.}

\item{family}{The family argument which is used to fit the glm model for the outcome.}

\item{reference}{A vector of reference levels in the following format:
If \code{contrasts} is not \code{NULL}, the desired reference level(s). This
must be a vector or list the same length as \code{contrasts}, and if not named,
it is assumed that the order is as specified in contrasts.}

\item{transforms}{A vector of transforms in the following format:
If set to \code{"log"}, \code{"logit"}, or \code{"odds"}, the standardized
mean \eqn{\theta(x)} is transformed into \eqn{\psi(x)=\log\{\theta(x)\}},
\eqn{\psi(x)=\log[\theta(x)/\{1-\theta(x)\}]}, or
\eqn{\psi(x)=\theta(x)/\{1-\theta(x)\}}, respectively.
If the vector is \code{NULL}, then \eqn{\psi(x)=\theta(x)}.}
}
\value{
An object of class \code{std_glm}.
This is basically a list with components estimates and covariance estimates in \code{res}.
Results for transformations, contrasts, references are stored in \code{res_contrasts}.
Obtain numeric results in a data frame with the \link{tidy} function.
}
\description{
Get regression standardized estimates from a glm
}
\details{
\code{standardize_glm} performs regression standardization
in generalized linear models,
at specified values of the exposure, over the sample covariate distribution.
Let \eqn{Y}, \eqn{X}, and \eqn{Z} be the outcome, the exposure, and a
vector of covariates, respectively.
\code{standardize_glm} uses a fitted generalized linear
model to estimate the standardized
mean \eqn{\theta(x)=E\{E(Y|X=x,Z)\}},
where \eqn{x} is a specific value of \eqn{X},
and the outer expectation is over the marginal distribution of \eqn{Z}.
}
\examples{

# basic example
# needs to correctly specify the outcome model and no unmeasered confounders
# (+ standard causal assunmptions)
set.seed(6)
n <- 100
Z <- rnorm(n)
X <- rnorm(n, mean = Z)
Y <- rbinom(n, 1, prob = (1 + exp(X + Z))^(-1))
dd <- data.frame(Z, X, Y)
x <- standardize_glm(
  formula = Y ~ X * Z,
  family = "binomial",
  data = dd,
  values = list(X = 0:1),
  contrasts = c("difference", "ratio"),
  reference = 0
)
x
# different transformations of causal effects

# example from Sjölander (2016) with case-control data
# here the matching variable needs to be passed as an argument
singapore <- AF::singapore
Mi <- singapore$Age
m <- mean(Mi)
s <- sd(Mi)
d <- 5
standardize_glm(
  formula = Oesophagealcancer ~ (Everhotbev + Age + Dial + Samsu + Cigs)^2,
  family = binomial, data = singapore,
  values = list(Everhotbev = 0:1), clusterid = "Set",
  case_control = TRUE,
  matched_density_cases = function(x) dnorm(x, m, s),
  matched_density_controls = function(x) dnorm(x, m - d, s),
  matching_variable = Mi,
  p_population = 19.3 / 100000
)

# multiple exposures
set.seed(7)
n <- 100
Z <- rnorm(n)
X1 <- rnorm(n, mean = Z)
X2 <- rnorm(n)
Y <- rbinom(n, 1, prob = (1 + exp(X1 + X2 + Z))^(-1))
dd <- data.frame(Z, X1, X2, Y)
x <- standardize_glm(
  formula = Y ~ X1 + X2 + Z,
  family = "binomial",
  data = dd, values = list(X1 = 0:1, X2 = 0:1),
  contrasts = c("difference", "ratio"),
  reference = c(X1 = 0, X2 = 0)
)
x
tidy(x)

# continuous exposure
set.seed(2)
n <- 100
Z <- rnorm(n)
X <- rnorm(n, mean = Z)
Y <- rnorm(n, mean = X + Z + 0.1 * X^2)
dd <- data.frame(Z, X, Y)
x <- standardize_glm(
  formula = Y ~ X * Z,
  family = "gaussian",
  data = dd,
  values = list(X = seq(-1, 1, 0.1))
)

# plot standardized mean as a function of x
plot(x)
# plot standardized mean - standardized mean at x = 0 as a function of x
plot(x, contrast = "difference", reference = 0)

}
\references{
Rothman K.J., Greenland S., Lash T.L. (2008).
\emph{Modern Epidemiology}, 3rd edition.
Lippincott, Williams & Wilkins.

Sjölander A. (2016).
Regression standardization with the R-package stdReg.
\emph{European Journal of Epidemiology} \bold{31}(6), 563-574.

Sjölander A. (2016).
Estimation of causal effect measures with the R-package stdReg.
\emph{European Journal of Epidemiology} \bold{33}(9), 847-858.
}
