% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_lgcp.r
\name{fit_lgcp}
\alias{fit_lgcp}
\title{Spatial or spatiotemporal log-Gaussian Cox process (LGCP)}
\usage{
fit_lgcp(
  locs,
  sf,
  smesh,
  tmesh,
  parameters,
  covariates,
  tmb_silent = TRUE,
  nlminb_silent = TRUE,
  ...
)
}
\arguments{
\item{locs}{A \code{data.frame} of \code{x} and \code{y} locations, \eqn{2 \times n}. If a
spatiotemporal model is to be fitted then there should be the third column (\code{t}) of the occurrence times.}

\item{sf}{An \code{sf} of type \code{POLYGON} specifying the spatial region
of the domain.}

\item{smesh}{A Delaunay triangulation of the spatial domain returned by \code{fmesher::fm_mesh_2d()}.}

\item{tmesh}{Optional, a temporal mesh returned by \code{fmesher::fm_mesh_1d()}.}

\item{parameters}{A named list of parameter starting values:
\itemize{
\item \code{beta}, a vector of fixed effects coefficients to be estimated, \eqn{\beta}
(same length as \code{ncol(covariates)} + 1 );
\item  \code{log_tau}, the \eqn{\textrm{log}(\tau)} parameter for the GMRF;
\item \code{log_kappa}, \eqn{\textrm{log}(\kappa)} parameter for the GMRF;
\item \code{atanh_rho}, optional, \eqn{\textrm{arctan}(\rho)} AR1 temporal parameter.
}
Default values are used if none are provided. NOTE: these may not always be appropriate.}

\item{covariates}{Optional, a \code{matrix} of covariates at each
\code{smesh} and \code{tmesh} node combination.}

\item{tmb_silent}{Logical, if \code{TRUE} (default) then
TMB inner optimisation tracing information will be printed.}

\item{nlminb_silent}{Logical, if \code{TRUE} (default) then for each iteration
\code{nlminb()} output will be printed.}

\item{...}{optional extra arguments to pass into \code{stats::nlminb()}.}
}
\value{
A list containing components of the fitted model, see \code{TMB::MakeADFun}. Includes
\itemize{
\item \code{par}, a numeric vector of estimated parameter values;
\item \code{objective}, the objective function;
\item \code{gr}, the TMB calculated gradient function; and
\item \code{simulate}, a simulation function. 
}
}
\description{
Fit a log-Gaussian Cox process (LGCP) using Template Model Builder (TMB) and the
\code{R_inla} namespace for the SPDE-based construction of the latent field.
}
\details{
A log-Gaussian Cox process (LGCP) where the Gaussian random field, \eqn{Z(\boldsymbol{x})},
has zero mean, variance-covariance matrix \eqn{\boldsymbol{Q}^{-1}}, and covariance function
\eqn{C_Z}. The random intensity surface is
\eqn{\Lambda(\boldsymbol{x}) = \textrm{exp}(\boldsymbol{X}\beta + G(\boldsymbol{x}) + \epsilon)},
for design matrix \eqn{\boldsymbol{X}}, coefficients \eqn{\boldsymbol{\beta}}, and random error \eqn{\epsilon}.


Shown in Lindgren et. al., (2011) the stationary solution to the SPDE (stochastic
partial differential equation) \eqn{(\kappa^2 - \Delta)^{\frac{\nu + \frac{d}{2}}{2}}G(s) = W(s)} is
a random field with a Matérn covariance function,
\eqn{C_Z \propto {\kappa || x - y||}^{\nu}K_{\nu}{\kappa || x - y||}}. Here \eqn{\nu} controls
the smoothness of the field and \eqn{\kappa} controls the range.


A Markovian random field is obtained when \eqn{\alpha = \nu + \frac{d}{2}} is an integer. Following
Lindgren et. al., (2011) we set \eqn{\alpha = 2} in 2D and therefore fix \eqn{\nu = 1}. Under these
conditions the solution to the SPDE is a Gaussian Markov Random Field (GMRF). This is the approximation
we use.

The (approximate) spatial range \eqn{= \frac{\sqrt{8 \nu}}{\kappa} = \frac{\sqrt{8}}{\kappa}} and
the standard deviation of the model, \eqn{\sigma = \frac{1}{\sqrt{4 \pi \kappa^2 \tau^2}}}.
Under \code{INLA} (Lindgren and Rue, 2015) methodology the practical range is defined as the
distance such that the correlation is \eqn{\sim 0.139}.
}
\examples{
\donttest{
### ********************** ###
## A spatial only LGCP
### ********************** ###
if(requireNamespace("fmesher")) {
data(xyt, package = "stelfi")
domain <- sf::st_as_sf(xyt$window)
locs <- data.frame(x = xyt$x, y = xyt$y)
bnd <- fmesher::fm_as_segm(as.matrix(sf::st_coordinates(domain)[, 1:2]))
smesh <- fmesher::fm_mesh_2d(boundary = bnd,
max.edge = 0.75, cutoff = 0.3)
fit <- fit_lgcp(locs = locs, sf = domain, smesh = smesh,
parameters = c(beta = 0, log_tau = log(1), log_kappa = log(1)))
### ********************** ###
## A spatiotemporal LGCP, AR(1)
### ********************** ###
ndays <- 2
locs <- data.frame(x = xyt$x, y = xyt$y, t = xyt$t)
w0 <- 2
tmesh <- fmesher::fm_mesh_1d(seq(0, ndays, by = w0))
fit <- fit_lgcp(locs = locs, sf = domain, smesh = smesh, tmesh = tmesh,
 parameters = c(beta = 0, log_tau = log(1), log_kappa = log(1), atanh_rho = 0.2))
}
}
}
\references{
Lindgren, F., Rue, H., and Lindström, J. (2011)
An explicit link between Gaussian fields and Gaussian Markov random fields: the stochastic
partial differential equation approach. \emph{Journal of the Royal Statistical Society: Series B
(Statistical Methodology)}, \strong{73}: 423--498.

Lindgren, F. and Rue, H. (2015) Bayesian spatial modelling with R-INLA.
\emph{Journal of Statistical Software}, \strong{63}: 1--25.
}
\seealso{
\code{\link{fit_mlgcp}} and \code{\link{sim_lgcp}}
}
