#' Deprecated functions in stplanr
#'
#' These functions are depreciated and will be removed:
#'
#' @name stplanr-deprecated
NULL

# Functions for aggregating two-way OD pairs into 'oneway' lines
#' Aggregate ods so they become non-directional
#'
#' For example, sum total travel in both directions.
#' @param x A data frame or SpatialLinesDataFrame, representing an OD matrix
#' @param attrib A vector of column numbers or names
#' for deciding which attribute(s) of class numeric to
#' aggregate
#' @param id1 Optional (it is assumed to be the first column)
#' text string referring to the name of the variable containing
#' the unique id of the origin
#' @param id2 Optional (it is assumed to be the second column)
#' text string referring to the name of the variable containing
#' the unique id of the destination
#' @param stplanr.key A key of unique OD pairs regardless of the order,
#' autogenerated by [od_id_order()]
#' @return `onewayid` outputs a data.frame with rows containing
#' results for the user-selected attribute values that have been aggregated.
#' @family lines
#' @export
onewayid <- function(x, attrib, id1 = names(x)[1], id2 = names(x)[2],
                     stplanr.key = od_id_order(x, id1, id2)) {
  .Deprecated(new = "od_oneway", package = "od",
              msg = "See updated od_oneway function in stplanr, and the od package")
  UseMethod("onewayid")
}

#' @name onewayid
#' @details
#' Flow data often contains movement in two directions: from point A to point B
#' and then from B to A. This can be problematic for transport planning, because
#' the magnitude of flow along a route can be masked by flows the other direction.
#' If only the largest flow in either direction is captured in an analysis, for
#' example, the true extent of travel will by heavily under-estimated for
#' OD pairs which have similar amounts of travel in both directions.
#' Flows in both direction are often represented by overlapping lines with
#' identical geometries (see [flowlines()]) which can be confusing
#' for users and are difficult to plot.
#' @examples
#' \donttest{
#' # this function is deprecated so examples are not expected to run
#' # keeping the example code in there for now for posterity
#' flow_oneway <- onewayid(flow, attrib = 3)
#' nrow(flow_oneway) < nrow(flow) # result has fewer rows
#' sum(flow$All) == sum(flow_oneway$All) # but the same total flow
#' # using names instead of index for attribute
#' onewayid(flow, attrib = "All")
#' # using many attributes to aggregate
#' attrib <- which(vapply(flow, is.numeric, TRUE))
#' flow_oneway <- onewayid(flow, attrib = attrib)
#' colSums(flow_oneway[attrib]) == colSums(flow[attrib]) # test if the colSums are equal
#' # Demonstrate the results from onewayid and onewaygeo are identical
#' flow_oneway_geo <- onewaygeo(flowlines, attrib = attrib)
#' plot(flow_oneway$All, flow_oneway_geo$All)
#' flow_oneway_sf <- onewayid(flowlines_sf, 3)
#' plot(flow_oneway_geo, lwd = flow_oneway_geo$All / mean(flow_oneway_geo$All))
#' plot(flow_oneway_sf$geometry, lwd = flow_oneway_sf$All / mean(flow_oneway_sf$All))
#' }
#' @export
onewayid.data.frame <- function(x, attrib, id1 = names(x)[1], id2 = names(x)[2],
                                stplanr.key = od_id_order(x, id1, id2)) {
  if (is.numeric(attrib)) {
    attrib_names <- names(x)[attrib]
  } else {
    attrib_names <- attrib
    attrib <- which(names(x) %in% attrib)
  }

  # separate geometry for sf objects
  is_sf <- is(x, "sf")
  if (is_sf) {
    x_sf <- sf::st_sf(stplanr.key[3], geometry = sf::st_geometry(x))
    x <- sf::st_drop_geometry(x)
  }

  x <- dplyr::bind_cols(x, stplanr.key)

  x_oneway_numeric <- dplyr::group_by(x, stplanr.key) %>%
    dplyr::summarise_at(attrib, sum)

  x_oneway_binary <- dplyr::mutate(x, is_two_way = duplicated(stplanr.key)) %>%
    dplyr::group_by(stplanr.key) %>%
    dplyr::summarise(is_two_way = dplyr::last(.data$is_two_way)) %>%
    dplyr::select(-stplanr.key)

  x_oneway_character <- x %>%
    dplyr::transmute(
      id1 = stringr::str_split(.data$stplanr.key, " ", simplify = TRUE)[, 1],
      id2 = stringr::str_split(.data$stplanr.key, " ", simplify = TRUE)[, 2],
      stplanr.key = .data$stplanr.key
    ) %>%
    dplyr::group_by(stplanr.key) %>%
    dplyr::summarise(id1 = dplyr::first(id1), id2 = dplyr::first(id2)) %>%
    dplyr::select(-stplanr.key)

  x_oneway <- dplyr::bind_cols(
    x_oneway_character,
    x_oneway_numeric,
    x_oneway_binary
  )

  if (is_sf) {
    x_sf <- x_sf[!duplicated(x_sf$stplanr.key), ]
    x_oneway <- sf::st_as_sf(dplyr::inner_join(x_oneway, x_sf))
    # class(x_oneway) # sf
  }

  x_oneway$stplanr.key <- NULL
  names(x_oneway)[1:2] <- c(id1, id2)

  return(x_oneway)
}

#' @name onewayid
#' @examples
#' # with spatial data
#' data(flowlines)
#' fo <- onewayid(flowlines, attrib = "All")
#' head(fo@data)
#' plot(fo)
#' sum(fo$All) == sum(flowlines$All)
#' # test results for one line
#' n <- 3
#' plot(fo[n, ], lwd = 20, add = TRUE)
#' f_over_n <- rgeos::gEquals(fo[n, ], flowlines["All"], byid = TRUE)
#' sum(flowlines$All[f_over_n]) == sum(fo$All[n]) # check aggregation worked
#' plot(flowlines[which(f_over_n)[1], ], add = TRUE, col = "white", lwd = 10)
#' plot(flowlines[which(f_over_n)[2], ], add = TRUE, lwd = 5)
#' @export
onewayid.SpatialLines <- function(x, attrib, id1 = names(x)[1], id2 = names(x)[2],
                                  stplanr.key = od_id_order(x, id1, id2)) {
  x_geom <- sp::SpatialLines(x@lines, proj4string = sp::CRS(proj4string(x)))
  x <- x@data

  x_oneway <- onewayid(x, id1, id2, attrib = attrib)
  x_oneway$stplanr.key <- od_id_order(x_oneway[c(id1, id2)])$stplanr.key

  if (length(x_geom) != nrow(x_oneway)) {
    id_old <- paste(x[[id1]], x[[id2]])
    sel <- id_old %in% x_oneway$stplanr.key
    x_geom <- x_geom[sel, ]
  }

  x_oneway <- sp::SpatialLinesDataFrame(sl = x_geom, data = x_oneway, match.ID = FALSE)

  return(x_oneway)
}

#' Import GTFS shapes and route data to SpatialLinesDataFrame.
#'
#' Takes a string with the file path of the zip file with the GTFS feed,
#' imports the shapes (geometry), route and agency data and returns a
#' SpatialLinesDataFrame for the GTFS feed.
#'
#' @param gtfszip String with the file path of the GTFS feed zip file
#' @export
#' @examples
#' f <- system.file("extdata", "beartransit-ca-us.zip", package = "stplanr")
#' # update file to latest version
#' # see https://code.google.com/p/googletransitdatafeed/wiki/PublicFeeds
#' u <- "http://data.trilliumtransit.com/gtfs/beartransit-ca-us/beartransit-ca-us.zip"
#' # download.file(u, f)
#' gtfs <- gtfs2sldf(gtfszip = f)
#' plot(gtfs, col = gtfs$route_long_name)
#' plot(gtfs[gtfs$route_long_name == "Central Campus", ])
#' \dontrun{
#' # An example of a larger gtfs feed
#' download.file(
#'   "http://www.yrt.ca/google/google_transit.zip",
#'   paste0(tempdir(), "/gtfsfeed.zip")
#' )
#' yrtgtfs <- gtfs2sldf(paste0(tempdir(), "/gtfsfeed.zip"))
#' sp::plot(yrtgtfs, col = paste0("#", yrtgtfs$route_color))
#' }
gtfs2sldf <- function(gtfszip = "") {
  .Deprecated(new = "read_gtfs", package = "gtfs2gps", msg = "Many packages read gtfs files")
  if (gtfszip == "") {
    stop("Zip file required")
  }
  if (file.exists(gtfszip) == FALSE) {
    stop("Specified zip file does not exist")
  }

  gtfsfiles <- unzip(gtfszip, exdir = tempdir())

  gtfstrips <-
    read.csv(stringsAsFactors = TRUE, paste0(tempdir(), "/trips.txt"))
  if (all(charToRaw(substr(colnames(gtfstrips)[1], 1, 3)) == c(as.raw(239), as.raw(46), as.raw(46)))) {
    gtfstrips <-
      read.csv(
        stringsAsFactors = TRUE,
        paste0(tempdir(), "/trips.txt"),
        fileEncoding = "UTF-8-BOM"
      )
    gtfsroutes <-
      read.csv(
        stringsAsFactors = TRUE,
        paste0(tempdir(), "/routes.txt"),
        fileEncoding = "UTF-8-BOM"
      )
    gtfsagency <-
      read.csv(
        stringsAsFactors = TRUE,
        paste0(tempdir(), "/agency.txt"),
        fileEncoding = "UTF-8-BOM"
      )
    gtfsshape <-
      read.csv(
        stringsAsFactors = TRUE,
        paste0(tempdir(), "/shapes.txt"),
        fileEncoding = "UTF-8-BOM"
      )
  }
  else {
    gtfsroutes <-
      read.csv(stringsAsFactors = TRUE, paste0(tempdir(), "/routes.txt"))
    gtfsagency <-
      read.csv(stringsAsFactors = TRUE, paste0(tempdir(), "/agency.txt"))
    gtfsshape <-
      read.csv(stringsAsFactors = TRUE, paste0(tempdir(), "/shapes.txt"))
  }

  if (nrow(gtfsshape) == 0) {
    stop("GTFS shapes.txt file is empty.")
  }

  unlink(gtfsfiles)

  gtfslines <- sp::SpatialLinesDataFrame((
    gtfsshape %>%
      dplyr::group_by_( ~ shape_id) %>%
      dplyr::arrange_( ~ shape_pt_sequence) %>%
      dplyr::do_(gtfsline = "sp::Lines(sp::Line(as.matrix(.[,c('shape_pt_lon','shape_pt_lat')])),unique(.$shape_id))") %>%
      dplyr::ungroup() %>%
      dplyr::do_(
        gtfsline = "sp::SpatialLines(.[[2]],
                                    proj4string = sp::CRS('+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs'))"
      )
  )[[1]][[1]],
  data = gtfstrips %>%
    dplyr::inner_join(gtfsroutes) %>%
    dplyr::distinct_(
      ~ route_id,
      ~ shape_id,
      ~ route_short_name,
      ~ route_long_name,
      ~ route_desc,
      ~ route_type,
      ~ route_color,
      ~ route_text_color,
      ~ agency_id
    ) %>%
    dplyr::select_(
      ~ route_id,
      ~ shape_id,
      ~ route_short_name,
      ~ route_long_name,
      ~ route_desc,
      ~ route_type,
      ~ route_color,
      ~ route_text_color,
      ~ agency_id
    ) %>%
    dplyr::inner_join(gtfsagency) %>%
    dplyr::do_("`rownames<-`(.,.$shape_id)")
  )
  rm(gtfstrips, gtfsshape, gtfsagency, gtfsroutes)
  return(gtfslines)
}

