% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/string_magic_main.R
\name{string_magic_register_fun}
\alias{string_magic_register_fun}
\alias{string_magic_register_ops}
\title{Register custom operations to apply them in \code{string_magic}}
\usage{
string_magic_register_fun(fun, alias, valid_options = NULL, namespace = NULL)

string_magic_register_ops(ops, alias, namespace = NULL)
}
\arguments{
\item{fun}{A function which must have at least the arguments 'x' and '...'.
Additionnaly, it can have the arguments: 'argument', 'options', 'group', 'group_flag'.
This function must return a vector.
This function will be internally called by \code{string_magic} in the form
\code{fun(x, argument, options, group, group_flag)}.\code{x}: the value to which the
operation applies. \code{argument}: the quoted \code{string_magic} argument (always character).
\code{options}: a character vector of \code{string_magic} options. The two last arguments are of use
only in group-wise operations if \code{fun} changes the lengths of vectors. \code{group}: an index of
the group to which belongs each observation (integer). \code{group_flag}: value between 0
and 2; 0: no grouping operation requested; 1: keep track of groups; 2: apply grouping.}

\item{alias}{Character scalar, the name of the operation.}

\item{valid_options}{A character vector or NULL (default). Represents a list of
valid options for the operation. This is used: a) to enable auto-completion,
b) for error-handling purposes.}

\item{namespace}{Character scalar or \code{NULL} (default). \strong{Only useful for package developers.}
As a regular end-user you shouldn't care! If your package uses \code{string_magic}, you should care.
If the function \verb{string_magic_register_*} is located in the \code{onLoad} function (see \code{help("onLoad")}),
there is nothing to do. Otherwise, pass the name of your package in this argument to
make all the new operation definitions scoped (i.e. only your package can access it and
it can't be messed up by end users).}

\item{ops}{Character scalar representing a valid chain of \code{string_magic} operations. It should
be of the form \code{"op1, 'arg'op2, etc"}. For example \code{"'80|-'fill"} fills the line
with dashes up to 80 characters.}
}
\value{
These function do not return anything. They register new operations to be used in the
\code{string_magic} family of functions by placing them in the options (later fetched by
\code{string_magic()} at run-time).
}
\description{
Extends the capabilities of \code{\link[=string_magic]{string_magic()}} by adding any custom operation
}
\details{
We try to strongly check the new operations since it's always better to find out problems
sooner than later. This means that when the function is defined, it is also
tested.

If you pass a function, note that it should work for non-character arguments in \code{x}.
}
\section{Functions}{
\itemize{
\item \code{string_magic_register_ops()}: Create new combinations of \code{string_magic} operations

}}
\section{Writing a package using \code{string_magic}}{


If you want to use \code{string_magic} in your package and want to make use of custom operations:
\itemize{
\item place any \code{string_magic_register_fun} and \code{string_magic_register_ops} in your \code{.onLoad} function
(see \code{help("onLoad")}). The .onLoad function is run whenever the package is loaded
for the first time. It's a function that you can place anywhere in your \verb{R/*} files
and which looks like this:
}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{.onLoad = function(libname, pkgname)\{
  # string_magic custom operations
  string_magic_register_ops("'80|-'fill", "h1")

  invisible()
\}
}\if{html}{\out{</div>}}
\itemize{
\item if you don't want to place the \verb{string_magic_register_*} functions in the .onLoad function,
you can, but then you \strong{must} provide the argument \code{namespace}:
}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{string_magic_register_ops("'80|-'fill", "h1", namespace = "myPackageName")
}\if{html}{\out{</div>}}
\itemize{
\item you must create an \code{\link[=string_magic_alias]{string_magic_alias()}} to create an alias to \code{\link[=string_magic]{string_magic()}} and use the
argument \code{.namespace = "myPackageName"}. Use this opportunity to change the
defaults if you wish. You can even override the \code{string_magic} function:
}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# creating an alias with the same name + changing the delimiter
string_magic = stringmagic::string_magic_alias(.namespace = "myPackageName", .delim = "\{\{ \}\}")
}\if{html}{\out{</div>}}
}

\examples{

# let's create an operation that adds markdown emphasis
# to elements of a vector

# A) define the function
fun_emph = function(x, ...) paste0("*", x, "*")

# B) register it
string_magic_register_fun(fun_emph, "emph")

# C) use it
x = string_vec("right, now")
string_magic("Take heed, {emph, c? x}.")

#
# now let's add the option "strong"
fun_emph = function(x, options, ...) {
  if("strong" \%in\% options){
    paste0("***", x, "***")
  } else {
    paste0("*", x, "*")
  }
}

string_magic_register_fun(fun_emph, "emph", "strong")

x = string_vec("right, now")
string_magic("Take heed, {emph.strong, c? x}.")

#
# now let's add an argument
fun_emph = function(x, argument, options, ...){
  arg = argument
  if(nchar(arg) == 0) arg = "*"
  
  if("strong" \%in\% options){
    arg = paste0(rep(arg, 3), collapse = "")
  }
  
  paste0(arg, x, arg)
}

string_magic_register_fun(fun_emph, "emph", "strong")

x = string_vec("right, now")
string_magic("Take heed, {'_'emph.s, c? x}.")

#
# using string_magic_register_ops
#

# create a 'header' maker
string_magic_register_ops("tws, '# 'paste, ' 'paste.right, '40|-'fill", "h1")
cat_magic("{h1 ! My title}\n my content")




}
\seealso{
Other related to string_magic: 
\code{\link{string_magic_alias}()}
}
\author{
Laurent R. Berge
}
\concept{related to string_magic}
