% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bk_control_limit.R
\name{bk_control_limit}
\alias{bk_control_limit}
\title{Determine control limits for BK-CUSUM by simulation}
\usage{
bk_control_limit(time, alpha = 0.05, psi, n_sim = 200, theta, coxphmod,
  baseline_data, cbaseh, inv_cbaseh, interval = c(0, 9e+12),
  h_precision = 0.01, seed = 1041996, pb = FALSE, chartpb = FALSE,
  assist)
}
\arguments{
\item{time}{A numeric value over which the type I error \code{alpha} must be restricted.}

\item{alpha}{A proportion between 0 and 1 indicating the required maximal type I error.
Default is 0.05.}

\item{psi}{A numeric value indicating the estimated Poisson arrival rate of subjects
at their respective units. Can be determined using
\code{\link[success:parameter_assist]{parameter_assist()}}.}

\item{n_sim}{An integer value indicating the amount of units to generate for the
determination of the control limit. Larger values yield more precise control limits,
but increase computation times. Default is 200.}

\item{theta}{The expected log-hazard ratio \eqn{\theta}{\theta} under the alternative hypothesis.
If \eqn{\theta >= 0}{\theta >= 0}, the chart will try to detect an increase
in hazard ratio (upper one-sided). If \eqn{\theta < 0}{\theta < 0},
the chart will look for a decrease in hazard ratio (lower one-sided).}

\item{coxphmod}{A Cox proportional hazards regression model as
produced by
the function \code{\link[survival:coxph]{coxph()}}. Suggested: \cr
\code{coxph(Surv(survtime, censorid) ~ covariates, data = data)}. \cr
Alternatively, a list with the following elements:
\describe{
\item{\code{formula}:}{a \code{\link[stats:formula]{formula()}} in the form \code{~ covariates};}
\item{\code{coefficients}:}{a named vector specifying risk adjustment coefficients
for covariates. Names must be the same as in \code{formula} and colnames of \code{data}.}
}}

\item{baseline_data}{(optional): A \code{data.frame} used for covariate resampling
with rows representing subjects and at least the
following named columns: \describe{
\item{\code{entrytime}:}{time of entry into study (numeric);}
\item{\code{survtime}:}{time from entry until event (numeric);}
\item{\code{censorid}:}{censoring indicator (0 = right censored, 1 = observed),
(integer).}
} and optionally additional covariates used for risk-adjustment. Can only be specified
in combination with \code{coxphmod}.}

\item{cbaseh}{A function that returns the unadjusted cumulative
baseline hazard \eqn{H_0(t)}{H_0(t)}. If \code{cbaseh} is missing but
\code{coxphmod} has been
specified as a survival object, this baseline hazard rate will be determined
using the provided \code{coxphmod}.}

\item{inv_cbaseh}{(optional): A function that returns the unadjusted inverse cumulative
baseline
hazard \eqn{H^{-1}_0(t)}{H_0^-1(t)}. If \code{inv_cbaseh} is missing, it will be
determined from \code{cbaseh} numerically.}

\item{interval}{(optional): Interval in which survival times should be solved for numerically.}

\item{h_precision}{(optional): A numerical value indicating how precisely the control limit
should be determined. By default, control limits will be determined up to 2 significant digits.}

\item{seed}{(optional): A numeric seed for survival time generation. Default
is 01041996 (my birthday).}

\item{pb}{(optional): A boolean indicating whether a progress bar should
be shown. Default is \code{FALSE}.}

\item{chartpb}{(optional): A boolean indicating whether progress bars should
be displayed for the constructions of the charts. Default is \code{FALSE}.}

\item{assist}{(optional): Output of the function \code{\link[success:parameter_assist]{parameter_assist()}}}
}
\value{
A list containing three components:
\itemize{
\item \code{call}: the call used to obtain output;
\item \code{charts}: A list of length \code{n_sim} containing the constructed charts;
\item \code{data}: A \code{data.frame} containing the in-control generated data.
\item \code{h}: Determined value of the control limit.
\item \code{achieved_alpha}: Achieved type I error on the sample of
\code{n_sim} simulated units.
}
}
\description{
This function can be used to determine control limits for the
BK-CUSUM (\code{\link[success]{bk_cusum}}) procedure by restricting the type I error \code{alpha} of the
procedure over \code{time}.
}
\details{
This function performs 3 steps to determine a suitable control limit.
\itemize{
\item Step 1: Generates \code{n_sim} in-control units (failure rate as baseline).
If \code{data} is provided, subject covariates are resampled from the data set.
\item Step 2: Determines chart values for all simulated units.
\item Step 3: Determines control limits such that at most a proportion \code{alpha}
of all units cross the control limit.
} The generated data as well as the charts are also returned in the output.
}
\examples{
require(survival)

#Determine a cox proportional hazards model for risk-adjustment
exprfit <- as.formula("Surv(survtime, censorid) ~ age + sex + BMI")
tcoxmod <- coxph(exprfit, data= surgerydat)

#Determine a control limit restricting type I error to 0.1 over 500 days
#with specified cumulative hazard function without risk-adjustment
a <- bk_control_limit(time = 500, alpha = 0.1, theta = log(2),
cbaseh = function(t) chaz_exp(t, lambda = 0.02),
inv_cbaseh = function(t) inv_chaz_exp(t, lambda = 0.02), psi = 0.5,
n_sim = 10)

#Determine a control limit restricting type I error to 0.1 over 500 days
#using the risk-adjusted cumulative hazard determined using coxph()
b <- bk_control_limit(time = 500, alpha = 0.1, theta = log(2),
coxphmod = tcoxmod, psi = 0.5, n_sim = 10)

print(a$h)
print(b$h)
}
\seealso{
\code{\link[success]{bk_cusum}}

Other control limit simulation: 
\code{\link{bernoulli_control_limit}()},
\code{\link{cgr_control_limit}()}
}
\author{
Daniel Gomon
}
\concept{control limit simulation}
