% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/das_effort.R
\name{das_effort}
\alias{das_effort}
\alias{das_effort.data.frame}
\alias{das_effort.das_df}
\title{Summarize DAS effort}
\usage{
das_effort(x, ...)

\method{das_effort}{data.frame}(x, ...)

\method{das_effort}{das_df}(
  x,
  method = c("condition", "equallength", "section"),
  conditions = NULL,
  strata.files = NULL,
  distance.method = c("greatcircle", "lawofcosines", "haversine", "vincenty"),
  seg0.drop = FALSE,
  comment.drop = FALSE,
  event.touse = NULL,
  num.cores = NULL,
  ...
)
}
\arguments{
\item{x}{an object of class \code{das_df},
or a data frame that can be coerced to class \code{das_df}}

\item{...}{arguments passed to the specified chopping function,
such as \code{seg.km} or \code{seg.min.km}}

\item{method}{character; method to use to chop DAS data into effort segments
Can be "condition", "equallength", "section",
or any partial match thereof (case sensitive)}

\item{conditions}{character vector of names of conditions to include in segdata output.
These values must be column names from the output of \code{\link{das_process}},
e.g. 'Bft', 'SwellHght', etc.
If \code{method == "condition"}, then these also are the conditions which
trigger segment chopping when they change. Only the following conditions can be used for chopping:
'Bft', 'SwellHght', 'RainFog', 'HorizSun', 'VertSun', 'Glare', 'Vis', 'Course', 'SpdKt'}

\item{strata.files}{list of path(s) of the CSV file(s) with points defining each stratum.
The CSV files must contain headers and be a closed polygon.
The list should be named; see the Details section.
If \code{NULL} (the default), then no effort segments are not classified by strata.}

\item{distance.method}{character;
method to use to calculate distance between lat/lon coordinates.
Can be "greatcircle", "lawofcosines", "haversine", "vincenty",
or any partial match thereof (case sensitive).
Default is "greatcircle"}

\item{seg0.drop}{logical; flag indicating whether or not to drop segments
of length 0 that contain no sighting (S, K, M, G, t) events.
Default is \code{FALSE}}

\item{comment.drop}{logical; flag indicating if comments ("C" events)
should be ignored (i.e. position information should not be used)
when segment chopping. Default is \code{FALSE}}

\item{event.touse}{character vector of events to use to determine
segment lengths; overrides \code{comment.drop}.
If \code{NULL} (the default), then all on effort events are used.
If used, this argument must include at least R, E, S, and A events,
and cannot include ? or 1:8 events}

\item{num.cores}{Number of CPUs to over which to distribute computations.
Defaults to \code{NULL}, which uses one fewer than the number of cores
reported by \code{\link[parallel]{detectCores}}.
Using 1 core likely will be faster for smaller datasets}
}
\value{
List of three data frames:
  \itemize{
    \item segdata: one row for every segment, and columns for information including
      unique segment number (segnum), the corresponding effort section (section_id),
      the segment index within the corresponding effort section (section_sub_id),
      the starting and ending line of the segment in the DAS file (stlin, endlin),
      start/end/midpoint coordinates(lat1/lon1, lat2/lon2, and mlat/mlon, respectively),
      the start/end/midpoint date/time of the segment (DateTime1, DateTime2, and mDateTime, respectively;
      mDateTime is the average of DateTime1 and DateTime2), segment length (dist),
      conditions (e.g. Beaufort), and, if applicable, stratum (InStratumName).
    \item sightinfo: details for all sightings in \code{x}, including:
      the unique segment number it is associated with, segment mid points (lat/lon),
      the 'included' column described in the 'Details' section,
      and the output information described in \code{\link{das_sight}} for \code{return.format} is "default"
    \item randpicks: see \code{\link{das_chop_equallength}};
      \code{NULL} if using "condition" method
  }
}
\description{
Chop DAS data into effort segments
}
\details{
This is the top-level function for chopping processed DAS data
  into modeling segments (henceforth 'segments'), and assigning sightings
  and related information (e.g., weather conditions) to each segment.
  This function returns data frames with all relevant information for the
  effort segments and associated sightings ('segdata' and 'sightinfo', respectively).
  Before chopping, the DAS data is filtered for events (rows) where either
  the 'OnEffort' column is \code{TRUE} or the 'Event' column "E".
  In other words, the data is filtered for continuous effort sections (henceforth 'effort sections'),
  where effort sections run from "R" to "E" events (inclusive),
  and then passed to the chopping function specified using \code{method}.
  Note that while B events immediately preceding an R are on effort,
  they are ignored during effort chopping.
  In addition, all on effort events (other than ? and numeric events)
  with \code{NA} DateTime, Lat, or Lon values are verbosely removed.

  If \code{strata.files} is not \code{NULL}, then the effort lines
  will be split by the user-provided stratum (strata).
  In this case, a column 'stratum' will be added to the end of the segdata
  data frame with the user-provided name of the stratum that the segment was in,
  or \code{NA} if the segment was not in any of the strata.
  If no name was provided for the stratum in \code{strata.files},
  then the value will be "Stratum#",
  where "#" is the index of the applicable stratum in \code{strata.files}.
  While the user can provide as many strata as they want,
  these strata can share boundaries but they cannot overlap.
  See \code{\link{das_effort_strata}} for more details.

  The following chopping methods are currently available:
  "condition", "equallength", and "section.
  When using the "condition" method, effort sections are chopped
  into segments every time a condition changes,
  thereby ensuring that the conditions are consistent across the entire segment.
  See \code{\link{das_chop_condition}} for more details about this method,
  including arguments that must be passed to it via the argument \code{...}

  The "equallength" method consists of
  chopping effort sections into equal-length segments of length \code{seg.km},
  and doing a weighted average of the conditions for the length of that segment.
  See \code{\link{das_chop_equallength}} for more details about this method,
  including arguments that must be passed to it via the argument \code{...}

  The "section" method involves 'chopping' the effort into continuous effort sections,
  i.e. each continuous effort section is a single effort segment.
  See \code{\link{das_chop_section}} for more details about this method.

  The distance between the lat/lon points of subsequent events
  is calculated using the method specified in \code{distance.method}.
  If "greatcircle", \code{\link{distance_greatcircle}} is used,
  while \code{\link[swfscMisc]{distance}} is used otherwise.
  See \code{\link{das_sight}} for how the sightings are processed.

  The sightinfo data frame includes the column 'included',
  which is used in \code{\link{das_effort_sight}} when summarizing
  the number of sightings and animals for selected species.
  \code{\link{das_effort_sight}} is a separate function to allow users to
  personalize the included values as desired for their analysis.
  By default, i.e. in the output of this function, 'included' is \code{TRUE} if:
  the sighting was made when on effort,
  by a standard observer (see \code{\link{das_sight}}),
  and in a Beaufort sea state less than or equal to five.
}
\examples{
y <- system.file("das_sample.das", package = "swfscDAS")
y.proc <- das_process(y)

# Using "condition" method
das_effort(
  y.proc, method = "condition", conditions = c("Bft", "SwellHght", "Vis"),
  seg.min.km = 0.05, num.cores = 1
)

# Using "section" method
das_effort(y.proc, method = "section", num.cores = 1)

\donttest{
# Using "equallength" method
y.rand <- system.file("das_sample_randpicks.csv", package = "swfscDAS")
das_effort(
  y.proc, method = "equallength", seg.km = 10, randpicks.load = y.rand,
  num.cores = 1
)

# Using "section" method and chop by strata
stratum.file <- system.file("das_sample_stratum.csv", package = "swfscDAS")
das_effort(
  y.proc, method = "section", strata.files = list(Poly1 = stratum.file),
  num.cores = 1
)
}

}
\seealso{
Internal functions called by \code{das_effort}:
  \code{\link{das_chop_condition}}, \code{\link{das_chop_equallength}},
  \code{\link{das_chop_section}}, \code{\link{das_segdata}}
}
