% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AllGenerics.R, R/alpha-diversity.R
\docType{methods}
\name{alpha-diversity}
\alias{alpha-diversity}
\alias{diversity}
\alias{diversity-method}
\alias{evenness}
\alias{evenness-method}
\alias{rarefaction}
\alias{rarefaction-method}
\alias{richness}
\alias{richness-method}
\alias{richness,CountMatrix-method}
\alias{rarefaction,CountMatrix-method}
\alias{diversity,CountMatrix-method}
\alias{evenness,CountMatrix-method}
\title{\eqn{\alpha}{\alpha}-diversity}
\usage{
diversity(object, ...)

evenness(object, ...)

rarefaction(object, ...)

richness(object, ...)

\S4method{richness}{CountMatrix}(object, method = c("margalef",
  "menhinick"), simplify = FALSE)

\S4method{rarefaction}{CountMatrix}(object, sample)

\S4method{diversity}{CountMatrix}(object, method = c("berger",
  "brillouin", "mcintosh", "shannon", "simpson"), simplify = FALSE, ...)

\S4method{evenness}{CountMatrix}(object, method = c("brillouin",
  "mcintosh", "shannon", "simpson"), simplify = FALSE, ...)
}
\arguments{
\item{object}{A \eqn{m \times p}{m x p} matrix of count data.}

\item{...}{Further arguments passed to other methods.}

\item{method}{A \code{\link{character}} string specifiying the index to be
computed. This must be one or more of "\code{berger}",
"\code{brillouin}", "\code{margalef}", "\code{mcintosh}",
"\code{menhinick}", "\code{shannon}", "\code{simpson}" (see details).
Any unambiguous substring can be given.}

\item{simplify}{A \code{\link{logical}} scalar: should the result be
simplified to a matrix? The default value, \code{FALSE}, returns a list.}

\item{sample}{A length-one \code{\link{numeric}} vector giving the sub-sample
size.}
}
\value{
\code{rarefaction} returns a numeric vector.

 If \code{simplify} is \code{FALSE}, then \code{diversity}, \code{evenness}
 and \code{richness} return a list (default), else return a matrix.
}
\description{
Measures within-sample diversity. \code{diversity} returns a diversity or
dominance index. \code{evenness} returns an evenness measure. \code{richness}
returns sample richness. \code{rarefaction} returns Hurlbert's unbiaised
estimate of Sander's rarefaction.
}
\note{
Ramanujan approximation is used for \eqn{x!} computation if \eqn{x > 170}.
}
\section{Richness and Rarefaction}{

 The number of different taxa, provides an instantly comprehensible
 expression of diversity. While the number of taxa within a sample
 is easy to ascertain, as a term, it makes little sense: some taxa
 may not have been seen, or there may not be a fixed number of taxa
 (e.g. in an open system; Peet 1974). As an alternative, \emph{richness}
 (\eqn{S}) can be used for the concept of taxa number (McIntosh 1967).

 It is not always possible to ensure that all sample sizes are equal
 and the number of different taxa increases with sample size and
 sampling effort (Magurran 1988). Then, \emph{rarefaction} (\eqn{E(S)}) is
 the number of taxa expected if all samples were of a standard size (i.e.
 taxa per fixed number of individuals). Rarefaction assumes that imbalances
 between taxa are due to sampling and not to differences in actual
 abundances.

 The following richness measures are available:
 \describe{
  \item{margalef}{Margalef richness index:
   \eqn{D_{Mg} = \frac{S - 1}{\ln N}}{D = (S - 1) / ln N}}
  \item{menhinick}{Menhinick richness index:
   \eqn{D_{Mn} = \frac{S}{\sqrt{N}}}{D = S / \sqrt N}}
 }
}

\section{Diversity and Evenness}{

 \emph{Diversity} measurement assumes that all individuals in a specific
 taxa are equivalent and that all types are equally different from each
 other (Peet 1974). A measure of diversity can be achieved by using indices
 built on the relative abundance of taxa. These indices (sometimes referred
 to as non-parametric indices) benefit from not making assumptions about the
 underlying distribution of taxa abundance: they only take evenness and
 \emph{richness} into account. Peet (1974) refers to them as indices of
 \emph{heterogeneity}.

 Diversity indices focus on one aspect of the taxa abundance and emphasize
 either \emph{richness} (weighting towards uncommon taxa)
 or dominance (weighting towards abundant taxa; Magurran 1988).

 \emph{Evenness} is a measure of how evenly individuals are distributed
 across the sample.

 The following heterogeneity index and corresponding evenness measures
 are available (see Magurran 1988 for details):
 \describe{
  \item{berger}{Berger-Parker dominance index. The Berger-Parker index
   expresses the proportional importance of the most abundant type.}
  \item{brillouin}{Brillouin diversity index. The Brillouin index describes a
   known collection: it does not assume random sampling in an infinite
   population. Pielou (1975) and Laxton (1978) argues for the use of the
   Brillouin index in all circumstances, especially in preference to the
   Shannon index.}
  \item{mcintosh}{McIntosh dominance index. The McIntosh index expresses the
   heterogeneity of a sample in geometric terms. It describes the sample as a
   point of a S-dimensional hypervolume and uses the Euclidean distance of
   this point from the origin.}
  \item{shannon}{Shannon-Wiener diversity index. The Shannon index assumes
   that individuals are randomly sampled from an infinite population and that
   all taxa are represented in the sample (it does not reflect the
   sample size). The main source of error arises from the failure to include
   all taxa in the sample: this error increases as the proportion of species
   discovered in the sample declines (Peet 1974, Magurran 1988). The
   maximum likelihood estimator (MLE) is used for the relative abundance,
   this is known to be negatively biased.}
  \item{simpson}{Simpson dominance index for finite sample. The Simpson index
   expresses the probability that two individuals randomly picked from a
   finite sample belong to two different types. It can be interpreted as the
   weighted mean of the proportional abundances.}
 }

 The \code{berger}, \code{mcintosh} and \code{simpson} methods return a
 \emph{dominance} index, not the reciprocal or inverse form usually adopted,
 so that an increase in the value of the index accompanies a decrease in
 diversity.
}

\examples{
# Richness
## Margalef and Menhinick index
## Data from Magurran 1988, p. 128-129
trap <- CountMatrix(data = c(9, 3, 0, 4, 2, 1, 1, 0, 1, 0, 1, 1,
                             1, 0, 1, 0, 0, 0, 1, 2, 0, 5, 3, 0),
                    nrow = 2, byrow = TRUE, dimnames = list(c("A", "B"), NULL))

richness(trap, method = c("margalef", "menhinick"), simplify = TRUE)
## A 2.55 1.88
## B 1.95 1.66

# Rarefaction
rarefaction(trap, sample = 13) # 6.56

# Diversity
## Shannon diversity index
## Data from Magurran 1988, p. 145-149
birds <- CountMatrix(
  data = c(35, 26, 25, 21, 16, 11, 6, 5, 3, 3,
           3, 3, 3, 2, 2, 2, 1, 1, 1, 1, 0, 0,
           30, 30, 3, 65, 20, 11, 0, 4, 2, 14,
           0, 3, 9, 0, 0, 5, 0, 0, 0, 0, 1, 1),
  nrow = 2, byrow = TRUE, dimnames = list(c("oakwood", "spruce"), NULL))

diversity(birds, "shannon") # 2.40 2.06
evenness(birds, "shannon") # 0.80 0.78

## Brillouin diversity index
## Data from Magurran 1988, p. 150-151
moths <- CountMatrix(data = c(17, 15, 11, 4, 4, 3, 3, 3, 2, 2, 1, 1, 1),
                     nrow = 1, byrow = TRUE)

diversity(moths, "brillouin") # 1.88
evenness(moths, "brillouin") # 0.83

## Simpson dominance index
## Data from Magurran 1988, p. 152-153
trees <- CountMatrix(
  data = c(752, 276, 194, 126, 121, 97, 95, 83, 72, 44, 39,
           16, 15, 13, 9, 9, 9, 8, 7, 4, 2, 2, 1, 1, 1),
  nrow = 1, byrow = TRUE
)

diversity(trees, "simpson") # 1.19
evenness(trees, "simpson") # 0.21

## McIntosh dominance index
## Data from Magurran 1988, p. 154-155
invertebrates <- CountMatrix(
  data = c(254, 153, 90, 69, 68, 58, 51, 45, 40, 39, 25, 23, 19, 18, 16, 14, 14,
           11, 11, 11, 11, 10, 6, 6, 6, 6, 5, 3, 3, 3, 3, 3, 1, 1, 1, 1, 1, 1),
  nrow = 1, byrow = TRUE
)

diversity(invertebrates, "mcintosh") # 0.71
evenness(invertebrates, "mcintosh") # 0.82

## Berger-Parker dominance index
## Data from Magurran 1988, p. 156-157
fishes <- CountMatrix(
  data = c(394, 3487, 275, 683, 22, 1, 0, 1, 6, 8, 1, 1, 2,
           1642, 5681, 196, 1348, 12, 0, 1, 48, 21, 1, 5, 7, 3,
           90, 320, 180, 46, 2, 0, 0, 1, 0, 0, 2, 1, 5,
           126, 17, 115, 436, 27, 0, 0, 3, 1, 0, 0, 1, 0,
           32, 0, 0, 5, 0, 0, 0, 0, 13, 9, 0, 0, 4),
  nrow = 5, byrow = TRUE,
  dimnames = list(c("station 1", "station 2", "station 3",
                    "station 4", "station 5"), NULL)
)

diversity(fishes, "berger") # 0.71 0.63 0.50 0.60 0.51
}
\references{
Berger, W. H. & Parker, F. L. (1970). Diversity of Planktonic Foraminifera
 in Deep-Sea Sediments. \emph{Science}, 168(3937), 1345-1347.
 DOI: \href{https://doi.org/10.1126/science.168.3937.1345}{10.1126/science.168.3937.1345}.

 Brillouin, L. (1956). \emph{Science and information theory}. New York:
 Academic Press.

 Hurlbert, S. H. (1971). The Nonconcept of Species Diversity: A Critique and
 Alternative Parameters. \emph{Ecology}, 52(4), 577-586.
 DOI: \href{https://doi.org/10.2307/1934145}{10.2307/1934145}.

 Laxton, R. R. (1978). The measure of diversity. \emph{Journal of Theoretical
 Biology}, 70(1), 51-67.
 DOI: \href{https://doi.org/10.1016/0022-5193(78)90302-8}{10.1016/0022-5193(78)90302-8}.

 Magurran, A. E. (1988). \emph{Ecological Diversity and its Measurement}.
 Princeton, NJ: Princeton University Press.
 DOI:\href{https://doi.org/10.1007/978-94-015-7358-0}{10.1007/978-94-015-7358-0}.

 Margalef, R. (1958). Information Theory in Ecology. \emph{General Systems},
 3, 36-71.

 McIntosh, R. P. (1967). An Index of Diversity and the Relation of Certain
 Concepts to Diversity. \emph{Ecology}, 48(3), 392-404.
 DOI: \href{https://doi.org/10.2307/1932674}{10.2307/1932674}.

 Menhinick, E. F. (1964). A Comparison of Some Species-Individuals Diversity
 Indices Applied to Samples of Field Insects. \emph{Ecology}, 45(4), 859-861.
 DOI: \href{https://doi.org/10.2307/1934933}{10.2307/1934933}.

 Peet, R. K. (1974). The Measurement of Species Diversity. \emph{Annual
 Review of Ecology and Systematics}, 5(1), 285-307.
 DOI: \href{https://doi.org/10.1146/annurev.es.05.110174.001441}{10.1146/annurev.es.05.110174.001441}.

 Pielou, E. C. (1975). \emph{Ecological Diversity}. New York: Wiley.
 DOI: \href{https://doi.org/10.4319/lo.1977.22.1.0174b}{10.4319/lo.1977.22.1.0174b}

 Sander, H. L. (1968). Marine Benthic Diversity: A Comparative Study.
 \emph{The American Naturalist}, 102(925), 243-282.

 Shannon, C. E. (1948). A Mathematical Theory of Communication. \emph{The
 Bell System Technical Journal}, 27, 379-423.
 DOI: \href{https://doi.org/10.1002/j.1538-7305.1948.tb01338.x}{10.1002/j.1538-7305.1948.tb01338.x}.

 Simpson, E. H. (1949). Measurement of Diversity. \emph{Nature}, 163(4148),
 688-688. DOI: \href{https://doi.org/10.1038/163688a0}{10.1038/163688a0}.
}
\seealso{
\code{\link[=beta-diversity]{turnover}}
 \code{\link[=beta-diversity]{similarity}}
}
\author{
N. Frerebeau
}
