#' Simple horizontal dead-reckoned track 
#' 
#' This function is used to estimate the simple horizontal dead-reckoned track (pseudo-track) based on speed and heading. This differs from ptrack in that the animals body angle is not considered. This makes it appropriate for animals that do not always move in the direction of their longitudinal axis.
#' 
#' @param A The nx3 acceleration matrix with columns [ax ay az] or acceleration sensor list. Acceleration can be in any consistent unit, e.g., g or m/s^2. 
#' @param M The magnetometer signal matrix, M = [mx,my,mz] in any consistent unit (e.g., in uT or Gauss) or magnetometer sensor list. A and M must have the same size (and so are both measured at the same sampling rate).
#' @param s The forward speed of the animal in m/s. s can be a single number meaning that the animal is assumed to travel at a constant speed. s can also be a vector with the same number of rows as M, e.g., generated by \code{\link[tagtools]{ocdr}}.
#' @param sampling_rate The sampling rate of the sensor data in Hz (samples per second).
#' @param fc (optional) Specifies the cut-off frequency of a low-pass filter to apply to A and M before computing heading. The filter cut-off frequency is in Hz. The filter length is 4*sampling_rate/fc. Filtering adds no group delay. If fc is empty or not given, the default value of 0.2 Hz (i.e., a 5 second time constant) is used.
#' @return Data frame track containing the estimated track in a local level frame. The track is defined as meters of northward and eastward movement (termed 'northing' and 'easting', i.e, columns of track are \code{northing} and \code{easting} relative to the animal's position at the start of the measurements (which is defined as [0,0]). The track sampling rate is the same as for the input data and so each row of track defines the track coordinates at times 0,1/sampling_rate,2/sampling_rate,... relative to the start time of the measurements.
#' @note Frame: This function assumes a [north,east,up] navigation frame and a [forward,right,up] local frame. Both A and M must be rotated if needed to match the animal's cardinal axes otherwise the track will not be meaningful. Unless the local declination angle is also corrected with rotframe, the dead-reckoned track will use magnetic north rather than true north.
#' @note CAUTION: dead-reckoned tracks are usually very inaccurate. 
#' They are useful to get an idea of HOW animals move rather than WHERE they go. Few animals probably travel in exactly the direction of their longitudinal axis. Additionally, measuring the precise orientation of the longitudinal axis of a non-rigid animal is fraught with error. 
#' Moreover, if there is net flow in the medium, the animal will be advected by the flow in addition to its autonomous movement. For swimming animals this can lead to substantial errors. The forward speed is assumed to be with respect to the medium so the track derived here is NOT the 'track-made-good', i.e., the geographic movement of the animal. It estimates the movement of the animal with respect to the medium. 
#' There are numerous other sources of error so use at your own risk!
#' @seealso \code{\link[tagtools]{ptrack}}, \code{\link[tagtools]{fit_tracks}}, \code{\link{track3D}}
#' @export
#' @examples
#' bwhtrack <- htrack(A = beaked_whale$A, M = beaked_whale$M, s = 4)
#' plot(bwhtrack$easting, bwhtrack$northing, xlab = "Easting, m", ylab = "Northing, m")
#'

htrack <- function(A, M, s, sampling_rate = NULL, fc = 0.2) {
  #*******************************
  # input checks
  #*******************************
  if (missing(A) | missing(M) | missing(s)) {
    stop("htrack: inputs for A, M, and s are all required.\n")
  }
  
  if (is.list(A) & is.list(M)) {
    if (A$sampling_rate != M$sampling_rate) {
      stop("htrack: A and M must be at the same sampling rate.\n")
    }
    sampling_rate <- M$sampling_rate 
    M <- M$data 
    A <- A$data
  } else {
    if (missing(sampling_rate)) {
      stop("htrack: sampling_rate is required if A and M are matrices.\n")
    }
  }
  
  #*******************************
  # end of input checks
  #*******************************
  
  # if there are any outages (NaNs) in A or M, interpolate over these and
  # remember the interpolated points to remove them from the DR track later
  A <- interp_nan(A)
  ka <- A$k
  A <- A$data
  
  M <- interp_nan(M)
  km <- M$k
  M <- M$data
  
  nf <- 4 * sampling_rate / fc 
  hd <- m2h(M, A, sampling_rate, fc)$h
  
  if (length(s) == 1){
    s <- matrix(s / sampling_rate, nrow = length(hd), ncol = 2)
    ksused <- FALSE
  } else {
    s <- interp_nan(s)
    ks <- s$k
    ksused <- TRUE
    s <- s$data
    s <- matrix(s / sampling_rate, ncol = 2)
  }
  
  
  track <- apply(s * cbind(cos(hd), sin(hd)), 2, cumsum)
  # replace NAs
  track[ka,] <- NA
  track[km,] <- NA
  if (ksused) {
    track[ks,] <- NA
  }
  
  track <- data.frame(track)
  names(track) <- c('northing', 'easting')
  
  return(track)
}
