% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tcgsa_seq.R
\name{tcgsa_seq}
\alias{tcgsa_seq}
\title{Time-course Gene Set Analyis}
\usage{
tcgsa_seq(y, x, phi, genesets, indiv = rep(1, nrow(x)),
  Sigma_xi = diag(ncol(phi)), which_test = c("permutation", "asymptotic"),
  which_weights = c("loclin", "voom", "none"), n_perm = 1000,
  preprocessed = FALSE, doPlot = TRUE, gene_based = FALSE, bw = "nrd",
  kernel = c("gaussian", "epanechnikov", "rectangular", "triangular",
  "biweight", "tricube", "cosine", "optcosine"), exact = FALSE,
  transform = FALSE, padjust_methods = c("BH", "BY", "holm", "hochberg",
  "hommel", "bonferroni"), lowess_span = 0.5, homogen_traj = FALSE)
}
\arguments{
\item{y}{a numeric matrix of size \code{G x n} containing the raw RNA-seq counts or
preprocessed expressions from \code{n} samples for \code{G} genes.}

\item{x}{a numeric matrix of size \code{n x p} containing the model covariates from
\code{n} samples (design matrix).}

\item{phi}{a numeric design matrix of size \code{n x K} containing the \code{K} variables
to be tested}

\item{genesets}{either a vector of index or subscripts that defines which columns of \code{y}
constitute the invesigated geneset. Can also be a \code{list} of index when several genesets are
tested at once, such as the first element of a \code{\link[GSA:GSA.read.gmt]{gmt}} object.
If \code{NULL}, then genewise p-values are returned.}

\item{indiv}{a vector of length \code{n} containing the information for
attributing each sample to one of the studied individuals. Coerced
to be a \code{factor}.}

\item{Sigma_xi}{a matrix of size \code{K x K} containing the covariance matrix
of the \code{K} random effects.}

\item{which_test}{a character string indicating which method to use to approximate
the variance component score test, either \code{"permutation"} or \code{"asymptotic"}.
Default is \code{"permutation"}.}

\item{which_weights}{a character string indicating which method to use to estimate
the mean-variance relationship wheights. Possibilities are \code{"loclin"},
\code{"voom"} or \code{"none"} (in which case no weighting is performed).
Default is \code{"loclin"}.
See \code{\link{sp_weights}} and \code{\link{voom_weights}} for details.}

\item{n_perm}{the number of perturbations}

\item{preprocessed}{a logical flag indicating whether the expression data have
already been preprocessed (e.g. log2 transformed). Default is \code{FALSE}, in
which case \code{y} is assumed to contain raw counts and is normalized into
log(counts) per million.}

\item{doPlot}{a logical flag indicating whether the mean-variance plot should be drawn.
Default is \code{FALSE}.}

\item{gene_based}{a logical flag used for "loclin" weights, indicating whether to estimate
weights at the gene-level. Default is \code{FALSE}, when weights will be estimated at the
observation-level.}

\item{bw}{a character string indicating the smoothing bandwidth selection method to use. See
\code{\link[stats]{bandwidth}} for details. Possible values are \code{"ucv"}, \code{"SJ"},
\code{"bcv"}, \code{"nrd"} or \code{"nrd0"}}

\item{kernel}{a character string indicating which kernel should be used.
Possibilities are \code{"gaussian"}, \code{"epanechnikov"}, \code{"rectangular"},
\code{"triangular"}, \code{"biweight"}, \code{"tricube"}, \code{"cosine"},
\code{"optcosine"}. Default is \code{"gaussian"} (NB: \code{"tricube"} kernel
corresponds to the loess method).}

\item{exact}{a logical flag indicating whether the non-parametric weights accounting
for the mean-variance relationship should be computed exactly or extrapolated
from the interpolation of local regression of the mean against the
variance. Default is \code{FALSE}, which uses interporlation (faster computation).}

\item{transform}{a logical flag used for "loclin" weights, indicating whether values should be
transformed to uniform for the purpose of local linear smoothing. This may be helpful if tail
observations are sparse and the specified bandwidth gives suboptimal performance there.
Default is \code{FALSE}.}

\item{padjust_methods}{multiple testing correction method used if \code{genesets}
is a list. Default is "BH", i.e. Benjamini-Hochberg procedure for contolling the FDR.
Other possibilities are: \code{"holm"}, \code{"hochberg"}, \code{"hommel"},
\code{"bonferroni"} or \code{"BY"} (for Benjamini-Yekutieli procedure).}

\item{lowess_span}{smoother span for the lowess function, between 0 and 1. This gives
the proportion of points in the plot which influence the smooth at each value.
Larger values give more smoothness. Only used if \code{which_weights} is \code{"voom"}.
Default is \code{0.5}.}

\item{homogen_traj}{a logical flag indicating whether trajectories should be considered homogeneous.
Default is \code{FALSE} in which case trajectories are not only tested for trend, but also for heterogeneity.}
}
\value{
A list with the following elements:\itemize{
  \item \code{which_test}:
  \item \code{preprocessed}:
  \item \code{n_perm}:
  \item \code{pval}: associated p-value
}
}
\description{
Wrapper function for performing gene set analysis of longitudinal RNA-seq data
}
\examples{
#rm(list=ls())
n <- 200
r <- 12
t <- matrix(rep(1:3), 4, ncol=1, nrow=r)
sigma <- 0.4
b0 <- 1

#under the null:
b1 <- 0

y.tilde <- b0 + b1*t + rnorm(r, sd = sigma)
y <- t(matrix(rnorm(n*r, sd = sqrt(sigma*abs(y.tilde))), ncol=n, nrow=r) +
      matrix(rep(y.tilde, n), ncol=n, nrow=r))
x <- matrix(1, ncol=1, nrow=r)

#run test
res <- tcgsa_seq(y, x, phi=t, genesets=lapply(0:9, function(x){x*10+(1:10)}),
                        Sigma_xi=matrix(1), indiv=rep(1:4, each=3), which_test="asymptotic",
                        which_weights="none", preprocessed=TRUE)
res_genes <- tcgsa_seq(y, x, phi=t, genesets=NULL,
                      Sigma_xi=matrix(1), indiv=rep(1:4, each=3), which_test="asymptotic",
                      which_weights="none", preprocessed=TRUE)
quantile(res_genes$pvals[, "rawPval"])
}
\references{
Agniel D, Hejblum BP, Variance component score test for
time-course gene set analysis of longitudinal RNA-seq data, \emph{submitted}, 2016.

Law, C. W., Chen, Y., Shi, W., & Smyth, G. K. (2014). voom: Precision
weights unlock linear model analysis tools for RNA-seq read counts. \emph{Genome
Biology}, 15(2), R29.
}
\seealso{
\code{\link{sp_weights}} \code{\link{vc_test_perm}} \code{\link{vc_test_asym}} \code{\link{p.adjust}}
}

