% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tclust.R
\name{tclust}
\alias{tclust}
\alias{print.tclust}
\title{TCLUST method for robust clustering}
\usage{
tclust(
  x,
  k,
  alpha = 0.05,
  nstart = 500,
  niter1 = 3,
  niter2 = 20,
  nkeep = 5,
  iter.max,
  equal.weights = FALSE,
  restr = c("eigen", "deter"),
  restr.fact = 12,
  cshape = 1e+10,
  opt = c("HARD", "MIXT"),
  center = FALSE,
  scale = FALSE,
  store_x = TRUE,
  parallel = FALSE,
  n.cores = -1,
  zero_tol = 1e-16,
  drop.empty.clust = TRUE,
  trace = 0
)
}
\arguments{
\item{x}{A matrix or data.frame of dimension n x p, containing the observations (row-wise).}

\item{k}{The number of clusters initially searched for.}

\item{alpha}{The proportion of observations to be trimmed.}

\item{nstart}{The number of random initializations to be performed.}

\item{niter1}{The number of concentration steps to be performed for the nstart initializations.}

\item{niter2}{The maximum number of concentration steps to be performed for the 
\code{nkeep} solutions kept for further iteration. The concentration steps are 
stopped, whenever two consecutive steps lead to the same data partition.}

\item{nkeep}{The number of iterated initializations (after niter1 concentration 
steps) with the best values in the target function that are kept for further iterations}

\item{iter.max}{(deprecated, use the combination \code{nkeep, niter1 and niter2}) 
The maximum number of concentration steps to be performed.
The concentration steps are stopped, whenever two consecutive steps lead
to the same data partition.}

\item{equal.weights}{A logical value, specifying whether equal cluster weights 
shall be considered in the concentration and assignment steps.}

\item{restr}{Restriction type to control relative cluster scatters. 
The default value is \code{restr="eigen"}, so that the maximal ratio between 
the largest and the smallest of the scatter matrices eigenvalues is constrained 
to be smaller then or equal to \code{restr.fact} 
(Garcia-Escudero, Gordaliza, Matran, and Mayo-Iscar, 2008). 
Alternatively, \code{restr="deter"} imposes that the maximal ratio between 
the largest and the smallest of the scatter matrices determinants is smaller 
or equal than \code{restr.fact} (see Garcia-Escudero, Mayo-Iscar and Riani, 2020)}

\item{restr.fact}{The constant \code{restr.fact >= 1} constrains the allowed 
differences among group scatters in terms of eigenvalues ratio
(if \code{restr="eigen"}) or determinant ratios (if \code{restr="deter"}). Larger values 
imply larger differences of group scatters, a value of 1 specifies the 
strongest restriction.}

\item{cshape}{constraint to apply to the shape matrices, \code{cshape >= 1}, 
(see Garcia-Escudero, Mayo-Iscar and Riani, 2020)). 
This options only works if \code{restr=='deter'}. In this case the default 
value is \code{cshape=1e10} to ensure the procedure is (virtually) affine equivariant. 
On the other hand, \code{cshape} values close to 1 would force the clusters to 
be almost spherical (without necessarily the same scatters if \code{restr.fact} 
is strictly greater than 1).}

\item{opt}{Define the target function to be optimized. A classification likelihood 
target function is considered if \code{opt="HARD"} and a mixture classification 
likelihood if \code{opt="MIXT"}.}

\item{center}{Optional centering of the data: a function or a vector of length p 
which can optionally be specified for centering x before calculation}

\item{scale}{Optional scaling of the data: a function or a vector of length p 
which can optionally be specified for scaling x before calculation}

\item{store_x}{A logical value, specifying whether the data matrix \code{x} shall be 
included in the result object. By default this value is set to \code{TRUE}, because 
some of the plotting functions depend on this information. However, when big data 
matrices are handled, the result object's size can be decreased noticeably 
when setting this parameter to \code{FALSE}.}

\item{parallel}{A logical value, specifying whether the nstart initializations should be done in parallel.}

\item{n.cores}{The number of cores to use when paralellizing, only taken into account if parallel=T.}

\item{zero_tol}{The zero tolerance used. By default set to 1e-16.}

\item{drop.empty.clust}{Logical value specifying, whether empty clusters shall be 
omitted in the resulting object. (The result structure does not contain center 
and covariance estimates of empty clusters anymore. Cluster names are reassigned 
such that the first l clusters (l <= k) always have at least one observation.}

\item{trace}{Defines the tracing level, which is set to 0 by default. Tracing level 1 
gives additional information on the stage of the iterative process.}
}
\value{
The function returns the following values:
\itemize{
    \item cluster - A numerical vector of size \code{n} containing the cluster assignment 
         for each observation. Cluster names are integer numbers from 1 to k, 0 indicates 
         trimmed observations. Note that it could be empty clusters with no observations 
         when \code{equal.weights=FALSE}.
    \item obj - The value of the objective function of the best (returned) solution.
    \item size - An integer vector of size k, returning the number of observations contained by each cluster.
    \item weights - Vector of Cluster weights
    \item centers - A matrix of size p x k containing the centers (column-wise) of each cluster. 
    \item cov - 	An array of size p x p x k containing the covariance matrices of each cluster. 
    \item code - A numerical value indicating if the concentration steps have 
         converged for the returned solution (2).
    \item posterior - A matrix with k columns that contains the posterior 
         probabilities of membership of each observation (row-wise) to the \code{k} 
         clusters. This posterior probabilities are 0-1 values in the 
         \code{opt="HARD"} case. Trimmed observations have 0 membership probabilities 
         to all clusters.
    \item cluster.ini - A matrix with nstart rows and number of columns equal to 
         the number of observations and where each row shows the final clustering 
         assignments (0 for trimmed observations) obtained after the \code{niter1} 
         iteration of the \code{nstart} random initializations.
    \item obj.ini - A numerical vector of length \code{nstart} containing the values 
         of the target function obtained after the \code{niter1} iteration of the 
         \code{nstart} random initializations.
    \item x - The input data set.
    \item k - The input number of clusters.
    \item alpha - The input trimming level.
}
}
\description{
This function searches for \code{k} (or less) clusters with 
 different covariance structures in a data matrix \code{x}. Relative cluster 
 scatter can be restricted when \code{restr="eigen"} by constraining the ratio 
 between the largest and the smallest of the scatter matrices eigenvalues 
 by a constant value \code{restr.fact}. Relative cluster scatters can be also 
 restricted with \code{restr="deter"} by constraining the ratio between the 
 largest and the smallest of the scatter matrices' determinants. 

 For robustifying the estimation, a proportion \code{alpha} of observations is trimmed. 
 In particular, the trimmed k-means method is represented by the \code{tclust()} method,
 by setting parameters \code{restr.fact=1}, \code{opt="HARD"} and \code{equal.weights=TRUE}.
}
\details{
The procedure allows to deal with robust clustering with an \code{alpha}
 proportion of trimming level and searching for \code{k} clusters. We are considering 
 classification trimmed likelihood when using \code{opt=”HARD”} so that “hard” or “crisp” 
 clustering assignments are done. On the other hand, mixture trimmed likelihood 
 are applied when using \code{opt=”MIXT”} so providing a kind of clusters “posterior” 
 probabilities for the observations. 
 Relative cluster scatter can be restricted when \code{restr="eigen"} by constraining 
 the ratio between the largest and the smallest of the scatter matrices eigenvalues 
 by a constant value \code{restr.fact}. Setting \code{restr.fact=1}, yields the 
 strongest restriction, forcing all clusters to be spherical and equally scattered. 
 Relative cluster scatters can be also restricted with \code{restr="deter"} by 
 constraining the ratio between the largest and the smallest of the scatter 
 matrices' determinants. 

 This iterative algorithm performs "concentration steps" to improve the current 
 cluster assignments. For approximately obtaining the global optimum, the procedure 
 is randomly initialized \code{nstart} times and \code{niter1} concentration steps are performed for 
 them. The \code{nkeep} most “promising” iterations, i.e. the \code{nkeep} iterated solutions with 
 the initial best values for the target function, are then iterated until convergence 
 or until \code{niter2} concentration steps are done. 

The parameter \code{restr.fact} defines the cluster scatter matrices restrictions, 
 which are applied on all clusters during each concentration step. It restricts 
 the ratio between the maximum and minimum eigenvalue of 
 all clusters' covariance structures to that parameter. Setting \code{restr.fact=1}, 
 yields the strongest restriction, forcing all clusters to be spherical and equally scattered. 

Cluster components with similar sizes are favoured when considering \code{equal.weights=TRUE} 
 while \code{equal.weights=FALSE} admits possible different prior probabilities for 
 the components and it can easily return empty clusters when the number of 
 clusters is greater than apparently needed.
}
\examples{

 \dontshow{
     set.seed (0)
 }
 ##--- EXAMPLE 1 ------------------------------------------
 sig <- diag(2)
 cen <- rep(1,2)
 x <- rbind(MASS::mvrnorm(360, cen * 0,   sig),
            MASS::mvrnorm(540, cen * 5,   sig * 6 - 2),
            MASS::mvrnorm(100, cen * 2.5, sig * 50))
 
 ## Two groups and 10\\% trimming level
 clus <- tclust(x, k = 2, alpha = 0.1, restr.fact = 8)
 
 plot(clus)
 plot(clus, labels = "observation")
 plot(clus, labels = "cluster")
 
 ## Three groups (one of them very scattered) and 0\\% trimming level
 clus <- tclust(x, k = 3, alpha=0.0, restr.fact = 100)
 
 plot(clus)
 
 ##--- EXAMPLE 2 ------------------------------------------
 data(geyser2)
 (clus <- tclust(geyser2, k = 3, alpha = 0.03))
 
 plot(clus)
 
\dontrun{

 ##--- EXAMPLE 3 ------------------------------------------
 data(M5data)
 x <- M5data[, 1:2]
 
 clus.a <- tclust(x, k = 3, alpha = 0.1, restr.fact =  1,
                   restr = "eigen", equal.weights = TRUE)
 clus.b <- tclust(x, k = 3, alpha = 0.1, restr.fact =  50,
                    restr = "eigen", equal.weights = FALSE)
 clus.c <- tclust(x, k = 3, alpha = 0.1, restr.fact =  1,
                   restr = "deter", equal.weights = TRUE)
 clus.d <- tclust(x, k = 3, alpha = 0.1, restr.fact = 50,
                   restr = "deter", equal.weights = FALSE)
 
 pa <- par(mfrow = c (2, 2))
 plot(clus.a, main = "(a)")
 plot(clus.b, main = "(b)")
 plot(clus.c, main = "(c)")
 plot(clus.d, main = "(d)")
 par(pa)
 
 ##--- EXAMPLE 4 ------------------------------------------

 data (swissbank)
 ## Two clusters and 8\\% trimming level
 (clus <- tclust(swissbank, k = 2, alpha = 0.08, restr.fact = 50))
 
 ## Pairs plot of the clustering solution
 pairs(swissbank, col = clus$cluster + 1)
 ## Two coordinates
 plot(swissbank[, 4], swissbank[, 6], col = clus$cluster + 1,
      xlab = "Distance of the inner frame to lower border",
      ylab = "Length of the diagonal")
 plot(clus)
 
 ## Three clusters and 0\\% trimming level
 clus<- tclust(swissbank, k = 3, alpha = 0.0, restr.fact = 110)
 
 ## Pairs plot of the clustering solution
 pairs(swissbank, col = clus$cluster + 1)
 
 ## Two coordinates
 plot(swissbank[, 4], swissbank[, 6], col = clus$cluster + 1, 
       xlab = "Distance of the inner frame to lower border", 
       ylab = "Length of the diagonal")
 
 plot(clus)
 
 ##--- EXAMPLE 5 ------------------------------------------
  data(M5data)
  x <- M5data[, 1:2]
  
  ## Classification trimmed likelihood approach
  clus.a <- tclust(x, k = 3, alpha = 0.1, restr.fact =  50,
                     opt="HARD", restr = "eigen", equal.weights = FALSE)
 ## Mixture trimmed likelihood approach
  clus.b <- tclust(x, k = 3, alpha = 0.1, restr.fact =  50,
                     opt="MIXT", restr = "eigen", equal.weights = FALSE)
 
 ## Hard 0-1 cluster assignment (all 0 if trimmed unit)
 head(clus.a$posterior)
 
 ## Posterior probabilities cluster assignment for the
 ##  mixture approach (all 0 if trimmed unit)
 head(clus.b$posterior)
 
}

}
\references{
Fritz, H.; Garcia-Escudero, L.A.; Mayo-Iscar, A. (2012), "tclust: An R Package 
 for a Trimming Approach to Cluster Analysis". Journal of Statistical Software, 
 47(12), 1-26. URL http://www.jstatsoft.org/v47/i12/

Garcia-Escudero, L.A.; Gordaliza, A.; Matran, C. and Mayo-Iscar, A. (2008), 
 "A General Trimming Approach to Robust Cluster Analysis". Annals of Statistics, 
 Vol.36, 1324--1345.  

García-Escudero, L. A., Gordaliza, A. and Mayo-Íscar, A. (2014). A constrained 
 robust proposal for mixture modeling avoiding spurious solutions. 
 Advances in Data Analysis and Classification, 27--43. 

García-Escudero, L. A., and Mayo-Íscar, A. and Riani, M. (2020). Model-based 
 clustering with determinant-and-shape constraint. Statistics and Computing, 
 30, 1363--1380.]
}
\author{
Javier Crespo Guerrero, Luis Angel Garcia Escudero, Agustin Mayo Iscar.
}
