% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sts.R
\name{sts_sparse_linear_regression}
\alias{sts_sparse_linear_regression}
\title{Formal representation of a sparse linear regression.}
\usage{
sts_sparse_linear_regression(design_matrix, weights_prior_scale = 0.1,
  weights_batch_shape = NULL, name = NULL)
}
\arguments{
\item{design_matrix}{float \code{tensor} of shape \code{tf$concat(list(batch_shape, list(num_timesteps, num_features)))}.
This may also optionally be an instance of \code{tf$linalg$LinearOperator}.}

\item{weights_prior_scale}{float \code{Tensor} defining the scale of the Horseshoe
prior on regression weights. Small values encourage the weights to be
sparse. The shape must broadcast with \code{weights_batch_shape}.
Default value: \code{0.1}.}

\item{weights_batch_shape}{if \code{NULL}, defaults to
\code{design_matrix.batch_shape_tensor()}. Must broadcast with the batch
shape of \code{design_matrix}. Default value: \code{NULL}.}

\item{name}{the name of this model component. Default value: 'LinearRegression'.}
}
\value{
an instance of \code{StructuralTimeSeries}.
}
\description{
This model defines a time series given by a sparse linear combination of
covariate time series provided in a design matrix:
}
\details{
\preformatted{observed_time_series <- tf$matmul(design_matrix, weights)
}

This is identical to \code{sts_linear_regression}, except that
\code{sts_sparse_linear_regression} uses a parameterization of a Horseshoe
prior to encode the assumption that many of the \code{weights} are zero,
i.e., many of the covariate time series are irrelevant. See the mathematical
details section below for further discussion. The prior parameterization used
by \code{sts_sparse_linear_regression} is more suitable for inference than that
obtained by simply passing the equivalent \code{tfd_horseshoe} prior to
\code{sts_linear_regression}; when sparsity is desired, \code{sts_sparse_linear_regression} will
likely yield better results.

This component does not itself include observation noise; it defines a
deterministic distribution with mass at the point
\code{tf$matmul(design_matrix, weights)}. In practice, it should be combined with
observation noise from another component such as \code{sts_sum}.

Mathematical Details

The basic horseshoe prior Carvalho et al. (2009) is defined as a Cauchy-normal scale mixture:\preformatted{scales[i] ~ HalfCauchy(loc=0, scale=1)
weights[i] ~ Normal(loc=0., scale=scales[i] * global_scale)`
}

The Cauchy scale parameters puts substantial mass near zero, encouraging
weights to be sparse, but their heavy tails allow weights far from zero to be
estimated without excessive shrinkage. The horseshoe can be thought of as a
continuous relaxation of a traditional 'spike-and-slab' discrete sparsity
prior, in which the latent Cauchy scale mixes between 'spike'
(\code{scales[i] ~= 0}) and 'slab' (\code{scales[i] >> 0}) regimes.

Following the recommendations in Piironen et al. (2017), \code{SparseLinearRegression} implements
a horseshoe with the following adaptations:
\itemize{
\item The Cauchy prior on \code{scales[i]} is represented as an InverseGamma-Normal
compound.
\item The \code{global_scale} parameter is integrated out following a \code{Cauchy(0., scale=weights_prior_scale)} hyperprior, which is also represented as an
InverseGamma-Normal compound.
\item All compound distributions are implemented using a non-centered
parameterization.
The compound, non-centered representation defines the same marginal prior as
the original horseshoe (up to integrating out the global scale),
but allows samplers to mix more efficiently through the heavy tails; for
variational inference, the compound representation implicity expands the
representational power of the variational model.
}

Note that we do not yet implement the regularized ('Finnish') horseshoe,
proposed in Piironen et al. (2017) for models with weak likelihoods, because the likelihood
in STS models is typically Gaussian, where it's not clear that additional
regularization is appropriate. If you need this functionality, please
email tfprobability@tensorflow.org.

The full prior parameterization implemented in \code{SparseLinearRegression} is
as follows:\preformatted{Sample global_scale from Cauchy(0, scale=weights_prior_scale).
global_scale_variance ~ InverseGamma(alpha=0.5, beta=0.5)
global_scale_noncentered ~ HalfNormal(loc=0, scale=1)
global_scale = (global_scale_noncentered *
sqrt(global_scale_variance) *
weights_prior_scale)
Sample local_scales from Cauchy(0, 1).
local_scale_variances[i] ~ InverseGamma(alpha=0.5, beta=0.5)
local_scales_noncentered[i] ~ HalfNormal(loc=0, scale=1)
local_scales[i] = local_scales_noncentered[i] * sqrt(local_scale_variances[i])
weights[i] ~ Normal(loc=0., scale=local_scales[i] * global_scale)
}
}
\section{References}{

\itemize{
\item \href{http://proceedings.mlr.press/v5/carvalho09a/carvalho09a.pdf}{Carvalho, C., Polson, N. and Scott, J. Handling Sparsity via the Horseshoe. AISTATS (2009).}
\item \href{https://arxiv.org/abs/1707.01694}{Juho Piironen, Aki Vehtari. Sparsity information and regularization in the horseshoe and other shrinkage priors (2017).}
}
}

\seealso{
For usage examples see \code{\link[=sts_fit_with_hmc]{sts_fit_with_hmc()}}, \code{\link[=sts_forecast]{sts_forecast()}}, \code{\link[=sts_decompose_by_component]{sts_decompose_by_component()}}.

Other sts: \code{\link{sts_additive_state_space_model}},
  \code{\link{sts_autoregressive_state_space_model}},
  \code{\link{sts_autoregressive}},
  \code{\link{sts_constrained_seasonal_state_space_model}},
  \code{\link{sts_dynamic_linear_regression_state_space_model}},
  \code{\link{sts_dynamic_linear_regression}},
  \code{\link{sts_linear_regression}},
  \code{\link{sts_local_level_state_space_model}},
  \code{\link{sts_local_level}},
  \code{\link{sts_local_linear_trend_state_space_model}},
  \code{\link{sts_local_linear_trend}},
  \code{\link{sts_seasonal_state_space_model}},
  \code{\link{sts_seasonal}},
  \code{\link{sts_semi_local_linear_trend_state_space_model}},
  \code{\link{sts_semi_local_linear_trend}},
  \code{\link{sts_smooth_seasonal_state_space_model}},
  \code{\link{sts_smooth_seasonal}}, \code{\link{sts_sum}}
}
\concept{sts}
