% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.tidylda.R
\name{predict.tidylda}
\alias{predict.tidylda}
\title{Get predictions from a Latent Dirichlet Allocation model}
\usage{
\method{predict}{tidylda}(
  object,
  new_data,
  type = c("prob", "class", "distribution"),
  method = c("gibbs", "dot"),
  iterations = NULL,
  burnin = -1,
  no_common_tokens = c("default", "zero", "uniform"),
  times = 100,
  threads = 1,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{object}{a fitted object of class \code{tidylda}}

\item{new_data}{a DTM or TCM of class \code{dgCMatrix} or a numeric vector}

\item{type}{one of "prob", "class", or "distribution". Defaults to "prob".}

\item{method}{one of either "gibbs" or "dot". If "gibbs" Gibbs sampling is used
and \code{iterations} must be specified.}

\item{iterations}{If \code{method = "gibbs"}, an integer number of iterations
for the Gibbs sampler to run. A future version may include automatic stopping criteria.}

\item{burnin}{If \code{method = "gibbs"}, an integer number of burnin iterations.
If \code{burnin} is greater than -1, the entries of the resulting "theta" matrix
are an average over all iterations greater than \code{burnin}.
Behavior is the same as documented in \code{\link[tidylda]{tidylda}}.}

\item{no_common_tokens}{behavior when encountering documents that have no tokens
in common with the model. Options are "\code{default}", "\code{zero}",
or "\code{uniform}". See 'details', below for explanation of behavior.}

\item{times}{Integer, number of samples to draw if \code{type = "distribution"}.
Ignored if \code{type} is "class" or "prob". Defaults to 100.}

\item{threads}{Number of parallel threads, defaults to 1. Note: currently
ignored; only single-threaded prediction is implemented.}

\item{verbose}{Logical. Do you want to print a progress bar out to the console?
Only active if \code{method = "gibbs"}. Defaults to \code{TRUE}.}

\item{...}{Additional arguments, currently unused}
}
\value{
\code{type} gives different outputs depending on whether the user selects
  "prob", "class", or "distribution". If "prob", the default, returns a
  a "theta" matrix with one row per document and one column per topic. If
  "class", returns a vector with the topic index of the most likely topic in
  each document. If "distribution", returns a tibble with one row per
  parameter per sample. Number of samples is set by the \code{times} argument.
}
\description{
Obtains predictions of topics for new documents from a fitted LDA model
}
\details{
If \code{predict.tidylda} encounters documents that have no tokens in common
  with the model in \code{object} it will engage in one of three behaviors based
  on the setting of \code{no_common_tokens}.
  
  \code{default} (the default) sets all topics to 0 for offending documents. This 
  enables continued computations downstream in a way that \code{NA} would not.
  However, if \code{no_common_tokens == "default"}, then \code{predict.tidylda}
  will emit a warning for every such document it encounters.
  
  \code{zero} has the same behavior as \code{default} but it emits a message
  instead of a warning.
  
  \code{uniform} sets all topics to 1/k for every topic for offending documents.
  it does not emit a warning or message.
}
\examples{
\donttest{
# load some data
data(nih_sample_dtm)

# fit a model
set.seed(12345)

m <- tidylda(
  data = nih_sample_dtm[1:20, ], k = 5,
  iterations = 200, burnin = 175
)

str(m)

# predict on held-out documents using gibbs sampling "fold in"
p1 <- predict(m, nih_sample_dtm[21:100, ],
  method = "gibbs",
  iterations = 200, burnin = 175
)

# predict on held-out documents using the dot product
p2 <- predict(m, nih_sample_dtm[21:100, ], method = "dot")

# compare the methods
barplot(rbind(p1[1, ], p2[1, ]), beside = TRUE, col = c("red", "blue"))

# predict classes on held out documents
p3 <- predict(m, nih_sample_dtm[21:100, ],
  method = "gibbs",
  type = "class",
  iterations = 100, burnin = 75
)

# predict distribution on held out documents
p4 <- predict(m, nih_sample_dtm[21:100, ],
  method = "gibbs",
  type = "distribution",
  iterations = 100, burnin = 75,
  times = 10
)
}
}
