% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vec-diff.R
\name{diff_vec}
\alias{diff_vec}
\alias{diff_inv_vec}
\title{Differencing Transformation}
\usage{
diff_vec(
  x,
  lag = 1,
  difference = 1,
  log = FALSE,
  initial_values = NULL,
  silent = FALSE
)

diff_inv_vec(x, lag = 1, difference = 1, log = FALSE, initial_values = NULL)
}
\arguments{
\item{x}{A numeric vector to be differenced or inverted.}

\item{lag}{Which lag (how far back) to be included in the differencing calculation.}

\item{difference}{The number of differences to perform.
\itemize{
\item 1 Difference is equivalent to measuring period change.
\item 2 Differences is equivalent to measuring period acceleration.
}}

\item{log}{If log differences should be calculated.
\emph{Note that difference inversion of a log-difference is approximate.}}

\item{initial_values}{Only used in the \code{diff_vec_inv()} operation.
A numeric vector of the initial values, which are used to invert differences.
This vector is the original values that are the length of the \code{NA} missing differences.}

\item{silent}{Whether or not to report the initial values used to invert the difference
as a message.}
}
\value{
A numeric vector
}
\description{
\code{diff_vec()} applies a Differencing Transformation.
\code{diff_inv_vec()} inverts the differencing transformation.
}
\details{
\strong{Benefits:}

This function is \code{NA} padded by default so it works well with \code{dplyr::mutate()} operations.

\strong{Difference Calculation}

Single differencing, \code{diff_vec(x_t)} is equivalent to: \code{x_t - x_t1},
where the subscript _t1 indicates the first lag.
\emph{This transformation can be interpereted as change.}

\strong{Double Differencing Calculation}

Double differencing, \code{diff_vec(x_t, difference = 2)} is equivalent to:
\verb{(x_t - x_t1) - (x_t - x_t1)_t1}, where the subscript _t1 indicates the first lag.
\emph{This transformation can be interpereted as acceleration.}

\strong{Log Difference Calculation}

Log differencing, \code{diff_vec(x_t, log = TRUE)} is equivalent to:
\code{log(x_t) - log(x_t1) = log(x_t / x_t1)}, where x_t is the series and x_t1 is the first lag.

The 1st difference \code{diff_vec(difference = 1, log = TRUE)} has an interesting property
where \code{diff_vec(difference = 1, log = TRUE) \%>\% exp()} is approximately \emph{1 + rate of change.}
}
\examples{
library(dplyr)
library(timetk)

# --- USAGE ----

diff_vec(1:10, lag = 2, difference = 2) \%>\%
    diff_inv_vec(lag = 2, difference = 2, initial_values = 1:4)

# --- VECTOR ----

# Get Change
1:10 \%>\% diff_vec()

# Get Acceleration
1:10 \%>\% diff_vec(difference = 2)

# Get approximate rate of change
1:10 \%>\% diff_vec(log = TRUE) \%>\% exp() - 1


# --- MUTATE ----

m4_daily \%>\%
    group_by(id) \%>\%
    mutate(difference = diff_vec(value, lag = 1)) \%>\%
    mutate(
        difference_inv = diff_inv_vec(
            difference,
            lag = 1,
            # Add initial value to calculate the inverse difference
            initial_values = value[1]
        )
    )



}
\seealso{
Advanced Differencing and Modeling:
\itemize{
\item \code{\link[=step_diff]{step_diff()}} - Recipe for \code{tidymodels} workflow
\item \code{\link[=tk_augment_differences]{tk_augment_differences()}} - Adds many differences to a \code{data.frame} (\code{tibble})
}

Additional Vector Functions:
\itemize{
\item Box Cox Transformation: \code{\link[=box_cox_vec]{box_cox_vec()}}
\item Lag Transformation: \code{\link[=lag_vec]{lag_vec()}}
\item Differencing Transformation: \code{\link[=diff_vec]{diff_vec()}}
\item Rolling Window Transformation: \code{\link[=slidify_vec]{slidify_vec()}}
\item Loess Smoothing Transformation: \code{\link[=smooth_vec]{smooth_vec()}}
\item Fourier Series: \code{\link[=fourier_vec]{fourier_vec()}}
\item Missing Value Imputation for Time Series: \code{\link[=ts_impute_vec]{ts_impute_vec()}}, \code{\link[=ts_clean_vec]{ts_clean_vec()}}
}
}
