% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SimFunctions.R
\name{Sim-Functions}
\alias{Sim-Functions}
\alias{sim}
\title{Tariff Simulation With User-Supplied Demand Parameters}
\usage{
sim(
  prices,
  supply = c("moncom", "bertrand", "auction", "bargaining"),
  demand = c("logit", "ces"),
  demand.param,
  owner,
  tariffPre = rep(0, length(prices)),
  tariffPost,
  subset = rep(TRUE, length(prices)),
  insideSize = 1,
  priceOutside,
  priceStart,
  bargpowerPre = rep(0.5, length(prices)),
  bargpowerPost = bargpowerPre,
  labels = paste("Prod", 1:length(prices), sep = ""),
  ...
)
}
\arguments{
\item{prices}{A length k vector of product prices.}

\item{supply}{A character string indicating how firms compete with one another. Valid
values are "moncom" (monopolistic competition), "bertrand" (Nash Bertrand),  "auction2nd"
(2nd score auction), or "bargaining".}

\item{demand}{A character string indicating the type of demand system
to be used in the merger simulation. Supported demand systems are
logit (\sQuote{Logit}) or ces (\sQuote{CES}).}

\item{demand.param}{See Below.}

\item{owner}{EITHER a vector of length k whose values indicate which firm produced a product before the tariff
OR a k x k matrix of pre-merger ownership shares.}

\item{tariffPre}{A vector of length k where each element equals the \strong{current} \emph{ad valorem} tariff
(expressed as a proportion of the consumer price) imposed on each product. Default is 0, which assumes no tariff.}

\item{tariffPost}{A vector of length k where each element equals the \strong{new}  \emph{ad valorem} tariff
(expressed as a proportion of the consumer price) imposed on each product. Default is 0, which assumes no tariff.}

\item{subset}{A vector of length k where each element equals TRUE if
the product indexed by that element should be included in the
post-merger simulation and FALSE if it should be excluded.Default is a
length k vector of TRUE.}

\item{insideSize}{A length 1 vector equal to total units sold if \sQuote{demand} equals "logit", or total revenues if
\sQuote{demand} equals "ces".}

\item{priceOutside}{A length 1 vector indicating the price of the
outside good. This option only applies to the \sQuote{Logit} class and its child classes
Default for \sQuote{Logit},\sQuote{LogitNests}, and \sQuote{LogitCap} is 0,
and for \sQuote{CES} and \sQuote{CesNests} is 1.}

\item{priceStart}{A length k vector of starting values used to solve for
equilibrium price. Default is the \sQuote{prices} vector for all values of
demand except for \sQuote{AIDS}, which is set equal to a vector of 0s.}

\item{bargpowerPre}{A length k vector of pre-merger bargaining power parameters. Values
must be between 0 (sellers have the power) and 1 (buyers the power). Ignored if \sQuote{supply} not equal
to "bargaining".}

\item{bargpowerPost}{A length k vector of post-merger bargaining power parameters. Values
must be between 0 (sellers have the power) and 1 (buyers the power). Default is \sQuote{bargpowerPre}.
Ignored if \sQuote{supply} not equal to "bargaining".}

\item{labels}{A k-length vector of labels. Default is \dQuote{Prod#}, where
\sQuote{#} is a number between 1 and the length of \sQuote{prices}.}

\item{...}{Additional options to feed to the
optimizer used to solve for equilibrium prices.}
}
\value{
\code{sim} returns an instance of the class specified by the
\sQuote{demand} argument.
}
\description{
Simulates the price effects of an ad valorem tariff
with user-supplied demand parameters under the
assumption that all firms in the market are playing either a
differentiated products Bertrand pricing game, 2nd price auction, or bargaining game.

Let k denote the number of products produced by all firms below.
}
\details{
Using user-supplied demand parameters,
\code{sim} simulates the effects of a merger in a market where
firms are playing a differentiated products pricing game.

If \sQuote{demand} equals \sQuote{Logit}  then
\sQuote{demand.param} must equal a list containing
\itemize{
  \item{alpha}{The price coefficient.}
  \item{meanval}{A length-k vector of mean valuations \sQuote{meanval}. If
    none of the values of \sQuote{meanval} are zero, an outside good is assumed
    to exist.}
}
If demand equals \sQuote{CES}  then
\sQuote{demand.param} must equal a list containing

\itemize{
  \item{gamma}{ The price coefficient,}
  \item{alpha}{The coefficient on the numeraire good. May instead be
    calibrated using \sQuote{shareInside},}
  \item{meanval}{A length-k vector of mean valuations \sQuote{meanval}. If
    none of the values of \sQuote{meanval} are zero, an outside good is assumed
    to exist,}
  \item{shareInside}{ The budget share of all products in the
    market. Default is 1, meaning that all consumer wealth is spent on
    products in the market. May instead be specified using \sQuote{alpha}.}

}
}
\examples{
## Calibration and simulation results from a merger between Budweiser and
## Old Style. Note that the in the following model there is no outside
## good; BUD's mean value has been normalized to zero.

## Source: Epstein/Rubenfeld 2004, pg 80


prodNames <- c("BUD","OLD STYLE","MILLER","MILLER-LITE","OTHER-LITE","OTHER-REG")
owner <-c("BUD","OLD STYLE","MILLER","MILLER","OTHER-LITE","OTHER-REG")
tariff <- c(0,0,0,0,.1,.1)

price    <- c(.0441,.0328,.0409,.0396,.0387,.0497)

# a list containing price coefficient and mean valuations
demand.param=list(alpha=-48.0457,
                  meanval=c(0,0.4149233,1.1899885,0.8252482,0.1460183,1.4865730)
)

sim.logit <- sim(price,demand="logit",supply="bertrand", demand.param,
                 owner=owner,tariffPost=tariff,labels=prodNames)



print(sim.logit)           # return predicted price change
summary(sim.logit)         # summarize merger simulation

elast(sim.logit,TRUE)      # returns premerger elasticities
elast(sim.logit,FALSE)     # returns postmerger elasticities

diversion(sim.logit,TRUE)  # return premerger diversion ratios
diversion(sim.logit,FALSE) # return postmerger diversion ratios


cmcr(sim.logit)            #calculate compensating marginal cost reduction
upp(sim.logit)             #calculate Upwards Pricing Pressure Index

CV(sim.logit)              #calculate representative agent compensating variation

}
\seealso{
The S4 class documentation for:  \code{\linkS4class{Logit}} and
\code{\linkS4class{CES}},
}
\author{
Charles Taragin \email{ctaragin+trader@gmail.com}
}
