% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/non_linear_bins.R
\name{nonlinear_bins}
\alias{nonlinear_bins}
\title{Create Nonlinear Probability of Survival Bins}
\usage{
nonlinear_bins(
  data,
  Ps_col,
  outcome_col,
  divisor1 = 5,
  divisor2 = 5,
  threshold_1 = 0.9,
  threshold_2 = 0.99
)
}
\arguments{
\item{data}{A \code{data.frame} or \code{tibble} containing the probability of survival
data for a set of patients.}

\item{Ps_col}{The column in \code{data} containing the probability of survival
values for a set of patients.}

\item{outcome_col}{The name of the column containing the outcome data. It
should be binary, with values indicating patient survival. A value of \code{1}
should represent "alive" (survived), while \code{0} should
represent "dead" (did not survive). Ensure the column contains only these
two possible values.}

\item{divisor1}{A parameter to control the width of the probability of
survival range bins. Affects the creation of step sizes for the beginning
of each bin range. Defaults to \code{5}.}

\item{divisor2}{A parameter to control the width of the probability of
survival range bins. Affects the creation of step sizes for the beginning
of each bin range. Defaults to \code{5}.}

\item{threshold_1}{A parameter to decide where data indices will begin to
create step sizes. Defaults to \code{0.9}.}

\item{threshold_2}{A parameter to decide where data indices will end to
create step sizes. Defaults to \code{0.99}.}
}
\value{
A list with \code{intervals} and \code{bin_stats} objects:
\itemize{
\item \code{intervals}: A vector of start and end-points for the probability of survival bin ranges.
\item \code{bin_stats}: A \code{tibble} with columns \code{bin_number}, \code{bin_start}, \code{bin_end}, \code{mean}, \code{sd},
\code{alive}, \code{dead}, \code{count}, and \code{percent}.
}
}
\description{
This function generates nonlinear bins for probability of survival data based
on specified thresholds and divisors as specified in Napoli et al. (2017),
Schroeder et al. (2019), and Kassar et al. (2016). This function calculates
bin statistics, including mean, standard deviation, total alive, total dead,
count, and percentage for each bin.
}
\examples{
# Generate example data with high negative skewness
set.seed(123)

# Parameters
n_patients <- 10000  # Total number of patients

# Skewed towards higher values
Ps <- plogis(rnorm(n_patients, mean = 2, sd = 1.5))

# Simulate survival outcomes based on Ps
survival_outcomes <- rbinom(n_patients,
                            size = 1,
                            prob = Ps
                            )

# Create data frame
data <- data.frame(Ps = Ps, survival = survival_outcomes) |>
dplyr::mutate(death = dplyr::if_else(survival == 1, 0, 1))

# Apply the nonlinear_bins function
results <- nonlinear_bins(data = data,
                          Ps_col = Ps,
                          outcome_col = survival,
                          divisor1 = 5,
                          divisor2 = 5,
                          threshold_1 = 0.9,
                          threshold_2 = 0.99)

# View results
results$intervals
results$bin_stats

}
\author{
Nicolas Foss, Ed.D, MS, original paper and code in MATLAB by Nicholas
J. Napoli, Ph.D., MS
}
