% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/relative_mortality.R
\name{rm_bin_summary}
\alias{rm_bin_summary}
\title{Bin-Level Summary for Relative Mortality Metric (RMM)}
\usage{
rm_bin_summary(
  data,
  Ps_col,
  outcome_col,
  n_samples = 1000,
  Divisor1 = 5,
  Divisor2 = 5,
  Threshold_1 = 0.9,
  Threshold_2 = 0.99
)
}
\arguments{
\item{data}{A data frame or tibble containing the data.}

\item{Ps_col}{The name of the column containing the survival probabilities
(Ps). Should be numeric (values between 0 and 100).}

\item{outcome_col}{The name of the column containing the outcome data. It
should be binary, with values indicating patient survival. A value of \code{1}
should represent "alive" (survived), while \code{0} should
represent "dead" (did not survive). Ensure the column contains only these
two possible values.}

\item{n_samples}{A numeric value indicating the number of bootstrap samples
to take from the data source.}

\item{Divisor1}{A divisor used for binning the survival probabilities
(default is 5).}

\item{Divisor2}{A second divisor used for binning the survival probabilities
(default is 5).}

\item{Threshold_1}{The first threshold for dividing the survival
probabilities (default is 0.9).}

\item{Threshold_2}{The second threshold for dividing the survival
probabilities (default is 0.99).}
}
\value{
A tibble containing bin-level statistics including:
\itemize{
\item \code{bin_number}: The bin to which each record was assigned.
\item \code{TA_b}: Total alive in each bin (number of patients who survived).
\item \code{TD_b}: Total dead in each bin (number of patients who did not survive).
\item \code{N_b}: Total number of patients in each bin.
\item \code{EM_b}: Estimated mortality rate for each bin (TD_b / (TA_b + TD_b)).
\item \code{AntiS_b}: The anticipated survival rate for each bin.
\item \code{AntiM_b}: The anticipated mortality rate for each bin.
\item \code{bin_start}: The lower bound of the survival probability range for each bin.
\item \code{bin_end}: The upper bound of the survival probability range for each bin.
\item \code{midpoint}: The midpoint of the bin range (calculated as (bin_start + bin_end) / 2).
\item \code{R_b}: The width of each bin (bin_end - bin_start).
\item \code{population_RMM_LL}: The lower bound of the 95\% confidence interval for the population RMM.
\item \code{population_RMM}: The final calculated Relative Mortality Metric for the population
existing in \code{data}.
\item \code{population_RMM_UL}: The upper bound of the 95\% confidence interval for the population RMM.
\item \code{population_CI}: The confidence interval width for the population RMM.
\item \code{bootstrap_RMM_LL}: The lower bound of the 95\% confidence interval for the bootstrap RMM.
\item \code{bootstrap_RMM}: The average RMM value calculated for the bootstrap sample.
\item \code{bootstrap_RMM_UL}: The upper bound of the 95\% confidence interval for the bootstrap RMM.
\item \code{bootstrap_CI}: The width of the 95\% confidence interval for the bootstrap RMM.
}
}
\description{
Calculates a bin-level summary for the Relative Mortality Metric (RMM) from
Napoli et al. (2017) by grouping data into bins based on survival
probabilities (Ps) and summarizing outcomes within each bin. This function
returns statistics such as total alive, total dead, estimated mortality,
anticipated mortality, and confidence intervals for each bin. For more
information on the methods used in this function, see as well Schroeder et
al. (2019), and Kassar et al. (2016).

The Relative Mortality Metric (RMM) quantifies the performance of a center in
comparison to the anticipated mortality based on the TRISS national
benchmark. The RMM measures the difference between observed and expected
mortality, with a range from -1 to 1.
\itemize{
\item An RMM of 0 indicates that the observed mortality aligns with the expected
national benchmark across all acuity levels.
\item An RMM greater than 0 indicates better-than-expected performance, where
the center is outperforming the national benchmark.
\item An RMM less than 0 indicates under-performance, where the center’s observed
mortality is higher than the expected benchmark.
}

This metric helps assess how a center's mortality compares to the national
standards, guiding quality improvement efforts.\code{rm_bin_summary()} utilizes
bootstrap sampling to calculate the confidence intervals via the standard
error method.

Due to the use of bootstrap sampling within the function, users should set
the random number seed before running \code{rm_bin_summary()}.
}
\examples{
# Generate example data with high negative skewness
set.seed(10232015)

# Parameters
n_patients <- 10000  # Total number of patients

Ps <- plogis(rnorm(n_patients, mean = 2,
                    sd = 1.5)
                    )  # Skewed towards higher values

# Simulate survival outcomes based on Ps
survival_outcomes <- rbinom(n_patients,
                            size = 1,
                            prob = Ps
                            )

# Create data frame
data <- data.frame(Ps = Ps, survival = survival_outcomes) |>
dplyr::mutate(death = dplyr::if_else(survival == 1, 0, 1))

# Example usage of the `rm_bin_summary` function
rm_bin_summary(data = data, Ps_col = Ps,
               outcome_col = survival,
               n_samples = 5
               )

}
\author{
Nicolas Foss, Ed.D, MS, original paper and code in MATLAB by Nicholas
J. Napoli, Ph.D., MS
}
