% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/relative_mortality.R
\name{rm_bin_summary}
\alias{rm_bin_summary}
\title{Bin-Level Summary for Relative Mortality Metric (RMM)}
\usage{
rm_bin_summary(
  data,
  Ps_col,
  outcome_col,
  group_vars = NULL,
  n_samples = 100,
  Divisor1 = 5,
  Divisor2 = 5,
  Threshold_1 = 0.9,
  Threshold_2 = 0.99,
  bootstrap_ci = TRUE,
  seed = NULL
)
}
\arguments{
\item{data}{A data frame or tibble containing the data.}

\item{Ps_col}{The name of the column containing the survival probabilities
(Ps). Should be numeric on a scale from 0 to 1.}

\item{outcome_col}{The name of the column containing the outcome data. It
should be binary, with values indicating patient survival. A value of \code{1}
should represent "alive" (survived), while \code{0} should represent "dead" (did
not survive).  \code{TRUE/FALSE} are accepted as well. Ensure the column
contains only these possible values.}

\item{group_vars}{Optional character vector specifying grouping variables for
stratified analysis. If \code{NULL}, the calculation is performed on the entire
dataset.}

\item{n_samples}{A numeric value indicating the number of bootstrap samples
to take from the data source.}

\item{Divisor1}{A divisor used for binning the survival probabilities
(default is 5).}

\item{Divisor2}{A second divisor used for binning the survival probabilities
(default is 5).}

\item{Threshold_1}{The first threshold for dividing the survival
probabilities (default is 0.9).}

\item{Threshold_2}{The second threshold for dividing the survival
probabilities (default is 0.99).}

\item{bootstrap_ci}{A logical indicating whether to return the relative
mortality metric estimate and 95\% confidence intervals using bootstrap
sampling. Default is \code{TRUE}.}

\item{seed}{Optional numeric value to set a random seed for reproducibility.
If \code{NULL} (default), no seed is set.}
}
\value{
A tibble containing bin-level statistics including:
\itemize{
\item \code{bin_number}: The bin to which each record was assigned.
\item \code{TA_b}: Total alive in each bin (number of patients who survived).
\item \code{TD_b}: Total dead in each bin (number of patients who did not survive).
\item \code{N_b}: Total number of patients in each bin.
\item \code{EM_b}: Estimated mortality rate for each bin (TD_b / (TA_b + TD_b)).
\item \code{AntiS_b}: The anticipated survival rate for each bin.
\item \code{AntiM_b}: The anticipated mortality rate for each bin.
\item \code{bin_start}: The lower bound of the survival probability range for each
bin.
\item \code{bin_end}: The upper bound of the survival probability range for each
bin.
\item \code{midpoint}: The midpoint of the bin range (calculated as
(bin_start + bin_end) / 2).
\item \code{R_b}: The width of each bin (bin_end - bin_start).
\item \code{population_RMM_LL}: The lower bound of the 95\% confidence interval for
the population RMM.
\item \code{population_RMM}: The final calculated Relative Mortality Metric for the
population existing in \code{data}.
\item \code{population_RMM_UL}: The upper bound of the 95\% confidence interval for
the population RMM.
\item \code{population_CI}: The confidence interval width for the population RMM.
\item \code{bootstrap_RMM_LL}: The lower bound of the 95\% confidence interval for
the bootstrap RMM. (optional, if \code{bootstrap_ci = TRUE})
\item \code{bootstrap_RMM}: The average RMM value calculated for the bootstrap
sample. (optional, if \code{bootstrap_ci = TRUE})
\item \code{bootstrap_RMM_UL}: The upper bound of the 95\% confidence interval for
the bootstrap RMM. (optional, if \code{bootstrap_ci = TRUE})
\item \code{bootstrap_CI}: The width of the 95\% confidence interval for the
bootstrap RMM. (optional, if \code{bootstrap_ci = TRUE})
}
}
\description{
Calculates a bin-level summary for the Relative Mortality Metric (RMM) from
Napoli et al. (2017) by grouping data into bins based on survival
probabilities (Ps) and summarizing outcomes within each bin. This function
returns statistics such as total alive, total dead, estimated mortality,
anticipated mortality, and confidence intervals for each bin. For more
information on the methods used in this function, see as well Schroeder et
al. (2019), and Kassar et al. (2016).

The Relative Mortality Metric (RMM) quantifies the performance of a center in
comparison to the anticipated mortality based on the TRISS national
benchmark. The RMM measures the difference between observed and expected
mortality, with a range from -1 to 1.
\itemize{
\item An RMM of 0 indicates that the observed mortality aligns with the expected
national benchmark across all acuity levels.
\item An RMM greater than 0 indicates better-than-expected performance, where
the center is outperforming the national benchmark.
\item An RMM less than 0 indicates under-performance, where the center’s observed
mortality is higher than the expected benchmark.
}

This metric helps assess how a center's mortality compares to the national
standards, guiding quality improvement efforts.\code{rm_bin_summary()} utilizes
bootstrap sampling to calculate the confidence intervals via the standard
error method.
}
\details{
Like other statistical computing functions, \code{rm_bin_summary()} is happiest
without missing data.  It is best to pass complete probability of survival
and mortality outcome data to the function for optimal performance. With
smaller datasets, this is especially helpful.  However, \code{rm_bin_summary()}
will throw a warning about missing values, if any exist in \code{Ps_col} and/or
\code{outcome_col}.

\code{rm_bin_summary()} assumes \code{Ps_col} contains probabilities derived from
real-world inputs for the Trauma Injury Severity Score (TRISS) model.
Synthetic or low-variability data (especially with small sample sizes) may
not reflect the distribution of TRISS-derived survival probabilities. This
can result in unstable estimates or function failure due to insufficient
dispersion. With small sample sizes, it may be important to use smaller
values with the divisor arguments and adjust the thresholds (based on the
distribution of the \code{Ps_col} values) to create bins that better accommodate
the data.

Due to the use of bootstrap sampling within the function, users should
consider setting the random number seed within \code{rm_bin_summary()} using the
\code{seed} argument for reproducibility.
}
\note{
This function will produce the most reliable and interpretable results when
using a dataset that has one row per patient, with each column being a
feature.

By default, \code{rm_bin_summary()} derives bin cut points from the full dataset’s
distribution. This ensures comparability across groups when \code{group_vars} is
used. To tailor results to a specific group (e.g., a single hospital), filter
the dataset to that subgroup before calling \code{rm_bin_summary()}. The function
will then compute bins and related statistics using only that subset’s
\code{Ps_col} distribution. When \code{group_vars} is used, and ff a group lacks
observations within one or more bins, \code{rm_bin_summary()} will compute
statistics only for the bins that contain data. Bins with no observations are
excluded from the summary for that group.
}
\examples{
# Generate example data
set.seed(123)

# Parameters
# Total number of patients
n_patients <- 5000

# Arbitrary group labels
groups <- sample(x = LETTERS[1:2], size = n_patients, replace = TRUE)

# Trauma types
trauma_type_values <- sample(
  x = c("Blunt", "Penetrating"),
  size = n_patients,
  replace = TRUE
)

# RTS values
rts_values <- sample(
  x = seq(from = 0, to = 7.8408, by = 0.005),
  size = n_patients,
  replace = TRUE
)

# patient ages
ages <- sample(
  x = seq(from = 0, to = 100, by = 1),
  size = n_patients,
  replace = TRUE
)

# ISS scores
iss_scores <- sample(
  x = seq(from = 0, to = 75, by = 1),
  size = n_patients,
  replace = TRUE
)

# Generate survival probabilities (Ps)
Ps <- traumar::probability_of_survival(
  trauma_type = trauma_type_values,
  age = ages,
  rts = rts_values,
  iss = iss_scores
)

# Simulate survival outcomes based on Ps
survival_outcomes <- rbinom(n_patients, size = 1, prob = Ps)

# Create data frame
data <- data.frame(Ps = Ps, survival = survival_outcomes, groups = groups) |>
  dplyr::mutate(death = dplyr::if_else(survival == 1, 0, 1))

# Example usage of the `rm_bin_summary()` function
rm_bin_summary(
  data = data,
  Ps_col = Ps,
  outcome_col = survival,
  n_samples = 10,
  Divisor1 = 4,
  Divisor2 = 4
)

# Create example grouping variable (e.g., hospital)
hospital <- sample(c("Hospital A", "Hospital B"), n_patients, replace = TRUE)

# Create data frame
data <- data.frame(
  Ps = Ps,
  survival = survival_outcomes,
  hospital = hospital
) |>
  dplyr::mutate(death = dplyr::if_else(survival == 1, 0, 1))

# Example usage of the `rm_bin_summary()` function with grouping
rm_bin_summary(
  data = data,
  Ps_col = Ps,
  outcome_col = survival,
  group_vars = "hospital", # Stratifies by hospital
  n_samples = 10,
  Divisor1 = 4,
  Divisor2 = 4
)

}
\references{
Kassar, O.M., Eklund, E.A., Barnhardt, W.F., Napoli, N.J., Barnes, L.E.,
Young, J.S. (2016). Trauma survival margin analysis: A dissection of trauma
center performance through initial lactate. The American Surgeon, 82(7),
649-653. \url{doi:10.1177/000313481608200733}

Napoli, N. J., Barnhardt, W., Kotoriy, M. E., Young, J. S., & Barnes, L. E.
(2017). Relative mortality analysis: A new tool to evaluate clinical
performance in trauma centers. IISE Transactions on Healthcare Systems
Engineering, 7(3), 181–191. \url{doi:10.1080/24725579.2017.1325948}

Schroeder, P. H., Napoli, N. J., Barnhardt, W. F., Barnes, L. E., & Young, J.
S. (2018). Relative mortality analysis of the “golden hour”: A comprehensive
acuity stratification approach to address disagreement in current literature.
Prehospital Emergency Care, 23(2), 254–262.
\url{doi:10.1080/10903127.2018.1489021}
}
\seealso{
\code{\link[=probability_of_survival]{probability_of_survival()}}, \code{\link[=rmm]{rmm()}}, and \code{\link[=nonlinear_bins]{nonlinear_bins()}}
}
\author{
Nicolas Foss, Ed.D, MS, original implementation in MATLAB by Nicholas
J. Napoli, Ph.D., MS
}
