% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/arima_pi.R
\name{arima_pi}
\alias{arima_pi}
\title{Prediction Intervals for ARIMA Processes with Exogenous Variables Using Importance Sampling}
\usage{
arima_pi(x, order, xreg = NULL, n_ahead = 1, level = 0.95,
  median = TRUE, se_limits = TRUE, prior = "uniform", custom_prior,
  custom_prior_args = NULL, nsim = 1000, invertibility = FALSE,
  last_only = FALSE, return_weights = FALSE, ...)
}
\arguments{
\item{x}{vector containing the time series}

\item{order}{vector of length 3 with values p,d,q
corresponding to the number of AR parameters, degree of differencing and number of MA parameters.}

\item{xreg}{matrix or data frame containing the exogenous variables
(not including the intercept which is always included for non-differenced series)}

\item{n_ahead}{length of the forecast horizon.}

\item{level}{desired frequentist coverage probability of the prediction intervals.}

\item{median}{compute the median of the prediction interval.}

\item{se_limits}{compute the standard errors of the prediction interval limits.}

\item{prior}{prior to be used in importance sampling for AR and MA parameters.
Defaults to uniform prior. Several Jeffreys' priors are also available (see \code{\link{jeffreys}} for details).
If "custom", a user-defined custom prior is used (see next arguments).
All priors assume that the ARMA parameters lie in stationarity/invertibility region.}

\item{custom_prior}{function for computing custom prior.
First argument must be a vector containing the AR and MA parameters (in that order).}

\item{custom_prior_args}{list containing additional arguments to \code{custom_prior}.}

\item{nsim}{number of simulations used in importance sampling. Default is 1000.}

\item{invertibility}{Logical, should the priors include invertibility constraint? Default is \code{FALSE}.}

\item{last_only}{compute the prediction intervals only for the last prediction step.}

\item{return_weights}{Return (scaled) weights used in importance sampling.}

\item{...}{additional arguments for \code{\link{arima}}.}
}
\value{
a list containing the prediction intervals.
 @references
\enumerate{
\item{Helske, J. and Nyblom, J. (2015). Improved frequentist prediction
intervals for autoregressive models by simulation.
In Siem Jan Koopman and Neil Shephard, editors,
Unobserved Components and Time Series Econometrics. Oxford University Press.
\url{http://urn.fi/URN:NBN:fi:jyu-201603141839}}
 \item{Helske, J. and Nyblom, J. (2014). Improved frequentist prediction intervals for
 ARMA models by simulation.
 In Johan Knif and Bernd Pape, editors,
Contributions to Mathematics, Statistics, Econometrics, and Finance:
essays in honour of professor Seppo Pynnönen,
number 296 in Acta Wasaensia, pages 71--86. University of Vaasa.
\url{http://urn.fi/URN:NBN:fi:jyu-201603141836}}
}
}
\description{
Function \code{arima_pi} computes prediction intervals for ARIMA processes
with exogenous variables using importance sampling. For regression coefficients,
diffuse (uninformative) prior is used, whereas multiple options for
prior distributions for ARMA coefficients are supported.
}
\examples{

set.seed(123)
x <- arima.sim(n = 30, model = list(ar = 0.9))

pred_arima <- predict(arima(x, order = c(1,0,0)), n.ahead = 10, se.fit = TRUE)
pred_arima <- cbind(pred = pred_arima$pred,
  lwr = pred_arima$pred - qnorm(0.975)*pred_arima$se,
  upr = pred_arima$pred + qnorm(0.975)*pred_arima$se)

pred <- arima_pi(x, order = c(1,0,0), n_ahead = 10)

ts.plot(ts.union(x,pred_arima, pred[,1:3]), col = c(1,2,2,2,3,3,3),
  lty = c(1,1,2,2,1,2,2))

}
\seealso{
\code{\link{tsPI}}, \code{\link{struct_pi}}
}

