% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/detect_outliers.R
\name{detect_outliers}
\alias{detect_outliers}
\title{Detects unreliable outliers in univariate time series data based on finite
mixture modelling}
\usage{
detect_outliers(
  data,
  S,
  model.select.iter = 10,
  outlier.detect.iter = 10,
  proba = 0.1,
  share = 0.3,
  detection.parameter = 1,
  out.par = 2,
  noise.par = 10^-5,
  mar = 1,
  max.cluster = 9,
  G = NULL,
  modelName = NULL,
  feat.int = list(org.s = TRUE, grad = TRUE, abs.grad = TRUE, abs.seas.grad = TRUE,
    lin.tr = TRUE),
  ...
)
}
\arguments{
\item{data}{an one dimensional matrix or data frame without missing data;
each row is an observation.}

\item{S}{vector with numeric values for each seasonality present in data.}

\item{model.select.iter}{denotes the number of
iterations to find the optimal number of clusters as well as
the optimal model of the covariance matrix.
By default set to 10. Results become more robust
with increasing iterations but lead
likewise to increasing computational time.}

\item{outlier.detect.iter}{denotes the number of iterations
for outlier detection based on variing subsamples
controlled by the share parameter. By default set to 10.
Results become more robust with increasing iterations but lead
likewise to increasing computational time.}

\item{proba}{denotes threshold ranging from 0 to 1 from which an
observation is considered being an outlier. Number of outlier increases
with decreasing threshold. By default set to 0.1, implying that in average
an observation obtains a probability in each iteration of 0.1 belonging to
the outlier cluster.}

\item{share}{controlls the size of the subsample used for estimation.
By default set to 0.3 (ranging from 0 to 1). Controlls the computational time
and the robustness of the method.}

\item{detection.parameter}{denotes a parameter to regulate the
detection sensisitivity. By default set to 1. The smaller the more outliers.}

\item{out.par}{controls the number of artifially produced outliers in
relation to the subsample size controlled by the share parameter. By default
out.par ist set to 2. By increase a priori it is assumed that
share of outliers in data increases.}

\item{noise.par}{controls strenght of noise added to feature matrix to avoid
zero variance issues by applying bivariate features. By default set to 10^-5,
strength of noise decreases with decreasing noise.par parameter.}

\item{mar}{denotes a margin controlling the number of adjacent
values around an identified outlier which are likewise considered
as outliers. By default set to 1 so that the one most closest neighbours
of an identified outlier on each side are also treated as outliers.}

\item{max.cluster}{a single numeric value controlling the maximum
number of clusters allowed. By default set to 9.}

\item{G}{denotes the optimal number of clusters limited by the
max.cluster paramter. By default G is set to NULL and automatically
calculated based on the BIC.}

\item{modelName}{denotes the geometric features of the covariance matrix.
i.e. "EII", "VII", "EEI", "EVI", "VEI", "VVI", etc.. By default modelName
is set to NUll and automatically calcualted based on BIC.
The help file for \link[mclust]{mclustModelNames} describes
the available models.}

\item{feat.int}{a list of logical values indicating which features should
be applied in clustering algorithm.
\describe{
\item{org.s}{denotes the scaled original time series.}
\item{lin.tr}{denotes linear trends based on seasonalities S.}
\item{grad}{denotes the gradient of scaled original time series.}
\item{abs.grad}{denotes absolute gradient of
scaled original time series.}
\item{abs.seas.grad}{denotes the absolute seasonal gradient of
scaled original time series based on seasonalties S.}
}}

\item{...}{additional arguments for the \link[mclust]{Mclust} function.}
}
\value{
An object of class "tsrobprep" containing the following elements:
\describe{
\item{original.data}{an numeric vector containing the original data.}
\item{outlier.probs}{an numeric vector containing the averaged probability}
\item{outlier.probs.mat}{a matrix containg the probability for each iteration
that observation is belonging to the outlier cluster.
Each row is an observation and each column an iteration.}
\item{outlier.pos}{a logical vector indicating the position of each outlier.}
\item{outlier.pos.aug}{a logical vector indicating the position of
each outlier including neigbouring values based on the mar parameter.}
\item{estimated.models}{a list containing each estimated model.}
\item{BIC}{an mclustBIC object containing the
Bayesian Information Criterion for the specified mixture models numbers of
clusters. Auxiliary information returned as attributes.}
}
}
\description{
This function applies finite mixture modelling to compute
the probability of each observation being an outlier
in an univariate time series.
By utilizing the \link[mclust]{Mclust} package the data is
assigned in k clusters whereof one contains outliers.
The clustering process is based on features, which are
specifically desinged for outlier detection in time series data.
}
\details{
The detection of outliers is addressed by
model based clustering based on parameterized finite Gaussian mixture models.
For cluster estimation the \link[mclust]{Mclust} function is applied.
Models are estimated by the EM algorithm initialized by hierarchical
model-based agglomerative clustering. The optimal model can be selected
according to BIC.
}
\examples{
\dontrun{
set.seed(1)
id <- 14000:17000
# Replace missing values
modelmd <- model_missing_data(data = GBload[id, -1], tau = 0.5,
 S = c(48, 336), indices.to.fix = seq_len(nrow(GBload[id, ])),
 consider.as.missing = 0, min.val = 0)
# Impute missing values
 data.imputed <- impute_modelled_data(modelmd)
# Detect outliers
o.ident <- detect_outliers(data = data.imputed, S = c(48, 336),
                           model.select.iter = 1,
                           outlier.detect.iter = 1)
# Plot of identified outliers in time series
plot(data.imputed, type = "o", col=1 + o.ident$outlier.pos.aug,
     pch = 1 + 18 * o.ident$outlier.probs)

# Plot of feature matrix
plot.ts(o.ident$features, type = "o",
        col = 1 + o.ident$outlier.pos,
        pch = 1 + 18 * o.ident$outlier.probs)
}
}
\seealso{
\link[tsrobprep]{model_missing_data},
\link[tsrobprep]{impute_modelled_data},
\link[tsrobprep]{auto_data_cleaning}
}
