% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ashdog.R
\name{flexdog_full}
\alias{flexdog_full}
\title{Flexible genotyping for polyploids from next-generation sequencing data.}
\usage{
flexdog_full(refvec, sizevec, ploidy, model = c("hw", "bb", "norm", "ash",
  "f1", "s1", "f1pp", "s1pp", "flex", "uniform"), verbose = TRUE,
  mean_bias = 0, var_bias = 0.7^2, mean_seq = -4.7, var_seq = 1,
  seq = 0.005, bias = 1, od = 0.001, update_bias = TRUE,
  update_seq = TRUE, update_od = TRUE, mode = NULL, use_cvxr = FALSE,
  itermax = 200, tol = 10^-4, fs1_alpha = 10^-3, ashpen = 10^-6,
  p1ref = NULL, p1size = NULL, p2ref = NULL, p2size = NULL,
  outliers = FALSE)
}
\arguments{
\item{refvec}{A vector of counts of reads of the reference allele.}

\item{sizevec}{A vector of total counts.}

\item{ploidy}{The ploidy of the species. Assumed to be the same for each
individual.}

\item{model}{What form should the prior (genotype distribution) take?
See Details for possible values.}

\item{verbose}{Should we output more (\code{TRUE}) or less
(\code{FALSE})?}

\item{mean_bias}{The prior mean of the log-bias.}

\item{var_bias}{The prior variance of the log-bias.}

\item{mean_seq}{The prior mean of the logit of the sequencing
error rate.}

\item{var_seq}{The prior variance of the logit of the sequencing
error rate.}

\item{seq}{The starting value of the sequencing error rate.}

\item{bias}{The starting value of the bias.}

\item{od}{The starting value of the overdispersion parameter.}

\item{update_bias}{A logical. Should we update \code{bias}
(\code{TRUE}), or not (\code{FALSE})?}

\item{update_seq}{A logical. Should we update \code{seq}
(\code{TRUE}), or not (\code{FALSE})?}

\item{update_od}{A logical. Should we update \code{od}
(\code{TRUE}), or not (\code{FALSE})?}

\item{mode}{The mode if \code{model = "ash"}. If not provided,
\code{flexdog} will estimate the mode. This is the starting point of
the allele frequency if \code{model = "hw"}. This should be
\code{NULL} for all other options of \code{model}.}

\item{use_cvxr}{A logical. If \code{model = "ash"}, then do you want
to use the EM algorithm
(\code{FALSE}) or a convex optimization program using
the package CVXR \code{TRUE}?
Only available if CVXR is installed. Setting \code{use_cvxr} to
\code{TRUE} is generally slower than setting it to \code{FALSE}.}

\item{itermax}{The maximum number of EM iterations to run for each mode
(if \code{model = "ash"}) or the total number of EM iterations to
run (for any other value of \code{model}).}

\item{tol}{The tolerance stopping criterion. The EM algorithm will stop
if the difference in the log-likelihoods between two consecutive
iterations is less than \code{tol}.}

\item{fs1_alpha}{The value at which to fix
the mixing proportion for the uniform component
 when \code{model = "f1"}, \code{model = "f1pp"},
\code{model = "s1"}, or \code{model = "s1pp"}.
I would recommend some small
value such at \code{10^-3}.}

\item{ashpen}{The penalty to put on the unimodal prior.
Larger values shrink the unimodal prior towards the
discrete uniform distribution.}

\item{p1ref}{The reference counts for the first parent if
\code{model = "f1"} (or \code{model = "f1pp"}), or for
the only parent if \code{model = "s1"} (or
\code{model = "s1pp"}).}

\item{p1size}{The total counts for the first parent if
\code{model = "f1"} (or \code{model = "f1pp"}),
or for the only parent if \code{model = "s1"}
(or \code{model = "s1pp"}).}

\item{p2ref}{The reference counts for the second parent if
\code{model = "f1"} (or \code{model = "f1pp"}).}

\item{p2size}{The total counts for the second parent if
\code{model = "f1"} (or \code{model = "f1pp"}).}

\item{outliers}{A logical. Should we allow for the inclusion of outliers
(\code{TRUE}) or not (\code{FALSE}). Only supported when
\code{model = "f1"} or \code{model = "s1"}. I wouldn't
recommend it for any other model anyway.}
}
\value{
An object of class \code{flexdog}, which consists
    of a list with some or all of the following elements:
\describe{
  \item{\code{bias}}{The estimated bias parameter.}
  \item{\code{seq}}{The estimated sequencing error rate.}
  \item{\code{od}}{The estimated overdispersion parameter.}
  \item{\code{num_iter}}{The number of EM iterations ran. You should
      be wary if this equals \code{itermax}.}
  \item{\code{llike}}{The maximum marginal log-likelihood.}
  \item{\code{postmat}}{A matrix of posterior probabilities of each
      genotype for each individual. The rows index the individuals
      and the columns index the allele dosage.}
  \item{\code{gene_dist}}{The estimated genotype distribution. The
      \code{i}th element is the proportion of individuals with
      genotype \code{i-1}. If \code{outliers = TRUE}, then this
      is conditional on the point not being an outlier.}
  \item{\code{par}}{A list of the final estimates of the parameters
      of the genotype distribution. If \code{model = "hw"} then
      this will consist of \code{alpha}, the allele frequency.
      If \code{model = "f1"} or \code{model = "s1"} then this will
      consist of the parent genotype(s), the value of \code{fs1_alpha}
      (now just called \code{alpha}), and possibly the outlier
      proportion \code{out_prop}. If
      \code{model = "bb"} then this will consist of \code{alpha},
      the allele frequency, and \code{tau}, the overdispersion parameter.
      If \code{model = "norm"} then this will consist of \code{mu}, the
      normal mean, and \code{sigma}, the normal standard deviation (not variance).}
  \item{\code{geno}}{The posterior mode genotype. These are your
      genotype estimates.}
  \item{\code{maxpostprob}}{The maximum posterior probability. This
      is equivalent to the posterior probability of correctly
      genotyping each individual.}
  \item{\code{postmean}}{The posterior mean genotype. In downstream
      association studies, you might want to consider using these
      estimates.}
  \item{\code{input$refvec}}{The value of \code{refvec} provided by
      the user.}
  \item{\code{input$sizevec}}{The value of \code{sizevec} provided
      by the user.}
  \item{\code{input$ploidy}}{The value of \code{ploidy} provided
      by the user.}
  \item{\code{input$model}}{The value of \code{model} provided by
      the user.}
  \item{\code{input$p1ref}}{The value of \code{p1ref} provided by the user.}
  \item{\code{input$p1size}}{The value of \code{p1size} provided by the user.}
  \item{\code{input$p2ref}}{The value of \code{p2ref} provided by the user.}
  \item{\code{input$p2size}}{The value of \code{p2size} provided by the user.}
  \item{\code{prop_mis}}{The posterior proportion of individuals
      genotyped incorrectly.}
  \item{\code{out_prop}}{The estimated proportion of points that
      are outliers. Only available if \code{outliers = TRUE}.}
  \item{\code{prob_out}}{The ith element is the posterior probability
      that individual i is an outlier. Only available if
      \code{outliers = TRUE}.}
}
}
\description{
Genotype polyploid individuals from next generation
sequencing (NGS) data while assuming the genotype distribution is one of
several forms. \code{flexdog} does this while accounting for allele bias,
overdispersion, sequencing error, and possibly outlying observations
(if \code{model = "f1"} or \code{model = "s1"}).
}
\details{
Possible values of the genotype distribution (values of \code{model}) are:
\describe{
  \item{\code{"hw"}}{A binomial distribution that results from assuming that
      the population is in Hardy-Weinberg equilibrium (HWE). This actually does
      pretty well even when there are minor to moderate deviations from HWE.}
  \item{\code{"bb"}}{A beta-binomial distribution. This is a overdispersed
      version of "hw" and can be derived from a special case of the Balding-Nichols model.}
  \item{\code{"norm"}}{A distribution whose genotype frequencies are proportional
      to the density value of a normal with some mean and some standard deviation.
      Unlike the "bb" option, this will allow for distributions less dispersed than a binomial.
      This seems to be the most robust to violations in modeling assumptions, and so is the
      default.}
  \item{\code{"ash"}}{Any unimodal prior. This will run \code{ploidy} EM algorithms
      with a different center during each optimization. It returns the center (and its fit)
      with the highest likelihood.}
  \item{\code{"f1"}}{This prior assumes the individuals are all
      full-siblings resulting
      from one generation of a bi-parental cross.
      This model assumes
      a particular type of meiotic behavior: polysomic
      inheritance with
      bivalent, non-preferential pairing. Since this is
      a pretty strong
      and well-founded prior, we allow \code{outliers = TRUE}
      when \code{model = "f1"}.}
  \item{\code{"s1"}}{This prior assumes the individuals are
      all full-siblings resulting
      from one generation of selfing. I.e. there is only
      one parent. This model assumes
      a particular type of meiotic behavior: polysomic
      inheritance with
      bivalent, non-preferential pairing.
      Since this is a pretty strong and well-founded prior,
      we allow \code{outliers = TRUE} when \code{model = "s1"}.}
  \item{\code{"f1pp"}}{The same as \code{"f1"} but accounts for possible
      (and arbitrary levels of) preferential
      pairing during meiosis.}
  \item{\code{"s1pp"}}{The same as \code{"s1"} but accounts for
      possible preferential (and arbitrary levels of)
      pairing during meiosis.}
  \item{\code{"flex"}}{Generically any categorical distribution. Theoretically,
      this works well if you have a lot of individuals. In practice, it seems to
      be less robust to violations in modeling assumptions.}
  \item{\code{"uniform"}}{A discrete uniform distribution. This should never
      be used in practice. Please don't email me that \code{flexdog} doesn't
      work if you use this option.}
}

You might think a good default is \code{model = "uniform"} because it is
somehow an "uninformative prior." But it is very informative and tends to
work horribly in practice. The intuition is that it will estimate
the allele bias and sequencing error rates so that the estimated genotypes
are approximately uniform (since we are assuming that they are approximately
uniform). This will usually result in unintuitive genotyping since most
populations don't have a uniform genotype distribution.
I include it as an option only for completeness. Please, please don't use it.

The value of \code{prop_mis} is a very intuitive measure for
the quality of the SNP. \code{prop_mis} is the posterior
proportion of individuals mis-genotyped. So if you want only SNPS
that accurately genotype, say, 95\% of the individuals, you could
discard all SNPs with a \code{prop_mis} under \code{0.95}.

The value of \code{maxpostprob} is a very intuitive measure
for the quality of the genotype estimate of an individual.
This is the posterior probability of correctly genotyping
the individual when using \code{geno} (the posterior mode)
as the genotype estimate. So if you want to correctly genotype,
say, 95\% of individuals, you could discard all individuals
with a \code{maxpostprob} of under \code{0.95}.

In most datasets I've examined, allelic bias is a major issue. However,
you may fit the model assuming no allelic bias by setting
\code{update_bias = FALSE} and \code{bias_init = 1}.

Prior to using \code{flexdog}, during the read-mapping step,
you could try to get rid of allelic bias by
using WASP (\url{https://doi.org/10.1101/011221}). If you are successful
in removing the allelic bias (because its only source was the read-mapping
step), then setting \code{update_bias = FALSE} and \code{bias_init = 1}
would be reasonable. You can visually inspect SNPs for bias by
using \code{\link{plot_geno}}.

\code{flexdog}, like most methods, is invariant to which allele you
label as the "reference" and which you label as the "alternative".
That is, if you set \code{refvec} with the number of alternative
read-counts, then the resulting genotype estimates
will be the estimated allele dosage of the alternative allele.
}
\examples{
## A natural population. We will assume a
## normal prior since there are so few
## individuals.
data("uitdewilligen")
ploidy  <- 4
refvec  <- uitdewilligen$refmat[, 1]
sizevec <- uitdewilligen$sizemat[, 1]
fout    <- flexdog_full(refvec  = refvec,
                        sizevec = sizevec,
                        ploidy  = ploidy,
                        model   = "norm")
plot(fout)

}
\references{
Gerard, David, Luis Felipe Ventorim Ferrao,
Antonio Augusto Franco Garcia, and Matthew Stephens. 2018.
"Harnessing Empirical Bayes and Mendelian Segregation
for Genotyping Autopolyploids from Messy Sequencing Data."
\emph{bioRxiv}. Cold Spring Harbor Laboratory. doi:10.1101/281550.
}
\author{
David Gerard
}
