#include "vctrs.h"
#include "ptype-common.h"
#include "ptype2.h"
#include "type-data-frame.h"
#include "utils.h"

static SEXP syms_df_lossy_cast = NULL;
static SEXP fns_df_lossy_cast = NULL;

static SEXP new_compact_rownames(R_len_t n);


// [[ include("type-data-frame.h") ]]
bool is_data_frame(SEXP x) {
  if (TYPEOF(x) != VECSXP) {
    return false;
  }

  enum vctrs_class_type type = class_type(x);
  return
    type == vctrs_class_bare_data_frame ||
    type == vctrs_class_bare_tibble ||
    type == vctrs_class_data_frame;
}

// [[ include("type-data-frame.h") ]]
bool is_native_df(SEXP x) {
  enum vctrs_class_type type = class_type(x);
  return
    type == vctrs_class_bare_data_frame ||
    type == vctrs_class_bare_tibble;
}

// [[ include("type-data-frame.h") ]]
bool is_bare_data_frame(SEXP x) {
  return class_type(x) == vctrs_class_bare_data_frame;
}

// [[ include("type-data-frame.h") ]]
bool is_bare_tibble(SEXP x) {
  return class_type(x) == vctrs_class_bare_tibble;
}

// [[ include("type-data-frame.h") ]]
SEXP new_data_frame(SEXP x, R_len_t n) {
  x = PROTECT(r_clone_referenced(x));
  init_data_frame(x, n);

  UNPROTECT(1);
  return x;
}

static R_len_t df_size_from_list(SEXP x, SEXP n);
static R_len_t df_size_from_n(SEXP n);
static SEXP c_data_frame_class(SEXP cls);

// [[ register() ]]
SEXP vctrs_new_data_frame(SEXP args) {
  args = CDR(args);

  SEXP x = CAR(args); args = CDR(args);
  SEXP n = CAR(args); args = CDR(args);
  SEXP cls = CAR(args); args = CDR(args);
  SEXP attrib = args;

  PROTECT_INDEX pi;
  PROTECT_WITH_INDEX(attrib, &pi);

  if (TYPEOF(x) != VECSXP) {
    Rf_errorcall(R_NilValue, "`x` must be a list");
  }

  bool has_names = false;
  bool has_rownames = false;
  R_len_t size = df_size_from_list(x, n);

  SEXP out = PROTECT(r_clone_referenced(x));

  for (SEXP node = attrib; node != R_NilValue; node = CDR(node)) {
    SEXP tag = TAG(node);

    // We might add dynamic dots later on
    if (tag == R_ClassSymbol) {
      stop_internal("new_data_frame", "Can't supply `class` in `...`.");
    }

    if (tag == R_NamesSymbol) {
      has_names = true;
      continue;
    }

    if (tag == R_RowNamesSymbol) {
      // "row.names" is checked for consistency with n (if provided)
      if (size != rownames_size(CAR(node)) && n != R_NilValue) {
        Rf_errorcall(R_NilValue, "`n` and `row.names` must be consistent.");
      }

      has_rownames = true;
      continue;
    }
  }

  // Take names from `x` if `attrib` doesn't have any
  if (!has_names) {
    SEXP nms = vctrs_shared_empty_chr;
    if (Rf_length(out)) {
      nms = r_names(out);
    }
    PROTECT(nms);

    if (nms != R_NilValue) {
      attrib = Rf_cons(nms, attrib);
      SET_TAG(attrib, R_NamesSymbol);
      REPROTECT(attrib, pi);
    }

    UNPROTECT(1);
  }

  if (!has_rownames) {
    SEXP rn = PROTECT(new_compact_rownames(size));
    attrib = Rf_cons(rn, attrib);
    SET_TAG(attrib, R_RowNamesSymbol);

    UNPROTECT(1);
    REPROTECT(attrib, pi);
  }

  if (cls == R_NilValue) {
    cls = classes_data_frame;
  } else {
    cls = c_data_frame_class(cls);
  }
  PROTECT(cls);

  attrib = Rf_cons(cls, attrib);
  SET_TAG(attrib, R_ClassSymbol);

  UNPROTECT(1);
  REPROTECT(attrib, pi);


  SET_ATTRIB(out, attrib);
  SET_OBJECT(out, 1);

  UNPROTECT(2);
  return out;
}

static R_len_t df_size_from_list(SEXP x, SEXP n) {
  if (n == R_NilValue) {
    if (is_data_frame(x)) {
      return df_size(x);
    } else {
      return df_raw_size_from_list(x);
    }
  } else {
    return df_size_from_n(n);
  }
}

static R_len_t df_size_from_n(SEXP n) {
  if (TYPEOF(n) != INTSXP || Rf_length(n) != 1) {
    Rf_errorcall(R_NilValue, "`n` must be an integer of size 1");
  }

  return r_int_get(n, 0);
}

static SEXP c_data_frame_class(SEXP cls) {
  if (TYPEOF(cls) != STRSXP) {
    Rf_errorcall(R_NilValue, "`class` must be NULL or a character vector");
  }
  return chr_c(cls, classes_data_frame);
}

// [[ include("type-data-frame.h") ]]
enum rownames_type rownames_type(SEXP x) {
  switch (TYPEOF(x)) {
  case STRSXP:
    return ROWNAMES_IDENTIFIERS;
  case INTSXP:
    if (Rf_length(x) == 2 && INTEGER(x)[0] == NA_INTEGER) {
      return ROWNAMES_AUTOMATIC_COMPACT;
    } else {
      return ROWNAMES_AUTOMATIC;
    }
  default:
    Rf_error("Corrupt data in `rownames_type()`: Unexpected type `%s`.",
             Rf_type2char(TYPEOF(x)));
  }
}

static R_len_t compact_rownames_length(SEXP x) {
  return abs(INTEGER(x)[1]);
}

// [[ include("type-data-frame.h") ]]
R_len_t rownames_size(SEXP rn) {
  switch (rownames_type(rn)) {
  case ROWNAMES_IDENTIFIERS:
  case ROWNAMES_AUTOMATIC:
    return Rf_length(rn);
  case ROWNAMES_AUTOMATIC_COMPACT:
    return compact_rownames_length(rn);
  }

  never_reached("rownames_size");
}

static void init_bare_data_frame(SEXP x, R_len_t n);

// [[ include("type-data-frame.h") ]]
void init_data_frame(SEXP x, R_len_t n) {
  Rf_setAttrib(x, R_ClassSymbol, classes_data_frame);
  init_bare_data_frame(x, n);
}
// [[ include("type-data-frame.h") ]]
void init_tibble(SEXP x, R_len_t n) {
  Rf_setAttrib(x, R_ClassSymbol, classes_tibble);
  init_bare_data_frame(x, n);
}

static void init_bare_data_frame(SEXP x, R_len_t n) {
  if (Rf_length(x) == 0) {
    Rf_setAttrib(x, R_NamesSymbol, vctrs_shared_empty_chr);
  }

  init_compact_rownames(x, n);
}

// [[ include("type-data-frame.h") ]]
void init_compact_rownames(SEXP x, R_len_t n) {
  SEXP rn = PROTECT(new_compact_rownames(n));
  Rf_setAttrib(x, R_RowNamesSymbol, rn);
  UNPROTECT(1);
}

static SEXP new_compact_rownames(R_len_t n) {
  if (n <= 0) {
    return vctrs_shared_empty_int;
  }

  SEXP out = Rf_allocVector(INTSXP, 2);
  int* out_data = INTEGER(out);
  out_data[0] = NA_INTEGER;
  out_data[1] = -n;
  return out;
}

// [[ include("type-data-frame.h") ]]
SEXP df_rownames(SEXP x) {
  // Required, because getAttrib() already does the transformation to a vector,
  // and getAttrib0() is hidden
  SEXP node = ATTRIB(x);

  while (node != R_NilValue) {
    SEXP tag = TAG(node);

    if (tag == R_RowNamesSymbol) {
      return CAR(node);
    }

    node = CDR(node);
  }

  return R_NilValue;
}


// vctrs type methods ------------------------------------------------

// [[ register() ]]
SEXP vctrs_df_ptype2_opts(SEXP x, SEXP y, SEXP opts, SEXP x_arg, SEXP y_arg) {
  struct vctrs_arg c_x_arg = vec_as_arg(x_arg);
  struct vctrs_arg c_y_arg = vec_as_arg(y_arg);

  const struct ptype2_opts c_opts = new_ptype2_opts(x, y, &c_x_arg, &c_y_arg, opts);

  return df_ptype2(&c_opts);
}

static
SEXP df_ptype2_match(const struct ptype2_opts* opts,
                     SEXP x_names,
                     SEXP y_names);

static
SEXP df_ptype2_loop(const struct ptype2_opts* opts,
                    SEXP y_names);

// [[ include("type-data-frame.h") ]]
SEXP df_ptype2(const struct ptype2_opts* opts) {
  SEXP x_names = PROTECT(r_names(opts->x));
  SEXP y_names = PROTECT(r_names(opts->y));

  SEXP out = R_NilValue;

  if (equal_object(x_names, y_names)) {
    out = df_ptype2_loop(opts, x_names);
  } else {
    out = df_ptype2_match(opts, x_names, y_names);
  }

  UNPROTECT(2);
  return out;
}

SEXP df_ptype2_match(const struct ptype2_opts* opts,
                     SEXP x_names,
                     SEXP y_names) {
  SEXP x = opts->x;
  SEXP y = opts->y;

  SEXP x_dups_pos = PROTECT(vec_match(x_names, y_names));
  SEXP y_dups_pos = PROTECT(vec_match(y_names, x_names));

  int* x_dups_pos_data = INTEGER(x_dups_pos);
  int* y_dups_pos_data = INTEGER(y_dups_pos);

  R_len_t x_len = Rf_length(x_names);
  R_len_t y_len = Rf_length(y_names);

  // Count columns that are only in `y`
  R_len_t rest_len = 0;
  for (R_len_t i = 0; i < y_len; ++i) {
    if (y_dups_pos_data[i] == NA_INTEGER) {
      ++rest_len;
    }
  }

  R_len_t out_len = x_len + rest_len;
  SEXP out = PROTECT(Rf_allocVector(VECSXP, out_len));
  SEXP nms = PROTECT(Rf_allocVector(STRSXP, out_len));
  Rf_setAttrib(out, R_NamesSymbol, nms);

  R_len_t i = 0;

  // Fill in prototypes of all the columns that are in `x`, in order
  for (; i < x_len; ++i) {
    R_len_t dup = x_dups_pos_data[i];

    struct arg_data_index x_arg_data = new_index_arg_data(r_chr_get_c_string(x_names, i),
                                                          opts->x_arg);
    struct vctrs_arg named_x_arg = new_index_arg(opts->x_arg, &x_arg_data);

    SEXP col = VECTOR_ELT(x, i);
    struct ptype2_opts col_opts = *opts;
    col_opts.x = col;
    col_opts.x_arg = &named_x_arg;

    SEXP type;
    if (dup == NA_INTEGER) {
      col_opts.y = vctrs_shared_empty_uns;
      col_opts.y_arg = NULL;
      type = vec_ptype2_from_unspecified(&col_opts,
                                         vec_typeof(col),
                                         col,
                                         &named_x_arg);
    } else {
      --dup; // 1-based index

      struct arg_data_index y_arg_data = new_index_arg_data(r_chr_get_c_string(y_names, dup),
                                                            opts->y_arg);
      struct vctrs_arg named_y_arg = new_index_arg(opts->y_arg, &y_arg_data);
      col_opts.y = VECTOR_ELT(y, dup);
      col_opts.y_arg = &named_y_arg;

      int _left;
      type = vec_ptype2_opts(&col_opts, &_left);
    }

    SET_VECTOR_ELT(out, i, type);
    SET_STRING_ELT(nms, i, STRING_ELT(x_names, i));
  }

  // Fill in prototypes of the columns that are only in `y`
  for (R_len_t j = 0; i < out_len; ++j) {
    R_len_t dup = y_dups_pos_data[j];

    if (dup == NA_INTEGER) {
      SEXP col = VECTOR_ELT(y, j);

      struct arg_data_index y_arg_data = new_index_arg_data(r_chr_get_c_string(y_names, j),
                                                            opts->y_arg);
      struct vctrs_arg named_y_arg = new_index_arg(opts->y_arg, &y_arg_data);

      struct ptype2_opts col_opts = *opts;
      col_opts.y = col;
      col_opts.y_arg = &named_y_arg;
      col_opts.x = vctrs_shared_empty_uns;
      col_opts.x_arg = NULL;
      SEXP type = vec_ptype2_from_unspecified(&col_opts,
                                              vec_typeof(col),
                                              col,
                                              &named_y_arg);

      SET_VECTOR_ELT(out, i, type);
      SET_STRING_ELT(nms, i, STRING_ELT(y_names, j));
      ++i;
    }
  }

  init_data_frame(out, 0);

  UNPROTECT(4);
  return out;
}

static
SEXP df_ptype2_loop(const struct ptype2_opts* opts,
                    SEXP names) {
  SEXP x = opts->x;
  SEXP y = opts->y;

  R_len_t len = Rf_length(names);

  SEXP out = PROTECT(Rf_allocVector(VECSXP, len));
  Rf_setAttrib(out, R_NamesSymbol, names);

  for (R_len_t i = 0; i < len; ++i) {
    const char* name = r_chr_get_c_string(names, i);

    struct arg_data_index x_arg_data = new_index_arg_data(name, opts->x_arg);
    struct arg_data_index y_arg_data = new_index_arg_data(name, opts->y_arg);
    struct vctrs_arg named_x_arg = new_index_arg(opts->x_arg, &x_arg_data);
    struct vctrs_arg named_y_arg = new_index_arg(opts->y_arg, &y_arg_data);

    struct ptype2_opts col_opts = *opts;
    col_opts.x = VECTOR_ELT(x, i);
    col_opts.y = VECTOR_ELT(y, i);
    col_opts.x_arg = &named_x_arg;
    col_opts.y_arg = &named_y_arg;
    int _left;

    SEXP type = vec_ptype2_opts(&col_opts, &_left);

    SET_VECTOR_ELT(out, i, type);
  }

  init_data_frame(out, 0);

  UNPROTECT(1);
  return out;
}

// [[ register() ]]
SEXP vctrs_df_cast_opts(SEXP x, SEXP to, SEXP opts, SEXP x_arg, SEXP to_arg) {
  struct vctrs_arg c_x_arg = vec_as_arg(x_arg);
  struct vctrs_arg c_to_arg = vec_as_arg(to_arg);

  const struct cast_opts c_opts = new_cast_opts(x, to, &c_x_arg, &c_to_arg, opts);

  return df_cast_opts(&c_opts);
}

static SEXP df_cast_match(const struct cast_opts* opts,
                          SEXP x_names,
                          SEXP to_names);

static SEXP df_cast_loop(const struct cast_opts* opts, SEXP names);

// Take all columns of `to` and preserve the order. Common columns are
// cast to their types in `to`. Extra `x` columns are dropped and
// cause a lossy cast. Extra `to` columns are filled with missing
// values.
// [[ include("cast.h") ]]
SEXP df_cast_opts(const struct cast_opts* opts) {
  SEXP x_names = PROTECT(r_names(opts->x));
  SEXP to_names = PROTECT(r_names(opts->to));

  if (x_names == R_NilValue || to_names == R_NilValue) {
    stop_internal("df_cast_opts", "Data frame must have names.");
  }

  SEXP out = R_NilValue;

  if (equal_object(x_names, to_names)) {
    out = df_cast_loop(opts, x_names);
  } else {
    out = df_cast_match(opts, x_names, to_names);
  }

  UNPROTECT(2);
  return out;
}

static SEXP df_cast_match(const struct cast_opts* opts,
                          SEXP x_names,
                          SEXP to_names) {
  SEXP x = opts->x;
  SEXP to = opts->to;

  SEXP to_dups_pos = PROTECT(vec_match(to_names, x_names));
  int* to_dups_pos_data = INTEGER(to_dups_pos);

  R_len_t to_len = Rf_length(to_dups_pos);
  SEXP out = PROTECT(Rf_allocVector(VECSXP, to_len));
  Rf_setAttrib(out, R_NamesSymbol, to_names);

  R_len_t size = df_size(x);
  R_len_t common_len = 0;

  for (R_len_t i = 0; i < to_len; ++i) {
    R_len_t pos = to_dups_pos_data[i];

    SEXP col;
    if (pos == NA_INTEGER) {
      SEXP to_col = VECTOR_ELT(to, i);
      col = vec_init(to_col, size);

      // FIXME: Need to initialise the vector because we currently use
      // `vec_assign()` in `vec_rbind()` before falling back. Attach
      // an attribute to recognise unspecified vectors in
      // `base_c_invoke()`.
      if (opts->fallback.s3 && vec_is_common_class_fallback(to_col)) {
        PROTECT(col);
        Rf_setAttrib(col, Rf_install("vctrs:::unspecified"), vctrs_shared_true);
        UNPROTECT(1);
      }
    } else {
      --pos; // 1-based index
      ++common_len;

      struct arg_data_index x_arg_data = new_index_arg_data(r_chr_get_c_string(x_names, pos), opts->x_arg);
      struct arg_data_index to_arg_data = new_index_arg_data(r_chr_get_c_string(to_names, i), opts->to_arg);
      struct vctrs_arg named_x_arg = new_index_arg(opts->x_arg, &x_arg_data);
      struct vctrs_arg named_to_arg = new_index_arg(opts->to_arg, &to_arg_data);

      struct cast_opts col_opts = {
        .x = VECTOR_ELT(x, pos),
        .to = VECTOR_ELT(to, i),
        .x_arg = &named_x_arg,
        .to_arg = &named_to_arg,
        .fallback = opts->fallback
      };
      col = vec_cast_opts(&col_opts);
    }

    SET_VECTOR_ELT(out, i, col);
  }

  // Restore data frame size before calling `vec_restore()`. `x` and
  // `to` might not have any columns to compute the original size.
  init_data_frame(out, size);
  Rf_setAttrib(out, R_RowNamesSymbol, df_rownames(x));

  R_len_t extra_len = Rf_length(x) - common_len;
  if (extra_len) {
    out = vctrs_dispatch3(syms_df_lossy_cast, fns_df_lossy_cast,
                          syms_out, out,
                          syms_x, x,
                          syms_to, to);
  }

  UNPROTECT(2);
  return out;
}

static SEXP df_cast_loop(const struct cast_opts* opts, SEXP names) {
  SEXP x = opts->x;
  SEXP to = opts->to;

  R_len_t len = Rf_length(names);

  SEXP out = PROTECT(Rf_allocVector(VECSXP, len));
  Rf_setAttrib(out, R_NamesSymbol, names);

  R_len_t size = df_size(x);

  for (R_len_t i = 0; i < len; ++i) {
    const char* name = r_chr_get_c_string(names, i);

    struct arg_data_index x_arg_data = new_index_arg_data(name, opts->x_arg);
    struct arg_data_index to_arg_data = new_index_arg_data(name, opts->to_arg);
    struct vctrs_arg named_x_arg = new_index_arg(opts->x_arg, &x_arg_data);
    struct vctrs_arg named_to_arg = new_index_arg(opts->to_arg, &to_arg_data);

    struct cast_opts col_opts = {
      .x = VECTOR_ELT(x, i),
      .to = VECTOR_ELT(to, i),
      .x_arg = &named_x_arg,
      .to_arg = &named_to_arg,
      .fallback = opts->fallback
    };
    SEXP col = vec_cast_opts(&col_opts);

    SET_VECTOR_ELT(out, i, col);
  }

  // Restore data frame size before calling `vec_restore()`. `x` and
  // `to` might not have any columns to compute the original size.
  init_data_frame(out, size);
  Rf_setAttrib(out, R_RowNamesSymbol, df_rownames(x));

  UNPROTECT(1);
  return out;
}

// If negative index, value is appended
SEXP df_poke(SEXP x, R_len_t i, SEXP value) {
  if (i >= 0) {
    SET_VECTOR_ELT(x, i, value);
    return x;
  }

  R_len_t ncol = Rf_length(x);

  SEXP tmp = PROTECT(r_resize(x, ncol + 1));
  Rf_copyMostAttrib(x, tmp);
  x = tmp;

  SET_VECTOR_ELT(x, ncol, value);

  UNPROTECT(1);
  return x;
}
SEXP df_poke_at(SEXP x, SEXP name, SEXP value) {
  SEXP names = PROTECT(r_names(x));
  R_len_t i = r_chr_find(names, name);
  UNPROTECT(1);

  x = PROTECT(df_poke(x, i, value));

  if (i < 0) {
    SEXP names = PROTECT(r_names(x));
    SET_STRING_ELT(names, Rf_length(x) - 1, name);
    UNPROTECT(1);
  }

  UNPROTECT(1);
  return x;
}

// [[ include("type-data-frame.h") ]]
R_len_t df_flat_width(SEXP x) {
  R_len_t n = Rf_length(x);
  R_len_t out = n;

  for (R_len_t i = 0; i < n; ++i) {
    SEXP col = VECTOR_ELT(x, i);
    if (is_data_frame(col)) {
      out = out + df_flat_width(col) - 1;
    }
  }

  return out;
}

// [[ register() ]]
SEXP vctrs_df_flat_width(SEXP x) {
  return r_int(df_flat_width(x));
}


static R_len_t df_flatten_loop(SEXP x, SEXP out, SEXP out_names, R_len_t counter);

// Might return duplicate names. Currently only used for equality
// proxy so this doesn't matter. A less bare bone version would repair
// names.
//
// [[ register(); include("type-data-frame.h") ]]
SEXP df_flatten(SEXP x) {
  R_len_t width = df_flat_width(x);
  SEXP out = PROTECT(Rf_allocVector(VECSXP, width));
  SEXP out_names = PROTECT(Rf_allocVector(STRSXP, width));
  r_poke_names(out, out_names);

  df_flatten_loop(x, out, out_names, 0);
  init_data_frame(out, df_size(x));

  UNPROTECT(2);
  return out;
}

static R_len_t df_flatten_loop(SEXP x, SEXP out, SEXP out_names, R_len_t counter) {
  R_len_t n = Rf_length(x);
  SEXP x_names = PROTECT(r_names(x));

  for (R_len_t i = 0; i < n; ++i) {
    SEXP col = VECTOR_ELT(x, i);

    if (is_data_frame(col)) {
      counter = df_flatten_loop(col, out, out_names, counter);
    } else {
      SET_VECTOR_ELT(out, counter, col);
      SET_STRING_ELT(out_names, counter, STRING_ELT(x_names, i));
      ++counter;
    }
  }

  UNPROTECT(1);
  return counter;
}

SEXP df_repair_names(SEXP x, struct name_repair_opts* name_repair) {
  SEXP nms = PROTECT(r_names(x));
  SEXP repaired = PROTECT(vec_as_names(nms, name_repair));

  // Should this go through proxy and restore so that classes can
  // update metadata and check invariants when special columns are
  // renamed?
  if (nms != repaired) {
    x = PROTECT(r_clone_referenced(x));
    r_poke_names(x, repaired);
    UNPROTECT(1);
  }

  UNPROTECT(2);
  return x;
}


void vctrs_init_type_data_frame(SEXP ns) {
  syms_df_lossy_cast = Rf_install("df_lossy_cast");
  fns_df_lossy_cast = Rf_findVar(syms_df_lossy_cast, ns);
}
