# ------------------------------------------------------------------------------
# vec_order_radix(<integer>) - insertion

test_that("can order size zero input", {
  expect_identical(vec_order_radix(integer()), integer())
})

test_that("can order integers", {
  x <- c(2L, 3L, 1L, 5L)
  expect_identical(vec_order_radix(x), order(x))
})

test_that("can order sorted vector", {
  x <- 1:5
  expect_identical(vec_order_radix(x), order(x))
})

test_that("orders correctly around the UINT8_MAX boundary", {
  x <- 251:255
  expect_identical(vec_order_radix(x), order(x))
})

test_that("ordering on ties is done stably", {
  x <- c(1L, 3L, 1L, 3L)
  expect_identical(vec_order_radix(x)[1:2], c(1L, 3L))
  expect_identical(vec_order_radix(x)[3:4], c(2L, 4L))
})

test_that("`NA` order defaults to last", {
  x <- c(1L, NA_integer_, 3L)
  expect_identical(vec_order_radix(x), c(1L, 3L, 2L))
})

test_that("`NA` order can be first", {
  x <- c(1L, NA_integer_, 3L)
  expect_identical(vec_order_radix(x, na_value = "smallest"), c(2L, 1L, 3L))
})

test_that("`direction` can be set to `desc`", {
  x <- c(1L, .Machine$integer.max, 3L)
  expect_identical(vec_order_radix(x, direction = "desc"), c(2L, 3L, 1L))
})

test_that("all combinations of `direction` and `na_value` work", {
  x <- c(3L, NA_integer_, 1L, 2L)

  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "asc")],
    x[order(x, na.last = TRUE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "asc")],
    x[order(x, na.last = FALSE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "desc")],
    x[order(x, na.last = TRUE, decreasing = TRUE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "desc")],
    x[order(x, na.last = FALSE, decreasing = TRUE)]
  )
})

test_that("all `NA` values works", {
  x <- c(NA_integer_, NA_integer_)
  expect_identical(vec_order_radix(x), order(x))
})

test_that("can order when in expected order", {
  x <- c(1L, 1L, 2L, NA, NA)
  expect_identical(vec_order_radix(x, direction = "asc", na_value = "largest"), 1:5)

  x <- c(3L, 3L, 2L, NA, NA)
  expect_identical(vec_order_radix(x, direction = "desc", na_value = "largest"), 1:5)

  x <- c(NA, NA, 1L, 1L, 2L)
  expect_identical(vec_order_radix(x, direction = "asc", na_value = "smallest"), 1:5)

  x <- c(NA, NA, 3L, 3L, 2L)
  expect_identical(vec_order_radix(x, direction = "desc", na_value = "smallest"), 1:5)
})

test_that("can order when in strictly opposite of expected order (no ties)", {
  x <- c(NA, 2L, 1L)
  expect_identical(vec_order_radix(x, direction = "asc", na_value = "largest"), 3:1)

  x <- c(NA, 1L, 2L)
  expect_identical(vec_order_radix(x, direction = "desc", na_value = "largest"), 3:1)

  x <- c(2L, 1L, NA)
  expect_identical(vec_order_radix(x, direction = "asc", na_value = "smallest"), 3:1)

  x <- c(1L, 2L, NA)
  expect_identical(vec_order_radix(x, direction = "desc", na_value = "smallest"), 3:1)
})

# ------------------------------------------------------------------------------
# vec_order_radix(<integer>) - counting

# To trigger counting ordering, get above the insertion order boundary and then
# have a range less than the counting order range boundary.

test_that("can order integers with counting order", {
  x <- (ORDER_INSERTION_BOUNDARY + 1L):1L
  expect_identical(vec_order_radix(x), order(x))
})

test_that("can order sorted vector", {
  x <- 1:(ORDER_INSERTION_BOUNDARY + 1L)
  expect_identical(vec_order_radix(x), order(x))
})

test_that("ordering on ties is done stably", {
  x <- c(1:ORDER_INSERTION_BOUNDARY, 1L)
  expect_identical(vec_order_radix(x)[1:2], c(1L, ORDER_INSERTION_BOUNDARY + 1L))
})

test_that("all combinations of `direction` and `na_value` work", {
  x <- c(3L, NA_integer_, 1L, 2L, 1:ORDER_INSERTION_BOUNDARY)

  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "asc")],
    x[order(x, na.last = TRUE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "asc")],
    x[order(x, na.last = FALSE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "desc")],
    x[order(x, na.last = TRUE, decreasing = TRUE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "desc")],
    x[order(x, na.last = FALSE, decreasing = TRUE)]
  )
})

# ------------------------------------------------------------------------------
# vec_order_radix(<integer>) - radix

# To trigger radix ordering, get above the insertion order boundary and then
# have a range greater than the counting order range boundary.

test_that("can order integers with radix order", {
  x <- c(INT_ORDER_COUNTING_RANGE_BOUNDARY + 1L, 1:ORDER_INSERTION_BOUNDARY)
  expect_identical(vec_order_radix(x), order(x))
})

test_that("can order sorted vector", {
  x <- c(1:ORDER_INSERTION_BOUNDARY, INT_ORDER_COUNTING_RANGE_BOUNDARY + 1L)
  expect_identical(vec_order_radix(x), order(x))
})

test_that("ordering on ties is done stably", {
  x <- c(1:ORDER_INSERTION_BOUNDARY, 1L, INT_ORDER_COUNTING_RANGE_BOUNDARY + 1L)
  expect_identical(vec_order_radix(x)[1:2], c(1L, ORDER_INSERTION_BOUNDARY + 1L))
})

test_that("all combinations of `direction` and `na_value` work", {
  x <- c(3L, NA_integer_, 1L, 2L, 1:ORDER_INSERTION_BOUNDARY, INT_ORDER_COUNTING_RANGE_BOUNDARY + 1L)

  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "asc")],
    x[order(x, na.last = TRUE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "asc")],
    x[order(x, na.last = FALSE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "desc")],
    x[order(x, na.last = TRUE, decreasing = TRUE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "desc")],
    x[order(x, na.last = FALSE, decreasing = TRUE)]
  )
})

test_that("can order all 1 value", {
  x <- rep(1L, ORDER_INSERTION_BOUNDARY + 1L)
  expect_identical(vec_order_radix(x), base_order(x))
  expect_identical(vec_order_radix(x, direction = "desc"), base_order(x, decreasing = TRUE))
})

test_that("all `NA` values works - ensures that we can compute the 'range' of all NAs", {
  x <- rep(NA_integer_, ORDER_INSERTION_BOUNDARY + 1L)
  expect_identical(vec_order_radix(x), base_order(x))
  expect_identical(vec_order_radix(x, direction = "desc"), base_order(x, decreasing = TRUE))
})

test_that("can order with many NAs first", {
  x <- c(rep(NA_integer_, ORDER_INSERTION_BOUNDARY + 1L), 2L)
  expect_identical(vec_order_radix(x), base_order(x))
  expect_identical(vec_order_radix(x, na_value = "smallest"), base_order(x, na.last = FALSE))
})

# ------------------------------------------------------------------------------
# vec_order_radix(<logical>)

# Really this just goes through the integer infrastructure. Just checking that
# it is working.

test_that("can order size zero input", {
  expect_identical(vec_order_radix(logical()), integer())
})

test_that("can order logicals", {
  x <- c(FALSE, TRUE, FALSE)
  expect_identical(vec_order_radix(x), order(x))
})

test_that("all combinations of `direction` and `na_value` work", {
  x <- c(TRUE, NA, FALSE)

  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "asc")],
    x[order(x, na.last = TRUE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "asc")],
    x[order(x, na.last = FALSE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "desc")],
    x[order(x, na.last = TRUE, decreasing = TRUE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "desc")],
    x[order(x, na.last = FALSE, decreasing = TRUE)]
  )
})

test_that("all `NA` values works", {
  x <- c(NA, NA)
  expect_identical(vec_order_radix(x), order(x))
})

# ------------------------------------------------------------------------------
# vec_order_radix(<double>) - insertion

test_that("can order size zero input", {
  expect_identical(vec_order_radix(double()), integer())
})

test_that("can order doubles", {
  x <- c(2, 3, 1, 5)
  expect_identical(vec_order_radix(x), order(x))
})

test_that("can order sorted vector", {
  x <- 1:5 + 0
  expect_identical(vec_order_radix(x), order(x))
})

test_that("ordering on ties is done stably", {
  x <- c(1, 3, 1, 3)
  expect_identical(vec_order_radix(x)[1:2], c(1L, 3L))
  expect_identical(vec_order_radix(x)[3:4], c(2L, 4L))
})

test_that("`NA` order defaults to last", {
  x <- c(1, NA_real_, 3)
  expect_identical(vec_order_radix(x), c(1L, 3L, 2L))
})

test_that("`NA` order can be first", {
  x <- c(1, NA_real_, 3)
  expect_identical(vec_order_radix(x, na_value = "smallest"), c(2L, 1L, 3L))
})

test_that("`direction` can be set to `desc`", {
  x <- c(1, 5, 3)
  expect_identical(vec_order_radix(x, direction = "desc"), c(2L, 3L, 1L))
})

test_that("all combinations of `direction` and `na_value` work", {
  x <- c(3, NA_real_, 1, 2)

  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "asc")],
    x[order(x, na.last = TRUE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "asc")],
    x[order(x, na.last = FALSE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "desc")],
    x[order(x, na.last = TRUE, decreasing = TRUE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "desc")],
    x[order(x, na.last = FALSE, decreasing = TRUE)]
  )
})

test_that("all `NA` values works", {
  x <- c(NA_real_, NA_real_)
  expect_identical(vec_order_radix(x), order(x))
})

test_that("NA_real_ and NaN look identical for ordering", {
  x <- c(NA_real_, NaN)
  expect_identical(vec_order_radix(x, na_value = "largest"), c(1L, 2L))
  expect_identical(vec_order_radix(x, na_value = "smallest"), c(1L, 2L))
})

test_that("-Inf / Inf order correctly", {
  x <- c(0, -Inf, Inf)
  expect_identical(vec_order_radix(x, direction = "asc"), c(2L, 1L, 3L))
  expect_identical(vec_order_radix(x, direction = "desc"), c(3L, 1L, 2L))
})

test_that("-0 and 0 order identically / stably", {
  x <- c(0, -0)
  expect_identical(vec_order_radix(x, direction = "desc"), c(1L, 2L))
  expect_identical(vec_order_radix(x, direction = "asc"), c(1L, 2L))
})

test_that("can order when in expected order", {
  x <- c(1, 1, 2, NA, NaN)
  expect_identical(vec_order_radix(x, direction = "asc", na_value = "largest"), 1:5)

  x <- c(3, 3, 2, NA, NaN)
  expect_identical(vec_order_radix(x, direction = "desc", na_value = "largest"), 1:5)

  x <- c(NA, NaN, 1, 1, 2)
  expect_identical(vec_order_radix(x, direction = "asc", na_value = "smallest"), 1:5)

  x <- c(NA, NaN, 3, 3, 2)
  expect_identical(vec_order_radix(x, direction = "desc", na_value = "smallest"), 1:5)
})

test_that("can order when in strictly opposite of expected order (no ties)", {
  x <- c(NA, 2, 1)
  expect_identical(vec_order_radix(x, direction = "asc", na_value = "largest"), 3:1)

  x <- c(NA, 1, 2)
  expect_identical(vec_order_radix(x, direction = "desc", na_value = "largest"), 3:1)

  x <- c(2, 1, NA)
  expect_identical(vec_order_radix(x, direction = "asc", na_value = "smallest"), 3:1)

  x <- c(1, 2, NA)
  expect_identical(vec_order_radix(x, direction = "desc", na_value = "smallest"), 3:1)
})

# ------------------------------------------------------------------------------
# vec_order_radix(<double>) - radix

# To trigger radix ordering, get above the insertion order boundary. There is
# no intermediate counting sort for doubles.

test_that("can order doubles with radix order", {
  x <- (ORDER_INSERTION_BOUNDARY + 1L):1L + 0
  expect_identical(vec_order_radix(x), order(x))
})

test_that("can order sorted vector", {
  x <- 1:(ORDER_INSERTION_BOUNDARY + 1L) + 0
  expect_identical(vec_order_radix(x), order(x))
})

test_that("ordering on ties is done stably", {
  x <- c(1:ORDER_INSERTION_BOUNDARY, 1L) + 0
  expect_identical(vec_order_radix(x)[1:2], c(1L, ORDER_INSERTION_BOUNDARY + 1L))
})

test_that("all combinations of `direction` and `na_value` work", {
  x <- c(3, NA_real_, 1, 2, 1:ORDER_INSERTION_BOUNDARY)

  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "asc")],
    x[order(x, na.last = TRUE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "asc")],
    x[order(x, na.last = FALSE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "desc")],
    x[order(x, na.last = TRUE, decreasing = TRUE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "desc")],
    x[order(x, na.last = FALSE, decreasing = TRUE)]
  )
})

test_that("all `NA` values works", {
  x <- rep(NA_real_, ORDER_INSERTION_BOUNDARY + 1L)
  expect_identical(vec_order_radix(x), order(x))
})

test_that("NA_real_ and NaN look identical for ordering", {
  x <- rep(c(NA_real_, NaN), ORDER_INSERTION_BOUNDARY + 1L)
  expect_identical(vec_order_radix(x, na_value = "largest"), seq_along(x))
  expect_identical(vec_order_radix(x, na_value = "smallest"), seq_along(x))
})

test_that("-Inf / Inf order correctly", {
  x <- c(rep(0, ORDER_INSERTION_BOUNDARY), -Inf, Inf)
  expect_identical(vec_order_radix(x, direction = "asc"), order(x, decreasing = FALSE))
  expect_identical(vec_order_radix(x, direction = "desc"), order(x, decreasing = TRUE))
})

test_that("-0 and 0 order identically / stably", {
  x <- c(rep(0, ORDER_INSERTION_BOUNDARY), -0)
  expect_identical(vec_order_radix(x, direction = "desc"), order(x, decreasing = TRUE))
  expect_identical(vec_order_radix(x, direction = "asc"), order(x, decreasing = FALSE))
})

# ------------------------------------------------------------------------------
# vec_order_radix(<complex>)

test_that("can order size zero input", {
  expect_identical(vec_order_radix(complex()), integer())
})

test_that("can order complex", {
  x <- complex(real = c(3, 1, 2))
  expect_identical(vec_order_radix(x), c(2L, 3L, 1L))
})

test_that("ordering on ties is done stably", {
  x <- complex(real = c(1, 3, 1, 3))
  expect_identical(vec_order_radix(x)[1:2], c(1L, 3L))
  expect_identical(vec_order_radix(x)[3:4], c(2L, 4L))
})

test_that("imaginary section is used to break ties", {
  x <- complex(
    real = c(1L, 2L, 1L),
    imaginary = c(3L, 2L, 1L)
  )
  expect_identical(vec_order_radix(x), c(3L, 1L, 2L))
})

test_that("can be used in a data frame", {
  x <- c(1L, 1L, 1L, 2L, 1L)

  y <- complex(
    real = c(1L, 2L, 1L, 3L, 1L),
    imaginary = c(3L, 2L, 1L, 4L, 1L)
  )

  z <- c(1, 2, 5, 4, 3)

  # as second column
  df1 <- data.frame(x = x, y = y)

  # as first column
  df2 <- data.frame(y = y, x = x)

  # as second column with a third after it to break ties
  df3 <- data.frame(x = x, y = y, z = z)

  # Base R can't do radix sorting with complex
  expect_identical(vec_order_radix(df1), c(3L, 5L, 1L, 2L, 4L))
  expect_identical(vec_order_radix(df2), c(3L, 5L, 1L, 2L, 4L))
  expect_identical(vec_order_radix(df3), c(5L, 3L, 1L, 2L, 4L))
})

test_that("all combinations of `direction` and `na_value` work", {
  x <- complex(real = c(3, NA, 1.5, 2, NA), imaginary = c(1, 1, 1, 1, 2))

  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "asc")],
    x[order(x, na.last = TRUE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "asc")],
    x[order(x, na.last = FALSE, decreasing = FALSE)]
  )

  # Base R is actually wrong here! It doesn't consider the imaginary part
  # when ordering in decreasing order with `NA` real.
  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "desc")],
    #x[order(x, na.last = TRUE, decreasing = TRUE)]
    x[c(1L, 4L, 3L, 5L, 2L)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "desc")],
    #x[order(x, na.last = FALSE, decreasing = TRUE)]
    x[c(5L, 2L, 1L, 4L, 3L)]
  )
})

# ------------------------------------------------------------------------------
# vec_order_radix(<character>) - insertion

test_that("can order size zero input", {
  expect_identical(vec_order_radix(character()), integer())
})

test_that("can order characters", {
  x <- c("xy", "x", "a", "bc")
  expect_identical(vec_order_radix(x), order(x))
})

test_that("can order sorted vector", {
  x <- c("a", "b", "c")
  expect_identical(vec_order_radix(x), order(x))
})

test_that("ordering on ties is done stably", {
  x <- c("ab", "ba", "ab", "ba")
  expect_identical(vec_order_radix(x)[1:2], c(1L, 3L))
  expect_identical(vec_order_radix(x)[3:4], c(2L, 4L))
})

test_that("`NA` order defaults to last", {
  x <- c("x", NA_character_, "y")
  expect_identical(vec_order_radix(x), c(1L, 3L, 2L))
})

test_that("`NA` order can be first", {
  x <- c("x", NA_character_, "y")
  expect_identical(vec_order_radix(x, na_value = "smallest"), c(2L, 1L, 3L))
})

test_that("`direction` can be set to `desc`", {
  x <- c("x", "abcde", "yz")
  expect_identical(vec_order_radix(x, direction = "desc"), c(3L, 1L, 2L))
})

test_that("all combinations of `direction` and `na_value` work", {
  x <- c("aaa", NA_character_, "a", "aa")

  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "asc")],
    x[order(x, na.last = TRUE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "asc")],
    x[order(x, na.last = FALSE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "desc")],
    x[order(x, na.last = TRUE, decreasing = TRUE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "desc")],
    x[order(x, na.last = FALSE, decreasing = TRUE)]
  )
})

test_that("all `NA` values works", {
  x <- c(NA_character_, NA_character_)
  expect_identical(vec_order_radix(x), order(x))
})

test_that("can order empty string vs ASCII value 1 'Start of Header'", {
  x <- c("\001", "")
  expect_identical(vec_order_radix(x), c(2L, 1L))
})

test_that("can be used in a data frame", {
  x <- c(1L, 4L, 1L, 3L, 1L)

  y <- c("zy", "zz", "abcd", "gfa", "zy")

  z <- c(1, 2, 5, 4, 3)

  # as second column
  df1 <- data.frame(x = x, y = y)

  # as first column
  df2 <- data.frame(y = y, x = x)

  # as second column with a third after it to break ties
  df3 <- data.frame(x = x, y = y, z = z)

  expect_identical(vec_order_radix(df1), base_order(df1))
  expect_identical(vec_order_radix(df2), base_order(df2))
  expect_identical(vec_order_radix(df3), base_order(df3))
})

test_that("can have multiple character columns in a data frame", {
  df <- data.frame(
    x = c("def", "aba", "aba", "aba", "q"),
    y = c("zy", "zz", "zz", "gfa", "zy"),
    z = c("foo", "qux", "bar", "baz", "boo")
  )

  expect_identical(vec_order_radix(df), base_order(df))
})

test_that("can order with varying encodings by converting to UTF-8", {
  encs <- encodings()
  x <- c(encs$utf8, encs$unknown, encs$latin1, "AC")

  expect_identical(vec_order_radix(x), c(4L, 1L, 2L, 3L))
  expect_identical(vec_order_radix(x, direction = "desc"), c(1L, 2L, 3L, 4L))
})

test_that("can order when in expected order", {
  x <- c("a", "a", "b", NA, NA)
  expect_identical(vec_order_radix(x, direction = "asc", na_value = "largest"), 1:5)

  x <- c("c", "c", "b", NA, NA)
  expect_identical(vec_order_radix(x, direction = "desc", na_value = "largest"), 1:5)

  x <- c(NA, NA, "a", "a", "b")
  expect_identical(vec_order_radix(x, direction = "asc", na_value = "smallest"), 1:5)

  x <- c(NA, NA, "c", "c", "b")
  expect_identical(vec_order_radix(x, direction = "desc", na_value = "smallest"), 1:5)
})

test_that("can order when in strictly opposite of expected order (no ties)", {
  x <- c(NA, "b", "a")
  expect_identical(vec_order_radix(x, direction = "asc", na_value = "largest"), 3:1)

  x <- c(NA, "a", "b")
  expect_identical(vec_order_radix(x, direction = "desc", na_value = "largest"), 3:1)

  x <- c("b", "a", NA)
  expect_identical(vec_order_radix(x, direction = "asc", na_value = "smallest"), 3:1)

  x <- c("a", "b", NA)
  expect_identical(vec_order_radix(x, direction = "desc", na_value = "smallest"), 3:1)
})

# ------------------------------------------------------------------------------
# vec_order_radix(<character>) - radix

# Have to get the number of unique strings above the ORDER_INSERTION_BOUNDARY
# to trigger radix ordering.

test_that("can order character vectors", {
  x <- paste0("x", seq(1L, ORDER_INSERTION_BOUNDARY + 1L))
  expect_identical(vec_order_radix(x), base_order(x))
})

test_that("ordering on ties is done stably", {
  x <- c(paste0("x", seq(1L, ORDER_INSERTION_BOUNDARY + 1L)), "x1")
  expect_identical(vec_order_radix(x)[1:2], c(1L, length(x)))
})

test_that("`NA` order defaults to last", {
  x <- paste0("x", seq(1L, ORDER_INSERTION_BOUNDARY + 1L))
  x <- c(x, NA_character_, "y")
  expect_identical(vec_order_radix(x)[length(x)], length(x) - 1L)
})

test_that("`NA` order can be first", {
  x <- paste0("x", seq(1L, ORDER_INSERTION_BOUNDARY + 1L))
  x <- c(x, NA_character_, "y")
  expect_identical(vec_order_radix(x, na_value = "smallest")[[1L]], length(x) - 1L)
})

test_that("`direction` can be set to `desc`", {
  x <- paste0("x", seq(1L, ORDER_INSERTION_BOUNDARY + 1L))
  expect_identical(vec_order_radix(x, direction = "desc"), base_order(x, decreasing = TRUE))
})

test_that("all combinations of `direction` and `na_value` work", {
  x <- paste0("x", seq(1L, ORDER_INSERTION_BOUNDARY + 1L))
  x <- c(x, NA_character_, "x", "aa", "x1")

  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "asc")],
    x[base_order(x, na.last = TRUE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "asc")],
    x[base_order(x, na.last = FALSE, decreasing = FALSE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "largest", direction = "desc")],
    x[base_order(x, na.last = TRUE, decreasing = TRUE)]
  )
  expect_identical(
    x[vec_order_radix(x, na_value = "smallest", direction = "desc")],
    x[base_order(x, na.last = FALSE, decreasing = TRUE)]
  )
})

# ------------------------------------------------------------------------------
# vec_order_radix(<list>)

test_that("list elements are ordered by first appearance", {
  expect_identical(vec_order_radix(list(1:2, "a", 1:2)), c(1L, 3L, 2L))
})

# ------------------------------------------------------------------------------
# vec_order_radix(<data.frame>) - insertion

test_that("data frame with no columns and no rows returns integer()", {
  x <- data.frame()
  expect_identical(vec_order_radix(x), integer())
})

test_that("data frame with no columns and some rows returns sequential rows", {
  x <- new_data_frame(n = 5L)
  expect_identical(vec_order_radix(x), 1:5)
})

test_that("can order with multiple pre-sorted keys", {
  df <- data.frame(x = 1:2, y = 3:4)
  expect_identical(vec_order_radix(df), 1:2)
})

test_that("first column has ordering presedence", {
  df <- data.frame(x = c(3L, 2L, 1L), y = c(1L, 2L, 3L))
  expect_identical(vec_order_radix(df), 3:1)
})

test_that("secondary columns break ties - integer", {
  df <- data.frame(
    x = c(1L, 2L, 1L),
    y = c(3L, 2L, 1L)
  )
  expect_identical(vec_order_radix(df), c(3L, 1L, 2L))
})

test_that("secondary columns break ties - double", {
  df <- data.frame(
    x = c(1, 2, 1),
    y = c(3, 2, 1)
  )
  expect_identical(vec_order_radix(df), c(3L, 1L, 2L))
})

test_that("secondary columns break ties - logical", {
  df <- data.frame(
    x = c(FALSE, TRUE, FALSE),
    y = c(TRUE, TRUE, FALSE)
  )
  expect_identical(vec_order_radix(df), c(3L, 1L, 2L))
})

test_that("orders correctly when first column is already ordered but second isn't", {
  df <- data.frame(
    x = c(1L, 1L, 2L, 2L),
    y = c(3L, 2L, 4L, 1L)
  )
  expect_identical(vec_order_radix(df), c(2L, 1L, 4L, 3L))
})

test_that("orders correctly when first column is already ordered but second isn't - character", {
  df <- data.frame(
    x = c("a", "a", "b", "b"),
    y = c("c", "b", "d", "a")
  )

  expect_identical(vec_order_radix(df), c(2L, 1L, 4L, 3L))
})

test_that("`direction` is recycled", {
  df <- data.frame(
    x = c(1L, 1L, 2L, 2L),
    y = c(3L, 2L, 4L, 1L)
  )
  expect_identical(vec_order_radix(df, direction = "desc"), c(3L, 4L, 1L, 2L))
})

test_that("`na_value` is recycled", {
  df <- data.frame(
    x = c(1L, 1L, 2L, 2L, NA),
    y = c(3L, 2L, 4L, 1L, NA)
  )
  expect_identical(vec_order_radix(df, na_value = "smallest"), c(5L, 2L, 1L, 4L, 3L))
})

test_that("`direction` can be a vector", {
  df <- data.frame(
    x = c(1L, 1L, 2L, 2L),
    y = c(3L, 2L, 4L, 1L)
  )
  expect_identical(vec_order_radix(df, direction = c("desc", "asc")), c(4L, 3L, 2L, 1L))
})

test_that("`na_value` can be a vector", {
  df <- data.frame(
    x = c(1L, 1L, 2L, 2L, NA, NA),
    y = c(3L, 2L, 4L, 1L, NA, 2)
  )
  expect_identical(vec_order_radix(df, na_value = c("smallest", "largest")), c(6L, 5L, 2L, 1L, 4L, 3L))
})

test_that("`na_value` and `direction` can both be vectors", {
  df <- data.frame(
    x = c(1L, 1L, 2L, 2L, NA, NA),
    y = c(3L, 2L, 4L, 1L, NA, 2)
  )

  expect_identical(
    vec_order_radix(df, direction = c("desc", "asc"), na_value = c("smallest", "largest")),
    6:1
  )
})

# ------------------------------------------------------------------------------
# vec_order_radix(<data.frame>) - counting

test_that("can order 2+ integer column chunks with counting sort", {
  half <- floor(ORDER_INSERTION_BOUNDARY / 2) + 1L
  quarter_low <- floor(half / 2)
  quarter_high <- ceiling(half / 2)

  df <- data.frame(
    x = 1L,
    y = c(rep(2L, quarter_low), rep(1L, quarter_high), rep(3L, half))
  )

  expect_identical(vec_order_radix(df), base_order(df))
})

# ------------------------------------------------------------------------------
# vec_order_radix(<data.frame>) - radix

test_that("can order 2+ integer column chunks with radix sort", {
  half <- floor(ORDER_INSERTION_BOUNDARY / 2) + 1L
  quarter_low <- floor(half / 2)
  quarter_high <- ceiling(half / 2)

  df <- data.frame(
    x = 1L,
    y = c(rep(2L, quarter_low), rep(1L, quarter_high), rep(3L, half), INT_ORDER_COUNTING_RANGE_BOUNDARY + 1L)
  )

  expect_identical(vec_order_radix(df), base_order(df))
})

test_that("can order 2+ double column chunks with radix sort", {
  half <- floor(ORDER_INSERTION_BOUNDARY / 2) + 1L
  quarter_low <- floor(half / 2)
  quarter_high <- ceiling(half / 2)

  df <- data.frame(
    x = 1,
    y = c(rep(2, quarter_low), rep(1, quarter_high), rep(3, half), INT_ORDER_COUNTING_RANGE_BOUNDARY + 1)
  )

  expect_identical(vec_order_radix(df), base_order(df))
})

# ------------------------------------------------------------------------------
# vec_order_radix() - error checking

test_that("`na_value` is checked", {
  expect_error(vec_order_radix(1L, na_value = "x"), "\"largest\" or \"smallest\"")
  expect_error(vec_order_radix(1L, na_value = c(TRUE, TRUE)), "must be a character vector")
  expect_error(vec_order_radix(1L, na_value = NA_character_), "can't be missing")
})

test_that("`direction` is checked", {
  expect_error(vec_order_radix(1L, direction = "x"), "must contain only")
  expect_error(vec_order_radix(1L, direction = c("asc", "asc")), "single value")
  expect_error(vec_order_radix(1L, direction = NA_character_), "can't be missing")
  expect_error(vec_order_radix(data.frame(x = 1), direction = c("asc", "asc")), "length 1 or")
})

test_that("`x` is checked", {
  expect_error(vec_order_radix(foobar()), class = "vctrs_error_scalar_type")
})

# ------------------------------------------------------------------------------
# vec_order_radix() - groups

test_that("groups can be reallocated if we exceed the max group data size", {
  set.seed(123)

  # The first column has all unique groups so 1 more than the default group
  # data size is needed and will be reallocated on the fly
  df <- data.frame(
    x = sample(GROUP_DATA_SIZE_DEFAULT + 1L, replace = TRUE),
    y = sample(GROUP_DATA_SIZE_DEFAULT + 1L, replace = TRUE),
    z = sample(GROUP_DATA_SIZE_DEFAULT + 1L, replace = TRUE)
  )

  expect_identical(vec_order_radix(df), base_order(df))
})

# ------------------------------------------------------------------------------
# vec_order_radix() - comparison proxy

test_that("ordering works with rcrd types", {
  x <- tuple(c(1, 2, 1), c(3, 2, 1))
  expect_identical(vec_order_radix(x), c(3L, 1L, 2L))
})

test_that("data frame comparison proxies don't allow vector `direction` or `na_value`", {
  x <- tuple(c(1, 2, 1), c(3, 2, 1))
  expect_error(vec_order_radix(x, direction = c("desc", "asc")), "single value")
  expect_error(vec_order_radix(x, na_value = c("largest", "smallest")), "single value")
})

test_that("ordering works with df-cols", {
  df_col <- new_data_frame(list(y = c(2, 1, 2), z = c(3, 3, 3)))
  df <- new_data_frame(list(x = c(1, 1, 1), y = df_col))

  expect_identical(vec_order_radix(df), c(2L, 1L, 3L))

  # Can only supply a max of 2 `direction` or `na_value` values which get internally
  # expanded to 3 to match the flattened df proxy
  expect_identical(vec_order_radix(df, direction = c("asc", "desc")), c(1L, 3L, 2L))

  expect_error(vec_order_radix(df, direction = c("desc", "desc", "asc")), "or length equal to")
})

test_that("ordering works with df-cols with 0 cols", {
  df_col <- new_data_frame(list(), n = 3L)
  df <- new_data_frame(list(x = c(1, 3, 1), y = df_col, z = c(2, 1, 1)))

  expect_identical(vec_order_radix(df), c(3L, 1L, 2L))

  # Can supply 3 `direction` values even though the 0-col df-col gets dropped
  expect_identical(vec_order_radix(df, direction = c("asc", "desc", "desc")), c(1L, 3L, 2L))

  expect_error(vec_order_radix(df, direction = c("desc", "asc")), "or length equal to")
})

test_that("ordering works with rcrd cols", {
  y <- tuple(c(1, 2, 1), c(3, 2, 1))
  df <- new_data_frame(list(z = c(1, 1, 1), y = y))

  expect_identical(vec_order_radix(df), c(3L, 1L, 2L))

  # Can only supply a max of 2 `direction` values which get internally
  # expanded to 3 to match the flattened df proxy
  expect_identical(vec_order_radix(df, direction = c("asc", "desc")), c(2L, 1L, 3L))

  expect_error(vec_order_radix(df, direction = c("desc", "desc", "asc")), "or length equal to")
})

# ------------------------------------------------------------------------------
# vec_order_locs()

test_that("`vec_order_locs()` is working", {
  x <- c(1, 3, 1, 5, 2, 5, 1)

  expect <- new_data_frame(
    list(
      key = c(1, 2, 3, 5),
      loc = list(c(1L, 3L, 7L), 5L, 2L, c(4L, 6L))
    )
  )

  expect_identical(vec_order_locs(x), expect)
})

# ------------------------------------------------------------------------------
# `vec_order_radix()` - Pre-existing tests

test_that("can request NAs sorted first", {
  expect_equal(vec_order_radix(c(1, NA), "asc", "largest"), 1:2)
  expect_equal(vec_order_radix(c(1, NA), "desc", "largest"), 1:2)

  expect_equal(vec_order_radix(c(1, NA), "asc", "smallest"), 2:1)
  expect_equal(vec_order_radix(c(1, NA), "desc", "smallest"), 2:1)
})

test_that("can sort data frames", {
  df <- data.frame(x = c(1, 2, 1), y = c(1, 2, 2))

  out1 <- vec_sort(df)
  expect_equal(out1, data.frame(x = c(1, 1, 2), y = c(1, 2, 2)))

  out2 <- vec_sort(df, "desc")
  expect_equal(out2, data.frame(x = c(2, 1, 1), y = c(2, 2, 1)))
})

test_that("can sort empty data frames (#356)", {
  df1 <- data.frame()
  expect_equal(vec_sort(df1), df1)

  df2 <- data.frame(x = numeric(), y = integer())
  expect_equal(vec_sort(df2), df2)
})

test_that("can order tibbles that contain non-comparable objects", {
  expect_equal(vec_order_radix(data_frame(x = list(10, 2, 1))), 1:3)
})

test_that("can order matrices and arrays (#306)", {
  x <- matrix(c(1, 1, 1, 1, 2, 1), ncol = 2)
  expect_identical(vec_order_radix(x), c(1L, 3L, 2L))

  x <- array(1:8, c(2, 2, 2))
  x[2] <- 1
  x[3] <- 5
  expect_identical(vec_order_radix(x), 2:1)
})

test_that("can order empty data frames (#356)", {
  df1 <- data.frame()
  expect_equal(vec_order_radix(df1), integer())

  df2 <- data.frame(x = numeric(), y = integer())
  expect_equal(vec_order_radix(df2), integer())
})

test_that("can order data frames with data frame columns (#527)", {
  expect_equal(
    vec_order_radix(iris),
    vec_order_radix(data_frame(iris = iris))
  )
})

test_that("can order data frames (and subclasses) with matrix columns", {
  df <- new_data_frame(n = 2L)

  df$x <- new_data_frame(list(y = matrix(1:2, 2)))
  expect_identical(vec_order_radix(df), 1:2)

  df$x <- tibble::tibble(y = matrix(1:2, 2))
  expect_identical(vec_order_radix(df), 1:2)
})
