% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vimp_regression.R
\name{vimp_regression}
\alias{vimp_regression}
\title{Nonparametric Variable Importance Estimates}
\usage{
vimp_regression(Y, X, f1 = NULL, f2 = NULL, indx = 1,
  run_regression = TRUE, SL.library = c("SL.glmnet", "SL.xgboost",
  "SL.mean"), alpha = 0.05, na.rm = FALSE, ...)
}
\arguments{
\item{Y}{the outcome}

\item{X}{the covariates}

\item{f1}{the fitted values from a flexible estimation technique regressing Y on X.}

\item{f2}{the fitted values from a flexible estimation technique regressing the fitted values in \code{f1} on X withholding the columns in \code{indx}.}

\item{indx}{the indices of the covariate(s) to calculate variable importance for; defaults to 1.}

\item{run_regression}{if outcome Y and covariates X are passed to \code{vimp_regression}, and \code{run_regression} is \code{TRUE}, then Super Learner will be used; otherwise, variable importance will be computed using the inputted fitted values.}

\item{SL.library}{a character vector of learners to pass to \code{SuperLearner}, if \code{f1} and \code{f2} are Y and X, respectively. Defaults to \code{SL.glmnet}, \code{SL.xgboost}, and \code{SL.mean}.}

\item{alpha}{the level to compute the confidence interval at. Defaults to 0.05, corresponding to a 95\% confidence interval.}

\item{na.rm}{should we remove NA's in the outcome and fitted values in computation? (defaults to \code{FALSE})}

\item{...}{other arguments to the estimation tool, see "See also".}
}
\value{
An object of classes \code{vim} and \code{vim_regression}. See Details for more information.
}
\description{
Compute estimates of and confidence intervals for nonparametric ANOVA-based variable importance.
}
\details{
See the paper by Williamson, Gilbert, Simon, and Carone for more
details on the mathematics behind this function, and the validity
of the confidence intervals.
In the interest of transparency, we return most of the calculations
within the \code{vim} object. This results in a list containing:
\itemize{
 \item{call}{ - the call to \code{vim}}
 \item{s}{ - the column(s) to calculate variable importance for}
 \item{SL.library}{ - the library of learners passed to \code{SuperLearner}}
 \item{full_fit}{ - the fitted values of the chosen method fit to the full data}
 \item{red_fit}{ - the fitted values of the chosen method fit to the reduced data}
 \item{est}{ - the estimated variable importance}
 \item{naive}{ - the naive estimator of variable importance}
 \item{update}{ - the influence curve-based update}
 \item{se}{ - the standard error for the estimated variable importance}
 \item{ci}{ - the \eqn{(1-\alpha) \times 100}\% confidence interval for the variable importance estimate}
 \item{full_mod}{ - the object returned by the estimation procedure for the full data regression (if applicable)}
 \item{red_mod}{ - the object returned by the estimation procedure for the reduced data regression (if applicable)}
 \item{alpha}{ - the level, for confidence interval calculation}
}
}
\examples{
library(SuperLearner)
library(gam)
## generate the data
## generate X
p <- 2
n <- 100
x <- data.frame(replicate(p, stats::runif(n, -5, 5)))

## apply the function to the x's
smooth <- (x[,1]/5)^2*(x[,1]+7)/5 + (x[,2]/3)^2

## generate Y ~ Normal (smooth, 1)
y <- smooth + stats::rnorm(n, 0, 1)

## set up a library for SuperLearner
learners <- "SL.gam"

## using Y and X
est <- vimp_regression(y, x, indx = 2, 
           alpha = 0.05, run_regression = TRUE, 
           SL.library = learners, cvControl = list(V = 10))

## using pre-computed fitted values
full <- SuperLearner(Y = y, X = x,
SL.library = learners, cvControl = list(V = 10))
full.fit <- predict(full)$pred
reduced <- SuperLearner(Y = full.fit, X = x[, 2, drop = FALSE],
SL.library = learners, cvControl = list(V = 10))
red.fit <- predict(reduced)$pred

est <- vimp_regression(Y = y, f1 = full.fit, f2 = red.fit, 
            indx = 2, run_regression = FALSE, alpha = 0.05)

}
\seealso{
\code{\link[SuperLearner]{SuperLearner}} for specific usage of the \code{SuperLearner} function and package.
}
