% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vmideal.R
\name{vmideal}
\alias{vmideal}
\alias{dvmideal}
\alias{pvmideal}
\alias{qvmideal}
\alias{rvmideal}
\alias{cvmideal}
\title{Ideal Distribution of Visual Meteor Magnitudes}
\usage{
dvmideal(m, lm, psi, log = FALSE, perception.fun = NULL)

pvmideal(m, lm, psi, lower.tail = TRUE, log = FALSE, perception.fun = NULL)

qvmideal(p, lm, psi, lower.tail = TRUE, perception.fun = NULL)

rvmideal(n, lm, psi, perception.fun = NULL)

cvmideal(lm, psi, log = FALSE, perception.fun = NULL)
}
\arguments{
\item{m}{integer; visual meteor magnitude.}

\item{lm}{numeric; limiting magnitude.}

\item{psi}{numeric; the location parameter of the probability distribution.}

\item{log}{logical; if \code{TRUE}, probabilities are returned as \code{log(p)}.}

\item{perception.fun}{function; optional perception probability function.
The default is \link{vmperception}.}

\item{lower.tail}{logical; if \code{TRUE} (default), probabilities are
\eqn{P[M < m]}; otherwise, \eqn{P[M \ge m]}.}

\item{p}{numeric; probability.}

\item{n}{numeric; count of meteor magnitudes.}
}
\value{
\itemize{
\item \code{dvmideal}: density
\item \code{pvmideal}: distribution function
\item \code{qvmideal}: quantile function
\item \code{rvmideal}: random generation
\item \code{cvmideal}: partial convolution of the ideal distribution of meteor magnitudes
with the perception probabilities.
}

The length of the result is determined by \code{n} for \code{rvmideal}, and is the maximum
of the lengths of the numeric vector arguments for the other functions.
All arguments are vectorized; standard R recycling rules apply.

Since the distribution is discrete, \code{qvmideal} and \code{rvmideal} always return integer values.
\code{qvmideal} may return \code{NaN} with a warning.
}
\description{
Density, distribution function, quantile function, and random generation
for the ideal distribution of visual meteor magnitudes.
}
\details{
The density of the \link[=mideal]{ideal distribution of meteor magnitudes} is
\deqn{
    {\displaystyle f(m) = \frac{\mathrm{d}p}{\mathrm{d}m} = \frac{3}{2} \, \log(r) \sqrt{\frac{r^{3 \, \psi + 2 \, m}}{(r^\psi + r^m)^5}}}
}
where \eqn{m} is the meteor magnitude, \eqn{r = 10^{0.4} \approx 2.51189 \dots} is a constant, and
\eqn{\psi} is the only parameter of this magnitude distribution.

In visual meteor observations, magnitudes are usually estimated as integer values.
Hence, this distribution is discrete and its probability mass function is given by
\deqn{
    P[M = m] \sim g(m) \, \int_{m-0.5}^{m+0.5} f(m) \, \mathrm{d}m \, ,
}
where \eqn{g(m)} denotes the perception probability.
Thus, the distribution is the product of the
\link[=vmperception]{perception probabilities} and the
underlying \link[=mideal]{ideal distribution} of meteor magnitudes.

If a perception probability function \code{perception.fun} is supplied,
it must have the signature \verb{function(x)} and return the perception probabilities of
the difference \code{x} between the limiting magnitude and the meteor magnitude.
If \code{x >= 15.0}, the \code{perception.fun} function should return a perception probability of \code{1.0}.
The argument \code{perception.fun} is resolved using \link{match.fun}.
}
\examples{
N <- 100
psi <- 5.0
limmag <- 6.5
(m <- seq(6, -4))

# discrete density of `N` meteor magnitudes
(freq <- round(N * dvmideal(m, limmag, psi)))

# log likelihood function
lld <- function(psi) {
    -sum(freq * dvmideal(m, limmag, psi, log=TRUE))
}

# maximum likelihood estimation (MLE) of psi
est <- optim(2, lld, method='Brent', lower=0, upper=8, hessian=TRUE)

# estimations
est$par # mean of psi

# generate random meteor magnitudes
m <- rvmideal(N, limmag, psi)

# log likelihood function
llr <- function(psi) {
    -sum(dvmideal(m, limmag, psi, log=TRUE))
}

# maximum likelihood estimation (MLE) of psi
est <- optim(2, llr, method='Brent', lower=0, upper=8, hessian=TRUE)

# estimations
est$par # mean of psi
sqrt(1/est$hessian[1][1]) # standard deviation of psi

m <- seq(6, -4, -1)
p <- vismeteor::dvmideal(m, limmag, psi)
barplot(
    p,
    names.arg = m,
    main = paste0('Density (psi = ', psi, ', limmag = ', limmag, ')'),
    col = "blue",
    xlab = 'm',
    ylab = 'p',
    border = "blue",
    space = 0.5
)
axis(side = 2, at = pretty(p))

plot(
    function(lm) vismeteor::cvmideal(lm, psi, log = TRUE),
    -5, 10,
    main = paste0(
        'Partial convolution of the ideal meteor magnitude distribution\n',
        'with the perception probabilities (psi = ', psi, ')'
    ),
    col = "blue",
    xlab = 'lm',
    ylab = 'log(rate)'
)
}
\references{
Richter, J. (2018) \emph{About the mass and magnitude distributions of meteor showers}.
WGN, Journal of the International Meteor Organization, vol. 46, no. 1, p. 34-38
}
\seealso{
\link{mideal}
\link{vmperception}
}
