#' Plot the prepared data into Highcharts.js plot
#'
#' @param data the data frame to be plotted (ranges + events), e.g. generated by `visime_data`
#' @param title the title for the plot
#' @param show_labels boolean, show labels on events or not
#' @importFrom highcharter hcaes
#' @importFrom highcharter hc_add_series
#' @importFrom highcharter hc_yAxis
#' @importFrom highcharter hc_xAxis
#' @importFrom highcharter hc_title
#' @importFrom highcharter hc_tooltip
#' @importFrom highcharter hc_legend
#' @importFrom highcharter JS
#' @importFrom highcharter hc_chart
#' @importFrom highcharter highchart
#'
#' @return a plot object generated by `highchart`
#' @keywords internal
#' @noRd
#' @examples
#' \dontrun{
#' plot_highchart(data.frame(
#'     event = 1:2, start = as.POSIXct(c("2019-01-01", "2019-01-10")),
#'     end = as.POSIXct(c("2019-01-10", "2019-01-25")),
#'     group = "", tooltip = "", col = "green", fontcol = "black",
#'     subplot = 1, y = 1:2, label = 1:2
#'   ), title = "A title", show_labels = TRUE
#' )
#' }
plot_highchart <- function(data, title, show_labels){

  # let an event be 1/50th of total timeline range
  data$end <- with(data, ifelse(start != end, end, end + diff(range(c(start, end)))/50))

  data$low <- 1000 * as.double(data$start)
  data$high <- 1000 * as.double(data$end)
  data$x <- max(data$y) - data$y + 1
  data$color = data$col

  cats <- round(tapply(data$y, data$group, mean))
  y_vals <- names(sort(c(cats, setdiff(seq_len(max(data$y)), cats))))

  hc_title(
    hc_legend(
      hc_tooltip(
        hc_xAxis(
          hc_yAxis(
            hc_add_series(
              hc_chart(highchart(), inverted =TRUE),
              data, "columnrange",
              dataLabels = list(enabled = show_labels, inside=T,
                                formatter = JS("function () {return (this.y === this.point.low ? this.point.event : \"\")}"))),
            type = "datetime"),
          categories = c("", y_vals)),
        crosshairs = TRUE, formatter = JS("function () {return this.point.tooltip}")) ,
      enabled=F),
    text = title)


}
