#' @title Create an igraph graph from network
#' 
#' @param net A named list of dataframes \code{nodes} and \code{edges} generated by \code{Create}.
#' @param directed Logical. Create a directed graph. Default is \code{TRUE}.
#' @param writeToFile Logical. Save graph to a file in the current working directory. Default is \code{FALSE}.
#' @param ... Additional parameters passed to function. Not used in this method.
#' 
#' @return An igraph object.
#' 
#' @aliases Graph
#' @name vosonSML::Graph
#' @export
Graph <- function(net, directed = TRUE, writeToFile = FALSE, ...) {
  cat("Creating igraph network graph...")
  if (writeToFile) { cat("\n") }
  
  # create igraph object from dataframes
  g <- igraph::graph_from_data_frame(d = net$edges, directed = directed, vertices = net$nodes)
  
  # searches the class list of net for matching method
  UseMethod("Graph", net)
}

#' @noRd
#' @export
Graph.default <- function(...) {
  stop("Unknown network type passed to graph.", call. = FALSE) 
}

#' @noRd
#' @method Graph activity
#' @export
Graph.activity <- function(net, directed = TRUE, writeToFile = FALSE, ...) {
  UseMethod("Graph.activity", net)
}

#' @noRd
#' @export
Graph.activity.default <- function(...) {
  stop("Unknown social media type passed to graph.", call. = FALSE)
}

#' @noRd
#' @export
Graph.activity.twitter <- function(net, directed = TRUE, writeToFile = FALSE, ...) {
  igraph::V(g)$label <- ifelse(!is.na(igraph::V(g)$screen_name), 
                               paste0(igraph::V(g)$name, " (", igraph::V(g)$screen_name, ")"), 
                               igraph::V(g)$name)
  g <- set_graph_attr(g, "type", "twitter")
  
  graphOutputFile(g, "graphml", writeToFile, "TwitterActivity")
  cat("Done.\n")
  
  g
}

#' @noRd
#' @export
Graph.activity.youtube <- function(net, directed = TRUE, writeToFile = FALSE, ...) {
  igraph::V(g)$label <- ifelse(!is.na(igraph::V(g)$screen_name), 
                               paste0(igraph::V(g)$name, " (", igraph::V(g)$screen_name, ")"), 
                               igraph::V(g)$name)
  g <- set_graph_attr(g, "type", "youtube")
  
  graphOutputFile(g, "graphml", writeToFile, "YoutubeActivity")
  cat("Done.\n")
  
  g
}

#' @noRd
#' @export
Graph.activity.reddit <- function(net, directed = TRUE, writeToFile = FALSE, ...) {
  V(g)$label <- ifelse(!is.na(V(g)$user), paste0(V(g)$name, " (", V(g)$user, ")"), V(g)$name)
  g <- set_graph_attr(g, "type", "reddit")

  graphOutputFile(g, "graphml", writeToFile, "RedditActivity")
  cat("Done.\n")
  
  g
}

#' @noRd
#' @method Graph actor
#' @export
Graph.actor <- function(net, directed = TRUE, writeToFile = FALSE, ...) {
  UseMethod("Graph.actor", net)
}

#' @noRd
#' @export
Graph.actor.default <- function(...) {
  stop("Unknown social media type passed to graph.", call. = FALSE)
}

#' @noRd
#' @export
Graph.actor.twitter <- function(net, directed = TRUE, writeToFile = FALSE, ...) {
  V(g)$label <- ifelse(is.na(V(g)$screen_name), paste0("ID:", V(g)$name), 
                             paste0(V(g)$screen_name, " (", V(g)$name, ")"))
  g <- set_graph_attr(g, "type", "twitter")
  
  graphOutputFile(g, "graphml", writeToFile, "TwitterActor")
  cat("Done.\n")
  
  g  
}

#' @noRd
#' @export
Graph.actor.youtube <- function(net, directed = TRUE, writeToFile = FALSE, ...) {
  V(g)$label <- ifelse(is.na(V(g)$screen_name), V(g)$name, 
                             paste0(V(g)$screen_name, " (", V(g)$name, ")"))
  g <- set_graph_attr(g, "type", "youtube")
  
  graphOutputFile(g, "graphml", writeToFile, "YoutubeActor")
  cat("Done.\n")
  
  g
}

#' @noRd
#' @export
Graph.actor.reddit <- function(net, directed = TRUE, writeToFile = FALSE, ...) {
  V(g)$label <- ifelse(is.na(V(g)$user), paste0("ID:", V(g)$name), 
                             paste0(V(g)$user, " (", V(g)$name, ")"))

  g <- set_graph_attr(g, "type", "reddit")
  
  graphOutputFile(g, "graphml", writeToFile, "RedditActor")
  cat("Done.\n")
  
  g  
}

#' @noRd
#' @method Graph semantic
#' @export
Graph.semantic <- function(net, directed = TRUE, writeToFile = FALSE, ...) {
  UseMethod("Graph.semantic", net)
}

#' @noRd
#' @export
Graph.semantic.default <- function(...) {
  stop("Unknown social media type passed to graph.", call. = FALSE)
}

#' @noRd
#' @export
Graph.semantic.twitter <- function(net, directed = TRUE, writeToFile = FALSE, ...) {
  # we need to simplify the graph because multiple use of same term in one tweet will cause self-loops, etc
  # g <- simplify(g)  
  V(g)$label <- V(g)$name
  g <- set_graph_attr(g, "type", "twitter")
  
  graphOutputFile(g, "graphml", writeToFile, "TwitterSemantic")
  cat("Done.\n")
  
  g  
}

#' @noRd
#' @method Graph twomode
#' @export
Graph.twomode <- function(net, directed = TRUE, writeToFile = FALSE, ...) {
  UseMethod("Graph.twomode", net)
}

#' @noRd
#' @export
Graph.twomode.default <- function(...) {
  stop("Unknown social media type passed to graph.", call. = FALSE)
}

#' @noRd
#' @export
Graph.twomode.twitter <- function(net, directed = TRUE, writeToFile = FALSE, ...) {
  V(g)$label <- ifelse(V(g)$name == V(g)$display_name, V(g)$display_name,
                       paste0("@", V(g)$display_name, " (", V(g)$name, ")"))
  
  g <- set_graph_attr(g, "type", "twitter")
  
  graphOutputFile(g, "graphml", writeToFile, "TwitterTwomode")
  cat("Done.\n")
  
  g  
}

# set output file name
# if wtof is logical use def as file name
# if character use wtof as file name
graphOutputFile <- function(g, type, wtof, def) {
  if (is.logical(wtof) && wtof) { 
    writeOutputFile(g, "graphml", def) 
  } else if (is.character(wtof)) {
    writeOutputFile(g, "graphml", wtof)
  }  
}
