#' @title Assessing the performance of acoustic distance measurements
#' 
#' @description \code{compare.methods} creates graphs to visually assess performance of acoustic distance measurements 
#' @usage compare.methods(X = NULL, flim = c(0, 22), bp = c(0, 22), mar = 0.1, wl = 512, ovlp = 90, 
#' res = 150, n = 10, length.out = 30, methods = c("XCORR", "dfDTW", "ffDTW", "SP"), 
#' it = "jpeg", parallel = 1, path = NULL, sp = NULL, pb = TRUE, grid = TRUE, 
#' clip.edges = TRUE, threshold = 15, na.rm = FALSE, scale = FALSE,
#'  pal = reverse.gray.colors.2, img = TRUE, ...)
#' @param X 'selection.table' object or data frame with results from \code{\link{manualoc}} function, \code{\link{autodetec}} 
#' function, or any data frame with columns for sound file name (sound.files), 
#' selection number (selec), and start and end time of signal (start and end).
#' Default \code{NULL}. 
#' @param flim A numeric vector of length 2 for the frequency limit in kHz of 
#'   the spectrogram, as in \code{\link[seewave]{spectro}}. Default is c(0, 22).
#' @param bp numeric vector of length 2 giving the lower and upper limits of the 
#' frequency bandpass filter (in kHz) used in the acoustic distance methods. Default is c(0, 22). Note that
#' for XCORR this argument sets the frange argument from the \code{\link{xcorr}} function.  
#' @param mar Numeric vector of length 1. Specifies plot margins around selection in seconds. Default is 0.1.
#' @param wl A numeric vector of length 1 specifying the window length of the spectrogram and cross-correlation, default 
#'   is 512.
#' @param ovlp Numeric vector of length 1 specifying the percent overlap between two 
#'   consecutive windows, as in \code{\link[seewave]{spectro}}. Default is 90.
#' @param res Numeric argument of length 1. Controls image resolution.
#'   Default is 150.
#' @param n Numeric argument of length 1. Defines the number of plots to be produce. 
#' Default is 10.
#' @param length.out A character vector of length 1 giving the number of measurements of fundamental or dominant
#' frequency desired (the length of the time series). Default is 30.
#' @param methods A character vector of length 2 giving the names of the acoustic distance
#' methods that would be compared. The methods available are: cross-correlation (XCORR, from
#' \code{\link{xcorr}}), dynamic time warping on dominant frequency time series (dfDTW, from
#'  \code{\link[dtw]{dtw}} applied on \code{\link{dfts}} output), dynamic time warping on dominant 
#'  frequency time series (ffDTW, from \code{\link[dtw]{dtw}} applied on \code{\link{ffts}} output),
#'   spectral parameters (SP, from \code{\link{specan}}).
#' @param it A character vector of length 1 giving the image type to be used. Currently only
#' "tiff" and "jpeg" are admitted. Default is "jpeg".
#' @param parallel Numeric. Controls whether parallel computing is applied.
#'  It specifies the number of cores to be used. Default is 1 (i.e. no parallel computing). 
#'  Not available in Windows OS.
#' @param path Character string containing the directory path where the sound files are located. 
#' If \code{NULL} (default) then the current working directory is used. 
#' @param sp Data frame with acoustic parameters as the one generated by \code{\link{specan}}. Must contain 'sound.files'
#' and "selec' columns and the same selections as in 'X'.
#' @param pb Logical argument to control progress bar. Default is \code{TRUE}. Note that progress bar is only used
#' when parallel = 1.
#' @param grid Logical argument to control the presence of a grid on the spectrograms (default is \code{TRUE}).
#' @param clip.edges Logical argument to control whether edges (start or end of signal) in
#' which amplitude values above the threshold were not detected will be removed when using dfDTW and 
#' ffDTW methods. If \code{TRUE} this edges will be excluded and signal contour will be calculated on the
#' remainging values. Default is \code{TRUE}. 
#' @param threshold amplitude threshold (\%) for dominant and/or fundamental frequency detection when using dfDTW, ffDTW 
#' and SP methods. Default is 15.
#' @param na.rm Logical. If \code{TRUE} all NAs produced when pairwise cross-correlations failed are removed from the 
#' results. This means that all selections with at least 1 cross-correlation that failed are excluded in both methods under
#' comparison. Only apply if XCORR is one of the methods being compared.
#' @param scale Logical. If \code{TRUE} dominant and/or fundamental frequency values are z-transformed using the \code{\link[base]{scale}} function, which "ignores" differences in absolute frequencies between the signals in order to focus the 
#' comparison in the frequency contour, regardless of the pitch of signals. Default is \code{TRUE}.
#' @param pal A color palette function to be used to assign colors in the 
#'   spectrograms, as in \code{\link[seewave]{spectro}}. Default is reverse.gray.colors.2. 
#' @param img A logical argument specifying whether an image files would be produced. Default is \code{TRUE}.
#' @param ... Additional arguments to be passed to a modified version of \code{\link[seewave]{spectro}} for customizing
#' graphical output. This includes fast.spec, an argument that speeds up the plotting of spectrograms (see description in 
#' \code{\link{specreator}}).
#' @return Image files with 4 spectrograms of the selection being compared and scatterplots 
#' of the acoustic space of all signals in the input data frame 'X'.
#' @export
#' @name compare.methods
#' @details This function produces graphs with spectrograms from 4 signals in the 
#' provided data frame that allow visual inspection of the performance of acoustic
#' distance methods at comparing those signals. The signals are randomly picked up
#' from the provided data frame (X argument).The spectrograms are all plotted with
#' the same frequency and time scales. The function compares 2 methods at a time. The
#' methods available are: cross-correlation 
#' (XCORR, from \code{\link{xcorr}}), dynamic time warping on dominant frequency time 
#' series (dfDTW, from \code{\link[dtw]{dtw}} applied on \code{\link{dfts}} output), dynamic time 
#' warping on dominant frequency time series (ffDTW, from \code{\link[dtw]{dtw}} applied on 
#' \code{\link{ffts}} output), spectral parameters (SP, from \code{\link{specan}}). The graph also 
#' contains 2 scatterplots (1 for each method) of the acoustic space of all signals in the 
#' input data frame 'X'. The compared selections are randomly picked up from the pool of 
#' selections in the input data frame. The argument 'n' defines the number of comparisons (i.e. graphs) 
#' to be produced. The acoustic pairwise distance between signals is shown next 
#' to the arrows linking them. The font color of a distance value correspond to the font 
#' color of the method that generated it, as shown in the scatterplots. Distances are 
#' standardized, being 0 the distance of a signal to itself and 1 the farthest pairwise 
#' distance in the pool of signals. Principal Component Analysis (\code{\link[stats]{princomp}}) 
#' is applied to calculate distances when using spectral parameters (SP). In that case the first 2 PC's are used. Classical 
#' Multidimensional Scalling (also known as Principal Coordinates Analysis, 
#' (\code{\link[stats]{cmdscale}})) is used for all other methods. Note that SP can only be used with at least 22 selections (number of rows in input data frame) as PCA only works with more units than variables. The graphs are return as image files in the 
#' working directory. The file name contains the methods being compared and the 
#' rownumber of the selections. This function uses internally a modified version
#' of the \code{\link[seewave]{spectro}} function from seewave package to create spectrograms.
#' @seealso \url{https://marce10.github.io/2017/02/17/Choosing_the_right_method_for_measuring_acoustic_signal_structure.html}
#' @examples{
#' # Set temporary working directory
#' setwd(tempdir())
#' 
#' data(list = c("Phae.long1", "Phae.long2", "Phae.long3", "Phae.long4", "selec.table"))
#' writeWave(Phae.long1,"Phae.long1.wav")
#' writeWave(Phae.long2,"Phae.long2.wav")
#' writeWave(Phae.long3,"Phae.long3.wav")
#' writeWave(Phae.long4,"Phae.long4.wav")
#' 
#' compare.methods(X = selec.table, flim = c(0, 10), bp = c(0, 10), mar = 0.1, wl = 300,
#' ovlp = 90, res = 200, n = 10, length.out = 30,
#' methods = c("XCORR", "dfDTW"), parallel = 1, it = "jpeg")
#' 
#' #remove progress bar
#' compare.methods(X = selec.table, flim = c(0, 10), bp = c(0, 10), mar = 0.1, wl = 300,
#' ovlp = 90, res = 200, n = 10, length.out = 30,
#' methods = c("XCORR", "dfDTW"), parallel = 1, it = "jpeg", pb = FALSE)
#'
#' #check this folder!
#' getwd()
#' 
#' 
#' #compare SP and XCORR
#' #first we need to create a larger data set as the PCA that summarizes the spectral parameters
#' #needs more units (rows) that variables (columns)
#' #so I just create a new selection table repeating 3 times selec.table
#' st2 <- rbind(selec.table, selec.table, selec.table)
#' 
#' #note that the selection labels should be also changed
#' st2$selec <- 1:nrow(st2)
#' #now we can compare SP method against XCORR
#' compare.methods(X = st2, flim = c(0, 10), bp = c(0, 10), mar = 0.1, wl = 300,
#' ovlp = 90, res = 200, n = 10, length.out = 30,
#' methods = c("XCORR", "SP"), parallel = 1, it = "jpeg")
#' 
#' #compare SP method against dfDTW
#' compare.methods(X = st2, flim = c(0, 10), bp = c(0, 10), mar = 0.1, wl = 300,
#' ovlp = 90, res = 200, n = 10, length.out = 30,
#' methods = c("dfDTW", "SP"), parallel = 1, it = "jpeg")
#' 
#' #alternatively we can provide our own SP matrix
#' sp <- specan(selec.table, bp = c(0, 10))
#' 
#' #and selec just a few variables to avoid the problem of # observations vs # parameters in PCA
#' sp <- sp[, 1:7]
#' 
#' compare.methods(X = selec.table, flim = c(0, 10), sp = sp, bp = c(0, 10), mar = 0.1, wl = 300,
#' ovlp = 90, res = 200, n = 10, length.out = 30,
#' methods = c("XCORR", "SP"), parallel = 1, it = "jpeg")
#' 
#' #note that "SP" should also be included as a method in 'methods'
#' #again, all images are saved in the working directory
#' }
#' 
#' @author Marcelo Araya-Salas (\email{araya-salas@@cornell.edu}). It uses 
#' internally a modified version of the \code{\link[seewave]{spectro}} function from 
#' seewave package to create spectrograms.
#last modification on feb-17-2017 (MAS)

compare.methods <- function(X = NULL, flim = c(0, 22), bp = c(0, 22), mar = 0.1, wl = 512, ovlp = 90, 
    res = 150, n = 10, length.out = 30, methods = c("XCORR","dfDTW", "ffDTW", "SP"),
    it = "jpeg", parallel = 1, path = NULL, sp = NULL, pb = TRUE, grid = TRUE, 
    clip.edges = TRUE, threshold = 15, na.rm = FALSE, scale = FALSE, 
    pal = reverse.gray.colors.2, img = TRUE, ...){  
 
  # reset working directory 
  wd <- getwd()
  on.exit(setwd(wd))
  
  #check path to working directory
  if(is.null(path)) path <- getwd() else {if(!file.exists(path)) stop("'path' provided does not exist") else
    setwd(path)
  }  
  
  #if X is not a data frame
  if(!class(X) %in% c("data.frame", "selection.table")) stop("X is not of a class 'data.frame' or 'selection table")

  
  #check basic columns in X
  if(!all(c("sound.files", "selec", 
            "start", "end") %in% colnames(X))) 
    stop(paste(paste(c("sound.files", "selec", "start", "end")[!(c("sound.files", "selec", 
                                                                   "start", "end") %in% colnames(X))], collapse=", "), "column(s) not found in data frame"))
  
   #if parallel is not numeric
  if(!is.numeric(parallel)) stop("'parallel' must be a numeric vector of length 1") 
  if(any(!(parallel %% 1 == 0),parallel < 1)) stop("'parallel' should be a positive integer")
  
  #if parallel and pb in windows
  if(parallel > 1 &  pb & Sys.info()[1] == "Windows") {
    message("parallel with progress bar is currently not available for windows OS")
    message("running parallel without progress bar")
    pb <- FALSE
  } 
  
  # methods 
  if(any(!is.character(methods),length(methods) > 2)) stop("'methods' must be a character vector of length 2")
  if(length(methods[!methods %in% c("XCORR", "dfDTW", "ffDTW", "SP")]) > 0) 
    stop(paste(methods[!methods %in% c("XCORR", "dfDTW", "ffDTW", "SP")],"is (are) not valid method"))
  
  #if flim is not vector or length!=2 stop
  if(!is.null(flim))
  {if(!is.vector(flim)) stop("'flim' must be a numeric vector of length 2") else{
    if(!length(flim) == 2) stop("'flim' must be a numeric vector of length 2")}}    
 
  #if bp is not vector or length!=2 stop
  if(!is.null(bp))
  {if(!is.vector(bp)) stop("'bp' must be a numeric vector of length 2") else{
    if(!length(bp) == 2) stop("'bp' must be a numeric vector of length 2")}} 
  
  #if wl is not vector or length!=1 stop
  if(is.null(wl)) stop("'wl' must be a numeric vector of length 1") else {
    if(!is.vector(wl)) stop("'wl' must be a numeric vector of length 1") else{
      if(!length(wl) == 1) stop("'wl' must be a numeric vector of length 1")}}  
  
  #if res is not vector or length!=1 stop
  if(is.null(res)) stop("'res' must be a numeric vector of length 1") else {
    if(!is.vector(res)) stop("'res' must be a numeric vector of length 1") else{
      if(!length(res) == 1) stop("'res' must be a numeric vector of length 1")}}  
  
  #if there are NAs in start or end stop
  if(any(is.na(c(X$end, X$start)))) stop("NAs found in start and/or end")  
  
  #if any start higher than end stop
  if(any(X$end - X$start<0)) stop(paste("The start is higher than the end in", length(which(X$end - X$start<0)), "case(s)"))  
  
  #if any selections longer than 20 secs stop
  if(any(X$end - X$start>20)) stop(paste(length(which(X$end - X$start>20)), "selection(s) longer than 20 sec"))  

  # If n is not numeric
  if(!is.numeric(n)) stop("'n' must be a numeric vector of length 1") 
  if(any(!(n %% 1 == 0),n < 1)) stop("'n' should be a positive integer")

  # If length.out is not numeric
  if(!is.numeric(length.out)) stop("'length.out' must be a numeric vector of length 1") 
  if(any(!(length.out %% 1 == 0),length.out < 1)) stop("'length.out' should be a positive integer")
  
  #return warning if not all sound files were found
  fs <- list.files(path = getwd(), pattern = "\\.wav$", ignore.case = TRUE)
  if(length(unique(X$sound.files[(X$sound.files %in% fs)])) != length(unique(X$sound.files))) 
    message(paste(length(unique(X$sound.files))-length(unique(X$sound.files[(X$sound.files %in% fs)])), 
                  ".wav file(s) not found"))
  
  #count number of sound files in working directory and if 0 stop
  d <- which(X$sound.files %in% fs) 
  if(length(d) == 0){
    stop("The .wav files are not in the working directory")
  }  else X <- X[d,]
  
  # If SP is used need at least 24 selections
  if("SP" %in% methods & is.null(sp))
  {if(nrow(X) < 24)  stop("SP can only be used with at least 24 selections (number of rows in input data frame) as PCA only works with more units than variables (NOTE that you can also input your own matrix with the 'sp' argument)")}
  
  #check sp data frame
  if(!is.null(sp))
  {if(!is.data.frame(sp)) stop("'sp' must be a data frame") 
    if(nrow(sp) != nrow(X)) stop("'sp' must have the same number of selections than X") 
    if(!all(c("sound.files", "selec") %in% names(sp))) stop("'sound.files' or 'selec' columns missing in 'sp'")
  } 
  
  #if it argument is not "jpeg" or "tiff" 
  if(!any(it == "jpeg", it == "tiff")) stop(paste("Image type", it, "not allowed"))  
  
  #wrap img creating function
  if(it == "jpeg") imgfun <- jpeg else imgfun <- tiff
  
  
  #create empty list for method results
  disim.mats <- list()
  
  if("XCORR" %in% methods)
  {xcmat <- xcorr(X, wl = wl, frange = bp, ovlp = ovlp, dens = 0.9, parallel = parallel, pb = pb, na.rm = na.rm, cor.mat = TRUE)

  MDSxcorr <- stats::cmdscale(1-xcmat)  
  MDSxcorr <- scale(MDSxcorr)
  disim.mats[[1]] <- MDSxcorr
  
  #remove the ones that failed cross-corr
  if(na.rm) X <- X[paste(X$sound.files, X$selec, sep = "-") %in% rownames(MDSxcorr), ]
    }
  
  if("dfDTW" %in% methods)
    {dtwmat <- dfts(X, wl = wl, flim = flim, ovlp = ovlp, img = FALSE, parallel = parallel, length.out = length.out,
                    pb = pb, clip.edges = clip.edges, threshold = threshold)
   
    dtwmat <- dtwmat[,3:ncol(dtwmat)]
    
     if(scale)
       dtwmat <- t(apply(dtwmat, 1, scale))  
    
      dm <- dtw::dtwDist(dtwmat, dtwmat, method="DTW")  
  
  MDSdtw <- stats::cmdscale(dm)  
  MDSdtw <- scale(MDSdtw)
  disim.mats[[length(disim.mats) + 1]] <- MDSdtw
  }

  if("ffDTW" %in% methods)
  {dtwmat <- ffts(X, wl = 512, flim = flim, ovlp = ovlp, img = FALSE, parallel = parallel, length.out = length.out,
                  pb = pb, clip.edges = clip.edges, threshold = threshold)
  
  dtwmat <- dtwmat[,3:ncol(dtwmat)]
 
   if(scale)
    dtwmat <- t(apply(dtwmat, 1, scale))  
  
  dm <- dtw::dtwDist(dtwmat,dtwmat, method="DTW")  
  
  MDSdtw <- stats::cmdscale(dm)  
  MDSdtw <- scale(MDSdtw)
  disim.mats[[length(disim.mats) + 1]] <- MDSdtw
  }
  
  if("SP" %in% methods)
  { if(is.null(sp)) spmat <- specan(X, wl = 512, bp = flim, parallel = parallel, pb = pb, threshold = threshold) else spmat <- sp
  
  sp <- princomp(scale(spmat[,3:ncol(spmat)]), cor = FALSE)$scores[ ,1:2]

  PCsp <- scale(sp)
  
  disim.mats[[length(disim.mats) + 1]] <- PCsp
  }
  
  #name mats changing order to match order in whic methods are ran
  nms <- match(c("XCORR","dfDTW", "ffDTW", "SP"), methods)
  
  names(disim.mats) <- c("XCORR","dfDTW", "ffDTW", "SP")[!is.na(nms)] 
  
  maxdist <-lapply(disim.mats, function(x) max(stats::dist(x)))
  
  X$labels <- 1:nrow(X)
  
  combs <- combn(1:nrow(X), 4)
  
  if(nrow(X) == 4)  {n <- 1
  combs <- as.matrix(1:4)
  message("Only 1 possible combination of signals")
  } else if(n > ncol(combs)) {n <- ncol(combs)
  message(paste("Only",n, "possible combinations of signals"))
  }
  
  if(nrow(X) > 4)  combs <- as.data.frame(combs[,sample(1:ncol(combs), n)])
  

  #create matrix for sppliting screen
  m <- rbind(c(0, 2.5/7, 3/10, 5/10), #1
             c(4.5/7, 1, 3/10, 5/10), #2
             c(0, 2.5/7, 0, 2/10), #3
             c(4.5/7, 1, 0, 2/10), #4
             c(0, 1/2, 5/10, 9/10), #5
             c(1/2, 1, 5/10, 9/10), #6
             c(0, 2.5/7, 2/10, 3/10), #7 
             c(2.5/7, 4.5/7, 0, 5/10), #8
             c(4.5/7, 1, 2/10, 3/10), #9
             c(0, 3.5/7, 9/10, 10/10), #10
             c(3.5/7, 1, 9/10, 10/10)) #11
     
  # screen 1:4 for spectros
  # screen 5,6 for scatterplots
  # screen 7:9 for similarities/arrows
  # screen 10:11 method labels
  
  
  if(any(parallel == 1, Sys.info()[1] == "Linux") & pb)  message("saving graphs as image files:")
  
      comp.methFUN <- function(X, u, res, disim.mats, m, mar, flim)
    {
    rs <- combs[,u]
       X <- X[rs,]
  
       if(img)
  imgfun(filename = paste("comp.meth-", names(disim.mats)[1],"-",names(disim.mats)[2], "-", paste(X$labels, collapse = "-"), paste0(".", it), sep = ""), width = 16.25, height =  16.25, units = "cm", res = res)
  
  graphics::split.screen(m)
  
  mxdur<-max(X$end - X$start) + mar*2
  
  #set colors for numbers in scatterplots and spectrograms
  col <- rep("gray40", nrow(disim.mats[[1]]))
  
  col <- adjustcolor(col, alpha.f = 0.5)
  
  
  col[rs] <- hcl(h = seq(15, 375, length = 4 + 1), l = 65, c = 100)[1:4]

  col[rs] <- adjustcolor(col[rs], alpha.f = 0.8)

  
  invisible(lapply(c(7:9, 1:4, 5:6, 10:11), function(x)
  {
    graphics::screen(x)
    par( mar = rep(0, 4))
    if(x < 5) 
    { 
      r <-  tuneR::readWave(as.character(X$sound.files[x]), header = TRUE)
      tlim <- c((X$end[x] - X$start[x])/2 + X$start[x] - mxdur/2, (X$end[x] - X$start[x])/2 + X$start[x] + mxdur/2)
      
      mar1 <- X$start[x]-tlim[1]
      mar2 <- mar1 + X$end[x] - X$start[x]
      
      if (tlim[1] < 0) { tlim[2] <- abs(tlim[1]) + tlim[2] 
      mar1 <- mar1  + tlim[1]
      mar2 <- mar2  + tlim[1]
      tlim[1] <- 0
      }
      if (tlim[2] > r$samples/r$sample.rate) { tlim[1] <- tlim[1] - (r$samples/r$sample.rate - tlim[2])
      mar1 <- X$start[x]-tlim[1]
      mar2 <- mar1 + X$end[x] - X$start[x]
      tlim[2] <- r$samples/r$sample.rate}
      
      if (flim[2] > ceiling(r$sample.rate/2000) - 1) flim[2] <- ceiling(r$sample.rate/2000) - 1
      
      
      r <- tuneR::readWave(as.character(X$sound.files[x]), from = tlim[1], to = tlim[2], units = "seconds")
      
      spectro.INTFUN.2(wave = r, f = r@samp.rate,flim = flim, wl = wl, ovlp = ovlp, axisX = FALSE, axisY = FALSE, tlab = FALSE, flab = FALSE, palette = pal, grid = grid, ...)
      box(lwd = 2)
      if(x == 1 | x == 3) 
        text(tlim[2] - tlim[1], ((flim[2] - flim[1])*0.86) + flim[1], labels = X$labels[x], col = col[rs[x]], cex = 1.5, font = 2, pos = 2) else 
          text(0, ((flim[2] - flim[1])*0.86) + flim[1], labels = X$labels[x], col = col[rs[x]], cex = 1.5, font = 2, pos = 4)  
      if(grid)
      abline(v=c(mar1, mar2),lty = 4, col = "gray")
    }
    
    #upper left
    if(x == 5) {
      plot(disim.mats[[1]], col = "white", xaxt = "n", yaxt = "n", xlim = c(min(disim.mats[[1]][,1]) * 1.1, max(disim.mats[[1]][,1]) * 1.1), ylim = c(min(disim.mats[[1]][,2]) * 1.1, max(disim.mats[[1]][,2]) * 1.1))
      box(lwd = 4)
      centro <- apply(disim.mats[[1]], 2, mean)
      points(centro[1], centro[2], pch = 20, cex = 2, col = "gray3")
      cex <- rep(1, nrow(disim.mats[[1]]))
      cex[rs] <- 1.4
      text(disim.mats[[1]],  labels = 1:nrow(disim.mats[[1]]), col = col, cex =cex, font = 2)
    }
    
    #upper right
    if(x == 6) {
      plot(disim.mats[[2]], col = "white", xaxt = "n", yaxt = "n", xlim = c(min(disim.mats[[2]][,1]) * 1.1, max(disim.mats[[2]][,1]) * 1.1), ylim = c(min(disim.mats[[2]][,2]) * 1.1, max(disim.mats[[2]][,2]) * 1.1))
      box(lwd = 4)
      centro <- apply(disim.mats[[2]], 2, mean)
      points(centro[1], centro[2], pch = 20, cex = 2, col = "gray3")
      cex <- rep(1, nrow(disim.mats[[2]]))
      cex[rs] <- 1.4
      text(disim.mats[[2]],  labels = 1:nrow(disim.mats[[2]]), col = col, cex =cex, font = 2)
    }  
    
    #lower mid
    if(x == 8){
      plot(0.5, xlim = c(0,1), ylim = c(0,1), type = "n", axes = FALSE, xlab = "", ylab = "", xaxt = "n", yaxt = "n")
      lim <- par("usr")
      rect(lim[1], lim[3]-1, lim[2], lim[4]+1, border = adjustcolor("#EFAA7B", alpha.f = 0), col = adjustcolor("#EFAA7B", alpha.f = 0.15))
      arrows(0, 5.5/7, 1, 5.5/7, code = 3, length = 0.09, lwd = 2)
      text(0.5, 5.36/7,labels =round(stats::dist(disim.mats[[1]][rs[c(1,2)],])/maxdist[[1]],2), col = "black", font = 2, pos = 3)
      text(0.5, 5.545/7,labels =round(stats::dist(disim.mats[[2]][rs[c(1,2)],])/maxdist[[2]],2), col = "gray50", font = 2, pos = 1)
      arrows(0, 1.5/7, 1, 1.5/7, code = 3, length = 0.09, lwd = 2)
      text(0.5, 1.4/7,labels = round(stats::dist(disim.mats[[1]][rs[c(3,4)],])/maxdist[[1]],2), col = "black", font = 2, pos = 3)
      text(0.5, 1.63/7,labels =round(stats::dist(disim.mats[[2]][rs[c(3,4)],])/maxdist[[2]],2), col = "gray50", font = 2, pos = 1)
      arrows(0, 2/7, 1, 5/7, code = 3, length = 0.09, lwd = 2)
      text(0.69, 4.16/7,labels =round(stats::dist(disim.mats[[1]][rs[c(2,3)],])/maxdist[[1]],2), col = "black", font = 2, pos = 3)
      text(0.85, 4.4/7,labels =round(stats::dist(disim.mats[[2]][rs[c(2,3)],])/maxdist[[2]],2), col = "gray50", font = 2, pos = 1)
      arrows(0, 5/7, 1, 2/7, code = 3, length = 0.09, lwd = 2)
      text(0.3, 4.16/7,labels =round(stats::dist(disim.mats[[1]][rs[c(1,4)],])/maxdist[[1]],2), col = "black", font = 2, pos = 3)
      text(0.15, 4.4/7,labels =round(stats::dist(disim.mats[[2]][rs[c(1,4)],])/maxdist[[2]],2), col = "gray50", font = 2, pos = 1)  
    }
    
    #in between left
    if(x == 7){
      plot(0.5, xlim = c(0,1), ylim = c(0,1), type = "n", axes = FALSE, xlab = "", ylab = "", xaxt = "n", yaxt = "n")
      lim <- par("usr")
      rect(lim[1], lim[3]-1, lim[2], lim[4]+1, border = adjustcolor("#EFAA7B", alpha.f = 0.15), col = adjustcolor("#EFAA7B", alpha.f = 0.15))
      arrows(0.5, 0, 0.5, 1, code = 3, length = 0.09, lwd = 2)
      text(0.53, 0.5, labels =round(stats::dist(disim.mats[[1]][rs[c(1,3)],])/maxdist[[1]],2), col = "black", font = 2, pos = 2)
      text(0.47, 0.5, labels =round(stats::dist(disim.mats[[2]][rs[c(1,3)],])/maxdist[[2]],2), col = "gray50", font = 2, pos = 4)
    }
    
    #in between right
    if(x == 9){
      plot(0.5, xlim = c(0,1), ylim = c(0,1), type = "n", axes = FALSE, xlab = "", ylab = "", xaxt = "n", yaxt = "n")
      lim <- par("usr")
      rect(lim[1], lim[3]-1, lim[2], lim[4]+1, border = adjustcolor("#EFAA7B", alpha.f = 0.15), col = adjustcolor("#EFAA7B", alpha.f = 0.15))
      arrows(0.5, 0, 0.5, 1, code = 3, length = 0.09, lwd = 2)
      text(0.53, 0.5,labels =round(stats::dist(disim.mats[[1]][rs[c(2,4)],])/maxdist[[1]],2), col = "black", font = 2, pos = 2)
      text(0.47, 0.5,labels =round(stats::dist(disim.mats[[2]][rs[c(2,4)],])/maxdist[[2]],2), col = "gray50", font = 2, pos = 4)
      
    }
    
    #top (for method labels)
    if(x == 10){
      plot(0.5, xlim = c(0,1), ylim = c(0,1), type = "n", axes = FALSE, xlab = "", ylab = "", xaxt = "n", yaxt = "n")
      lim <- par("usr")
      rect(lim[1], lim[3]-1, lim[2], lim[4]+1, border = "black", col = adjustcolor("#4ABDAC", alpha.f = 0.3))
        text(0.5, 0.5, labels = names(disim.mats)[1], col = 'black', font = 2, cex = 1.2)
        box(lwd = 4)
        }
    
    if(x == 11){
      plot(0.5, xlim = c(0,1), ylim = c(0,1), type = "n", axes = FALSE, xlab = "", ylab = "", xaxt = "n", yaxt = "n")
      lim <- par("usr")
      rect(lim[1], lim[3]-1, lim[2], lim[4]+1, border = "black", col = adjustcolor("#4ABDAC", alpha.f = 0.3))
      text(0.5, 0.5, labels = names(disim.mats)[2], col = 'gray50', font = 2, cex = 1.2)      
      box(lwd = 4)
    }
  }
  ))
  if(img)
  invisible(dev.off())
  on.exit(invisible(close.screen(all.screens = TRUE)))
  }

      
      options(warn = -1)
      
      #parallel not available on windows
      if(parallel > 1 & Sys.info()[1] == "Windows")
      {message("parallel computing not availabe in Windows OS for this function")
        parallel <- 1}
      
      if(parallel > 1) {
        if(Sys.info()[1] == "Windows") {
           
        u <- NULL #only to avoid non-declared objects
         
         cl <- parallel::makeCluster(parallel)
         
         doParallel::registerDoParallel(cl)
         
         a1 <- foreach::foreach(u = 1:ncol(combs)) %dopar% {
           comp.methFUN(X, u, res, disim.mats, m, mar, flim)
         }
         
         parallel::stopCluster(cl)
         
      } 
        
        if(Sys.info()[1] == "Linux"){    # Run parallel in other operating systems
        
          if(pb)
        a1 <- pbmcapply::pbmclapply(1:ncol(combs), mc.cores = parallel, function(u) {
          comp.methFUN(X, u, res, disim.mats, m, mar, flim) 
        }) else
          
        a1 <- parallel::mclapply(1:ncol(combs), mc.cores = parallel, function(u) {
          comp.methFUN(X, u, res, disim.mats, m, mar, flim)
        })
        
      }
        if(!any(Sys.info()[1] == c("Linux", "Windows"))) # parallel in OSX
        {
          cl <- parallel::makeForkCluster(getOption("cl.cores", parallel))
          
          doParallel::registerDoParallel(cl)
          
          a1 <- foreach::foreach(u = 1:ncol(combs)) %dopar% {
            comp.methFUN(X, u, res, disim.mats, m, mar, flim)
          }
          parallel::stopCluster(cl)
        }
        
        
        } else {
          if(pb)
            a1 <- pbapply::pblapply(1:ncol(combs), function(u) 
      { 
        comp.methFUN(X, u, res, disim.mats, m, mar, flim)
      }) else       
        a1 <- lapply(1:ncol(combs), function(u) 
      { 
        comp.methFUN(X, u, res, disim.mats, m, mar, flim)
      })
      
      }

      }
