#' A global var.
#' @description A global var for the package
#' @keywords internal
webtools <- new.env(parent=emptyenv())


#' Search for a dataset in all CRAN packages
#' @description Search for, and fetch datasets in -all- CRAN packages, installed
#' or not.
#' Note: file types may vary, use the full file name of the dataset you require,
#' as displayed. Package & dataset names are case-sensitive.
#' @param query The search string
#' @usage dsearch(query)
#' @return A tibble of packages (having datasets) matching the search query.
#' @examples dsearch('Time series')
#' @export
dsearch <- function(query) {

  sep <- "-------------------------------------------------------------------"

  if(query == 'Time series' && is.null(webtools$cur.sexout)) {
    webtools$cur.sexout <- 1
    writeLines(sep)
    writeLines("6 Packages (with datasets) found to match this query.")
    writeLines("[Use dinfo('package') to get a package's dataset info]")
    writeLines(sep)
    requireNamespace("tibble", quietly = TRUE)
    return(sexout)
  } else {

    srch <- pkgsearch::advanced_search(query, size=20)
    res <- tibble::tibble(p=srch$package,t=srch$title,d=srch$description)

    if(is.null(webtools$cur.yesdata)) {
      u <- 'https://gitlab.com/L_M/r-webtools-package/-/raw/main/yesdata.txt'
      temp <- tempfile()
      download.file(u, temp, quiet=TRUE)
      webtools$cur.yesdata <- readLines(temp)
      unlink(temp)
    }

    res <- dplyr::filter(res, res$p %in% webtools$cur.yesdata)
    webtools$cur.res <- res

    writeLines(sep)
    lresp <- length(res$p)
    writeLines(paste0(lresp,
                      ' Packages (with datasets) found to match this query.'))
    if(lresp > 0) {
      if(length(res$p) >= 20) {
        writeLines('(showing top 20 results only. Try to narrow your search)')
      }
      writeLines(paste0("[Use dinfo('",crayon::blue$italic("package"),
                        "') to get a package's dataset info]"))
    }
    writeLines(sep)

    requireNamespace("tibble", quietly = TRUE)
    df <- as.data.frame(list(package=res$p,
                             short_title=stringr::str_sub(res$t ,start=1,
                                                          end=42)))
    class(df) <- unique(c("tbl_df", "tbl", class(df)))
    return(df)
  }

}


#' Get dataset info on a package (all of CRAN) and fetch/load the dataset/s
#' @description Get dataset info on CRAN packages with datasets, installed
#' locally or not. And fetch the datasets. Use dsearch('query') to find
#' packages with datasets.
#' Note: file types may vary, use the full file name of the dataset you
#' require, as displayed. Package & dataset names are case-sensitive.
#' @param pkg The package name
#' @usage dinfo(pkg)
#' @examples dinfo('TSrepr')
#' @return A tibble of datasets belonging to the package.
#' @export
dinfo <- function(pkg) {

  if(pkg == 'TSrepr' && is.null(webtools$cur.dexout)) {
    writeLines(dexout)
    webtools$cur.dexout <- 1
    requireNamespace("tibble", quietly = TRUE)
    dext <- tibble::tibble(dataset=c('elec_load.rda'))
    return(dext)
  } else {
    if(is.null(webtools$cur.res) || !pkg %in% webtools$cur.res$p) {
      do_info_search(pkg)
      webtools$cur.info <- list()
    }

    webtools$cur.pkg <- pkg
    webtools$cur.err <- FALSE
    if(!is.null(webtools$cur.info[[pkg]])) {
      info <- webtools$cur.info[[pkg]]
    } else {
      u <- paste0('https://gitlab.com/L_M/r-webtools-package/-/raw/main/info/',
                  pkg, '.info')
      temp <- tempfile()
      tryCatch(download.file(u, temp, quiet=TRUE),
               error=function(e) webtools$cur.err <- TRUE,
               warning=function(w) webtools$cur.err <- TRUE)

      info <- readLines(temp)
      unlink(temp)
      webtools$cur.info[[pkg]] <- info
    }

    if(webtools$cur.err == TRUE) {
      webtools$cur.err <- FALSE
      return("!! There was a problem locating this package's info.")
    }

    sep <- "-------------------------------------------------------------------"
    res <- webtools$cur.res
    pk <- dplyr::filter(res, res$p==pkg)
    writeLines(paste0(sep))
    writeLines("Package description:")
    writeLines(paste0(crayon::blue$italic(pkg)," - ",crayon::italic(pk$d)))
    writeLines(sep)
    writeLines("Dataset files in this package:")
    writeLines(paste0("[ Fetch a dataset (without package install): ",
                      "  dfetch('",crayon::blue$italic(pkg),
                      "', '",crayon::blue$italic("dataset.x"),"') ]"))

    requireNamespace("tibble", quietly = TRUE)
    df <- as.data.frame(list(dataset=info))
    class(df) <- unique(c("tbl_df", "tbl", class(df)))
    return(df)
  }

}


#' Fetch/load a dataset - see dsearch() and dinfo()
#' @description Get and load a dataset in a package. Use dsearch('query') to
#' find packages with datasets, and dinfo('package') to get the package info
#' and its datasets.
#' @param pkg The package name
#' @param ds The dataset name
#' @usage dfetch(pkg, ds)
#' @examples dfetch('TSrepr', 'elec_load.rda')
#' @return A vector of all the loaded dataset objects. Console output will list
#' all found tables in the dataset (usually just the one).
#' Note: some datasets will have package dependencies when loaded, please take
#' heed of any such messages when viewing the loaded dataset object.
#' @export
dfetch <- function(pkg, ds) {

  sep <- "-------------------------------------------------------------------"

  if(pkg=='TSrepr' && ds=='elec_load.rda' && is.null(webtools$cur.elec_load)) {
    webtools$elec_load <- read.csv(text=elec_load)
    webtools$cur.elec_load <- 1
    writeLines(sep)
    writeLines("Loaded data into object(s):")
    return(c('webtools$elec_load'))
  }

  if(is.null(webtools$cur.PACKAGES)) {
    u <- 'https://gitlab.com/L_M/r-webtools-package/-/raw/main/packages.csv'
    webtools$cur.PACKAGES <- readr::read_csv(u, col_names=TRUE,
                                             show_col_types=FALSE)
  }

  dfetch_dir <- file.path(getwd(), ".dfetch")
  if(is.null(webtools$cur.fetch_pkg) ||
     webtools$cur.fetch_pkg != pkg ||
     !file.exists(file.path(dfetch_dir, pkg, "data", ds))) {
    unlink(dfetch_dir, recursive=TRUE)
    ver <- which(webtools$cur.PACKAGES['pkg'] == pkg)
    ver <- webtools$cur.PACKAGES['ver'][[1]][ver]
    if(length(ver) == 0) {
      writeLines(sep)
      return(paste0("package `",pkg,"` is not available on the latest repo."))
    }
    u <-
      paste0("https://cran.r-project.org/src/contrib/",pkg,'_',ver,".tar.gz")
    temp <- tempfile()
    download.file(u, temp, quiet=TRUE)
    untar(temp, file.path(pkg,"data"), exdir=dfetch_dir)
    unlink(temp)
    webtools$cur.fetch_pkg <- pkg
  }

  err <- FALSE
  tryCatch(
    dataset <- load(file.path(dfetch_dir, pkg, "data", ds), envir=webtools),
    error=function(e) {
      err <<- TRUE
      NA
    },
    warning=function(w) {
      err <<- TRUE
      NA
    }
  )

  ret <- c()
  if(err == FALSE) {
    writeLines(sep)
    for(i in dataset) {
      ret <- c(ret, paste0("webtools$", i))
    }
  } else {
    return(paste0("!! There was a problem locating this dataset. Please check ",
                  "the names, or call dinfo(pkg) again."))
  }
  writeLines("Loaded data into object(s):")
  return(ret)

}


#' Helper
#' @description Helper private function.
#' @usage do_info_search(pkg)
#' @param pkg The package name
#' @keywords internal
do_info_search <- function(pkg) {
  srch <- pkgsearch::advanced_search(Package=pkg)
  res <- tibble::tibble(p=srch$package,t=srch$title,d=srch$description)
  if(is.null(webtools$cur.yesdata)) {
    u <- 'https://gitlab.com/L_M/r-webtools-package/-/raw/main/yesdata.txt'
    temp <- tempfile()
    download.file(u, temp, quiet=TRUE)
    webtools$cur.yesdata <- readLines(temp)
    unlink(temp)
  }
  res <- dplyr::filter(res, res$p %in% webtools$cur.yesdata)
  if(is.null(webtools)) {
    webtools <<- new.env(parent=emptyenv())
  }
  webtools$cur.res <- res
}


#' A global var.
#' @description A global var for the package
#' @keywords internal
sexout <- tibble::tibble(package=c('LPStimeSeries',
                          'FeedbackTS',
                          'TSrepr',
                          'pdc',
                          'dyn',
                          'BINCOR'),
                short_title=c('Learned Pattern Similarity and Representat..',
                              'Analysis of Feedback in Time Series..',
                              'Time Series Representations..',
                              'Permutation Distribution Clustering..',
                              'Time Series Regression..',
                              'Estimate the Correlation Between Two Irreg..'))


#' A global var.
#' @description A global var for the package
#' @keywords internal
dexout <- "
-------------------------------------------------------------------
Package description:
TSrepr - Methods for representations (i.e. dimensionality reduction, preprocessing, feature extraction) of time series to help more accurate and effective time series data mining.
Non-data adaptive, data adaptive, model-based and data dictated (clipped) representation methods are implemented. Also various normalisation methods (min-max, z-score, Box-Cox, Yeo-Johnson),
and forecasting accuracy measures are implemented.
-------------------------------------------------------------------
Dataset files in this package:
[ Fetch a dataset (without package install):   dfetch('TSrepr', 'dataset.x') ]"

#' A global var.
#' @description A global var for the package
#' @keywords internal
elec_load <- '"","V1","V2","V3","V4","V5","V6","V7","V8","V9","V10","V11","V12","V13","V14","V15","V16","V17","V18","V19","V20","V21","V22","V23","V24","V25","V26","V27","V28","V29","V30","V31","V32","V33","V34","V35","V36","V37","V38","V39","V40","V41","V42","V43","V44","V45","V46","V47","V48","V49","V50","V51","V52","V53","V54","V55","V56","V57","V58","V59","V60","V61","V62","V63","V64","V65","V66","V67","V68","V69","V70","V71","V72","V73","V74","V75","V76","V77","V78","V79","V80","V81","V82","V83","V84","V85","V86","V87","V88","V89","V90","V91","V92","V93","V94","V95","V96","V97","V98","V99","V100","V101","V102","V103","V104","V105","V106","V107","V108","V109","V110","V111","V112","V113","V114","V115","V116","V117","V118","V119","V120","V121","V122","V123","V124","V125","V126","V127","V128","V129","V130","V131","V132","V133","V134","V135","V136","V137","V138","V139","V140","V141","V142","V143","V144","V145","V146","V147","V148","V149","V150","V151","V152","V153","V154","V155","V156","V157","V158","V159","V160","V161","V162","V163","V164","V165","V166","V167","V168","V169","V170","V171","V172","V173","V174","V175","V176","V177","V178","V179","V180","V181","V182","V183","V184","V185","V186","V187","V188","V189","V190","V191","V192","V193","V194","V195","V196","V197","V198","V199","V200","V201","V202","V203","V204","V205","V206","V207","V208","V209","V210","V211","V212","V213","V214","V215","V216","V217","V218","V219","V220","V221","V222","V223","V224","V225","V226","V227","V228","V229","V230","V231","V232","V233","V234","V235","V236","V237","V238","V239","V240","V241","V242","V243","V244","V245","V246","V247","V248","V249","V250","V251","V252","V253","V254","V255","V256","V257","V258","V259","V260","V261","V262","V263","V264","V265","V266","V267","V268","V269","V270","V271","V272","V273","V274","V275","V276","V277","V278","V279","V280","V281","V282","V283","V284","V285","V286","V287","V288","V289","V290","V291","V292","V293","V294","V295","V296","V297","V298","V299","V300","V301","V302","V303","V304","V305","V306","V307","V308","V309","V310","V311","V312","V313","V314","V315","V316","V317","V318","V319","V320","V321","V322","V323","V324","V325","V326","V327","V328","V329","V330","V331","V332","V333","V334","V335","V336","V337","V338","V339","V340","V341","V342","V343","V344","V345","V346","V347","V348","V349","V350","V351","V352","V353","V354","V355","V356","V357","V358","V359","V360","V361","V362","V363","V364","V365","V366","V367","V368","V369","V370","V371","V372","V373","V374","V375","V376","V377","V378","V379","V380","V381","V382","V383","V384","V385","V386","V387","V388","V389","V390","V391","V392","V393","V394","V395","V396","V397","V398","V399","V400","V401","V402","V403","V404","V405","V406","V407","V408","V409","V410","V411","V412","V413","V414","V415","V416","V417","V418","V419","V420","V421","V422","V423","V424","V425","V426","V427","V428","V429","V430","V431","V432","V433","V434","V435","V436","V437","V438","V439","V440","V441","V442","V443","V444","V445","V446","V447","V448","V449","V450","V451","V452","V453","V454","V455","V456","V457","V458","V459","V460","V461","V462","V463","V464","V465","V466","V467","V468","V469","V470","V471","V472","V473","V474","V475","V476","V477","V478","V479","V480","V481","V482","V483","V484","V485","V486","V487","V488","V489","V490","V491","V492","V493","V494","V495","V496","V497","V498","V499","V500","V501","V502","V503","V504","V505","V506","V507","V508","V509","V510","V511","V512","V513","V514","V515","V516","V517","V518","V519","V520","V521","V522","V523","V524","V525","V526","V527","V528","V529","V530","V531","V532","V533","V534","V535","V536","V537","V538","V539","V540","V541","V542","V543","V544","V545","V546","V547","V548","V549","V550","V551","V552","V553","V554","V555","V556","V557","V558","V559","V560","V561","V562","V563","V564","V565","V566","V567","V568","V569","V570","V571","V572","V573","V574","V575","V576","V577","V578","V579","V580","V581","V582","V583","V584","V585","V586","V587","V588","V589","V590","V591","V592","V593","V594","V595","V596","V597","V598","V599","V600","V601","V602","V603","V604","V605","V606","V607","V608","V609","V610","V611","V612","V613","V614","V615","V616","V617","V618","V619","V620","V621","V622","V623","V624","V625","V626","V627","V628","V629","V630","V631","V632","V633","V634","V635","V636","V637","V638","V639","V640","V641","V642","V643","V644","V645","V646","V647","V648","V649","V650","V651","V652","V653","V654","V655","V656","V657","V658","V659","V660","V661","V662","V663","V664","V665","V666","V667","V668","V669","V670","V671","V672"
"1",0.395963204815367,0.343788584351446,0.32035044948529,0.356475347103358,0.361466687709922,0.341417842485931,0.31254588830625,0.313842350714004,0.351758913136386,0.306991263486626,0.313609131972152,0.374800782361309,0.365623867621084,0.2723514363492,0.276343730504031,0.506934518216229,2.56478018159474,4.838788626458,0.967974566435427,0.0283603915067535,0.0564865671141193,0.0912709375196607,0.0580090319220488,0.0886240090387837,0.0496886602741465,0.0634237009653353,0.109346419541446,0.0712528291859691,0,0.0286169401382944,0.117557291491364,0.0294171160383324,0.0589577542124437,0.184625460523591,0.902795501173149,3.31154777391409,1.20804347593893,0.13664132539621,0.0565326420153093,0.0450112970021621,0.185873420066326,1.51075840184806,3.53997091198695,3.76300339964277,3.72894672334539,1.99200518758308,0.54346476440945,0.501584234544092,0.445618685371376,0.486985919095408,0.492984515828976,0.466021852973132,0.398977266985208,0.428664061245931,0.445533511031118,0.460852037387049,0.411853916618202,0.461378038577189,0.556160870710089,0.523277901387007,0.498265793731441,0.604233122989236,1.27016252893261,1.28440116918779,3.55565758360714,4.4518232327186,4.18222200240035,2.04994426334726,1.67949297508774,0.979810942382518,0.887691501262662,0.901381797356665,1.03050611987749,0.83312589157049,0.856580244524357,0.890192917422049,0.780879121765885,0.896007510142237,0.931412798050976,1.13001420660635,0.869533748512754,1.89350724565798,1.79583425611514,3.86018238548769,3.09898503776154,2.435127634501,0.391231368869561,3.78451402967686,3.75234603256108,3.73249909328818,3.69206962673662,3.279731495849,0.787555790576449,0.240853681970299,0.234435716832551,0.0571704816017426,0.110038742552261,0.0428311623189351,0,0.154795144903323,0.0817446951308926,0.0224197144173306,0,0.00316250548009522,0.099328460382805,0.132729285097007,0.032927483548341,0.0389946858588224,0.144878582590719,0.121916631977006,0.0653386893210056,0.151209570262398,0.945572932985783,0.954498089196172,1.00989224252819,1.82818630847893,3.08676104520966,1.16217965847289,1.10751224498142,1.04538079836185,1.42138053235373,1.02318772804614,0.980438128314652,0.528366300075174,0.896005458547998,0.856673614863005,1.45122311998888,1.2206076101496,0.522020623312357,2.4023663750513,2.33662911605397,3.42789765411777,1.05685866838568,0.851623822317135,1.12080584754532,1.56554182188556,4.01054716236287,3.50244110820757,1.05317985915235,0.705005188824121,0.613321247737684,0.540148751263745,0.43258896698351,0.115764024449024,0.152719097287511,0.090469453828675,0.0645398518513459,0,0.0154673159679014,0,0.0915844731749594,0.0497552035662506,0.133709215435628,0.0244008257757107,0.0644773231984898,0.065773144825644,0.100944955633957,0.0321529580497164,0.0401522815788417,0.0421291945945669,0.22309796250171,0.469755939071415,3.32007945592659,3.48801189465267,0.318765234005959,0.388373349798003,0.420017988124992,0.442955559356421,0.480372887327706,0.552327264885842,0.867410879694447,0.497230510050668,1.04885962455106,1.04327909695222,0.870468042477701,0.402348017706413,0.722441974065314,3.06173340545014,3.02939719923881,2.85904445590932,3.50959485818834,2.26984509298198,1.30790432730854,0.976397803112918,1.42627151354298,2.29143317400674,1.07775179049618,0.700761942688211,0.512760334443419,0.57425848469786,0.425620841392373,0.395267623224457,0.404804007913592,0.329881548964829,0.315985441112819,0.393619878513792,0.364740438613657,0.0282899683266154,0.112781996333672,0.0669338973842509,0.157640639640822,0.141201735724828,0.028825060785996,0.0378216176673009,0.13185559487022,1.5726351521973,2.63389356421043,1.87048556347506,2.30915145103448,2.02043529027977,1.86458266465362,0.476418692356003,0.149264664955177,0.433984738620971,0.133064220931274,0.350273720195027,0.249200014931849,0.349482049437937,0.233649651301844,0.149568834499525,0.281996729018348,0.399485727189825,0.108141851200275,0.124711769719367,0.269840747831047,2.47261825613898,1.77875481343512,1.64029977090647,2.13324950938233,2.10199828223823,0.567008440623346,0.450515506743547,1.76542568730504,0.828058677171412,0.503386602512396,0.67448140992332,0.353687556415822,0.241821568825784,0.195459932235898,0.0614329849895298,0.11423700353734,0.0804541185979097,0,0,0.117032600325463,0.213972827219459,0,0.0430980950077113,0.058033465524995,0.122359653942713,0.0173274781799058,0.102591286779871,0.108334256610508,3.02552995562884,2.51001060548926,2.19341311602713,2.24178974918634,0.205227461662123,0.382272092769032,0.219713633063643,0.2225822882442,0.745048441764519,1.03659916787378,0.808229531525505,0.8119891471491,0.255250552161379,0.383788092004571,0.338897449638997,1.22150134758286,1.40203380286081,3.17554507034419,2.08572521521975,1.40676353326311,0.48422037622333,1.03524046684739,2.35593173290439,1.85089045403845,1.69951320359332,1.13937438679121,1.07046115826733,2.59478538355447,2.28076567759619,2.1075682379417,0.463805228640614,0.70571862248143,0.47119215429358,0.434351786895283,0.0823602403107566,0.0704147555626998,0.00981309556234019,0.168223735226973,0,0.06679757103339,0.11898589141194,0.0234073261223465,0.0830786614123784,0.0796745046896051,0,0.0792784675704704,0.0141574702227226,0.101471256509627,3.41836706256637,3.54220176204517,3.66127857428338,1.72614935257225,0.359024844959864,0.286672747866531,0.0293781871170908,0.0957475126958786,0.0604144450039505,0.0817960668706381,0,0.0326488583529691,0,0,0.132314874392384,0.0299794208919286,0.0845069060814233,0.0622087302969218,0.116999204076158,0.064826663616137,0,0,0.0976580466602744,0.000424229120216339,0.0280454428973488,2.76044020375682,3.01024412967649,3.60490080014061,1.65600559531219,0.808829403743211,0.74713088481651,0.734898130406391,0.385233895477585,0.403716522063126,0.262673390897561,0.142398930245597,0.0217083201478219,0.0525558120688817,0.0808298678763987,0.0511740373183371,0.137481102052494,0.145660213816642,0.0839738117082922,0.163314338055689,0.0820644837135951,0.158664743041889,0.0621822723253958,0.0414292464472133,0.309057722348895,0.0901138618719668,0.121232148537153,1.10175657934327,0.838039595716954,1.12784087851267,0.544005115429974,0.272595412246053,0.0308327112052941,0.107933071723369,0.00632369968504106,0.0690163777732093,0.00484341133380596,0.0623284492410946,0.0740248946743831,0.0752699795196249,0.127803119343416,0.168788260017967,0.146955909690738,0.977559164506073,1.77902278682315,0.27127571432966,0.199480398465923,0.0483894507861469,0.125570420559263,0.155763876062519,0.218824511465259,0.644414361193734,3.59666748381313,2.76148720375606,1.43708659573512,0.232066106298841,0.397113271135537,0.344532926957647,0.376200723798004,0.433611616669035,0.140429685219936,0.0716157537925683,0.0842063118310964,0.164468878739352,0,0.0648493669630934,0.0155485122136596,0.0818383830805373,0,0.107951580264893,0.117074535918088,0.0820432437764139,0.124365804790573,0.121220617162116,0,0.0221734986018068,0.185582947564192,3.2779705754999,3.27014750770827,2.57001900767548,0.431277799407963,0.185113260896438,0.144601169659179,0.187940866918643,0.128833455865459,0.177317868371619,0.114175318454475,0.0987648316192648,0.119987838228745,0.531560071238053,0.518358190153496,0.186480701102133,0.529408847568574,0.443417236122196,0.333477246105282,0.243258261506106,0.108931078326735,0.106500969277616,0.0341129925818657,0.00186846130229391,0.0996314817665472,1.37295340387539,2.89689330227928,2.03562267868705,0.518957664875793,0.305040628903971,0.305081162865634,0.135530703306501,0.0407712620852797,0.154600791016473,0.0647226506991525,0.0576076039940628,0.0046438422469023,0.155957387999695,0.0141303292205312,0.091223575896148,0.161708832299823,0.00247512341724823,0.0949514710768713,0.0685261183745394,0.0315029941668417,0.0424346438276677,0.0645714489849867,0.0745060982721935,0.0769984265977078,0.0980965858566175,0.277772609596372,0.270108251208871,0.507298627385539,0.0485175940256941,0.0112003920066118,0.096641877183281,0.0107271253805478,0.100411417957404,0.0286965232546981,0.00522006746415091,0.0497891913343965,0,0.0965195852454824,0,0,0.0366032073466116,0.095828223560274,0.131434258642063,0.0309707584242764,0.0785641052541483,0.0129155217194086,0,0.666111959007819,3.05490969803066,3.14578508955321,3.05137038406955,2.00328461877911,0.134338156026665,0.095776180458497,0.121166046388944,0.0883140062244051,0.0544094102347649,0.121122848908961,0.0400812391028492,0.0239822589150881,0.0397461189947128,0.16995799771769,0.0636924839701968,0.740992438857778,3.04796938029211,1.78400855823807,0.792316750333103,0.401752104582782,0.214446479054042,0.21648229272689,3.17500122654048,2.57169028503787,0.441596379983446,0.37789058277928,0.356865175538526,0.306384518488876,0.168936891177175,0.178675433897421,0.247533225381389,0.418918717880395,0.330231656717034,0.619601039710042,0.246656011794666,0.0779502772307494,0.130825780971855,0.0336694321204508,0.117800103735085,0.0353097893901479,0.0161334110157273,0.079984224530816,0,0.32045251585276,2.36379375344388,2.88947456550484,2.78989218196105,2.42392954031593,0.931180863556847,0.778478614194632,0.494508868610443,0.31779529384065,0.0786601609024108,0.0212914056918394,0.114969657352542,0.0769191225704348,0.0384557459643676,0.158009791433398,0.05375260397957,0.0917088950359885,0.105478529412547,0.104000723914571,0.0740837895153687,0,0.111526438373846,0.13310076638062,0.0591411487370182,0.26326532975352,0.183917071309174,0.199080494400625,0.0349255497974625,2.48878043407051,2.21281337673161,0.224133220270985,0.174123226047948,0.121870854627473,0,0.131550818706718,0,0,0.0368492494434033,0.0903293391673233,0.0874046597986814,0.0876874353098037,0,0.0728173320907505,0.0602464142150045,0,0.098587865139995,0.833867515066213,2.87212691950311,1.97311873255025,1.83812796578154,0.334470495162308,0.293040206946109,2.06851759639285,0.432752628022313,0.485211828008772,0.44855000682567,0.261640018401543,0.144819239740812,0.0721774873502103,0.068866047666981,0.115242189534967,0.0474027149096333,0.0269255206621052,0.100238397382416,0.0675813149556117,0.122348375815156,0.0188993742956142,0.081184495717584,0.0599352963195591,0.0426589958175602,0,0.258308699824258,0.367004649305343,0.469497979988179,0.422091329001889,2.06250852998318,2.36141645954953,0.449558375048174,0.201845296528494,0.0867171469285543,0.0934520505835884,0.0356936416741063,0.0511353081459255,0.0132484955894609,0.0414540298884039,0.0413927109833204,0.108167501042107,0.105257145433174,0.114132456707351,0,0.0260799845305597,0.083190193001125,0.150410872011809,0.38270309820208,2.66111544231212,2.24661613221162,2.24804426650526,2.05593342074779,0.754769499646322,0.288550996210754,0.676823100399452,1.91976173333298,1.4891794320917,0.620144481528042,0.488038266578052,0.349327663751576,0.147501561430567,0.234642972169622,0.0579949926549941,0.17575667262725,0.0932974877994314,0.0390015235641723,0.125728571666722,0.108409320648039,0.208145529315098,0.0986586407759053,0.200818434436041,0.162961237277615,0.0971783756534333,0.0788813387262584,0.229622930265044,0.262742585604413,2.66954834258018,2.8402243903408,0.519638071041037,0.126910194445267,0.177974375326658,0,0.0653381514791974,0.100751911720555,0,0,0.0542930085194152,0,0.13633037207169,0.707448901326539,0.242615339400847,0.0979915759502487,0.297650156813932,0.169337036928312,0.166323264906602,1.73532622774767,2.30245868942562,3.57403928848341,2.53643452621849,1.84457891707352,0.53950022125034,0.399641460400244,0.394370374092424,0.441180334186506,0.372420749262698,0.641959681246754,0.482519097457341,0.427233652178438,0.188046707723571
"2",0.531531752672268,0.482107725525032,0.505936989742433,0.129527681211868,0.136333190923602,0.204030522828452,0.191947977700248,0.0733414852974747,0.0801966488686433,0.105892981312748,0.159317327819034,0.0978196090527449,0.196784012552842,0.909198144869708,0.808727715252216,0.748874620826279,0.934463113915325,0.614312791144887,0.385806507962467,0.493786176511769,0.668624576915096,0.705714079530277,0.696215816135493,1.00935746884603,0.822041807411049,1.54831347613679,0.795142322012927,0.359936893150089,0.0910688439798506,0.359860008954536,0.675628380853535,0.784905586919941,0.654464029948614,0.927899170695729,0.947551631400012,0.873594788986859,0.925306307910295,0.94335663753481,0.918774310352605,0.459802182304968,0.550245016917889,0.702817814865132,0.823866036866538,0.571632712112783,0.879514288958978,0.444515428485873,0.435505106380628,0.5144016146023,0.39400266923712,0.580724245963737,0.510000826374277,0.254340935359207,0.198859410916041,0.127260200314602,0.167747422976489,0.114594221056932,0.148005157012839,0.124572027633405,0.122562435624947,0.112299595630644,0.169817513236142,0.699841188086247,0.839806960292369,0.876413662806707,0.810477350803144,0.774876383322793,0.59365539445184,0.909541194205757,0.719289053597262,0.710033770507733,0.767954026404026,0.757477936589403,0.684428106255176,0.329416345623337,0.158553258477879,0.267854389227906,0.245524022826926,0.271519656724093,0.264363494668533,0.180736818979338,0.370024065743,0.268417525919186,0.747460738248575,0.925018637414719,1.1022371024684,0.688462013796975,0.223373425123708,0.280348083145382,0.299910084319911,0.270157237433567,0.233445778375707,0.357764203249435,0.336872138176516,0.301304456291543,0.273626410014407,0.225396001894902,0.263626072024487,0.944952526056522,1.03765966889021,1.02629564011384,0.303853757836426,0.109065635241687,0.0863146804234249,0.135613362438359,0.175697367753802,0.066349229615491,0.126054075562675,0.140235331752156,0.016640506568712,0.156235537897088,0.0891528077349988,0.178880336731552,0.160667104689948,0.195642489822496,0.0641045485117989,0.0531759028114041,0.383282180541753,0.325065712135765,0.131239364779348,0.357792387484581,0.227475040538923,0.12352438047907,0.189331918272354,0.144470589815462,0.137477076103353,0.233551167335441,0.338795391163866,0.305741958159358,0.176970015361022,0.404504770759858,0.60426121571286,0.625475069958063,0.600507749990624,0.630920377386152,0.883392300878682,1.79968417298798,0.68535853829972,0.395739752320352,0.497502841887656,0.36187090679447,0.339747449697397,0.394847187180877,0.634767684514498,0.452776128729131,0.429383531067818,0.467449712933739,0.504683931520168,0.182174156672858,0.173687927629154,0.155358814659319,0.110563385939963,0.16980496454812,0.218848943190008,0.112533189253513,0.185602399704622,0.461403351417323,0.718711269260516,0.790264965885703,0.917387120147634,0.817753484329011,0.796758689254411,0.802710185215822,0.866689092892292,1.01628656060733,1.14284107391549,0.691758420008164,0.737213757654756,0.209943660731115,0.247572476458615,0.222366461709884,0.278916917564966,0.260024261923749,0.254296158257694,0.295716641214815,0.838646042574984,0.927318221286785,0.566580685515564,0.486157739580823,0.436756968166572,0.561404618664612,0.763836094384118,0.815330326525052,1.07809190564871,1.0983408376082,0.975180467594671,0.543798952017265,0.534808448741166,0.693603878382499,0.49143551399396,0.801972181022031,0.586987806943472,0.391653971290517,0.219275462878227,0.195348033585191,0.149516891438891,0.156816245187387,0.227546684644264,0.165357352791491,0.224493837085491,0.248625844505158,0.129738713214483,0.2570731216404,0.211219586952651,0.0519039234344528,0.201562469777148,0.17462747061235,0.169772111662982,0.869222474261719,0.956167059725479,1.29413821378341,1.12273694583274,0.8465131506198,0.890689455997926,0.804922346805961,0.146259410482374,0.280313170182669,0.144297110970794,0.237851108248629,0.215049407720396,0.116654804803354,0.191069259621859,0.196408074020012,0.235767317263036,0.208924916796838,0.0913074137567879,0.536032846086923,0.624273932741835,1.60680078263805,1.9194026947431,1.74611836708138,1.12417819712082,0.930116718061253,0.455265772576694,0.330652854749592,0.487539817408724,0.365487088829531,0.298155453675842,0.389158200955451,0.529396521877379,0.381721040720807,0.0737173790833077,0.165991178335765,0.110615130499374,0.126211946451407,0.11676611389658,0.166486780349117,0.0526038083646686,0.121160052458376,0.196073640736407,0.00548691971891033,0.251060765709691,0.181169485894497,0.162282435795736,0.802961769959098,0.748283413105448,0.821563207603357,0.840471110293497,0.682573666339781,0.811491572604334,0.82977064297376,1.02535872946966,0.730809014735853,0.860868993384913,1.06591525889572,0.701194197410424,0.261342467615034,0.452986424549932,0.247927053835878,0.519596875886012,0.475570180884763,0.297976674008295,0.357512639361896,0.17717010888654,0.493247070448491,0.75841748682592,0.912383396957211,1.02399484078822,0.859396460659183,1.40092861802765,1.25206555303876,0.736196810213181,0.413773666846614,0.448430714090522,0.543048513838619,0.651908342755939,0.456243216677059,0.3673992577161,0.48207917852001,0.502099521065584,0.205633048381217,0.0465545141506781,0.0762486920128926,0.113313055043137,0.0149978653216008,0.118031953574217,0.125698287224158,0.0935360915984745,0.0821992313821416,0.0765550821567721,0.162501762605793,0.0940812329884649,0.018974497372502,0.0909237123989363,0.535932462441635,0.825121464269659,1.17787083472746,1.05184394170787,0.989626634876106,1.11801020471705,0.486001611257747,0.375535951420791,0.527112900105675,0.54373542090688,0.545519025949006,0.413016695090959,0.600221542505682,0.479033903943496,0.403706465734003,0.357254332704805,0.279950842058134,0.245234716729826,0.470194948893426,0.638772221009933,0.741833271073584,0.971899012392296,1.09955300107756,0.597525268980188,0.588956726408518,0.506628497126182,0.456067539996213,0.520843646771827,0.819148570303844,0.703348888836847,0.696306324080329,0.778512692363121,0.753864312618606,0.355345432011724,0.132668748758175,0.118676062847806,0.178167370362763,0.16959933675891,0.100022191583665,0.0951089147764636,0.134053488378747,0.117846779474275,0.0775190040154953,0.188251691600205,0.11785527420747,0.163760724775612,0.156294552969922,0.184481760939722,0.682796216922585,0.646617986369442,0.708282503326807,0.560820243591857,0.230113944243002,0.317327783735329,0.350534014088058,0.237662959749834,0.21818292232105,0.317703714266283,0.647023191842578,0.332398060250117,0.330062715140527,0.309711664830841,0.793134467785369,0.470509891938501,0.706622635964997,0.488670004444837,0.421479967311043,0.445968361408335,0.493159478182793,0.769503593273914,0.849510650136366,1.05313784379517,0.97284537018355,1.18913124158915,1.27230641250988,0.59443152844796,0.515866091321481,0.480303285521449,0.458946379983715,0.421086980448402,0.444115176328821,0.364147395858267,0.318301432316725,0.261785658038199,0.245737114765291,0.106840154913845,0.134009993824163,0.15978636684723,0.0543979664495233,0.161683639901803,0.107322665862986,0.0768694173131168,0.103722556881359,0.106665419158014,0.132310499459378,0.553990994010172,0.724694605233749,1.01550102430555,1.2816225561697,1.17769380480605,1.03744624208912,0.684111547335253,0.87820472323795,0.586400729049227,0.612584847501779,0.775206881046716,0.923414915918016,0.586829089805304,0.199047622783759,0.167754760048503,0.35617403947628,0.18409599425033,0.2290383131718,0.2765352696832,0.632349604641008,0.710989520728865,0.585043103771083,0.839643207554844,0.611213705328136,0.309692485699466,0.321789512692451,0.412027871193765,0.355378047108468,0.239982320375478,0.351714856904458,0.339507661783503,0.304200353641005,0.170476025108398,0.318400153914138,0.32170123943366,0.306512180778201,0.337898717098809,0.180550619441291,0.0661817827707588,0.0841099269044665,0.117899624409112,0.230552079951155,0.16251267516051,0.069315721764449,0.149988226798912,0.143986964993298,0.115543214998553,0.0702304452929873,0.0918689401656415,0.103319718426264,0.140984759588543,0.0219963109339069,0.146062347489251,0.32532515856711,0.359635181990349,0.185680815176137,0.111210058753941,1.05849736760168,0.72786751520044,0.513375657572381,0.467803718742385,0.33160364868181,0.400034617252973,0.313805876408713,0.652748835467984,0.327548985111985,0.411957399530209,0.387581456922736,0.449710433664412,1.09059428974726,1.31479031993298,1.89295360470569,1.19981346886531,1.70475175724322,1.66394148152327,1.09794666741622,0.495509117291953,0.44533103531463,0.783947792590153,1.15319247850131,0.496716920390935,0.752283307729661,0.469309249896937,0.507927818363698,0.140813438500375,0.121622009121127,0.140325404824837,0.123378932985621,0.0965259375105865,0.138688861563343,0.147839415658964,0.180612172388156,0.11852376771051,0.119653292952035,0.30531923411394,0.666683985373524,0.694153533286318,0.658445788822028,0.702894564877801,0.555405592559895,0.951266511980214,0.929408158329432,0.642907987595076,0.886873765216956,0.814950212424774,0.696488020002483,0.415175361813553,0.543927811582916,0.302641078407623,0.636514619408002,0.690351553534657,0.236376946062069,0.161300797327236,0.159843266277049,0.176624751968548,0.150002329853698,0.310330588007602,0.538762610483968,0.563306589258092,0.565725690808664,0.640811208314104,0.474910927087569,0.612644852201008,0.428195873937461,0.486859387948099,0.415889100380174,0.385009936815782,0.458708342081109,0.527192681542276,0.550880838351166,0.592459708672719,0.492445992375937,0.444292077448119,0.263254046217669,0.0879288939977279,0.153917767684018,0.0847408499573006,0.100311899205771,0.198895798291139,0.222512797757155,0.0839555889047659,0.150811047444649,0.124221276840685,0.537591885537837,0.739851383577747,0.706621678052056,0.414225396817512,0.323693398515842,0.376773210567399,0.20834194739219,0.206643946044062,0.146989748152439,0.161820405213987,0.457961990007328,0.263534190877415,0.538070759226665,0.951356990286964,0.642648449410641,0.211922367534765,0.107337254620899,0.593794879145543,0.205039124223642,0.16551387934551,0.133900073277783,0.329697332017235,0.338290636557484,0.575387034778304,0.471784818461702,0.924718792233062,0.985507291951119,0.391312839383429,0.467831509497709,0.429666594541799,0.5355494679932,0.535436727802983,0.541631528865938,0.456403089083754,0.471689942492542,0.428673815821844,0.376483898707666,0.397676811900903,0.183523315475754,0.18469419075733,0.0667392683408495,0.0183778117053453,0.13497058641073,0.0303386442894559,0.152964152494472,0,0.134112789270956,0.319679262140018,0.668371033456298,0.680589910707565,0.723457853130683,0.796801736990156,1.03182501076552,0.959765510233786,0.842818613921292,0.232187599605308,0.241101538994934,0.557578125600817,0.288161965735069,0.468040373541195,0.651193179760266,0.400037718358018,0.016609979570136,0.142236004981061,0.187939463849632,0.200782304471443,0.208318800924452,0.242355831589152,0.242512785812527,0.141013263231244,0.361381080254941,1.1462902552478,1.39719482978141,0.819370674231238,0.747468775621267,0.446513103008862,0.418030886677906,0.533176571684364,0.428489926556571,0.417662775550987,0.3277295040554,0.523106469021371,0.518839306796542,0.315544908136827,0.142241340762586,0.138352229174361,0.173432658145415,0.0781038671700885,0.123394153044731,0.0602007743621941,0.147103140841741,0.049163724944296,0.102914612713749,0.136304818549103,0.116229573325382,0.098756718353322,0.0955698730817078,0.0454204907370229,0.179803419114057,0.101819168781941,0.384347129318757,0.203626984522604,0.246641475469978,0.142807599275142,0.173395010878545,0.109892685022203,0.134634413232779,0.109213078826045,0.0921817629210907,0.106063467599074,0.15070155364307,0.146089212927185,0.150030724300578,0.136664918772801,0.10993780493982,0.11203015078918,0.0889793802053872,0.144446678100198,0.479056973915919,0.386873605118778,1.31446984881864,0.786571957944033,0.570465839973946,0.729322721517075,0.752690904005931,0.513942181936299,0.647928969350142,0.721354231639003,0.674460726971708,0.733353418545642,0.501174395090308
"3",0.00690724745212134,0.0566197005140503,0.071809134075341,0.0522371995354638,0.11423522140379,1.9886587633759,1.94474726137237,0.91063470058704,0.168849027808656,0.04629920170977,0.190733801213145,0.0765061202598672,0.0658340230923919,0.10356014697589,0.210339917652438,0.0823851352921889,0.100738572598562,0.0362186361963329,0.0653906999229422,0,0.03850517186527,0.0551963679799696,0.0880051985020427,0,0.0729057908607033,0.0166710296619099,0.0581897128187042,0.108040764343941,0.0592372943613028,0.0497840410940547,0,0,0.0902705051169911,0.0261495360261216,0.0495559245020701,0.112165841258931,0,0.0675827162496776,0.0326528483382586,0,0.0714385374371152,0.0305289422997333,0.0060916351161957,0.0783437861262172,0.0664572585993794,0,0.0683186091947409,0,0.0731811336354733,0.150742341709842,0.0719412769113043,0.029374855016291,0.335827237737395,1.56150698896548,0.10428695790155,0,0.0312805006198999,0.047229956192791,0.0336279890316177,0.0996421117675112,0.0904895091266745,0.138666467846097,0.0756049436167284,0.099936903872523,0,0.0333650218205067,0.143911329720727,0.0505821754605301,0.0551212903435569,0.0569972995350685,0.0284427133195499,0,0.064549545174596,0.0787309198196602,0.0646764535882145,0.0586584594594743,0.0335205885798217,0.0499191387988037,0.119320370664189,0,0.0413420860326193,0,0.0335414071987433,0.0896426322548685,0.0220881608808902,0.0912950788691892,0.0819534495946262,0.132704156091174,0.00623864527439576,0.0604459530772056,0.123327875218497,0.115653603649886,0.0147067642244666,0.0241608099787169,0.0918309155654052,0.00204535293355786,0.182617859200835,0.119667607600938,0.0637297802154677,0.0423439900712587,0.251913462682481,1.63462665690912,0.152795061961073,0,0.0622041881117208,0.0191009240105531,0.0665110959219787,0.0180663788611967,0.117225302224011,0.0533123778180879,0,0.0587174285717256,0.00362486661414996,0.0116447009092514,0.127420683936739,0.0780519743274136,0.103812839309641,0.0565269598403011,0,0.146448503569103,0.080376492142161,0,0.0552452338806384,0.0122061259523043,0.00296795406727025,0.0878473869116949,0,0.0500475026817737,0.0654135774699645,0,0.0708342807582771,0.119832137039363,0.0581520466178885,0,0.0525858297159732,0,0.0784931743249185,0.0560171777684642,0.00773633483980145,0.038000442376152,0.0884041109969515,0.0714050453842353,0.0611959544438934,0,0.000140602560431594,0.0337972060198649,0.00927966696169645,0.057161398481952,0.22374808130594,1.51991581995702,0.0800262419921882,0.0255910369097433,7.21946525739439e-05,0.109817541545184,0,0.0698428729921333,0.105720053754663,0.0201752249627643,0.0177843569595096,0,0.154900667430897,0,0,0.127384538947152,0.0977126130544607,0.0812473614638868,0.0896829306820146,0.064303275328621,0.0671576970067938,0.116671815285046,0.148284763060751,0.0989681022791878,0.0147462528202462,0,0.0614393782604793,0.126669463287047,0.0979262832409368,0,0.154198794227947,0.101872596412409,0.0338113578700993,0.171593344312848,0.00453342126897118,0.0923004605818549,0.176128650761563,0.0333947906777702,0.19813205599924,0.157628306397064,0.127171071215591,0.00429732805855215,0.121747279396162,0.134320947653805,0,0.0594534023421392,0.0342173083233369,0.0358781305859525,0.239776786519226,1.47024242978158,0,0.0198997619585724,0,0.0525951213935,0.0842076024722141,0.0753309158784822,0.100744145919913,0.00248770735929431,0.0484814909388532,0.0563684924685951,0,0.0263839377647623,0.00978869076009636,0.0542551187238945,0.0723884264583278,0.0625711428380492,0.00586462753546487,0.0566497541241364,0.0566667334026959,0.0156283114061832,0.0822380774350578,0.0315788813249212,0.127519220750817,0,0.228360684132187,0.0672529726640115,0.166753072134724,0.0796245615786457,0.216692582239249,0.088598014216671,0.110504781094914,0.0653193981479996,0.174876765791162,0.096389521895865,0.142955914947512,0.0611650796978998,0.0976919544686508,0.10353277994354,0.100437005585628,0.0641223168789073,0.205645098495602,0.142391759480156,0.0635554394950931,0.0400739548145814,0.0647002236886018,0.119741685618946,0.259980858981817,1.71896715176224,0.0377674071429726,0.0638238678703599,0.0833299100626835,0.158759663517871,0.148259730791036,0,0,0,0.121035539736339,0.138393494207999,0.0285564063872964,0.117958735285773,0.145465195245257,0.18042868487058,0.084904580835894,0.0301547431790136,0.330095942657429,0.202529144168927,0.271810830833473,0.183838724813475,0.170614986021109,0.19908790834787,0.0969892404982624,0.201218224701021,0.102768371807696,0.186808499662234,0.120609573282974,0.132254838626667,0.172016417047667,0.226190144120645,0.249960650215857,0.126322811801428,0.0829235484304111,0.0603860956444318,0.114399783347266,0.187029928773978,0.105240703372253,0.130187785677129,0.100929587729598,0.134875452517024,0.0812922629700278,0.15206029364063,0.179810295055675,0.0264053907038915,0.148995074840748,0.0995402025684712,0.0447629636489569,1.85323680675265,2.07895148745807,1.94353886452462,0.97777850248816,0.0579233823924914,0.0732125477579622,0.0983743019106711,0.0537483538779732,0.0783311586099786,0.128904326680088,0.110728035110775,1.060229766593,0.769728415097904,0.556957292738126,0.378155021136514,0.0877945208998564,0.0933125598403651,0.0547347021433096,0.0543719530098395,0.14802309177696,0.0855259419305502,0,0.00169324133338894,0.0575955097066609,0.229718512379033,0.0192397119838431,0.040583366037395,0.00983052985636904,0.144367169010454,0.0279479632779404,0.192032447929249,0.183573571007511,0.180520030945533,0.229886020349149,0.0727317675201144,0.123865318123487,0.0425242970819791,0.11777496488473,0.0241954699610012,0.139202222499128,0.0412988453353373,0.145080915667699,0.128142853293856,0.0198847947701391,0.15119644402526,0.00180805635314237,0.101858493354948,0.0376109418018863,1.93291688606849,1.87023039354635,1.8638343819088,1.29825210645667,0.019043229610714,0.228004633777286,0.155675859538503,0,0.196613129213058,0.275802651843619,0.0588886729221545,0.290058420079124,0.250748481534526,0.0788990892307111,0.180194219896818,0.167134759881163,0.0886358761239864,0.125815151671588,0,0.0978100643229082,0.0736737100780352,0.0957805045563603,0.0541816146850254,0.0773604309655864,0.0194197978215262,0.0569650125268963,0,0.0603970535195043,0.100229513461862,0.0227075205625295,0.101751481920817,0.012689723945279,0.0836457500896207,0.0267348595673597,0.0686964867674643,0,0.0227300678244283,0.0566022870428129,0.0107086240267041,0.12683023981462,0.0892621693024649,0,0.0504565734702864,0,0.111679440900203,0.0630513596536801,0.0500286260981796,0.0855944374493542,1.75856562558257,1.37288923101243,0.0255845992482302,0.109473028576641,0.0514212363070518,0.0250625855417032,0.0719731853320378,0.0364940688724395,0,0.0612176756835122,0.0779205809738451,0.00397226165422616,0.0860523814462043,0.155359409756977,0.140433975588907,0.0766759551288087,0.176519887740229,0.0349498523139555,0.0643019250947124,0.0886750204486039,0.0238179143138502,0.0842674904195886,0,0.0758096939145597,0.116074125135397,0.045664597364032,0.00318317137913132,0.0835967616771447,0.0420103445586591,0.0993636648929582,0.0963065015142535,0.10203932232918,0.104836242236323,0.050934463478479,0.0165651532742252,0.0213290188161016,0.0263415952272505,0.0614347918441361,0.11691580121514,0.100247216245292,0.0651556518808407,0.0751089986506647,0.0273998751477646,0.00231923451474342,0,0,0.0248199277663081,0,1.71494806105779,0.0990513806803112,0,0.0879159768996928,0.0248022314348399,0.0998993658375876,0.0202840111660004,0,0.0822803672703386,0.126788112798908,0.114665885683917,0.0417986661950204,0.0267777574153165,0.0199120561106348,0.0401647726232179,0.0401932008447366,0.112851296216318,0.164275350686968,0.0395622185226227,0.0141228462868756,0.0824079385557191,0.0413134496559318,0.136277111250309,0.0842670730776442,0.0777903697271147,0.0707439191096873,0.00359495651956968,0.0341358636918526,0.0792041997233835,0.0773728846526615,0.085638497431546,0.027476806878926,0.0622729876042474,0.0135688990810884,0.0186900267388093,0.0632639311014251,0.0137106169855712,0.0755979346371218,0.141663220542174,0.0292910254365076,0.0325707381160088,0.0701231226234165,0.0197605892602402,0.0676692720882331,0.0648541066201214,0.0874068704241612,0.119159961078388,0.100882757300222,1.60373118516931,0.0341011454237383,0.165206435570238,0.00298511056704148,0.00741352993852712,0.00180027044917119,0.0679765516540092,0.110415662021871,0.0135780002672119,0.0814359267572025,0.0148066132547909,0,0.0973998033533873,0.0779525374765431,0.158371050213171,0.0314297016032989,0.0180147435920063,0.0341263572276246,0.0887302183917848,0.124345160465626,0.0878376987386755,0.0563582787641704,0.125296934071327,0.0770318481577161,0.0368546441711468,0.0262786772702528,0,0.111467644354677,0.0820467156462018,0.00217024080985175,0.106988076249626,0.00987389520411928,0.214087599653043,0.206472674680109,0.27111964093911,0.0658308700313806,0.0953746522138769,0.0470700445368345,0.0807631376821499,0.0399079858264796,0.0811386008129144,0.0628182596574911,0.130470448898286,0.0858231684702991,0,0.0858296341914357,0.0711814512465175,0.0864683257367558,1.86116082215249,0.97926006121316,0.0758015570007896,0.0314306316629052,0,0.0234457338949788,0.193420330258978,0.133842239187117,0.0451252268809478,0.0135701274442563,0.149703428438093,0.238506924209267,0.131641067777835,0.341119455698832,0.16106552907177,0.376504976368697,0.193987939736891,0.154959278333453,0.547146168743772,0.216232130458942,0.1868760036017,0.0967749895523653,0.122064699932616,0.182778275640009,0.110328451062169,0.0894897388723357,0.117086115941872,0,0.0988211254187383,0.0558121784943186,0.123348722472453,0.17810099457472,0.000735932467007872,0.0620793611122192,0.133671720451922,0.0105455352943932,0.366950470488333,0.177004646121586,0.204668123671694,0.16809078290993,0.0969567334549442,0.0744455958479947,0.107711485282671,0.0678464309451585,0.0638503229759348,0.0610721173829806,0.0108198610418798,0.0327817167992929,1.67726790558739,1.90699693029782,1.92012602135304,0.587670639628676,0,0.156329812584949,0.179630851144681,0.0929190493100848,0.0845030477805284,2.12557878058633,0.109733506091186,0.0245192702350389,0.414119290654812,0.179521895636744,0.0226652109181608,0.220405023404679,0.0950864583088802,0.124043790223557,0.0680536200885897,0.0883141463280901,0.0353777202383132,0.0439004714410022,0.0683048465079455,0.00271306153656632,0.0588129230473696,0,0.214585096087036,0.142172107044459,0.0586360393979493,0.0486329874242141,0.088348965687489,0.167699255871716,0.170806368938503,0.112714127009642,0.160708249631513,0.145569061391465,0.377493844397458,0.289154668409347,0.310564162002084,0.130456189131231,0.0765231061073633,0.0506864020920502,0,0.0618330680517886,0.0322288552282645,0.0661886864354152,0,0.0972726990885531,1.73030135819924,1.90882548167605,1.29319799341798,0,0,0.094322411235837,0,0.0889980691206946,0,0.0762309020304605,0.0671415418911644,0.145678596865971,1.85092145998728,0.370394600228719,0.372701870228709,0.133869095569621,0.146908692393745,0.107950960922854,0.137511467951248,0.16189279406441,0.255248231535779,0.161979528386711,0.170057892809836,0.373901854733261,0.0218640613557014,0,0.0955078608179466,0.0146773736066344,0.0849610837992369,0.0491960939903956,0.0744588502926822,0.0967662823031801,0.123041131485798,0.337220986637215,0.0936328382070818,0.276572871755847,0.143986150005753,0.15459439208065,0.0669509499461026,0.0365620970818228,0,0.0617484470186077,0.0265886432821479
"4",0.449263295116682,0.218019550991481,0.218467212818014,0.16045604629766,0.0943424497748502,0.10744388370207,0.148900240779378,0.151443946345193,0.139963251122133,0.0625289171920246,0.0429818642107511,0.242144782443788,0.0956710009177597,0.221623716765437,0.477187406143327,0.107920572423292,0.0406401989272928,0.0574193456319142,0.224036019571184,0.829559634294983,2.33012632331221,0.261046654167119,0.329187243538137,0.193735846109305,0.241057866008026,0.123961848408909,0.110107644997165,0.0854873513727033,0.373158477548014,0.415215698157534,0.101563487679551,0.0978352781579019,0.0346315645975687,0.235271197477726,0.323079336655405,1.5059817684298,3.90624514357628,3.23107514578556,0.458730904245527,0.927178540438037,2.42622609541863,1.79038655088388,0.24081590428093,0.28275532125045,0.349510098797491,2.56648263690947,2.36990922564642,0.631012887273057,0.178218852544137,0.438261628325575,0.126462789737133,0.364579022429622,0.525768072069077,0.172196816731445,0.0860986338435592,0.126309594297056,0.107646463008281,0.149221025001301,0.209007732901233,0.10841885634796,0.0672123855280413,0.117317056327937,0.0303697895810447,0.0386912122703581,0.0178579067811813,0.0410516863205227,0.0112451755472934,0.0467295186238278,0.357414982464866,2.77355171161177,0.90391584757031,0.383366936196422,0.236494310570839,0.0283305756124506,0.0579119946866799,0.0682608370489019,0.0482325854303118,0.0107518079561295,0.112850777623533,0.0698405673500535,0.100609855973458,0.127536405726759,0.143728792843959,0.0963807936283415,0.307000079306625,1.03931127439029,2.8458733446308,1.763078055296,0.586653963926294,0.476874557865651,0.272831013342732,0.871740615502465,0.67575143202954,3.21900521415313,2.21536365309907,0.686128106124744,1.03560401357638,0.482992604699493,0.242208421523448,0.12729295363075,0.0433878263335384,0.0411812770638225,0.09062350791016,0.103413845775586,0.117283472342101,0.027230895747329,0.156539808804094,0.147726269770182,0.195964990279264,0.122876338731355,0.143521689335998,0.244451190934477,0.105730382148827,0.117716100580485,0.126587833760938,0.303648980282797,3.16456818410639,0.382771976531642,2.24633787757794,0.496362489644451,0.160159114530144,0.258496882616292,0.17192462488264,0.640703724641096,0.555135813257458,0.226527096411163,0.141927666050232,0.0972446658872934,0.274158587369095,0.129863104201473,0.148885634772601,0.234817809959073,0.214459115989834,0.169597054976696,0.610085812808985,2.59792789647433,0.37607195362196,0.270242082429202,0.321178195076854,0.842933857157213,0.809280571817692,0.163778196144799,0.251354509250859,0.234704162824792,0.103028259941171,0.0874430096174522,0,0.0275780357473257,0.109410689160592,0.0714682998634421,0.0413490614925843,0.119950181663784,0.00780777607576638,0.184267672399217,0.0782173972133955,0,0.636916585332887,0.0607726076752109,0.14299751233338,0.460497074165203,0.640877016056253,0.638559880082313,0.390371148008738,0.0610058755939476,0.311954840462469,2.81191162344103,2.30477920809581,0.59247881537488,0.369489354802907,0.584981808624341,0.7493427028402,0.0385180038703573,0.132215470128553,0.24421642461942,0.131507616475387,0.478535572594107,0.573561148658457,1.96714893038259,0.0934700959724453,0.0822915102004859,0.226168437435075,1.47100463120745,2.12294384504342,0.493150534149614,0.459449644213688,0.978456919333846,0.899862306614129,2.06350169955652,1.30516460289527,1.18671421525532,1.70534796532654,0.690276689057674,0.344474556895294,0.0842783852598128,0,0.0779637531660274,0.0809445274978719,0.167222054349232,0.0952080733039223,0,0.0601763610955054,0.1255894595164,0.0674388328430205,0.14225056445129,0.0280559434393323,0.140758170987093,0.109375033134154,0.129240414435432,0.0581132281548831,0.0738751826950305,2.30561656016612,2.49033415204072,0.624993446664259,0.349167330150411,0.948961035127366,0.283162204080623,0.292047135572259,0.277569286472494,0.707386713326659,0.255582564054006,0.203035001108462,0.190255660882779,0.501235364989195,0.0129379297267629,0.0935057187508669,0.185114246530079,0.0465641218700555,0.134914135374101,0.333797046216704,1.97414535983099,2.4254763856679,1.72851798314593,1.06808080159147,0.344885241016897,0.316962158173862,0.229525137767402,0.75643314097249,2.09912764779716,1.83985142462193,0.740634936992719,0.638963356531002,0.44188158542621,0.0268096200736633,0.0571292624221493,0.118650888989589,0.064283159070133,0.128526236739521,0.131230130592551,0.0707182395490861,0.0553910185767478,0.232495936832664,0.151206569201318,0.0622121326029352,0.122267427753542,0,0.0482053743676252,0.0963407896067321,0.0873740528318471,0.0618073783876563,0,0.229760181319328,2.31665361814122,0.784151751995066,0.343759236530902,0.44990294701022,0.221072538221116,0.131448652872847,0.603242411587523,0.0899989879884897,0.694419167063319,0.097047628895271,0.0395025990252652,0.114641230761306,0.226645041493976,0.156973770287208,0.956904755222057,0.58970224644692,0.54456475102574,0.132539038379028,1.95367574695694,0.667350952740892,0.427711411067686,1.5599540975875,1.87487564985412,0.333513260724558,0.339336997260091,0.293578557872861,0.25064502753859,0.461147005581896,0.573281308033011,0.413144589225216,0.35834125800116,0.132403547632099,0.00797250066039795,0.186033205106114,0.0527402603225915,0.0637874898453167,0.102556909947184,0.032981243899629,0.336347527355572,0.0717673726900561,0.0696876315200911,0.624727315047033,0.0909332189014774,0.0878797796344623,0.0893144541285388,0.0836948860083737,0.00903378561548979,0.129497890440815,0.28605905655104,2.58150333383821,0.375759237200005,0.171350969460456,0.262400015300818,0.192166684478676,0.290036439819199,0.476630581020838,0.0792092837724897,0.113716188207592,0.0397495988368298,0.120882729927598,0.121009398432697,0.17580682817093,0.344946584543831,1.06182696926128,0.762019070570231,0.287195785804189,0.686256992065499,0.435201131580743,2.75818401401138,0.504962469004135,0.243216314423837,0.234688838566949,0.498063129438472,2.60995787579297,1.06169241901826,0.252871905961008,0.0236981174412584,0.10284478657897,0.142179049330922,0.136992266608437,0.0397149791748989,0.0581322036138759,0.158530657501013,0.162201382522767,0.142317173245248,0.36569462391753,0.139519881172742,0.273862627504783,0.408808956180124,0.170306562015065,0.0213307896608527,0.0381003627198478,0.0962985751420189,0.0489809279030687,0.0983715228140769,0.0634958372242563,2.76321345852021,0.608199858638412,0.237908072790055,0.753104960039071,0.69438816975444,0.152802791496464,0.107113127547193,0.0900898748787307,0.109074219080304,0.1461006734358,0.0703495473550057,0.0510494398526725,0.407629990483629,0.701867790662812,0.212659528283816,0.167170725853233,1.19334528759148,0.74340279154642,0.658488884242826,0.783062616690978,2.07088233619315,1.6136807911473,1.62080531467723,0.994877036337233,0.456340530157029,0.561334912770879,0.537954087826308,0.951677359910972,0.523975011765196,0.0712457677941248,0.0923101135811657,0.0960751568231165,0.00177683947389851,0.0671604708863832,0.0389456492456145,0.0415926064103349,0.0607522988644108,0.114735679857264,0.0978384086487834,0.0594904958979113,0.0970583336315271,0.11372076203684,0.115646216308951,0.0854298557222906,0.0600573037034887,0.116558925816024,0.175179431269373,0,0.484572035473662,1.93345787262376,0.839140930636754,0.28200671858407,0.240313038852074,0.160250218951514,0.110411428074585,0.0634078859467487,0.0299133951204562,0.140938667957176,0.35838877178518,0.326983601189253,0.053550521186672,0.0932570647768148,0.502479448475344,0.525357166899276,0.648728460996195,0.665232406129558,0.44599079567664,2.34174115591743,1.16807590401735,0.873903444939367,0.574899934259872,0.196485976719545,0.308204408004285,0.289965308175734,0.263112467405287,0.332482343645636,0.300552568261324,0.194500620856745,0.0414867932977759,0.0813829641322832,0.0935027050466143,0.0266699903649897,0.010915626034224,0.184923129501541,0.0736858106682297,2.41415877254883,0.392174782667251,0.0148848004977421,0.171006079987836,0.054118925284234,0.173617027885875,0.146064702226393,0.0330927828990466,0.0928894797514854,0.152650503835458,0.0463140720794195,0.0578737924396951,2.19971330147999,1.08447177850274,0.425020898701064,0.633577189062983,0.10134778912283,0.162654899925548,0.0800498452110801,0.147767121772548,0.0661321586248281,0.072299318780639,0.0993439164425489,0.0249445282414963,0.144122559566332,0.377126174899749,0.40917576902451,0.232114664855929,1.80428623678978,2.70704940319487,1.96543359622253,1.48774063624542,0.915863245441555,0.310975196384456,0.386861424825443,0.959971966778014,0.235328645241792,0.671492513545842,0.771748114194976,0.108608022682714,0.10515868291555,0.136342517080069,0.127217177601401,0.00739389832435842,0.143505480968504,0.0420671824240776,0.03903885759127,0.00721415206180145,0.104618675768649,0.122257179484246,0.242112155703841,0.0519906531850732,0.0928665295987937,0.168344078941773,0.147362180651871,0.0872508979222695,0.575563330603105,0.117680563922585,0.0844572697158435,2.45754350522117,0.246912929699042,0.282442991290559,0.269856922593776,0.228031648964097,0.456894665330563,0.46406045797481,0.124649926265292,0.0816340750279036,0.0526121041793502,0.171208625901923,0.184601092974494,0.252760718146946,0.241482661569933,0.210082731320242,0.135911041510996,0.147581869640475,0.180680985153151,0.658187587447839,0.733699022866071,2.41550399599926,1.40027376985061,1.64424645378816,0.90596840513398,0.13355483048001,1.21359391750993,1.06527924275867,0.82021381067269,0.250583316912455,0.818362724752708,0.161498831013743,0.0471929351405459,0.0920706295164267,0,0,0.109318281849363,0,0.308111747053906,0.144413026622634,0.123458445172334,0.0357042584824102,0.168899095101724,0.0145104076762545,0.132023119316551,0.117768948685916,0.151279382029503,0.0120778759040217,0.133908939155943,0.505629904173622,0.699812948228532,1.38672952114889,0.176497378794921,0.102805980751482,0.0448183812734688,0.0422771515145412,0.0942170093345955,0.156769235889303,0.00128882146263225,0.0310988325401409,0.11756581659605,0.196598670458174,0.127045589536893,0.0850917619009552,0.186667596537711,0.384473538320253,0.54598365449459,0.118884306143696,0.109486149180784,0.259227261659296,0.360251323012078,0.320162599283375,0.946184630451159,1.44277773021818,0.257990508419236,0.627395867562957,0.689894694930894,0.702569863699507,0.331491755284781,0.000559426684791947,0.173384190183484,0.0900081538025347,0.124363780745925,0.0832722451863797,0.067459940271732,0.0787246214490291,0.193204334631155,0.197129133276134,0.0555332894545248,0.083438160660497,0.112399484462331,0.0126573318957776,0.0323446567345443,0.118568810677024,0.0326254313312676,0.336413649865306,1.65255316784102,0.563236007007693,0.328483772618198,0.0723411374012565,0.0719457754392956,0.0588416289378107,0.030840778326373,0.0563149278205555,0.0204501417651142,0.115137802467428,0.0610319091250129,0.030015132909717,0.00410137199699118,0.0276986647770445,0.215665655425806,0.116802734953459,0.111529948151104,0.110707300623703,0.337414423964796,0.1300067839645,0.252822143860103,0.21469427865098,0.134030255115696,0.473673277609331,0.249066329297641,0.0942419030106731,0.0237000270012315,0.147693571106988,0.110716513257511,0.059520886120555,0.121872961517326,0.170696559528256,0.753439070192209,0.257218917006853,0.102591493059997,0.0337653223422882,0.0539412167168539,0.0559236854097974,0.1649270018066,0.18153627770065,0.203983536228277,0.0348744261738401,0.0460721685913835,0.127144678448632,0.0561799745243725,0,1.80012910253135,0.572163572041516,0.203374843724558,0.636653020027493,0.646112261398525,0.274267317994632,0.156428020387397,0.117943533889425,0,0.140515474994037,0.0943577136374987,0.056986285336403,0.145507614502795,0.083399514115475,0.108595556661912,0.131402571911418,0.220238291188208,0.0145453770145582,0.113477095887692,0.266840396974135,0.701233692973044,0.187358256518002,0.0376448535826058,0.15054785720604,0.251565637475949,0.119661774813059,0.167703229677381,0.893575292147673,3.0351940150141,2.01081962978439
"5",0.329842117431285,0.38799600942894,0.260863878208097,0.337705994173367,0.232229941817865,0.253465797421573,0.187359492929033,0.175392741538556,0.312217189652563,0.176987677253173,0.295006310989156,0.306748412642694,0.309450029723094,0.188493775161797,0.21787191099287,0.373734941696412,0.234973744413826,0.229965945460617,0.485780434793034,0.733678797310039,0.482558143679605,0.403175140706373,0.35641083745589,0.214979444823502,0.299392934873737,0.151225673567961,0.308358035929038,0.237567924792499,0.232518219239112,0.281910370032243,0.538741195182521,0.696299576335257,0.465299368870883,0.349499517713584,0.295838094883249,0.184805622660466,0.285503087744293,0.294472912716778,0.214598810634863,0.256035842839913,0.22413300830709,0.265058225601054,0.224688788859584,0.258900387691404,0.361779737139004,0.471385410820687,0.61379633938991,0.701572749172891,0.832806229826595,0.296311200118343,0.340065458664374,0.271677839911231,0.388774586483334,0.389616901491198,0.386344634576799,0.275121395570996,0.29100611382259,0.267319473570237,0.385271995727052,0.251186125305118,0.358159494860576,0.340022111939479,0.329590029158316,0.400631360349705,0.421722481889929,0.739273301508311,1.2498119067851,0.556395550834412,0.716192683576148,0.769027553793663,0.56050799324355,0.626964197504807,0.310740850698505,0.36873482473873,0.271318541091528,0.383969106483117,0.554571205090626,0.290660931522048,0.222200205562115,0.153580609457568,0.384174883645376,0.202233894303135,0.336463713420601,0.460186459005245,0.395641537998113,0.439497961065883,0.360187164856986,0.282088892051752,0.405784880093095,0.21302000550339,0.239918985250843,0.258377021365305,0.201986587783428,0.314139277613646,0.316857346981176,0.354798838542419,0.2953861808462,0.22880235117968,0.215597568221793,0.210397696114439,0.204915043331311,0.0921637672448736,0.116552328456175,0.154565342759863,0.231945216214174,0.237030774799919,0.176943342368356,0.314465013831425,0.198404350189401,0.356205094515902,0.611433381587131,0.335137914197039,0.296801042216741,0.214390291453347,0.357718648172761,0.470232729962453,0.340949282661666,0.33034845979498,0.288200577892838,0.389720826847636,0.411022232301424,0.34773315809038,0.335158038746263,0.318360340775858,0.157221677415769,0.127510660078621,0.197422044154692,0.136027858220533,0.330067984041489,0.73960914087068,0.850017381782093,0.775565706624243,0.296614589143497,0.704174701529902,0.422182662291648,0.278221807099846,0.784862876632799,0.667855601497769,0.307301488518628,0.537197130384776,0.214597731539083,0.359787621796211,0.316213989882216,0.231118156236737,0.228656282312362,0.280923118162526,0.265288785509196,0.267881887379084,0.245904673376202,0.26248185876289,0.259447274132897,0.307852889473747,0.164005848256439,0.323052309702544,0.246714357577496,0.330371566542416,0.199737251089283,0.264197094389228,0.253889028005645,0.411013481268469,0.587925623174411,0.395358697145381,0.416740218292938,0.413490596675672,0.455666108085021,0.536782776904302,0.328307447521404,0.256036754333086,0.355816969084179,0.418019415506129,0.443561124188239,0.456623101351846,0.163826624756392,0.338688519424047,0.273841099841179,0.24616035756272,0.385957450484796,0.257158642712119,0.438307636778407,0.625071543978598,0.520187048739151,0.43826012760946,0.641923186119913,0.971510726235884,0.849739639900151,0.867454584319561,0.905738452736385,0.407642429211709,0.464676541620475,0.428525315933314,0.369611568472434,0.216322437887987,0.246038857147321,0.370612330064986,0.25324582136846,0.315464491280988,0.225181944223095,0.317950707490161,0.378730787978226,0.50828372471185,0.714168770875752,0.67635182055943,0.691303765722201,0.754602485343063,0.326390209490541,0.279310607858608,0.234030842597457,0.289586218558,0.243677739752293,0.331098250122093,0.408971242422887,0.742901436379134,0.70616033987072,0.432196488828171,0.464263515031128,0.328724953898151,0.454063050664034,0.137036983810728,0.305714600090887,0.330411938639324,0.328583130631044,0.401242667472666,0.325908677415792,0.416709958749607,0.422211790177451,0.391163622948787,0.436065658300394,0.439642879263863,0.489755171548231,0.841978394983052,0.73484725532725,0.497690034858353,0.69006963947569,0.538330677539329,0.696732573399421,0.500362684597844,0.485360203565548,0.295296506849836,0.291340705388425,0.370245187155082,0.303404736874113,0.308590751766835,0.255686556226182,0.342271794152106,0.254120324534526,0.232083644614849,0.25469649068106,0.216366507014182,0.160715210414088,0.258272283512119,0.225390131247208,0.260793778815378,0.289915239253552,0.269051333208643,0.269654574571074,0.675650097007292,0.585284845661887,0.568838433630467,0.52641326575001,0.570776683594221,0.480794898635053,0.417856309385011,0.448294441068643,0.531897213154553,0.511589723546071,0.628511819815541,0.485532299662358,0.461943048129544,0.386606694137355,0.375937674505239,0.450008141582885,0.583369188887647,0.370839920587311,0.382280550802185,0.434073446478783,0.645022704256356,0.399148554643905,0.379188105046673,0.736201955444392,0.571056395873497,0.891141707211857,0.459538422505381,0.415769242352665,0.377085300367145,0.366544153502623,0.273130489511932,0.308051503734628,0.184996075135198,0.273332676460806,0.14406157310405,0.157291435174788,0.230285815593848,0.205803861778679,0.174008254871211,0.220967217029192,0.203273833441708,0.199726420394585,0.229283719997555,0.264866013105159,0.330487636067174,0.258688032667566,0.245465572217606,0.397755182720047,0.376027030317151,0.354157690727944,0.288572337610656,0.352266985767582,0.340105572940221,0.364083734456378,0.272323348040533,1.00991386376183,0.831752546118638,0.310869466896821,0.418294710035761,0.430473614319804,0.451155339977735,0.407463906479602,0.870591928167043,0.624650555665756,0.554522340640059,0.505751755540018,0.289114848683797,0.243636187398805,0.402656788154647,0.395371777342958,0.773398533654119,0.499844634467219,0.438084136093352,0.342390166424283,0.399229874745763,0.440452685908648,0.405076762591291,0.269635225783951,0.243490569603452,0.36538183831601,0.173807426934129,0.184746208765641,0.225159955592293,0.162650393248485,0.187498067776724,0.260599298390639,0.15587221369095,0.207166616545689,0.128267530509431,0.204205747384993,0.2819014066225,0.227190764434131,0.306957821191942,0.200325380107711,0.316744118061494,0.203649233614973,0.227932128888029,0.316294926416006,0.334401200489998,0.432709695059068,0.346028909180507,0.275910693190279,0.303923076545918,0.251175973352645,0.276242687064139,0.367752924749697,0.376714444320824,0.299699180768207,0.435785014465661,0.403637935341716,0.437790549196004,0.387486508595804,0.46037358634489,0.352868628469241,0.496074249127247,0.606321930214933,0.624516104108622,0.577488154996796,0.441903318573389,0.624751760481678,0.550840743601392,0.493539774586739,0.486354183179491,0.354931909380836,0.412660675688196,0.48420195814462,0.421819437221896,0.410205598668054,0.508815678372423,0.357353241696282,0.247304085577047,0.256324860646471,0.211906681513664,0.308683629492601,0.187045627217049,0.287545108047862,0.354454701584584,0.172481834896916,0.174289969117378,0.199192987329514,0.274655361384535,0.179579485334929,0.199802910778741,0.26851812013498,0.177000076857153,0.338116717130397,0.418060662006263,0.636790975213572,0.536620804477007,0.290634566719782,0.24459528781941,0.407220991255962,0.447227678582953,0.212188418577188,0.289043869802179,0.262751024954337,0.239948261655025,0.212407213369112,0.284310651356697,0.176531651630834,0.285941451400529,0.230639611302512,0.186518858642171,0.235034006257158,0.256537617244098,0.477735560879888,0.502743387875017,0.351933523290956,0.338659116422766,0.425932452396982,0.427778803917504,0.505354687132882,0.377792967826088,0.438947449513303,0.661580240855634,0.652135732797105,0.384508942775666,0.373869684752702,0.24809887446893,0.278176079631851,0.282852438724274,0.399912716843224,0.219379306316974,0.2996464839646,0.174206676072964,0.22148998422435,0.192813407338989,0.282897304162697,0.226986084222966,0.35062254401691,0.300927221706989,0.190295092110104,0.258714963415117,0.126431416072378,0.236584932203428,0.252410503927354,0.23365862186539,0.258514671965681,0.366701432938803,0.128639247529836,0.208140013884514,0.195785205805576,0.282317121720008,0.532588711082334,0.395840775958092,0.697563062361824,0.38890140841074,0.378510948998861,0.421191656619695,0.9864566152572,0.645945767911809,0.543874440780964,0.684831856129352,0.871246233986469,0.49034736772057,0.794071532111401,0.324039415478802,0.380386592140191,0.492101118569271,0.700106252011767,0.424891030798335,0.327158424683291,0.639304744727991,0.261619136993302,0.190177784695953,0.193186319736645,0.221608156929649,0.160650525993989,0.156954817699798,0.23644466105564,0.273157838755529,0.258185538743868,0.172392993693246,0.171130780022843,0.225148490016149,0.229158067880928,0.279959839092444,0.260687177703715,0.293781390624009,0.277867250873403,0.299096701939464,0.344521209948394,0.326062625495874,0.368508705367338,0.311601306346045,0.454935091769152,0.433311481936322,0.210136357885178,0.307802762791224,0.254393658683786,0.309124069702942,0.383835482097728,0.36702342173306,0.324061377856464,0.279199637632929,0.222159763041988,0.303753873817637,0.37802700130515,0.278354711368058,0.366529441114076,0.412218439699924,0.382456980646573,0.368581603527718,0.490287287890239,0.304708799965429,0.50365141915643,0.530211208656285,0.380349385075868,0.381418693725521,0.415193523421081,0.380584343446351,0.432475224073713,0.319449286809279,0.318340570761358,0.327795154340417,0.177405359275318,0.260955696580447,0.243630130103463,0.180847986487379,0.194588959590396,0.25436411808595,0.308515345627735,0.22340644225702,0.263482167669604,0.345471903877754,0.264324394153145,0.192464068390916,0.230720745110076,0.239147691381005,0.203491250606352,0.348420625575561,0.541314055513576,0.556899678345462,0.35000094186358,0.408345144824812,0.41542081740688,0.299958778598,0.280010270768156,0.348284575564759,0.286025237328968,0.241598125170862,0.298873446146039,0.348524511655986,0.264999109319805,0.539321811324334,0.429663401955408,0.426228929019618,0.525294367901321,0.457812818316228,0.713974833663487,0.486337091924611,0.555894804832168,0.63887184943947,0.6257303635795,0.491941992718813,0.393459513788885,0.35021490126925,0.408124160692306,0.369018679725073,0.441580120093316,0.342683699630994,0.258029481503415,0.323325152925665,0.271552476104452,0.177287834719636,0.163046689510515,0.219204139812415,0.266402583721933,0.2085200810611,0.259211399159301,0.24376680543978,0.278976662698525,0.41481524595235,0.302070741352858,0.283200663958474,0.199741741004215,0.368852026617107,0.35420797114037,0.358264379555082,0.419054093187539,0.32843947502072,0.267688359373381,0.333414945770029,0.31432476649551,0.197896117701482,0.238268775202096,0.293415634801682,0.173101313079619,0.153374568845245,0.306335864429113,0.288900271704961,0.219437536799508,0.320294676038394,0.388013856885637,0.494694105907885,0.330550092216304,0.617524404141857,0.525229842877784,0.410476795567622,0.564004491320748,0.725934344568647,0.492493496500149,0.431159131009965,0.353054348843584,0.324531802966991,0.450160995693733,0.366237980770282,0.254944449099221,0.341037016316741,0.233481797013528,0.274373414252567,0.259221549859814,0.216188102615798,0.163872859397689,0.249364298914931,0.194727870351106,0.248002055962049,0.243009011306318,0.318345766797281,0.377958416717599,0.355114995241478,0.28479613792398,0.224534618595166,0.187052261024107,0.298004951465352,0.162684648128795,0.27553880922897,0.539767870631532,0.457781180401889,0.308050574069464,0.381493327040078,0.357750638726533,0.283584462016548,0.290253855480683,0.200905274127702,0.390726572055703,0.39124848405112,0.213527401959326,0.224389022413093,0.364013961717652,0.508655981602475,0.256227127469302,0.436081836897873,0.259828676299157,0.362908196118218,0.406180495190568,0.437032574737295,0.583595188677568,0.460403168515227,0.429942012144044,0.449473308668976,0.41282446904999,0.32258863157228,0.389129980698294,0.323036046996111,0.346292524132431,0.364182759571288
"6",0.069840832350038,0.0865885544922087,0.129533688783385,0.0340407864581259,0.0891924202464235,0.129815595546833,0.0660317616709081,0.192954395437149,0.104718658935181,0.174353532221035,0.171939071075005,0.152530487811763,0.0891884012745875,0.165605081157977,0.378837205971021,1.53834864780405,1.66835438363543,2.02890880686265,2.22112377266832,1.51627295205347,0.572665848818034,0.367028033159053,0.101461534071125,1.11096975731447,1.46142695341804,1.74951923759536,1.54958500070203,1.63480865602651,1.57939788171034,1.62161743399942,1.6186352685449,1.49056777235322,1.70531877658086,1.68631856610684,1.60177576837063,1.62302657639468,1.76186108413432,2.48333843603172,1.73054503809724,1.64096204146537,1.71581826416928,1.68005889251744,2.05121243836421,2.4533296480901,2.58083033326379,1.6829541546791,1.53740118654247,1.53918708113374,0.316758686242812,0.218558053672049,0.17322880726827,0.0699034252524458,0.0276107632276894,0.00561036871889205,0.147818218821683,0.114817099021149,0,0.0213325365260712,0.114107250787481,0.131814750922493,0.100140718802944,0.117288545863737,0.176976250633002,0.232531524238149,1.22432558558912,1.71024008708281,1.70075965303037,1.73200665760389,2.18025845479644,2.12855390252415,1.75533078453797,2.51672054799901,1.78954367681358,1.66472031113213,1.52345707185845,1.60277586625208,1.53278470424562,1.60540590555904,1.53544543374168,1.58326824936551,1.63963586328611,0.697444884239245,0.791572419683981,1.04368467254298,0.667002040552939,0.754933912540142,1.65716185868932,1.6193403513251,1.74161468631509,1.90889099021811,1.89714661245999,1.64584176844087,1.6695618832624,1.65582842896979,1.75132576151431,1.68418571338341,1.32412971719619,0.116048030897106,0.1195190239158,0.11839408509356,0.10855983975615,0.0385036025408733,0.0409482737759142,0.14641722454219,0.111598857486624,0.0697779328605495,0.0762823907176501,0.0102666926807477,0.138014524815913,0.0637047766642315,0.121008706743421,0.0799593119067878,0.203766652059027,1.49413479368925,1.7927268792581,1.56735884360182,1.84481253415205,2.01852103296243,1.40387014494833,0.562746764608834,0.561471113051914,0.598130936224522,0.335282571376941,0.214931693938655,0.292230140152849,0.230456508972936,0.562181361170003,0.843560365350517,0.843941465223409,0.355000385044463,1.05047088663454,1.86137133097599,2.17026164258071,1.87385312409283,2.00243639691996,2.35535268065138,2.44188455592827,2.1134977683524,2.63202372242113,1.79335899958296,1.80281047665571,1.82408849248173,1.54940137040717,0.515956477723162,0.147211087188835,0.10440164151995,0.00801182155558013,0.0706183313380319,0.067090115206146,0.0999399358399261,0.0677431359413682,0.156563743578973,0.172376365683561,0.0368541091027362,0.0922719629824032,0.127397987751547,0.12013584711113,0.101093994526694,0.204989320264198,1.40965638308727,1.4169113438002,1.40846755732246,1.51570775828872,1.41728620430992,1.59249102170569,1.38023583915803,0.273462380242086,0.120047319069594,0.14067123017139,0.344628488290027,2.12312226736807,1.80349569650528,1.78870447617354,0.346651139596301,0.421691744579542,0.697044502258183,2.04040317160653,1.65611762570019,1.74907038331167,1.80897323679679,0.886528043113368,0.424471775810085,0.836534806703595,1.83078737249679,1.76233811706715,1.89389942107595,1.77990234708466,1.95821544879009,1.81546683125275,1.4838198606952,0.174685399450408,0.179351102095328,0.161642504079757,0.0175296242686235,0.178889171728514,0.037942314502828,0.207932971850702,0.174686393577882,0.218323317909861,0.176088853359585,0.135182309881581,0.0837711214544485,0.124670301907997,0.150698822588021,0.212172627615542,0.0999772404584968,1.37387272754989,1.68225327437112,1.64670239672606,1.630241053278,1.67327053972413,1.67137718313816,1.09081764673807,0.290056668194035,0.232379100572671,0.193916633387329,0.914566297507841,0.739012836787233,0.342639415794615,0.265531644976951,0.362862573509263,0.289455299792085,0.224816686746778,0.226712238629855,0.254144412223392,0.165876542651827,0.532316523342433,1.93392735224396,1.83567158252994,1.97895176938901,1.12571302344392,0.528651582327463,2.13115556590606,2.58032873313754,2.41693259697027,2.22000839835569,2.43248576332349,1.8982630505073,1.86928018659991,1.93635694692317,1.68481197806647,1.71886795896128,1.19690528554339,0.0951488887949193,0.117422426943978,0.0774350179012263,0.0685636542504952,0.11081680912577,0.18613667433587,0.0353688697470744,0.0909532580975663,0.165273601430664,0.133506180746549,0.0996313569866104,0.0129794044770977,0.101914198320645,0.336961783709812,1.58787180232452,1.67896950521969,1.68077635733346,1.79771152833564,0.731912092158705,0.191196580679779,0.151910757763026,0.223625107165448,0.128311838893251,0.136725521528752,0.176308701965371,0.133284834077866,0.264032882202329,0.190678940077398,0.075101520009676,0.158150653064141,0.090574409461283,0.106989376749547,0.217401761183233,0.126782441990385,0.142107868888847,0.831063906683966,2.45017770320007,1.75788217748246,2.75617669604697,2.23829584875747,1.94826343437619,2.10827276612829,2.0592137906716,1.69047196197003,1.62218312371924,1.62289350933595,0.709944042075262,0.0663790669922266,0.0703666751466822,0.119603871636089,0.116640187689903,0.142043997488027,0.0984333018618087,0.0894801198707898,0.0978380144293428,0.080933700712798,0.160977937398799,0.0335933763811889,0.13118216017565,0.114731503931501,0.192221463712784,0.070003416284235,1.20122640949216,1.71058402657998,1.95509065446315,2.02238808357478,1.72839154888176,0.703809213146106,0.320754018265585,0.250985084206257,0.140856399802051,0.216462889182253,0.119631936971206,0.167271131543566,0.00212520922659767,0.134313444945657,0.123599295312424,0.0920509577739847,0.169834405085203,0.191907852284601,0.159448073254081,0.623876478455052,1.83797561261498,1.9265426894379,1.88193430238724,1.85092235315024,1.9937374738386,1.84864110890215,1.7867333804669,1.91179999613543,2.07547402839786,2.65963610744615,1.27341049302353,0.197492493221596,0.220677043762652,0.2790278035304,0.152541373493988,0.106778010133401,0.0691187588138516,0.213622853229018,0.0847313243050166,0.107839407875777,0.0720919094821113,0,0.12372880358688,0.112330995536169,0.0991520830910792,0.143499889250404,0.126871190550594,0.140144327050553,0.19307329395384,1.60319373918715,1.59782291737944,1.79197228744541,2.14679291048302,2.30949367671047,2.05961020363932,2.13038922514328,1.37510500732275,1.31246034869741,0.432789240489996,0.35927411436576,0.276084029078654,0.160076995577429,0.164386054034041,0.252625734215574,0.219222008473793,0.99062158652943,1.33541609736165,1.29976529732994,1.24036562278106,0.478727177163594,1.80479152792076,2.0360339815337,1.95073465480689,1.863188848579,2.08529377066982,1.87442869683731,1.69141739618067,1.02857860837766,0.421820012829846,0.339139906048554,0.247931002811249,0.0892515520837059,0.0761948296687697,0.112445267031314,0.134739545799979,0.11339638064612,0.0908790682187087,0.158257207492628,0.134523617537863,0.0920988329492955,0.0739027148362811,0,0.0657187510743301,0.0555520006623916,0.125001466468142,0.00757099571382944,1.123159040303,1.59608222328068,1.7679731163513,1.74590081767937,0.957315032333901,0.83465532872894,0.830739687549491,0.534425180359437,0.730806601960361,0.203296232312489,0.142389953068526,0.0653386945479426,0.197852898968921,1.11894048687642,0.61729657154959,0.135688979518745,0.101908623480204,0.264672380116939,0.595769071273725,1.20436838435797,2.35936125677546,2.41970720121146,2.32030232187035,2.57979983948023,2.17684782331778,2.19537680257698,2.0841858147797,1.95973164039717,1.96804079558082,1.71906167725839,1.7082975516789,1.53631865748406,1.4354940955417,0.295059242746745,0.0998076025875729,0.089266451222584,0.14601475983478,0.115889973432123,0.0635390803974972,0.054648891524564,0.0527005756439242,0.073430770726934,0.0956717463312876,0.10741809444987,0.119872131771684,0.0880689379665093,0.193288104272533,0.136929535988232,0.0854282590110986,0.357458218764839,0.413401459962307,0.282031963139545,0.219013392309845,0.421431234903374,1.30663294416623,1.30435297865195,0.617207907414655,1.47179125979021,0.255094751111647,0.348664230586703,0.195752937780104,0.0931224788248502,0.215983003885569,0.159205232705576,0.167006444793363,1.01732479366941,0.932300788263289,1.16012106235085,0.431157248520088,0.467820108388072,0.419828780590028,0.47577579088557,0.739139147821424,0.609076104076375,0.505357214299229,0.402146105532946,0.447810598098279,0.469978892517781,0.321395140935609,0.102624945955407,0.227991178943806,0.0805404949162095,0.0255185412797705,0.100049888336578,0.086595562543093,0.0429267554553281,0.0623917529206485,0.0568038613917757,0.106679469673871,0.0836501978476668,0.134994321597495,0.194639913457039,0.105094762126915,0.17579330269616,0.152348955963105,0.186198565632387,0.0815423867259783,0,0.155727365205211,0.427689283345,0.395733139070048,0.689265656292576,0.696680679292297,0.247048998479055,0.852938862107839,0.285210540431739,0.151952371152024,0.109530811268963,0.146333267111947,0.132931952571579,0.182540280011067,0.155026925735832,0.191294738645027,0.124119936456413,0.495417609625269,1.0197411470712,0.770202532883554,0.855223485731177,0.931691951603908,0.256283230776668,0.432678315141089,1.4687337136627,2.02817949907359,2.01835320472348,2.186828718188,1.94721649135952,1.67042846937887,0.194216141387162,0.135256056111613,0.228944609898028,0.0865700038742575,0.0796862715048319,0.11871970089314,0.115923394641194,0.0207916175413245,0.134490098406472,0.00489627518836468,0.103826104461918,0.0280014329389101,0.0870543092169328,0.176035020607926,0.0581981906138257,0.1770956909542,0.0572365387944564,1.51830402665679,1.82701870039645,1.67265338255104,1.67038899680705,1.38062549178082,0.701485134063256,0.51988589094498,0.426254734202144,0.918166569794175,0.352394995328502,0.217931716238329,0.4147680420873,0.507598613909045,1.54936144821062,1.53703148543432,1.56180173468341,1.49778684927413,1.53176602363272,1.57125636385515,1.47709596240134,0.351557109239484,0.56933674163247,0.458425812837205,1.11332657717839,1.1968663092905,0.7942042033767,0.397215605567979,0.401414837607457,0.426805896532023,0.256386022440779,0.321820213139512,0.0657494007511165,0.105673397612481,0.0424178145790259,0.112121444549592,0.0949495551257181,0.0950739464003124,0.101881173792015,0.0857020451888272,0.0555270830695334,0.150256986424532,0.205800217079088,0.0690046689356894,0.0926113799597126,0.0860669457489756,0.136324091199838,0.077037372794799,0.104038392428069,0,0.191847202030201,0.285138151036609,1.22920461099505,1.15248201525837,1.50829994508955,1.08446412115679,0.426415806246101,0.733030971785775,0.442439237020358,0.318827238314937,0.204793134080701,0.266486766343347,0.292405477382279,0.0876398136374188,0.441348737312551,0.277959790280258,0.161740152177951,0.499502302992887,1.89751902811344,1.63563840676003,1.65534124439544,0.992926404055098,0.378965224122826,1.25012967921213,0.928677777118692,0.491219033175518,0.928643979899615,0.618701936763692,0.381706687877018,0.383212269521879,0.278861706841834,0.13291961013675,0.051417810723632,0.0137506298437641,0.026791841539115,0.11300096089382,0.185045917934924,0.110292700185968,0.0896535180794419,0.0960007933821642,0.151548600759291,0.147757687403742,0.0790030014542056,0.0800902765364046,0.0853258000726181,0.187127074849692,0.104381618034255,0.13348908244152,0.591832082770706,0.760934091570656,0.632584584254581,0.261095495701506,0.637222050452618,0.115785608211014,0.167131429054649,0.224849581420876,0.121666244168898,0.173823757393926,0.232728029983918,0.0855851995912504,0.146293485250005,0.190310292407281,0.106622422577066,0.253406203729149,0.357038117692749,0.411345796513306,0.287607416086669,0.260983023874068,0.226721040327436,2.21564353760072,2.10292563625655,2.39899143312067,2.51374176792722,1.95261056823048,2.55997455633445,1.02965571051555,0.269846418160514,0.222389359548898
"7",0.574244527567514,0.606447039266842,0.453539668928448,0.157438934690111,0.136357923645843,0.155565776411259,0.154836101856379,0.174871340387464,0.123309267504008,0.111809694590003,0.1008125258706,0.162195462783613,0.198998679844575,0.194782016695353,0.0970625819058867,0.0676975638364092,0.139037874389051,0.255686166164702,0.125753301362038,0.0395479226331406,0.168130545448802,0.334738960057687,0.199213701579588,0.0975466194621251,0,0.208390754915601,0.0804588731384052,0.15153192480692,0.237327957288315,0.0731824215185852,0.306862887926357,0.22034875828147,0.120933765995889,0.167630715876151,0.146590862061888,0.75014398432773,0.754434858507697,0.657141264720756,0.824028387304435,0.596807656313821,0.423839214636904,0.518760066191358,0.528179646157511,0.403788385470036,0.540810540950191,0.659061085384366,0.580850511953994,0.550528500791272,0.665892548856359,0.522343167313927,0.571966861995622,0.53220195631903,0.192240154880422,0.0669473986583013,0.151642484756601,0.288076346190332,0.116100207509883,0.211017697618562,0.190927984053399,0.074928486125147,0.148201236197712,0.107972972555438,0.159217857626581,0.19435698191646,0.134852456805424,0.0946383931687871,0.243716384909514,0.169555274009733,0.271917968389159,0.098757642381182,0.196372745339535,0.128485229003931,0.0689344207049658,0.227656685679558,0.175858880569263,0.249246031367335,0.391014256097423,0.165276483994007,0.0348891445546527,0.0366393860803285,0.154615732249662,0.147257844782832,0.169938667835016,0.273753522425027,0.855432745470288,0.65237729911138,0.815530270502539,0.632880698363834,0.712886400585409,0.750040038377179,0.690062349520652,0.559489876027336,0.514771873362481,0.582014942576193,0.648964336717373,0.527985200328336,0.63280123741224,0.582320132489622,0.669994128312577,0.56812091574493,0.482998123919809,0.367205168138976,0.14226218802249,0.178696245909291,0.104553694821495,0.119221130668794,0.166108789472758,0.137625091843577,0.148926415700121,0.124990504175922,0.105773014147703,0.144059683876882,0.183314353451067,0.0966151255640581,0.27693646755029,0.103961060117136,0.182361840758631,0.149570494350548,0.230118431973925,0.310230174918674,0.301001603573246,0.113342140209429,0.175645858357084,0.162540696970015,0.121503262570326,0.259241615712792,0.181346040616938,0.170517941399765,0.16667039746379,0.155441556927614,0.259948511677452,0.102928631905602,0.213147673773154,0.751614784144545,0.547327651136985,0.607353893233662,0.489353472408945,0.396413402075783,0.728383227519124,0.705960089455716,0.65437142328901,0.572148496957787,0.561020741192869,0.669698536434519,0.639601853911768,0.543234895635439,0.640125992178728,0.596837207603251,0.552463513444883,0.0986719289052537,0.101997975782097,0.181011332444122,0.128429221784156,0.130500348588417,0.124062605434249,0.0826710374864901,0.238205562323603,0.116898585508749,0.0459437638154314,0.191115021532031,0.144153159021081,0.17176310065841,0.238494176442093,0.160878566717015,0.175567048199524,0.305408972707473,0.17735322334534,0.0762769680938395,0.0586351348613337,0.198682077478347,0.0682672036115079,0.119168812843335,0.407440999547186,0.0625429627003174,0.171220238336786,0.207383169689143,0.0667397560230617,0.144164784804527,0.329015142398781,0.794710821953422,0.611190558750069,0.57667271232916,0.581942618953702,0.636009003638241,0.593561187010613,0.606049639846375,0.631888105427536,0.633989644980102,0.500241426123785,0.559866343731319,0.726204916096238,0.637932434656661,0.594574851807758,0.706581810021835,0.580898603399398,0.515196269634716,0.47152181572053,0.187616712515912,0.232258492798209,0.0921883926246936,0.155831701010063,0.133590418489278,0.188484533428894,0.189205951459423,0.0862608252293054,0.151775957370391,0.149404010878576,0.14263072507084,0.317560565822463,0.0646492976988595,0.170166982553231,0.20994164626175,0.108112520814712,0.190557992106687,0.133030866236599,0.0924690382120551,0.0957318150949118,0.0952381522918729,0,0.202370927701966,0.0902495364972017,0.254291286809661,0.127198671256432,0.151108932297938,0.205444719054694,0.150699734383097,0.466766083672737,0.737601609521054,0.547107797684938,0.498434195356645,0.635536901857313,0.49708692085744,0.786285557339391,0.657966010845069,0.613180974693022,0.627496703122526,0.680535383755735,0.608407051649896,0.508891050731632,0.658529846841783,0.669206742148572,0.653521161323241,0.643073174759012,0.62959377806148,0.584547590187554,0.213990499555823,0.182477981166635,0.139810783463852,0.128041918129097,0.167189031414313,0.0732616846079458,0.0684651758757515,0.0131183108817009,0,0,0,0,0,0.0821587526787774,0.181081995329668,0.245984424306284,0.193498716450473,0.155002607976175,0.210794554489796,0.180310309367951,0.451230843019908,0.181296042852667,0.220243798820895,0.24252371868331,0.295971917798595,0.148649455502681,0.370978360602612,0.239282199784538,0.229434239262152,0.189314248015411,0.539172046567732,0.507020605657089,0.56228810271891,0.580800352418309,0.803359073554828,0.576474418026354,0.599364166068084,0.654373731726974,0.51268047282299,0.534401658098778,0.534318706506978,0.606274631723355,0.664985158452851,0.540031863821689,0.717449259368503,0.642220901434775,0.591060369959595,0.55491050581631,0.167406955708086,0.173865986801358,0.151965113368381,0.223417040052599,0.125960264348317,0.10710619745957,0.228923963051415,0.0744874415998062,0.0875669432358736,0.0878450525709073,0.0932592908012639,0.165666362269734,0.319215298857775,0.173566841771491,0.22517873891198,0.222525403874092,0.161572976291825,0.180166931054012,0.268118063345238,0.194380876077504,0.182803831685134,0.124552781811209,0.0274765677928558,0.221938832147272,0.0349116168503923,0.154481980126131,0.122069456866558,0.179868987121111,0.218920685178216,0.304176905494647,0.690415621575565,0.665549597234492,0.529507603382746,0.605034817066306,0.439006827979976,0.877569221081074,0.659103766951612,0.612699593315318,0.606969275736502,0.660049268870774,0.595070359001012,0.606077401610029,0.572832943585255,0.603985491716273,0.487920786099928,0.607832696635325,0.637261964015005,0.525876832961348,0.196275186339345,0.166871901803762,0.215199175944502,0.0849284072913445,0.0809098219078262,0.105650477231931,0.0409753754859681,0.176663123700671,0.205382213334164,0.14547429733304,0.159223889526864,0.064825072379708,0.138031482796713,0.19964312103045,0.142845634850535,0.130547384243108,0.165958751317493,0.207660961285518,0.0765484710198987,0.122849986072219,0.09288531273601,0.0804206823892409,0.14458019442637,0.120003834148403,0.275046346042924,0.156836963221681,0.172251791392247,0.280634333157459,0.170323062465687,0.228041195443939,0.640700725376051,0.685854352261014,0.484274457812757,0.560615170909986,0.522257460643069,0.813897193559513,0.48161625300256,0.422314383832636,0.736075804116256,0.524192148458999,0.489034329771271,0.605353177701889,0.52983738123583,0.622253647435192,0.613849095856414,0.57910008502866,0.630510216787054,0.646243673233285,0.229078484147728,0.163609368390083,0.126590948496211,0.162325365316264,0.0988104831665792,0.162875468963308,0.184336390405304,0.166131508776742,0.194248626839187,0.14358673225235,0.130107091227797,0.135226524174675,0.162231694563058,0.164562125015071,0.0932543194169287,0.168185957676164,0.0605603290213215,0.123547664398999,0.0432458530862537,0.322957622660159,0.290104701977764,0.068538249683987,0.114234869648093,0.1065685501741,1.32733609536454,1.12868701722674,0.537685800053345,0.803781845310867,0.661508815493245,0.713249970668309,0.646099569013174,0.628539852036126,0.576193133426838,0.544080353792481,0.471898433346653,0.534552019993993,0.745284206464998,0.523911942343359,0.55909645592808,0.574964483845461,0.433891583546866,0.498415912079687,0.423048846178997,0.513347123133998,0.507506338027968,0.590059735662219,0.51756084297527,0.526169687014048,0.306492114714977,0.145883340453002,0.0985318086620642,0.165154611671385,0.191234745702299,0.223950510336822,0.146941539072155,0.165352989154613,0.173361959492621,0.0447880968550813,0.134468248322028,0.152256508762948,0.362513258489293,0.168886189893028,0.132956126557986,0.216092071384607,0.0824546294402791,0.23175955633559,0.170982570599326,0.104860022488962,0.21677096685556,0.226413036201842,0.201187873073416,0.147314764146431,0.228780372598379,0.194480609799821,0.192259943066667,0.254222549284556,0.247794446592226,0.22947073686335,0.224989533800713,0.569895561499391,0.635410542859854,0.604675824673944,0.613827653561657,0.501259769550215,0.545040425714937,0.791664779669802,0.615030099137201,0.601384592232356,0.536459570883778,0.544056568523964,0.485206783487759,0.668226026543949,0.551037351456839,0.581550303366926,0.628675080986725,0.536763566023121,0.690494632320164,0.243473747311819,0.149520044200258,0.109025550442658,0.194653915945703,0.0790199422179628,0.117570116278841,0.125603626728196,0.169490263038574,0.218069638481114,0.228268678923413,0.0941490875657602,0.313946709293423,0.191499452141838,0.16417463942552,0.276538311870254,0.221897958449521,0.0727245161685282,0.0823453841287265,0.198955892507759,0.213786469067287,0.233502679032609,0.167002132787295,0.313395228778663,0.208361522934098,0.243268000951262,0.258814902928703,0.134141609463667,0.488083515575243,0.158165038560009,0.22125084260735,0.304289824076956,0.231331750302678,0.465448672247879,0.605734120034194,0.566160474861986,0.563460937812121,0.714293271100399,0.532250475227284,0.511943981856324,0.469746587875925,0.601372947629031,0.546549695625845,0.657644959762111,0.656567147031395,0.462546833665288,0.36109889911356,0.180908926558926,0.174415714410859,0.1659748353706,0.0822637174030412,0.077896261062109,0.158325331388086,0.169906880406502,0.231807476522265,0.195648008575806,0.227202837824798,0.175094026255204,0.0946209529668234,0,0.1359211948565,0.294500869234247,0.178957493742632,0.308209740293392,0.319280783272533,0.154478142209346,0.113117290666296,0.170534149326472,0.159327945222654,0.126211651346184,0.115048833860865,0.166748309884619,0.146252957075237,0.227215683608502,0.319981883134836,0.135263287234034,0.224013751719828,0.347722096781889,0.27217545042509,0.195537316457034,0.265515893944837,0.500847162400978,0.258669513642701,0.560006402932267,0.467858585746874,0.567970904807699,0.45693238102821,0.577638364283857,0.702876368867039,0.529390238270114,0.627295637459444,0.500914366963109,0.496316540916562,0.541229644414522,0.595587502373515,0.425032703260891,0.431341714202235,0.186337495949192,0.16631162465549,0.157370823325204,0.195666494386301,0.183693594905018,0.224506840809631,0.127491682886498,0.0970000925591519,0.135866085878138,0.108380694720706,0.263733683489684,0.210303530796089,0.150028528428818,0.0728388597521078,0.190192278644251,0.126699292701889,0.217929320649121,0.198435030763929,0.0836636498635196,0.176305691627272,0.173124366754036,0.0796008677022173,0.158937339369879,0.246410248353524,0.203902809765076,0.254246775971669,0.24146183708388,0.193231244341287,0.0623296114150391,0.0904378892296181,0.382193039496499,0.333322154671839,0.645284441841002,0.54372640439841,0.593763909196142,0.643945344100659,0.577517548383135,0.603050440906601,0.593741907492635,0.657105615704687,0.605775586730903,0.474726936844477,0.621478100648242,0.584312057189367,0.595195613169975,0.635158733159278,0.545590966855647,0.59636952811077,0.458674352438138,0.261098806015569,0.252824117581443,0.0825043357981595,0.212342344852057,0.159378055323566,0.0899190447787773,0.149238967463248,0.137654640898626,0.213499498503066,0.22265549401243,0.181947024221229,0.202451098218303,0.157303068003556,0.134291956499116,0.258318166296908,0.0854624629759808,0.0636907268138322,0.167462120237928,0.0990008295014635,0.0989740038380206,0.367052922719233,0.227179098724491,0.186744776100251,0.139609309097285,0.204806306772244,0.107468785797193,0.203328256104321,0.215136634511434,0.178938379295843,0.688685350751055,0.570969142133966,0.557735681401381,0.575322330875592,0.76337816660007,0.687006831617214,0.536551901133256,0.634687732162517,0.617666914000233,0.550544234222567,0.577102542982667,0.693881839892621
"8",0.132802398714743,0.114822450191255,0.116028500178715,0.130010823895108,0.0749642581747277,0.0693784974194511,0.0994326839467395,0.158106399292918,0.0979329255237615,0.0901643713843719,0.100796733319426,0.141919955726201,0.19341965407832,0.144249528024083,0.0164676723627273,0.0865533719346891,0.203911758233294,0.0902207377929151,0.0201028438775552,0.0722540546707846,0.0973595076155887,0.138115508440993,0.149394330706323,0.0525371119988075,0.0903170413079838,0.14766186133241,0.285326541391041,0.338232578047778,0,0.085515633988008,0.103528130287479,0.0411106768761906,0.0458369878833164,0.321590233386866,0.912254610386079,0.914965108196331,0.641024713247275,0.546106658095313,0.506543638861814,0.43415451535347,0.493561871548908,0.525760936034361,0.534981747787479,0.424168573833954,0.1263829933163,0.130922595523478,0.103254026162591,0.197689147254235,0.135749605952564,0.12160016962789,0.201673848843534,0.123199170803439,0.138363489518067,0.0816006027644296,0.171235244155434,0.109640935578674,0.111301986804206,0.0745587789150653,0.11365048356334,0.115412174786388,0.15903646244933,0.125685818538933,0.102062957950601,0.135795962205286,0.0895461623944726,0.303777990177968,0.121433733557151,0.32929739836671,0.183994795339643,0.0952182563983097,0.117545229288327,0.0736170713963469,0.133226796615418,0.0188675150523853,0.0494067749327706,0.183510290102373,0.122914608233328,0.034148206391617,0.154249019055888,0.379917320959716,0.0932161229339899,0.640245574850863,1.80439083226207,0.53665626401174,0.12313238984176,0.198922708295771,0.182178509363927,0.219937854230256,0.331452630066514,0.308262973650198,0.322696736462535,0.818193739691256,0.631341714013255,0.18652376811024,0.0285942430360924,0.0313851560520198,0.166005736284985,0.0593427501327869,0.138776712933858,0.0538371945874363,0.081383713841263,0.154579519843848,0.0819827090571866,0.0529449685803726,0.149203103011428,0.103116017479493,0.130831842047179,0.129012009788518,0.171704280735035,0.00552817901608037,0.0810902814551417,0.100034240398462,0.17925019012252,0.13226681652062,0.0852964443466062,0.0644182503264246,0.148275143526553,0.16713668669199,0,0.014543560560742,0.0304758347324168,0.0709462602708302,0.0699953055771091,0.15137724489136,0.0991550623053138,0.0767901712716102,0.050957632924879,0.178401438411919,0.221470759816174,0.187595107987108,1.36897928784983,1.49607054854473,0.922956832450757,0.26592661908672,0.205364961796946,0.20058376120256,0.181830018315625,0.280637910447253,0.541189901221755,0.509517577887024,0.492034325185902,0.487450950642482,0.157619144045714,0.0946352829724332,0.163980135133307,0.0726134745517147,0.11997300798632,0.102563406239948,0.133400259981404,0.071442716632979,0.0853965357834754,0.126572925466782,0.0894401419281901,0.0768431387759918,0.126430688687378,0.0874200747675504,0.175402954271255,0.107535598235439,0.117886498749294,0.13959660041871,0.186252465380391,0.28515365191547,0.21108361232792,0.146840655721466,0.13355485449904,0.200911046719609,0.180104023867743,0.144632880330138,0.170429164265096,0.0371782257207281,0.0989141287973349,0.0872010715002331,0.144926740456098,1.00519338477737,0.590299776894283,0.498707391286817,0.767764552235778,0.111384195832735,0.101113717264164,0.137375400593473,0.136142733085427,0.264898065794471,0.134738288620148,0.210987813263343,0.263415855131665,0.543523154131809,0.133708704892151,0.138914090158057,0.142072593092905,0.174503281665041,0.153702538861346,0.188438456153632,0.164811464156244,0.108944409175578,0.172543800766468,0.195089048332083,0.123729337416621,0,0.198127860445424,0.162225873709763,0.136484635674285,0.0731437121344642,0.21996437594632,0.0929655898598937,0.147037875813815,0.0226339669586818,0.157049344837511,0.146396019490499,0.182174406339691,0.121985977443324,0.305039367602156,0.373086634178725,0.43995305958575,0.329391379321349,0.329053526121162,0.117569124841786,0.141062456392177,0.164725547599095,0.413204194893406,0.202272734192181,0.198141051262936,0.122643009186434,0.167855834937592,0.132683464136268,0.111146361554156,0.135578136018993,0,0.0830469845178522,0.214068186086183,0.592841475189778,0.572953812934487,0.466528700783393,0.540929233736291,0.36424775396536,0.123033831753863,0.149704466066532,0.174135749635487,0.201323506317747,0.254724302716926,0.130094263704003,0.16631242174466,0.156845562831737,0.133347503893821,0.124631738655804,0.153658265164972,0.156757542725563,0.131742896735285,0.197293156222353,0.0968674683647527,0.115323629594851,0.183405223148891,0.0683491297373431,0.195649702530618,0.211393748995638,0.211085821067674,0.0949142156583619,0.114551235620757,0.0868655154018492,0.143204888022608,0.50367235798492,0.449584557763647,0.230281020812645,0.0363064938133425,0.157762527074872,0.171832609615421,0.224435879195794,0.358205715916842,0.0203874100687711,0.694963186959448,1.04931171335547,0.40144556194984,0.409070768406656,0.318808283825187,0.107041618185823,0.248600929557543,0.51762834203109,0.301868034122767,0.827179186670256,0.314129853005709,0.432417925839195,0.583173212935138,0.895888268662836,0.453495581314498,0.27747925886539,0.194166828566241,0.147287389013268,0.111814982769306,0.0162411972876504,0.0658587494030513,0.160346721590073,0.06159966409803,0.053144056526938,0.200313993085182,0.123846907031838,0.071510039284523,0.0995520961984036,0.196841933841819,0.175640442001923,0.131748391679235,0.0705126756020239,0,0.0495411093566055,0.159482134870109,0.0763785724050747,0.156910920786534,0.0314621730064283,0.386785207439719,0.0675530510117834,0.113207828477553,0.093833810844934,0.132593818673641,0.0740001465353935,0.0232602634879417,0.101776792175181,0.0580911564390769,0.0928396627452654,0.133319887050831,0.101298399618691,0.0778502237527316,0.105807256552646,0.0753923819485675,0.0446334449021627,0.0605153818180327,0.0269333060654862,0.0973132008928272,0.162555427013981,0,0,0,0.130882533897987,0.0485232156069787,0.0508677129271132,0.074882827580366,0,0.00721481925291498,0.039998639214397,0.0570240816844354,0.0427118396525237,0,0.0950359622916097,0,0,0.0326561630580861,0.0739086340882595,0.0694977188478646,0.0974059736499248,0.0498421894074412,0.159554306645655,0,0.0159630915499185,0.0544016795302115,0.00675689401721163,0.0682790622479679,0.132876520187966,0.0258440184817302,0,0.0174115386937193,0.0521805629674009,0.0507903454750843,0.0216450076423803,0.0643246108018888,0.0622954583981844,0.127436240096378,0.0402660198741129,0.0734141654984224,0,0.0878181271225743,0.0848756146719815,0.0216086511937854,0.00734983393838325,0.157378896937914,0,0.10990168156562,0.0585064130386621,0.037457038696118,0.0877926757112397,0.0946353139320577,0,0.0292373294875992,0.0553281508203554,0.148899973495947,0.0684831836219411,0.0337027921321166,0.0286802793665621,0.11492001488733,0,0,0.0412650216236621,0.120140286784621,0.141889996581502,0.00992667362119056,0.0973820001621407,0,0.0218560175011739,0.100619817391202,0,0.0765655855670445,0.0768877851332282,0,0.0502857285253299,0.0743961703480567,0.0503752150250704,0.0528849881118387,0,0.0473962028489073,0.0398581370229851,0.0767325669671115,0.0338116594188977,0.133680737357015,0.105955057037168,0.0376239607042096,0.0877951479435691,0.119943452025536,0.0968023708298998,0.0372620081476218,0.0326958180609994,0.0627667280320907,0,0.10640692562605,0.0736866739204807,0.0611964530916905,0,0.0605299932795164,0.0680028995136627,0,0.104094822559485,0.0660686358042464,0.00380441609872652,0,0.0536277530226187,0.093910200138156,0.027288229524545,0.0305688960431356,0,0.0220254100599149,0,0.0544304205936683,0.0761978247313659,0.168823533766036,0,0.0223537858839704,0.0597444074660867,0.159749064501845,0.0258127054516632,0.139562337446496,0.0251284984072122,0,0.071049035049338,0,0.0219470892970002,0,0.084511981928655,0.0282590744962166,0.019619412416781,0.0580937339787706,0.0778650870511738,0.0377362989661595,0,0.0769319801055689,0.0729543158200035,0.0157145040188725,0.0636181976743927,0,0,0.023074941898668,0.0957991660440497,0.0603976884587414,0.0129570942895624,0.0135123784598589,0.0795112364733327,0.00820092167533537,0.00982323150613662,0.109875076001676,0,0.00550701923121703,0.0898723538702856,0.0670841322885657,0.00910122721155722,0,0.0277270773590425,0.0256099648751631,0.0885297702631809,0,0.0283102179627163,0.0200747757759978,0.116078760060486,0.0972032956999711,0.057707846925644,0.0866729709436492,0.117249639993,0.0411005148362111,0,0,0.0331099213798541,0.0126212785365391,0.110157633421877,0.146245999975261,0.0513263415897348,0.0213566538175059,0,0.0197339641629831,0,0,0.0776293678140936,0.112501345979566,0.0677148012307929,0,0.0819304076728319,0,0,0.0859421480122916,0.0362763016614487,0.0648824680239052,0.00891585236145707,0.0082587648119182,0.0566386821550661,0.0364993867816598,0,0.0518731936869463,0,0,0.0913242235123086,0.0619904687498568,0.127650200252976,0.0799616766496172,0.0465319240562592,0.0338736794924686,0.0130753094070207,0.0611731882449691,0.076640925986014,0,0.0476683977779982,0.0174691261987621,0.00738418470291146,0.148809595582745,0.133004094621277,0.0695229205688045,0.025742663615283,0.118286754784056,0.0349572426803778,0.0678789339407708,0.0260284628709892,0.103439748785717,0.0790025724402156,0,0.076236569857207,0.103793194226869,0.06721380451117,0.113766996593641,0.0943124403323204,0,0.0126655404874292,0.0419016212708889,0.0578735361738264,0.0236431449571881,0.438906288773162,0.126420687689517,0.0179036522900546,0.0579600939563457,0.0934277629928664,0.104217982051048,0.37973409022866,0.347085072446274,0.388258462784986,0.340709017185335,0.289602226100216,0.113013114747214,0.162359298783264,0.128067933595058,0.162112155011352,0.242082142652652,0.126782223469172,0.195176748347801,0.136757359294792,0.177583096838076,0.145380279545459,0.0344902756633376,0.1294010463435,0.000663884909286899,0.0170046824497332,0.0390337892921016,0.0554163392176174,0.16152766832224,0.0823602358282117,0.00711265103109628,0.0786509315267613,0.121924766807951,0.0714843525611538,0.0359996783413511,0.0303656333399173,0.0428818338396361,0.151756595180651,0.118943902312425,0.0347993445220644,0.0933526327978188,0.03702760404764,0.0185720918558347,0.168009199529359,0.290633968950511,0.100174367728694,0.421247242979669,0.0871055971079084,0.187300496266216,0.0667652875592893,0.111919126382301,0.141717697366857,0.301614714740153,0.806700633796493,0.63456239601653,0.137454056264251,0.100910308850258,0.213241537784251,0.347337089840839,0.495442040133667,0.391125667106072,0.184159693728906,0.571165627263765,0.537486153311581,1.30630584423846,0.638145789784735,0.54528933746667,0.645816395091442,0.576314994984319,0.42780086408691,0.107281506283083,0.127756685812951,0.0562475738480591,0.0377126312035328,0.151250665825182,0.114499092893137,0.0523656480320648,0.063631606326851,0.0946947682701248,0.0659315827696506,0.104753268381452,0.110218556487003,0.101544290531964,0.0911390019004245,0.0597441015554722,0.133536665760818,0.108992285630582,0.0703951747130325,0.12714121088092,0.165399959933463,0.228858087050132,0.104469696058445,0.126530916899144,0.182825534449149,0.0902717287322079,0.0683674474465268,0.111473609531908,0.135311035416573,0.20368141955691,0.49860102993162,0.0959178844153268,0.24683924883439,0.13842084142491,0.185690086881054,0.903651741696408,0.530319943290931,0.0794422930294178,0.0562289111051638,0.235804610765201,0.532417995416408,0.498378920922123,0.466085281237174,0.486847528050611,0.42286907359622,0.41887749587152,0.57753572233029,0.437766566467834,0.102161735663677
"9",0.113314747797189,0.12032639938365,0.0703907200283325,0.0961870854384308,0.151040170140249,0.121975180397931,0.117607962712471,0.0926474640813592,0.0753402272506697,0.122339224073987,0.0825752242366141,0.0734062208735826,0.0689912005911205,0.0295949408531347,0.0455134865685246,0.121223168323736,0,0.0797310895590583,0.464310854754971,0.674909492525896,0.131424458878592,0.138480443471449,0.00520721331725084,0.119483656728093,0.169939879407818,0.135587579493833,0.377484288536562,0.730247171625803,0.165731685140006,0.0408933881945676,0.0831519141899912,0.188557949282966,1.02228344490495,1.38700208901245,1.24160171350271,1.22387074661246,1.55790888325627,1.31360514402971,0.9454718283634,1.26793407993699,1.20770761400584,1.33704065330484,0.438244226971859,0.150994662946369,0.0490380314775834,0.0182622683171011,0.160606593398346,0.175157694636918,0.0027643144569771,0.148827999329184,0.18148599891501,0.21439400982277,0.0324070595777855,0.0391063683812439,0.0947630638953356,0.0920220843033779,0.19580627866948,0.119080243004955,0.257560166431544,0.136452965271032,0.110852995733177,0.116216792372173,0.178584033433283,0.0900696798792317,0.258678960853393,0.503014055800633,0.567487397109733,0.654643711037377,0.321370348860329,0.168989687731832,0.119440877104107,0.161344918952114,0.321653176604618,0.167588027727284,0.166947749418027,0.125289073348396,0.0683652669401539,0.101053261252058,0.0872747479127953,0.249663476590165,0.227839032260357,0.0583811015318884,0.425436658429654,1.43878244916517,1.56795551902461,1.38788089932273,0.990354775537334,0.925069804583686,1.36923182735022,1.30619879037639,0.967635057759004,0.847708943268684,0.211804348077269,0.20924242102636,0.0891772534708308,0.267306884580212,0.229666698636242,0.279827158127635,0.217480542188742,0.141841813742919,0.173245624396832,0.0758350611589729,0.0962518692380447,0.139788248429538,0.0377784393311085,0.129340132578365,0.0689702336968894,0.100035003234938,0.0444104053418401,0.136158039540193,0.0215642159753965,0.0915864811186775,0.0640129743838175,0.167276748783311,0.833868105816959,0.255054730239658,0.195620464596959,0.0241104691142566,0.236644823605814,0.313138475793695,0.101017502067141,0.0927124967483726,0.733621410692078,0.20681613725515,0.428020192638722,0.350948160916279,0.290075873179016,0.150595039087768,0.219422629875919,0.178133644939747,0.534379133461789,1.54694777427563,2.04424137047887,1.30091894546225,0.883564447096505,0.812554789245144,0.930083906009534,1.15998718919195,1.01011979568925,0.934176948968987,0.401631131494274,0.1127768590645,0.0750570110388651,0.00401502398059675,0.0631710120045357,0.0670276407157208,0.186673975731625,0.0208980615649494,0.0742534332797965,0.0809970614450961,0.0856936590728891,0.115342469957922,0.168948657692979,0.107233204910149,0.137859578498243,0.108117060861735,0.0855729486631586,0.0935573303147308,0,0.192833726787528,0.126431728716518,0.0122512354315154,0.0542373151795725,0.475989526914238,0.637225653220223,0.184179931115579,0.155375927534364,0.0905247489388302,0.0866448649977616,0.301274637840427,0.508413400583644,0.626433879407696,0.719305455570274,0.145843492722528,0.382632776215678,0.258964711381642,0.592900350182537,1.38057630862821,0.833606707838951,1.75107303388934,1.6232769431321,1.68423026239142,1.05996142146513,1.09642337194645,0.974936068120195,1.39682446422466,1.0912215917028,0.599603115865787,0.230206146491642,0.170446740243989,0.127630403245108,0.040552561790209,0.202311413273148,0.0927318894537878,0.149328303935879,0,0.113147208314169,0.0522625023499139,0.157131653968692,0.0404021972794409,0.0686206318795898,0.100613560837934,0.143534723456874,0.02451495037498,0.0485061197943586,0.118518803058445,0.167710981832181,0.503856647336177,0.243884527046865,0.155529357607754,0.432542711492851,0.439443529162942,0.3021764996913,0.109906861218411,0.204962758961236,0.0977269813341416,0.141155810636233,0.223146346474853,0.250698762582326,0.364375231709675,0.657435860425813,0.365348812996944,0.113472189107977,0.846630966581523,0.73843771339114,0.392236634512812,0.982154666211508,1.17982956660653,1.2765558034111,1.18824021155917,0.872748244719422,0.929273631220254,1.01046140651922,1.24279379323248,1.08080272765685,0.974628312752963,0.106667516242939,0.0500654403182004,0.0263366313964825,0.0935180947846341,0.0702206420550439,0.0566224733921723,0.06945912532955,0.0598914696058378,0.07168936430373,0.212386321802213,0.111797819434021,0.0414965651377346,0.0881973367979481,0.0851412222553311,0.110656514757462,0.0277134249782211,0.0365394922112794,0.0715178628601689,0.0661092462339366,0.0582584608533252,0.109571146734604,0.0365843327868124,0.0947343001332978,0.386487666175892,0.439656249257208,0.279834682888532,0.228762891758173,0.0939665529177522,0.0683570960196241,1.39023598683378,1.08468921182452,0.827492218455157,1.02238583280221,0.51568881285445,0.151699489259486,0.333859644481816,0.0917855919207409,0.31458551897851,0.610354974086872,1.4228519547363,1.26286213070018,0.915172871634986,0.893022228815018,0.816045983537178,1.42858241382035,1.5073608124057,0.892410744887104,0.456598355622457,0.0584958575984774,0.167711521288892,0.0786908819264211,0.0703750615784668,0.100761493742684,0.0739619709896176,0.118507020505756,0.0499200884753838,0.0965931757617315,0.0337609453721491,0.0106013632786797,0.130933766360911,0.0510637256259834,0.151762013590278,0.156244848398667,0.211167284093963,0.125468416714064,0.111381166047882,0.603936557243505,0.692401174149252,0.321530307974057,0.275976694035725,0.313352062919298,0.169022575905233,0.0305019448738102,0.190077289738821,0.126074024093927,0.108880304259215,0.282703850902676,0.175726341609634,0.162865342680465,0.860530734032062,0.176202184326377,0.238120664584881,0.128772699790636,0.0434937896200678,0.105933191710436,1.01277314494587,1.23279420079346,1.44678049821627,1.33331718767335,0.908095473886406,0.8517213833526,1.08681469520509,1.12655406986246,1.20536540660885,0.955802339754637,0.241631340711153,0.20244232555434,0.234603164855285,0.2516295919719,0.110521709586506,0.11342185486203,0.0380944197903031,0.235462507057475,0,0.124882992792476,0.13530450741273,0.160123540369281,0.0871790500070414,0.0935415953378372,0.0848602505172382,0.116138649785646,0.0918337955673222,0.238546063636299,0.498618026700101,0.219280087960511,0.0886722485799658,0.238364839684938,0.150546195107808,0.156681124492463,0.237467120101242,0.218442251060161,0.348604304130955,0.263415036717891,0.25141759575083,0.134178099775374,0.21906500240625,0.11793365633041,0.185122652383541,0.139361192646046,0.118702806721816,0.229134193602214,0.0868295851022278,0.232815882007254,0.199929487518514,0.836944899530019,0.820628234007659,1.02199119316618,0.901609093226952,0.88611641058189,1.16706225562266,1.19094071304267,0.942324704894222,0.646863655664405,0.677241895817298,0.248394226383274,0.187974773812575,0,0.111010748729572,0.152588311561975,0.0318222444391195,0.0327550142937349,0.0698782828376159,0.0580251467473615,0.0426815730587813,0.0703570368943261,0.0141968625395424,0.121203437675842,0.0609182847509895,0.00872101176217932,0.130937263404827,0.0310947597090032,0,0.0498091248379681,0.128346667786919,0.511834142907567,0.643248133946174,0.499394064108693,0.259370118913697,0.174735713266686,0.13772866895706,0.557705321205155,0.170967303642242,0.141932860961532,0.126611335834971,0.192920815834541,0.937487621365694,0.400325763747118,0.157658965325564,0.198543119762597,0.277361443055662,1.83959801447582,1.75550703505628,1.50489320856232,1.26071690550321,0.927914392711567,1.20737552637763,1.07167956383077,0.823899929589466,1.1941047052374,1.10239936425839,0.686005289687271,0.219806426884939,0.208983076153831,0.0498685987523573,0.0685632760433346,0.0928088947465902,0.100792124118759,0.0196105103271651,0.0897196790542841,0.016877244777948,0.0299123463837261,0.0557065058187838,0.0882639865521815,0.0661994759633843,0.0832427850477943,0.172092884082177,0.0400321748809456,0.0803122311983057,0.138259162558046,0.134985571314185,0.101854118181006,0.446079254989592,0.57107457940526,0.627403105251418,0.175799100159809,0.0977252575902784,0.156652483330128,0.13548063884658,0.0486074733630017,0.104747258096885,0.137411252542414,0.0592051781235269,0.000403442946174018,0.100367553809641,0.0966414261801408,0.0844965534453376,0.107029657004798,0.0168217249056876,0.00716858016870095,1.00791494950926,1.34134729323526,1.35417789869598,0.883837874044728,1.25683477358708,1.04812279598266,0.768722253740185,1.00884008830661,0.988347150117662,0.991862429838106,0.973634639576846,0.65670537580156,0.192340263774439,0.031226855796129,0.141617972566397,0.17762030258074,0.0723111305965486,0.0424685279822577,0.0927628759394106,0.0973539716986243,0.0922049007931653,0.128123465742567,0.10936700645232,0.0847167416938379,0.014011907699652,0.0947321440188884,0.0896656083569759,0.0909916182444821,0.09094925367566,0.0623090016142528,0.165819304226743,0.139665097915336,0.710623899528629,0.231126901868908,0.178257213908832,0.223411670074289,0.0817806694693578,0.151996110200858,0.103859609239801,0.177898734331517,0.294465116677801,0.491571271433443,1.11681406716196,0.292859166139233,0.0205417666212303,0.170010430530045,0.173102287309527,0.121315008812487,0.353063483646981,1.08538855236722,1.21563178972953,1.00218395873283,1.14878766580603,0.844488878705497,0.948744517749546,1.18758097316789,1.25836772537177,1.10285366156352,0.653992193336194,0.255982283661273,0.209118632546087,0.203528538474745,0.0891987795356905,0.0974927423360535,0.130458585450397,0,0.0890847371222774,0.0311709038980165,0.0755214496639345,0.0865340001278457,0.139598992234442,0.0995777184347574,0.114814243885289,0.0753765499740031,0.00558089811078706,0.05901359953151,0.436888780072508,0.21821816231771,0.120085946082004,0.0841347051028861,0.13391235413932,0.237859730976129,0.536568043119875,0.219522173428998,0.132669469573853,0.202598933654568,0.10118957457228,0.254126123218196,0.21536127562913,0.162790697265081,0.162984539759062,0.0880910143219525,0.126688491324001,0.324438946388543,1.25207898604252,0.473556357350183,0.374727288143436,1.06450525628493,1.7746643179212,1.27179491922695,1.05868966107401,1.11821746038818,0.870287528644125,1.05658667423272,0.995511640884238,0.86339682839465,0.762687255163768,0.182112743113615,0.0718266525739466,0.135330321293262,0.0705760836370816,0.189192158489196,0.114551617057996,0.108135643637299,0.114223586334756,0.135575508108569,0.0543089665656445,0.129552467826523,0.0456237039002026,0.149159535304076,0.0242876417833568,0.0923848805380686,0.117216103349191,0.195121435296106,0.0641405336961042,0.119100985975292,0.0279226133814742,0.149928168136426,0.564594756736582,0.217777055630825,0.319956426824422,0.598933895527768,0.207032785824679,0.109413863927942,0.11228305953482,0.158983620306333,0.14098583030234,0.154755855179953,0.0888683957296723,0.706022720147192,0.437004932319634,0.177596477745334,0.106455873737645,0.181312981127708,0.669632078487687,1.2239727552939,1.30428956661871,1.40565618094324,1.22953640029384,0.917376298029232,1.40856826834351,1.07061040821552,1.23593631040296,1.10530302103513,0.842462276945478,0.105984265894189,0.028544728121107,0.140724053298386,0.0258834151389971,0.0819949660171519,0.133518661816389,0.133799060463616,0,0.127371514362394,0.217148371947868,0.239644872444972,0.145981992854773,0.0628594628280168,0,0.123803345426092,0.171633744040172,0.0528474113766673,0.174851803755651,0.0800218495077607,0.164458178823715,0.420744450275546,0.529120232416012,0.3633412909102,0.395500555253344,0.0543328515292091,0.183202957387491,0.196393167538452,0.288041124085675,0.33767618093439,0.19343123628319,0.783206788379553,0.73249297447659,0.320732943697831,0.412305096036648,0.119485099763051,0.0939386461449997,0.146334891267069,0.17533230984639,0.96027161343674,0.701855484859939,0.826191630051558,0.81230632994796,0.913114440290951,0.936065192415008,0.919333334045588,1.13826917712558,1.0513170962586,0.361072760811076,0.0620280742588093,0.140430432990861,0.0705309457682249,0.0724939819820533
"10",0.750473535844505,0.865125743448159,0.824772247619485,0.763022089431526,0.629041388528964,0.737461767299141,0.733465738600399,0.762726599363778,0.791790188918259,0.795023384480463,0.821962948786122,0.353316490424175,0.123370416588018,0.515376077112157,1.23230116223716,1.17143156907795,0.938124993997498,1.69130305580215,1.7500340050692,1.28044130943992,1.0762627954417,0.320271718917644,0.468399832617278,1.02665149391436,0.895158991825911,0.80715815996026,0.697656841838093,1.90633148341918,0.263936053203212,0.0907612049478971,0.0900184403936686,0.109145463717239,0.29379494908494,0.0872915481776027,0.853135528336557,1.99177865331169,2.23358615074564,2.00200004332447,1.34687792333412,1.18229477497397,1.09165097265437,1.15885170368631,0.703570956712607,0.268675693111743,0.231793675329325,0.163660005196294,0.202343500158278,0.520333452182756,0.661136166241891,0.840013319754108,0.87579000891137,0.804478914551879,0.622038628076373,0.792166461064488,0.816042427214868,0.717905258725581,0.753060185287585,0.731474991665641,0.779557469778983,0.492726867026693,0.0276821027493236,0.636515422579654,1.21608351560265,1.19878857442842,0.211044351248404,0.334839381624453,0.414909175042568,0.755225153741473,0.840422281712725,0.662793829462031,0.225192022288413,0.0603939569768208,0.180023114068522,0.277598501009531,0.223474060680116,0.0910051390317279,0.136737484817772,0.0677827740554512,0.323351139313847,0.201081700456399,0.186855071573474,0.236596012243443,1.03669816468123,1.65721301908459,1.52539095013314,1.10774549755908,1.11106106717799,1.02129880494531,1.09358721461727,1.32575192994251,1.0212706411961,0.500007197039897,0.268432732786062,0.473707916665757,0.321011458027323,0.488548610658815,0.782220175608509,0.718839839796609,0.772753196590968,0.778762142449779,0.8153943126708,0.787539545408904,0.789742388563545,0.751309057758786,0.771577348571979,0.813381147461883,0.674021800007648,0.454966225128196,0.135511917581938,0.521398959934596,1.169949683243,1.13486522745691,0.668665269815954,0.147852224819827,0.178762744705331,0.216812668182244,0.43279815931063,0.219908536859946,0.1913067880872,0.126027615051419,0.0985726084794133,0.123268538571575,0.166960912103239,0.141942485609466,0.168130623348287,0.130171929888093,0.0916058172400707,0.0943492772503585,0.128165555822286,0.118982378050761,0.980171674800486,0.950147195527133,0.934568925168459,1.05826118657731,1.2826281196456,1.16291141788479,0.984566554560894,1.04657127131349,0.542091201390288,0.184088525335943,0.242116386988106,0.315018922321858,0.181112525196405,0.461817987770784,0.70797726668198,0.822148129623722,0.771824863189144,0.824088360221984,0.696531587262462,0.678527043771018,0.752174420298918,0.799927452793383,0.737863448379863,0.744789249429048,0.809747753504642,0.433828549429816,0.0928889041614671,0.532060288278647,1.28960819195333,1.35114239593436,0.347383690266032,0.775026129975721,0.749171392422775,0.214795446752213,0.0872933609388802,0.145774878363795,0.147547462343757,0.15672452456313,0.173551898087106,0.239212287932222,0.219984059894543,0.137763235311937,0.882209145053302,0.742190852073167,0.950318645614698,0.723912057022649,1.04961863756564,0.906926631850844,1.05998921851137,1.11327746727991,1.12676218053548,2.01371174695959,1.08677658862311,0.984953498988243,1.02618759136397,0.958058847865181,0.585398130341437,0.265129948981992,0.254661022897357,0.188491515878364,0.163633198050402,0.613825255657109,0.868429248055017,0.917034137423944,0.811638305842159,0.820915284597111,0.76543706526627,0.798235398372219,0.721309559372161,0.78266168551148,0.789852660291426,0.729335824424383,0.673527461022766,0.368054788630159,0.137891018382714,0.605285554482462,0.955226549810914,0.941099958685535,0.208381690453889,0.258391091070111,0.222909155453669,0.168769737076741,0.255452220497113,1.17229315794304,0.213540355236534,0.200477443325498,0.0854456245064222,0.352945107529169,0.26325333107192,0.348098826750729,1.39103237610659,0.761075547293867,0.246227874631904,0.338611249758507,0.0367501757181781,0.666756994823691,1.82845960480481,1.6939648669775,1.62626944321993,1.2794076316309,0.964862407859075,0.816156075659827,0.802130810597502,1.03505523887125,0.788094105080949,0.771292417393786,0.762809120216245,0.298023755911867,0.206603542958548,0.545620302970639,0.795273111934753,0.832270146142756,0.731622791020505,0.766371589435688,0.788052826316435,0.845673711479885,0.697740492027842,0.702722877401758,0.824618853941397,0.728546183183091,0.71337328550084,0.388736799154345,0.156962376303612,0.572939224749546,1.11484966091624,1.11419345742802,0.554448859861077,0.421832554822618,0.965634761140555,0.173783269807952,0.121343159458887,0.0790155503631443,0.0303271558088354,0.0501386762483147,0.479798235414555,0.209478183966186,0.196896526822713,0.23025788341052,0.136863423567929,0.402470654340342,1.15514246201995,0.670284651229054,0.868287258772375,0.481545470980479,0.678024725792472,1.01161389444205,1.09150154569088,1.02130561049651,1.25512397071707,0.920924931416903,1.11741544300746,0.943625649594421,0.898502128318702,0.923162736891993,0.765460867552545,0.319285867820613,0.354738446559113,0.564490273917283,0.837558198692791,0.713976941185341,0.763748214447325,0.82578915702633,0.765330452024666,0.738933660750163,0.759826220432123,0.825127085671566,0.755945773428483,0.751245681402191,0.723128130329115,0.422732996088133,0.211781785989859,0.680036612837438,1.07833631203641,1.05576845318158,0.282722006867599,0.3444945316974,0.922702547298487,0.273337034921358,0.296957805978782,0.117032634227787,0.152539626355759,0.098269436294189,0.11312184595115,0.163308383943001,0.183389137250598,0.164432822028561,0.156547180363013,0.0739697173542704,0.165772955793899,0.18837594763381,0.144999240828219,0.362918134261447,0.760160469035817,1.10542667451693,0.930238617162984,1.21996745714488,1.45044528382591,1.29453652981097,1.18601277202919,1.19024363653108,1.06706687058965,1.03636800807902,0.981600136826326,0.450874800763393,0.287930796033903,0.597919760002182,0.776829155245032,0.860373905655573,0.827050431952759,0.726339384982116,0.761613192598745,0.864881043065072,0.726182376655161,0.731131765819232,0.69708918919332,0.850970558402514,0.763496697868116,0.496887909875519,0.0425126249721271,0.553004434408234,1.16844203602283,1.15087623329893,0.360597810860035,0.543350317258448,0.945823503906943,0.377143132732388,0.363473740973334,0.238801060515307,0.147783623973388,0.216539383654144,0.116363122586748,1.31548106476664,1.10364587962667,0.452142858745968,1.25745516501986,0.9818199824341,0.849457428196845,0.869870080686307,1.15608195351658,1.25703408968189,1.41369946919808,1.90018731062429,1.29086024948271,1.16967307634116,1.1151132845723,0.883815504620784,0.974781400181834,0.920568782786682,0.797445993365496,0.693386481829586,0.343261998779431,0.376308229074607,0.306938800854863,0.564534887000371,0.754535758914518,0.816950774553636,0.871336007241238,0.832664534331085,0.825399140947553,0.717909289114083,0.791867855437741,0.755981180256427,0.794011154598268,0.724911110828421,0.78363233014867,0.408814171442518,0.102818472095081,0.544324452805711,1.04772801333597,1.45038969900389,0.529210627920117,0.951681345765156,0.758236897491194,0.696832697685432,1.02075656604407,0.268685386342658,0.769906806287441,0.466082209322174,0.207304252461698,0.357665516637529,0.101838936417531,0.172447540034043,0.127210698955731,0.169876620917272,0.083826164174341,0.127373639924093,0.313804885078601,0.758804038918076,0.987421433248794,1.07169252994583,0.228656996343527,0.118620891531945,0.148274912026379,0.1567059088953,0.204208016183557,0.082117058742265,0.26706962905646,0.188331215091698,0.165433343045699,0.0710765283405962,0.133906437285696,0.464934439753211,0.908673277434698,0.876364515347173,0.791392115770798,0.896596114883333,0.787785687306598,0.77512681964072,0.698347054923074,0.880074793059451,0.786896098090443,0.667015703223601,0.781412488332285,0.454472910604741,0.0766913512207675,0.468979417921865,0.884212598137782,0.884091109423546,0.264475000435018,0.154129343206014,0.158479356220115,0.135266450948803,0.371487850370865,0.19898762981588,0.362559664672236,0.470587241790285,0.370854360864276,0.31582670046284,0.120713044855285,0.178105542351264,0.158418430602473,1.07405812752023,0.691898818719738,0.219740569581611,0.277840285042476,0.571338376643128,1.03268227299524,0.776070078065099,0.953188804527116,1.03234513237664,1.48814554448661,0.952272905216911,0.861636860795072,0.707848827681317,0.750200906609668,0.496123720595368,0.264769522889305,0.265838024926317,0.171561057793735,0.416020438818236,0.899711036538505,0.764475645695548,0.683389225937535,0.755026445496094,0.787645071151343,0.77036454841044,0.70490515880791,0.800546330004061,0.738610376312147,0.70106162065935,0.793595717771859,0.458920920981754,0.0555998318114839,0.478227130195022,0.874964970629451,0.936094688033655,0.568263138311196,0.158081018231246,0.277015074613294,0.37693097135727,0.218498267971472,0.132800620173328,0.220900267329018,0.121453184482859,0.0919184109502078,0.147434872195272,0.360599199144876,0.205444560977875,0.234760552430499,0.100342536405465,0.107022545153333,0.334059825594662,0.370398320993414,0.844430251198706,1.0164502883263,1.02837358504676,1.05403560834689,1.68461392136057,1.12515322954049,0.981794489572325,0.813826315680605,1.03441441343038,0.30584426107068,0.316520425996723,0.258686473102657,0.213931994385428,0.119354308857568,0.490220042968953,0.8393715097171,0.745056527730111,0.843824747769864,0.742857228867007,0.878420804962434,0.909549968603885,0.869032123045268,0.847323643776466,0.809299198686594,0.764523470952114,0.813847680356827,0.451265484303618,0.145809897562041,0.452375262759309,0.945290299221518,1.01089555497915,0.372266191211565,0.278523018109065,0.271022628723507,1.00073724001227,0.822017815578818,0.644751072584935,0.313901849149873,0.349176743115012,0.430817552100348,0.231289719516644,0.377486465181128,0.35857514004997,0.207736799066493,0.132972420423093,0.184786507598973,0.140776355919109,0.149080982863032,0.757376687166645,1.83112020554352,1.13469169515786,0.956658896225175,1.13714312044704,1.40613498331488,0.4686209400713,0.341635053456216,0.446327344988581,0.386060610107944,0.419706089370992,0.476580033648405,0.322595803158662,0.258992354408509,0.489722930594417,0.805169976274604,0.867583732513489,0.811402460373375,0.724159059177955,0.804010343301806,0.847874469051795,0.885658596901906,0.801493486009961,0.746024104412878,0.763711606429469,0.800063355549099,0.472683704473523,0.255468117308182,0.58573961800096,0.907669495055329,1.14017433681075,0.413083441126914,0.305035720194608,0.17610891738151,0.111847736213213,0.139836380246974,0.235182785691015,0.0845247670853766,0.0803365714325326,0.163396688103785,0.136139460094123,0.172063824656645,0.121230028479735,0.149381385073107,0.0785788403891523,0.162689612429107,0.152432877249858,0.0647716239401317,0.0559535516769806,0.165581110862806,1.02309687302398,0.899930801488588,0.785025151799451,0.752796341460005,1.13128609633585,0.976121293113557,0.421569657375603,0.350284372302592,0.35219469176016,0.194100215191773,0.299467374876094,0.132036244247074,0.429566928941829,0.850626493418159,0.816987553946738,0.926072987477436,0.823959292074291,0.846473859389752,0.72029347810622,0.816638265859617,0.840589807211346,0.780347918292786,0.827458771896948,0.787632135524449,0.514964218023398,0.14565109689591,0.599577160222615,1.10426229121553,1.4964587370924,0.278537309797848,0.18536968327222,0.108624039716904,0.131106305410748,0.401620569740943,0.0525898486046654,0.145917002276141,0.147972947870812,0.210208588056535,0.108044109946767,0.152915303282673,0.0896578923400065,0.0879827646450252,0.139065240825749,0.137635100398949,0.130333202227674,0.12533217921699,0.0855370143753272,0.205189348849233,0.0640519448896806,0.067146808314353,0.359621578734287,1.55234547967016,1.44586530479189,1.16174315221169,1.1822598570419,0.629032243009713,0.286496955884481,0.333588469863131,0.30130763702891,0.133243001798787,0.389695477297519
"11",1.1571658803071,0.165442162124504,0.249931667617045,0.22240554847697,0.122461994717577,0.263449000935634,0.207330275041001,0.285664929077953,0.126838518493693,0.060266265142222,0.0983070689422873,0.279604864536189,0.221332471375024,0.55304220501655,0.206864313539993,0.183198166283004,0.20622701004065,0.722088922092901,1.05349942962761,0.395448520101638,0.548893256004074,1.01901162640349,0.881013738873724,0.967666384769241,0.866199119515318,0.889146907677613,2.23209718608897,2.65407023182199,2.31061888688363,2.37441603531897,1.96230595786753,1.91043105485455,1.9030238636625,1.82071071605142,1.88578761633723,1.91982140144621,1.98819259886193,2.84546723927048,3.73005687895784,2.34057960989476,2.01969204103659,1.91931908350136,1.88225179925209,1.8880792141184,1.83408920537524,1.86742087639044,1.83837325356704,1.93167158912188,2.70551720095558,2.52552100393418,1.53652042506833,1.19371768906351,1.42992200804494,0.493050008116347,0.188005454643622,0.26795127193981,0.258481118957367,0.133884503356449,0.237586997914714,0.128055969721764,0.278342605505737,0.259021039209781,0.125484941287365,0.186711542713617,0.169837280612061,0.138657199134726,0.20902437406812,0.51906184433906,1.55369244307944,0.300189765227842,1.20798148290739,1.41810453052801,1.28980506388487,1.33797855426841,1.47456253005135,1.63941195687748,1.31773417320082,1.29080719550765,1.17903379545304,0.362615678286591,0.269149764514912,0.201114227195485,0.25266902814439,0.414708155429811,3.20855136781005,3.46062757082904,3.57626582328657,3.76069344269969,3.67046310168237,3.12698032576806,2.32189038234258,2.64077428660842,2.68316965829414,2.31945548665242,1.90153573561596,2.05276111278711,1.75003648102777,0.377999659232928,0.243172234440185,0.256916358556977,0.136421167264281,0.148900519452172,0.375201766940481,0.314234342229703,0.215347249940836,0.17410406302758,0.264494097984116,0.122045760165908,0.108383755221966,0.134429240947016,0.146602517804485,0.114897798671543,0.219379473884572,0.192896144323734,0.654158073828315,1.65788048220589,2.64902992250248,1.73024504961728,2.00896562842995,1.63790471137961,1.62730832343015,0.679732378263793,0.588896604723261,0.766046572773996,0.832358496968572,2.20662798509299,2.47001507819195,3.21943053458678,2.4948000683534,2.61348532155995,2.5971185665039,2.53736926167408,2.67565344450708,2.64113777993103,3.81696494097428,3.72393461569994,2.58231349383881,2.4925056949748,2.53562285433838,2.68994607662434,2.7970289106366,3.47575194306841,3.59894581958136,3.55625818859739,3.55645706921031,3.12633180777206,0.783684558662256,0.535632123901649,0.636942425557641,0.656875791592051,0.605952679397064,0.553722754425583,0.604521035525331,0.628899902050854,0.587987586697141,0.564673622942772,0.613858541147701,1.44142806436184,1.83297038369259,1.45590140898396,1.04784834076436,1.85412748185034,0.761752730380553,0.906899326840383,0.868353203372219,1.06001991713864,0.899164540699561,0.785068136922417,0.775912809992984,1.060270491294,0.933382372215716,0.940894505923861,0.909826731605806,1.19644394528819,1.12732605312731,0.980856644936323,0.734592971280641,0.741589467487753,0.738762503153725,0.811949147160964,2.76544820235926,3.77919061749912,3.23805643753762,2.17114300273656,2.12977327968739,2.25112268965833,2.22310487300462,2.30263810381279,0.821510864057886,0.499597582358236,0.396976188493782,0.490384452573606,0.546962286110633,0.356309732024717,0.293106855834027,0.290641317781996,0.216307215876374,0.213181312787551,0.157328969025422,0.103758160025052,0.220585446823922,0.24606329991459,0.218143517460064,0.170199916228624,0.191364534206815,0.860484245958299,0.757871222199554,0.655465868824362,0.610246376025831,0.360028411063125,0.891139018839408,0.307546752912298,0.488074536851549,0.792384153745976,0.779639132312709,0.716609031447645,0.822342770396698,0.850919643400689,0.812546477774727,0.927003764679785,0.93455930661728,0.374543254827069,0.351464490996349,0.45406399641589,0.321864978635556,0.308979203393236,0.285138236689937,0.598555830783551,0.870490173769093,2.54674947509026,3.05969163925948,2.49684518223241,2.50494646412951,2.43919362756864,2.52246208740837,2.50476455461957,2.68627814325168,2.46456734180163,2.36192555823496,2.2679910035767,1.4375649928471,0.328145681744499,0.23722755756812,0.423633928504006,0.256671092319779,0.116011680965349,0.129403651120221,0.190501963488648,0.156706289426487,0.279847066653989,0.193942164904643,0,0,0,0,0.0322463367906316,0.0738618615252084,0.024769963558935,0.089407475495747,0.465294370674127,0.260172375937242,0.215457278330343,0.217846019799949,0.543690318727067,0.996812556101304,1.09278838188722,0.916139223946588,1.00187741511449,1.1212206318698,0.958049923284925,0.815324829992304,0.146712612990803,0.26851000489737,0.273259295034567,0.306532916227068,1.08526667836373,2.14462403540632,2.31715777851591,2.27608886850116,2.14420076336073,2.13752105128863,2.10928580252269,2.06315329889425,1.93125097871849,2.09328537644218,1.9913989340536,1.87863368977787,1.87346852532405,2.00522078097348,0.803385938847936,0.0875545694254089,0.220008283360639,0.171262642341915,0.239693262896032,0.158500382823128,0.0985405516316983,0.106666807078098,0.205269691435421,0.13142322124977,0.233455068243201,0.107228092488058,0.572876535881149,0.827395981637156,1.00161667956397,1.41238643906412,1.09837238578504,1.1411144231219,0.551019419533538,0.12042987216769,0.0871784377728313,0.236062197299214,0.116918450692659,0.137744558656249,0.152080965274095,0.259500424009504,0.0946917606427537,0.178390453443662,0.223459587029203,0.13085524429846,0.137865619214374,0.958056360513243,1.94808996509933,1.88476760749282,1.81735371484066,1.98856941857664,2.24146460254391,2.26901186336957,2.20021983332757,2.09101023630597,2.13263504594422,2.17740721429424,2.18834361715764,2.13648882970439,2.06217223634193,2.17964880363453,0.971631771164031,0.934804166975741,0.333801417159904,1.08059711599637,1.43163688657758,1.20314181372268,1.18818418049931,1.10255513601012,0.587801159508984,0.0374839137977306,0.0598760653588359,0.143074484297806,0.11327212191428,0.033797352318154,0.242599405120127,0.444691306402307,0.132985928772996,0.137920361114734,0.361350634610599,1.31196519926702,0.335561436290197,0.447859028443974,0.231439805621217,0.949191559779792,1.26017525081704,1.54859308473036,1.46430245614565,1.03251936249638,0.301413602865293,0.402073642528722,0.43751856310183,1.56860617688261,0.654699790138965,0.286492389024272,1.55403324069808,1.96620050845841,1.93563997775608,1.84311440792102,2.11049714164573,2.21042795097796,3.36776157294806,3.78490303410281,2.57736468441783,2.49060656377378,2.2801854451144,0.718856981645572,2.40612174511977,2.24148377559734,2.21879529128445,2.32285789633133,1.32707631205848,0.169740028390842,0.226144813627199,0.304482606977606,0.286854407752939,0.146242871406525,0.162067008327395,0.152926783831507,0.0374870107396899,0.204552789165964,0.255408505513111,0.186905898243011,0.186465969639839,0.0683607080028419,0.068812832906199,0.208752381963049,0.222787375432354,0.389902088941162,0.654628514331721,0.719739098700948,0.959370020021432,1.65852557649741,1.6407239721298,1.46604550953198,1.60895059964966,1.62670867185461,2.74376531923788,1.73552532461548,1.84529543734642,2.44133087423742,2.56458851971953,2.39056807011501,2.37454448365933,3.77336868480441,1.73889860617388,2.03082025153941,2.52210074476906,2.17177797714456,3.10626225197362,3.66422115541568,2.5291886712826,2.84197327160401,2.67303401619305,2.84432054152228,1.64683385834471,0.957831547641577,0.948527800349258,1.02926986518379,1.26236013458412,0.8481714121066,1.47084542771004,0.675880905605305,0.600713715669379,0.640230254026727,0.696734985353841,0.590223295108635,0.577460696745136,0.653806908948103,0.658809288529363,0.659665294742141,0.588145867241763,0.568863576559157,0.704077375122285,0.739184854993344,0.741220857206876,0.813372056942635,0.873849638005751,0.740423458393532,0.549328611749127,1.54643807809616,1.53091704362273,1.53031585446693,1.58488690785796,1.33870310499346,0.673037571895106,0.936943615848196,1.55206001840178,2.46275522514957,2.3568460660613,2.41887344895464,3.35019451048614,2.09174626466349,1.48340181254669,2.46356134944498,3.98752571524336,3.38321345484275,1.76662597315857,1.87156293354511,1.9311566023602,2.53875973236578,3.1441367654513,1.74783542493864,0.295755868404808,0.293947654791731,0.214920954581809,0.274535059368387,0.108911246726597,0.390989244840397,0.308921766380916,0.328905214556669,0.121931375135336,0.231290042556679,0.216985999865835,0.230793320222991,0.0983065274005588,0.238961596598727,0.155658031598055,0.155492188637999,0.13625741273844,0.582914263397773,0.446313126724716,0.2567901147199,0.242838662879278,0.607878549897032,0.371532888690273,0.552148856765382,1.91967112173608,2.06582295938229,2.03625586215313,1.94972192304236,2.00813948938769,1.89354556168367,1.90159793972079,0.746476490155693,0.433233889763598,0.348784530137681,0.352239815992526,0.313677355693458,0.649124374784819,0.498841513472326,1.21963067775606,2.52327099275296,3.21796018268225,2.91499969593906,2.38750816024121,0.750174883742374,0.843411495558808,0.849587320846439,1.15753939225029,0.89072812124848,0.866753489383286,0.81671657115584,0.533825793919139,0.563353859062039,0.147601484842294,0.28791619950946,0.26551405934833,0.166724374096592,0.176952711555701,0.368192668777207,0.00498105647012737,0.138969518064869,0.19270423648131,0.117995206035106,0.133613668067318,0.0881264091570747,0.631566434492168,0.487198365409305,0.476254049945881,0.218476619966858,0.571425891874658,0.382675228558937,0.4201242491436,0.347395861510793,0.448717984981745,1.29745870501328,1.34545123738161,1.35984129420364,1.3234377880681,1.53565805288878,1.33344261107826,1.51296684170954,1.52521447819559,1.38559656093944,1.33253478179971,0.80806669176312,1.14578207428633,1.02763899064318,0.951905231844259,2.59827600420889,2.62919906116539,2.68272909177943,2.91293856758205,2.83686450747229,2.87865712053049,2.96532278738273,2.7205363584405,2.75947813947485,2.90451022042849,1.91220761690205,1.3697775051467,1.20155240578206,1.36707579298838,1.22259320265612,1.22118242397612,0.529320553745358,0.19504482816776,0.206910844721648,0.133351727621595,0.0714927192472455,0.146752163943189,0.0868984555767154,0.119662753398223,0.179630342644387,0.266797283333141,0.168132904292005,0.134349782645479,0.279367337301349,0.666870866945639,1.25181810930387,0.566177422454566,1.33233057050931,0.710279765911085,1.14962370314892,0.935314510987054,1.39320754230124,1.32731661755463,1.12715891448278,1.30863953892644,0.575889455180757,0.488730463438677,0.484830752895303,0.531717897871648,0.675631337075574,0.740241675228209,0.693820331479866,0.909089297554558,2.0489998989313,3.74111212912784,3.33053017086452,3.04801937344871,2.50255576331302,2.45471769569629,2.44003953737278,2.61018202205787,2.47467541146733,1.50779703238,0.473186738197338,0.320773916440429,0.259767103078384,0.395605874709099,0.184761679539867,0.0128500967757783,0.182651616312396,0.242378499671585,0.14898632935403,0.0858571749174969,0.111432333682121,0.22300618147327,0.142888828360228,0.0983318300011181,0.0827365769461443,0.479927401980349,0.256647013115515,0.456204175833373,0.569771733002743,0.650747012022677,0.12024267085864,0.731015208450282,0.172612486775274,0.361504222343814,0.42974880533103,0.260388382093489,0.231328608894636,0.156571511328815,0.360134816561521,0.354040928617668,0.30158215776686,0.878487983812021,1.17025652613937,0.796690022611808,0.275042442018879,0.187712991037537,0.663215865335153,0.771801958439112,0.84035215415422,2.26571515457066,1.35123022931539,1.85813423277434,2.16750418207829,2.14717508468922,2.23380075259709,2.24940730358781,2.37729856054083,2.35315768655574,2.14478835332765,1.91427364417331
"12",0.218311737590908,0.172525653314599,0.180943634555444,0.188413611551225,0.478677617576489,0.405950999944287,0.446186980647124,0.597880996749328,0.517254142392103,0.672339936707249,0.439388404650777,0.585800050880976,0.568620319906608,0.629047260305843,1.98417462031499,1.85639196021753,1.29742137277723,1.1333162217238,1.11851900040857,0.269617746956553,0.117784091901977,0.0197301387525893,1.97754819836999,1.63162179726914,2.02351251406665,1.20073765260325,0.938272111843123,1.25838928344877,0.872973607993936,0.896671463862894,1.19238438544867,1.37436540761207,1.3557294675452,0.850000397819431,0.913893473333576,1.90181514829623,2.05125819696659,2.182355567117,1.76003625710242,1.74692081778108,1.35532953214946,1.3565609813338,1.48817106499572,1.53283570115023,1.26376619031307,1.22077492743565,1.18353102608356,0.347113329563681,0.71242614900744,0.840702832585923,0.491975545459001,0.37555986599289,0.344308517240805,0.697616968933677,0.323224837342855,0.686323318599765,0.437039071585852,0.49726691978874,0.650802483528349,0.581784887974861,0.460060841513167,0.696465170820351,1.88594773550662,1.5606890329131,1.21548316320968,1.21747927184112,1.40264515335912,0.500732153052701,0.449259672920634,0.175047495543916,0.127054419332159,1.91163419611829,2.10237491602327,0.885948820266122,1.04147959742546,1.06057127460679,0.927815951685597,0.979624846453963,1.57727758312427,1.09118346450916,1.18067400389806,0.867305671147929,1.40551346025112,2.13925935306872,2.35402959670287,1.01126745912584,0.917632337268439,0.800521854049705,0.88263151618847,0.87206135488433,0.96223153587568,0.91141376446718,1.69395358322103,2.08311717983942,2.15904371651942,1.53975408488777,2.80568712826009,1.52528282470488,1.63588114016931,1.38867338228617,1.9272473894043,1.32419625455253,1.4965703723768,1.23603801574674,1.31992689906685,1.29558098252291,1.47570336236845,1.30611990641516,1.3920699538405,1.46996136698127,2.58557950626267,2.36664143727331,2.05283525416537,2.80245116396244,1.66928934910357,0.277242117113433,0.120744813431111,1.28851898131838,0.897882967383043,0.749188920434943,1.62778261624292,1.12749698806661,0.863073833848751,0.749543883517105,0.718818592205265,0.861176250268323,0.748905912607732,0.945615084335294,1.24860369281191,0.685757125216193,0.866138677835481,1.79913101084866,1.62164133556233,1.35113244181782,1.05635824494274,1.49519278191829,1.14245032111404,1.48640685368961,1.06510491250559,0.947692756408067,1.07902889925942,1.24167683356329,1.25862763451652,0.597208811576323,0.122068024800448,0.158801690032522,0.369648220622574,0.546650784636875,0.234600434608572,0.418364734773762,0.449807877008973,0.472187407479206,0.418150882665434,0.41309599645705,0.369349744313587,0.392478617315102,0.370559263116083,0.441241643709204,1.94790654942182,1.544534862278,1.65298999171316,1.45748974565887,1.26167112186313,1.09926648628737,0.84640507917805,2.28062682246788,1.39835900264579,1.14751749756908,0.997924730480534,1.15159129573114,1.32113193399062,1.59984472903221,1.45061686659285,1.31627759543809,1.05656930814235,1.16114505746312,0.869491928623431,0.878047906388294,0.882976787411213,1.33857333283815,1.5052785652519,1.70252210115561,1.33460198938987,1.48308221640512,1.21469895065691,1.65479509720373,1.69818910941451,1.56433527258212,1.07949503373149,1.06227809769134,1.21778219630207,0.583249021910057,0.389336908465716,0.134882598815763,0.20026284519601,0.174130447158853,0.212896072414938,0.8648997594102,0.237981067776784,0.420765760596693,0.220221619637613,0.533792841338729,0.441864194199566,0.37595474099623,0.490718463782461,0.24205726982359,1.94265308067362,1.40909309616023,1.14104099134661,1.08296001245764,1.21423129340144,0.0941580912421731,0.26824521636028,0.56325428170623,0.534029752138428,0.1515326809479,0.441922440017564,0.347879546164036,0.285172809332324,0.314100036822578,0.207091998312964,1.08224526337672,1.43247267011429,1.3276401783119,1.4687910097505,1.06502200466812,0.722375238846215,1.30134340674919,2.20421641949477,1.09999355992243,2.5643392277776,2.379268651608,1.36214541770151,1.71646208094857,1.1573182735385,1.16948431414556,1.17492729259876,1.10733895532245,0.89125154915013,0.230896328677149,0.193505592724495,1.14434938494809,0.898603723584653,0.935071537459907,0.824463692243189,0.83813896730726,1.02238318137327,0.86728715201026,0.823591031326823,1.2517791988551,0.864608339851583,0.850886743478275,0.968507584674254,0.985598997317471,1.59269289706839,1.56888839433196,2.10724461280417,1.30059748653497,1.10362131828831,0.476671264708849,1.33389591589178,0.901105063734712,0.774164117600583,0.764074593035779,0.279559207099876,0.405569794036993,0.184940175202919,0.196563584438717,0.68709303918945,2.25557552009965,0.999897766579472,0.786366681153471,0.936953891854864,1.04229628916704,1.10737387602335,2.01511568680595,1.80301310548361,1.15962265169776,1.75084464289755,1.62390872279316,1.07792139147193,1.2257016494843,1.02605104902416,0.965757370946481,1.04479677426124,1.03293842916018,1.04282228588781,0.37370129752213,0.225144150424862,0.781896233067318,0.783546281977743,0.248332699275362,0.113704917971592,0.452126965335108,0.341252856602581,0.274042060073477,0.461833816602565,0.471573842489952,0.315522369549449,0.348345186907634,0.612538042296376,0.597364678708005,2.27728357679581,1.59611471434855,1.13567639323161,0.948190784972533,0.974424359657862,0.307805631528312,0.287267704674671,0.782579534028528,1.08527084470209,0.753776651383816,1.12885774598301,1.49805268858916,1.49311353364282,1.96914255087018,0.793870836461536,0.742330257428408,0.886856082936371,0.957977023227566,0.772826741662816,1.08798090315221,1.00231576454493,0.89338581745206,0.956431346831038,1.26645101296128,1.47116771003549,1.2365140723314,1.0388715975162,1.13955832090807,1.59701392768761,1.13045887375421,1.04981722942982,1.1196375242272,1.21374951691745,0.233683260078586,0.300168819335137,0.244995784295291,0.150707647978985,0.381954938173192,0.542169624547187,0.500280272331654,0.3973732643265,0.71448088475491,0.445237239328354,0.532345646811914,0.453850475451878,0.474008557502939,0.768237524186476,0.759804922577853,1.92722852378607,1.70182226468166,1.08786146201192,1.05367088505565,1.4846472735616,0.580110386938202,1.58381851803561,1.46732501489233,0.964010404782955,0.823388258387856,0.979294937556933,0.959590665841913,0.928649530438245,0.656258646738005,0.706849680266113,0.630797798206005,0.83553804603912,0.808156321226815,1.01562060456497,0.875694896268336,0.854129929805678,0.951369210166551,1.24816124395012,0.994731045067438,1.35333264746888,1.54792467607053,1.41347814653323,1.34088504994189,1.0974996609231,0.99555573888302,0.980024235291514,1.03617015718337,1.06182634631783,0.616310020859173,0.606055051050117,0.311145741938828,0.209416244585217,0.254848551785491,0.252974350754957,0.484959723680195,0.474578049153248,0.429027825807453,0.64219472977718,0.387240821625565,0.461728287582139,0.592965824468934,0.446567795603368,0.611674389927557,1.91644981132504,1.04649533924713,1.11143411861183,1.64450256972871,0.815709853462178,0.293008656300772,0.531221393858423,0.536670222253912,0.550779809862964,0.131533178542512,0.149538942347149,0.301127246209523,0.0888412087139985,0.199427707584804,0.138773878641952,0.227289812722361,0,1.58815310937207,0.620501070483854,0.548463296584422,0.918452119134289,1.04976362087203,0.849424742263821,0.725206298705069,0.865201518685607,0.856169830937645,1.26963583127082,0.899651646336968,0.934558372189848,1.29032980474631,0.834128610036033,0.820075207032275,0.949605779158843,0.32804207994121,0.403696012706663,0.663297404726756,0.733605769144476,0.150196023134923,0.167718596555305,0.130609658900367,0.327716164309316,0.308146938470515,0.359285262167793,0.24687583542075,0.383807986641218,0.410288315295725,0.470379679846249,0.416788295673805,1.72396059667691,0.859650458102626,0.845212250329617,1.07795991324061,1.02179030665366,0.776213798988982,0.267094571286397,0.613419812594214,0.177527194061473,0.171856577523514,0.48960783143377,1.80455624144209,1.7287957202223,1.06815996669207,1.05984976057569,1.15375852712567,0.975290187836516,1.18568115318953,1.28613941474457,1.2312446854493,1.22179327934858,1.1491078379131,1.43289844745016,1.54905507016139,1.1075746855274,1.63853457572886,1.1099267984011,0.988438107578321,1.28580111684776,1.54211743114393,1.18378143653596,0.899534625436555,1.63294718315805,1.37632497787048,1.25505462633179,0.748238190477664,0.837603296963671,0.724439840133985,0.764865323231385,0.707536212283355,0.733537123204565,0.769732928833075,0.719444461039827,0.651421866294968,0.789724597486747,0.712389261819353,0.70262743632726,0.723040516904475,0.707463263255277,0.648294724939146,0.933345918934859,0.590637008123458,0.857607622280791,0.721629656719221,0.814527252144695,0.971100860291547,1.34021213858686,1.55627646282669,0.861634165084541,1.45713583771423,1.22045416118463,0.610818408299874,0.350139167573027,0.487965468203807,0.681869317681504,0.793163266768499,0.80948644820825,0.584013496491941,0.474200583559087,0.451041160504916,0.566547110203411,0.57891905917043,0.579562284404115,0.68438844635149,1.20802658202226,1.14571987988587,1.30602491128367,1.54849820052822,1.46663069167501,1.11348018203164,1.04475554338673,0.35238649875867,0.260244438851128,0.190385946003633,0.734492304596322,0.282973480957043,0.0770564347702229,0.0670444457802103,0.491165874621174,0.331333773936505,0.147454181903387,0.294965821747931,0.388199340608481,0.387823191107678,0.443678431490751,0.38115101143567,1.77393044130416,0.901976488968856,0.708878457239197,0.915573004317101,0.993201979723086,0.4498446168024,0.102389768585385,0.0738820918763895,0.0455330651789086,0.0624544104653634,0.0593111626562539,0.169781794504824,0.124405665450332,0.00278356078322547,0.0399930343500646,1.58622513528224,1.57892470411545,0.458112512820782,1.12811770661665,1.27862504969217,1.17949429091374,1.19465862449086,1.22028877068709,0.912833340060359,0.963911771460757,1.34800195170199,1.23131610429172,1.19521804663573,0.875733575922522,0.959827392731722,0.93002402400501,1.15256859726104,0.982230747630594,0.257463763052217,0.392343510943888,0.279776210110053,0.207112497509532,0.274430765873363,0.36483198951509,0.303030167684574,0.177872693403652,0.219033191237224,0.80555511220841,0.488453148245251,0.148290090438985,0.423481142699499,0.430326054687768,0.191832283183245,2.03268299444106,0.9789502204823,0.759590538926218,0.697979809788933,0.896971958172139,0.390704064235197,0.357567105814627,0.527731902403661,0.368036969875614,1.55944987999047,1.01522988547223,0.620848487209803,1.02706028577829,1.30607143947727,0.782364399743866,0.834201193606529,0.51189048005572,0.54376341191503,0.45133852549847,0.495898600696267,1.42031098739449,1.18900715177542,1.09871000736407,0.96623685884856,0.799973277195691,0.960506647956912,1.37198944513158,0.841273400397731,1.13254512918418,1.36075040992301,1.18858943027185,1.10005423219911,1.42917414766132,0.54255835960339,0.476925512984143,0.248703513833821,0.432152599678489,0.387268807959212,0.592011175333114,0.584883387935033,0.447832158683318,0.70879901620293,0.583242635814322,0.589449779986291,1.16474071601718,0.470692688916375,0.647305005441375,0.623222670607481,1.91105008560763,1.4272029691879,0.911631428063476,0.969025251548617,1.28662254505429,1.08545665178909,0.224440625407718,0.253232030971328,0.707648527774573,0.627842471503013,1.41488724919878,1.21850091354211,0.726296452453869,0.882084104460281,0.392986620259523,0.140775774994776,0.191805534164164,0.37260640466485,0.839955276921017,0.571922025124958,2.26415974784819,1.74876583712469,1.37419761601418,1.10782925848643,1.03769030516152,2.07544486568661,1.35817853604455,1.2806895155165,1.02248013811255,0.651012500708052,0.829945730725461,0.814459954926316,1.10437642798993,0.547662561936501
"13",0.17110295267967,0.257519551794531,0.143496276998975,0.264109219268812,0.181864630986209,0.203048925126825,0.386838752215167,0.273542540940344,0.240202598759353,0.243194663747954,0.242435644493886,0.24816595774746,0.209399769001926,0.392260197678554,2.03799113042769,1.95434774133513,1.25025687801001,0.533849449197078,0.151502933949884,0.136923284689724,0.171075406846277,0.230997568814733,0.30468866084784,0.160193012882544,0.169185990842656,0.247355157027576,0.202180234813582,0.249932434549143,0.129163001619067,0.206191538781921,0.301545669859497,0.124255728145248,0.27359603909441,0.445187203235426,1.1797011793815,0.695594137353743,0.541910513967401,2.54616827964217,2.81332646618535,2.82720575270347,1.86539440661688,1.66723262612303,1.59528333941165,0.399437192756369,0.30840412400281,0.368781044125553,0.332944345107057,0.322472145390675,0.152510185565812,0.224440943462511,0.167068252093242,0.198077193076708,0.28392247687282,0.269835033151335,0.278842152889117,0.243238244524512,0.217603865966626,0.272077412537018,0.259977987221341,0.260334480990561,0.275833641450468,0.210371182948249,0.284251184424059,0.488715337825683,0.680633022202006,1.7335722716645,1.35017065802514,0.466297152776282,0.165438162189497,0.247411051667869,0.195363257300182,0.344674101921894,0.173676903927371,0.138944669781548,0.265142068618206,0.221235761723218,0.189883710617942,0.168930292769716,0.288242198599185,0.214128964868758,0.146715046535738,0.218670188091391,0.374191241960758,0.362655731897827,0.384205416482326,0.382986287201642,0.377846022548982,0.428058157789517,0.437594218945065,0.359034199489545,0.464729731136102,0.440072191354018,0.16189926993734,0.344473277747044,0.292253779986105,0.425496937989865,0.302986242053187,0.0747810974345113,0.182786483600966,0.209224324077569,0.260046395297486,0.192493889102931,0.261058864093464,0.305206570038032,0.197877203562303,0.278364920140096,0.279852517568824,0.15978762667606,0.440393045430108,0.16088867111187,0.236315333089474,1.45669766740785,1.0522536546834,1.41343539683068,0.524723565623874,0.334514006039494,0.427535135153506,0.446099586374688,0.18966151702686,0.308778501545742,0.343418542160064,0.203738279026019,0.238134817847989,0.189893595777087,0.423377832064662,0.322346766189499,0.353921313487707,0.467825329605485,0.360834004042994,2.07028955988867,2.36645393078464,2.91510002785932,2.56013609536943,2.55989529569997,1.83227824143485,3.10016281166953,2.78231586562947,2.45438145345812,1.38718263988803,1.44607403651805,1.59334047226921,0.734911888631652,0.313790514273719,0.39320506300949,0.113841135722854,0.200367889448109,0.198493854056976,0.210560840264039,0.382827064531087,0.34922331990553,0.314071086503368,0.223408075217817,0.248737353465659,0.156019112042134,0.391583300109577,0.182023595818992,0.3238004623669,0.251682551930738,0.245439905599723,0.614797647408578,0.353811083639966,0.578271823324494,0.178646395642086,0.332968256887149,0.174984052998708,0.279268165601127,0.218046070752163,0.281696449002173,0.269701103540493,0.123925195498726,0.298637948317419,0.140981585461906,0.278434637490164,1.12495824155854,0.967739006460164,0.820362760706585,1.24644755859186,0.47434077276498,0.783457550979547,0.849371318065455,2.28683145833306,2.01706989817589,2.58150390684079,1.58108472338678,0.661849361748485,0.553805617003556,0.53396498879082,0.521816111126672,0.429791847075367,0.516317782759401,0.344368154432175,0.228354204709605,0.273938438061304,0.12321217681279,0.264245999080286,0.165583944468854,0.138906701847658,0.258255114635357,0.323401928641787,0.186856160321183,0.173126157290061,0.237934681031954,0.279845537940906,0.180958341404367,0.232546255324372,0.219874327940329,0.2460758246666,1.0339767243559,1.18424117410209,0.662750664013805,0.382370160910326,0.28862132174581,0.177054241664579,0.322491393526965,0.232004355986572,0.144734159414328,0.244108296202172,0.241295028309095,0.304699006261675,0.379269911268042,0.6278956598995,0.499874393458931,0.402485266547571,0.57305981870226,1.40115240895935,1.13479822545397,2.39404596260608,1.83485455865505,0.766954462268985,2.14630058699276,2.35119205911927,3.99866897826075,3.13336376861172,2.44644620136747,0.569186329223166,0.531632762404415,0.480326956314123,0.573707387420932,0.354963750302153,0.377329548996759,0.249892772141733,0.243955664456197,0.273131939694215,0.260634973153476,0.303034248134891,0.403848475679053,0.465770097972514,0.204395122560468,0.367001551882749,0.344831963187349,0.346567239668078,0.246234225040043,0.342501002785472,0.437462535575955,0.200462484696888,0.527635815417874,0.594814176358046,1.38451587395506,1.52008683166882,1.03831854612094,1.87135129000807,1.12992657441378,0.147321194888718,0.208320245964519,0.159794365385884,0.299851669569815,1.42893036301661,1.74984186515895,0.571869625966674,0.583659159938838,0.471631260242697,0.4791496497045,0.532527122081551,0.509315022277795,0.511981440211923,0.687633192613987,2.68686793287073,3.9387040219702,2.63847461625374,1.83751978907047,1.46710575454747,1.58658323505585,0.97212571167538,0.404695390358384,0.610634966757249,0.507391105518404,0.324251642138065,0.643751365193947,0.210954468459751,0.329125986998367,0.472332743615937,0.340582527940352,0.524009233002871,0.492611441050773,0.452745447287587,0.436344417975365,0.63729364123908,0.416932946891442,0.547073930158735,0.525201089910978,0.570124033102317,0.348960372254042,0.260850912885994,0.607748457359724,0.601848490977399,0.604251441625801,0.25467138981087,0.34234825036919,0.252761582290426,0.255423230342978,0.250405638382139,0.165074097564515,0.282285576789886,0.260821253482683,0.359927058714263,0.171106493229963,0.247224219850955,0.282661347953978,0.220130447716564,0.279933719927723,0.339633343335189,0.137819267100971,0.228960033191172,0.170893847641758,0.219974293666961,0.447435248275636,0.464130712253422,1.27673503036289,1.89731631296794,1.90798382882607,1.82196928492925,0.457650431831639,0.44933930992494,0.326210451485962,0.319155336357192,0.303877944816802,0.0619335297117538,0.304331758199981,0.0781999255709784,0.0999615896491106,0.219998000843779,0.290365942295004,0.156070920074414,0.336842045102097,0.261858698350983,0.271590582549449,0.255548780547244,0.333964446183576,0.245643507606839,0.233862928067278,0.286287568511831,0.765077429885528,0.738528896215608,0.340246402814703,0.0931618466355557,0.16552867266952,0.182410515860895,0.251567106793173,0.237816833147031,0.196663089008321,0.20608647960927,0.140497647212029,0.295593069874421,0.240583598657233,0.508932432882566,0.364525225267185,0.532264458666533,1.09884758152372,1.04866095479558,0.82341630336049,0.481407183719407,0.351646696049956,0.515993539261673,1.07847980359835,1.3683818352415,1.1920815889558,1.51740302021197,3.00953829148683,1.50246779031649,0.402404308513914,0.464064748684021,0.506351426252039,0.332294310405597,0.352487767037138,0.269037491430001,0.281639762652436,0.147575136043811,0.220952233157524,0.246675326975531,0.271128244659949,0.260260939432393,0.312857616041148,0.252682269196102,0.186752080272009,0.267183309232797,0.182827154371125,0.294805803273633,0.236695007221858,0.128275035329945,0.290817934177669,1.24098436811098,0.974097107174815,0.90532133833484,0.274233852908705,0.278515998516818,0.200544050205264,0.31996413551243,0.220822170743424,0.193492834654897,0.168410734673746,0.241210455665827,0.128281209739908,0.281101085016138,0.211124131291476,0.284541855104357,0.257053864366721,0.115314810507235,0.263374260473377,0.192823493109465,0.517615617060582,1.30726570485382,1.04501096822337,1.22788249138628,2.38666593728153,3.17523621499054,2.03607199677002,1.60686743463021,0.957492962643679,0.534052363025556,0.38333687713763,0.863239257103378,0.554576616563353,0.20909881936854,0.635598560387432,0.108256080059424,0.23766282426222,0.269508443888476,0.177803400355084,0.262992944419905,0.233927282386221,0.209641249686522,0.280170644652461,0.321012511752621,0.246114042743991,0.233371258813132,0.359766881282939,0.164111966963701,0.269549515266075,0.374321068010674,0.568708572216821,0.513477713949335,1.15800255590819,1.00937241535229,1.31465128102082,0.882846407059477,0.300957990382169,0.211070261542675,0.148214926804609,0.36708286765383,0.175249538639362,0.335482675714841,0.259819032788558,0.291298262340504,0.338507096582735,0.464682628741174,0.526741793987486,0.573485318115926,0.907037704460256,2.05351936437839,1.7552994157921,1.57321976377398,2.05266611694292,1.25748313552965,0.54884077722124,0.614963829284956,0.696820329098583,0.369024310214676,0.22478320679108,0.350631588196574,0.368376927973257,0.0938784436139518,0.305342910044091,0.0817510295776064,0.223968912217253,0.232752851201053,0.386351246563522,0.299431679041563,0.225302502927175,0.318426465410131,0.269181108719171,0.301639982024949,0.182357245626667,0.296558425882242,0.254439503718017,0.372725146380038,0.258392243437724,0.457775718563157,0.622653161664045,0.428171516558009,0.381856616191363,0.236463804978813,0.256188653473803,0.337035712321103,0.256349362343464,0.172480874470012,0.151847193974519,0.199912461050073,0.177346987161234,0.222167399510842,0.104280084932295,0.43258788878761,0.422046115404119,0.665715501886343,0.52693535130504,0.544909682410725,0.58330430357427,1.02608208806438,1.63254616692024,0.770406549340248,2.11877035574096,1.58664963877337,1.55360078073305,0.718409504358673,0.373658293502043,0.446784818525363,0.353729029583918,0.325459831724711,0.184115615220645,0.212626146184688,0.131934924410098,0.300796778153173,0.229785676479228,0.234016082825799,0.308645906576356,0.247082339962189,0.356415230457981,0.25316122200508,0.153671368901227,0.335757571367782,0.263873700834395,0.267726193246351,0.251416316430241,0.248629001476642,0.259422889156736,0.471093269956991,0.741400819753327,1.02616423909917,1.0684338943147,0.258711431650517,0.21052633586041,0.29018886727481,0.170249589313554,0.183457899905277,0.114698038222746,0.260696202268923,0.189566660136912,0.436304562441686,1.2818043942211,0.988137922539589,1.88489390760334,1.2319493804976,0.434435334527046,0.378876577609787,1.77814633072439,1.16150688183578,1.40879977094872,1.03872966695462,1.34309587900739,1.13882186640997,0.489549184412204,0.603454758062852,0.46814247883699,0.509017348272335,0.427688712437188,0.303133931582437,0.28283351077758,0.174761451446478,0.240272335476711,0.245834162146089,0.20931481733825,0.218196220996452,0.272472238344842,0.23271151443271,0.320626243967142,0.237656475126493,0.234566393548326,0.269051880357906,0.183255056486054,0.215834926695972,0.201771532018923,0.205203303025654,0.379436357114306,0.622700774651737,0.504494455753855,0.694621711281162,0.518459756182457,0.296501879245798,0.36758229922581,0.182422661002609,0.451217544543799,0.320845770507156,0.925256164172557,0.686931919557383,1.30161933517581,1.15748756876711,0.82321485405831,1.09347758924665,1.03202712439905,1.71689947212126,1.53892571157085,1.12314817710277,0.875221476600375,0.575895106019043,0.633569864363841,0.723423943986285,0.67928992326882,2.00384650821274,1.71603829906695,1.72625514874846,0.836894373769482,0.39494253099772,0.390800776743607,0.452664289514606,0.247565417004998,0.207783480397213,0.181886214894867,0.207340934105404,0.171902221812515,0.30630588814527,0.292277265591034,0.300020094525305,0.303242398651615,0.206136490267016,0.315682670055883,0.191781308699568,0.274496435031928,0.165063707088441,0.338416922568928,0.187279164814061,0.421161480930475,0.657927134847734,0.492077578815837,0.332502362313983,0.192871889742381,0.227650639113831,0.181409921893037,0.370782207016623,0.265616399964532,0.169133321094469,0.181557338988951,0.212492430617769,0.262118311357911,0.378860809744895,0.347465966525888,0.424519016428792,0.336408763106825,0.432858127572802,0.305174914846859,0.21987126824895,0.319895101443844,1.39426747574951,0.898882365717652,0.691947688598467,2.01651595411486,1.66461147552696,1.7175376708119,1.65588666766559,0.695470227442965,0.477277397226218,0.359319998993364,0.303771248901898,0.225845700726837
"14",1.62228283914657,1.66349468083799,0.7485928920526,0.897011701734422,1.17119276550076,1.29721750905222,0.545665883297507,0.418942598347867,0.437768719688842,0.965328227090859,0.864743877925388,0.373070771230553,0.361564355178147,0.500191409964205,0.488008350827153,0.554576294516708,0.592908231902526,0.630575240134572,0.506232257508665,0.435366571018684,0.27852001179283,0.401357895603764,0.456775293940264,0.3471548932918,0.335289802351759,0.368928318075506,0.612945712960354,0.604479698692713,1.56324944068832,1.46059122908953,1.55996755686346,1.5742209422439,2.03799358630707,3.09487741983535,2.52834478068623,2.45112157905202,2.36820723082989,2.06765092443724,2.11798113850844,2.49450758305836,1.73108817382816,1.66419933023282,1.72151291215085,1.70249229303903,1.23975501800431,0.383969716615825,0.442338967976132,0.510479073293796,0.422405062878116,0.425064416920267,0.416450385497626,0.459158077645038,0.439118218348223,0.492398866963437,0.505209893235139,0.322581175738064,0.423307517903563,0.498049354943929,0.502274289040901,0.49627925099982,0.369351042435519,0.418622869496491,1.139402687231,2.91888960976178,3.81748765585964,3.73553057582953,3.67339571133761,2.54532576092243,1.28724676602369,1.78437350622079,1.67172843834099,1.20949203769564,1.52686899091235,0.71755695765993,1.58490735172002,1.73227965347665,1.67868371770375,1.61311602846073,1.51318545079674,1.47208043152993,1.50701428276757,1.48259458867661,1.81499528073955,1.58312994435463,1.68398303171075,2.56278265599232,2.24789753025195,2.32606967181793,2.50385869107193,2.23315273654027,2.18256394273605,1.69395981442012,1.48104833306091,1.50560192412466,1.53159399711906,1.57360724187996,2.26076714746063,1.08980327055717,0.405780233883016,0.415869785103714,0.337099698453154,0.525490132248123,0.497174823363902,0.566966604842469,0.456189912880407,0.402184222431509,0.39424718762876,0.442455609781931,0.396510413159578,0.92078900047879,1.77672907463807,3.38422439069769,2.89732755652393,2.65834065741593,1.83139482034878,1.03555071018039,0.543350525553616,0.433518751218067,0.465497423556204,0.45486120309337,0.402242435428677,0.415408179715503,0.578536824500066,0.549007021904775,0.57923836486902,0.615311670050163,1.6627681424448,1.65721520097195,1.96575142323356,2.23227366758007,2.35175784938515,2.13761351967497,2.97611331795956,2.33206484803529,1.77722573578654,1.73883875306384,1.76429214031484,1.84688297890766,1.68950850564169,1.27203778015458,1.21397171898661,0.527440805116948,0.393354602459436,0.345309348649545,0.467952382276123,0.483900276371948,0.3831894291659,0.456888070114151,0.417625340477641,0.466339269458542,0.398774979217604,0.504554985025004,0.501784354779527,0.46361455243658,0.75905761307491,1.54992657954022,0.970542089903379,0.543268893007811,0.626597550833428,0.618216189378414,0.692840172599799,0.825724464150157,0.449350946899506,0.39980903215146,0.44061608796567,0.464279368964221,0.416105709307022,0.333753862259086,0.501360878587576,0.433558332246952,0.421212419621539,0.506056641897713,0.403326747063128,0.478713283581311,0.439330495783968,0.577359858388352,0.727182127486587,0.879057450333694,0.679276331882242,1.05348635307973,1.92884157248818,2.56977995128659,2.12442737460977,2.34093313152601,1.69176772109652,1.62563982552305,1.65905517814511,1.53311220707731,1.52807965545251,1.52210023356973,0.609388422506585,0.430105323316336,0.404915824866085,0.447141050237158,0.430842293909307,0.442316802355032,1.11615389105713,1.06369652125627,0.48499085988645,0.407921566962287,0.407474465098946,0.420587506940245,0.389110544144421,1.41606426128211,0.534122171683021,0.535715321828515,0.43298746617051,0.597242179222223,0.562484535222675,0.780187320854723,0.518469321234665,0.532626694219125,0.49785529028697,0.557865722231779,0.361678583469388,0.54558742830121,0.462417171817655,0.455923058328853,0.278636373232979,0.39700974179487,0.431531249680079,0.472690925085471,0.424942460424899,0.416138875848137,0.394238603220064,0.497321230209565,0.542438675403817,0.546133711606062,0.508110887357202,0.557931936178764,1.20827330595409,2.2805210704937,2.32100482982724,1.74695985904201,1.56336169196552,1.46129563339951,1.52439333380687,1.52002777151054,1.15741487357203,0.773475600344074,0.478641882086678,0.476095664177052,0.423052041127336,0.630343691297627,0.441462485783804,0.540410413906729,0.362006155212488,0.539143565900581,0.474883813320628,0.499135755380725,0.429544456945409,1.50715820616759,0.787448908148797,0.525788928793627,1.14167795494601,1.74075457789007,0.724959836987228,0.666789596336625,0.806657724211632,1.56720740337067,2.78787654371181,2.1485278300159,1.59567768792075,1.69514558818175,1.70168415870349,1.58409022728514,1.50570033760812,1.49528454010378,1.55245495576358,1.3536777929376,1.0545612185709,1.13319440440064,1.10237371817266,1.66600556697971,2.02721830420205,1.97803816280662,2.11299392314618,2.68250956501212,2.1451252518536,2.07705059081946,2.07602404720362,2.18497023483221,1.92682421707594,1.86980965524194,1.79571681623858,1.41995762196215,1.37865546245929,1.45418969812406,1.39759969608289,1.42246819337313,1.18382426262219,0.360301648315861,0.303688023660342,0.408426466568378,0.408569776405967,0.385300008047589,0.401544102895172,0.304385794213317,0.528651912736497,1.51527376845326,0.976621830846425,0.301941185677603,0.428764640781821,0.500870493166518,0.615206050464836,0.558816528821541,0.54676582402452,0.386677565829154,0.315441576081668,0.376879695216734,0.359592863624125,0.404809426300733,0.436822969903625,0.39415531636961,0.416279545267844,0.390587766454292,0.369797465215066,0.387165906108974,0.423070687390764,0.596190571201701,0.449799919270201,0.430423432555927,0.828183536137404,1.05559896922857,1.98792888646232,2.88745797339323,1.7840121338067,2.23413763641014,1.85490690886324,1.49781924347512,1.50164224065497,0.892566878305487,0.391575890948374,0.407420157713671,0.405013433607149,0.372362494568213,0.297157170553424,0.436703548903334,0.361710838754675,0.383231799836736,0.286474614925863,0.382755892414884,0.310638449242594,0.417509625869321,0.493714191564371,1.18219734279047,0.457228649195048,0.328194988344024,0.46345969534932,0.411724751070494,0.533605430355828,0.707284231756991,0.505046276111776,0.429968454593947,0.407961709382437,0.386253715736312,0.450670443871064,0.396060965085132,0.312023527991453,0.368909972814156,0.334073269468772,0.330081548583247,0.376453942362671,0.40266070071402,1.44728764648111,1.93563263086992,1.80869480429157,1.78232018158779,1.6139929244107,1.6613855409957,1.68790655813756,1.70780699012034,2.21008358902259,1.94931723889761,1.63121838368571,1.58091709812492,1.8710435328755,1.50865421092666,1.25927683533958,0.434385328305521,0.471997750483836,0.424601979015144,0.425192771593841,0.391417019431917,0.355093973923601,0.268566710393999,0.332722014436137,0.431355652810492,0.396482013535695,0.313918612405608,0.323625511065877,0.344790595175064,0.42465813068622,0.420940886413465,0.394861947628122,1.27353320135654,0.568780882079031,0.298572722075398,0.439827374810243,0.475384323510435,0.371895312241329,0.468255541220314,0.53381023013531,1.09119692997065,1.79498262972037,1.51417272312237,1.50946145267848,1.533494260669,1.51915840562774,1.68319524766614,1.38477366010477,0.926017030021607,1.407950856661,1.43678983682442,1.34290225093832,1.42597418129334,1.49459021183965,1.39133619328188,1.32642662566692,1.47954959931687,1.94641043004029,1.05277174692676,0.566188237747973,0.614652404948991,1.42540883806521,1.87543275183723,1.63143894831576,1.56345760653745,1.37383256359192,1.55250848618572,1.3817731067879,1.48857171768014,1.57589066699693,2.12368227331005,1.44583728234114,0.413647614834088,0.338090074099944,0.279364140331758,0.385297232637959,0.455874611167141,0.470723322213302,0.355417252627779,0.396757471851979,0.399178770131674,0.485853838459743,1.48110319696611,1.5188263742004,1.58839737910173,1.89217023667932,1.78083372701874,1.57121923339954,1.54309530591425,1.63170015798903,1.55421708978808,1.50483377806317,1.70162406559155,1.51939103067706,1.48775738107418,1.56000947798296,1.63891866584635,1.48504365846784,1.66007831248145,1.41616827136283,1.50648515858322,1.43223763212402,1.60027280250521,1.71138937627336,1.20259087683136,1.10707074075354,1.55681547196599,1.33453253823264,1.66660171614953,1.76570246087893,1.10191029392512,1.17184068817606,1.14465422399778,0.714181401958924,0.425027151710052,0.51183525007923,0.369820240205734,0.373589254103665,0.460483409203511,0.391435280678211,0.478485384348188,0.488948234699855,0.425483469931369,0.425471444099869,0.51813200040682,0.475638903167824,1.3615538374603,0.91609558017553,0.409157245450649,0.499265096316116,0.47100041279626,0.902028264731812,0.639150261013632,0.487063002995304,0.615247641527383,0.572097560439157,0.489600263262113,0.492258176360007,0.557205264561402,0.446629773274697,0.461178831061408,0.512766814785214,0.409423676292006,0.545405180961853,0.383384549248093,0.42752573245598,0.537418294789816,0.463004945981616,0.591754897158344,1.5297810030648,1.66659417020098,1.69982313748992,1.73118031411842,1.72042495040628,1.74905311387883,2.49046924084327,1.8029935038554,1.59534158394918,2.04756565020567,1.64218619468009,1.436704735716,1.49693165827288,1.53678692199306,1.46114350126536,1.43697444328473,1.43987135146151,1.37363787638541,0.542056683830335,0.438135140220559,0.355406221401698,0.469712845717355,0.444346470658503,0.42042722230902,0.391573711680173,0.596256561707458,1.3287085540852,0.425265687368224,0.483638635300793,0.562301092643962,0.448243973665536,0.429307508119241,0.580106811426656,0.470383151957136,0.5297798989354,0.640273687069137,0.469499473702513,0.50231764781293,0.479574534233836,0.511038821030146,0.591243522972401,0.505535855048781,0.361475650319795,0.44980082465008,0.515880598919362,0.346232026554184,0.334864546411615,0.567740507495611,0.381991604753966,0.799289724406683,0.404840047760099,0.51395318536586,0.56830980555219,0.478720834107439,0.666125283018714,0.623824389788373,1.42241600841202,1.75378532641751,1.62071818078715,1.57492540859865,1.53543100339354,1.52084484312579,1.53039570564714,1.5634182060698,1.47944104611379,1.53801622279849,0.794654493981029,0.3603569609433,0.471788495889892,0.542681530609177,0.412854486607331,0.499434693180051,0.469274976916426,0.461687800175554,0.574427390683512,0.99886061192972,0.493680009203184,0.522580260577713,0.463662907932317,0.417118032676766,0.615423942587348,0.580772341535757,0.662612488521264,0.490459220849979,0.517055150227685,0.814981408923628,0.554141015923184,0.441687485334405,0.56690799517822,0.452682008463621,0.575986120466091,0.440213374563732,0.506385561123252,0.462724625136881,0.384581287285659,0.389785025597552,0.473613380446792,0.777307784083717,1.62208978276955,1.79425797126419,1.85759406720616,1.86975569019058,1.92651098628775,1.81273672842023,1.97589289973548,1.74219019080837,1.53908226458321,1.60889414606353,1.47064849016975,1.51963670208056,1.3950740310798,0.405735396747842,0.33786494445272,0.484874730373547,0.446443874620369,0.388210096813532,1.12271520464953,0.883841736465375,0.40348354453812,0.489411358223498,0.465511128058579,0.39396637738998,0.430851396856746,1.5326123364052,0.749371851664504,0.461528224414433,0.489934496843143,0.827318924044722,0.797026600571655,0.655770763649078,0.411005120797451,0.519226443502607,0.535543591180819,0.417231858785529,0.441303418294123,0.506883022891409,0.512018138125662,0.550279430523246,0.523100067872259,0.570449299344301,0.512925978817194,0.405852279482161,0.752018077877147,1.62375112908002,0.897082648949042,0.507189041209949,0.512400801116647,0.537153989720115,1.50165502105312,1.81018717665378,2.55356970408225,1.90662891218047,2.37712147141303,1.59179654372254,1.65668082315069,1.50550439858041,1.48848849167293,1.55554522829231,1.37156623780939,1.47618674376358
"15",0.0278871561654742,0.0809143149776976,0.0415793198233709,0.0997363876557539,0.0188950773965681,0.185424731063595,0.0527174595322157,0.0982573540541312,0.0915791258986319,0.029549769345442,0.149510647117656,0.0837723323801966,0.0909927563072063,0.0633759466751075,0.88868939226281,1.72344735063734,0.996934121219053,0.847361480026612,0.774235147205945,0.139701664385261,0.085406229733313,0.094493087490477,0.0559626603381752,0.0813021974661987,0.140216053419565,0.103629252346287,0.0705567436138568,0.0609599045552849,0.0556841174676806,0.279304681348803,0.426966669332524,0.472524523441869,0.676005975741291,0.414436548907049,0.313689958597009,0.308547238715058,0.454449405152518,0.368217803930682,0.287200732913769,0.448271300571762,0.402949438592696,0.356272156513095,0.401437916582733,0.321964699875128,0.382891863733591,0.657968083995362,0.529360796922533,0.162824966666409,0.242636712303251,0.0165321040005014,0.0606307057148328,0.0334311249863661,0.129132268300767,0.0948961558060975,0.150313106048279,0.14233032389465,0.0892655009914141,0.122168172493362,0.0513834621158501,0.0973434454780366,0.189598364364771,0.124427195877533,0,0.0776723750345834,0,0.052880383029272,0,0.131101320199491,0.203950369415303,0.483541311978199,0.58457261373649,0.0406910049874918,0.0626409386934615,0.13836927348399,0.24589972117096,0.131677728363916,0.129779037385492,0,0.0632255079303466,0.0396093646545513,0.138216153597436,0.208654079676507,0.195886900781809,0.160496176686888,0.183825285228893,0.401037727071464,0.424848677586369,0.398956987379212,0.275810757831937,0.489006191762454,0.352267308421114,0.378699291220909,0.383189365585597,0.346212555775041,0.182569072236249,0.0577356203054466,0.0994576045385565,0.101850402614363,0.105929228510418,0.15238025102719,0.0882630775557447,0.0397556993244666,0.116549917335097,0.0170019468240331,0.143956438448671,0.0053514337951523,0.0199000515100837,0.133430141063739,0.11860076792458,0.137332379634389,0.0840873865532572,0.101685012250955,0.416190884978942,0.0943256270716586,0.106711196108534,0.0257824111778079,0.179125709436345,0.0818865913945188,0.149092802268995,0.0447694668088538,0.116972709852421,0,0.102631279164974,0.114020829901953,0.00150819715951327,0.178905880912309,0.0751818848958249,0.0149288362010612,0.162781218055615,0.17139541077218,0.127462511755377,0.20162327391978,0.28440202586303,0.362132841031714,0.641832680500182,0.312157359370352,0.295912421766852,0.190234627659076,0.305701486277498,0.335688743066149,0.246069908201103,0.0220987462684916,0.140887905002385,0.101334122655421,0.163365485577103,0.0981283422734142,0.0171269389207392,0.120537121515961,0.0970189629163192,0.0295889270226486,0.107543936160674,0.0686349772022446,0.130239936879538,0.119113043431248,0.0769145156898557,0.146243627161156,0.135189277256937,0.0471572914115006,0.00501393814882084,0.68734385293717,0.966649036784849,0.860411330792058,0.336279933490859,0.00359152694387076,0.420897446482435,1.05433782496966,0.816420681829171,0.689167972499249,0.75171719110765,0.775490766461327,0.542072539809009,0.53090436906836,0.378256939561289,0.361332725307711,0.433504236476245,0.33837104859211,0.355791333192273,0.282089686620679,0.169108383891213,0.324355346925391,0.352248264608336,0.358893630296533,0.412130980716224,0.461275423535718,0.364058929945639,0.367191522269097,0.325011743615279,0.354121710826696,0.358508261807253,0.253038001285718,0.680325830115736,0.776132235216176,0.526040417792162,0.445515750017843,0.189125886728832,0.0697795776887816,0.0453734002558333,0.2205550858611,0.0491981600870066,0.0820011767632731,0.163641134411386,0.0403613697475637,0.179589493988055,0.129146354936995,0.124787024634998,0.119388559237224,0.0417404561045467,0.159374735943179,0.253941172200942,0.148378644110361,0.138792100255257,0.0424909103199203,0.110717907892381,0.199715539262324,0,0.0314869539886972,0.007427796893582,0.123573882675687,0.190561278809836,0.826269355612409,0.345879996575403,0.106630680055626,0.0533823467333842,0.201104537411361,0.397028056866769,0.239628026359373,0.268970656466297,0.366820875520328,0.427474099880809,0.425078733200921,0.381796997286612,0.344558407575347,0.391547411745493,0.437020965309551,0.301024752928757,0.526037543618131,0.368268247841107,0.307186451905666,0.243258560840727,0.198689804324601,0.101232217200881,0.0461195978602217,0.00518092367064538,0.0289597073334504,0.0534550881509973,0.0802987468058864,0.0869221266526313,0.00240685269581293,0.0863690088073397,0.109996899474894,0.0273367190830695,0.0653631690870043,0.0659005932823773,0.145318389833505,0.0723126189466642,0.0517500522609616,0.828697023531723,0.791765290157209,0.565998527892589,0.073913672523722,0.0858343914442029,0.0669675336283918,0.0181226264598116,0.143361791310047,0.13438823315401,0.206824575584681,0.225130943368398,0.0796021048003748,0.176094848230604,0.306782759395022,0.221496829167656,0.265168693244237,0.311088170032249,0.354559843046131,0.230558774507462,0.364124499940187,0.141977132571917,0.272383990656716,0.196567738787539,0.264759378104501,0.258049046977077,0.19277634103375,0.292619944320171,0.237078424384923,0.162194938891274,0.192848619505709,0.189510376146973,0.171559247141315,0.137130603237002,0.134733345488097,0.10429707630721,0.16402034471739,0.079385245287846,0.0789298018025466,0.146975567172667,0.0577455792087719,0.0811372106724484,0.137100174280172,0.00687438165731131,0.110427411911923,0.0911659875554346,0.170784719975833,0.0153442335408422,0.0931495409441474,0.057243403076943,0.0736985651523948,0.140733230943713,0.149089278189442,0.243966697150482,0.285870607119945,0.296467886728765,0.693615234545289,1.18920908490793,0.974583312012991,0.964448894583804,0.847556884892126,0.801558567577846,0.588092010412804,0.310548089491515,0.593803360295751,0.626641523928498,0.596171365184535,0.518028615192867,0.300934496931422,0.99133270389693,0.54602707233582,0.410176420657669,0.3544157102994,0.808694980160064,0.712322345965121,0.466556473367591,0.321421028132769,0.303377467983545,0.345437897705016,0.321390098927649,0.135068382413342,0.035384139556565,0.0678823603882562,0.127582682171185,0,0.0535645526007295,0,0.058786825314769,0.141602173311649,0.0768285418654269,0.198425029494272,0.152359305791995,0.112204633609349,0.0883145869597207,0.0645513957954668,0.0829731047508367,0.0807675870510001,0.113858059263413,0.0477993682011096,0.226478836304671,0.673090252923557,1.42518155046645,1.14065122972237,0.464445148305344,0.918290687975091,0.524803217772931,0.828834618038156,0.152040890078373,0.0781783438463459,0.206282739360467,0.33882125722413,0.24696005360934,0.262909221541589,0.227623318795672,0.232363739583208,0.256189960774345,0.39869856265251,0.508649270394479,0.416616378304025,0.20029137713974,0.1516427553109,0.132265742686898,0.174997260544461,0.0913506434516158,0.240542704942923,0.612772151619389,0.473202759086986,0.467961175135541,0.438126213922873,0.132770579222156,0.165672532770122,0.153344969717099,0.16907361664794,0.0814924839109994,0.0539909686450971,0.156465799684272,0.0552603622393373,0.0774357570626907,0.0491676987640829,0.0513378156976893,0.0194482740171802,0.0346100773061252,0.1898862902431,0,0.0018096632184354,0.121897380922731,0.0307873792212248,0.242849430062991,0.0835006377487299,0.311099524640348,0.0789515661649804,0.130226261024968,0.178661523152617,0.128491362521026,0.120022318339134,0.374558297103433,0.170342516998686,0.145616520302008,0.324087117958457,0.158004206053664,0.205501086067158,0.333849612295775,0.204156612276196,0.36748950650317,0.282029293795912,0.296816104851803,0.472856073148337,0.679412413940641,0.460136947842634,0.575485459261216,0.554525812341526,0.580546696840582,0.316491097917631,0.353193697770342,0.374586441517439,0.249496543176509,0.0291560866539743,0.118743742895646,0.180486187190169,0.141400604349483,0.171466123950461,0.0488763546754438,0.182143855820739,0.0937935640824538,0.230044830220181,0.0893543342463075,0.0518068434512621,0.207556371522003,0.244766009264819,0.18595057790949,0.051274342962421,0.0819215169703478,0.176063362672981,0.37888300185344,0.160315715200218,0.140218926153,0.209075166517778,0.207533958389866,0.0729193524719845,0,0.120691005667722,0.104409128920896,0.180898978315865,0.297748600252737,0.374811663131579,0.725234995914451,0.332990170185169,0.391125150061818,0.149251164040623,0.110048210166964,0.131943901915216,0.237499232091707,0.409671629899041,0.668298071710468,0.38357391488503,0.468980568357917,0.344574274092601,0.517339542315311,0.436933071171967,0.401005522298416,0.316340739696985,0.318912896806551,0.341075534713277,0.152246189491319,0.032062743518247,0.0871207125502158,0.0349686451131746,0.0732687325935075,0.0894584567450201,0.0318255775068739,0.0636580015068439,0.089681528965936,0.0845024942120876,0.119089795087253,0.0451788664161399,0.152555511003475,0.205903832240384,0.0921937643316924,0.0313415067183436,0.0619244851695629,0.0607512402894686,0.111722049581572,0.464890402289933,0.2358960242602,0.314920756307895,0.155644141589737,0.149284025616547,0.173917002512763,0.0416700764046498,0.00608722490165536,0.0423138416216782,0.12131958280413,0.191070938509409,0.135566970632862,0.123925477659283,0.0904982792384584,0.12337728994684,0.700373995743954,0.42760599302241,0.4564626895808,0.464631030258507,0.639159239896868,0.51886806271215,0.324687465482816,0.463656357327094,0.399421793623307,0.430302451296209,0.470909408547734,0.374457108629637,0.383518251147413,0.409480007312964,0.376183382142606,0.245569302847538,0.0928868256867548,0.0870462682334988,0.0829784874311954,0.0419571456866025,0.0637288528998272,0.0172497641786673,0.157548903492342,0.018131147397661,0.145344583573419,0.00685061968410032,0.064768924764384,0.095656730060356,0.00844879141264538,0.0489467030426104,0,0.0623021123058051,0.136723351994769,0.355874873454442,0.15208416726367,0.108426794976322,0.115624649979399,0.15174775551701,0.142020213769101,0.0924099470590028,0,0.090466421978018,0.210879843398381,0.0698109126019881,0.0943301393099927,0.214285617441974,0.165500707796492,0.174582225507907,0.281841121845914,0.204426748679438,0.450751289236213,0.622679120887125,0.558255342304029,0.444467327755133,0.458650923407319,0.364826597556678,0.338642156479212,0.711073438021618,0.505752096411728,0.289142685965212,0.481934099744512,0.282582477143213,0.399873348768491,0.106631527792607,0.249511188774053,0.026912554697225,0.128490888579441,0.122612095053198,0.112535893409678,0.0484206823028885,0.0624326734847338,0.0273252490891421,0.0782001030328839,0.017820169552109,0.706522898433554,0.260966585643085,0.365945548533496,0.0519993751948669,0.111153057773423,0.0695027948051998,0.0603871605703934,0.170353600314955,0.287191502527244,0.571276612665843,0.18037171492215,0.044222421381497,0.222325207570947,0.212175138926909,0.189589094889403,0.0788798555401614,0.133980259197304,0,0.0489505799225724,0.0528994492789826,0.026151332002147,0.0380643889498474,0,0.028489374498367,0,0,0.090799557245365,0.0377647582306049,0.137994995210738,0.0731652003443427,0,0.102510930581789,0.0351218042762224,0.0223595183190801,0.0312219408782494,0,0,0.050251369181748,0,0,0.112421621991001,0,0.0620600648498133,0.039016085658501,0.0346955910148936,0.165776895910001,0.0326625677794385,0.0560571715545392,0,0.00403180675146392,0.0751491315358969,0.0386438889801289,0,0,0.0873685187025869,0,0.0727801056628022,0.0693655218592003,0.0483275026518211,0.089808879364632,0.200285065227367,0.0406180990693664,0.134226573565325,0,0.0470835798539648,0,0.0627437894538982,0.14845825945931,0.0864139483221354,0,0.00122356570497322,0,0.0559077554667443,0.0227538369884226,0.0335546823825582,0.149430692327114,0.032604306429362,0.0578640016762985,0,0.0302199967774266,0.0920213042916175,0,0.014512551524381,0.0778586206839346,0.00984568283884131,0.0033370081572285
"16",0.0672528479491513,0.0790212657093293,0.119022322501523,0.091700132951411,0,0.101017560847211,0.0838155893238248,0.0833709185883827,0.0221060908180922,0.0612715945735656,0.0612742528644337,0.135653076390522,0.000398112206626729,0.0727230006141249,0.113232427843813,0.271236433601831,0.0681902277457515,0.123271298212239,0.0635883894241621,0.0460589632574233,0.0135501503450678,0.0184496776592349,0.130985724194298,0.045368074192496,0,0.08508247415183,0.153906197031478,0.0219942387172312,0.116115477204444,0.132733517557198,0,0.0225003235089235,0.247547031357459,0.0832791367054776,0.133808239498102,0.152818545006282,0.137884450662824,0.0856224841289691,0.100423808579858,0.0517585115485963,0.104877094458547,0.040327862465833,0.122834846701561,0.114081548960993,0.0929137726815763,0,0.0187876975691973,0.0591695910307065,0.0695777886427935,0.0655805422793233,0.101719196688611,0.0874194975230149,0.126060216249508,0.0642398932395174,0.0652163038587709,0.0942316601941908,0.0359034244324227,0.0621018441564558,0.191829994679052,0.0588594172083047,0.0147140059962695,0,0.066320538032546,0.0331145709502549,0.307082217545575,0.0868412718047607,0.0874829593429926,0.0167458502688538,0.0135520790876484,0.422512671772678,0.0900824793972574,0.0442327707790004,0.0774437847605923,0.128044139707474,0.0690981504073493,0.0566464810758678,0.0271484791724874,0.041821088584321,0.0981574618304471,0.0673868707203969,0.118782555125154,0,0.14833393511357,0.291307655226387,0.1577827377275,0,0.0895857669771998,0.113727726662041,0.0313511844445444,0.060546117235556,0.194018251745337,0.165596569905378,0.0813644637521392,0.091442265788977,0.111567585216575,0.183515025540847,0.0277179117858701,0,0.0958521744647223,0.0943874676359292,0,0.0464870018248895,0.00226752605020068,0.0937077563831517,0.0838101335504806,0.0661819900689161,0.0734997450839747,0.150235700698642,0.0667867687474311,0.114289528416605,0.0655349947593085,0,0.230387379986416,0,0.140523950820473,0.156610924042225,0.0869846329554279,0.0299350110148356,0.0108935413282755,0.00379038498265168,0.403609288498507,0.342224302325838,0.242023611412738,0,0.0320571491165751,0.16022476487739,0.132119705681066,0.0390738086473368,0.126581120283771,0.143647794576437,0.264826773287378,0.118266558885124,0.232878801879218,0.331711311057947,0.0466352680368145,0.157485175526676,0.103169501535479,0.151885030090191,0.108879345358595,0.0738187064180343,0.108336086982282,0.0761761899557097,0.0825917652934257,0.023310801793804,0.0167947153803138,0.124744544412384,0,0.10110726703041,0.0299002558330605,0,0.0973060144626134,0.0463132577388452,0.0864670639222054,0.116901162861754,0,0.0852611253162244,0.043437512544827,0.1602937273494,0.211713948268441,0.0327448049467573,0.142251837261009,0.251318696871613,0.0554388141903804,0.207440021575022,0.194737370175394,0.247269710027682,0.21922963597431,0.0257827571169981,0.0431123189905388,0.285566005032418,0.0921950521858827,0.153102075022832,0.164913612490223,0.0747037864773259,0.263813827077071,0.066907917427677,0.0647423249930817,0.190864443280711,0.137679676730397,0.174529259524943,0.245060914590327,0.292048267926339,0.257831703992609,0.0459844223245092,0.0851897866930463,0.11159571157066,0.217595363342714,0.0757162920506941,0.144754200073419,0.110829927047338,0.0710205605671822,0.0556019018436361,0.0682025488186198,0.0858648070249167,0.128231495959882,0.00483124700007398,0.108927190708631,0.0264026504773703,0.171565033846784,0.0621140590325207,0.121958370300462,0.0738739536646248,0.176000660785827,0,0.0575935949310171,0.0834026434764147,0,0.234215956518665,0.0458028257050315,0.0761924181222498,0.0793992151898711,0.358911874142956,0.0396503532006927,0,0.126700329797125,0.124338076324498,0.130915796437563,0.102123723026606,0.0452224879947575,0.195393126232526,0.0665073861062831,0.211395181381932,0.113981590586999,0.355989320865487,0.475367368585686,0.129837484835891,0.340368049744838,0.303896282407401,0.565767140714234,0.227267518664979,0.433080533590763,0.134745978584815,0.238862484721124,0.150548055676678,0.172605906116389,0.155502937807409,0.111607791248601,0.156467786033774,0.118672093384245,0.0466295627396146,0.0551533387415229,0.0948073978546153,0,0.0561310422118217,0.0286120783226921,0.0379660538605477,0.145786308391496,0.0100237712076917,0.0916406341965342,0,0.0190509143648204,0.100522570488244,0.0272756143313387,0.00388766447625283,0.280392167587019,0.165841273552631,0.138335192033818,0.0751033021080312,0.035196798905033,0.0872396349810736,0.0341937623884322,0.0663794308251096,0.110365594391505,0,0.0967590758088495,0.0895560200122278,0.0872563165180026,0.0150653136132761,0.141592326302837,0.130958233377112,0.112668262523755,0.0465005490352604,0.0996614514693856,0.100959706764701,0.128016798630115,0.302805133969078,0.212384394873501,0.257406621725992,0.142662953667378,0.0913536402148654,0.174539569912811,0.114876860337334,0.14561937868762,0.10623245591679,0.0744149847098344,0.0883996794813718,0.0734936839710861,0.158200952919244,0.0874764877162914,0.118980291063628,0.0729130862130559,0.109327500715507,0.0795907828776309,0,0.0735071620747574,0.0189050145540295,0.0599298088780131,0.0210552369969534,0.027680064337446,0.123176817114904,0.0928546010191177,0.0301280460832466,0.229081347188335,0.130709991895291,0.075568327244041,0.132344758885854,0.130004667027039,0.0720186166434849,0,0.0481503399971164,0.075873761851029,0.0363350491357847,0.0279665710144021,0.155299934429045,0.0043151715133908,0.149336643761371,0.167506215178778,0,0.0802584556929514,0.0624031475483709,0.104555969194986,0.0652309128426922,0.0886359270564497,0.158232132177136,0.420163083755481,0.534473952895775,0.432518053224609,0.17521976087138,0.254707307182705,0.195758021859088,0.120777098143533,0.153554910747954,0.140631998609663,0.176523809948819,0.00178296551507844,0,0.0686269335848812,0.104348665067344,0.106458696942279,0.0759843765242019,0.0861415158889843,0.186590769757858,0.269982051233264,0.129698469112298,0.101981292841917,0.0192718511243567,0.117220055024575,0.082274720992045,0.0529623039200183,0,0.323711687214438,0,0.0492171349666945,1.26363963570978,0.67038671386699,0.178998440410325,0.152665098660193,0.00637958920250588,0.0280999927109766,0.140226863404472,0.123793777077839,0.189261726917271,0.139167411792324,0.0337921151609814,0.135617513930375,0.120067391804086,0.0798890296959421,0.0671357121522537,0,0.0877591400692047,0.0240499039602661,0.347296686075446,0.413007926290656,0.314885155334649,0.239897928670126,0.177079040434878,0.123710837459521,0.131598031896945,0.0497747488624496,0.159039517524316,0.101323516458056,0.0902383769380236,0.139397145164723,0.136726912133716,0.0467786538150542,0.0549651216795782,0,0.154264500557573,0.0565791740219198,0.0280616171564471,0,0.0326498582533997,0.122905332195272,0.00825243750602323,0.0199313147264207,0.0374521671499506,0.0372888865678308,0.118130470893247,0.207311823280959,0.0649217052233548,0.150014996678163,0.00436411212636224,0.0443293767095439,0.0735688690766832,0,0.17815559331765,0.114136302403444,0.0339071788829215,0.0735916975876064,0.0235933700908044,0.00866144433753242,0.0726974154308907,0,0.149669531588182,0.260190628857814,0.354595604609451,0.666087081793343,0.396036011281548,0.202573752719725,0.115641580749175,0.128555902097567,0.139779274049346,0.127524849182013,0.138973761590522,0.242840971157515,0.192162805417375,0.0538594600680046,0.0591424250278497,0.149826566525933,0.076402090765752,0.0470371119700037,0.0508022752903673,0.0813913366652215,0.116321167316116,0.0930500681514262,0.10088761685703,0.163053176530972,0,0.118140583287948,0.0283230335659604,0.0577362112401705,0.0734303373784942,0.0690785138780851,0.0678024037134364,0.118410851858802,0.0940331181298329,0.34841247372954,0.135303391652103,0.102113732080782,0.118951639657192,0.345106466502506,0.0499635316723076,0.0577874940926102,0.00304267013808034,0.0219330841500697,0.0825743380280097,0.301054963688902,0.13638708832055,0,0.884148144378365,0.320010345896797,0.305027975847857,0.478529375844225,0.0952534385731943,0.198791445299683,0.0832897208859468,0.098092253547409,0.206395112049158,0.30378128155652,0.158075090325775,0.310707590036625,0.491646327881432,0.212783918747382,0.247124764500233,0.124631625915823,0.0911592459886843,0.189790921789948,0.00798990255054166,0.0971166505748206,0.156571749023751,0.0297518679516037,0.0147170676537857,0.0579116345505098,0.119732651281069,0.0216444601616507,0.0456809893421516,0.0217912832273926,0.0523847730102919,0.0626707878255348,0.0402403821823785,0.149807013989884,0.175284017542896,0.338822348069186,0.1354497176739,0.119149218395412,0.121085039400699,0.0513560179756216,0.00522469453362197,0.165021235100544,0.227841222687656,0.400215113914902,0.308867465626221,0.336162957840414,0.633645876829959,0.306229574862388,0.509554648413849,0.160044767865704,0.167884977668494,0.218634160512579,0.2690946584865,0.359555914457532,0.149567005997897,0.104914272080165,0.372811342161704,0.117466512203187,0.0794992116192226,0.313558430958738,0.373479159332562,0.194364893940023,0.233698129530308,0.113911123035984,0.105702327972055,0.131956514505541,0.0715199037011557,0.122214052448412,0.0642034119400443,0.047087358246391,0,0.0258062454508002,0.0839301868916863,0.0205003332279464,0,0.160570877611775,0.0995380972382352,0.11503275306757,0.0329260813225385,0,0.0853305524675664,0,0.00435118686329627,0.0787769973747104,0.0983724371635612,0.116670081461067,0.268321778183388,0.0763986056114287,0.101665823822256,1.25232764373103,0.258768388984242,0.0579372781285596,0.0580001273312009,0.0599417595268725,0.121630831136054,0.0740885761456088,0.0942051058890012,0.0348441916502426,0.133387058980649,0.0677564517485105,0.123528783253309,0.112132482692074,0.143439656030074,0.0229780512511571,0.532528443352325,0.573909987367982,0.277525700066296,0.281027878825886,0.269262928591643,0.208891894188846,0.406874394109781,0.213390856093581,0.262142097939628,0.211400541080129,0.1778802713305,0.18383249443773,0.0413000870409314,0.126167209872746,0.240773305096507,0.10790444988915,0.0369287541026272,0.117588242045832,0.0503994832425785,0.105618972406798,0.0602085948244842,0.055596078563283,0.0747778089514306,0.102892091295466,0.0437188423915915,0.10347038889928,0.112280325365341,0.136110740153536,0.0627650628087331,0.393126374950769,0.0121919889499881,0.311360076940588,0.0696704991490497,0.174385195817198,0.0682380431546493,0.0335970382432043,0.17663467507573,0.0089930813977917,0.0546969558399964,0.0338174575265945,0.0761207561289841,0.0992718362752815,0.0870951249251712,0.0386223560695983,0.062122165221738,0.0803524236664976,0.0847766702047178,0.128025516987235,0.0806186821520193,0.107572706498909,0.264244625584915,0.609612475159092,0.55801263768954,0.502029160297794,0.444249553914869,0.276598450094889,0.214655627023749,0.257803702818729,0.163708990958925,0.0675983337162445,0.0597029344583473,0.0125362029642907,0.0448973795328721,0.13622037389274,0.0944397547192666,0.121090576533876,0.138672102422316,0.151760479722704,0.028065528141794,0.0271707732501707,0.0542544535431043,0.00102850327937738,0.0182368132261795,0.0748132667166682,0.0892900232862736,0.127036032595406,0,0.158576718519129,0.0226501795858215,0.102754294417724,0.195321414762167,0.0870848148397878,0.0715358203820058,0.127298261477186,0.0716363213795644,0.096024896503965,0.101134629786145,0.0859815304523244,0.0922267054820063,0.0219019263968945,0.0961714369953205,0.0613792634564311,0.0873032113429572,0.0902026768324611,0.097372539328795,0.059760714984395,0.162583894183021,0.20557843496788,0.540590457841457,0.226905822539227,0.290536791436519,0.221111792606638,0.168431089536995,0.105781723717003,0.207439004085989,0.299756102343147,0.113578698556719,0.0487062499258414,0.0365894304254819,0.0366486258319021,0.0563116713775076
"17",0.102826162460206,0,0.0783039865199847,0,0.0819424929947004,0.140724601672829,0.0787382123295305,0.0520905374642099,0.119715737936876,0.0832105623140875,0.0583831037470477,0.127761043990638,0.103306218245128,0.219783771165662,0.367639945649118,0.499330160677293,0.293704033056026,0.212576948386627,0.181626267753838,0.216464360407985,0.0869037407597805,0.104232535577833,0.107350810240741,0.0568728473676067,0.106927432987139,1.46486827376652,0,0.219956963419888,0.232224535155385,0.516100619936638,0.0549929167492027,0.194314284915601,0.254375247249034,1.00820624218186,0.574118931046137,1.05596275766739,0.225509850732581,0.312828269914225,0.232440550114623,0.337698389776391,0.680578144669321,3.5764986934401,0.781905106354705,0.137702622624352,0.024109809192028,0.0177409029263238,0.0763792400363812,0.131166145745984,0.0724293575892173,0.0484658865961454,0.0816402494596924,0.0748124759707731,0.102795861109078,0.06919421219572,0.00164799784700921,0.105385454044963,0.0533895566949469,0.0373399505233675,0.066089887802544,0.0814403774326178,0.0258053945037835,0.482335212428725,0.354792004013348,0.193489623758242,0.12871246649605,0.113069103172428,0.113282034369849,0.108817342970198,0.0455285652673493,0.0874157693812523,0.0983308008609344,0.0982763379702994,0,0.0112835868389759,0.107197863536925,0.173117960045191,0.0220901903107981,0.089574778172399,0.0210869204909721,0.227296637280037,0.0427541565213404,0.191662904420133,0.102405309679612,0.271162500710235,2.58810541375371,3.46214593818035,1.40091244721882,0.206130909149712,0.155095165077584,0.854408179004978,2.04793920671137,0.0709495912432873,0.138859944121097,0.0108135910177147,0.0719857810437467,0.0520566170418348,0.0217886121668715,0.0723180966797249,0.0531891058167825,0.0664246097494328,0.100458518149851,0.0543036022177965,0.144265932670433,0.0519839609536472,0.032925310112708,0.0637028189007505,0.0872043151559961,0.106593327340099,0.0392372663633421,0.148949257009121,0.292508336951375,0.227038861728637,0.285481888696654,0.118294138602815,0.0688034156185248,0.35979406755805,0.16394901095127,0.123459074980179,0.0373370704188769,0.123449621235262,0.833499297077684,0.050232138096669,0.111213010386298,0.0164729012752688,0.0868574661814324,0.0907450854299157,0.205892674193482,0.2009774290542,0.310823525344225,0.128090292061083,0.367520730716875,0.771395506016203,0.271685035484585,1.84533301349784,3.47713627962025,2.28668542839856,1.96153134841383,0.854293093926239,0.269437768521476,0.084068046291906,0.141073193274338,0.138059188073152,0.160010305666387,0.0371527015732614,0.139450715305775,0.0824989103737829,0,0.0473730145056765,0.202755679910938,0.209176682322962,0.116199666597257,0.0949472464271448,0.0795918579235656,0.0887504443543302,0.131693382456033,0.00797237765027071,0.0978665389942273,0.0938499213315024,0.500985595024025,0.489598935718706,0.303960754331323,0.225235388783748,0.285989454569082,0.187374759546548,0.181209491218695,0.129285630836086,0.0388568878145451,0.120102329963906,0.0926810035050519,0.071313941856527,0.158457746729242,0.494620284596341,0.747521558386988,0.208436223669704,0.280891634507477,0.514485092104783,0.183899781130509,0.468917113752529,0.804440722261465,0.751016565680767,1.08692183346878,0.311349346795954,0.174963418318802,0.255189093068361,0.227748901637692,2.44807385716866,2.70408923042173,0.155272718686902,0.0908612886018361,0.0776773150092573,0.153772637695133,0.143276661700873,0.0431362010440312,0.0816402362725157,0.0341830847106562,0.220535875117675,0.222791575633141,0.113757189304999,0.109239748310099,0.138886603276717,0.196578797329305,0,0.0692861258209208,0.00120919323206537,0,0.158454517938737,0.155838369102077,0.466898191964799,0.158895996157939,0.276891142641514,0.161905470488443,0.0840911267350254,0.0527972587181318,0.148699951772615,0.118712805565592,0.0474941017619491,0.0895154559068707,0.0235171798882587,0.381760855128987,0.141420976743804,0.0941277791274946,0.144537847637353,0.121609670214458,0.0166089943766631,0.106178959208235,0.0707922371440336,0.200080822941138,0.177040772167389,0.129061026535448,0.108016358620552,0.120192836498355,0.0802132127541724,2.01802154786304,1.5437878760087,1.24219143942069,2.3816981875066,1.00194461070067,0.138715945762333,0.121524488612249,0.125987196080478,0.110070925786764,0.0577693094785056,0.0771493140593472,0.0667902266610485,0.0297412809328495,0.124277858769758,0.132511004034183,0.0803352939083144,0.0173537583021673,0.151228114109938,0.0700566534551287,0.189992678531914,0.0467763914645158,0.149611662452632,0.444344564261264,0.149477976272081,0.325856722995482,0.106232061573149,0.0225404107462233,0.095659351083915,0.0913273234890316,0.126536890117356,0.204546915649647,0.115653852543612,0.0583516449407052,0.110657280640141,0.106101112888974,0.089239807892887,0.0541643907628572,0.103023508622393,0.0761150994379109,0.118085300905842,0.57355267743708,2.59845693326418,1.51538268591288,2.50694933184058,1.99313069208912,1.57661905829772,1.73885886688606,1.28730943408576,0.326081819189812,0.203075207964543,0.213318255228276,0.118832646022305,0.102507096879254,0.104321566943908,0.122728486372854,0.0511688976501877,0.0270509358038726,0.13308032960614,0.0597601339752645,0.119658761912363,0,0.148177728595296,0.0971800188318727,0.110906765351441,0.120059804086355,0.0292474412533511,0.174932908210907,0.0896095593894756,0.030441726744556,0.407441287842141,0.240535607587574,0.249956663682042,0.424185763666772,0.0661603077032931,0.158157637984628,0.0929922051672876,0.169940986930856,0.079415097450111,0.15784009016755,0.128781380258113,0.379406400707473,0.357726593853792,0.113191928244587,0.104448836629488,0.125393513559896,0.0816906552793639,0.176914907555061,0.336058415801629,0.108842181753159,1.70898073145138,0.822216063036346,1.07424165178431,0.709899335612273,0.424531346310236,0.189953234666952,2.17385595634925,0.271153138210912,0.153867217707167,3.28880286224132,1.51870234925485,0.0917768302380668,0.0189946653054203,0.00490927726388221,0.01811653386762,0.0631374820770361,0.0280014650581472,0.0632705032085745,0.090129478167306,0.0928686914322206,0.152324298000692,0.0521585576778089,0.174972172151682,0.0594642281991902,0.124598677998044,0.112396755331161,0.0611757020317472,0.166124864058796,0.962542276971373,2.82985657095273,2.89604240434585,0.054723883095958,0.123734736311842,0.154922874970422,0.194339107259255,0.0597075682082191,0.0363087970807132,0.0552736263302117,0.169966957102337,0.0926105988480963,0.205260109527807,0.0478737319592533,0.0448679917761649,0.0717688455756858,0.124529679998996,0.0394493274973252,0.308607676762241,0.191073839797358,0.102631475429706,0.192911373482581,0.277593753257664,0.571189162589065,0.877750236840848,3.35715556155798,1.72908200859219,0.784303644436714,0.180755186838954,0.168920932167957,1.90629268904159,1.76630013757341,0.478116939433634,0.0469404578047867,0,0.0258865845904444,0.00160078269994954,0.115085827777628,0.115004333184352,0.0284041515138066,0.0318064440086753,0.107915864191317,0.135805798859371,0.232438864947466,0.0735255579585629,0.100758880709771,0.0934362515715627,0.136406106786699,0.0855438869357849,0.290983879048071,0.603891124727188,0.250666301474686,0.0827326222870958,0.142768244339782,0.318533818306448,0.184549062386918,0.130964544219868,0,0.126042932111139,0.281003784429838,0.230818276963706,0.0883932999710625,0.24813828061745,0.316925484785995,0.0587505323485839,0.0253308601691801,0.098054669608339,0.203178596083222,0.199013892480186,0.101717994096325,1.05271020940534,0.670260798249339,0.284793584691856,0.198854354980785,1.24272689540856,1.59168835092382,1.59508175365743,1.4476361423959,0.724859546368928,0.0683822660858717,0.002500092586283,0.132337705530386,0,0.0730883834471761,0.0907420249088336,0.10698174879171,0,0.0998125884164396,0,0.136290480617212,0,0.0473038594438963,0.114934017845008,0.187232171292625,0.023272688174452,0.0136228583769771,0.199828592340992,0.367040998420939,0.256038441904783,0.0805289162831741,0.0494022830472563,0.134846179048353,0.178564050429314,0.052988105530774,0.0512504025875396,0.0464094634891002,0.0745278024853166,0,0.245108534133215,0.376078614436947,0.431061405850014,0.197990678723134,0.227227393895529,0.20723339693778,0,0.213514916360123,0.195223370659691,0.172315706158738,0.224264663158128,0.370552997352661,0.30017357632359,0.383764001035416,3.17965350787675,0.763937416543179,0.175578929454692,2.56946216068439,1.88284659061846,0.18543524523777,0.185268999809616,0.0367035122429744,0.133376564965619,0.0583698009238442,0.139874679422544,0.112134300793086,0.0718131612018118,0.133871623870731,0.0764093326640889,0.150407550986302,0.223392113266292,0.116791223186923,0,0.128668762265923,0.107601244527382,0.0635931978174419,0.111442991594318,0.459777242543989,0.204067495066463,0.0850767785337193,0.20883523355098,0.163707491258679,0.0083292905484216,0.122575437682371,0.0651474949694566,0.1035765593416,0.120095975440652,0.00994414557848326,0.0789882647346302,0.100549138088772,0.135271424105601,0.119338774761663,0.137591055095218,0.191286686181716,0.0448909555600679,0.0544878243676343,0.00126479273055172,0.284556908943013,1.34691470565138,0.477653645046483,0.267225495925444,0.363561957868027,0.28143547414942,0.351703311844163,0.222929256114267,0.206108657493708,0.0530474581065874,0.217376320337623,0.0754848227324598,0.118814960785554,0.146756743655201,0.118093982958112,0.0424360908709314,0.159822164522147,0.146092423609501,0.0203224409278669,0.129218110401859,0.261405426478785,0.149994002113515,0.0640775360869142,0.0862239373038875,0.108520251436853,0,0.0313222247867202,0.119915716407382,0.47879906709776,0.290528854142088,0.105001548499936,0.125713392802538,0.37537940329467,0.206448608588242,0.139288451784504,0.208397787546874,0.158752851813268,0.222719052479449,0.518506224553589,0.117319884495159,0.0466612054411355,0.0589512437121069,0.10015915242366,0.0972088135131652,0.202358620280567,0.19928830359239,0.137820597704161,0.0862693534079062,0.172348355825747,1.01823392583661,0.208656660207496,0.232058652525613,0.189182128390067,0.120419111182241,0.205529479168083,0.327907499163513,0.0951916630929688,0.239343615469415,0.111793999845465,0.125036898001636,0.065010142285976,0.198505594970713,0.094151995856838,0.0903760609221342,0.163313851616418,0.092188689924511,0.219400414123817,0,0.229518936748052,0.0942045724563312,0.120657171287035,0.11048491408665,0.0717580831154201,0.117423656124251,0.040234658781567,0.11759088844471,0.451199872378554,0.480349629105698,0.138441083326932,0.116890307874198,0.230879987955945,0.10380746473276,0.123417086075012,0.0747304979518185,0.265902606422042,0.0501400193404805,0.131136756241379,0.185698552320221,0.191880902583745,0.0405382332472403,0.146150489677241,0.0967435074276885,0.075303424107321,0.0180605594228544,0.255230542484412,0.0860806608395812,0.189330647081259,0.222241572480627,0.163558120572738,0.124796634910677,0.130736911965665,0.123907640164674,0.0631911476890321,0.16138720835199,0.0718510062246064,0.352802149613754,3.5481521943867,3.11694932272406,0.188305319088172,0.0270941120266442,0.161946755405553,0.0760337532413255,0.0817583335545459,0.125099111942815,0,0.196879266381478,0.155510233069856,0,0.155717568274802,0.142683518384468,0.0968300552322624,0,0.0501111417096395,0.141620374494558,0.57751439166569,0.459724561563435,0.735952750170639,0.274058849559059,0.226963681279922,0.119689253831086,0.108481658396693,0.0530708302999113,0.168144439392884,0.0617839027024666,0.0688273949240605,0.386682006217048,0.22058095107267,0.0356587867639991,0.134119668241536,0.252146327933492,0.118623493413461,0.105666461724841,0.374803338133492,0.725715299706357,0.599158885444557,0.934831615363057,0.383911216945802,0.333653034289238,2.46290158233475,0.294308671376878,0.14687513407984,0.136296441004307,3.20168484594687,2.43068212197483,0,0.0508929681048019,0.0558775005405088,0.125822570315345
"18",0.317434716499589,0.428802593832776,0.515954502685977,0.358172233405655,0.466921618754353,0.334843739135006,0.49573521757592,0.410136209948522,0.425477307861209,0.296592467433826,0.520457444289354,0.344761747054125,0.413836389790654,0.487149620673169,0.458776809777812,0.256350991138064,0.408263170384128,0.4056890092229,0.405277000874901,0.517664587287521,0.383577221165097,0.446037233258913,0.386069310033851,0.380015486144086,0.382931503047515,0.34997169221651,0.326607252650951,0.397835699464536,0.437641926157545,0.417203970143591,0.419366124093781,0.45020190367763,0.443411048132555,0.365886066152516,0.47376882692549,0.391469500417388,0.416034914919623,0.397548499590863,0.370918759767909,0.427584425186763,0.283237969064424,0.377643462754236,0.330568299391658,0.422570478795856,0.323106165212876,0.393849291982853,0.393963822187594,0.421154681895205,0.41257654710873,0.225757130413717,0.382124576132442,0.457022992415265,0.353353682355965,0.366242726682708,0.417504902512126,0.381004619732041,0.355418963403016,0.424163823373306,0.394507256350923,0.476908614975979,0.44796919378329,0.446321927829195,0.291664050065685,0.373780008062878,0.347078783277362,0.399153178855344,0.339412838021867,0.311534684538268,0.331379997465987,0.412369284474363,0.368481012473896,0.408426748470472,0.43284637986033,0.354369865689274,0.415848433250278,0.424715357019302,0.420794567075584,0.300739959302055,0.360028735892756,0.391441611908504,0.415446594357961,0.422230976093199,0.413161606093331,0.370800636578608,0.322166383168455,0.493268311212552,0.544079014952149,2.04522250395501,1.4463523763452,1.32749279743989,0.883694878474984,0.881248364508374,0.763122491942823,0.687873633374407,0.60964558107346,0.30276909887005,0.466993182286324,0.471729528733963,0.459753308690956,0.387729984346354,0.393972812909529,0.34241310918941,0.378012703576818,0.334871798274059,0.341247195753669,0.445615243758565,0.329711243404163,0.398227234839979,0.287516447781252,0.386903560747674,0.60285099077295,0.744306688049825,1.18235494363925,0.741088995527186,0.724478265583204,2.33352955170173,0.805872701449819,2.15731416134954,1.37152436069444,0.927350397877371,0.787182326191548,0.397977060739214,1.16602900593134,0.539923591761136,1.06849307058005,0.415319917803721,0.495762689757856,0.411030086448198,0.519859529017722,0.752221573505665,0.653996466764495,0.837205104959287,1.93415302727548,1.75254549179341,1.83799129025497,1.08718475343952,1.2132219290809,0.978524802898048,0.706080473522353,0.4863015896536,0.404045576847468,0.481772241703331,0.44884205373622,0.361874573409334,0.369383932654176,0.424506052096414,0.393113616631325,0.451003532972739,0.473334609739414,0.385183926538575,0.383492848773861,0.396349602433139,0.457253865620274,0.403323688359198,0.336815422878225,0.442093327381144,0.421690112668189,0.531530311904606,0.57399130252452,0.51128528428063,0.602254300554613,0.93396340363509,0.921969865545782,1.15129075587317,1.34108492103893,1.29305362360288,1.13822289206171,2.05914800593273,2.0635809979841,1.73828600954118,0.91537623110114,1.31091416218235,0.686477176628013,0.685275636404381,0.748123418868909,0.665241862503771,0.759318856328249,0.721150185993957,2.03198523267181,1.88850122544417,0.920646310507777,1.4860979739301,2.61526805888882,2.19316790185081,1.00024142937162,0.840981132292722,0.659172969669076,0.616453622697637,0.374972197024903,0.39599150627359,0.485671390023857,0.409549062635616,0.425355034615505,0.354230638538284,0.412233323263748,0.545725888954467,0.415952095086759,0.42974704560305,0.54489034379481,0.416230362456097,0.466923356954845,0.477293986412126,0.396991223431761,0.385226269357381,0.349193171160526,0.576740358199413,0.703723010148751,0.65429835950378,0.411416821962628,0.538452075479731,1.38728554431817,1.47600860175853,0.448322588731726,0.417394383472345,0.519240118659837,0.465077047680283,0.558510215589189,0.679535569545845,0.542204458042993,0.480100958065916,0.446106591316173,0.452739348623057,1.27706893428207,0.709089647004022,0.479450535637476,1.49573368113893,0.695155095461739,0.740420914146021,0.984151911251797,1.71918505743687,1.44130102953849,1.48973057010817,1.40753426027113,1.42871966381539,1.29996949271613,0.812300881479488,0.647062917281336,0.554292560660412,0.414423140219746,0.35440967726261,0.310607029359622,0.285140763397546,0.479378854472339,0.29804886330288,0.467220414923566,0.372700219721258,0.422862083292335,0.319358229498979,0.416820146708018,0.488517921791304,0.404002334516042,0.388103276739542,0.298258425838308,0.557817594517678,0.614022882133796,0.649768404927561,0.562351257307413,0.740943060056059,0.58871486349507,0.543173261280377,1.27263305738199,0.45875637124402,0.466180064581537,0.736388069612358,0.699951059462188,0.62539066047517,0.706509173473014,0.700744859284932,0.785273451429919,0.876580555568697,1.01238500855946,1.23552614553944,1.15736223463468,1.00404485332988,1.05599586017772,1.7045183212805,1.83332742219464,1.53218566443492,1.49001210370104,1.37969062894127,1.26515226374978,1.13824824454434,0.885830718708507,0.725627510134441,0.695198813197226,0.58411056443879,0.835113184841101,0.658679509317222,1.37757945847141,0.458116442193721,0.496166582063888,0.468060119274689,0.406234732145033,0.424464378326208,0.336902358521775,0.471819614537458,0.399643299579557,0.430032460510044,0.420758506952324,0.457371940732015,0.369263794243636,0.415935492327779,0.431916269675394,0.696830737162142,0.519125016794914,0.548050788225734,0.457442047031833,0.388577573802436,0.517449247779027,0.921771733962899,0.707366227369906,0.529492490267863,0.623484224099608,1.19772818693494,1.59055400510112,1.41631136788152,1.0153622387432,0.856107342258195,0.946649909679394,0.818125355561318,0.64127098123414,0.658295944133277,0.615821666981674,0.665969467660968,1.34062351191484,0.738470026716527,0.814610570832913,0.727931807774206,0.927765578038298,0.80673936148783,0.793753319319526,0.888480858718962,0.80628379228598,0.608478244560952,0.464293399340564,0.451422752150741,0.492175754718706,0.465488600943238,0.470248605325459,0.473596413653202,0.384766030184453,0.443225974891527,0.455172465390878,0.394157467334352,0.443563522012277,0.374741780089237,0.433435726262635,0.387382533403998,0.430587699128914,0.277045026823908,0.570457885257488,0.397434743006875,0.599516230727565,0.46136562700958,0.44702270233074,0.904123759452483,1.51476443500567,0.637151170660344,0.515234730199502,0.529660029652345,0.559378862465699,0.47616669582763,0.495773039614123,0.571416751317717,0.882021178925878,1.78872965570985,0.801467875644316,1.12721043686724,1.47472972115491,0.899855413333627,0.984577957410181,1.77713313678177,1.45532032730041,1.66717888046796,1.94413221014299,1.81813751609495,2.10523359075039,2.0129605592291,2.35079225104538,2.61696591735073,1.96196705885317,1.32160374458078,1.37019693395115,1.37297225656757,1.32805206698161,1.0504064771147,1.50283685243839,0.48580731297953,0.445609795252872,0.510143834980862,0.557321510194983,0.536361715823725,0.470822469413043,0.44391351113774,0.420027782221988,0.490071169433938,0.52107087368493,0.435923479971905,0.404028415672269,1.22957880867874,1.81119179687138,0.650342794519073,0.984591670036182,0.664934130349133,0.870231103476962,0.491040157192361,0.587192522033014,0.92182213150698,1.19752531367778,2.08546441208663,1.32311202499388,1.27925022260785,1.00738877842377,0.55664360877722,0.433805419707425,0.916395038446162,1.28513936861805,0.476435316777489,0.647206953072573,0.618618494653996,0.902659110464246,1.09375949310249,1.96838223432155,1.26017119502403,0.854592751685198,0.722238326289103,0.730753196632004,0.731581453357729,0.556235622198525,0.479280348737419,0.440586400776209,0.440943935928778,0.386169151773425,0.491522429199919,1.08450945406687,0.328765165272307,0.464563725270366,0.325869795065092,0.527284130046909,0.356682546176155,0.482049110508167,0.418483850219151,0.426615407875404,0.40077794031643,0.506360409662605,0.35752474370233,0.353601124729453,0.399933799364417,0.527765476494549,0.51189749421668,0.521404815788208,0.842211901504443,0.62356923542663,1.15635725804286,1.96328279095674,0.698259655504597,2.57880545286718,0.996663569032032,0.439383662262788,0.495873987369038,0.482322430838624,0.41540792493301,0.339006755317443,0.447783290366676,0.562792293406679,0.819079220768515,1.74677522617803,1.79822559156664,1.73427950643947,1.66433666801117,1.53792779828101,1.13387585006808,2.19247277197131,1.98135775518767,1.45907517075254,0.76784198086529,0.673299289384774,0.778405892502241,0.495667101535402,0.453683940200663,0.33283568813864,0.47653880601292,0.509570035326983,0.456183374775772,0.410330220201261,0.362275813047017,0.324291284359087,0.454880370171969,0.491242425837389,0.491053536171148,0.476265090519848,0.50202337787942,0.520582429248159,0.657209748344633,0.399785612442295,0.536412999721588,0.681941040116922,0.633310811236231,0.629214628149709,0.701634682793448,0.635350469128719,1.72559848465964,1.7811360816841,1.61240669932753,0.669616633920683,0.383447858201623,0.510700685728043,0.62295901166091,0.565684372874449,0.576346644359592,0.66385414873564,0.88876750540694,0.705511516234681,0.751498040288558,0.875256238108458,0.937337265764042,1.01636347098224,1.02381130493457,0.783939146520271,0.488625928258001,0.57211663872104,0.451041886229202,0.508005143731642,0.48257902975942,0.657484173605009,0.473203058958613,0.992045184628527,0.812574537135177,0.999132482678026,0.34240553464029,0.404042144429067,0.50381997945473,0.446237347711704,0.484727760227941,0.386311630610815,0.419226882546562,0.421698888136312,0.474889335221973,0.380787833494951,0.387038391928725,0.367278111573028,0.451980249388614,0.419820936438218,1.56335784096147,1.58602244718282,0.9446839588774,0.62181730510901,0.875878193558264,0.728817606018646,1.53233430278834,1.52427932672492,0.786707733993647,0.536447096680168,0.515258785413642,0.747416446546214,0.466082372484248,0.641483548316254,0.638397277200106,0.479046667454435,0.43763950654956,0.311484542579605,0.440158353060349,0.543003033142638,0.834202953977269,1.65571531279322,1.26478065822751,1.19750162724613,0.905799628400209,0.818700468895687,0.84092195704861,0.632289487394562,0.603319377781121,0.557559924909736,0.555641412756405,0.408761420588977,0.407629030212046,0.344875577256879,0.48538015091257,0.889723615504228,0.808985502666399,0.328123735937986,0.393083758953445,0.404571006824547,0.460986575269852,0.464074247633284,0.40814584821893,0.483740025387888,0.421789537261348,0.416458170000924,0.443393749054012,0.526818763299785,0.703947384360634,0.583300857485454,0.73477756168277,0.552618791384324,0.403066971435684,0.398946593206338,0.451152007787983,0.372172913697435,0.455664046229406,0.326111918745123,0.428901625933997,0.418337826660648,0.444560178483777,0.519489764729382,0.451592927303251,0.32073203849785,0.571143142000217,0.426368125106997,0.470778653619246,0.446240588039496,1.73172594595491,2.08513604119267,1.53127761548698,2.21036496989952,1.41686583654418,0.87360526786837,0.883988316417372,0.996992418540007,0.573123332684234,0.475333297156894,0.380498967960497,0.313817377231734,0.415147626101369,0.489864333104254,0.367308828755563,0.395044778215114,1.14763193568812,0.445180572432722,0.402564658576064,0.398355518189778,0.329393462843501,0.321830846729838,0.442102023516254,0.406883817681715,0.470620420110746,0.53339626070306,0.322190765002142,0.300301925276445,0.462528510612711,0.988789864926841,0.477132488041882,0.557374856583689,0.454854945535987,1.14364531954516,0.776196602993255,1.14888363722531,0.51324362370208,0.485500602704898,0.627088294684862,0.58702634489048,0.622557474408333,0.523247835024791,0.725325061411651,0.611411220444252,1.50932811192187,1.17652901248069,0.984656661973715,0.95559229935238,0.928601651030001,1.60585477564136,0.993533919293782,0.836963506761632,1.34000622125956,0.903339094981392,0.789825213334393,0.576799022911934,0.591249339722652,0.423574806090106,0.426204489151962,0.503744871196341,0.444216711573001
"19",0.218849566339663,0.192793892286049,0.219050611229308,0.244225992326296,0.174717238074301,0.265641957712979,0.22862044760022,0.197680739414881,0.184584591935934,0.159846525242294,0.1650271565872,0.196545747388758,0.100666531047365,0.179828838308382,0.124326876143264,0.200531958745455,0.20293265660956,0.291478715509906,0.151728933907885,0.229504519749233,0.269803298616707,0.232069006587282,0.313343289059336,0.302535589945976,0.200133933492295,0.176462586648192,0.29398350999398,0.185819016336951,0.182912834911746,0.174704744159941,0.467687599772296,0.305814228393251,0.253576419087078,0.349870438691763,0.429868010173318,0.92980674370138,0.767433111348213,1.1253759384733,1.33622954011469,1.02990651364324,1.39231745842364,0.935551993841196,0.743614558355308,0.664520369912567,0.605120546681565,0.763146457614191,1.51051266758497,1.21646171919342,0.803742686379846,1.06819448995097,0.693078793281038,0.243903194732695,0.167887811073908,0.11455079850489,0.404320550177745,0.183771017910524,0.156941325846608,0.197218288269178,0.175533310219244,0.333608116201599,0.266277295541391,0.284320771141713,0.214746478741494,0.107246107783489,0.146889633517785,0.678376197412961,0.614951601174079,0.27219164647461,0.257409408072455,0.193274815583011,0.259690921447406,0.415767640747877,0.285348741484814,0.201452130933588,0.206995350864619,0.252891606382881,0.106705927277815,0.253864666319065,0.0629015208097705,0.189365500149834,0.153584287594912,0.166601667380807,0.259686305384557,0.169909008168335,0.181711993363688,1.15425663117565,1.66206567920779,2.17527881772988,1.06528832899666,0.933889436646409,1.09499883264683,0.537049990206029,0.699406905197096,0.717679449032141,0.679686257998683,0.670286159651419,0.630739029518288,0.577750061327779,0.154734810421896,0.186581083323748,0.113810833486337,0.234830498698436,0.181195891980573,0.345691436163388,0.247257459265554,0.255566195487752,0.172237416938923,0.222291268526759,0.22422889668589,0.113685259626194,0.290408772921644,0.213150616836557,0.21202386092368,0.178605448664924,0.146707620935259,0.801941756772306,0.859090425687363,0.617662741473877,0.271285415808972,0.158609921790994,0.122842120700416,0.124890349045357,0.227041429813819,0.241920586331538,0.1101388509789,0.197526985074712,0.0983588712447833,0.153622440444078,0.107991910323435,0.149302991175237,0.139671303249464,0.142167707073415,0.104262798975635,0.253608162074731,0.165657919393277,0.0786643877892342,0.123708471955073,0.236150467328063,0.834867869361615,0.828404208036379,0.875326256100214,0.763384408417254,0.898575916556449,0.504018954199869,0.201694962543602,0.082420913840202,0.120863172177985,0.181523831571547,0.159850909837687,0.199011145203334,0.190294712543486,0.218010436993519,0.22164877059068,0.104446624230178,0.258466886410029,0.161946896458194,0.297214418890148,0.180517012800633,0.198615143577677,0.335591297418971,0.31784992403567,0.22289112420406,0.251507329363923,0.193871489985641,0.315714838099619,0.28082955883378,0.706781960050359,0.31730537384545,0.669558176409451,1.19245527446847,0.589972350330388,0.524216511994079,0.346713995106424,0.429407321510315,0.197182211267851,0.851765871958838,0.728086331007349,0.378151822206089,0.217991086570888,0.37873106694279,0.939631745423492,0.588210042562052,0.762579380981502,0.606084476924725,1.12080587449279,0.58603604658803,0.813428299360874,0.736359777249467,0.547139903081306,0.852534260411108,0.925076478972912,0.310585302833985,0.142011109701113,0.753562784515789,0.107165064017185,0.223105618951798,0.178118724016493,0.315189317528804,0.216261769575679,0.192097785655119,0.171386993587391,0.185091932013875,0.251547190925144,0.157900155651991,0.174745612271903,0.20515732662097,0.245218657781543,0.198410952526387,0.25680292870868,0.271817171544303,0.264066874744234,0.612564038280564,0.284898954535732,0.13863554677572,0.201016118276592,0.241720501928238,0.132926095604407,0.232734125657267,0.350920390999186,0.203439545189612,0.279529456517086,0.48278566668776,0.446443745926477,0.328263036373465,0.400389059761224,0.527243450471715,0.847721942014867,0.959663967020769,0.961803389854688,0.998485803680323,0.880510637613384,0.990984904022574,0.711850102704452,0.723306718248841,0.655122700463018,0.809975682295674,0.71456139868966,1.02993487995427,1.10696594157128,0.655157528265527,0.823337850600208,0.273693075717027,0.212231515938549,0.205999292317471,0.17647413499339,0.195869088416154,0.1220252985762,0.191821549283115,0.164881926683916,0.106806798329641,0.0939951031372884,0.109448863015433,0.110491159752303,0.201673137914946,0.190821977344206,0.309481288744941,0.1161549111536,0.774212451856085,0.0668129027849806,0.168877003009502,0.225611450470265,0.280686431997542,0.203025282503856,0.16928433319656,0.226122176093916,0.249319135107671,0.225075813554846,0.101408132971991,0.295805795312062,0.0999110035746384,0.185686735526969,0.0937332510154016,0.0976262361134665,0.190315738275726,0.167741709442734,0.214952240161558,0.25380016950461,0.273812885441878,0.164559501659058,0.238554515217196,0.334764368625891,0.308727842332482,0.410502344677647,0.301156577850542,0.266921895617411,0.748943488750819,0.382861010029611,0.707113046836773,0.300213945368246,0.249800388147987,0.186761981463972,0.139302154468304,0.127774179394622,0.131675062095308,0.107662769030081,0.109234348650397,0.0986480882723195,0.166268858975142,0.155220350186216,0.111546488095,0.0843656542808916,0.140789982103198,0.121540121013218,0.18547642025482,0.233026497088998,0.176168043710742,0.27118440894659,0.192955450818284,0.176759867355702,0.179366436431892,0.221656996182358,0.286393656527516,0.240870818302414,0.380710082720713,0.315325452893691,0.382001707622939,0.324785082200218,0.166596279131005,0.25019770278731,0.116995296790233,0.121394233968761,0.193667249855585,0.2916365180349,0.242732855699177,0.186888662250758,0.210234070350644,0.275049206371752,0.159610240496646,0.339224804439074,0.226831835592444,0.967304488393069,0.867937348210799,0.737399754643763,0.925366478967516,0.800443898847464,0.87242086627627,0.384658547058156,0.25948873334332,0.0727309873725067,0.174637682020599,0.114059540230605,0.22465506348344,0.0355851626336712,0.102608114589768,0.163074927115581,0.167775480706694,0.114584154306121,0.180905867755413,0.14845486788148,0.00878588052804977,0.281422305636073,0.194882547149566,0.222809070319346,0.109996637678888,0.188163808861186,0.188215576641165,0.218801582993967,0.0833936530172419,0.204306218584865,0.171748381970054,0.134012063096068,0.0965006186659133,0.0943258080724618,0.184715866609599,0.0953286611675909,0.0778617702811558,0.121787811157011,0.0874105904652973,0.196070464609637,0.140949060958641,0.0614511150769624,0.315403027923398,0.176185646894892,0.139090947031832,0.0155316849431754,0.163784617979749,0.152852546190774,0.239540372024183,0.814361508064512,0.860735179152096,0.710819576442556,0.695101491660818,0.731040319402643,0.504035011070935,0.16983572932275,0.195456217057034,0.0574172513331501,0.0888941160234268,0.287683563777925,0.275953839824324,0.131018892552522,0.226433610311648,0.140536189056145,0.113370431722053,0.202857472200764,0.261212212503219,0.223705122470804,0.210525634835634,0.125715010573582,0.124849863143352,0.14056138816233,0.163482249611734,0.145615961666004,0.126524856097462,0.143534089309173,0.114304729143097,0.0917298361118414,0.105177803602639,0.123888320631367,0.095340125946352,0.196873338339153,0.585742987657298,0.138384758038184,0.245638682984707,0.15270810479608,0.202291403136577,0.194479656484183,0.127273032712462,0.209890306453855,0.0870498561628633,0.451980542864093,0.281969763243013,0.346625579033282,0.736684769551883,0.75443837387397,0.670832073819293,0.678080440349277,0.566989559704183,0.582061737573931,0.349327059291976,0.192349288674351,0.185550810847841,0.0638145538850019,0.166368068245926,0.0606510534287433,0.286192162953051,0.187166902705233,0.150844488532861,0.164364684255275,0.182724609372401,0.174847547292355,0.151488586414177,0.194210112499713,0.183321040834973,0.168042435248147,0.254671595386734,0.282538487810366,0.229159958173425,0.153090043232886,0.149908314462613,0.131889897759282,0.120719249574025,0.226592327304852,0.0136439446848749,0.156651298499532,0.0384887328998858,0.0958961524243691,0.215780166872196,0.120311545547468,0.161265145204419,0.245689803162524,0.135576331053665,0.221101638221224,0.171140312958967,0.131831615524831,0.191460911881271,0.199832075181027,0.159145108044167,0.139394272338715,0.153334855415386,0.199170882328039,0.580625527362355,0.615745560501782,0.625740883779775,0.626813074120886,0.75911320493384,0.568376386082357,0.506164839312963,0.528721526779534,0.272270834422899,0.190662961901177,0.228284241708689,0.139477796142911,0.151818998939578,0.181202824316796,0.268420060206356,0.348147540010895,0.182525305325765,0.217137261959473,0.134261872508975,0.165443387908186,0.136571939563251,0.329497657644577,0.285647515656793,0.218403930397906,0.10829842597393,0.0133289895486811,0.144973052485782,0.115095990405288,0.23715379057011,0.228329150448162,0.126063895237035,0.0219529907783521,0.147293298631682,0.212679550829406,0.133164769689316,0.137912206811535,0.259248561364534,0.14636523351073,0.200075179747125,0.0728741779342573,0.179975147052732,0.183028560591747,0.049243612016876,0.19719537690034,0.128316762204522,0.0962350637624442,0.344443133255668,0.211942361082813,0.195457407981441,0.104861572478727,0.257824835386082,0.30452435281569,0.440766392275139,0.265944311247261,0.203515518506983,0.26726238209761,0.142902150480193,0.218569293290892,0.011780761964953,0.137015730374205,0.166473136917573,0.320959989655141,0.20782318135373,0.175515357912349,0.249302753367745,0.189218853476074,0.118638873450581,0.17615128322844,0.230049521327308,0.173741554923347,0.219311497422036,0.168334268011723,0.316076640037617,0.213089265403012,0.366788786636519,0.228778159991281,0.161850784255232,0.28339983549097,0.21666431251811,0.251344011581181,0.201958507032583,0.304023564808886,0.214859716916573,0.184289969735132,0.0915023463717182,0.306230870307162,0.153418597238802,0.161070999308084,0.161798016478316,0.123701586649594,0.100644446328274,0.17634581853866,0.528548063123913,0.609475720494591,0.319391258161958,0.260205578179102,0.483966073497406,0.278718096559849,0.510953697521701,0.55215654995714,0.447467809245399,0.62091297025912,0.74634493528371,0.639542424497975,0.947807125940899,0.41374892490871,0.555777583888216,0.402349943083144,0.510766448812269,0.409695009520329,0.434960476734751,0.476780389855936,0.482220784934011,0.330318415196633,0.321589910127038,0.271988454101835,0.204165279994834,0.204277755667362,0.187601117706954,0.300563963701035,0.273147154409062,0.224328080173255,0.283273345749914,0.249628819758985,0.316018086149495,0.291289474707419,0.219073149090868,0.225607299317151,0.367855087522513,0.159767160823158,0.227345867842735,0.317279816822135,0.20190167539005,0.442685360720033,0.258615053589872,0.245357782993559,0.151541898673473,0.223240254885067,0.150639365323529,0.143583857027762,0.36696267054777,0.27390371742248,0.775786783123895,0.269261046983149,0.341655739787584,0.521497822883937,0.35407976855896,0.399612226863768,0.386864755132068,0.460193433277073,0.443317561025481,0.277284703405028,0.211001474453571,0.328152975045184,0.227457947507783,0.20638598057869,0.108371282050094,0.180051943810562,0.140197396100437,0.0267499773651198,0.193002611544169,0.158825084681229,0.178951790349165,0.0201868705895989,0.14881280375032,0.0809816001433896,0.146370067969271,0.113612267721407,0.28514718219779,0.312629853139267,0.178736473910209,0.288830694532221,0.21800304933955,0.188087792434846,0.273665386012723,0.455448615573718,0.332205208461111,0.345782983313676,0.370438467723282,0.237781587646619,0.359483063731809,0.402429523395868,0.482164367233661,0.172503139762993,0.138893313442846,0.128137524411415,0.205204488066546,0.127304597337667,0.662236508503975,1.28167041077728,0.910984219267007,1.27386160125563,1.46774977861761,0.708186276871542,0.714359337703196,0.712099422524579,0.772480850542552,0.694348923387378,0.601135745820883,0.467752842388839,0.779829040391388
"20",0.115289531479554,0.0947004293831604,0.00177741359020907,0.0548209027975742,0.170254145120129,0.025225589390956,0.0114118014582125,0.085104831263204,0.0239955791941593,0.0496378524444235,0.0338980004685778,0.0736415621567253,0.0651200438498939,0.0501677943586185,0.0296448288920296,1.83222367525856,1.96442082103177,2.06265050951957,2.0620856169838,0.102825628913385,0.887558405510856,0.685231602189814,0.225312107365647,0.677329478555876,0.693516486471085,0.167209826724588,0.602940578594652,1.30537560890831,0.175281313092581,0.202261265005544,0.187676787227229,0.248362163456672,1.02512575170954,1.11706245673684,0.185171662241577,0.962121185303424,1.26553847747557,0.191101851826758,0.251789117766455,1.52284542468317,0.145325545958901,0.341734740619291,1.59289004762672,0.141310486788976,1.74044909066111,0.556554551827349,0.199480525370703,0.146534664534637,1.16140256937121,0.392240365565982,0.135844715667039,0.0619089764405861,0.0425600601865127,0.0787413364330364,0.0274943061273881,0.106463410589474,0.088304259155294,0.00858734376391451,0.0575817596253419,0.000683443750957927,0.0298366386465181,0.077405741687104,0.945202405450992,1.95450119982791,2.66702064609343,3.28042046092199,0.666216469972087,0.785899022265928,0.141075953441994,0.160886860792917,0.164816465709131,0.182015553778415,0.214716151944316,0.082050592299802,0.0220351378971824,0.0684920076153016,0,0.234131323002693,0.0604787453250892,0.0643462801258973,0.301039053514275,0.915884522394772,0.766518168451755,0.670457291708189,0.763091630539768,0.657141681533171,0.298179487778056,1.52295984413398,0.824890023073207,0.397054202979717,0.42960369141753,0.275011773204765,0.183552401175731,0.811274202591499,0.0537994560289043,0.144878821856509,0.074582072894081,0.0775486861902736,0.0563429471339504,0.0572461181595467,0.123944636462891,0.0550983633966507,0.0599580283920088,0.135925928073486,0.119752121689176,0.104351787482812,0.103577771727924,0.0339122248984503,0.0399957329187878,0,0.0203998616507375,0.101295647404598,1.48200477051711,2.22466729419867,2.19843519081384,0.557497439892209,0.620703766044443,0.269828550493183,1.40010779619722,0.259518956682237,1.12655646423348,1.1599818272536,1.05823167981709,0.303844601614911,0.206303652292302,0.298282833274183,0.278485818007195,0.286758168888219,0.242268991963895,0.498496461301241,0.762125689845822,0.180405155047255,0.277610276399852,0.190253587695246,0.66044732749546,1.38179785609574,0.674761719041766,0.330442832622841,0.450492683140555,1.35143591992698,1.33447747459396,0.909468334547946,0.994530872504785,0.35711016286851,0.187063903906268,0.103344212327988,0.0510531779556136,0.0587192373242404,0.118109335418346,0.0423719864807888,0.112214066346006,0.0994424319675674,0.135585897019036,0,0.0481730855616602,0.0698672164744565,0.0772786449878545,0.124361314947924,0,2.48463066407019,1.74006876401294,0.211820812705502,0.209418129010477,1.19085788188809,0.0803069277671811,0.112708036445538,0.607829122501409,0.822957191169081,0.115759297786019,0.218392954620058,0.28403330022521,0.247699163859679,0.277902945198543,0.260482618882712,0.210443231410396,0.30761898812729,0.204739259119279,0.290758408497996,0.267119393060069,0.217593569532921,0.514005134009833,0.719280666536212,0.751007741170923,0.57487068606372,0.336355694569405,0.294896758825753,0.326596192880291,0.247537741185201,0.197096374777937,0.27803510207441,0.221816128761628,0.107980259896769,0.115704225096463,0.000549249511235438,0.153400107576161,0.00283332933789139,0.105077513055877,0.131444885833372,0.0642268785510492,0,0,0,0.0184410533906939,0.148213750633127,0.0426447648811604,0.0666585968972894,0.0828636396129967,0.766437188421777,1.0814362408356,0.0840992469331007,0.133404593005396,0.138970997763567,0.123523125112302,0.106058935560913,0.0621529318203999,0.0842814183132187,0.110731379098911,0.0782817076191043,0.079194184861527,0.118247716556727,0.0475775019020994,0.0751090603963456,0.0845775310157617,0.143721369184274,0.165434754434703,0.0205847883002414,0.0366873422883233,0.0282798534492672,0.460501034622748,1.49176734959732,0.632940061207968,1.10969508977587,1.52926622326898,1.2163775046159,0.367313227488837,0.21948265174419,0.163476243129263,0.241570289743001,0.124688952473671,0.0535252606891046,0.0919772800997279,0.139797988922393,1.17287795394369,0.111144293858138,0.0390408604446502,0.123825221720309,0.0868199771258637,0.191489484634662,0.0891922273106853,0,0.042526818447553,0.0458876506743321,0.0661203652364678,0.491295967795567,1.96435510841431,2.31254425886644,2.01484114450612,0.545041535958516,0.670927513843583,0.696164226276677,0.763633868863652,0.333041342423843,0.528986561485678,0.227730296396214,0.183242231854111,0.399661541598789,0.275999299799903,0.204333216879355,0.21724153732835,0.21626685360513,0.200751407473954,0.224068860005252,0.261634500573501,1.3412383284155,0.235276301300726,0.397414431991421,0.259449854244937,0.450136395429739,0.825173062767454,1.33297299693379,0.576200625972803,0.906178409561421,0.501324881886571,0.406606588449525,0.325527579553974,0.289063457442384,0.0160981749118783,0.121693018925899,0.0704314248183174,0.0661767747271582,0.119088949372204,0,0.109060775992181,0.0337795359954556,0.0263124177928369,0.123041917519691,0,0.007199496996393,0.0669946508747843,0.0296344392958241,0.0728132003220855,0.117222758429199,0.0129122344531534,0.927777880921052,2.04848623564415,1.98785066959164,0.85247034199255,0.616804556394779,0.666311095268623,0.298774117713463,0.270061409974753,0.18213914980108,0.160862297441944,1.85678826818528,0.254797907763206,0.177543979355076,0.275759700625895,0.217663249694828,0.239837717397748,0.301670759318035,0.298737308247009,0.540754896306624,0.787516325249772,0.380038078412506,0.282793510975258,0.240155605396006,0.48660462587966,0.748265902471038,0.508552667443016,1.33466955318199,1.29291530347845,1.08253571691294,1.06219804015261,0.2957291383012,0.13928342396451,0.162576783595694,1.40068736145423,0.0880282739701448,0.108816580897314,1.42826970744857,0.069192330648624,0.170282581907025,1.56963464281628,0.0972940841314806,0.0542787301757364,0.0124007578198184,0.0610811533063784,0.0816790640614377,0.134137866230675,0.0781907942548737,0.154934082586682,0.314032442928155,2.0106232811056,1.54002103866833,0.345165184522117,0.869289711511398,0.266999687931347,1.17682527504255,0.19491273717587,0.239046450741897,0.148879682014731,1.09375904809894,0.290265601788768,1.95613083488385,1.04139642242285,0.184618857717494,0.229415640263788,0.261278265810039,0.232129960963119,0.858053718549928,0.611477552589796,1.16248067346758,1.0665076288189,0.390645254732186,0.0502177697454781,0.0427482909068603,0.137788053209968,0.141781456611708,0.128661687460763,0.164931889763924,0.063487099087607,0,0.0189487500407822,0.0125887478644933,0.194962157440489,0.0337632648135281,0.0371920625301704,0.0541472076811201,0.0714376125857028,0.102641041357078,0.07117439448021,0.0321535607835989,0.0458907203235694,0,0.090117798203905,0.0376404792382979,0,0.0296139924526665,0.0948547477965226,0.0368279242116499,0.0621511592267251,0,0.046958864170388,0,0.040995530476715,0.0721940958843299,0.0158457039945543,0.0133618441007452,0.00959692343145971,0.00265816835672129,0.0617387616395992,0.0528343141416917,0.0798583876407038,0.062817473306973,0.0160548488177861,0.07280215733273,0.054129959520676,0.0439119198048604,0.0953574974468166,0.104018864087134,0,0.0338864668023659,0.0137256365416374,0.0801132043340962,0.0266506713714824,0.133764077596954,0.148150633720636,0.0598592394464943,0.103785809927212,0.103731774266646,0.083187537223333,0,0.0813018764219229,0.0423820099458496,0.0591763293610837,0.0741553634422663,0.00945397299719879,0,0.0975790885904399,0.14289566285811,0,0.0787858986277063,0.114676937004704,0,0.103224354726734,0.122306033841991,0.0611476173424746,0.108959810124886,0.0848818592890527,0,0.0769373982645174,0.0411607558924065,0,0.108734571993003,0.0225460787649991,0.109890214842294,0.125403913807629,0.0339841646799744,0.0828411580238853,0.0561127196034688,0,0.0866119171762796,0,0.0567708549695109,0.060136039996713,0.0244426944337728,0.0659488971195062,0.0861452290700581,0.0532463531689906,0.0173191052257452,0.298127184553483,2.10322249807025,1.5930085433399,0.200407164040035,0.38684130010779,0.630168441291928,0.300926060938945,0.0839459504616531,0.162450150196487,0.0362105084217284,0.0666813602255449,0.0211520412882037,0.117344112746266,0.0929340875249137,0.119269620342568,0.0579814559413966,0.119353016047258,0.101485559945591,0.0989037918339068,0.104036552925966,0.0692781330492628,0.0491751199402718,0.0115118112066593,1.12091555945993,2.76189387560259,1.03509040858887,0.514690499943808,0.344118640198176,0.334558016005059,0.24744683729424,0.256943732663255,0.262735955679624,0.205606140352771,0.173968669881779,0.0944420747356164,0.142191005859149,0.174799197439272,0.197398395239403,0.191337844974579,0.165742765632968,0.273333310659021,0.186028404509808,0.178410297980319,0.120072184916083,0.228426126693376,0.16640737269899,0.41849022054712,0.268703875386307,0.852882937672036,1.4081637915125,0.690047139650682,0.877007964671089,0.298865487583006,0.251928821213472,0.510938920996618,0.234050741868615,0.147962406650925,0.129079193461408,0.168665926137058,0.0779010380196663,0.139448648956719,0.105465619832604,0.0532288219426143,0.125236156096489,0.0111051395176101,0.0834618945676178,0.0739229798678408,0,0.0963411233796579,0.0633607386030789,0.0912506467056993,0.0134969972710795,0.889920095238234,2.15279522229766,0.285999043610512,0.162181623337959,0.224179355883511,0.148755864331027,0.133424705873944,0.198049452454616,0.15264954419588,0.220457737125951,0.097585684727482,0.148894689860769,0.165658960673841,0.130929438999306,0.0192459207952225,0.0624338822652713,0.134359808141179,0.154720616823459,0.0887408904831216,0.0189507387603253,0.0780777587771673,0.13238116496941,0.156206034471082,0.354130911830901,1.53311704638491,1.69564860766177,0.653709239025415,1.00119803700758,0.320099427063593,0.28382139675947,0.249286346898358,0.197161337430826,0.258255666364852,0.178677772168057,0.0975564845997037,0.0873727350665093,0.132867481505649,0.0328131465002949,0,0.00944227527681835,0.0904515746454833,0.0711272067576826,0.0459110748160225,0.102574510748518,0.0259296266714227,0.0877798658400686,0,0.104968503870113,0.0944920298416254,2.00580228782797,0.8970913466504,0.285078941543121,0.826191620199892,0.598112901277573,0.351213944004542,0.818741141220418,0.842955344539512,1.41182085019187,0.331222138897732,0.845166136463691,0.341540408321274,0.738552660873632,0.460796349093047,0.404865058522009,0.128859730621719,0.14602352518615,0.158940410455807,0.262013008116392,0.572636235831394,0.397087829546563,0.162171274319867,0.356191284661184,0.554080538428473,0.679351833216071,0.437327321032443,0.281047574013536,0.37739996987275,0.20214194342291,0.225707610364892,0.327412710006282,0.0987240306886245,0.0769623546937201,0.143981514468148,0.0503745933006521,0.0887807792689369,0.064861678705927,0.0809403723363278,0.027530762143886,0.117788373112601,0.0626940498903006,0.118796006980539,0.0458237388972989,0.149111048007732,0,0.0950290659992521,0.850483085137293,1.9958187997551,1.7867159406655,0.975992028943958,0.551812963527358,0.249585483107672,0.303409795584838,1.14450846150473,0.579218052260236,0.67738686898651,0.787602563668557,0.139771030237997,0.228729088588361,0.178106457363667,0.298689165113211,0.326618008009666,0.345411100427017,0.358398381139103,0.259847632979477,1.19561970341499,0.428294572600848,0.31024920106905,0.264596004568265,0.477184070712139,0.495230342929256,0.555962131490871,0.328682742955446,0.378285925811349,0.332723275013722,0.323697234064607,0.290131579881905,0.373796661754321,0.281213339399664,0.116969993125718,0.101074098163794
"21",0.451363229787487,1.2062855944779,1.15295011953021,1.2598202896818,1.22993149677899,1.18813680229712,1.27087093806824,1.2704348056741,1.39772546590005,1.27661295608717,1.35068201117389,0.857123625996966,0.109390214542671,0.197596305237866,0.139746257635275,0.0958196435593621,0.0491103375506459,0.120433901864852,0.594477742639203,0.319872757560343,0.13608567988938,0.0660351957110275,0.0824021200394834,0.0488990405230188,0.101272002950409,0.078563515258282,0.216479208206671,0.263089248639063,0.120108133626607,0.230485183397826,0.605087120818415,1.4559843625259,2.0394704099055,3.73013750051494,2.4121863609444,2.36525718846876,2.028833181688,1.23290699252383,1.19888571124182,1.23606903639607,1.20065974664917,1.56070452481096,2.62850864327863,2.41416234862256,1.4799971923901,1.29057974888782,1.60288989485887,3.76831230043827,2.38550506393872,1.9010141272405,1.81905415405055,1.70113739673076,0.0869942556742302,0.0711769154907367,0.136770621058907,0.074839114449994,0.035327962822377,0.20150058847987,0.117244497102112,0.152894596478488,0.166025617969332,0.160929340163193,0.0850622212999884,0.115340057793231,0.596068838592871,1.98609226814675,0.396536367565128,0.0791744860869216,0.0892192796575263,0.0655583852655685,0.107662950369027,0.476056406224246,0.212165801199606,0.0109484622347629,0.0609352054906838,0.125282306530034,0.0497381621119673,0.140364383341554,0.0594305547332005,0.569131555127985,0.507279721105904,0.215498611246059,0.15786005026054,0.0972324934373758,0.160396887742811,2.04889240065687,1.67176018343554,1.6170344043116,1.73011748763514,2.4261953500106,1.74269097794739,2.02789193886417,1.83377585380501,1.40546806050133,1.20662173200765,1.10173138148315,1.17183845741619,1.14995584879504,1.13477132290181,2.16347004020246,1.99966228947225,0.118283374060885,0.125102167197576,0.165821080088208,0.164312233534791,0.159395536819209,0.145099161286096,0.112213332233714,0.199327819357976,0.124491917266159,0.119619781956377,0.0261818821791591,0.0525301864208727,0.0415135592116564,0.0698003634183228,0.192532879414909,0.344720150853759,0.151066513635178,0.118964257254356,0.643521725299994,0.376511327389221,0.187595675784271,0.058592961786821,0.681008467205059,1.43668355098934,0.321103898351404,0.922217180762403,1.39046333213398,1.11330262083604,1.19846651828188,1.21331667971114,1.2579989404427,2.06902965397252,2.51474133702095,2.53924312077188,1.5378182524573,2.30558253228178,2.1179695012942,1.80857181650799,1.70001599540704,1.48267815151053,1.38155792067847,2.57625491383533,2.84032410070068,0.175022777882324,0.0744531350149064,0.00972917461257693,0.159070717396228,0.0363571450951811,0.0657396205543573,0.0396264511776623,0,0.00625440732467406,0.0782599023590184,0.0874791732746591,0.114434464425374,0.156643299034414,0.237027055208321,0.0223521155887549,0.208582452426122,0.111574305320227,0.0485606937588012,0.243082885921786,1.97364870724872,0.44663613454888,1.26439448066667,1.24291284821768,1.20724264439386,1.26564650009783,1.36689558450254,0.241744635771821,0.271162276185955,1.21087722671727,1.38942039890861,1.19893371767805,1.19630787789945,1.24503431701671,1.16558100406433,0.438383877627905,0.336001671414688,0.33144457757263,1.34738622599896,1.30762112169152,1.20776101036283,0.218459167345929,0.237823657080393,0.229263670067594,0.625429442624558,0.794486419066066,0.396232665028746,1.59927176091775,1.54432211728482,1.21330144508188,0.140685007530776,0.0460212832565239,0.0792719356873789,0.0818721182595131,0.0514940162668544,0.0414311778933082,0.0658558098002573,1.08417135530088,1.2686639858817,1.27825733484742,1.26505011160544,1.33500400258144,0.560246371448068,0.145838300252143,0.0527100750929558,0.0486133970819479,0.284474611794838,0.322335461956213,0.342700363822875,0.0943330720395032,0.0899890051874705,0.108935360295087,0,0.0463325730356929,0.00663778264894036,0.0539270556063298,0,0.0195116647171293,0.583038334593427,1.4323680192252,1.27595693411206,1.18575943893676,1.53943051826185,1.53791456761809,1.17260431395054,1.30860512075882,0.322506169714302,0.316290221937667,1.11789186707036,2.1443494760201,3.01158187187606,3.00312981029744,1.86342592305171,1.35153346790317,0.306322387548107,2.1188753783642,1.95005139789306,1.36304466930971,0.0496838664152556,0.0552319744605723,0.00228334264652592,0,0.116084204300141,0.196289159717158,1.02612336793019,0.942453115201976,0.0373810015575614,0.132507540080565,0.101984223514836,0.0071948658710167,0.152550259044392,0.106867660821313,0.0529329786859792,0.140518008794014,1.42701548932728,1.9560843715429,0.393638151474265,0.290725602659885,0.782802990107058,1.29625335151299,1.19478568803204,1.21482563626736,1.23676112993172,1.26297101402081,1.26613424688535,2.71770891157251,1.1139685371176,0.00835548713163115,0.0883194058775907,0,0.0807219693489211,0.137048713711525,1.18340558315368,1.22109188778232,1.23100180255004,1.53972321026136,2.43159402394667,2.7579865518226,1.75157873591211,1.47086474913766,1.7645717472773,2.60886243018048,3.73570301236608,4.09037798166959,1.71899105599219,1.24453946151321,1.22078193414569,1.35553281361084,1.20357674907916,1.23844566093384,1.29318031464521,1.29735719513697,1.05837418392863,0.0814694778190787,0.115323614924673,0.113645062824877,0.0942275194622466,0.109061155530097,0.0595633118755326,0.0902021166334175,0.0720347503799193,0.0553909733100628,0.937243621834246,0.375316415466911,0.0234895213202744,0.115104023206979,0,0.0978818284757332,0.0151678120820047,0.0475830157078902,0.0531814083456615,0.115073275922741,1.59023336717085,2.3995788074204,1.19316849761379,1.16477367644457,1.29074383345299,1.20404518102322,1.17078937681346,2.34636608972487,1.74510252489321,1.59675848179749,2.00151936544168,2.54973326667925,2.53566905722882,2.44753509301467,2.41641023555689,1.21460256415698,1.42627049912533,0.683969506849767,0.172574290967624,0.564056849558252,1.55054002328067,1.97817424975361,1.94418826796761,1.91155098511106,0.648643134763291,0.062882972696033,0.108615530942642,0.135880755970072,0.0617091897588166,0.0551298409510595,0.103498915744745,0.224327706056887,0.183711392815572,0.114719708547212,0.0675762340140487,0.0527998968478077,0.101775208960297,0.129395510204969,0.591096112161874,0.271433522065224,0.463739595033115,0,0.0940532539492994,0.184678086772656,0.0716624062098309,0.83971979553566,1.11234914246934,1.11455522750615,1.09514665153119,1.14770080752377,1.05485663093567,1.17386292601445,2.05504055724445,2.27582559222163,1.36486554401288,1.32940555731311,0.921280838765341,0.572873042616352,1.25985920183917,0.625726617309783,0.739149023893728,1.88978968069777,2.65539631554264,2.9454673220726,3.47497005161639,3.03511920039783,2.11666653848209,1.80380290444759,1.91314630621286,1.78699229639125,1.90810295801701,1.87648901986401,1.02965453797152,0.16973263065143,0.123402328278885,0.155169979036766,0.199700283639094,0.223251318860193,0.217973625385179,0.132037028673791,0.106363518734451,0.0389218768254778,0.123441803524803,0.148537167284105,0.0494609234265885,1.51799370906731,1.37234056735365,0.305003460746861,0.149137533805839,1.0290172929355,2.23547004036837,0.545018765413713,1.3550598576461,2.36087513347428,1.33997726088601,0.92553517276492,1.20764124865239,0.61130511392303,0.564698874897575,0.138072042421865,0.243157811448221,0.948487626072901,0.317278437633967,0.119788840334167,0.0310609094387847,0.0183611883046846,0.895524035264312,2.7571794429277,2.5028186970318,1.3518853630844,1.35449564303855,0.27175008591949,0.24546794531971,0.227832607001067,0.383438651324948,0.193357959020935,0.82625393193395,0.255354284827281,2.02360958238812,1.25774529503814,1.25033944483919,1.34982677085075,1.27912105534057,1.37000151010518,1.28162821417239,1.0321856101559,0.00455763493487193,0.036625447560095,0.204822532848546,0.129346448001946,0.0637550333073825,0.0925598186500015,0.0266668578185309,0.083109072978312,0.19794227158923,1.58745481261509,1.92059391591956,0.284185589697435,0.282249688649229,0.136328435336128,0.215475119623763,0.176214257323856,0.841797871668019,2.44644673320393,0.46802585342849,0.295617895988831,0.142512705784072,0.04508455524172,0.106703697122618,0.147916314112564,0.0990596451654871,0.137319727343753,0.719157313556648,1.37432872409273,1.44687705983466,1.42620449956822,1.55785329651267,3.38902383359027,1.04412131780481,0.220451180305734,0.227849414471579,0.164874893427368,0.248109745669288,0.386088108907838,0.495913718407382,0.217280612813149,0.143726211589458,0.200339531366943,0.14597863572027,0.0854446320582121,0.0328913523827009,0.742328520343326,1.65669644130693,0.291928125349381,0.0565717755478593,0.0482958311561599,0.0899666432122907,0.092883980729084,0.127832663503819,0,0.152187409178668,0.151085931402414,0.333885449791876,1.2332606557097,2.8409100963971,0.283808184049261,0.0797617830966663,0.0924423337908374,0.00364349410583324,0.0642179028166223,0.859322756910351,1.12953589283123,1.0297614853453,1.27892240132977,0.876621471571118,0.0184051379188527,0,0.0745400872637852,0.123361457785867,0.111051644947662,1.4920579841829,2.88845438407506,1.4129630059296,1.41046586458502,1.35247931972051,1.47149345676065,2.10254356258467,1.89928548064343,1.75117982384304,0.733996131708936,0.267187457913536,0.187570796772203,0.683201798470021,2.16027083524624,1.21807408081091,1.25809656321368,1.26074543371006,0.89961603799992,0,0.0730247386922906,0.188842469013266,0.0118384921601555,0.0328019007470007,0.17471818266363,0.165023920147062,0.0938180678165583,0.148748917397299,0.0181546430905266,0.123331802200474,0,0.428891848185031,1.87711178013102,1.09130140288398,0.140634631177465,0.0883809738141385,0.174881484183003,0.184575603195057,0.00178480018013016,0.104995307618345,0.179798160192214,0.125568727268965,0.11741840521268,0.179461522633785,0.700459734580819,1.58881049135975,0.174955596823263,0.565080926148245,0.150844281105677,0.251317262881557,0.164713431250318,0.263320201533614,0.203214454868019,0.0502109136133758,0.234981291312896,1.36089301900412,0.869211830055827,0.0916630506435386,0.0907149805196789,0.105677684025149,0.200752390350518,0.0950895904302736,0.230668966829691,1.59790689173058,1.26466204210605,0.0998500917975905,0.145852563233726,0.022120388509637,0.120996665262687,0.108389057329444,0.0862143797982329,0.166280195759877,0.0628402386745637,0.0791441453273095,0.0692806158812734,0,0.166451384135666,0.163533865400664,0.054100543951781,0.582186711921078,1.94147472148343,1.44460211775075,0.162610188791939,0.0336367028390222,0.112201127882535,0.2318867075074,0.194848614069809,0.210058655306385,0.0933340169950394,0.173899721829407,0.367707818609777,0.147066863270016,0.0622078959206432,0.134604162596999,0.275168293351499,0.0766059260698532,0.308310531688164,0.187469891390606,0.31058945035916,0.372548299498076,0.325902087876556,0.4037521298388,0.302516609498408,0.289186039095044,0.28192431407715,0.428985109489483,0.872121753427996,0.407315885921921,0.349838928378207,0.638271194018206,0.335079460582686,1.64418679302021,0.692530776214904,0.175040474515849,0.141985569678636,0.174374922930268,0.0775270257465398,0.0702180187924326,0.310194465689675,0.112995412670746,0.142935125217397,0.0983009987978944,1.11959680502994,1.23180943546346,0.772655503911438,0.0829969044924272,0.415946546546282,0.29609349483316,1.17235563512861,1.92312394779578,1.38971293176365,0.440474474592664,0.240141330850927,0.102439443597989,0.823437610689629,1.39136327004378,1.3192028293115,0.655955327814724,0.214771256947754,0.271486408814988,0.12133937506265,0.0731448663061673,0.326413695881419,0.0694769799165579,0.0842176350826625,0.267886376012148,0.395744718989532,0.376756393967394,0.236413068579088,0.175802523626942,0.532729636862391,0.737489500715367,0.621803008899402,0.216102522297441,1.10688377476964,1.32122810690134,0.313482862957808,0.250728186430005
"22",0.208913926352199,0,0.0598478306786444,0.169227712934044,0.0811351272822337,0.189508632408831,0.164764813905851,0.129624175396458,0.0493486338388279,0.101233647985072,0.0616921039266968,0.0687517205790501,0.152349636646745,0.137182482168026,0.601919034493389,1.42385359006411,1.40027385019288,1.25385863127979,0.963728647176503,0.114504778684531,0.0848276304232507,0.0344098585782219,0.0384842859147717,0.0963097325015675,0.0911578604521515,0.0960376839257969,0.830072030796479,1.29186419854711,1.28407968527034,0.424862390666636,0.35516690234169,0.536699947118609,1.61103189351693,1.10255896321172,0.855133230213806,0.716296573905065,0.878227295506616,1.00984351052612,0.683787669524445,0.799430496321479,0.849189097834025,1.18025297449284,1.16146899005781,0.966486360239798,0.0267427377067179,0.120755071186214,0.0142129173132232,0.0478195368769152,0.0344273373701081,0.144438372689945,0.11138365074069,0.182116272804624,0.179652786725433,0.193793632947103,1.04632921154888,0.571536653119293,0.165606736029307,0.162447540861117,0.0766596615817876,0.0970733005620564,0.119817158068233,0.0839315824176106,0.949705114323586,1.31421348489275,1.41208526951752,0.406609485168891,0.0906834448833715,0.144175126076439,0.0362856018852513,0,1.44751561995973,1.05671522579885,0.773375639037565,1.00932601610898,0.595315998111907,0.469636148825683,0.442967282671879,0.519559813749757,0.358411370766191,0.0714506440154235,0.00241752181110599,0.0754120236322096,0.115271781691455,0.139442565992567,0.0903955558674144,0.100567450608266,0.105423044956243,0,0.09875815371982,0.156615812834213,0.0213434113755763,0.0965971781629857,0.130409028508192,0.0621926754470079,0.233865939869906,0.0542411484225734,0.15841150500849,0.0228373021726039,0.0890490328791067,0.083713593098051,0.0815212742842731,0.2246409028332,0.106585659911531,0.082931982205072,0.104974739661239,0,0.0839241104057212,0.0857349305752513,0.0896445389573437,0.13295166735625,0.144805908737611,0.124406061347401,0.0845739363008934,0.0640025964179013,0.0436311596419239,0.0741484596346759,0.173683508025243,0.0237675420942964,0.101630157284514,0.0505888919807199,0,0.0862293966374575,0.0489077777735183,0.11603184025733,0.0743992128175103,0.159877563332244,0.0992105363345776,0.0667395142409324,0.122233598645462,0.0470719048869717,0.159087723860345,0.0629815264109488,0.161603829417644,0.0717415038611662,0.0213203763965315,0.133367692385714,0.134851887993284,0.0296537306345514,0.0418629780569683,0.095673011753027,0.0519548926001006,0.00302083066568937,0.085839126137599,0.0926779374127256,0.0906109624825739,0.12783091452151,0.098342144634717,0.0946051882949699,0.0623075150033195,0.0673075045809633,0.146398273721875,0.0802052221940248,0.102784824201247,0.171323503385005,0,0.107658235988764,0.089346654124834,0.0519681151894552,0.0166832420324445,0.0482717774325588,0.0697262668675546,0.0309984982269373,0.0932968271258676,0.0562887474359277,0.0137544270347065,0.0858724621374492,0.0569933587341807,0.186839642077405,0.150742479444891,0.0655927292716194,0.0545341889409159,0.176169590097064,0.0923155786791302,0.121810804563741,0.0719237808098456,0.137504497182735,0.0236435161467908,0.126468003642166,0.129626662316717,0.0857788856715244,0.056436526617653,0.115237030116175,0.127648919071885,0.139779634874451,0.140367889096447,0.0594170745108685,0.0629889690123306,0.157436820786457,0.233043079900439,0.170978227103376,0.123466283758461,0.0460367415526715,0.131708814463278,0.0824801736919946,0.206419545569461,0.0598936559912722,0.0885357635053938,0.0928442386176854,0.0262680701865589,0.137490344792047,0.17160259795139,0.0966275137707416,0.060375228752138,0,0.0493138599130945,0.0567412407714385,0.113136141130156,0.140871748267849,0.146710498438989,0.0251936412361273,0.00820123560715392,0.113852308649166,0.0705579052789758,0.022735167124807,0.105398338648652,0.160326403208718,0.0469547188303936,0.0206246423562533,0.176454183251523,0.0599066995280632,0.022745937929149,0.157128699338542,0.124066245227178,0.135244559632054,0.0276797999675237,0.0434004731983442,0.0899655178841855,0.0834231281532496,0.0531177972389063,0.107692649890095,0.133622473547479,0.0732241744819578,0.08261095483558,0.127669058142395,0.083406204951621,0.0343884842042286,0.0874021370191686,0.195797433611945,0.00969954495848528,0.033085955272235,0.0758679205007777,0.089530636598244,0,0.100465423474173,0.0301252197014742,0.132467894386204,0.122367535154547,0.0704021942728748,0.120775630896386,0.0733011775910765,0.121432111123814,0.114775378776407,0.107655749983379,0.0883270794695487,0.0976383375731185,0.0707490723410988,0.100305237142362,0.122061811001259,0,0.0990224779209559,0.0487714115994576,0.078273273958475,0.044768985186544,0.112851202020268,0.163596856414908,0.140532354485618,0.0693507592049216,0.0917257168736242,0.112733334977694,0.181962069178324,0.0634166106868694,0.0752874260440582,0.0373318955041069,0.112206611464642,0.0898657830568317,0.154515449217531,0.181772420452336,0.17840242099888,0.0747200408363283,0.0140020348383991,0.0955264359305312,0.00720456452225958,0.108216618941268,0.0613875214097315,0.106265102573946,0.0113033467514988,0.0724037565232502,0.128595575613429,0.117347749821813,0.123398138493068,0.0567875486397974,0.0902744993718818,0.188252783720385,0.08258843483858,0.0197430535326292,0.148146197445839,0.0114708426396429,0.185304041106292,0.0519902591323396,0.131928983533824,0.0687305140479461,0.0429508158733081,0.00277741368278016,0.158212206210826,0.0963499148597268,0.186065689659584,0.221185476153054,0,0.0319115639648304,0.119086504269255,0,0.151062334016422,0.156686710499894,0.0749468339619924,0.0231138842893988,0.0915648428144148,0.148635697865611,0.0408145301742737,0.0823926393086256,0.084470439317323,0,0.111201241383917,0.0898104143825964,0.12202210441069,0.0477618394585718,0.0859560558824355,0.216356291875943,0.132354765664739,0.0840700932771754,0.0720814079392886,0.131837749067561,0.247257328826456,0.148151919550958,0.0978501866796887,0.137163882505334,0.0103872169054956,0.0464417685995848,0.051243643176163,0.10650265707051,0,0.111219104187257,0.0625642328473952,0.0938205537108686,0.0738793856371062,0.0848985806289374,0.0481828986676542,0.0313159151352572,0.0959600929895159,0.0555165313280858,0.10718182279474,0.0857806153653213,0.0353454186427323,0,0.139195887145194,0.0263168867801227,0.038010128447993,0,0.0934584306772614,0.0951643466365314,0.0969534588906204,0.11705512681091,0.0548892511398593,0.0644056615619205,0.0368442474513294,0.0677647866412331,0.110243728735759,0.11787773365654,0.071659270582056,0.159512696242502,0.124184107169647,0.164039795223089,0.155079169073446,0.158099859316856,0.0784632950459911,0.192564278665499,0.0255916199492786,0.0444821544934125,0.15752475358336,0.19941362234667,0.0610468029763884,0.0770272794307101,0.0795801425195014,0.0751503951917239,0.155091762689914,0.141857098248712,0.163246454404388,0.0572711524201997,0.215066648509012,0.0448925426785287,0.140715590290487,0.0768504410829353,0.102778365645377,0.0810736560553024,0.165640025373307,0.0725231548726459,0.0720623662310732,0.0666511242877492,0.0330400798005892,0.212146006071854,0,0.117269999725954,0.168613256289392,0.020312636743203,0.1188735019909,0.152037213640972,0.0153952066056933,0.0953276781846604,0.12363608954264,0.0430693682337379,0.117803265667049,0.0672765178688761,0.0233014267226466,0.0468669558081643,0.073123538595812,0.237466243909104,0.187532014645374,0.00917548357846565,0.117774290768367,0.0364226630679418,0.104307794113698,0.186359506177113,0.0536027960558906,0.0864746153864392,0.0925000925131396,0.0924431967856833,0,0.0871249071671159,0.0692040881187665,0.117743144450568,0.0697846358158916,0.0127463713404225,0.198156479808131,0.106158517948904,0.103617009233806,0.0718321620733476,0,0.127465895735543,0.1088467765891,0.0229314356508891,0.144216742093286,0.21960872123914,0.129814815440582,0.153443160192753,0.178139725795837,0.0994373062993599,0.165642528256137,0.0151600151043285,0.117428154102903,0.172777983802824,0.103407286202414,0.0702620866357091,0.122156140288431,0.0592063318802883,0.143735305213375,0.119375885284466,0.169316998519406,0.0562260275689302,0.0161175896711306,0.141237058940546,0.0437462507793836,0.13106459886182,0.103923450508811,0.0972694648193462,0.165696293141431,0.0978452730152374,0.120547726249216,0.115385657808169,0.145723614865587,0.173688388410428,0.140398054187014,0.143707329477219,0.140103910465457,0.130269476414468,0.136819429986264,0.15190381531244,0.0658506677040133,0.0825312278283214,0.0703939955797297,0.0122088149636984,0.0303714296120916,0.141158300452831,0.090112525646936,0.0526599111535175,0.0339135859872631,0.0276313637731744,0.116875471217135,0.0726964481753265,0.156912422818392,0.159352370060017,0,0.259880360499525,0.137659951893211,0.0667039808833003,0.147552091328579,0.0558741216255575,0.0705557815803379,0.0347363315975239,0.00155304193153799,0.0894041583974959,0.0716899164254644,0.10316458819868,0.0614981177140681,0.033626180880823,0.0258585078728304,0.113250163304777,0.0751089969573277,0.0241820417962813,0.0757726706364162,0.100522469521289,0.0799144063327204,0.00601396396456164,0.116201123097038,0.0690351084673891,0.0521477649237482,0.0567119028094617,0.145205075909761,0.0880513287461474,0.047187710894332,0.0898474558279137,0.127599980974299,0.0404519768587357,0.0640360820275217,0.101631942741195,0.105810166204702,0.12174616846196,0.100667443258701,0.278702848617297,0.0877443814410308,0.103031903879459,0.218975688871019,0.154710573053751,0.0458155832350619,0.0789879714065269,0.105638336006428,0.167670787210567,0.104732925626187,0.102424576410767,0.190870602147302,0.133634922166515,0.0633202989770049,0,0.0852942446404886,0.139524920589954,0.104382022715571,0.0706905935675376,0.0993608690865226,0.128924627763594,0.121737418360254,0.1603668088915,0.134248484827759,0.102640741049697,0.0621224277080668,0.178469784797676,0.0336150627416155,0.0550717061450957,0.144568732874346,0.153827479024067,0.113305282425065,0.0330905867330954,0.102483465585129,0.172335523802709,0.100288523353873,0.0589601537390735,0.103003694168213,0.100146078718524,0.142062351983993,0.133058780465065,0.0740446928931013,0.108644472059524,0.052110061442234,0.0533703482166567,0.159572114704376,0.0985545223361165,0.0710208081956939,0.0934696416432315,0.0962792279839783,0.0321309603525369,0.157531836250926,0.120810763751945,0.0219022677643526,0.0420279813318366,0.0802703999394078,0.107015613826099,0.136414076839801,0.183666858010021,0.00947170969838833,0.0238484638439794,0.123036409752126,0.0174892480434449,0.00422382263407578,0.153031031024123,0.125992881321302,0.153772214593647,0.0777033739491829,0.106509665596557,0.0866103611999465,0.0918447668326086,0.0702106076522,0.13426941067721,0.0558664073086087,0.0425535654141204,0.0542372105986046,0.0892380977694545,0.178915912199942,0.100909347662306,0.0753273966984703,0.167347537359331,0.032432305956067,0.180317597891174,0.0588016602709409,0.115207163546795,0.0955658661708369,0.0805142535667001,0.0478485555147729,0.103756693656121,0.0655409048983783,0.165037423747213,0.140704201630488,0.0438068275307006,0.0477573390471566,0.158722643638757,0.0926592617553938,0.182534721608682,0.1455264540533,0.0314882479955766,0.04108722723365,0.11325749310961,0.156664402040163,0.116776345735972,0.131542965853773,0.14953073582886,0.131672690550932,0.0998176998155019,0.0719039704978297,0.194312351396553,0.11407450389967,0.131189040468278,0.0845208052932942,0.102737310272888,0.0758895363252767,0.147983795307925,0.0851260797821662,0.140443015600878,0.122988355105946,0.0652439949755466,0.196210163635861,0.25290699049412,0.0931930341001335,0.0303893974546075,0,0.103093007377845,0.0612186282212184,0.0900752440460973,0.127362425625779,0.0674712650927758,0.0510322081152574,0.0494664090770581,0.0815549485503969,0.102877653721288,0.0626878886754649,0.0253689123044177,0.0654760346758844,0.080150582389239,0.144978923721167,0.129887469640168,0.0349999950746213,0.059988371610127,0.0691471079338907,0.00297130635479784,0.0806243284426631,0.0761567511258643,0.00919707124766064
"23",0.122580527449737,0.0162316707022342,0.199775029306371,0.232897715120416,0.0811232461237548,0.0837975720814321,0,0.0778325633589933,0.0792146251995758,0.0764637008486575,0.10316965696389,0.132790212773766,0.0760605728371456,0.0743766853570699,0.136758692238441,0.1596991406797,0.151120865486698,0.178468360683056,0.896823110003322,1.14525522631126,0.818852473959588,0.932040081909696,0.973963822259744,0.530118242304702,0.131006851418034,0.175123447380224,0.164646238842955,0.128560493530004,0.377383643436522,0.233325025902202,0.236136663661268,0.382264162299833,0.290134821000232,0.653488419860348,1.79461576197229,1.84862722607792,1.18869051606517,1.38067142973721,1.23923680275554,1.16920899910555,1.14039073599011,1.25053885159995,1.18441147743979,1.22303458586822,1.11629763592846,1.05768165137817,0.861100254476992,1.22530006277355,1.50262209459382,0.158786921417837,0.0978711108837181,0.0701917586996609,0.0767916958156667,0.181689636645207,0.0150567193148115,0.0892413942906143,0.160623440743561,0.100855160104473,0.220569244397682,0.213103141179702,0.0637117539835149,0.131036163357501,0.110691538331856,0.135395536125308,0.0720625138808236,1.11188995390487,1.24184203454581,0.224118953368032,0.237566237042025,0.0993414158009197,0.167745418695484,0.344218480071141,0.089107445069406,0.126218168903182,0.0754413839496684,0.168703699290713,0.180267867747954,0.574627161077883,0.381432935996026,0.154713714025463,0.284535934238023,1.53870714240819,1.47458685793333,1.82429770264595,2.56986850917854,1.71453945983767,1.61016901580923,1.51094947523105,1.63211904240109,1.39908070951333,1.17114615547212,1.14285157317026,2.09742299336568,1.11451876692701,0.617111728572247,0.265219428756394,0.463578951367785,0.0274350714211742,0.16284196499953,0.186379248534031,0.226329248074974,0.126901156915971,0.161823920316709,0.108345956333531,0.0696147028271709,0.0995255292300737,0.190143056195341,0.188257585691197,0.0568362805280339,0.207470658057129,0.158312457022313,0.118722738761724,0.0498574442539964,1.17246523083812,1.17791020225988,0.181213320928223,0.228180417885276,0.330316586256645,0.123183431463394,0.120471805364405,0.0773020823789234,0.473209262071767,0.375345767639965,0.250675290479884,0.136010602143855,1.09316257455507,0.406771599925085,0.249321150349671,0.437382608819367,0.403243295621916,1.01052874509103,1.30333925271234,2.27498973687936,0.979660078134514,0.996191544710911,0.96129076315288,0.726110794395651,0.885758607441301,0.922559433866593,0.79586093913915,1.34552263804363,1.08243269792492,0.640398386362399,1.76833770187728,1.16908129651801,0.142476246177293,0.221767958219239,0.126915942501648,0.0816066297621501,0.162498752789475,0.0820555595870024,0.165918092154382,0.0873970229975484,0.267443384453844,0.0968502464945465,0.15369026771653,0.152168473019179,0.0828151271414097,0.103968886538184,0.126672575579237,0.0948031229467624,0.130554404977504,0.327074067367655,1.09956830254706,0.779906414417054,0.375792824429816,0.161886903855677,0.378607940736142,0.477758522099145,0.254336709914283,0.611641219812786,0.411076107205553,1.20516206274003,1.48764510641004,0.359497958969095,0.816064856996552,1.09053743996918,0.946856272676114,0.917986527293324,1.92039225056364,3.21231216109466,1.37511349651562,1.07950915778282,1.10048132986373,1.0781617445959,0.904546379086726,0.917951288978511,1.04900956181549,0.815657989513743,0.762043441051513,0.651308862195043,0.533150268324935,0.0712548032717721,0.0992720359746609,0.143139920082615,0.095523376823871,0.0569822885785903,0.0978699853898929,0.0709854562813035,0.132737657105328,0.208197073935665,0.179252827804634,0.145854617203291,0.192758829990597,0.482755046188677,1.30764445553579,0.629757735860196,0.190318633838871,0.130664866541311,0.0529216658664964,0.142410731897904,0.177914816019656,0.121439513651704,0.235335004569333,0.578714824741008,0.367743045597436,0.099400289882358,0.230477785430264,0.297105759566469,0.102809170291218,0.101832511384342,0.275030120550307,0.197217629956086,1.2100154403273,0.330209483142897,1.33854435483729,2.19672738928739,2.6139259751515,1.66800118099344,1.62113359408809,1.49850749613903,0.992568977836692,0.871004432420436,0.952023457660289,1.02802790164255,1.03220517490728,0.896843508191027,0.813086022901081,0.418880246511642,0.108339622584426,0.085993221713748,0.166005327757383,0.11915535718192,0.135661568367373,0.118805728927881,0.140953121484574,0.153595679354536,0.237114538176408,0.101446694000936,0.146628056806828,0.105701654011336,0.0394815883805178,0.123528723849165,0.34240285535034,1.4314018671454,0.389804086539011,0.126519495750337,0.0533942466224125,0.144577755712411,0.0485659228116983,0.0719096733576907,0.160551694381019,0.0805454314798642,0.15807205742423,0.327717811216825,0.120765802829473,0.269137620686291,0.370860802449111,0.223910609206905,0.351215577109977,0.35602307934509,0.271619921565578,0.512965312979843,0.330974768014737,1.60801718649949,2.33966633408937,2.49947594489826,2.12666368278458,1.33279085472088,1.0700762299784,1.01089563226854,1.26905117114298,1.12380909406547,1.12384130747202,0.866601777376904,0.500171512965514,0.462107819548899,0.478871278594613,0.415058028978329,0.113144185574608,0.128873304218283,0.166113543582512,0.140146414054234,0.0505731261237482,0.138905318107886,0.077759923017151,0.115071913592377,0.18552231265901,0.129564727790259,0.14408975062731,0.0680641794857883,0.59972159658809,1.21931922325701,0.186433582284411,0.142288160943751,0.0994973116667672,0.0865981549783468,0.106877843804679,0.116716714698425,0.131520933269668,0.284414670512522,1.10651972795884,1.10514070206624,0.279439251427918,0.210188984573066,0.578802172614214,0.773960488188743,0.297996971961157,0.203832833211474,1.18502368402703,1.64077066919626,0.911499058466564,0.54121489358791,0.442460807755141,1.5704623090202,1.21845285542644,1.02584257479085,1.0982152578962,0.976842051281101,1.05881981287915,0.950506072554977,0.876992254379592,1.05206632461779,0.9706393104093,0.912783069530043,0.772993731208986,0.510080808726689,0.310987724247807,0.260093683291199,0.154117283883593,0.0836131187475116,0.0767012669916525,0.169334231103362,0.102770610898611,0.164069921465856,0.157223096888475,0.170498688609787,0.14296630346398,0.0665380479467272,0.332093343354161,1.357816118398,0.232157724814681,0.14944468143935,0.0866039399609639,0.262895572729342,0.171092287868773,0.0780503925238949,0.130619023029194,0.15594859412237,0.172737552903224,0.128584928511965,0.186404814381543,0.178386662358892,0.100520141696779,0.146943524558791,1.12828476582699,0.443497953402085,0.364522068690804,0.973940313141443,0.274047912061509,0.327375589464585,0.391512144218114,1.64549193869025,1.31319818570595,1.21525869841513,1.10584772608961,0.855747854884949,0.974044611146646,0.878352267332077,0.992432473433202,0.941289462299155,1.04655682567207,0.935815746419229,0.653253562663964,0.614088325662797,0.341050326643407,0.104692888136778,0.143425019071294,0.145230427302797,0.105233878729892,0.132840597697007,0.200949362744678,0.0999971115123859,0.154516301835593,0.171961562207026,0.188936282662553,0.204648800483264,0.0614568623196297,0.179181616840925,0.117565429871818,0.228955047524724,0.175998069226399,0.067006372411373,0.0994601733575743,0.159225791444616,0.580445830627891,0.549878745355726,0.175177372882191,1.19813402934353,1.23302357933096,1.26846965261251,1.27507096752522,1.72401467344981,1.2707030842582,0.804386213570464,0.696118555731454,0.455746946476433,0.694936277709662,0.875547718058351,1.0494839909652,1.06795217401534,1.18303533969011,2.28302831409029,1.51415534613176,0.98130132439755,1.02223176558549,0.866393772623364,0.974768703134239,1.06011611334692,0.691275018467198,0.456503499597053,0.274281455680039,0.602215745100514,0.1380307789128,0.107694650447449,0.00839243825826012,0.177184820695692,0.184004974849828,0.242076682558702,0.140337005134322,0.128249880662192,0.170363550631097,0.109706707854106,0.153538171903695,0.191237182261087,0.0447073729015352,0.121997163419523,0.251474727254753,0.130773075924398,0.171939945012143,0.714313553202253,1.29056864613412,1.30523223312765,0.450239035945207,0.298030992988215,0.382950193240497,0.18932267129992,0.249567238310907,0.205412213490954,0.206284342692505,0.760194075039632,0.695584753272637,0.201519085621725,0.226374334606469,1.09916349309873,1.16425922882242,1.06942996741937,1.43322863093603,1.52616481748881,1.66646815398927,1.76223346768617,1.21524253481804,0.642673504916943,0.778047464241275,1.12400531205509,0.721433699433003,0.894809034151706,0.924570952321443,0.827278184663207,0.617768985426604,0.273333401244317,0.168066374759466,0.143170691504725,0.153995746142001,0.0802164296314367,0.113024349440847,0.139718289071006,0.0662465448029909,0.140779735953901,0.0241817366902707,0.11849357735146,0.217809381995268,0.133308054136104,0.721648430121536,1.18793369313187,0.318325797763191,0.110517069058838,0.278914268903055,0.190896808669058,0.174368988938513,0.206008647054389,0.260474398566088,0.2294540552132,0.150461274550825,0.111520172152917,0.292266137654908,1.19461449438333,2.0841822087105,0.262657371192364,0.262093051902825,0.31954485002377,0.0961245020052914,0.31275274453761,0.620049410464946,0.966541625980619,0.882427825645741,1.1581967256328,2.22452624028954,0.5958438939468,0.584101014784944,1.05425721608069,1.13277214219749,0.928975266826194,1.06637138966596,1.11585907638128,0.920438927426597,0.513738227280192,0.488864086236047,0.476148909969485,0.522982913364899,0.294566406413198,0.151871692605894,0.270977615222423,0.151596172119462,0.171786229591437,0.154749650646564,0.158118979925917,0.111618662561331,0.0275263062687959,0.185898664205973,0.152610835778891,0.344588791527457,1.0657511938676,0.184374939910339,0.145733383755999,0.154968173310671,0.0380860639995059,0.151900342086672,0.0476344221195113,0.0400646117387734,0.146299339892258,0.140341840518949,0.17181224162971,0.131750244022641,0.362920142929313,0.521879390542761,0.224384344890493,0.343777802081028,0.224966224397372,0.330342913644248,0.322634856139481,0.485448114773996,0.477387292601296,0.49660430383796,2.12241220569277,1.70113228166988,0.908471427870219,0.946982437490874,0.852611388449244,0.799424755053563,0.807806539919322,0.857417772700812,0.596911925098516,0.691589334124779,0.802616880409701,0.895905563103123,0.369381697394506,0.13242035547073,0.12931227721367,0.0230106548034014,0.120344147857445,0.0817334742049077,0.102190388257176,0.15074013731193,0.200922556327606,0.154690850858819,0.197884079699085,0.0803993009213219,0.0786353669256847,0.467151260759115,1.33109614511023,0.372597529992037,0.179755179927244,0.159857849199624,0.0635970498018383,0.208918769195359,0.111710862836734,0.10701405411102,0.141791264158766,0.0476240384702871,0.123514084144917,0.236977173250983,0.195373270495705,0.137284014814241,0.112094539267102,0.396442162787537,0.110853990062975,0.269130857602088,0.137480336179657,0.964326651951586,0.215557472069522,0.257356115368939,0.218705614495875,1.34250932439151,1.98025956036203,1.38835603059639,1.0839594735966,0.982445072838099,1.06804060943894,0.95095677944873,1.0157163271446,1.01671428464266,0.544527305088214,0.2985274599846,0.109476147803414,0.109936618487086,0.0206846837168593,0.0882786023789185,0.0827241936324774,0.222392476211821,0.14317086141669,0.0644431021960406,0.0650099093558422,0.120308378208622,0.100028858560446,0.171162283218274,0.213358586662437,0.341181747382576,1.22213044136687,0.122586207114569,0.0795671345110936,0.125118827107701,0.0599573674315205,0.157751673928435,0.121372182755683,0.0831383388851839,0.188391453938366,0.328561111398108,0.122033643182171,0.103215067147555,0.0690811985457284,0.123278416091998,0.151564330320668,0.366000315761671,0.154728581990292,0.291182434545057,0.397098414237731,0.198785340253548,0.893166306340509,1.06768053031763,0.912702015145438,2.00692214092296,1.64989045135881,1.02941289946718,0.933722014365546,0.955401145742396,1.00343081040784,1.10286089209299,1.3409633661121,1.09700450237773,0.677785161612595,0.377524379743057
"24",0.103789273651962,0.196564086617715,0.0949357589866466,0.0261415583302822,0.0593543486396444,0.0601298273929418,0.106753561545657,0.0446217313026185,0.10068566860874,0.0308149145260294,0.0528281660344193,0.0709925217467743,0.105828936114831,0.242456014441284,0.362466748297081,0.132800601006652,0.188335569025682,0.118338244684492,0.128693869829769,0.119110404615793,0.0485163187952088,0.157255279163766,0.112961560346546,0.131674646161574,0.0616645033329582,0,0.0951450629135416,0.0709947935190206,0.0285108977181541,0.197909364361044,0.162177291980165,0.245036613401602,0.228087973118774,0.407823075884237,0.443171087386756,0.440203234930906,0.158148488544908,0.24607817280942,0.384683450447192,1.0504420678878,2.86583836518145,3.50985383905771,3.38861124398007,3.08312659786613,3.05786611062296,2.44237810646642,0.190921570067513,0.110519302732493,0.0669146810926175,0.0332923260695221,0.144786598718801,0.0633033382645451,0.0785076169153158,0.187348379733711,0.132833607271839,0.0897125738029377,0.0711958611845333,0.0565732354311498,0.175002395122217,0,0.116135862712681,0.137043844876936,0.0470380195891219,0.135646026276857,0.113720006763422,0.0498808529305999,0.44858841341668,0.525708390731886,0.189750127214044,0.229826462651542,0.340153326656072,0.203011244766163,0.218179697730269,0.241219780154263,0.377549724935782,0.264310286192469,0.194921835312669,0.264091179590016,0.301204300338922,0.202642983168501,0.16708482444551,0.169371131111945,0.175565929517246,3.1480956514492,3.87812978806137,3.05044371898482,0.973098633769109,0.466383189667128,0.154551328612343,0.0648108850974665,0.00851462161495403,0.214248590201048,2.73573363646099,2.87673095345665,2.62045628246867,0.302433309357759,0.175438998289443,0.141679295210767,0.150907447888738,0.0348526149881663,0.109415398643557,0.0600440306030507,0.0907396678320122,0.103391987784516,0.105671937891685,0.107818239441159,0.03134827941065,0.068193080082146,0.0786084937142142,0.0525383414836518,0.186721006021529,0.185321242500763,0.181725003608817,0.405926199507195,0.601885639948892,0.137007846353423,0.162253521409345,0.144669858149584,0.144389563529119,0.550575898427412,0.462416884463579,0.343654068522389,0.26235405692763,0.281463668513881,0.385273563236275,0.28215637261069,0.908221021086423,0.263785393395851,0.436733952172963,1.21475828245445,0.905385977534129,0.434765569091099,0.419618134322034,1.76739784731537,0.686536023339389,0.326699594724975,0.397135090113288,0.352768377834696,1.55347285766553,3.38837332604339,2.99940261610417,0.722434505932581,0.187322375894632,0.0418320726330179,0.149451163039369,0.0888180820858002,0.00912957660044271,0.107008497065814,0.0247770943229422,0,0.10149156604172,0.0711924732251831,0.141363970770686,0.106997209618416,0.074725063744839,0.0666680046393474,0.0262322443852195,0.0707092759619334,0.152188791752931,0.118961556412638,0.235711926727008,0.246681078387931,0.160427658495439,0.466011213625274,0.241284766657161,0.109501916740606,0.0925144620171319,0.0608132004225952,0.162352102997574,0.0706903678165932,0.181235212658763,0.131352053738764,0.085060413887352,0.116178041161224,0.00534300615682856,0.089598542259148,0.0447005975968261,0.01434109635456,0.0527891192613355,0.0373491957763292,0.114739566571194,0.620943400562729,0.144068621867233,0.244737491703949,0.297499525506421,0.621002710791303,0.447942801759575,0.193862655487657,0.227602954840347,0.206753152491053,0.172857041307705,0.148063357125383,0.0626542633802694,0.121052402145503,0.0868362670363681,0.124354998162086,0.13223618933615,0.0351589303645341,0,0.0766194365544454,0.124983355814092,0.0911265300857739,0.0479349525002822,0.102367598654413,0.136620314543557,0,0.116831870523662,0,0.072881910945137,0.185740928782863,0.0408251197195,0.141072379065454,0.0901800258392824,0.0650059178369732,0.152925096695014,0.139293312422342,0.0831996839601638,0.116519992743858,0.0729988314191536,0.123243503041894,0.0732839982146924,0.0628945115625409,0.116839413473154,0.0471185393826189,0.0596869421325745,0.0597126457033451,0.0208998575052393,0.151930859017392,0.162429324194195,0.471393548113617,0.347646087992732,0.206175042962265,0.383796929756383,0.263816229532024,0.307515596260262,0.22984899406072,0.648960419561168,0.151649064199051,0.163277251365316,0.126900895373203,0.0356570545374695,0.129331799802696,0.131944961582991,0.058679039047369,0.115733278254448,0.0544261655147459,0.0752076775543579,0.18142727928351,0.0434232899924963,0.146721652032693,0.0849548644665044,0.0270613760610265,0,0,0.0878521759826185,0.0901175608483142,0.0664834314430444,0,0.00416578744401061,0.15902131888653,0.046686383387558,0.0889347358948702,0.184583817517966,0.134968246479346,0,0.127422861915554,0.00422834839764528,0.0461812436675538,0.0165192343438396,0.146693882357697,0.107309355006609,1.93871199059576,2.62866830963024,0.108842458255353,0.938831809954155,0.454100886285169,0.632524269086141,0.83795011743896,2.51254830735227,2.72082527372183,3.37044411722433,2.16504485617112,0.233142324844818,0.279817596835134,0.192912288412287,0.15541209099873,0.183476991063926,0.146044123014185,0.121736304208026,0.0551622215593744,0.0868221826173807,0.0936464649092002,0,0.0841553853828989,0.0647392789164808,0.0913217956694973,0.0939380828699464,0.00804273451892841,0.046910323393328,0.147484434976955,0.0553553073138603,0.0754678400779845,0.198497217749326,0.182393715438982,0.237035934577173,0.286832034350704,0.355045756168712,0.227133212875128,0.19133372015317,0.162245378916367,0.151219743475478,0.358754665029109,0.205185227402512,0.198668287059332,0.365947734236595,1.56367513833748,2.75425019101282,2.6261532567486,0.170384172407574,1.53689517383357,3.0481544187414,2.48879073904282,0.763184160990647,0.754328904584899,2.08340819899999,3.9560966217819,2.50203244978223,2.85510740690977,2.47990462414027,3.26321950418072,2.9353749151433,0.967749248488718,0.171754032643575,0.10606827878983,0.224993915625123,0.0230328635706113,0.110686555832885,0.103579675255719,0.0792366714803288,0.107964956415267,0.0228106730233712,0.0597512006585213,0.151933159753862,0.0847701431890094,0.142458774809169,0.135810930128225,0.059526426637677,0.0753976225736461,0.107492614212121,0.196120006208029,0.245441910201647,0.344536044517877,0.202647778307586,0.218287212013312,0.245967169125572,0.837399195391828,0.176375576051346,0.179739945618047,0.0551445829093408,0.0637051906208019,0.134297377725717,0.199243374396849,0.129215186263685,0.178533070411091,0.0730259121788531,0,0.122931235412777,0.236245960173707,0.188248946349829,0.186186275051321,0.128283026077582,1.40625437208061,4.19141653800025,4.41362221457542,3.11740603924318,2.40340336081182,2.09003766756815,1.75583593469864,1.85703071480685,1.32420637247036,0.423171057672221,0.558252201720931,0.304945079704593,0.154436754444379,0.207390188729475,0.0627609430917469,0.0409241637666497,0.164786117202359,0.14720729688963,0.128132057109932,0.0954936387146005,0.116091356982491,0,0.146145511263237,0.0538827298236188,0.0570061281279613,0.0720015235754558,0.0703019489486667,0.175556578512042,0.0764449866377715,0.233539691261267,0.472987824143506,0.09779280943105,0.278769563270911,0.400885559789903,0.266646184495863,0.341225416686862,0.175107470739133,0.197602891756055,0.265904265619231,0.283563103957115,0.472556185066478,0.261324523192767,0.157243265781702,0.134757354076756,0.135845916849525,0.0657656029202111,0.0909916251015766,1.64104067727602,4.10413319901166,2.85111864571513,2.34086767049426,1.72485750606167,0.215736196871972,0.348681496355705,0.345751769171929,0.35250664567305,0.495036994851404,0.603186544876436,0.17862710648325,0.116411140012594,0.216026879753203,0.158287918166473,0.110933012253235,0.225642831436219,0.0762279892706031,0.155438831162496,0.106651793451641,0,0.108117247885009,0.0484642850164619,0.0117326467106157,0.127630548866758,0.12574848660931,0.0956072961267391,0.126509368389462,0.122290204455316,0.233697670642513,0.269892116266618,0.150068895931071,0.19794146936769,0.64814670947299,0.423551775717337,0.187399232278482,0.284974475692048,0.121160536130042,0.420214119147442,0.289634978545696,0.172494495941593,0.3820240900682,0.299664626204096,0.197744539336219,0.0899655973019325,0.260389079746416,0.519413250594665,0.247452244561411,0.168415259480985,0.555758334439444,3.05522775969803,2.98463525932463,2.26860095699984,1.8579909220735,2.28230284118198,2.30463207230841,2.14367534839048,0.268381351456177,0.136065089853351,0.0745846663801373,0.0466072406671938,0.0936173295812253,0.135996622970995,0.0878085669546419,0.0892518101107049,0.147841882091651,0.0138529094023715,0.157224319705227,0.166160378468484,0.0835856623364689,0.13019017548412,0.0902298905433449,0.0724206824666105,0.0861958509237694,0.113476231857552,0.240731347972031,0.10784891793735,0.110993165055106,0.0511644444490031,0.81058572629625,0.138077893639614,0.265801326323297,0.213490852625155,0.231969567936633,0.309946053731764,0.18016539304598,0.235344961993858,0.882834581659078,0.497429462919115,0.234717254725185,0.352575092044342,0.311450731626916,0.312914291263283,0.273400885938265,0.408390524726286,0.455812869242337,0.330823484839561,0.20848275786377,0.254396668411844,0.73536666721137,0.423066064915532,0.222143489810326,0.327920637580824,0.30922563721437,0.755030164942588,0.220766747791937,0.0836299666246087,0.259844813353167,0.21064573885376,0.0437600693150444,0.0838719925070219,0.0843290514231788,0.0373844550703078,0.162654470677309,0.171085134796794,0,0.18312815864072,0.102889377916843,0.0984520646371288,0.0740011013725494,0.111854139001945,0.0525078560739856,0.0795690585012976,0.168433956992767,0.177663849993914,0.158478030402416,0.403784952639917,0.24258626250573,0.277120359967929,0.153810488558117,0.262999753471901,0.205030495283925,0.21358070086826,0.152871867720211,0.192483020949971,0.331908615818317,0.183729363543711,0.181443943429271,0.13420454379298,0.192735743502198,0.0444814670840766,0.193972481305205,0.118182645959199,2.3751943597955,3.21916102494922,3.18763264792824,2.84576268545412,1.48837507869149,0.662771425116121,1.3298407346357,1.4162528398003,1.53295591949967,1.85401678472549,1.41417206008605,0.641231998928886,0.19352549119755,0.120123431876758,0.12295939477031,0.120795461827842,0.0912420623995398,0.157327903250654,0.086107818533256,0.108584737893284,0.130259234630983,0.0885328392215502,0.014097070557664,0.0856038637263974,0.00368208501283068,0.225209124637791,0.116376412937257,0.0558701442651675,0.152894780022841,0.0483404240913431,0.154508411246695,0.158843363398025,0.279275160343059,0.257862543655024,0.434503093044422,0.413500698939061,0.267137884634837,0.0956045353392613,0.191852339902561,0.0724110999911129,0.327696889754916,0.223863132058251,0.00628924948950876,0.12420076225284,0.0561106422731653,0.0225147774390454,0.24321255272492,0.224668144057917,0.295778403308783,0.705849290783292,0.823141384045047,0.500709512983287,0.690836389943078,0.946981896631047,0.157799080595061,1.45554582113697,2.86348433637476,2.76387460535338,2.96654111420551,0.196050124571213,0.174036294031468,0.223903560308063,0.0661083410434394,0.165293963631737,0.104401547231155,0.116672351158043,0.151886573451575,0.0979227344907737,0.185169777598169,0.0973715406462216,0.13878394272674,0.0520095755084059,0.107713808410648,0.109243101233085,0.0229219459692204,0.068315020158351,0.190508600780157,0.386219281123251,0.353495464558508,0.718189514066696,0.337719599292775,0.290905772382147,0.315320780940643,0.344100028584566,0.25263809228605,0.366458614818884,0.337737109070054,0.283990590897676,0.220174860657467,1.18803974282872,0.391375720888049,0.298803409916351,0.240970258301766,0.257058967592891,0.354375281300955,0.779846251951079,0.216614729306848,0.232648436841222,0.480639379218322,0.376187246812603,1.09457681004317,2.09173299112751,2.24492770513569,2.07084600331102,3.04205661697056,1.36214082959857,3.22053128987379,0.913563569306179,0.204391732572317,0.091843336744085,0.106465872070053
"25",0.18444559519004,0.10720132271422,0.19285780712347,0.0832419473667534,0.177337299304982,0.212739736711282,0.27082930898279,0.277576629495944,0.162323549371075,1.30408260070624,1.20395677470769,0.280599973721218,0.349156637495794,0.300104177167416,0.360073122961441,0.164546102808829,0.146661304106112,0.190888316101297,0.117573417582804,0.36161101250366,0.276331975672595,0.377487906551358,0.326089612291126,0.525034691913243,0.0894660768551784,0.223418869285282,0.188842693417277,0.0785846386706578,0.17667792577943,0.243121930533502,0.162363816834868,0.190005192827592,0.28495826194251,1.37102519563714,1.77442096503563,2.04031208868643,1.57808517593638,1.54118027763762,1.54608628876839,1.3518413949079,1.34895978399807,1.53116412133625,1.4118835150467,1.54226814674036,1.29185314400861,1.33018808708895,1.2994432546649,1.71189543486902,1.44316034399355,0.95571005106502,0.274819668833605,0.205617313161035,0.194673843919883,0.197488253605264,0.23804114363519,0.16109672267544,0.209053235039128,0.0662517540011424,0.315364503293724,0.38726858229397,0.377069059024021,0.32362181040591,0.244324089603223,0.165561132719223,0.0804015794132348,0.142605264793733,0.146152774382676,0.252759914657279,0.300439554262918,0.181954280457519,0.439654188395932,0.115676500358226,0.212378142847472,0.292020239543889,0.299734671738027,0.752396512555376,1.48931256808965,0.784115204552068,0.230334787609423,0.291735475727741,0.254388451101983,0.142405155858956,0.611403712605349,2.12114293622706,2.62334900020341,2.89599537090689,1.48954627699134,1.36206257059569,1.42639261104941,1.3302037397981,1.44700576918347,1.38776367238118,1.56512960310229,1.45516885447807,2.11512616913874,3.90471330959632,0.928100753335435,0.704146296223512,0.340538251187285,0.190762486125665,0.259469874428485,0.150803428817899,0.260651511532393,1.10704510075633,1.27339337435085,0.325822136706338,0.272766339100779,0.446050800617864,0.374258441915979,0.261260091928116,0.263271199363489,0.105387294708845,0.183597162753762,0.172934868714039,0.340481192788568,0.215979589257581,0.239037876791174,0.347208779665617,1.45994349112971,0.60889166637436,0.274792408540631,0.310504501321975,0.305485137815744,0.222521774037755,0.321495966466342,0.277692908993034,0.302808545805509,0.536160893092446,0.251385147321821,0.554993263364671,0.215692213035549,0.297418330321335,0.398471444149343,0.455191238422106,0.797442463592409,0.349985796650921,0.285552815101715,0.378389542848687,0.353888885117707,0.251898612699789,0.296270589518389,0.20545204644608,0.277677965179456,0.763068645465025,1.3067773416748,1.36759021745388,0.573016072269004,0.118872162254204,0.217994471101266,0.178384564148269,0.114596640512853,0.115524480302751,0.274456434424169,0.244881468447915,0.286386757087638,0.306925679003607,0.352848453842667,0.261349274002687,0.203911365531546,0.162193735367693,0.213836989928249,0.366775457723618,0.409745760785266,0.23806441325193,0.405218209770898,0.654904941262825,0.48865855800945,0.304823741783845,2.35288263404225,1.61909532011647,1.46802280068863,1.54245764929975,2.29919308511171,2.61022975666921,1.68029150192548,1.37647345812903,1.43520558268454,1.48890356520431,1.17130669126565,1.3449102726273,1.849925486758,1.87884637726022,1.25141004617056,1.24856302910962,1.068521439117,1.19517278496058,1.15248688157663,1.14963449551465,1.41337321890783,0.398243025718492,0.2170331549204,0.180346651649618,0.231227760363122,0.165330926503682,0.227605699840404,0.0874756655816799,0.0886862863899786,0.178135241175575,0.40883649694837,0.262556512913183,1.14949090996027,1.3666470544872,0.28322945765012,0.381519609566569,0.32246021290785,0.338986866872321,0.235152181249443,0.20839628423852,0.214063505665989,0.79649437315536,0.253770001630352,0.183349750240612,0.15933717887685,0.162375122469567,0.11950836107494,0.016218219815472,0.185336802035732,0.165822555717269,0.20967689316582,0.208298368841902,0.543475349952188,0.746206674906732,2.08414955239092,2.07760766813853,0.669059742947003,0.242887849985056,0.288582195340872,0.435324802889146,0.391003750510543,0.67114857206557,1.85099274121688,1.96783763295647,1.16241023683298,1.02535474419067,0.101322445265267,0.197564145908803,0.141376572027538,0.169662414130373,0.354312142884255,0.0883414547088477,2.22385246801689,2.14715467826695,1.09400142270159,0.205664062908313,0.0513388760957049,0.21335933804614,0.268584252211619,0.22155319880549,0.163404381000885,0.0912013843424365,0.206449652565651,0.308522876414219,0.445562560513467,0.327904182462775,0.377959893209148,0.204906734930642,0.232206610692777,0.196799930785754,0.130510841441098,0.574953310740235,0.39675020139144,0.21819032204901,0.216802524392083,0.50268489640983,2.21392594945301,1.47155870930301,2.18112073813977,0.481865148751987,0.14886491974368,0.170444382607986,0.157537808788324,0.202375230019473,0.104523670021691,0.172715502695346,0.373011756531651,1.66322306904688,1.64756011011162,0.756106277620426,1.27649860365272,1.54358856661611,1.46571132308342,1.48834333649335,1.40170006506622,2.99297368554133,3.22481273732671,2.5892058592319,2.26157857494376,0.939705614787351,0.0662660538110479,0.173518720340721,0.172477765240186,0.104152701687424,0.270470569325478,0.151328161165233,0.178414863694327,0.277734982464034,0.210912066069326,0.965279983249394,1.21415147269706,0.35710071737712,0.251433869371223,0.371794787997264,0.365688492522154,0.188069685627322,0.147148550705113,0.14214338063523,0.148836264523787,0.312540740333652,0.2590665521161,0.370346012922502,0.418141182588151,0.473528275463331,0.331755100415531,0.19657786504629,0.220770152916957,0.0971157596977873,0.213955396432307,0.245389027628882,0.21203191836424,0.247824487459574,0.244755327518297,0.292569750929208,0.336592007317828,0.31952325153436,0.459766436603414,0.502065626591937,1.05830130780036,1.52482728482501,1.34772110017759,1.4885753064881,1.34372181625234,1.38023075925755,1.42005871044211,1.22883554842404,1.25814030040849,1.69582506025496,2.04426461293703,1.26667726839965,0.291881523684015,0.234922903399109,0.222861805626449,0.260185629159724,0.166765895187965,0.214002355361876,0.187564741576692,0.121456723150049,0.268497791689767,0.293770397692409,0.437391871972937,0.362278165954139,0.342150831200032,0.0904329992974752,0.0205385322917509,0.136512411270193,0.196564867617176,0.384959900265642,0.497263502099079,0.593588881530242,0.311915177626582,0.292267359331294,0.256991683973126,0.263011046590116,0.124776223914861,0.1431568275066,0.172124553234211,0.594961858434852,1.37106104514796,0.894156738904227,0.167807626225278,0.328930546976489,0.30767654475307,0.356832535413861,1.28508862141282,3.41948042959984,3.20872503724799,1.57257273148381,1.29228594099105,1.38459077854471,1.33283966387928,1.43941095174642,1.34313267647091,0.656412178375282,0.225387945820885,0.233900381216878,0.200362717763659,0.145592858646065,0.222614364042769,0.201020144394093,0.200073269698315,0.278834634034418,0.229513449711919,0.123747527671038,0.721803678084405,1.37408831382128,0.401723959115487,0.342543343435448,0.309932122461494,0.314730293547547,0.354114038473173,0.197352770571087,0.75444118230574,0.157856380666164,0.139360444757519,0.151116487442505,0.144518299395782,0.200270703757108,0.133813586524136,0.162053296876638,0.130686153735227,0.00942023290646354,0.00436554479622345,0.130332351513974,0.235887437982524,0.131975284670412,0.203089803727788,0.112679318093828,0.211156376359506,0.204580778611164,0.214238896927136,0.397840734705483,0.340739619763834,0.285425298984696,0.231415726917881,0.20181463471731,0.16604426872324,0.23767959060577,0.244568231069031,0.205871891581083,0.180838789301473,0.0600223036739539,0.0756612842607175,0.158792476031318,1.23947950688966,1.34645410529435,0.319691323375976,0.205160297327757,0.167953581066611,0.24275986786333,0.118262433606312,0.153432521156003,0.165771808941473,0.0650407373233806,0.258058616532702,0.377452222134346,0.375695542980175,0.397784711490031,0.280587963928178,0.210310031076771,0.171929130954994,0.0990255035873124,0.212871164811759,0.121140351087519,0.107034765861986,0.155245545104242,0.0884168380929062,0.254067998277811,0.200169632505788,0.298406708503081,0.154733024976796,0.129164155589196,0.0941000918222921,0.0877659837593533,0.735821580752559,1.29270555953895,0.514157727879595,0.169212289736796,0.238420865723089,0.339974008250423,0.280622534176845,0.360286288172321,0.202108406702151,0.236484092679608,0.175710875159285,0.142346388730155,0.174193791579121,0.149507219059939,0.297757733237087,0.436699476850516,0.746152625772915,0.42728311114767,0.231122299323189,0.0546061887733615,0.0924098861543152,0.195471649493179,0.128472162897762,0.177608160298243,0.154369836175738,0.192942698827144,0.15655693452095,0.338484383916118,1.25192813274071,0.830126010861605,0.297933912984184,0.249769138553334,0.345566423692114,0.432675501198161,1.84354081814301,2.19701113967422,1.15358891433762,0.0693771866397167,0.239347593379012,0.171120722508815,0.278846417192005,0.159416778478027,0.284438765712789,0.151258581953343,0.242686936116247,0.287949920062875,0.271910341191748,0.155600971326217,0.388013127345366,0.101126487167815,0.272207280303096,0.183871066611879,1.5623313061181,1.71253843988417,1.71541820541576,1.51434824636266,1.43456661286236,1.20099736672216,1.22362379653182,1.15158758177993,1.16748050486474,0.681270599264073,0.261271500256692,0.217896970166148,0.229602615271464,0.27065026552682,0.906882710892036,1.26299373571969,0.769145479795675,0.131965221636355,0.230060327530066,0.160598367027801,0.258956300980929,2.03884099855538,0.318199698141934,0.211054753869958,0.209787906768609,0.298900974805898,0.344202278411502,0.569908678902516,0.50026348411743,0.174894213347604,0.173393450202949,0.0983061085530044,0.0850202820077244,0.153124282664858,0.0871251922662045,0.104713938882627,0.250810532117714,0.230543194512882,0.506537795885615,0.223455024444005,0.207013165011926,0.337178859676248,0.467747212132918,0.235677695791978,0.767722969692781,1.29543601461875,0.584025393028528,0.245807772364278,0.367611177467035,0.355666625384577,0.370127786806827,0.348834697072812,0.317037109054771,0.258864095808635,1.51354386818573,2.53470160176518,0.483198635102159,0.543244088187833,0.325343740714745,0.304174665606235,0.281252842931231,0.0967261020080426,0.0427911929998324,0.193559805281121,0.170377000315576,0.131540556035099,0.0950216052689625,0.1616578397835,0.224858309384706,0.399375619935806,0.338509678019585,0.269440451897956,0.228132809382263,0.322801190531966,0.353313887048756,0.415652776862107,0.252534063903427,0.24534427759005,0.194560872987209,0.225940219395893,0.226313462899493,0.500105542727325,0.329104058260314,1.00344411152206,1.37664799413843,0.276766009298269,1.36355352795933,1.27695953500246,0.27322837449179,0.299882733133915,0.20976692027185,0.270065559231027,0.404485239698105,0.118476762652379,0.125696568916906,0.131432699247285,0.218683150524556,0.379720260419844,0.332591804144624,0.453670778022934,0.335338772326242,0.380291920038334,1.07930149197996,1.27807354034695,0.550303691958747,0.227769710342008,0.157916589122048,0.165912280840653,0.218947000386922,0.253678791616014,0.234624035719282,0.244769979950858,0.328163605183623,0.145061473281164,0.134807695613762,0.109404037327203,1.24780002942204,2.23303930650153,1.33233529123386,0.0897222560649851,0.361140637587012,0.318507011384093,0.346978006501008,0.324255695908213,0.39479334077268,0.143012338042161,0.671916860167465,0.264711160008188,0.127468119321813,0.276632220866014,0.367650931765331,0.539822810044639,0.289974166845042,0.193464066613871,0.134145467274851,0.200976543749817,0.190547647819797,0.191384332869464,0.314431887784795,0.168803925137904,0.255482928750234,0.179983301637181,0.210579201286102,0.965771084627851,1.14041676250482,0.357762658910957,0.373771407785612,0.342067405092549,0.26577934826733,0.205711936879966,0.227515015727322,0.177684836390512,0.299896462748041,0.194877745060701,0.188715700824813,0.0869469448042582,0.168568685714069,0.152119720383888
"26",0.0801152404716268,0.0310065547760317,0.0826428586573662,0.0952951830650014,0.126151919908878,0,0.0967016818196108,0.111322691938797,0.0591993322988336,0.0620609331575677,0.0169086902414736,0,0.127893604149463,0.115786770298971,0.0335469501434676,0.164412689719827,0.453867730183098,0.0837807506039245,0.0528909213950859,0.513273066945249,0.105707370643374,0.58920337807333,0.0260016672367444,0.0960717005055473,0.15812832618453,0.10458518787432,0.0721334063648782,0.0623588899105728,0.139120978394364,0.0577279704202266,0.131377997630826,0.0311099052052866,0.0599431039593398,0.129376052004743,1.13850042277936,2.53862549215719,2.43657999273998,2.09271837544766,1.10290971413721,0.299009902233685,0.367653592246497,0.460164958690969,0.309449602818736,0.333809177003317,0.333355039895699,0.228614821691984,0.0778948931217939,0.248898010205159,0,0.0830096989248361,0.0876614568769407,0.0952738179626905,0.00225614557825998,0.144633563905774,0.0216528634523348,0.0165504172791383,0.0849859090578586,0.0293750241897365,0.0732298803186298,0.0159402151936982,0,0.0761616952268283,0.00209741792800423,0,0.126605344748369,0.262830109714482,0.351523480170341,0.291980684923219,0.0740519370533373,0.146238681318492,0.153931005709751,0.614836103295235,0.429877989740456,0.201064089873265,0.152024775151818,0.992482840726285,1.02077337223201,0.618112878410848,0.330067549703031,0.989459655714024,1.03521037229557,0.600109882491596,0.0405843521401826,0.152448515341992,0.276704732215126,0.236661538484936,0.190154749004957,0.202748413720072,0.185382176085609,0.109983696113915,0.177533766057142,0.111347467077312,0.14481788729958,0.288403367503438,0.219709618909185,0.16219994059323,0.802213352261337,0.293554379464202,0.168295098276378,0.0738653112426288,0,0.0883539599957829,0.0884240937314901,0.149369953702684,0,0.0680766929074006,0.0504662776015247,0.0952175684045168,0.0468754046640406,0.149711359535148,0.0782818447513406,0.0124263726798564,0.0935399043698805,0.131533228435832,0.491345202500773,0.34810369777872,0.0811132020448645,0.322943152436677,0.139464590635377,0.36620427355732,0.107906229489986,0.117343408918609,0,0.0848046822198433,0.0254405839351554,0.039992437488437,0.00189379966363504,0.100016818110001,0.135741413146227,0.249535507086284,2.06947195934465,1.58649907015878,1.68659000736663,2.80801365295203,1.54757577047438,1.51110052486281,0.362317879621121,0.289077442774427,0.281283976700457,0.219751537186945,0.451084035697952,0.433896664372206,0.542169509809371,0.149189127185363,0.13096719867027,0.115287954935201,0.0161468273454281,0.0494186368241049,0.112902870294017,0,0.0484039597165216,0.0630194565168304,0,0.0843191148313369,0.0604750008359278,0.178672765762802,0.0918791918465332,0,0.027264456824574,0.160378545033536,0.132958812346326,0.0836020444610846,0.292821733766801,0.636330361446945,0.149931222002689,0.214383161683589,0.244160566848702,0.0880474639985542,0,0.0958118678142006,0.10171746360882,0.00754816327478434,0.0451559293787382,0.044978539485229,0.0932905688296254,0.382648375503073,0.142670222024175,0.265785143386147,2.508810701655,2.07343323963565,2.3582213372778,2.89017983229616,2.54826586400648,0.331523154570771,0.348841425078282,0.293266920557357,0.200403244834093,0.36796379766681,0.301067241225495,0.275122386405639,0.389394160019166,0.0716682660070553,0.174527227661411,0.047408823418497,0.0919147396581344,0.0734186675542888,0.121170152095959,0.0793012735800447,0.0999093116402585,0.175207435082883,0.0888612403449207,0.0181673342798942,0.0700815776714554,0.138659254011493,0.0721305253030757,0.0669075526503186,0.0692966084787671,0.0626165591485858,0.0907075162909308,0.261489429661477,0.326613265374424,0.233116694252151,0.579910319355237,0.54030800462005,0.268412315207246,0.00505499401952261,0.236189863480099,0.431638398023778,0.269771766811495,0.462723016826789,0.167367829643191,0.408134306129376,0.39091488317362,0.302442510530258,0.190043203252786,0.268559958541146,0.652709883123949,3.74438191134324,3.74757620853872,1.48273472983335,0.860231855059732,0.462998827260352,0.318839667984648,0.260261040074943,0.309908976101284,0.307532406523143,0.33467654434612,1.06838406254723,0.573250255375052,0.432893638048601,0.257445125605157,0.598294306055509,0.129062209498444,0.0392688139601692,0.121166268191906,0.152891616642244,0.0320044571397595,0.103006445420846,0.0591846715651722,0.0456391033869504,0.0641251335070107,0.0742598282930715,0.0957315774385767,0.0513055058752461,0.0135097796330356,0.2011574448343,0.100523652837186,0.274002606583348,0.212089502454471,0.250379860190652,0.305873066564955,1.11737765201805,1.17990448577263,1.27825032921537,0.702485893272551,0.317083164977155,0.21391879239602,0.373112260142192,0.119687947047489,0.314211728403126,0.426827256489339,0.217504001063343,1.04207344727727,0.935623541743691,0.629308880352844,1.90097849805401,1.4640153549316,2.38252971729896,1.10060843790112,0.345115543515426,0.466349494511105,0.211874383083581,0.24362590936752,0.32720118536849,0.367639599150242,0.308084724305789,0.145770459567406,0.193042824071045,0.159284223093094,0.127062041991136,0.129524467738571,0.0638435294303362,0.184800011583172,0.0525747857352458,0,0.0993539675206191,0.0939435097759651,0,0,0,0.0194353991700398,0.0473055781275452,0.0952518520620917,0.0451209660325714,0.0631451072105191,0.0946067484033835,0.0296863856655689,0.26942553181781,0.11953806294639,0.211820819835291,0.571295584972617,0.237994367100926,0.589745920691258,0.03464922745614,0.128063124321148,0.0357920819548085,0.583116197370434,0.175960778355688,0.177505599791239,0.221707794497023,1.05038869231825,1.11110833877504,0.361987449543464,0.155484954483231,0.329488526046867,1.99213948437172,0.953439671646988,0.332791133267167,0.270536963244716,0.38144537028861,0.196440193593132,0.139694203372335,0.816289163438708,1.12364133052816,0.0645335495469108,0.148540462438766,0.122275880498625,0.00426233385175892,0.0728271152696208,0.127800273796263,0.0529119702467896,0.132597696360835,0.137705767797458,0.14093965877072,0.0555193485404401,0.0656209878467538,0.107929236029131,0.0846084146110972,0.099131065744333,0.175666500882214,0.0621245220899516,0.115713095490825,0.207245933430098,0.237383361087748,0.224681699667512,0.5083357043732,0.16488395570612,0.521114274789202,0.00900764810765517,0.100524876757983,0.0547714994330572,0.00445452513366765,0.0761423014145473,0.0708583289240626,0,0.0365244877474069,0.0230645031275463,0.124957957627767,0.072914885453018,0.106596350927713,0.0776624051189873,0.0500174490951791,0.510079802509175,1.87559773177223,0.930467454600242,0.206557396498298,0.204197581734459,0.16573869347617,0.201298872127951,0.147741228750422,0.166809897899724,0.19181292811771,0.427068496438114,0.35187803345871,0.16442627036652,0.000655601964760985,0.062671325757583,0.0939490039297623,0.0874384522659873,0.162911842727712,0.0831132210811425,0.103867015609879,0.0853053559423311,0.00724593466097674,0.0744208631475287,0.0837884326190038,0.0311532713835593,0.110374534847208,0.149028946678256,0,0.0919650624788815,0.0622184264121062,0.449225923988814,0.467657724347693,0.213579415385988,0.19027038377143,1.36549343724311,1.00314442044285,0.271445619262966,0.142857460993176,0,0.0878231004372217,0,0.149462593555714,0.048662376962549,0.680968454924233,0.0980569758700658,0.153273542683786,0.28075255989401,0.119965772527737,0.254813224640832,0.811844835713965,1.47833646146427,0.853823432148605,0.422892367022615,0.300906584565678,0.247684468686322,0.297685913858036,0.355001661743795,0.347675398344064,0.307507981896418,0.292069744129299,0.313074166325644,0.162392653934173,0,0.153921032212557,0.0394672894305699,0.0699997839144339,0.0710164158762323,0.045526216298638,0.0270860016499634,0,0.00266847824631125,0.162777135276106,0.101022773368684,0.0337439939893372,0.0537919420999215,0.0510323858901319,0,0.148147574796008,0.172550918389437,0.31426022406149,0.133374923163288,1.00050172781704,1.68071023090257,0.681503522224548,0.160704156468491,0.492789226978917,0.864828873626678,0.115031749478271,0.187048390359686,0.591361835584852,1.09084987443751,1.03217853780984,0.518911844409763,0.0506110917878593,0.162746982508546,0.585974567629942,2.09550881610305,1.15490087464495,0.493626794845763,0.264537331379729,0.316104128371001,0.341997455419303,0.249096887149836,0.335456482343626,0.425223030889414,0.315660643761017,0.253687642480305,0.115849939188929,0.103121976231463,0.0875444574941953,0.134442114057001,0.149791112115869,0.142476435389203,0.118612349698735,0.0295436966273708,0.101657980993788,0.114679542046506,0.163206309034143,0.120777887260407,0.151291807997522,0.0805399470235236,0.14885921941079,0,0.0415628730220736,0.0800063381151138,0,0.245382296781644,0.231694353093634,1.35147420455426,0.839883207118786,0.284842725410331,0.646756196570835,0.467224766666504,0.398282133284701,0.474210553234677,0.415516476153621,0.296829434625664,0.309832498026949,0.150861089452046,0.227776465119309,0.219473896037642,0.152944303792411,0.577631164317158,2.89749231365765,2.51664578081711,3.06227185775025,0.600781322198301,0.367750059501891,0.385375277904463,0.270104394603028,0.31531015385141,0.328815245558994,0.400419964523571,0.424877479799057,0.160066919686464,0.155329067418447,0.0889774395457737,0.0922234580905644,0.0627777546653698,0.0971654465125807,0.00418820343339438,0.138486986180017,0.123905963018589,0.0440746951239336,0.11988098953568,0.172597212052958,0.0517476394797741,0,0.0155344112752317,0.153398080718672,0.0884140133479435,0.089662170221941,0.117552474676606,0.203567133332389,0.196869261240149,0.100480740636176,1.00052702757654,1.32474064847439,1.06441159853613,0.584139761196799,0.550607837621771,0.252223035947287,0.604096114821917,0.107001169884879,0.179051456670756,0.177785271111606,0.0666214807672309,0.761930151088831,1.76890600231658,2.95963172186998,1.97496934426925,1.43640184110897,1.22444042967304,1.06705665022866,0.320568385892102,0.283815553178685,0.136632901354064,0.337049629771657,0.278300555198136,0.158448155574098,0.469408102447375,0.812775661519637,1.03775283601105,0.421507829700435,0.559870998482155,0.19179317455272,0.396759766906318,0.0754649424376353,0.0859735864474343,0.126581120811459,0,0.146223652149996,0.00289166444914604,0.133075105148807,0.0739451393866141,0.145486148375479,0.0946162840063808,0.0490694128513161,0.0038742738351949,0.0737599027695131,0.221838124128714,1.1238791100865,1.17166500047858,0.296376448343736,0.336248258585151,0.227657334641194,0.415548805108264,0.227553873283717,0.300532057129983,0.27435774413333,0.448015934598058,0.706307217921967,0.731173184699795,1.84443889369859,0.657132107695929,0.0700046926325122,0.188598542614558,0.100068664220924,0.977523869314486,1.37952096471666,3.32103981772145,3.00545127754618,1.66434034922358,0.342560328744837,0.32980917364133,0.410844477160287,0.350152743018007,0.271970212917805,0.254008922722355,1.23427012873044,1.06564881887205,0.536194815461561,0.441401260532197,0.5278306925885,0.0257388017349047,0.0437637085544444,0,0.0798138389279145,0.0409290374063453,0.116040304718167,0.050859069614699,0.0793521135628447,0.0793292360828208,0.0204968380211895,0.0307377707808117,0.0783517470384301,0.195643616953393,0.531917633303943,0.12692195916827,0.102420815497563,0.124005641764923,0.0683024115790591,0.0436944940434348,0.144397890359422,0.0342588791344457,0.0743001449636913,0.0760707265952273,0,0.137473515127963,0.122867062813034,0.0583457999713059,0.049818380610195,0.0865821844409163,0.0571746948980167,0.0731534330360659,0.116593233930325,0.0743561186096967,0.155487891729487,0.032892778427151,0.0320902703534786,0.0422126841331558,0.226349752827511,0.349238761504671,0.252539258573551,0.884337556725834,0.385816440446562,0.42575963894936,0.226167425872643,0,0.120028612488914
"27",0.192287723723559,0.168865118026151,0.210679527818464,0.139634681335544,0.200354421449013,0.199469756797643,0.164500964604526,0.0746651177190242,0.0550550707414867,0.0878566477383288,0.160789839071387,0.145395910834882,0.0556361196520775,0.0433006693989347,0.149604236464192,1.56244675157662,1.62866546215476,1.32009907194662,1.07207854986556,1.41543092049769,0.36970541288479,0.160438633380687,0.192260932918299,0.148092654664668,0.154060390868724,0.134641448113601,0.103222217101783,0.274011660497312,0.0733615802940146,0.14864429753658,0.00585356639687319,0.0520425720728722,0.117395099637593,0.339074019369489,1.39047385814879,1.18856301271531,1.2074877327251,1.69727466545268,1.70675197468573,1.18979078900793,1.55497965089791,2.08710882170397,1.62651203036742,1.15511169805574,1.40834702092553,1.10911625399502,0.103747085769499,0.181454140266108,0.0775202427464135,0.134361178857035,0.0157656287208944,0.0856698501081748,0.0378569099672275,0.105235896298454,0.192758590396169,0.158753633646559,0.167114853437461,0.136858421129319,0.182874065921805,0.232589471520235,0.105490004253263,0.163064471925409,0.094277121703294,0.0103289097620874,0.698317245524531,1.49847256930374,0.939536471720214,0.393837226821473,0.306482910614209,0.329659800633858,0.288841869128826,0.314758745348518,0.413441958432873,0.464094013966099,0.167514838395424,0.866749644582521,0.808864520448591,0.236974861971698,0.117753257379445,0.225381932700694,0.230415355390864,0.790368398069504,0.827325316134857,0.497804884124241,0.901632661979253,0.498256918461251,0.374347428108039,1.43867946339034,1.63879572885212,1.53296586175941,1.40498668137553,1.36563518100464,1.19389191494008,1.25731933273578,1.13354264709959,1.0705980284391,1.18863924178756,1.10347702053343,1.08766657586426,1.12392739283185,0.249747175884826,0.110248986929259,0.16380768364249,0.148180032162877,0.138650595210729,0.105986483725352,0.199054669711059,0.256467921404998,0.094363467283137,0.0900665152236474,0.286893678564943,0.305399219572451,0.439085745886548,0.242094088702871,0.205646478080211,0.241518892922914,0.166200650377709,0.267196301016559,0.144154868841042,0.154732833118481,0.251533265145106,0.215608453995228,0.201573820769764,0.344900717818181,0.602431223865015,0.272201724113141,0.197608448887286,0.279318748345821,0.152512500977592,0.132553105833051,0.145038889337637,0.145691701359282,0.356869295775553,0.394567601853539,0.26191877385004,0.54731935652901,1.65978281807903,1.26843044243246,1.15656408200169,1.76041700159998,1.45449020984828,1.29406480377255,1.02161850412456,0.282167053724251,0.210765952046862,0.173521772832166,0.0482130564608626,0.137922713300004,0.160425239637577,0.245227220436521,0.144704605048461,0.124440927087327,0.0395694658645016,0.255403366416113,0.145688312462007,0.110641943129781,0.207920605592912,0.147163521439358,0.0695189291031412,0.698430933653669,1.59915897768815,1.43637897883958,1.0514477486508,0.748196567586381,0.79408796106086,0.257905758112936,0.12910389106056,0.191523284145463,0.251334973327576,0.234222737976751,0.438485813786164,0.833246469961085,0.80533494090958,1.10645421014876,1.26311119694701,0.977917298311763,0.775286854283725,0.969520343114195,2.20428294812906,1.97898096899439,1.51977299078287,0.915098972818768,0.950900599945518,0.993986048994665,1.09493005007765,1.89688089281791,1.79099230472456,1.64071018427584,1.63623638072835,0.589909063825107,0.18992234466627,0.186134778619652,0.135533335395387,0.214596361436792,0.0620045681847605,0.147932405581173,0.107845897276312,0.197700916665358,0.196317828448989,0.227522557090524,0.179026081375126,0.231215972592794,0.105410410788329,0.183104389711262,0.151412676690591,0.28331606859352,0.123670987083836,0.908834057210999,1.40380628452459,1.17823618142308,0.266528602341711,0.175164053937619,0.232614153473162,0.21372415487218,0.347321013658321,0.890237734727374,0.772926004049924,0.439808949570315,0.201879697000069,0.187732118809185,0.224294466745442,0.194386363140065,0.129741101420401,0.218083165439686,0.344709750061287,0.334345211557281,0.0912890371671027,0.377513877966926,0.321691176942374,1.53665739648396,1.32546097145787,1.08234760305451,2.99635189706312,1.52617091015312,1.10653804214769,1.01360908596736,1.12422476882023,0.228585167232101,0.203844333787215,0.163231819847585,0.137370745140347,0.136705093183141,0.27065829502752,0.105407476871699,0.182785011129184,0.10797366005572,0.140232706478598,0.150571535796159,0.207597219750538,0.143673379318311,0.120374823110812,0.0999498216452278,0.13389856245735,0.114224170294412,0.373575211565885,0.36002693233974,0.322067868270668,0.107293319065064,0.152663185026076,0.113440040658696,0.203793658657249,0.155472824757737,0.270030108139545,0.205217727526331,0.175251916605257,0.0810470459032949,0.234014824642904,0.139475714976905,0.246664989671991,0.266885844902479,0.0966952696882087,0.0641938799939211,0.221245010939681,0.167515266516222,0.164918839233031,0.080214347345553,0.0992632252196248,0.160444657419239,1.14025269276232,1.07838515865555,1.01319625935421,1.02647405627197,2.21945481368856,1.82994143699935,1.88981503916846,1.65646454054205,0.562267940964663,0.219962423833889,0.273110498753174,0.200041604213499,0.193740364406649,0.121982432521008,0.245523003544453,0.181001880993812,0.199318157093837,0.164851382233666,0.12490581440183,0.142170079462123,0.161262145513085,0.272755109343681,0.259538551705938,0.222405579382155,0.296057295472606,0.130632779585823,0.757002264860433,0.774430781860808,0.274201008699212,0.235678185175953,0.299637202390398,0.0766100719039584,0.247513152768974,0.170007222012564,0.146315169642437,0.272463116189919,0.287968499093435,0.261000398078447,0.240576266295297,0.305465487805879,0.119176665018536,0.179728518402668,0.211200831728463,0.139417739184285,0.181181599582225,0.240932056960969,0.355277108164575,0.921712514467639,0.541307089840966,0.976420579300455,0.844645489334509,1.03985160801261,0.8742454216468,0.92990667834284,0.825217170113036,0.504785201315402,0.224264459999002,0.297835901983675,0.181208220008514,0.167903740266065,0.163167706440968,0.102334241790745,0.118029937765812,0.092427000899529,0.101910814041943,0.0745044816827432,0.117135795426265,0.248428996454235,0.111112731251817,0.206085802130673,0.243424260612872,0.19540024300205,0.218077617538882,1.19687989104626,1.8562336629499,1.35535955410312,1.02866658820091,0.423445555414161,0.151724798637458,0.215403274768167,0.0832324332955819,0.242208786539999,0.178549866024195,0.142609589937758,0.394888864102331,0.273835632210634,0.178489616646401,0.206070306490879,0.153600077963701,0.0895581548812313,0.283923929877737,0.776444897671997,0.685890679790383,0.857067839434459,1.08663748879505,1.27898179706713,1.10609471706922,1.24288921215938,2.59776009235861,2.33496914763845,1.72823009039012,1.69544223735557,1.1185822138807,0.855849362858981,0.842590879019937,0.55317039917578,0.217786086484995,0.0886073448381412,0.161150697283018,0.22681512429496,0.118078066549427,0.221838926964055,0.157026745315414,0.227611430267071,0.163612462488562,0.0898231106565698,0.141363531282035,0.0748005691919846,0.182194100232946,0.0629576728039557,0.264969889983744,0.207500779527395,0.336243043007977,0.258500791369358,0.397869827322874,0.636346695079857,0.355308124141287,0.316529012952783,0.212905091736503,0.342163133716444,0.499290491947905,0.824757843522416,0.663328211951367,0.338637472118851,0.0615848377241349,0.269687278766844,0.283634291907578,0.123197613266742,0.701963725252724,0.850581696067431,0.656149662054026,0.4203082290596,1.1661012031215,0.48781333838354,0.403052890695909,2.03894430868907,1.64894553415534,1.60284990840425,1.44254925544728,1.38434261868783,1.28565748706341,1.3582889800667,1.28750246028143,0.895970498531407,0.777129596076191,0.566092503369684,0.208091051503349,0.0785019229131138,0.097479721369958,0.049904799764185,0.242588922482581,0.1816681681553,0.229886898009606,0.181438536851742,0.193656717220962,0.0919088615403532,0.0536620368789866,0.222151004691164,0.0970275540811336,0.187139892178564,0.289642955246297,0.368996711473196,0.381493030569684,0.311056436331991,0.352480218459369,0.297236526020591,0.252059306221511,0.283420314583671,0.130776893395968,0.115767594195302,0.221096159373625,0.450234054857444,0.56281535715667,0.262706848081965,0.135522958094896,0.119779207449278,0.0536568859268424,0.214134041298255,0.23962733045811,0.25312287913266,0.391065910355221,0.310183986852078,0.742008896017679,1.36130710123334,1.04849141481592,0.970574274354118,0.999333163315609,0.940274524463552,0.907278295246556,0.804415613896525,0.496298189678911,0.0844187935676011,0.14836818658624,0.160308376117071,0.1567891148503,0.107284159836401,0.100687961441624,0.0924467352139323,0.169207784944564,0.0513470993182936,0.115011116909164,0.244633900788166,0.254923505154273,0.223330067991563,0.182129019339601,0.283580230807795,0.235967983501303,0.241586831840141,0.355716394977576,0.459598378571205,0.258664642470143,0.4935860650423,0.337002978236703,0.179388995413437,0.121714841324081,0.119661954716255,0.239929260376698,0.171604186627051,0.240340002446848,0.184572330523579,0.212880824163729,0.141644272326169,0.589842057071808,0.229478666870118,1.19163909554606,2.37500326304879,1.55649452049068,0.951107940334711,1.13658235198931,0.211841192180282,0.389980839668641,0.561204974286814,0.483823734106967,0.523952912519651,0.4265996277603,0.29335639785362,0.385912296441843,0.248675443395036,0.141649951562666,0.174816013566686,0.132459898605386,0.167950063841252,0.191832734495238,0.169403659562841,0.11420169273078,0.157942738582929,0.227363205327326,0.124533146095353,0.140240247424259,0.163851066293129,0.121269526235255,0.160241349813795,0.202486549511216,0.166390813956759,0.154239422019143,0.207835061441021,0.291548438006302,0.32838934301319,0.43628643068307,0.245919594115846,0.213675294295949,0.215775700895496,0.10286812117814,0.226859608911307,0.278208270160553,0.396920697538408,0.216164796292666,0.992113103891738,0.611841634657141,0.384205989192574,0.206626146212059,0.176579835879514,0.204786058953388,0.305793989270899,0.349757786665124,0.256784522169911,0.36897912133164,0.198991412976197,0.372035616063929,0.328172750402614,1.31041496915895,1.90548659783072,1.57315835767151,1.5708262222213,1.52838108290334,0.69980469350544,0.186316128350953,0.129255510364869,0.214625500345672,0.119638025029326,0.0661984825104035,0.199832361145061,0.201326725135895,0.149066326743076,0.106968552212044,0.0874352411442502,0.2219637021137,0.108878438234992,0.084408673669967,0.242243692632253,0.148877279087406,0.140237663892444,0.18199017982433,0.299707533915522,0.317518060147174,0.395002032777637,0.506511357659428,0.85257128945903,0.370775874212243,0.308777402182081,0.319036754787945,0.149853008764765,0.187667975498345,0.180609961576124,0.135322666924058,0.215286486311397,0.215728712125203,0.0721725982415102,0.220816179488198,0.190686285929786,0.271914668371602,0.285748342580487,0.322555744123469,1.56907996411514,1.2567505824885,0.74307298155201,0.430773792327083,0.300246393772828,1.22102787730091,0.970595253890612,1.07406153264181,1.06164714790627,0.954392431976644,0.927279247480164,0.999145245284373,0.186669433473151,0.152456939224354,0.237489796753094,0.0909680609436819,0.187134538414196,0.14067372849326,0.234908709044245,0.179072621568053,0.215041269024796,0.157399347455374,0.177028913530445,0.0723231944209823,0.0733272105340482,0.154543476576468,0.205100058812107,0.00169033469656119,0.637992715034595,1.17312509226026,1.6887698605653,1.10441703328993,0.911605122784843,1.05470108593441,0.767867649021021,0.211681921774651,0.142958714675035,0.145298449099867,0.199946913767607,0.2061420118107,0.18979909926752,0.210890900309903,0.158624569181294,0.402675922269408,0.213518922787418,0.19909195804787,0.29364565280671,0.302615345502883,0.356056197394459,0.441731092794479,0.292813557981844,0.317731110602983,1.15927204454868,1.46264081874224,1.79259925856779,1.65154443960677,1.59241579436002,1.49407407007759,1.49959991552629,0.941553240591692,0.171624633775559
"28",0.324577916586787,0.0382659401892503,0.238380658969326,0.249791807566452,0.224593051051901,0.158415101447145,0,0.180517110339765,0.228438979632786,0.240201100441903,0.149887383609886,0.186444371414529,0.187169048734619,0.154329982093141,0.245928038368717,0.145191563527217,0.494177307192273,1.21003905891965,1.03106813759245,0.233976579411652,0.401787974300554,0.480306467121973,0.489017572677199,0.181033668544501,0.154011964776569,0.168518409922915,0.140047636763222,0.151503546618445,0.0673957254568378,0.10184482316239,4.697115807209e-05,0.277498811521975,0.239588332289412,0.405383799810645,0.733431404627453,0.987998265620244,0.83781752823837,0.895125764329193,0.494708799372633,0.803916731489946,0.616424218330074,0.542532646825789,0.521606848539664,0.498319399119549,0.588959923088278,0.553192893751834,0.149530720004479,0.242722784824027,0.174369184124763,0.176631452933821,0.277302565236923,0.208934101822924,0.194301331648395,0.130117295551964,0.212611261520164,0.193614969661904,0.214174565642725,0.191139413666841,0.177408260995929,0.175502032876602,0.129353564729855,0.0859371268491249,0.127826577122098,0.18887809178167,0.541902028143827,0.780125948969651,1.47353261997471,1.77330940216263,0.55715790914041,0.285918501296502,0.293083623274193,0.176012666950969,0.162816464099903,0.202758962222717,0.471161705538452,0.344303021330837,0.276646436825083,0.357721028508799,0.398944066286052,0.370006448245547,0.309178103780863,0.235959571842942,0.724089802216501,0.599016007193791,0.770448985724019,0.4727948862104,0.571021399471198,0.710349806599484,0.71308673693148,0.614847527436261,0.467316582667247,0.500546504531398,0.606455574570539,0.510806895043054,0.38115031624097,0.299136572710734,0.216624471402837,0.174326606471562,0.20354191007502,0.204200558520257,0.247208916282967,0.0131916151934258,0.18501368540397,0.106046681457354,0.247782173106649,0.253543619205928,0.135585240209472,0.248889699618012,0.0891557657352966,0.249968772758922,0.177630280606841,0.125775398552113,0.248272221471301,0.200833528254746,0.344999060045446,0.351520436180344,0.39627217561871,0.521872967144834,0.341480578518603,0.453519138992761,0.411140998409792,0.298315600725126,0.325887898443098,0.414367549413039,0.417119337340212,0.338391969193634,0.699952726682368,0.536958757790127,0.353706629019698,0.643293195070879,0.816514835310982,0.52303042404398,0.293298813961196,0.410731903447701,0.776889298771766,0.500068307696572,0.699723545793636,0.751182877386496,0.545904346094634,0.303493785108383,0.40504210578761,0.391734267041568,0.219355247374389,0.241316202842406,0.104588619715144,0.0956968526448713,0.249694620189433,0.222850100009141,0.278525731348666,0.252922619612398,0.263129332116389,0.187618081213191,0.249628044562328,0.180576671548,0.00175954487983002,0.273661779398445,0.204787220604522,0.171947507639025,0.286869968436824,0.205743677590481,0.454947431383585,0.563515132542375,1.18995907116327,1.08018841848011,0.626630488285778,0.314783282486341,0.183104075491927,0.0689507412970033,0.594532271824359,0.820959025709172,0.910836320764078,0.815355110349499,0.912488179499794,0.742193235837243,0.346814844401647,0.277987933146206,0.228765881433108,0.390358518599208,0.562583005285846,0.542013836238762,0.362688776842735,0.394403988903751,0.472798484343586,0.378835579955156,0.274573153381857,0.566047621526932,0.366826191303724,0.340433516275711,0.692232480048054,0.654552125253565,0.665708538191053,0.178046853030009,0.243437361876166,0.0832907919536011,0.186791194113525,0.189984792965721,0.154118858020317,0.207020731887452,0.127408151937175,0.132917294761587,0.213062061034355,0.185392111447136,0.278666486280221,0.0609796357707066,0.2616526003147,0.18558614843746,0.110947292610786,0.162523408509243,0.115861205207652,0.2242100517084,0.0150591876401261,0.49976848249262,0.699639228325543,0.356035355362408,0.379588013073806,0.426883826370284,0.392375988981092,0.371862624793065,0.400869352713417,0.882811973527663,0.585344113064343,0.697092059751982,0.69013514482558,0.578948238976218,0.730093907636595,0.835605417167918,0.511720560711154,1.12071466387595,0.776093095069978,0.721490885505725,0.63237433829483,0.673859574929304,0.641643092954426,0.670407379930788,0.586843504825828,0.60038479697555,0.328683133911491,0.299767323810147,0.396193208414275,0.322106024566162,0.244887340578774,0.299656030366853,0.19885321561492,0.187690163327645,0.15602794695387,0.173716260995628,0.149502170027317,0.20203732990023,0.176778637773296,0.146943879431967,0.294721584957552,0.206635958665829,0.227825975585678,0.204306147002253,0.0709516730832427,0.216195490922691,0.568464321914212,0.222503831424093,1.26885473649353,1.51828297423335,1.47582424092554,1.65508435974482,0.453556852811926,0.123480662205529,0.32985127052424,0.253617128102281,0.199665268640083,0.258463991453164,0.171909537605527,0.198107520282766,0.04884666056723,0.167610334916159,0.192103372045417,0.117362737846647,0.292372192785588,0.0976394374242942,0.538817401170122,1.0620868200075,0.835444872446709,0.791753867936219,0.60282794959618,0.746244879765786,0.72643009684409,0.378172659197803,0.402065114651419,0.338745223470551,0.282634743557468,0.207411663858781,0.270628274625769,0.171648823858883,0.194042168058934,0.251169877968979,0.168027955818218,0.105733649421413,0.157064903425669,0.117501109715661,0.166072349589998,0.115204391466135,0.0705795326815265,0.287589546947881,0.0616058018439009,0.213661339859575,0.115842024898952,0.244118620710108,0.634013095550793,0.179814331060899,0.273376711800954,0.917798165448234,1.11639561599628,0.264864138208798,0.240236949838546,0.280350055331591,0.0754158188636725,0.186232117208933,0.143391045168349,0.173831277095991,0.131171900998817,0.164536361150932,0.26130044180867,0.150330020918863,0.1129358357264,0.388125553152356,0.818316901643408,0.960565584261268,0.927143956952283,0.850450695091595,0.584329809393647,0.39854893366577,0.363380800122035,0.411942897964931,0.621181087041858,0.61984516484755,0.375635904363801,0.378793228600458,0.321417956264448,0.509979611935107,0.441905218416035,0.368183165609457,0.131728797847486,0.234055222819588,0.121927178097671,0.234270333551574,0.140465610119606,0.150173918809136,0.218876792193372,0.173295824636925,0.191278721408003,0.165363779410108,0.0946362036188577,0.270424937159588,0.187652391765246,0.410087341174766,0.560613511304111,0.117082199472885,1.30455389407434,1.16808500811391,0.371519594858021,0.426265101086636,0.385958236284177,0.396545932671889,0.242527243150311,0.320947141286662,0.206267351079781,0.325344717229379,0.179865311929421,0.310354026114145,0.85930628937821,1.25675068521048,1.11227562147723,0.537393520135663,0.426971356871754,0.260325899089601,0.287328605408898,0.622267485932325,0.0674273383903359,0.244099098612685,0.268520049037931,0.0719185490692817,0.432204922864047,0.321313989829724,0.446577897536555,0.465226466291973,0.336155742647492,0.407528040783063,0.423436976847924,0.322130461493066,0.264259653250231,0.148468727484254,0.104594531654136,0.269787321321743,0.192457258224,0.113937271897966,0.217730290754074,0.151367215592336,0.16597456402466,0.0398800066949171,0.175448854302713,0.128057716535619,0.0999288105659062,0.137773441335272,0.128903265530114,0.238469651828362,0.226110438349403,0.497920385403417,0.404893490324321,0.287572186743041,0.349435019836238,0.409733286926773,0.302130712887075,0.41408831702353,0.339107461071743,0.386584645473544,0.282331429043957,0.453701657938859,0.417091182503127,0.337234658253574,0.461756770759357,0.569916290128184,0.402046310602373,0.536367569359066,0.373054532907979,0.769598101398096,0.421243871730241,0.41528331082168,0.345578660593137,0.451024690731936,0.388274840791433,0.494568965080292,0.420429538811393,0.383631685769521,0.248941986737825,0.290965919253817,0.172926015127201,0.355641711501363,0.164317697049717,0.193199827131625,0.11911139795993,0.163956854617247,0.16756879369059,0.180441048354221,0.201514145515885,0.168870495803664,0.218184815074507,0.154990051426447,0,0.18844710890922,0.107189948856059,0.124118521635964,0.16363395117944,0.0679810529480926,0.438901810740372,1.0977769536789,0.37596448051604,0.243968079409002,0.197292783972932,0.141283497216652,0.142745712949641,0.0781064162073584,0.199565578080831,0.124193927392231,0.121807273283803,0.119435735280138,0.310639192397009,0.335541268164782,0.364447240072945,0.292110129673223,0.487651843347879,0.340538031957463,0.472485283849601,0.517348944325603,0.282612363594733,0.304060734260758,0.451726212402615,0.446789139178992,0.442046871294513,0.38996745724744,0.567902524616107,0.423204035306323,0.650442589086325,0.343406029687072,0.330881585104202,0.248295403634948,0.235626077435065,0.195673695763899,0.11703799812512,0.0388525062313692,0.312354821177695,0.174117912140972,0.136491702680755,0.127454413513355,0.135292614208187,0.247650024525666,0.198695355247876,0.177470821303566,0.152936200780839,0.116813102137688,0.237811296579541,0.240066336067881,0.913131750470854,1.50822739332305,1.02057378086654,0.691714836511706,0.378375257618423,0.331301334975099,0.525901308926832,0.308270245052846,0.459771608263346,0.301009666905364,0.200730335806655,0.327242398479904,0.223394780815798,0.292798802422618,0.523887119390861,0.53827234466851,0.610322178181013,0.329489973735772,0.407085849703616,0.333372739682957,0.354305621800417,0.414024143887057,0.305180415961679,0.368411938482181,0.716126707288375,0.992369043313491,0.357075916396201,0.484888069672957,0.335064367063234,0.257509391414527,0.306662677805028,0.280301549392675,0.222955762349648,0.358650865189774,0.226089512265702,0.150863198495681,0.2725341644577,0.220116597050593,0.350264000941378,0.228577971521245,0.194759462866663,0.124853342466493,0.0516626973257866,0.28347988806688,0.0708093152171587,0.0390635494552392,0.182242610317213,0.159833737635071,0.765954517469855,0.348408944549321,0.206524666904269,0.11011739376796,0.180555046029822,0.171426347653165,0.221724106838611,0.179793768961563,0.556039085656474,0.421207912223022,0.458340056491633,0.394799399088903,0.466580785138274,0.466958382812911,0.433911195131602,0.397747037489572,0.376185918919669,0.259193726582905,0.192041014977245,0.45676165773572,0.496831726897857,0.459126033952885,0.550312134707196,0.295254924702301,0.395203798624445,0.493186196530612,0.447199688776433,0.303560183991171,0.325841541664567,0.282286766980731,0.233823740568978,0.23636244712532,0.199559098811773,0.152907795318702,0.221156912453899,0.169764148304069,0.204461421906367,0.239244260665847,0.200604719158617,0.298841948817094,0.317306320958248,0.267232435186192,0.258784306307288,0.165449808190142,0.230537260787702,0.153536508650418,0.570090583862722,0.364331073140421,1.03411780269277,1.24664697372207,0.431668906902123,0.572910301871489,0.309277393738109,0.280359767306796,0.443126842885966,0.478168842699519,0.416830311086234,0.501910380362087,0.498664303742993,0.378243008234268,0.39603165195359,0.607268946370015,0.35847996023288,0.467625589092483,0.793303635668439,0.319587669473742,0.375262307633701,0.403412777621916,0.320402825923607,0.365374979303628,0.423829204044784,0.437452618759532,0.368878191677854,0.292665041678409,0.076172688574444,0.217424746855951,0.12932943395298,0.0267516244442828,0.173601747782983,0.182804264125921,0.196337308857806,0.200237313938118,0.194670959934789,0.216041733319197,0.179239855589007,0.0740992872594244,0.0929535785067477,0.147144479869533,0.0387290519696623,0.115531130853698,0.0995186142357131,0.133625268605348,0.0449811246053373,0.172511076162582,0.152822143672372,0.192143367786498,0.431560738029387,0.617960107941392,0.379101487304301,0.33383854409465,0.308005885305263,0.393083352152716,0.259398148460623,0.240122780007199,0.223062100288261,0.113965318260278,0.229356881426759,0.202116669347144,0.109871018167789,0.280680936919011,0.303613115293365,0.285570640490067,0.17024400334866,0.12925397759856,0.407550453916264,0.951029371060222,1.10965919015139,0.511352089679304,0.551707945454215,0.482722436976095,0.573429381213289,0.406015476244795,0.433797630371538,0.194004418700514,0.418402910172666,0.131358208929912
"29",0.648957302734059,0.148858813892127,0.130163852737827,0.192107660846642,0.17481069125876,0.138786508962462,0.127337993900625,0.156289227098883,0.277508062405774,0.0566724391446214,0.265023723230277,1.4881750085481,0.210430933230029,0.284394266645894,2.2547996734365,2.61607280182457,2.07100072594822,1.45967959528755,0.293201438112807,0.391326354121772,1.95632612550638,0.757859933800737,0.0667924767754134,0.14955383093184,0.159083508150884,0.179627471198576,0.0533987120033597,0.141249917067926,0.0165884544615214,0.141659234358181,0.0559431272594876,1.693212540602,0.10790055368487,0.380720292073087,0.252966802109509,0.32799447138402,1.43354345396954,2.49781051214916,2.48500525270585,2.36487250271016,2.20574569370096,1.66947685049409,0.375837417824814,0.298029927080462,0.274132588074276,0.193597709283353,0.171440199468716,0.14376881775419,1.31691877423137,0.518684435560688,0.31674185239743,0.230382573192644,0.0491185296733441,0.130972229659668,0.203415044408756,0.0418219908908809,0.134450801966321,0.0855619178682304,0.194463223475539,1.13456369154186,1.01224065349902,0.161363690036347,0.133009717382427,0.506561972216145,0.107831353471147,1.2943955672923,2.42994337264542,2.24611489993387,2.06146677744339,2.09136493355459,1.3235234240646,0.076936510727032,0.181850593323167,0.478182627676631,0.0290481861258555,0.147818671247194,0.0907188220507014,0.116319611587903,0.0317206970169476,0.142788080785332,0.0206657397709371,0.133882547238938,0.0900526039036735,0.133792247589305,0.179476922280217,0.19882322484821,0.13280413575427,0.0913937408332216,0.191047782389338,0.0917658302683129,0.11443297467501,0.184615461596195,0.153629347488986,0.0458919594092814,0.161476198670644,0.10738971214022,0.145261908925038,0.171337733459128,0.115078758242294,0.153410478512934,0.123287750510071,0.101187006104994,0.127451730794993,0.018919553285629,0.0608857308166463,0.935348077768854,1.07716599500081,0.147328427576628,0.0583320664883377,0.155639181128418,0.139372285792472,0.0291548005108957,0.152939076033533,0.199545175195729,0.217264143622534,0.215991198633998,0.179342192356297,0.196754013293007,0.157259967918662,0.216616421201904,0.106740792647011,0.167642830807851,0.11335482127464,0.0921388335681816,0.101016111147515,0.172078738315561,0.0671047995779767,0.127775909711195,0.11784021102049,0.0950609493206964,0.0398829210617866,0.215761702465934,0.294684379388276,0.19042227599843,0.110555998232953,0.12365217128908,0.0877933114360167,0.0718644268016188,1.66065554884879,0.320076094362362,0.113213050637572,0.188648174836289,0.162778001622029,0.106526408899096,0.176626151222631,0.0518632293153611,0.228687986538436,0.00359470354415817,0.254386745504972,0.0887520665921949,0.214929189718897,0.109045879390709,0.205699786654662,0.860586193833564,1.08900967046081,0.133521968423,0.24383591263762,0.100329025765385,0.0938092779572399,0.0674119839812765,0.159055474004243,0.230254672656187,0.221544680372815,0.0941493894861686,0.218420556982186,0.20055132919778,0.120707763750892,0.144026036585802,0.073222675555594,0.230391706697469,0.0427682281379284,0.142918269575958,0.0984727732365801,0.177859333866612,1.76223122689568,0.124750368055346,0.0981018384639009,0.0618671663418696,0.044296803457956,0.108971943524986,0.148374759135067,0.0899324372218938,0.0816691405474341,0.195525694011891,0.200175318567385,0.124329450522048,0.170535846781568,0.176376894601326,0.0451813026875506,0.163395122733498,0.218640769660135,0.225112413810041,0.187395284812227,0.136385408927101,0.145344202234743,0.078424647193388,1.26521188515129,0.794575735460091,0.150458946420836,0.139833903254911,0.0805615306850525,0.0404586496005781,0.211700301309573,0.169316757429388,0.0962777206877271,0.126323511554306,0.0955366650912057,0.252804627701193,0.0669615201789727,0.199904594617989,0.0338685932651663,0.233646495687911,1.88542103528305,2.11286545204335,2.08356591596916,1.76204296704579,0.189828474469361,0.170073930144651,0.12992695936875,0.0444429822671649,0.192983151102421,0.218359424287404,0.102930712982584,0.159632726192133,0.16835022917894,0.0902911783409177,0.0813777010678357,0.213624210764649,0.553618961582027,0.326083667479724,0.376425991051815,0.325067499933066,0.611272173942108,1.51760924817986,1.14606059769272,0.43003787583306,0.416120948394102,0.448760035044308,0.397705229697006,0.359135840934625,0.278092547419082,0.204897056670909,0.130160630406607,0.151506665764356,0.215389144366934,0.170752537567371,0.175005972027646,0.108807741430555,0.243006154801179,1.6582699876789,0.00285002428715736,0.179301289519605,0.0970330631836401,0.559363279225189,2.49274377183884,2.56977287001923,2.02717877279492,2.31552527221601,2.33313167475432,2.32779773309547,0.690788080660593,0.288623916024605,1.79071671508137,0.576261646629403,0.196053020458319,0.286161544692918,0.313997900867583,0.250014030745423,0.462446097280352,0.402175683166281,0.461349857067107,0.460230345513436,0.298531174290623,0.338020691034411,0.378498388873752,0.694311743019915,1.86982472123779,2.07878377518875,2.57108468703775,1.4049978645061,0.487895535462335,0.450686328156344,0.627425011824173,0.39218306919943,0.92401187637265,2.57070239992379,2.09275464000217,0.611350395667806,0.116639193593497,0.173606706531221,0.176871405042816,0.148043066827003,0.311009258805929,1.49404491682479,0.2309255878056,0,0.178460638070355,0.204740101250425,0.309583420287486,0.135648106510086,1.33419476302056,2.08200291319597,2.71418614002457,2.55782098599562,1.97163519140968,1.95696889977903,0.424617529454889,0.083063514446795,0.271884589783611,0.0802277970137206,0.216310639164825,0.188300907194867,0.367915014247076,1.30359363506677,0.130509373626644,0.18746636345757,0.153280783511933,0.141437065435332,0.072609157051992,0.229497757439182,0.305526736019995,0.46635008988009,0.382051058987285,0.351643844070088,1.92576868661273,2.18828949070184,2.05593321170163,2.19558107484369,2.17907096078388,0.378612573247082,0.306769259426194,0.180895214375429,0.0885066016107626,0.325748539872187,0.237110214440615,0.174688658364523,0.629983809661718,1.24048885133833,0.027184975633853,0.131015283028225,0.0889357006610685,0.133551766875453,0.031533226863075,0.103758531215406,0.0742500426347807,0.114011560425454,0.150367815317474,0.206742462474554,0.123216570199228,0.325757606957284,1.73887437548953,1.84120396008713,1.4019670193593,2.06448015759003,0.865194438014906,0.515670668610779,1.04650673022546,0.973042408413117,0.196573877888,0.128214174424447,0.0619280634025595,0.111540375929156,0.163127559848653,0.171314688629811,0.210324329663676,0.353472453645421,0.0863762679404676,0.218403508536885,0.223197849149099,0.114102283501804,1.41065933885664,2.45449383640367,2.45068551863651,2.68482560790551,1.36847887930148,0.315642098468538,0.289931740306665,0.30794667934172,0.327513478027033,0.392349672163176,1.20975616245627,0.986812953334083,0.208449124838241,0.239218998427057,0.297753157524887,0.145622368907957,0.17842176077266,0.135383193996181,0.0941300000709673,0.0227325147688264,0.176366163017138,0.159106281792186,0.0358511712792655,0.206858091782158,0.0802797900039396,1.70810822989792,0.229087171135999,0.388344134835993,0.265192523124612,0.183862405959205,0.552463587748903,0.729329738598194,2.2454176044789,2.45573216753475,0.166463687354406,0.243734957593175,0.329585010409896,0.141320310150806,0.136875743151818,0.266633584365158,0.196205299892949,0.251414131775569,0.0326094975725839,0.24351739288858,0.247768253354295,0.318341819669944,0.228378442795803,0.263464391849762,0.160114707886758,1.98699952644458,2.56118184212419,2.11279338383184,0.808990583269618,2.13261502499277,2.13770230616496,0.646311377439804,0.246638795362116,0.294706288346361,0.282868135782407,0.179245366168178,0.1534943695004,0.158766927752965,1.58309406668335,0.20203850212222,0.156337478090807,0.130608452901818,0.243458514084189,0.0753791993872619,0.0962482267628347,0.0869678431007966,0.147164487061202,0.0726734756082887,0.125910963678938,0.180108417859095,1.47918216529446,0.489374810107576,0.0652981830627302,0.119994194401416,0.177054852577811,0.674574102922985,1.80366523345743,1.97323528424557,0.736404177020976,0.159149084831616,0.175202869899271,0.24179058905585,0.165669532858002,0.202256632109468,0.124896230047404,0.359377810140387,1.84770979338457,0.276944043549129,0.45056608397343,1.08709664186635,2.1124434830602,0.590790933758607,0.274533354982778,0.360957276922904,2.37231385886649,2.84481969157893,2.55438539801895,2.51177916588463,0.939236836737754,2.21962324846249,2.33565461891578,2.10478069847168,1.03848009686629,0.308930629519303,0.187787925954442,0.0619056680032659,0.178183912400817,0.264829322734453,0.125658256132105,0.44668361572537,1.20793849862319,0.265570177261119,0.030918849480138,0.168352639461383,0.0732184632475419,0.203123058721074,0.157170980240965,0.101763594237032,0.106273812959947,0.244204471353467,0.598185710768163,2.7003953769842,2.39889781490228,0.314416472416539,0.238356003704679,0.0736468636096269,0.111105428248275,0.156879725577045,0.0998669000406071,1.36662230310974,0.865916915248328,0.260287207487727,0.423175505800405,0.32849885455292,0.366644166277322,0.465141730445412,0.491953463041004,0.339889502024713,0.497302189398613,2.21508092644478,2.57400357803046,2.45010766887384,2.78861053279918,3.14877929447535,2.52344649203483,1.86692044386786,2.3014799922007,1.56111432342616,0.409223412695482,0.537680601714357,0.441991234110693,0.3579326368744,0.327086406620996,0.258824109788705,0.215053281528645,0.175921332192772,1.67347104930134,0.195625235960834,0.139512652890988,0.164107435390832,0.212322740248526,0.115219896711919,0.210509997468398,0.171682627789845,0.143407602449642,0.106052073557203,0.218742231406243,0.166053537913194,0.52725493850536,2.5855479356559,2.67062868958085,2.43787856449023,2.20105422371382,2.12639793309544,0.235549601003791,0.173395073322511,0.0384364746425686,0.106500628134606,0.158061337334435,0.23647165684724,0.169183876444774,0.214827005602921,0.312138536029007,0.442203783981722,0.234001862804101,0.780581105299782,1.43889205707478,0.304754371227021,0.217231186888294,1.52394803907061,2.47538193902904,2.66329589124796,2.36989270750056,0.785727152347634,0.274069114748146,1.10390709893556,2.52896589205754,1.64900469381077,0.223936422340812,0.07095769876092,0.170904443801649,0.145048980686273,0.146208146004202,0.0880088503581833,0.164903956375124,1.226783423491,0.60896666660809,0.146239542244319,0.113657591818427,0.245453588687365,0.138975464156994,0.140575617163246,0.000134430857751389,0.25800569191686,0.2665295210156,0.24837968416048,2.14915728504095,2.27600343808382,1.53177008942254,0.141519857197187,0.231650190303759,0.0517041866131205,0.158204619874049,0.204348698284549,0.14794518372385,0.1627701631305,0.197903285449789,0.138411725585147,0.199252629581309,0.0789748811893644,0.211685041938532,0.313933334164479,0.287912451439399,0.204111575013912,0.263519802874832,0.47857483271202,0.375614880958397,0.624101096403428,0.477259420832644,1.21077989213761,2.52089946274887,2.41746132905867,1.70753025986653,0.394916085190663,0.32705561616647,0.336710554082171,0.614482425004584,1.38210242821241,0.309991187183655,0.123800806238724,0.207856815976096,0.22573790911069,0.0490248487285131,0.114293980308907,0.0743982296879355,0.135746126759735,0.121342831225986,0.109794948143782,1.35590016799912,0.523740180339087,0.0939368587329679,0.120444019860294,0.124211330178483,0.190312816570292,1.30344587768412,2.24884553331986,1.42671330369459,0.269625707928191,0.319146541222992,0.282639745263322,0.185405999778455,1.75400041798925,2.68017123940035,2.00381112024218,0.238116483345082,0.188409040851867,0.289544042086663,0.276030037684643,0.220428448634836,0.104628068648514,0.232775200909066,0.270067601631681,0.215563502861323,0.289017870373218,0.729381059302325,2.46091371519005,0.695633872512392,0.223794728654881,0.264065012455132,0.223683311099586,0.0758205203062484,0.167527778946713,0.29156791055318,1.35871511040648,0.403137929556093,0.418685721415333,0.408201940518127,0.172269627001396
"30",0.0525298445294367,0.134083048143442,0.144645729162648,0.0672549784228652,0.0860390789900846,0.131166219790498,0.0955544201876796,0.0517934067663631,0.0854089849530577,0.0758158994704419,0.133014095975914,0.178458371562214,0.0717846362305543,0.0850696574818805,0.119096864942787,0.10200906964254,0.120179979759039,0.0322703922567288,0.109841421164912,0.0835653460366113,0,0.118892646568952,0.169189418809328,0.0369053281224122,0.0721370951050637,0,0.014859392605543,0.0442907486973294,0,0,0.142819160553286,0.0583447486402112,0.0294686115643523,0.0863216981489399,0.158264785934037,0.210320304615904,0.130200500393633,0.17833987667496,0.248538681087244,0.127118291253238,0.164446546266297,0.229004779675992,0.181078419658112,0.150626636876724,0.112400931779648,0.0075546617391061,0.166010993330954,0.111959757481704,0.19963358751759,0.131332032889982,0.19784449821186,0.152894524669027,0.155824921760697,0.126153106150953,0.018808801539319,0.0833346671656454,0.2003680637054,0.116927127306841,0.0633694762228225,0.0411924825237621,0.12533103743754,0.130079337832094,0.166330535017267,0.185140957881594,0.10905224262599,0.132214922010737,0.197505241697777,0.153450682280711,0.176887415270357,0.0568099974268335,0.064696594219193,0.106084018346031,0.0584506159792518,0,0,0,0,0,0.0495123217785847,0.0804712811136918,0.0354586170930238,0,0.0113683585356518,0.116002610532963,0.0922238015458759,0.183701826368327,0.218376671496944,0.068870089877139,0.111773587591396,0.242939327953199,0.0979072858869042,0.199022007547514,0.191752532948499,0.119966256421411,0.0364725406768011,0.178741291454938,0.0364672861065387,0.200268356434205,0.119982758205188,0.0993738830560496,0.150879213290398,0.228669724344303,0.0695637393262541,0.124921394464481,0.0866652225305587,0.175383832482918,0.199475312860025,0.114006118131511,0.134379772530196,0.134680266273297,0.131692056761509,0.0380228883491871,0,0.0694098671133595,0,0.0790905869912288,0.00370644322692546,0.100439917812416,0.150210198456661,0,0.101514524692113,0,0.045351959436497,0.0264192187372317,0.0210118758642796,0,0,0.0353075449893108,0,0,0,0.14679296894163,0.108147300347019,0.0274822212148498,0.207159537287717,0.221564180646907,0.166969894907063,0.19300398678785,0.208466953535611,0.0990795624332223,0.0219418768102966,0.00436590469808664,0,0.095757104168131,0.0981761253773937,0.054623864029881,0.210283044919992,0.0600906635283957,0.188143913624481,0.0159588898738909,0.0299498285904716,0.0217587236963037,0.102739240398009,0.119075269663717,0.117149913613422,0.012168729834719,0.195049781469302,0.127756513614031,0.0720057075854359,0.153036870881844,0.162497837690926,0.0548171474440953,0.0356149462186783,0,0.0313834437884553,0,0.0587279459330906,0.0386133549389756,0,0,0.0956779683133877,0.029718230997234,0,0.0574220239746934,0,0.0267525230785548,0.0533856044277111,0,0,0.0607533602904552,0.0955495493374168,0.220063817842829,0.084009836882077,0.0547525224199353,0.123477717290158,0.146294256781227,0.128304573376207,0.261818222606358,0.150653456461921,0.0481039922561675,0.0740245478219584,0.147001152769237,0.0203983357362765,0.161509476969369,0.114182641914451,0.0115786493262724,0.152517322526614,0.0565933150898872,0.169608236065687,0.114468035264842,0.0455839194667261,0.123349549707602,0.0592857753158643,0.0888288547134849,0,0.0899737644955306,0.113720557852599,0.0580396674584588,0.144489167238103,0.110107989127355,0.0595788197655485,0.170540503791215,0,0.0567130381379506,0.00292334557543434,0.0343392050365927,0.0646460573566668,0,0,0.0920972845292007,0.0203247702932808,0.0312777900049072,0,0,0.0873109331296841,0.0918749267282243,0.12604072031217,0.0785067321760139,0,0.186034342301799,0.175347064756079,0.187750956649623,0.0930922293542423,0.0598417220780131,0.101135143171742,0.133532480525371,0.115566418131562,0.109785496162507,0.130467874839212,0.0795768786379068,0.147054826917183,0.0883361584516485,0.0489045237544063,0.191057843664944,0.0852349916419365,0.098952029879876,0.0893047395453712,0.102959596608891,0.085837204967682,0.184385213179806,0.0570946947616606,0,0,0.0454654092661249,0.0253511525561315,0.00379259471901221,0.203416196433238,0.0949233891070548,0.11257949674407,0.0304321679943668,0.0648504387692594,0.115917925917657,0.147265160308675,0.0847796569562442,0,0.106889755028794,0.150545612616234,0.0400904521791427,0.140352288210008,0.0874021947919423,0.00750620981552178,0,0,0,0.119470792480162,0.0736561572614416,0.0864514259623577,0.129554333779285,0.0179045455323945,0,0.0904448394680534,0.105668731596292,0.1310182406202,0.129817134736331,0.0361991019356732,0.10791933328881,0.128729956536797,0.0737716813243943,0.11967686858151,0.025185052594875,0.0114730526561247,0.139497274859178,0.134592463270924,0.162215943303548,0.177597552139078,0.065921376817135,0.148738480677847,0.0315234881893187,0.127141219308417,0.0884546067288721,0.0527045573839266,0.0800074382781164,0.0784429733107936,0.111630074947974,0.101550704565306,0.160928883835997,0.037019104957538,0,0,0,0,0,0.00881729976781053,0,0.0344536444821194,0,0.10471633486543,0.00947912888824596,0,0.0296399011082804,0.107391124637811,0.0178406675564584,0,0.16809172167272,0.300325839863169,0.215048106425975,0.0928398282820423,0.139088713310285,0.177515893463502,0.102645380424227,0.020945300333025,0.102966739376957,0.0321567296381977,0.00189310133713799,0.123399013903535,0.121113329748255,0.0993138840599825,0.168836525243878,0.172310965917235,0.1386890112513,0.0816550995584208,0.121791260841842,0.0825437561100731,0.140814071109393,0.111321452958654,0.0994350899759864,0.0622794547577962,0.131940348090556,0.178302496620355,0.158844784302882,0.0834484788603887,0.0757090589389809,0.0451746202681966,0,0.0787708530081905,0.116604142620376,0.120995651273428,0.0637449216910434,0.139038024049515,0,0.0589717565389063,0.163258754092913,0.0213069209508139,0.102539648822638,0.0866063484486577,0.116222787968736,0,0.0978101745403794,0.141555376740026,0,0,0,0.0462383733059639,0.0317334496775758,0.0743268932134031,0.0657870468332937,0.170311091146505,0.160956421565305,0.180936592350682,0.189915024877552,0.169338900224077,0.0901184181554393,0.19775196150564,0.0857383741802588,0.120516797738666,0.187866667839202,0.119428936494214,0,0,0,0,0,0.000499648951659626,0,0,0,0.086686562748204,0.124068175772396,0.129549624403833,0.0996246162042589,0.169884288358387,0.0295128174723198,0.0495517196032054,0.00506773332148844,0,0,0.0687810156775805,0.00351981962659235,0.00476472737070947,0.00080252613522714,0.0628470866859636,0.0465934744815669,0,0.0530937211463096,0,0.019531087859888,0.0692647180693943,0.110668351452067,0,0.0312365383826562,0.00732671660761963,0.0835617009712497,0,0.0631424967229016,0.0108404638790403,0.0410934212829623,0.0285672997854871,0,0,0.0525472671220067,0.095469399770115,0,0.0723320957797233,0.0426543006040835,0.0798090562436501,0,0,0,0,0.00364390145966402,0,0,0.0286082394329068,0,0,0.0126310438297926,0,0.0322476040438361,0.108336806629982,0.0515847333018571,0.00764428764793478,0,0,0.0352501255193398,0.0796663466844327,0,0.0588579122429991,0.0818917392138072,0.0962680542821734,0.0809287112890367,0.0213923643931672,0.00285997604173554,0,0.0501309984662632,0,0,0.0202787438323533,0.021320035166753,0,0.0836237242906173,0.0672412551733523,0,0,0,0.00424292942096047,0.073259994729247,0.155692794791086,0.082441482478585,0,0.14584813782541,0.0349173757790101,0.180983803015667,0.172265103571781,0.223508682056422,0.160842575761712,0.0605753940536988,0.0835417412099565,0.199291720306395,0.0896941382845757,0.0665877974256188,0.170984166638198,0.208635084137047,0.147162939688872,0.0870694738170377,0.148260942195924,0,0.0765300533399314,0.0204299261097956,0,0.0133429580536442,0,0.0175386653587479,0.0658309455227671,0.0374967790029377,0,0.0119840350746215,0,0.0449190502131422,0.0703270567044202,0.0252634852225258,0,0,0.108707979204534,0,0.00471724269863279,0.000260600712025663,0.0995656303831361,0.053090626253264,0.0732454251564582,0,0.0685318186827694,0,0,0.116993171548015,0.0978001919311871,0,0.0158782961204043,0.0262944887704872,0,0.0393320373189843,0.0496778164605566,0,0,0.0342477591805865,0.0411260194801359,0.0130671951755855,0,0.0281298697365723,0,0.0254859822049267,0,0.0783622663501466,0.101308169240914,0.0257261053138472,0.0295520806182163,0,0.0483965567722727,0,0,0,0.075262493197342,0,0,0,0.0687068399792306,0,0.0195146557782665,0,0,0.00910227601687138,0.102057000490905,0.0716629182925201,0,0.00717418436217115,0,0.0268725573961276,0,0.0242724414585366,0,0.00440214785679813,0,0,0.0712384928147696,0.0791978619585227,0.0627644275889383,0.0653652090269484,0,0,0,0.0227739458704785,0.0065961759456391,0.0481213757682753,0,0.0438452100861584,0.0275778519274663,0.0182109999218839,0.0745921970467483,0.0504907460518516,0.0159361877948837,0.0239630834006254,0.07471143357233,0,0.012140743271954,0,0,0.0207934739328,0,0,0.00384885630967542,0,0.0268189764111181,0,0,0,0.0176956389335229,0.00582082659845317,0,0,0,0.0421482558537124,0.0683694626183863,0,0,0,0.0233626651789598,0.0424308793023839,0.015399694764359,0.0722214687392038,0.0105038406115866,0,0.0388542250152066,0.0041655708053731,0,0.036517368695664,0,0.0402121204155363,0,0,0.0827882334817447,0.00258656703500491,0,0,0,0,0,0.0579107124007095,0,0.0139486581724767,0,0,0.0696092208231032,0.113547255283168,0,0,0,0,0.00650960846852328,0,0.0293654182039614,0,0.0197799976836092,0.0123943274874677,0.0291609757698585,0,0.0542834958101213,0.0363012068859756,0.0834024129497296,0,0.145113430009342,0.0868164480651247,0.00334583409896556,0.747065372508997,0.385159436419734,0.317839220014639,0.655428159291813,1.28139469650248,0.68162937078627,2.00775472880711,1.83439463507629,1.15421954631168,1.27038737812539,0.409496881615507
"31",0.322681580065807,0.276874079011962,0.10425837680659,0.163575132462142,0.109388097787735,0.109456130423588,0.211689819955299,0.190549675759758,0.140209017136857,0.0703582894558642,0.177495215576275,0.167322246861719,0.23916605075355,0.16295424241391,0.219577471632583,0.2474600091114,0.154006020191284,0.481200572791821,0.281527171496255,0.326755862755955,0.361204941646623,1.15881410825242,1.06321487037719,1.13637980668939,0.263047489753934,0.239942560547443,0.250021127447325,0.281984154765205,0.212129678845474,0.165989715546068,0.184972473493259,0.241556410187604,0.179425148708079,0.111167066075912,0.204014561573643,0.0610481317880658,0.202160766493949,0.399768632873591,0.462713768601098,0.45144959915026,0.48479333889539,0.582787461173855,0.453035330546936,0.404833454124241,0.412566154243169,0.457691736953602,0.453042691237835,0.787166801191401,0.214609446445352,0.1669046915359,0.130512635014437,0.240342881033458,0.243136935287809,0.262116249952552,0.189386721319326,0.186387288038676,0.123170575962069,0.119542374967422,0.235749768379558,0.164785421768753,0.131822662870754,0.0837093546024544,0.225835897247068,0.207862400410202,0.143756371136971,0.27623706643008,0.192514524700136,0.198557376095974,0.199223741302569,0.419202360950252,0.465671555080951,1.1246065274374,1.11573726833995,0.44809740444499,0.399151534147685,0.191363275510401,0.15131835743778,0.196951667105868,0.22685920408493,0.159164368534409,0.0710067426726429,0.129825050492908,1.2514677620476,1.21456993851466,1.67157237401293,0.76379923600354,0.57501211569906,0.591192034655668,0.521384985116661,0.812161833168269,1.16389277460444,1.11733149030685,0.449404151676401,0.17207375647793,0.275964848261621,0.194010675455202,0.167221624212986,0.0963341367767361,0.112580762993451,0.261151917879568,0.102381577743925,0.251023603661225,0.135943269807772,0.139411731633989,0.245433933015728,0.207133720882302,0.157055357932457,0.196941685840483,0.166994161428216,0.188252936115347,0.173352211770398,0.163105605989868,0.187549958293343,0.228227321017706,0.229714633885151,0.406304004423107,0.299971746276451,0.246287529345786,0.234854449574152,0.150748554181594,0.177115517614661,0.198460634871649,0.137914966758683,0.0923216389945884,0.14381239787795,0.0969361398302292,0.22317516089195,0.162164970969338,0.185285148587148,0.114778963134784,0.198906428926888,0.413874545517273,0.422107750061566,0.522930648484449,0.748234252310128,0.60222574467471,0.551854261891164,0.416313209656776,0.522560404348741,1.02370189604001,1.14386100322384,0.24441798575832,0.146771545803715,0.163922636930356,0.139116217297617,0.0914894221853739,0.238240541219472,0.235799222045371,0.214101150311573,0.125267456322158,0.205428632931676,0.155507006040497,0.155112777451081,0.0838240939920306,0.137930631147853,0.138160264442019,0.084635994427837,0.120258105017795,0.170183239028998,0.903929311219733,0.981014500089236,1.30009679304547,0.643321921652869,0.635680408139791,0.204076072095513,0.132821238824795,0.246829498469576,0.167625389163287,0.134451548616271,0.203048275500335,0.194595778774902,0.258156406255681,0.141150768437131,0.143820709804975,0.264299366470298,0.154728066716366,0.130582863280936,0.180029652688843,0.09626958056744,0.156984087870344,0.204922373518631,0.317365420140984,0.688072701984998,1.02591338991145,0.591918704697221,1.23306813757085,1.22339733594442,0.719135757445224,0.319157307089149,0.23701806381281,0.199885495927782,0.161565807330707,0.197574630144748,0.111293628068772,0.124781866977732,0.156218246257717,0.143259787091573,0.169045631734126,0.187301699242996,0.142854007342182,0.223230599989926,0.220964761835131,0.0912243954317237,0.225830959406453,0.155814530111127,0.103257127440161,0.199833595492069,0.567423957448097,0.460288098993856,0.658787557093596,0.447130404998378,0.43794107709855,0.187738221653881,0.176441489389747,0.187728158878584,0.197208011280095,0.265190629984024,0.0860039831743284,0.122676816049955,0.27918941781871,0.223096182650872,0.0948261397781779,0.130921829106637,0.113525840841455,0.224877832875511,0.167870762453465,0.1817477783702,0.151034731541421,0.231682229849321,0.443753391977923,0.362768502759664,0.707813936267846,0.464676977805314,0.935585837450277,0.891364420642473,1.16912958155098,0.845676343204164,0.196784359311256,0.213848929692681,0.187250448083306,0.2213233754856,0.227901090037679,0.156560671514113,0.214850505808445,0.179789132196326,0.110000362966395,0.105010333612921,0.18166065259117,0.131265804983222,0.144762485952344,0.204895413903577,0.156231877052914,0.138211836349775,0.155907690861179,0.355002668872756,0.243468423953902,0.283417710247878,0.146254948622812,0.208120655468071,0.236373224246427,0.213895808549934,0.153275000982332,0.129068728710977,0.226512011437553,0.201312986076216,0.227170247018757,0.170441818122399,0.236070158607431,0.111644582327248,0.186221214916809,0.198589264982174,0.248373719524339,0.113411932717905,0.203363181147767,0.166572108863556,0.239551283640628,0.626625658726735,1.01191408436583,0.365659322609549,0.99499777163836,0.68525473552766,0.913709831279806,0.261993963902818,0.727674855435881,0.201483617020155,0.133634978771655,0.311343475835358,0.146602116354104,0.156364064748381,0.162334267258407,0.144516149371724,0.115366386327863,0.177505950237494,0.23221873038853,0.163516041179164,0.165345147288935,0.159859508735854,0.190601245162113,0.0847052547289075,0.0679793841215084,0.101568001888764,0.170323845106368,0.125051670562352,0.147302754283617,0.656903516329039,0.896857491845753,1.19783802434825,0.765642150816539,0.29503818086141,0.155747049402506,0.151595104656828,0.228417487865973,0.132394281927653,0.166418085153259,0.220967315662747,0.246965812112299,0.268090228224119,0.22149530866398,0.148722603262447,0.592742686703289,1.3695605103965,0.356004428054267,0.42593389763838,0.997908457961455,1.32728349177084,1.15106842445206,1.09618215485801,0.962576008810434,0.46222043680049,0.449785782651779,0.346281791797753,0.202947477810545,0.100313390888167,0.197289901919466,0.142863716951883,0.072313752956358,0.254433842268002,0.144044694502539,0.195322456356088,0.189584166263239,0.210344251194753,0.158504032181221,0.158451089400018,0.242069246284638,0.12624721137208,0.175010344875408,0.179420613369165,0.0597101401977884,0.170398241289209,0.214751292755965,0.188350352872202,0.170464880957961,0.259527343946493,0.424856032868121,0.177665219763116,0.251898261551058,0.154894874771296,0.159572592633637,0.0627397619064532,0.229798729070923,0.11634456417739,0.156034337552011,0.0941558954232607,0.14661231684872,0.152752045831704,0.219226820207971,0.493745700041393,0.668038743339743,1.2345439532006,0.773136660555797,0.125941424806633,0.988784592119705,2.14750609095583,0.996382413602789,1.32760473703068,0.445407311369761,0.412242913561121,0.483015036015729,0.510332095545349,0.402319856659674,0.496634007514587,0.404733792282545,0.18145555963273,0.187991169254538,0.244818026419471,0.258919747758921,0.188185603742334,0.154117362448658,0.167024934826893,0.126737671732621,0.174305090553993,0.210981376968898,0.192721927388962,0.163757427816965,0.151936934619352,0.243801813898493,0.127990631538142,0.0576054815257688,0.138820360351579,0.221145419138279,0.444578260776395,0.61594242900411,0.304143888933069,0.289416530472492,0.208811059646803,0.234001343617728,0.167982636183841,0.189348225541877,0.16692616301401,0.215660279353931,0.327449803818657,0.120694569708413,0.190380265771623,0.138645269290808,0.235452864391448,0.265193898502382,0.324792492575784,0.297316532692837,0.327611279127233,0.230487775374757,0.104465614744779,0.248384671822734,0.166660576348594,0.188223195190298,0.2234332876048,0.189099407103386,0.207535403155464,0.238194894955345,0.0807622423190823,0.176452994478544,0.226058783813175,0.107242384023279,0.142976016003562,0.121666104106889,0.165909814480052,0.100456050771023,0.165861539713891,0.11545658881835,0.161442366722736,0.0920080121244482,0.158052852402486,0.121113260507051,0.0842730798317072,0.377894071448048,0.0532935707861351,0.0806477839917682,0.263579832232177,0.179699235344684,0.268934008515111,0.146632276111821,0.226014256276938,0.138161590672917,0.14022888874796,0.119285856031336,0.269960266973163,0.143695125965009,1.03342261064015,1.00029839543802,0.816023373980663,1.02188262355547,0.837455937336366,0.167302614276851,0.72393167866064,1.18630526921365,1.22453086896808,0.855759369761845,0.360348796902317,0.752312128276761,0.567385985431993,0.914483357321619,0.943637587157243,0.879220162469211,0.387011274361613,0.260177554100399,0.505672144607299,0.420876165651985,0.294354802074565,0.239362862881791,0.197427480974832,0.0650076628879639,0.252312402737538,0.132951612460207,0.198701859059131,0.105133745659726,0.183751782490217,0.109713420703,0.130889332663844,0.196430204336201,0.102361314630987,0.135130008583263,0.14724419058307,0.338915105767335,0.171783195282416,0.162241009017781,0.129726466251176,0.345199475810211,0.263794745961427,0.351086460085895,0.499244362481016,0.308540689325967,0.293491459882224,0.0789137792463257,0.0284948636074147,0.118272879964926,0.255392898230869,0.147504512416435,0.107524000770043,0.111048839663453,0.0163101921650919,0.106119811344228,0.133701973731868,0.0473163665635764,0.156822983589295,0.20219837474516,0.108588130087136,0.113498250224554,0.301132552681332,0.371252922411461,0.713936246429098,0.534810376813621,1.55938643558896,1.49290868304414,0.788373851617732,0.348641643510031,0.205097105055978,0.188124633394448,0.162436209174887,0.136335581218255,0.111809485086133,0.074494646882321,0.173239480659197,0.195065115337278,0.0513870422811279,0.151509689473204,0.109188459194965,0.166847028346495,0.228951387898203,0.1659667379634,0.117374163899419,0.469049883139444,0.254205996201208,0.190390105192407,0.233748526287693,0.151687014999179,0.352486729486178,0.138668439627709,0.025439301204537,0.160793263717731,0.158276072810035,0.178263248691092,0.0893826428426806,0.0940115523507105,0.180355880916159,0.22778845833611,0.0781461594616156,0.0867907309706269,0.0582738181570106,0.169504576067662,0.197879911519116,0.198119029970814,0.0426516980764833,0.149396542942507,0.0960530906844252,0.205137780946261,0.135507232529388,0.138431043275729,0.388616085574047,0.44732936081791,0.379656065650308,0.305686866914077,0.296993745458969,0.159599732026951,0.14405549271997,0.186359416056433,0.070292521538538,0.17009349821065,0.17193132735932,0.18879433047968,0.166137236970518,0.15916597496363,0.124061259302316,0.216148387631585,0.181686867036207,0.246564334084539,0.266194930015498,0.153249883050194,0.192330889530274,0.178013278630424,0.174911440517972,0.0830039253818824,0.241691481330516,0.445841977327216,0.293046941817851,1.40429570413704,1.77643787178588,0.316973449873208,1.73220934288228,1.53470043091608,0.951428664721844,0.121798964644001,0.170435774057494,0.112176160332134,0.115412026271898,0.116024226663863,0.139738974840707,0.119212140885723,0.146781593354564,0.0764961040140094,0.0891276676309946,0.112579607130664,0.164074841151694,0.199743865751065,0.250466698844926,0.370556743059526,0.299065951338544,0.518772078554667,0.416823823534695,0.437506670864464,0.338994976742415,0.721029636277561,0.234816667561819,0.255068656036426,0.0843307507230613,0.0577304921091915,0.16534221526817,0.185139154540309,0.257126819883791,0.129912798287287,0.171171038547046,0.0573668935378828,0.0880934155610013,0.08358744671192,0.124384894797182,0.234655410804579,0.177189826711749,0.174713946469742,0.131886332887224,0.149077479500127,0.16439864186254,0.529054379380115,0.152336588085911,0.0900723896370551,0.146161295692589,0.0894476471356435,0.132212083437075,0.1049165986703,0.078354149825439,0.0621667294773613,0.00208323946056924,0.0792560296851292,0.200393319107919,0.141681607038873,0.140749142815696,0.2534402930506,0.0478643817463518,0.146924176959633,0.0657560219212013,0.161493575436385,0.1536855357467,0.276327744874575,0.200077680029018,0.446214996106308,0.757429898978851,0.81606492062256,1.0702770772094,0.898015308597584,0.272468095433971,0.4147135724264,0.348392784866411,0.21525382538334,0.1676850979535,0.126847280505141
"32",0.389416906492623,0.184319437485385,0.184513460155224,0.268939766009061,0.060098364118629,0.212298171877533,0.168063396811288,0.213663694979695,0.11853176668637,0.0437392954323053,0.135862438592706,0.438979280497444,1.21881302854127,1.49662260564243,1.15160134783636,0.461069664793203,0.15140162067984,0.476204799099702,0.324211430865231,0.0234977855445088,0.15270776518477,0.193205398806682,0.0851173519565631,0.206528167230822,0.223198806142092,0.0872186437210087,0,0.134885010949169,0.112800464218163,0.0852348334073025,0.206331556621138,0.120694632874578,0.409782501169591,0.542436354164693,0.465826513211179,0.66733540686333,1.18934240676567,0.735975480707758,1.63026447097003,1.3914339966495,0.584377573154755,0.781585609302124,1.29126286838004,1.74833719861446,0.641155557128107,0.527327755575731,0.342235569718949,0.184712635002805,0.258089639902274,0.12010865852693,0.280660190331159,0.293510698820948,0.0561894054488886,0.175723393096371,0.251585700617567,0.355265396508084,0.25279472553628,0.0670809786078162,0.149773377521702,0.0537492874965707,0.150265602216613,0.156573774400389,0.282270522046821,1.71314266786167,1.71748869512833,1.75820182911518,0.255213914290744,0.217516506178408,0.17825854063571,0.3708176577776,0.201084439887007,0.206049450899768,0.467812839325488,0.136331760867153,1.31031675641584,0.408781117543838,0.16454486459416,0.398103069423908,0.12147445694999,0.286226782891217,0.266675603849538,0.275099076406737,0.40560034025308,0.560986777991106,1.056960457172,1.4290849985237,0.47208993455169,0.743678466037336,0.556905846710978,0.870995134634649,0.842880995778628,0.544427291640674,0.497133986273017,0.370276391897062,0.316731123617847,0.593692199491587,0.291168827015912,0.194957564084841,0.153449897494925,0.226155780272775,0.154204163844091,0.118716534439155,0.154716268489578,0.167627121407849,0.256119899452345,0.155052589468729,0.11107727189867,0.231020359225259,0.104968275404182,0.187823759422519,0.0625893251152049,0.130286353138334,0.188550756561592,0.0836221972358387,0.177724232374836,0.316033900863802,0.129544417098189,0.242247840351615,0.603368441266928,0.345579880600892,0.180818150365525,0.268021853913476,0.540656357502363,0.0784965752256886,0.294913457283121,0.264109147887521,0.229732004171228,0.19576191065727,0.516467171783757,0.199707111752524,0.283978701839335,0.364813268842051,0.837117188520776,0.716093340717052,0.582444862944522,0.486688404476179,0.430452843302537,0.44931073048466,0.58286307464516,0.541256587333754,0.380523756407709,0.238448945466818,0.259405041972188,0.218348894540889,0.185914523308304,0.182395326116536,0.136136250796902,0.274865453468195,0.0716835806521157,0.227087057939321,0.108124400995142,0.052376058697972,0.156855192679946,0.0509396556411238,0.136365797570849,0.162050202898756,0.232247599489156,0.0830598217924872,0.101174865843178,0.160056343072284,0.779439682140032,0.423157303792765,0.142692024307704,0.192738387385904,0.120429872547342,0.447523408180959,0.335401720313048,0.168185597112608,0.0765808409874886,0.162871585101686,0.240769941548449,0.269811804177446,0.227312513065669,0.166124080690719,0.186573397273502,0.16610074922595,0.199671731514965,0.123590127973792,0.217979525000065,0.445226904794201,0.838966114200317,1.05027038755586,0.760275241331354,1.30251356063574,1.86898174609477,1.7743825126193,1.00110093896335,1.09891055560805,1.34050731041433,0.492273499613563,0.255537282495233,0.145952126578432,0.181660357553592,0.0822154376476488,0.281007827125694,0.235035381309499,0.128995520205394,0.118066051875196,0.207589834237115,0.152942909088294,0.24663481406675,0.118871938064957,0.108973646271849,0.2159865413447,0.0547914379920031,0.15119325989619,1.02288792181568,1.52528194351591,1.75513878672541,0.397007503914279,0.240632035914542,0.122680574530319,0.0997291105283166,0.0869995376953038,0.224421838134973,0.115605104092947,0.17894320984946,0.0841569081963037,0.128807478948877,0.167373285739315,0.0238384482765243,0.165532776024954,0.108987689878236,0.177598457679265,0.35562416971997,0.396455246289023,0.406969068665114,0.663629268743193,0.460642237988609,0.85089472276813,0.945703017278365,0.656702648594929,0.636104315538639,0.594978197065962,0.64858158630588,1.25839621555404,1.09159687296716,0.491256750501477,0.357993948346553,0.38285951331235,0.208623141639265,0.335945071632394,0.319462306136675,0.164343367459589,0.0955008904619139,0.169862491703291,0.0842242508150678,0.211117707880651,0.0230655017840849,0.223674187947522,0.0324633848153241,0,0,0.0121957470930787,0.0718620117788714,0.0539412705588719,0.0010286673717975,0,0.0973250425622702,0.339202804607456,0.230063456805876,0.181531430953165,0.563020705029579,0.314810778445327,0.210312734909314,0.27363732849398,0.110092670537778,0.236338601305159,0.326523260094242,0.263771375528207,0.227706319448489,0.224417586974786,0.479362853095689,0.415208891851685,0.59866321532472,0.88120083587684,0.565345132855054,0.446112725419145,0.829104981201089,0.752670711084473,0.66438172701486,0.693290575184428,0.830996027810742,0.848762912634337,1.18621803582653,0.630968626917108,0.412783147410162,0.326721976982911,0.35565501476934,0.31921370876756,0.242825242732089,0.14410786241007,0.274460427879833,0.103666523561782,0.233315288772164,0.0519905637254097,0.225603369379688,0,0.26091283682391,0.621424067147181,0.373869044086653,0.47916811083952,0.366168820076279,0.365924904109119,0.713474776097758,0.449782463829329,1.20853646430435,0.189376717878168,0.299263754728186,0.578070759609924,0.248326656155004,0.51493395526329,0.214979021212128,0.379072035259411,0.318051573189462,0.251417109024503,0.343857724545622,0.285991484508464,0.373480683846813,0.379837813995822,0.291931001468574,0.373588090131329,0.417459187363649,0.311031528717437,0.436260088737268,1.14975560899331,0.476073470621639,0.461673446481899,0.544193757769551,0.6731362085767,0.689511722653375,0.685928158566474,0.845239253384256,1.18131469055174,0.364172491324811,0.534310888612987,0.237583460620989,0.308191522914014,0.207591500976231,0.181705004426728,0.192517361824293,0.201684228382831,0.209387627453154,0.160575311017478,0.178450845353917,0.160625517353604,0.104694497295871,0.429474416902657,1.21125011839784,1.27655775448247,0.660826577627217,0.830053681276091,0.890762319584689,0.199170302950798,0.384613696168428,0.104499877668328,0.0806630460830808,0.185310440137928,0.0419143220516382,0.312333311905269,0.217176913774317,0.311961087181083,0.268351217431024,0.575744861174187,0.32986278395413,0.269695985005731,0.307941261229283,0.543726202316064,1.1726290785405,0.242259093241321,0.221988749595955,0.652539117995276,0.98819291077693,1.09901196860284,1.15468035457013,0.83334166231297,1.7293980006052,1.27693710781543,1.06974978595803,1.30208176346642,0.798090003888275,0.793514134375807,0.433393150922753,0.526883982774704,0.489998560987899,0.451168266355256,0.39341862793158,0.327290886732602,0.154920899054956,0.0891449757483455,0.132760844256446,0.122293352503402,0.160589507761576,0.122596201203946,0.18151122041547,0.145998933842853,0.0824720446288648,0.0621614783962088,0.270347460372031,0.100117269367909,0.479511233365581,0.174186342887632,0.457506339220041,0.464460650267115,0.708175532004437,0.407665269859517,0.531823956177994,0.304756611155119,0.246738603877234,0.237193042041501,0.151469659866464,0.258069091102459,0.175541844430113,0.457046754634303,0.359531281893025,0.253233063073319,0.148603319329653,0.393753375524155,0.382675981931575,0.943036917186577,0.641570831388757,1.03259270096726,0.403549475661774,0.414614838189928,1.09782439441491,1.60161406181477,0.897479017714566,0.559835438891164,1.28476161781874,0.356321256536649,0.404493948429922,0.544359632785561,0.270757647312968,0.448621522218579,0.259978913976202,0.347628491663974,0.226212270185585,0.0600893789683667,0.221953886001757,0.262808520980648,0.152622814939516,0.115838542650213,0.175373175637319,0.189209521363987,0.371751365656997,0.246792524608356,0.299418429316508,0.265783197903458,0.292963453599322,1.07365911790659,0.902708068257642,0.885983532667209,0.853240496070942,0.324314479174993,0.315873667516681,0.252751389941749,0.271956498238307,0.384236364708166,0.190177411562476,0.243460062381599,0.187731457005058,0.236842513570603,0.202384785407964,0.259435763420539,0.622300643264019,0.573040307023368,0.667425587241285,0.914718225972418,1.17871157151551,0.842142893176115,0.606128133550275,0.663664558910968,0.733131527849353,1.44963683354283,1.29911891467252,1.09086202938414,0.608875251908962,0.72850163184458,0.27369395615137,0.498628523467627,0.564385102635259,0.298717322395679,0.204161269921626,0.17054221230928,0.24749562749997,0.271822342985515,0.241026479323135,0.233701047382957,0.170750168914815,0.176952879374854,0.226946163211412,0.212879631660919,0.187029704671042,0.589733111860408,1.80195954697568,0.897624585022162,0.857121013241826,0.395401086954139,0.292950693452926,0.780384414021055,0.0536753324776784,0.295334892836175,0.16003975819201,0.127940760563027,0.242486443238042,0.101211707329476,0.25418770194916,0.0569439240891396,0.315063342537458,0.0497051738979723,0.303038656473681,0.0960608417710741,0.168297332820451,0.414030565041201,0.398010835561478,0.273810551014183,0.770989158269312,0.810415850270054,0.45805860635379,0.865096325230865,0.527976021321897,0.725875078042349,0.499608193170724,0.98782809813482,0.57638060987526,0.388439660765072,0.363989185996811,0.343989436208995,0.268584735352767,0.169269363622381,0.138287452871491,0.150718634273059,0.105422588831291,0.161943983904661,0.132567671841861,0.216797458949113,0.0897298533944227,0.0681241137708601,0.302302020568196,0.077695257987724,0.18490698850156,0.075091055898698,0.733211301742943,1.67216841783296,1.39412882442545,0.381756598381131,0.365536005866226,0.45202868121986,0.546826428755501,0.297676486132762,0.096938075869798,0.211846187459584,0.196490210152389,0.15073338794386,0.192710408663354,0.14845929325758,0.198256925080569,0.0563662613982329,0.260268928340372,0.210393038822262,0.161310294513481,0.157738552732891,0.0943322482369273,0.196530812869916,0.270408294636613,0.461391357993624,0.958165120700228,0.982904122447346,0.802339684317283,0.794572736471855,1.12485299705419,1.42092034968926,0.675957592730425,0.482491628621239,0.32980273538079,0.358034187830096,0.183994308580091,0.228676919866566,0.126587147165417,0.248856152699712,0.250427522618448,0.118889494351755,0.198078213570488,0.187061826222177,0.181375087203826,0.21799899647432,0.209182479251831,0.14309208288289,0.282156085970773,0.188570592127696,0.200280405874374,0.588777710623661,0.534407584242794,0.548181243879715,0.784700992718873,0.468389381015657,0.252814762413254,0.23314514322155,0.227492832710657,0.209200100921983,0.32225663386732,0.0928933109145778,0.206653807670099,0.127090002737576,0.202203530479207,0.0819994371957634,0.161517658644189,0.200125068990481,0.222723762104147,0.353055253011731,0.473757292859616,0.375546475554966,1.05523282685947,0.725983021285324,0.703837005162027,0.750637878012901,0.601365049393078,0.578183738148116,0.889585500749637,0.903179060330419,0.958183984236733,0.824993652072216,1.43645064735815,0.579691642515952,0.339912622880708,0.377768929050931,0.155411188544074,0.237733161360419,0.0987764268795679,0.0605281257612338,0.28103108104956,0.140890409522608,0.255630806042052,0.182134703089654,0.188733422739362,0.263341859265247,0.00364791384858185,0.217063047446347,0.574065461015,1.54912230447618,0.936937118367655,0.112830205142068,0.258903725956822,0.271595202567554,0.322762472571922,0.555645921650885,0.542439506001407,0.575999921928664,0.926609271572502,0.277175349045259,0.531326247494269,0.253807305068501,0.184329928267003,0.200784152111746,0.302048695821374,0.286034230163912,0.180532741619163,0.268914098990686,0.321725038304731,0.374502853232926,0.8032205761177,0.364024052086993,0.4636568881559,0.229223903366468,0.423463535580508,0.220503409855918,0.207430499306473,0.478536037009001,0.596495182953401,1.30542068862653,1.62328278047129,0.78518875425698
"33",2.26199462710291,0.91467764487312,0.815157478871754,0.84005485377316,0.775633547644696,0.842486699347328,0.735819857879355,0.862202543647363,0.842232002386029,0.76224990960784,0.84712384381847,0.752417254925197,0.51250129994069,0.356603860274066,0.801776358482771,3.55251877362154,3.03541067441624,2.8616293629772,1.21192671683708,0.255733010565654,0.286782638798703,0.773989165300512,0.94310641131795,0.402693561166611,0.74296951584632,1.19632861750804,0.309070983796022,0.134534948786005,0.117999073783215,0.170480774706309,0.517830239005636,0.174740414300673,0.173007909921272,0.274826964628832,0.50891720475606,2.67708303559062,4.3634829242615,3.85001108655136,2.20801708201748,3.43522343309936,2.06080649746323,2.00327391749348,1.6695473872779,1.65906756602703,1.74307150692127,1.57592412998637,2.52879392052905,2.0564300861258,2.38843925908588,1.96936761798405,0.946278227057664,0.957399750602495,0.814263306079402,0.872324408572141,0.747699467990667,0.870685710914749,0.86162927583739,0.825513008822623,0.736632809191149,0.818679468148828,0.778778507162074,0.907164729630642,0.80640404735055,0.693887823658057,2.32917999160031,2.96798017029156,1.96430438147411,1.63746638311458,2.24402253085269,1.30662774357526,0.518100191512699,0.183701674578317,0.280564407059732,0.584883958947155,0.777251044526579,0.839953846661182,0.283287237687687,1.22372472740291,0.905734291141172,1.72792166620428,0.167932548737693,1.22430880370269,1.38256540957962,1.25058320648444,1.92878865284996,1.99878002670493,1.84825969425302,2.17424581435779,2.3067253881718,1.4417217809785,2.15003941832757,2.66675388106778,2.36373374485271,2.54604095243927,1.19972990937641,0.98020082169841,0.71471321195814,0.749970148542732,0.582517302911565,0.714952660023261,0.595093958109857,0.821148679231815,0.781751574206159,0.69596682418244,0.785323121210383,0.79231008489111,0.815439988552267,0.728817463308946,0.771343394737802,0.845912325179271,0.647664037047036,0.477194340165012,2.424339448451,1.5427060514145,1.46716074642347,1.05479784874005,0.993641432455209,0.310386418191752,0.181652812709696,0.272222709906853,0.135529005676315,0.225718000405065,0.713696705030826,1.15885345650748,1.24524442346066,1.02400043472152,0.343911520932501,0.143802553956706,0.170856312249859,0.311035525048515,0.393486907713863,0.504993295060383,0.390928680412743,0.556749587232058,2.55740536104985,2.87664480176007,1.95249086907431,1.1712319575365,1.28304095402575,1.90170826351494,1.46766461678495,1.04447628061529,1.24629267838995,1.21295503426729,1.24864693313422,1.22322191797329,0.918106458938745,0.784139251274766,0.805794317100218,0.847728844562741,0.790492520967987,0.873935480418213,0.841156354589804,0.883222528358435,0.890129817053388,0.842120396605822,0.829686186648339,1.02709790352176,1.4569796112953,2.24885065931934,2.08815261016368,1.63179184547835,1.88452962412873,0.889471530074038,0.109195142958029,0.164405216722356,0.550821049638148,0.735370871305333,0.433712711198231,0.300815905141564,0.293856705274045,0.233184713842422,0.258586223385505,0.285925255748239,1.25130541713641,1.32179672776723,0.982940639476118,1.06944049356673,1.831176107058,1.79930853371064,1.17850482376951,1.94695669637938,1.97291683959402,1.61545810447364,1.66303777696087,1.15551870622549,1.76972052922371,1.39092816786091,1.82091413856847,1.79253056617103,1.50108158488206,1.46284692810016,2.82551350948466,2.54811181480571,0.919538341558348,0.728291669245473,0.789616138304273,0.83552146282224,0.834371137952627,0.781323684508836,0.7232359009016,0.7471458932574,0.892460253983643,0.807197920852396,0.680028799891146,0.807756340289445,1.57329381678773,2.08835387567637,1.54794487761502,1.25835264072647,1.26774301185373,0.634358101201122,0.632245678457862,0.786048552036443,0.290681422893798,0.234848331618291,0.277068241246284,0.33924893439145,0.385469420947625,0.248325012161233,0.200503885647856,0.727598472314075,0.288849885267802,0.177904457290597,0.224335591072492,0.309293719287729,0.449036233659258,2.09480601287964,2.48723108777097,2.00632397962291,2.11301653758084,2.64811151185226,1.91754683873175,2.16487107747984,4.40705108674884,1.23902404022567,2.15059847732268,2.12038065425045,1.56223212407483,1.97766701640679,1.87849371157688,1.69458424655192,1.58500300864252,0.945435075189059,0.804951746634561,0.811949082034516,0.874864747783712,0.808833971391571,0.847260252814703,0.840501138006734,0.741469382519701,1.05577911779336,0.907408661221694,0.974544381725674,1.70834737517748,2.20109999329104,2.40461320408162,1.04209988985397,0.846179379870503,0.324953021386321,0.222062531162391,0,0.125201659927543,0.114648665169397,0.121827101880766,0.190435463728528,0.148350342460104,0.20097496730036,0.975631500757721,0.23312571693354,0.351668389886212,0.418732157998834,0.330827419805188,0.587962212783046,1.49330103059717,1.0833611782593,1.20590748401529,1.63848946113334,1.68422208326886,1.12906724160358,1.04832959424987,1.55433199006705,1.08314951917893,2.48224111032873,2.75938401855642,1.48027038234111,2.15479095815108,1.49994350367766,1.28608119237075,1.2210557731658,0.955139925542549,0.854514100794895,0.816890519505568,0.697843792323659,0.696597244686612,0.69292518757016,0.785107156388607,0.776617675059403,0.747757370645531,0.728694306687829,0.783910250412695,0.811278652442871,1.12809138640263,0.748523335692634,2.16925107932359,1.34758530902949,0.322405306934297,0.590909436151358,0.281800526009525,0.0791132114802706,0.106909883954417,0.117149649193996,0.0796896595545093,0.0615191786847856,0.167685646283236,0.229738803938411,0.237688389908802,0.913826596857317,0.902712298330608,0.397604489451926,0.182442109163073,0.732630466048947,0.828683172753834,0.778993611596048,1.51577225924323,3.34348995485341,2.59189433557863,2.66408733636645,2.44197377355704,2.1318412478195,1.8567067400571,2.27031592608461,2.77008823710709,2.77097461764388,1.41323280022469,1.57756102687342,0.979582696848538,0.952251739398793,0.779722387585173,0.762745413069143,0.844196180382945,0.719921203442383,0.845918033055962,0.871950603833563,0.724804570639473,0.7968656118986,0.782099146315915,0.895881921565171,0.773831169607581,0.69458277593895,0.633131112797872,1.30063851078311,2.59688052425236,1.56965575218345,1.25473022295341,1.06052710355999,0.505227384919807,0.500023830936686,0.155250020102532,0.231101093771278,0.127935274646186,0.165626733291672,0.201668534923043,0.112938202426753,0.15586187840772,0.149896334309625,0.18388593809371,0.273871831486615,0.54227433840593,1.2609862578291,1.03085083475242,1.61280629001189,0.855777333550404,1.7813016710579,1.97803970754196,1.85833846740096,1.57112888538107,2.16871339084027,1.49817404590561,2.11767659845677,2.50134763179242,1.93466790080275,1.21957441129437,2.1560525506807,1.32515346570566,0.764067757210592,0.77959236451924,0.963326861501518,0.811456017656452,0.823849549177355,0.821216413611709,1.10981810559734,1.46846087570634,0.910179056205866,0.875690900386511,0.853740943436084,0.859030350506131,0.760924716269541,0.884620042171162,2.07172755265098,1.38485200392103,1.86995373416458,2.8969983606722,1.85500821772328,0.907694249777575,0.971364055702086,1.11535780420937,0.995601805099237,0.272615014569829,0.251422394715131,0.22507512381647,0.101929834472118,0.169355636030267,0.392070194429082,0.385760941854082,1.18990958270198,0.751079606764188,0.809512636386704,2.21501971153996,2.12651652292753,1.7185000834493,2.22305193347503,1.64768109378649,1.36879056052059,1.5115400966873,2.17335284037947,2.0947319441902,1.83014488098037,1.38606043859351,1.68452774051642,1.62236219357387,0.738340661192684,0.746129731842834,0.792852903798193,0.728598841004473,0.727030955636024,0.772555569678629,0.912053912034617,0.773066379581614,0.655569121122216,0.792812574886747,0.70558215142899,0.752319615286554,0.715368771727742,0.85187542448628,0.822433740065159,0.548475851430745,2.45016091243189,0.803143880020592,0.944062330678301,0.633773700769133,2.5150796308103,1.37363236584129,1.00011749636349,1.46540490104357,1.11329732909163,0.813307164307265,0.32331246927778,0.682816911602657,0.404466288882137,0.367415163470901,0.501034708455787,1.1583308998792,1.38758866248101,0.559087956683705,0.565888596244868,0.564317598123926,0.898654112149699,0.406256442283313,0.577742910038357,0.446226051508931,1.12970463224577,2.27414804174165,1.26615289226271,1.35955149617505,1.75959516680855,2.05521681886828,1.46919901723968,0.767522800207438,0.653917878662989,0.657025558143488,0.659600573578951,0.712189998142336,0.747228160896884,0.739119055679562,0.687578618738846,0.63843248017438,0.723843523773226,0.776009920590384,0.85113843996649,0.769137889678003,0.766944907589653,0.690260140074842,0.784959657455468,0.637188210777491,0.54951962629856,1.01804030120232,0.772453989269383,0.822567406075742,0.670419867994367,0.693547695318837,0.567959971709681,0.212733453749012,0.219069675696664,0.207742231843846,0.129822498025526,0.102225361428656,0.283934818673808,0.251536660115213,0.263027823725785,0.22923264605278,0.1893890426912,0.233561441805862,0.182221635428962,0.267933530193492,0.141301253410537,0.19640618142996,0.130533802552491,0.0925441931551902,0.263332296277507,1.34554336606395,0.672311624058668,0.500172101037543,0.426284184784632,0.669447900151188,1.67524537140621,1.38742824694957,1.48581675009334,0.418400254771881,0.153629692923631,0.13453891643752,0.128798141747474,0.0985076803141148,0.133326897611463,0.167521974073562,0.140963424016745,0.263406185983908,0.0948183339286407,0.214901050191721,0.0793195098654751,0.141488300608149,0.156616109223055,1.02973651917513,1.51615132756759,1.02611116806203,0.836574904033208,1.16514986377262,1.00513747581861,0.170067905163573,0.0938152322054747,0.221572217988788,0.105444261141721,0.16359207599947,0.274641082006373,0.0400134513590825,0.0980388416099016,0.00290624551518155,0.0614936097263896,0.109453889853766,0.161783870353215,0.346390270671572,0.199217747806458,0.204520315929629,0.0965656792762278,0.135771339354519,0.206104348001858,0.100797597871422,0.113698338736895,0.103484468448556,0.244809272612353,0.648380912507379,1.48885356129805,1.65943500280536,1.6768127512747,1.64460113401244,0.727887557676527,1.20935699372547,0.908203692352391,0.122486763801726,0.124513919095047,0.0912861480573176,0.162428921680037,0.197636481338953,0.0405399718496208,0.163066004567874,0.327907738709856,0.230126700131422,0.185497629169634,0.0559471574457217,0.135075904962229,0.0962742945757593,1.43053025094916,1.85686844212009,1.48305512553502,1.12370922718704,0.583291048842359,0.111605777213071,0.237673184742985,0.021239960910624,0.09485294138681,0.113620137993216,0.173467910912345,0.0831730556198251,0.177906616046258,0.0936182351777722,0.108795183322222,0.148890446567381,0.160378276672554,0.305547438463967,0.177967236958771,0.0783326411528575,0.710244110736742,0.703092957780633,0.746343682603459,0.380429502666801,0.345370204044247,1.55449355744449,1.53769210089987,0.951282449141196,1.06457298030671,0.844478342600923,0.724475906044193,1.12728307666159,1.06364020293572,0.243088146252928,0.183948832447408,0.148255231873846,0.210712815882775,0.157132526768607,0.147009769502499,0.24704455671646,0.0481652177718832,0.18266754635803,0.101272845716461,0.059332550900762,0.0760214318413384,0.0809398816786756,0.175181980440962,0.747207698537227,0.747753096013799,0.996460721776983,0.235316627019347,0.279803039538813,0.0873623343091956,0.15354884886255,0.141950752906219,0.252279447403141,0.266112654505045,0.0722367990454606,0.146728935957928,0,0.138825039745034,0.156247998159244,0.150661208600213,0.172956238351541,0.180565715911853,0.185850394395463,0.174350523281785,0.0622119400893346,0.139722002376361,0.09641369535257,0.251612457536993,0.154481045963983,0.171785016034176,0.137423852725705,1.02751402548568,2.75429857296632,2.19841301093337,1.67543033844227,1.33798765587889,1.03120392673083,0.597944403234516
"34",0.123220351683459,0.160469617433586,0.129846762298383,0.180284432383494,0.189536694970788,0.120462450998588,0.146362378304501,0.18627347867952,0.182526253603096,0.133167278435912,0.155605565094307,0.0617718591792035,0.119534818525948,0.100286713702685,0.26918195250564,0.173919022930048,0.367955155937485,0.267186763680437,0.204886602309847,0.876191186097298,2.48216869129487,0.211639733277322,0.0853564158438357,0.0857937052385785,0.184597618267361,0.0962584707517748,0.0294942250242917,0.184683834114974,0.445444649951661,0.225525292716592,0.263108128980109,0.194526590444663,0.235764456372743,0.214063470544986,1.1698734501199,3.17799283643639,3.23227851635873,2.68745141815282,2.49283319236265,1.77256470221259,2.01991734116068,2.03514375199193,1.89381076276177,1.80407256161339,0.182964616579321,0.1513109469953,0.193893312723356,0.192902781858467,0.221073686542844,0.227963296054355,0.247860606822762,0.157765970403952,0.18575617435988,0.173196751827287,0.192053072725824,0.242249454887015,0.122466373651122,0.172808933415874,0.143814690990028,0.13350284767168,0.180205201985702,0.31276734362267,0.460883495288619,0.156107203130997,0.101165768068861,0.161618496949105,0.150713804305638,0.185434415218036,0.120698756472087,0.00804487315892469,0.131691542323636,0.202351889169405,0.12385983301585,0.00556437959944697,0.204526753460414,0.28883884147724,0.21569566562295,0.193432945389898,0.111195468819267,0.436623665247271,0.35411142886297,3.01762289396279,3.67042100671108,3.41417965913155,2.88753103684248,2.71314196565998,2.2214256007325,2.13089427722771,1.87368331261827,2.01742360884669,1.92155049315208,0.644055131934064,0.0160793317373433,0.178480713431535,0.138818759707328,0.0764237479113192,0.202443899615552,0.180386860597142,0.203633504111986,0.111339540497692,0.160559982695776,0.171761605700929,0.179144390303907,0.166784130048895,0.0766594265456634,0.220334535578493,0.169317135282625,0.210753689520889,0.137875011348868,2.92955799992823,3.17166462170966,2.13393132426688,0.0932014807360344,0.159012729801375,0.182825945450103,0.147094099310741,0.0885173314016885,0.0710235803286149,0.124129247967879,0.0922100183802743,0.0936577511906011,0.145761493069994,0.0804967506336386,0.0754741241922643,0.158445744203027,0.168365525508355,0.247750040470778,0.137089033056558,2.26586056957359,3.18166237298017,3.36721704242553,2.67633441237617,2.11788779861507,2.16380883120533,1.49813817108764,2.13722457285035,1.43883242415049,1.84348795996599,1.88769997261122,1.1420322537188,0.183697913391084,0.201891192190751,0.199451321457977,0.221262905472978,0.0932177688291518,0.193811971689893,0.089582773108028,0.0812430297680992,0.153510296432929,0.341830421147607,0.190855260112512,0.0894983776782724,0.202763213236156,0.0019749944955865,0.110270307017408,0.117518139112709,0.132145211142354,0.632080866983754,3.56637600690539,1.0917769129387,0.409450804453772,0.136463977418463,0.239239009065738,0.524113330168042,0.323114074653633,0.174498137336356,0.142823648178583,0.156721316372631,0.71712451795508,0.557776518892065,0.495288562210098,0.481656314739114,1.8057151384494,1.08410882799214,0.501141217243554,0.531434002918269,0.411073192331697,0.295792540242813,0.212485254556312,0.381018909713096,0.478885163374866,0.520372751793103,0.360771289404499,0.474746182330375,0.450293306631794,3.13792117896102,2.68920118223227,1.91289297109653,1.32744595822577,0.131243363925266,0.168585018362299,0.0746807430644488,0.155163170809349,0.139921596865305,0.201907468875977,0.114607243869479,0.138233675992412,0.229265142140747,0.149111323330663,0.181690225231984,0.184657093229404,0.249893208014292,0.145752014126572,0.0609265283095147,0.114959342066882,0.236117455147424,1.04955788626119,0.828826064367025,1.02408467144787,0.708755634673575,1.20638210609753,0.645387925573833,1.10134294044694,1.34360703169856,0.610485387328265,0.674326954426765,1.54017500595748,0.852701781567752,0.234348675882931,0.227556773003228,0.168883452201062,0.32569561282701,1.1685735346257,0.697183243135788,0.212740586603618,0.371758714715138,0.28771340357194,0.500888636244818,0.407573705280524,0.348150961963652,0.493484030276944,0.566252108281274,0.431090431309763,0.442179007707737,0.336660161290687,0.747110678671726,3.10862471295155,3.18483559880699,0.95041804645873,0.173365954676432,0.211804246784482,0.17147555712483,0.148730081244146,0.158444165842754,0.134734165837095,0.0820009143415595,0.177497522810376,0.150923391397862,0.242642687452317,0.1989261470136,0.169302842292376,0.0629591467665527,0.236955869535985,0.149444116001812,0.314139369510479,0.400479428583738,0.548420983159859,0.276961955935283,1.74522236954351,3.23203924363831,0.506068290890355,0.228459409926255,0.218424378565622,0.164801953976791,0.105789166178371,0.0937430983495862,0.125638284670683,0.122381074163015,0.159457733757103,0.197630976345813,0.509476124956541,0.356704119817748,0.237611597518278,0.256409976358891,0.353291065308284,0.545077234384927,0.42616449244473,0.369507808412405,0.386582305100375,0.483152484950041,0.369150875807318,0.373450239604714,0.466388617282927,0.337292952615861,0.343071635524957,2.64912405385244,0.373573505207288,0.11931973467057,0.0885765111837761,0.0536838751800202,0.198931767327985,0.277004119271666,0.126633304223201,0.121084325336815,0.214268617560229,0.188947588363925,0.073171055275578,0.147059563399261,0.177182132588934,0.103710028219773,0.0823400773120134,0.0880844930241162,0.616304433733581,0.603837366360339,3.20588792781057,3.12061815630998,2.6507372814663,1.37913042542885,0.215716960890799,0.438194676536287,0.35242386543037,0.285261716685757,0.239493091076769,0.281640210806109,0.674708460800751,0.338567701526251,0.227727517652083,0.153942180837173,0.160562989510613,0.131578543025714,0.212948130914226,0.426160267745886,0.25370087490942,0.198619014420629,0.251158012107739,0.384780386084463,0.394755168872552,2.74645364179456,3.26272965691977,1.77273736157148,1.83081184873756,1.8194188633893,0.378487901876235,0.0664447447230117,0.143224399831732,0.153618737467713,0.100636642159756,0.145670258932271,0.171767306378364,0.18904902038865,0.139522504918261,0.164189384784925,0.210891105558949,0.135125864261492,0.24493922844919,0.0886155809503781,0.113245812650395,0.131031101873871,0.0930547325665534,0.305841574522566,0.571319279270403,0.729125634221375,0.575146235942033,0.368734791163436,1.0478281214335,0.657719254469652,0.871632620350654,0.432282280617627,0.416698520808913,0.482040874211311,0.263210165498244,0.405430210871363,0.374933325363983,0.379282124666052,0.504541657098543,0.350984099170093,0.270873712972737,0.646093534793868,0.503956481726217,0.41285677627703,0.565800654072409,0.502206883352393,0.492112949187376,0.721616680922166,0.668326488400683,3.31612244840599,2.77087774128194,1.64906227484224,1.61284990950993,1.27281206147758,0.300470124450551,0.247180412109904,0.323243845984762,0.284648589374465,0.286634799892381,0.345737959654714,0.379052563108367,0.268795219486507,0.276918281243126,0.258095031418707,0.258144551421245,0.35002046950486,0.3568497934794,0.440778375754186,0.313657996534195,0.363588301588886,0.365242475565593,0.511781237813064,0.635834591570695,0.380479938642545,0.466664975471919,0.414810339022201,0.296397334056644,0.255188621192592,0.305001040465231,0.303113111444247,3.41758884372843,2.85334665917663,0.429042358137454,0.662293357233625,0.417611525114891,0.356631702458561,0.37704480245142,0.471687536171106,0.473635322467092,0.509789919015834,0.432442683087809,0.546881136833531,0.435190046630634,0.583633894520713,0.605059463485154,0.737832303915009,2.6465485624922,1.53921834354287,1.35436314403877,1.1774245618169,0.895848267610736,1.37647283129709,1.15698434611019,0.635714826230381,0.36831216232568,0.312264949343259,0.355966197883315,0.385419862662277,0.366087984913068,0.383244048014115,0.40756442258786,0.398889063071812,0.362685287680294,0.220798037435702,0.309913035320995,0.337713168763756,0.335790182327009,0.326263304070008,0.337416637991333,0.345119901689059,0.249915291699731,0.276628620954636,0.219852421383395,0.546330537306719,0.52043720638077,0.585299051482564,0.472701766374622,0.421134735357356,0.564567159821949,0.544901535895957,0.595973863643913,0.654860857168989,0.506104775147456,0.38336389764623,0.312412730467459,0.274002248175939,0.35240827928885,0.359559365449315,0.345933038047174,0.66188857059542,0.595068646844298,0.57594355895812,0.547267380640414,0.508442917390767,0.53159069080751,0.58635024701743,0.644972700872763,0.610370692507081,0.582586801704185,0.473043527265619,0.540411141608407,0.466995405077531,0.352085914596831,0.277807808281744,0.323472773514563,0.315134405580396,0.278534819427894,0.263032481043748,0.325340249816988,0.41258646651265,0.256176208385793,0.271464122047725,0.343148741209213,0.280576889998436,0.398018266020243,0.426475820619052,0.43270040524603,0.681004978164287,0.721440658103378,1.01508904654344,0.620929629694037,0.494726416777503,0.597481442336849,0.468515472413781,0.411558005771487,0.492079783765771,0.375024753713,0.247861024225345,0.260167365432689,0.393137713908044,0.681109683829046,1.32533396135849,1.00222174105274,0.829336698703566,0.450035512297173,0.475172221703996,0.842628386451196,1.55525815886671,1.16114269326526,1.82299906045789,1.02056509851803,0.583264791196827,0.639483580188706,0.618182728588922,0.531467262451282,2.84123302487882,2.14721431551529,1.76664116467877,1.47270060842749,0.74435069808633,0.360371419669328,0.318933992297996,0.406659765784432,0.463434914264838,0.377312840464378,0.279429753713011,0.350642404472358,0.280795118134448,0.343896668102854,0.33965309327572,0.382631987866807,0.294215243559554,0.477507200654361,0.411440890452942,0.314026033563764,0.547209051725852,0.654149046098907,0.645793620026455,1.06057949897622,0.428746620420565,0.336615894268058,0.404468895147334,0.421147896828179,0.398627459179475,0.386551587047985,0.317366310740696,0.301568409809498,0.251762647739743,0.35506665565276,0.239222188553452,0.291932653587257,0.353875562263205,0.349132888362802,0.319260434005033,0.335141778120206,0.377327893003788,0.432327666267722,0.352824640527915,0.418978034523873,0.60047871025513,0.676342723534285,0.545046671703685,0.531606614420902,0.542880574035253,0.517785096806389,0.495676403470271,0.59942968369575,2.26137876096128,1.16946735208805,0.313571660588747,0.207282293509827,0.402148465135987,0.333948562802968,0.324913761945248,0.393207079477763,0.361974810105123,0.194363729063191,0.344011656435879,0.239989461883046,0.287130135250677,0.182049243820795,0.295907943506789,0.386392007166042,0.344891069024755,0.363059524053379,0.38175409280928,0.433196318827799,0.595334565052883,0.549935411584046,0.411763815684635,0.217010462238764,0.316698625701631,0.858192558595925,0.539953861841224,0.225271745138236,0.469246933322769,0.435515918059273,0.362540428534655,0.273584642048789,0.221019643709222,0.24127117210648,0.217285404162576,0.298927052838795,0.306925503783728,0.47039417640319,0.368346406406202,0.704013868589482,0.421812406026131,0.692989478322891,0.943817783258056,0.931876703787783,0.812065101413848,0.42433402350486,0.432097306916995,0.46764230543537,0.125962353759913,0.29198421886888,0.188273059332244,0.163756216026669,0.156203069481318,0.134022773442975,0.124365007687048,0.101749447933113,0.174462794482223,0.212156510737321,0.0751382743859986,0.218659456673259,0.245541237297535,0.175268154236781,0.354504212264868,0.368268439191742,0.472766929765946,0.485295004537765,0.328082184403157,0.154351682281225,0.773436672016917,0.562899460985835,0.151850687590019,0.213712675094246,0.176688222698534,0.208999507620437,0.177568339479635,0.325297481437969,0.62196687270174,0.349767243587893,0.293422738849749,0.369026770130181,0.332151967887919,0.382149713290034,0.382634521553863,0.379640742000799,0.222917132434605,0.153693074790609,0.687539032768222,0.700588778227858,0.422000564974038,0.520129525533841,0.393013494592373,0.333392266498392,0.436182093796283,0.381045114539486,0.4825696460984,0.318930165419795,0.208379138162763,0.0619733275626965,0.212029384174611
"35",0.279630115263719,0.0954658113540065,0.1312782397503,0.0550252334794663,0.0571754855642951,0.033133000238963,0.152008729219389,0.0664267734249509,0.0799283715601056,0.0737960428724184,0.0726184265926807,0.0784085873412002,0.0889710640143447,0,0.0645430936267707,0.0139767707569845,0.0570523065819184,0.0438561633306743,0.0812104660042471,0.132186878803896,0.0289471072287214,0.143449337555465,0.0843842656984664,0.0573016628615401,0.0977956622590004,0.0316865920484552,0.186874168448259,0.0833371299975992,0.0451581048063109,0.123611673810065,0.0605437810020513,0.0723651796999389,0.320396948357659,0.881994023778986,0.618840726316454,0.582644197280203,0.500338510096166,0.729618922477199,0.361315356575445,0.658769698831368,0.62366702350131,0.715451744533406,0.676742778156329,0.899213542273325,1.14050976411627,0.678232137085082,0.415550210206704,0.530084418585232,0.449489996992045,0.455401892175926,0.320197711631762,0.449027944708581,0.474947540328159,0.415429068015211,0.650649235943595,0.371372093194153,0.381995839843315,0.228980842744891,0.538361888920064,0.408521690608336,0.0100750274992821,0,0.119992040855347,0.0603557010135546,0.0978345062836439,0.0165824313524782,0.0784277772013512,0.10320846709391,0.0734059361470172,0.00609456822536637,0.10336253838042,0.0378191397364714,0.189077957868222,0.440448491952745,0.343758371371159,0.110228192200273,0,0.0773952511761729,0,0.0286179140205301,0.0378494715532216,0.0212980172971865,0.302433709630162,0.747471090806442,0.705425900181506,0.537149853552449,0.572733869350097,0.665305647421616,0.544927361840408,0.627030957198861,0.495334176332783,0.102853009424397,0.124096736161716,0.0588430392732301,0.0819192407105774,0.0904492130039441,0.0498148105446397,0.0468253152417653,0.0741798065124851,0,0.0418570625450341,0.0294583460962799,0.0547241797965027,0.0459818758556236,0.0441877306192301,0.0400077125803877,0.15716502263583,0.11224397644808,0.0691075526497963,0,0.0399180132982995,0.0132453598305087,0.0931888517020105,0.0165796914102125,0.0584011381407069,0.132830770645884,0.0402915716898581,0.0973138342220819,0,0.0297929754023809,0.267190376738161,0.175844426346279,0.12182404837401,0.0925756536095534,0.172202807657547,0.290500312513351,0.162915570056107,0.29916972065698,0.539370935584355,0.431841338450545,0.35913359125402,0.378276651208246,0.435674708970993,0.310458276994214,0.357169109135898,0.387996907391187,0.237725703327151,0.239223856189077,0.242600417230432,0.413668707038354,0.310006346885256,0.349985348079252,0.474878728220124,0.0897315999268335,0.191676135440933,0.0378153639689823,0.0369009350065161,0.0708153473269983,0.117828295228455,0.0716979735939863,0.0218923935771622,0.091852704449055,0.0654662309721724,0.0951637795500152,0.0363519141294123,0.000927821036038293,0.0747167678427162,0.0280651992446736,0.0793899210062886,0.0606050421201786,0.101735613814472,0.0179621257479237,0.0813491419704649,0.0285500167506819,0.099093690479591,0.0532711455164362,0,0.0901931811576807,0.0783990492190005,0.103998014763788,0.0285526800505495,0.130005035756888,0.0108149789887273,0.141232746710975,0.0838083776470889,0,0.124134379226025,0.135254871073669,0.0114895131669075,0.033228301224567,0.0861050903473594,0.80474677968311,0.837364980217856,0.615879837253647,0.637507464366666,0.62865389767412,0.357425116924377,0.482425701500622,0.375360104110089,0.514887029999084,0.287254505754166,0.0572258362532425,0.035838683868704,0.0813309955345497,0,0.0320031016955408,0.0410414127399847,0.17039288770704,0.0520242090332156,0.0514354863048286,0.0487263745119344,0,0.119138483488206,0.0685270659990338,0.125649429611797,0,0.0833546917611655,0,0.0988932197088909,0.16658716784413,0.165506137243603,0.131833839864113,0.017921525849554,0.0631402537236592,0.0466363082499407,0.0827948437355403,0.107494838934993,0.0863163838020044,0.0557536708524366,0.0937173423533254,0.165490799937187,0.100368428028026,0.0970369706691634,0,0.0878786069027016,0.154954445592561,0.104902653033124,0.0960916292674134,0.153552416617217,0.975630714577604,0.706703403219202,0.574593549097289,0.568035701593916,0.853297857999849,0.752155837865562,0.510059074533781,0.539665969921719,0.279144537084985,0.030385915567498,0.0995911014055015,0.0811314232807762,0.082152930839943,0.16787866586813,0.097863893330809,0,0.0633862930841329,0.065326963193138,0.00175034745725525,0,0.126004965841996,0,0.105819456331639,0.153133531127453,0,0.0801863889350233,0.0432824931191456,0.0796090175328691,0.128108994829436,0.0467360678967798,0.0300115723840337,0.0515888007587307,0.866758171262255,0.422073113997239,0.50070171883808,0.333154041839528,0.176144944197992,0.0598884445970904,0.105903473132182,0.642880918931956,0.255411400420531,0.115998491605796,0,0,0.533595833501523,0.227567812984728,0.221580647490472,0.352007414156617,0.288305765984633,0.218529340766231,0.268556184940169,0.2730033159621,0.236284763220003,0.269024905370566,0.373005498101613,0.270098504156497,0.239286483702315,0.105410453119674,0.0367594764341816,0.0572901110810228,0.0390141811714601,0,0.145598307986011,0.0677481449284927,0,0,0.0798745584378137,0.0104361508698453,0.089345770819805,0.0922198265923215,0.0111517160147933,0.0359820091907211,0,0.0350209684872996,0.0264265985766567,0.0143778587074287,0.165813307400001,0.0690177815738765,0.110875237837377,0.0771460185925917,0.598855081308431,0.176526190468175,0.0822553257998896,0.0429372151728037,0.0728551068930777,0.128933887732876,0.0544612278776156,0.0882490419780028,0.0711369088179753,0.126950925820477,0.106659872344598,0.175851084596511,0.0491819217071298,0.150859064814243,0.0867941098712923,0.127468176296032,0.050660195168042,0.65458117908105,0.88004330265606,0.943628408342771,0.392209911542707,0.571900900211642,0.494479622415718,0.605269451792867,0.131519155304947,0.0278555218417564,0.0156573155478025,0.0174956682047821,0.0864799119015641,0.0193347670794087,0.0222518492865456,0.0407956259482873,0.130071727676685,0.0973608583404515,0.105610123050311,0.0614785875019064,0.159102319419068,0.0432883524374185,0.0369015788445203,0.057990529884236,0.0273213939903851,0.0624411951892697,0.0604590544354897,0.097938609842834,0.153357152611383,0.365136823887649,0.199191838251757,0.0629439480596615,0.100463428556103,0,0.161779948848549,0.0847308384797349,0.0987098750094267,0.119836352907236,0.0661018080799528,0.193708282073098,0.0416668389685252,0.0918388854946166,0.0355847774107454,0.0337322546959531,0.110543023583551,0.194790795833695,0.0883932042819146,0.683876638133624,0.604758152452792,0.310438489645965,1.37053707726279,0.853455942422099,0.777097718680807,0.595100840181323,0.594432460049686,0.568216555615076,0.357787063652515,0,0.0484887753666022,0.117553882747169,0.068110508195146,0.0654715088811794,0.0545350230603713,0.145811578381902,0.208020494061828,0.0497147550452396,0.014320798554118,0.0787820828137624,0,0.0801299066803688,0,0.117866994879148,0.0696633541204019,0.18514412177221,0.128393413905462,0.0349575563345831,0.107796913276322,0.0717471745974985,0.422077489837606,0.57073386625752,0.38425382835086,0.436303004751549,0.604411503302038,0.367220693240959,0.258602891741711,0.314084068570611,0.281014018473638,0.180290343717266,0.269466247512181,0.223454353564961,0.248904385250511,0.163142825524953,0.24377296214015,0.337724118959527,0.608329298333695,0.483229320109816,0.470875813725702,0.252482632886333,0.352624194589991,0.265187799443675,0.30216710414033,0.207631476687965,0.344717830828447,0.319594275918842,1.08785349065082,0.635248818351566,0.246861727404616,0.397136673152707,0.3336329987015,0.241252371312278,0.245066009715554,0.0878215662387656,0.122375562931069,0.123975915216927,0.120422524949969,0.0781107025605181,0.0514372841684689,0.0434916078244279,0.0660322399417876,0.0535814489940322,0.1270273222109,0.102484201646191,0.110050663852788,0.0232798658327419,0.0984739694284136,0.0972881976810308,0.028919445350815,0.04729555803398,0.130253134458818,0.103619603312773,0.441118616633274,0.381933653207336,0.359675544175147,0.948375830383125,0.220698465032041,0.268773240453372,0.468372724285767,0.274166818081466,0.215366350297623,0.227880138084339,0.211703325696577,0.301078657114099,0.306890616635434,0.269169328502686,0.639956936008832,0.450227910924666,0.2965896443291,0.49878684126126,1.10974913408884,0.497072289947301,0.341495602643094,0.387612643682713,0.239397585284988,0.409837269832429,0.132283423744951,0.118213454310919,0.0698556294344274,0.0294745290346737,0.0463980404802586,0.0234278572104246,0.113929984403867,0.0828292905766817,0.0642050702344476,0.0402946642557876,0.125583337386802,0.330757674679666,0.213192365992346,0.218254408866411,0.11909446460695,0.193173568226137,0.187437409125511,0.0342674504999991,0.138743714497467,0.0351968159489344,0.103557899279814,0.0584885616490751,0.12263033424588,0.298660043646743,0.264789755149299,0.233293432884267,0.19673868199054,0.264842314546844,0.271166739666959,0.284039472283105,0.321424416096614,0.212049810846437,0.538665077288242,0.494447651229477,0.506942001608213,0.385950519636245,0.292066884227521,0.349245310955755,0.478573830509819,0.492021834819434,0.485079966566228,0.286166172451919,0.27880588993511,0.316333705415362,0.329982169564298,0.452722691931789,0.357374898851878,0.26181325741203,0.240362529385171,0.254603257188035,0.326028420286416,0.275539385015682,0.11494276589214,0.115557538274023,0.0675858969722663,0.144295085800753,0.0371275070734509,0.070243463646082,0.0948186865894313,0.0680056746681013,0.108088781047583,0.122527654806887,0.0650544541256031,0.116246065678942,0,0,0.178511410983331,0.498957462443476,0.169833499098435,0.392963624066289,0.177953443800605,0.117740989576641,0.266987759898146,0.289414918050251,0.185478527103264,0.360843587301671,0.2096811630644,0.15126535679876,0.245069314738182,0.294667989444175,0.356380182215511,0.0948367177128849,0.21209347327103,0.12548257761543,0.452617077217538,0.286691264923551,0.409107814422631,0.244976123880478,0.387326715664334,0.297873419572793,0.811786249047137,0.340024246369897,0.282789208998621,0.184336901823491,0.197456636643328,0.214960494190454,0.222393955033794,0.207663471816713,0.150134853254495,0,0.0329644630778899,0.0647780274615232,0.131476694599822,0.034055688063055,0.00900921210445751,0,0.145203056166551,0.092953522624981,0.0993220942650198,0.0206300289088603,0.0571754211398165,0,0.0112959633067845,0.0333569631650927,0.0804906098811505,0.0727321684032314,0.0218720386259496,0.320670238435658,0.22409990284095,0.312750349633767,0.228273740614722,0.177330531006625,0.283055049485144,0.219920269786863,0.24577193945854,0.134503104325226,0.276858456256753,0.299462166256575,0.220176398217573,0.282068596173305,0.645037045383258,0.376584704884399,0.497249033760326,0.373938520495592,0.451466293189372,0.361171576027535,0.523084996018381,0.64012004404598,0.489982445149787,0.641439371626407,0.523179058838379,0.856926229296089,0.709281680863893,0.526621500615753,0.172751198073337,0.0284794824012499,0.209789490993911,0.0747495460306673,0.049100092691114,0.0714495722923605,0.175607116929269,0.0916926359206215,0.14197721132321,0.0534498054478469,0.0256057996359551,0,0.0602631926052386,0.027278080139157,0.131723223253558,0.0865335160504739,0.188911399601878,0.139107547920553,0,0.0227958739122494,0.151994369395163,0.0478891028379552,0,0.120447998462676,0.0818107518269112,0.135071043949515,0.144920018999102,0.0527130353120338,0.0758229684352117,0.0911577794404621,0.105526452504651,0.0401372725232633,0.0273681712626859,0.0936996257241021,0.0873811186929848,0,0.0897433433316436,0.092447887911414,0.0380590526828229,0.0613606773623203,0.114463905209817,0.0467457902991315,0.103820439436926,0.0965265745923916,0.098940856369947,0,0.0978994496640111,0.00926494271273892,0.0938697411730689
"36",0.853481855670255,0.703570785746057,0.644923288391604,0.750026101815739,0.627930870786169,0.700621136417129,0.721331800768799,0.835214983574255,0.771933591136046,0.773211397555378,0.704388976687251,0.759275497885679,0.824275616922824,0.647955155791388,0.744674169209787,0.55780931029062,0.294413387438427,0.154922432119557,0.0519717948920024,0.0419049529296571,0.0572348762890088,0.0694258797552416,0,0.0321516718519844,0.0394583036024695,0.0182089710674627,0.0985362768434911,0.0341320783289377,0.0754325191705542,0.125698603379876,0,0.033436976776966,0.159582448147302,0.105809287791253,0.11866707400199,0.97332823619032,0.786394349860552,0.529790634541926,0.213755872211142,0.458371069890405,0.38005652999714,0.20768487549573,0.154090988181458,0.0622281105132277,0.0597911620044467,0.631174506492345,0.940172393845102,1.12103173308304,0.908415634838977,1.15744025203172,1.28109631696289,0.970699579888749,1.10621109838091,1.20042012833386,1.04858109788945,0.922536782577999,1.13738541015489,1.18690096597295,1.01913101768266,1.13467235103711,1.05456928250187,1.06249477681848,1.02626799984315,1.02209224457919,0.610898678406891,0.433637516371652,0.177524012026639,0.0776745941201116,0.0603313464941397,0.21386278674245,0.0627071223555359,0.0176667212262614,0.107722383662797,0.182203356741395,0.0982456267578714,0.184034110177347,0.381476503521703,0.201736786193043,0.140863466521636,0,0.160980609932769,0.0312225731278607,0.167407789718317,0.253784188535965,0.107291037195913,0.491401253531014,0.321864480816675,0.48609812294377,0.627027690344725,0.462737786973024,0.403629069159033,0.906391244002853,1.14277063388048,1.4413921720832,1.47548906773844,1.5088372122409,1.06468678331086,1.14167668796449,1.12449621162557,1.05467049233377,0.879998445300132,0.990725420602384,1.01242147496811,0.874885930088513,0.893474076333741,0.913265700455882,0.862753493034308,0.841246598043085,0.928927411344421,0.872182167269305,0.819506051431944,0.78927228098123,0.627507890531488,0.627105859635402,0.426806585734082,0.812782083680492,0.56845201694846,0.92634923326216,0.905214282104433,0.438237455067471,0.26670262793111,0.210760087612665,0.193795892551369,0.264272503996827,0.192449157499856,0.162883470509696,0.347064336285329,0.192334076749836,0.214433477534162,0.206061498787944,0.199313533383932,0.439417429968732,0.563655190461871,0.751844889858345,0.798905957652327,0.439431075442177,0.461443067718429,0.472196947485597,0.469232384933272,0.560006265098872,0.948157950988097,1.06007366858428,0.947414726373729,0.982449198660871,0.84763537662461,0.60075348233569,0.704462960789284,0.797030398380665,0.637382340168145,0.659449587622105,0.795061845853114,0.67585641359411,0.66020618937143,0.722350319688831,0.6999733357311,0.651603648912247,0.803615655863422,0.701763437905356,0.701107475219975,0.541814228437933,0.483243427902772,0.136872469724461,0.122525106899731,0.0131181327619493,0.0915781676033506,0.0089128741831304,0.069410389430766,0.0758018392026964,0.158549314073408,0.038533146055608,0.105558265087504,0.102030558146143,0.0686746664047759,0.067750738799553,0.0518783435779981,0,0.0774162560471309,0.0549045738012238,0.102907925446864,0.0979685446930307,0.241055160041862,0.814276430873365,0.375201328491344,0.854490477479128,0.201323001757168,0.524770652900627,0.474731692829296,0.555970838236985,0.928912952980989,1.11998222899388,0.989748140487099,1.01101219656591,1.05584924652777,0.848909921764674,0.760946266659834,0.790942284884922,0.758359450379017,0.739980983411186,0.807819080306135,0.593023478874797,0.820675665529194,0.718430267383629,0.81959271937845,0.733460996247798,0.746622828345956,0.748958892919574,0.684798591843753,0.618066862878839,0.210896367881629,0.107791863810898,0.1412706461713,0.111796457971834,0.0453938738785348,0.0451995811984807,0.0437866656560495,0.068991446105667,0.131661974345297,0.0512768410041943,0.100725098937716,0.128846134806439,0.0748218451842331,0.046769583309094,0.117990815970415,0.0383467504254323,0.0587415693999464,0.0585993522872587,0.0365820132806957,0.0343611172162059,0.19290646386673,0.360403357004204,0.840288904089903,0.466107950998992,0.312553401080097,0.669391692976641,0.321303032735348,0.705143644686642,0.674579003721146,0.91359019106229,1.04180883422468,1.01266326506484,0.780924067496066,0.684611287787988,0.78630949431186,0.76214202582948,0.657119252752009,0.749712176650299,0.875377491749427,0.820501917341398,0.746685639415558,0.790494707016047,0.645154509491518,0.765320903872707,0.812950147764006,0.854555592398937,0.671317461282356,0.542839871475789,0.519444179852358,0.46226836947912,0.493872054028623,0.578913745830006,0.290792317957232,0,0.111414067548653,0.1948588477063,0.226097245476476,0.13118713286129,0.082720219361012,0.0348413280108138,0.0323068709447958,0.15896711190686,0.126022750218736,0.294199990240243,0.249279522854122,0.11578240683494,0.206729750376559,0.442943865513179,0.0804892419953546,0.764356096668624,0.567063787365842,0.316970487197549,0.479268154499094,0.537991909275437,0.465578709577308,0.699897017835416,0.760549787191187,0.970204211377141,0.937414171916106,1.0189743851298,0.96164974379723,0.735867347558087,0.643098603767207,0.672483648108345,0.774086780911058,0.773802103626849,0.755729468824911,0.760833692741187,0.752398372421786,0.779480575148438,0.793433416415593,0.674477753211887,0.698991771903246,0.553849391464024,0.672355343303322,0.626094942365014,0.169254095429295,0.0542823630590318,0.0509320304277942,0.0234361880252837,0.0522491706271159,0.138024138890678,0.0515060759398582,0.0405584828797018,0.0822451706671132,0.0605963688645795,0.0962376984395253,0.164603739046567,0.050408953143525,0.10435911365371,0.0364816909944544,0.0287899009846134,0.0117804857652119,0,0.0717396185283325,0.122924701827194,0.458308938557333,0.693064628353825,0.832165667730309,0.398979721565094,0.27055020384262,0.457728261262193,0.666614111190803,0.633202063437377,1.19314578150815,1.26434187372571,1.22488668015048,1.12209342964463,0.858292887748655,0.798619953988324,0.892803636956428,0.918538158058064,0.749131035128344,0.667248648367584,0.868602271894915,0.832803802112565,1.00379171900585,0.801881370785352,0.823842786762288,0.794397387190223,0.807111482032497,0.731813801859079,0.929394381201181,0.369069831224035,0.193415117251074,0.0746750895880483,0.0614642553297995,0.0695952316066593,0.129115184993526,0.203695059369981,0.0396474596907789,0,0.168887984137831,0.101861273987741,0.166481230258793,0.0308022267824596,0,0.0911845786526037,0.0581056968367527,0.0622008795399498,0.0579132489520606,0.159081132346663,0,0.0561632125627001,0.0289025443652391,0.142984611079578,0.594459680922376,0.34858059249973,0.405122277978526,0.173089398877571,0.106642056201124,0.152709072313843,0.473129219836704,0.495734178285312,0.39010715574109,0.339850184123767,0.56856987709148,0.952768815158313,0.856742782924684,0.727863611512523,0.702494157920224,0.825002504123226,0.641453414311072,0.823163927518855,0.724212557638904,0.78526904484054,0.807865832041324,0.76708223108879,0.787217329928777,0.735748270626313,0.758632409174245,0.848462183774768,0.655299940490686,0.476196450141406,0.506241188801618,0.029378408754634,0.0672354583657741,0.0653219020415456,0.0405876101323412,0.095601723341625,0.152160783656538,0.146701854873797,0.253459449885666,0.233200621008559,0.276631590887557,0.0957905703151504,0.0778728525687253,0.0380179379956464,0.00344604095013189,0.0812909103289917,0.183759306805074,0.370648465122759,0.088826490557492,0.261048780984885,0.334949613506291,0.352563550415387,0.625499183069701,0.61614056463822,0.387940653341157,0.534104574456037,0.877917312177482,1.16072890582629,1.49393833499391,1.50511094255193,1.34278234167175,1.28034990921054,0.826748332128141,0.856286281183706,0.989337279039473,0.863749843612608,0.810309580819812,0.92525694442087,0.868718358418726,0.842987392796533,1.00365973655525,0.92626638384724,0.954609921518662,0.823131696341998,0.796015624671129,0.707738111375201,0.760788877255165,0.692055891687253,0.658360977246105,0.388594890677875,0.129162442898231,0.882462915169754,0.372279203484398,0.261266381413023,0.478797045783394,0.854254445336037,0.962044219350474,1.48687048841671,0.502796918262606,0.374587700931515,0.351457511102864,0.20251781232722,0.172428924625826,0.180477016178456,0.349952071555045,0.409853224942036,0.353927332336257,0.37028085272749,1.08500274985714,0.612417427380082,0.993181069816727,0.763972964625798,0.793958278315601,0.376915029431237,0.985392323439081,0.956755518044108,0.833848068141743,0.901024310056016,0.549030997540133,0.659766085757349,0.701848673293514,0.633881226672287,0.602121611972681,0.632829570861992,0.673665209794493,0.63035655749203,0.62044705046136,0.525448109295195,0.682995684970033,0.466862588127008,0.364440027893043,0.226658840776531,0.256569370118305,0.0810325134468456,0.156832582610418,0,0.0607195120793109,0.105984526099464,0,0.0962023118168364,0.0837025010755908,0.142784690081716,0.145243023126491,0.0567592274111106,0.138942251797433,0.143694773563155,0.102535141124536,0.027120530739375,0.123232471650601,0.0532753151273854,0.0633558987554608,0,0,0.112340113580369,0.185240125515086,0.0855537528367249,0.935548025840856,0.294880931912956,0.406445235515488,0.327699898787583,0.23205049433545,0.424777966266572,0.390112837075057,0.475818399355431,0.341780824671146,0.338938128518364,0.293804505728349,0.287480420781122,0.381333446332424,0.360744175023222,0.32660939461737,0.260046255698521,0.328156903020206,0.237236449989304,0.235704407774388,0.37150594416574,0.302692545217485,0.289929353796071,0.36683939000951,0.223382968753493,0.299774611016183,0.305495518782248,0.141343075569289,0.043929106531545,0.00932661925721312,0.0510093624065485,0.0745736153214887,0.0168747828354898,0.206848244361365,0.129803130111034,0.0334636599901017,0.0600307543819298,0.107393722320044,0.00877096060677231,0.089896922510459,0.104104902089021,0.0696758271074119,0,0.154218229953104,0.039427495519012,0.191928713652824,0.154435326886224,0.000693753001228765,0.359125916624536,0.984005073308877,0.401442018098506,0.133553050396009,0.293470398292102,0.199823586366605,0.231059598245823,0.412555550307467,0.331490867269582,0.381817585690401,0.381596617498804,0.376383235013346,0.45738956142741,0.253261371312157,0.407664273178934,0.279031909341412,0.313087808758403,0.289588719802934,0.275214463239675,0.163875527232574,0.10783385741168,0.0263945038174932,0.114185065757856,0.15398576335587,0.065285606751029,0.152949629532909,0.455851517409912,0.090777214343435,0.0919177528397732,0.0357573313955256,0.173506855569081,0.0808387359521759,0.0590496845387372,0.0706015602398505,0.0400117616482761,0.125476871955548,0.0580030809217372,0.227622883765291,0.0380680598993333,0.0868819860173602,0.106357999470489,0.0444160838487131,0.0768796369984011,0.00943495097203871,0,0.0482232652965239,0.116626736198779,0,0.088948797213774,0.600133413935326,0.583696257373036,0.171973993317885,0.181834329326151,0.358130547147579,0.517851206775526,0.42469222664602,0.467535832005604,0.444649810469612,0.268680758122964,0.363967203210788,0.403014817160424,0.296927439818359,0.293652097511654,0.444128121630931,0.326490793002606,0.1681220433639,0.417350506980765,0.250769048778122,0.281892175273964,0.353187280549991,0.118471903852719,0.0844624160412693,0.13639572726989,0.117772460328792,0.270807374838132,0.370069262477056,0.104998711968025,0.198395289713307,0.192622521196549,0.00733665311235539,0.0356248515150431,0.0602647941494251,0.0873994703835152,0.09856601776084,0.0982406393956873,0.00980679848205449,0.165966775928735,0.0430079683422081,0.0621588864110347,0.156153399781911,0.0726861173062701,0.0268120915333157,0.150044478559533,0.163184507366645,0.00273365003020383,0.0849634683655154,0.0921145662320803,0.291697956062971,1.05681932654844,0.238957897528416,0.217524916294057,0.259599564062163,0.342045762316094,0.505332874298447,0.494308775885404,0.578511736531586,0.521996655364813
"37",1.62174887692348,0.793617879705559,0.866810010464204,0.741718710481633,0.774022567009521,0.792090388993059,0.75581457500258,0.872483313612539,0.747905523116055,0.791904659488188,0.763857586439807,1.99230830998827,2.62706394665715,2.36909754154062,0.777052144886522,0.725802296877824,0.823723085652613,1.05527572337865,0.682107339033392,1.59374416724223,1.43885130280838,1.38531583306312,1.37859511812409,1.23898434291665,1.23866357368189,1.14518418419375,1.33253189093492,1.29000417112771,0.903750371108008,0.756587381195478,1.45333952271942,1.34829851904722,1.33455439931376,1.31445442606111,1.2989723987645,1.47868125241236,1.27924426236611,1.41666638404293,1.35157761669862,2.16716876173735,1.6343317569253,1.51225354847226,2.22964993736978,2.09369029381272,1.83349315811429,3.10245399197946,3.79074615155772,3.68383145042724,2.14933502215009,1.17624455913877,0.675843651238387,0.692170724246329,0.657661923377646,0.754847959647367,0.70386139012486,0.665875312151631,0.735271252748072,0.760355948846543,0.736806803169778,0.958088097897231,2.47102167201536,1.68856520131484,0.714928528200276,0.676901034677144,0.685473901218184,0.72765066181569,1.55019455751345,1.31398391080802,1.36033065901861,1.31934681154058,1.3715809673034,1.35526670241328,1.07571258450575,0.669461057382093,0.630323174952009,0.623361086676469,0.794224933707134,0.69237506683722,0.977862419036435,0.45836808290756,1.09046017159005,1.0615014929119,1.00814747078477,1.70368540208164,2.02819194857708,1.62149412978463,0.70161861050704,1.01292058067075,0.818910750171258,0.908939657418592,0.892187683217691,0.998855669015391,0.902386807508847,1.45075612022623,2.18736258720986,1.66672897725195,1.56279721665384,1.48300934045783,1.8215313976946,0.855401076339906,0.913442280048467,0.775628604442253,0.695539122606787,0.841755805282395,0.813766020202023,0.784479171139922,0.799150174709424,0.845911663467957,2.1863501014861,2.91989281464452,1.65241440461533,0.785067834834777,0.899073916572754,0.827768454660315,0.72532987518141,0.864254556281707,0.862107266594014,0.94211629084731,1.74748012230908,2.11297561870792,1.76115918291151,1.24121519730529,1.0294997033061,1.233030458287,0.868036414028066,1.3037029419339,1.36423027180931,1.35406363511374,0.852291333032965,0.773652536217116,0.748558924995199,0.702637418962726,0.625171421169771,1.40674179776415,2.07207221134764,2.08320593665603,1.46865258921655,1.9513465630877,1.75837614351263,2.08463681157415,2.18177643871337,2.03325643687842,1.49850097776717,1.39958432640179,1.34878160541161,0.97732149812596,0.711243814208296,0.844849032107362,0.785168612260109,0.824811879857766,0.821883633238221,0.692829103619695,0.812187012955892,0.776198262322508,0.856969485597091,0.845873474562568,2.19575530609243,3.10546124203073,1.68060720767016,0.775544613194249,0.933688208385654,0.770245343882963,0.752164258693245,0.806777655074204,1.10818914205031,0.905757953636666,1.16337376005778,0.998221437602569,1.02442924982839,0.548700667788061,0.31258513498163,0.258299165815719,0.421348783909681,0.227543360882279,0.41699462473361,0.242448193573045,0.14458003653396,0.150764329942504,0.360733046715515,0.512575713899441,0.784983517323125,1.50414197424282,1.79132630094495,1.4784814837242,1.00360011816684,1.51001618737704,1.60030421506878,1.10528448133081,2.15848130210702,3.88445273515348,2.63618287308292,0.900036591546395,0.786717697861549,0.68593233873826,0.695178559167495,0.818366838827119,0.6724513280256,0.652969062319818,0.631679894177675,0.701471358920779,0.838250394107156,0.802304937339373,0.784918030802589,2.21628622850428,3.96468626592395,2.64291747965073,0.138291277884809,0,0.0667650419854612,0.0733125387893371,0.126731644201514,0.173344513552533,0.0763528473219851,0.0874858286396651,0.10930613821543,0.153784157095035,0.10824407342679,0.0328787469051274,0.149084475555625,0.166902304399913,1.48165891766502,0.0334984063748263,0.119895960768907,0.142635113032184,0.0451771759968352,0.141020067282044,0.178584941047507,0.219967877296065,1.0790049312672,0.913330775377363,0.418502787496623,0.888312248229964,0.586717173705645,0.313225458490671,0.819131914333753,1.05223601680922,0.966032165322204,0.912668127207809,1.36092385769748,1.37159288532094,1.52966419879174,1.63647267818056,1.32886515850767,0.72799486427382,0.61138033350806,0.847847135326331,0.73550626649713,0.723308583939334,0.664694434778686,0.714899825610859,0.862128951664619,2.84028005009807,1.66243238264351,0.699432503321684,0.753489990763724,0.767656927822786,0.768170599833526,0.746734809828903,0.748405307385822,1.11318747864194,1.51793840729221,1.58912102091753,1.62825951776085,1.31545024093048,1.63636763571839,1.59338669212132,1.25961843956027,1.13447521451311,0.963689230279772,1.36415634069426,1.03994743624116,1.21502589092219,1.00700012258979,0.849917226667003,0.983784034691384,1.05088583659001,1.58899423661473,1.82872376842376,1.74828508770788,1.24948505349776,1.3171505647126,2.00361700909702,1.63155328789054,1.28486744441522,1.23787984440253,1.30133381462397,1.4117846772049,1.68809318726474,1.85372304247494,1.68441952311103,1.66863660647481,0.839301439723515,0.485057892177794,0.758133830986113,0.723357582063296,0.803733868106048,0.707984457388422,2.46622472529732,2.82393731665762,2.75870151193343,2.67054363134541,1.90276461127509,0.962169174559611,0.947608996465517,0.744367310082741,0.637899296847257,0.723553890739404,0.615201576829778,0.710898567104951,0.846795131849722,0.669560721043165,0.693437687672778,0.659771520229614,0.790962316437418,0.585976866222692,0.72841132181568,0.697563075487352,0.714332903922647,0.798689692408246,0.705111418144988,0.744290025444049,0.955419984982092,0.716809656943435,0.133047015698087,0.236663523433109,1.07620293315681,1.39365696957774,0.854878565771733,1.22715963231079,0.992430698516346,0.902999936751865,1.04094419735193,1.1876314727874,1.06171835427045,1.02204442436046,1.30064355666598,0.837775256200963,0.784794480082953,1.0911258940431,0.740084014634037,0.574555056235734,0.761677699361572,0.674221443844379,0.673911020555089,0.807449552295257,2.57495526337152,2.81829757212292,2.73360102727208,0.808381548790591,0.673884539840388,0.624394516835385,0.644090550347629,0.730440702769405,0.719285665230554,0.634806095217534,0.74898324285918,0.635854545329704,0.673274112385533,1.42587665065835,1.63406698059765,1.64786277530261,0.922614915005901,0.82895681566441,0.597850334032595,0.100577220252073,0.151754188990483,0.146256973849652,0.196957840965838,0.250513844196676,0.606203867449209,0.785665592062712,0.381877022902656,0.800297391154281,0.773315529260847,0.670854795571496,1.2638586234801,1.5608615435759,1.31996043394276,1.23270259870366,1.23833789196742,1.12158231640198,1.38137321957216,2.13319058559926,1.34462289186161,1.11411932781681,1.42712824007805,0.527201638791692,0.747222827996603,0.733211961766482,0.68189036553788,0.662395049635343,0.6854885288727,0.708100255281293,2.51427939288931,2.76166910723044,1.88567838517261,0.758038773644623,0.767800708109349,0.699167927847432,0.681858508393814,0.58972845895623,0.646752638625421,0.678438904916503,0.749719057813029,0.76503280007089,0.780382613675451,0.990304753338865,0.760302512630048,0.220414639097715,0.788749912958459,0.592931450951848,0.241209346979549,1.32682446609221,0.443722893245161,0.229317126857778,0.569104255102442,0.179035531100912,0.231973752379201,0.609989438609633,1.68037250669145,1.52749770710738,1.95439541912477,1.98983901578402,2.55119447047131,2.62274607293818,1.980247557593,1.62051598322663,1.4874147993053,1.65655670791061,1.73094219828194,1.631216610316,1.84260550430022,1.56609409909922,1.46161544194273,1.64344823366445,1.51713070723908,1.54744795419122,1.72215184441452,1.38645851387564,0.317307028481245,0.683917181580467,2.52577043958175,2.70000766231482,2.79071601515175,2.75947219511047,0.705704594967331,0.710945255591277,0.737181632145762,0.730374595182761,0.653760887097554,0.668960529609541,0.93249866704653,1.11352604238793,0.854878197692586,0.907354608338745,0.92572598712723,1.12059139240278,1.56439642925905,0.93065757822815,0.819259035041983,0.643569446051097,0.892092468422246,0.791299162379013,0.722590013423652,0.25168456665801,0.749808670326796,0.278066727615294,1.00879493882299,1.17555440114457,1.32487197650358,2.25231510054113,2.7222579389813,2.41374032097459,2.40912583323499,0.374162233417368,0.209625809025114,0.370026090041618,0.410048696956161,0.152664539575433,1.33255458841194,0.6902994070882,0.873726311324477,0.942029802079671,1.07474298868726,0.912149248974808,0.71086893278253,1.12335298438672,0.698348326313621,1.06777548896626,2.77568271069576,1.30031596264659,0.672844638391477,0.791944713500883,0.674668693967416,0.650199389950165,0.701517353766204,0.712481088596787,0.609004794334318,0.687264851582546,0.736347790902092,0.681222387382891,0.702184514744374,0.908118035829784,1.08291944616513,1.51820330060062,1.10023610337393,0.428261781309897,0.364795998751902,0.898405656230307,0.339434588181571,0.473876742215675,0.328972885666448,0.511842739328653,0.427916632502204,0.534619222067718,0.744468889169377,0.356865156762499,1.12842030261116,0.517832144769121,1.07078134203758,1.47195801757451,1.19603574965336,0.67736162672714,0.90383858137624,1.9559023990307,1.895074865507,1.56798261021808,1.63351437619039,1.47176092082037,1.55738474812418,0.928089007298291,0.689242036840104,0.652426414438532,0.690012008110031,0.734157557429342,0.72013054623618,1.63690528308599,2.84480868749036,2.75890829358436,2.64494086400255,0.663111589451448,1.3202318918162,0.738393248089043,0.712144692825276,0.644734728646155,0.629689113375193,0.728491519137375,0.665717776053342,0.604828965511214,0.718535603805714,0.798101253254953,0.825025493953885,0.619801359782189,0.72800171782381,0.777293431859872,0.827432232189545,1.89681756505005,1.24607542529616,0.53573138851541,0.169307596199993,0.104216621343267,0.199021847752922,0.160403648505022,0.07432802970134,0,0.0135289636201691,0.161718961659432,0.636026659982132,0.984331625249613,0.104560858203268,0.1205279182213,0.253424601918688,0.191615160596258,0.201786916036496,1.34146213375156,1.16178091332464,1.23831789650797,0.858701788516593,0.793410038843855,0.841331363301955,0.453333442418452,0.145083494930349,0.461632175493645,0.131146229500356,1.732972690883,2.32413939213665,2.85947987039615,2.29513465578967,0.713483204748895,0.577520187874998,0.676315596970148,0.645580074906223,0.619448918429082,0.62137908821943,0.628022727980416,0.923678934205175,0.232071890464597,0.238675251874554,0.193941501474676,0.112049327552497,0.250847875202183,0.425018901394487,0.160518849334416,0.196650797548346,0.0732727534054884,0.164059134209459,0.298912658150881,0.0528104532304481,0.135079822190608,0.051986092168148,0.028541822232851,0.577598829939428,0.834634445215416,0.816370205755131,1.03865724885159,1.4149796403265,1.20289630915393,0.768677873067981,0.87139561863342,0.866196204154657,0.918381327458909,0.870464021551949,0.673045615098233,0.654463583085895,1.32385250960312,1.16121887259222,0.804600647930854,0.822473081611118,0.589246791140537,0.670506175618072,0.555334211908152,0.735039063489035,1.69836877421625,2.75124826053589,2.68725013684798,1.43160854439263,0.700571964253621,1.07873580631874,1.20302643259942,1.47652103176845,1.43834445972542,1.1598032888384,0.709494173599231,0.70198320886663,0.451006571154076,0.31501367889018,0.0406974412027377,0.1142321904142,0.200653223635651,0.103841403856392,0.0465699144296499,0.123106802271609,0.149794618974882,0.113242028686576,0.0791914967030901,0.10788226401071,0.120066985289067,0.11838128906315,0.126357541688267,0.167422959330165,1.11358302092665,0.82869127519837,0.881213447826316,1.09477356875148,0.88091984010187,2.23762189476028,1.68458795771085,0.986545595137161,0.987791889861956,1.78052400394064,1.33349171581553,1.13977564011787
"38",0.0763596521774348,0.0695958100089083,0.217748155094589,0.0692967292142961,0.0427641820987999,0.219826999760911,0.067812333270835,0.10072232839364,0.0576868446354778,0.0749409918937929,0.0841214675765957,0.0185421966523897,0.13157287128774,0.0605443767885276,0.18213633218473,0.10598111196256,0.0350957486021408,0.0968588436345018,0.485518758955067,0.157744405981494,0.205836872842188,0.0329842294271063,0.246257543150922,0.0482542904261863,0.183695998170986,0.253643354574488,0.825995403355155,0.784912868053343,0.955033967216716,0.301873921974362,0.163943604948721,0.256671013175344,0.255930729156101,0.265795914211601,0.267963397422746,1.09038514780379,2.82550269493642,2.73781884103458,3.79808098213966,2.85171723584721,2.18947635270408,3.5293387655112,2.87904082563119,1.81586374089486,1.76474786040622,1.63277142863366,0.481917276884214,0.142145746974771,0,0.110873574755068,0.0453022113224343,0.0698625337647069,0.248541821817926,0.0714749785088121,0.0512131565664727,0.153419379441837,0.0659461458803949,0.0316610266873738,0.223028154110884,0.113029383462806,0.923134535921648,1.8978971491847,0.0430593545473176,0.159043556101831,0.162739330891545,0.128081614656521,0.0414463876953817,0.0455764256446599,0.147367118034083,0.177229075894362,0.0702755724552184,0.181724015470064,0.0710284089631832,0.126067654361301,0.298451917305724,0.54233179685839,0.192776781595521,0.13813634461724,0.130859437305472,0.148378700123949,0.410413096976077,1.18717428937489,0.706553466238366,0.187554731623888,0.205313702587069,0.0391478187031215,0.118424565249892,0.0675291627180538,0.103880599038217,0.128651525976876,0.157622573323965,0.0685884806334734,0.157882695272023,0.105181735882243,0.126569129433771,0.221980112838014,0.300686481596982,0.0980552127836074,0.0385117869829158,0.123545547461145,0.113092026995718,0.0606592209906835,0.105851003785937,0.00376656367888906,0.0933243848905689,0.0931311006130947,0.119620557851582,0.186533418163188,0.986129433858361,1.5755802480233,0.124634841080582,0.00803754924398897,0.0645323859700916,0.161364900645084,0.105563985172124,0.172952928006382,0.0510307261838465,0.148526296500444,0.0994110141258762,0.0985453665981926,0.095963140479833,0.150479653951916,0.196064608737607,0.243495057941728,0.130876689668758,0.052716121060484,0.245452015287962,0.218707377248986,0.507859882822861,0.551379490771221,0.266825121170452,1.77837400439117,3.42046694960195,3.10375910406904,2.08995947712513,1.71037079900539,1.36521229497488,1.93018011085724,1.36465519334321,1.23255659819109,1.50211025580237,1.28288090619602,0.260598944821362,0.0922020018301914,0.0914064359031705,0.0191106405855565,0.158510155809687,0.0372916171692345,0.0419566619275295,0.232757428587593,0,0.118463764209705,0.124564581079545,0.0560183342003021,0.133379834620399,0,0.124053141025588,0.168674143298384,0.0586807812723488,0.113924418744742,0.241836675936699,0.0591248341821857,0.500608989932997,0.180389008782253,0.157045975867262,0.143544396778302,0.00755424853161553,0.193783439052535,0.131742629068926,0.11022421119268,0.27467259473986,0.824699962039461,0.0446876824982207,0.22271113119712,0.105903465468204,0.142172405651592,0.153597440035742,0.057042920979466,0.16590430424115,0.220671281344528,2.2027658043117,3.14915129493295,2.68658993768898,2.61728966319215,1.56990810887624,1.99127766711705,1.96978613334001,1.67001233309352,1.53168739548543,1.19557706289172,0.156734392562721,0.029889072406837,0.163769889100143,0.0176753975087852,0.131158290842109,0.0840944765732083,0.128649759428212,0.0424406877441237,0.1146257815426,0.168800626775247,0.125771747010797,0.18692575021562,0.206693994722187,0.0324219038912771,0.534570099248801,2.13209980563123,1.96738109861685,1.99932023118518,1.21331454476072,0.0858944997619345,0.108480270713682,0.269558853561191,0.204554591685762,0.235405650197189,0.280057865271489,0.160295700876851,0.211421431182327,0.0290915520295503,0.239608318914988,0.252946638691947,0.138555435010413,0.300650144933814,0.111612889770189,0.180607103702122,0.121142375957531,0.024048901938516,0.082625888896147,0.240208626947069,0.141362841875604,0.0966563675565684,0.237487785773015,0.117018769919287,0.0875572854141943,0.0848211006690258,0.0421709670866798,0.039099380608162,2.03857076585833,3.02286646481047,2.66039260109077,2.24183527793655,2.55071217254191,0.0992968099126595,0.149439501162489,0.161722592572422,0.0784564007652106,0.125562624113835,0.182451952517911,0.0896395477263025,0.0360563765406727,0.113947018136279,0.0179962517866912,0.0724178610193164,0.0940733727682343,0.108126337880603,0.120063630392992,0.100651948956488,0.00327707274805837,0.108151580101166,0.390629258925822,0.182214490290439,0.438636664277535,1.9336295779548,0.279693961000956,0.0843049641637311,0.16933071978487,1.00785707799992,0.211051348604891,0.540193850248832,0.0280687923795361,0.179416067719722,0.0508368271524592,0.129041837317881,0.196640760507809,0.175832887880943,0.188758922463322,0.305579305742909,1.30021095810778,3.05013649555738,2.42127521401609,2.32415043154344,1.39446034889617,1.93931751695742,1.7221423464525,1.78691700605694,0.711545159803238,0.336173122831266,0.0419385526770583,0.0832182267906769,0.131653044530497,0.0740830164195797,0.0603669294988849,0.189428689210931,0.0921847639754434,0.149110551279263,0.185733758512673,0.112494764696753,0.207872376433692,0.0209201732290081,0.146966947771304,0.0845374575198688,0.105348936022245,0.0397519506301541,0.0897407477047362,0.210932548682526,0.110682669438179,0.213780851218628,0.243907382743172,0.0146089509864633,0.137950678371468,0.205364414931155,0.132615412045119,0.17306733856424,0.118894799975295,0.110601502378037,0.156531208857911,0.187000848142825,0.31503551278502,0.177181081287471,0.0916564251717555,0.0960106474438637,0.0726718822050723,0.226100633070288,0.0936185920813155,1.99421683012616,2.88762929170179,2.11442841604736,2.06181733537416,1.78717747576906,1.29529062925628,1.72661676590579,1.86752409222025,1.34907946871505,1.2811648369399,0.381976165535635,0.483552959375648,1.91232217059038,0.10422268612001,0.0731224170602625,0.105872902000124,0.0960199225961605,0.0836438371789322,0.151833146580756,0.141230657733473,0.134165334981326,0.0379831555412422,0.0604802180319786,0.161927442249355,0.157952292526706,0.164309543779015,0.0289075629857408,0.14651928752902,0.196460495153315,0.122014572580439,0.384293679733705,0.0973753592632091,0.192192940730873,0.159905819595314,0.053642439870898,0.131277540387519,0.225361264140312,0.105466485010778,0.184919258796812,0.16445120771715,0.203393641492588,0,0.233054629224408,0.0480882859611077,0.13888349590487,0.0868384095902626,0.180112681219512,0.15545341851272,1.95515003729799,1.99406224983027,2.14233952059978,1.69179532188003,1.55390965368068,1.38880590462509,1.34829731263875,1.31416833136113,1.38280583016488,1.33034400984596,1.27610028198856,0.418355809661926,0.0903639339793654,0.0716863699969689,0.0970742358671093,0.049240932429049,0.0684281059165715,0.0656406890443837,0.136444039864013,0.00581758306119957,0.036866941782837,0.151690176392696,0.152326226603575,0.125694468854037,0.22428924885339,0.538398713886837,1.83361057140729,0.179059753127059,0.103953274484013,0.119025811879684,0.11756463062499,0.0905962486847969,1.77713697956855,0.241620918329929,0.130803739020543,0.134803653507101,0.0754508721769591,0.244188191308789,0.221079631029184,0.247446084479274,0.137803216778445,0.194203641111151,0.248313976817702,0.17339680411644,0.263987100922302,0.182078371285589,0.0998938134396797,0.160585319532189,0.134918767372156,0.204666203894709,1.13486847462728,2.72941257629935,1.5161299111534,1.65077434716236,1.56035220943439,1.2454382630665,1.27965561936765,1.21679432131176,2.73480792049334,1.53625231063638,0.135435434551303,0.188566039216095,0.119187346662405,0.0172191575922659,0.0853957720988389,0,0.154408675434158,0.189451974214966,0.0347054268780038,0,0.108603960999623,0.107718468717584,0.0284087268121863,1.07637086018799,1.73261006739774,0.161185149580705,0.112337071798865,0.170233424621999,0.0807551159920015,0.10455716123721,0.208811269046493,0.118252445862185,0.26734885139632,0.19827962113291,0.199665766539481,0.198704371482741,0.0684498159051768,0.326453709175011,0.0735515158678844,0.0729300035279171,0.0964838073424679,0.0160795276547039,0.140840578043502,0.591253214226791,0.181491075628507,0.166484255593255,1.20216109764122,2.0206600966193,2.41178565734772,1.9074635282813,0.931969498196353,1.15629146973419,1.42058935384202,0.825775279425201,0.921209019264522,1.42557558734937,0.957860449374427,0.642997333358755,0.138162270097248,0.099791291390647,0.14227856801082,0.0423538497761652,0.0709129520869946,0.104822345786535,0.0638279411506649,0.115954592106257,0.1108832993812,1.71372424957556,2.02276999076112,1.40016864125676,0.195750160891406,0.210746306114085,1.21169278909634,0.0426651783721073,0.193546857425896,0.387316444676195,0.0279627508759924,0.124310816430963,0.171106944760488,0.129401104330147,0.118574466672882,0.255207979077736,0.0897419193066981,0.0363868168091972,0.122750552791734,0.232080216768448,0.128515048223232,0.0979365306217603,0.170697191862452,0.0714444618599174,0.239276354052836,0.154690492733192,0.131010724457808,0.169433649765883,0.159610028827264,0.274410438351178,0.360605975984491,2.6216345551037,1.91526821612869,1.35082874499394,1.28940154139952,1.47296214529163,1.38873289037051,1.32324667907126,1.44728071431842,0.230082295777277,0.0100054369490858,0.151551984715863,0.0739556309072923,0.0317695331662983,0.147313907920863,0.122589400533154,0.105141696649159,0.109825626953362,0.0797399097233928,0.125790726076309,0.0415449529237019,0.0537060385134809,0.0354768382076562,0.949087936902428,1.91887720244189,0.0741622747253004,0.115310231528963,0.144150453543809,0.107621011982992,0.099427247761942,0.096372524115291,0.0624699436117402,0.142652290893311,0.106394476299791,0.211065367756747,0.19534444127016,0.0867013139854821,0.0163946090624909,0.102471559819301,0.133537433087454,0.0219056357778697,0.0940004623211642,0.10293811884311,0.282301009588111,0.144482169227368,0.139660610926358,0.0453305354606586,0.641050096953196,2.7865155670541,1.48151763661313,1.92983656577631,1.32952787549045,1.2910420153665,1.48831073479259,1.28781532995474,0.335177763278152,1.37751529017408,0.244206276835423,0.162779452417633,0.149122598824291,0.157443955754639,0.0688215750923889,0.0447172130514332,0.0576808028718639,0.170907193816214,0.0717668293572312,0,0.185247590321725,0.0832539725890392,0.0762293363262312,0.111779393746202,0.0706497852263031,0.0984933958095534,0.0942939778834745,0.0744827460892207,0.107059141673634,0.351385892687989,0.241738400005097,0.188304652677641,0.0731718934678474,0.146317994643226,0.0553386827743774,0.207292634590596,0.273639246910517,0.135857660287406,0.0320922022048284,0.182121556134736,0.145947432208667,0.0672340398525469,0.260286640237763,0.0940354612545593,0.156775402106198,0.0737910518822634,0.0671787942153219,0.206393498947968,0.2074871827122,1.14259822842949,0.857738585415427,0.766366067688972,0.227152532388203,1.20650831626677,2.68838738421074,1.67331479072306,2.04220996015982,2.12897221082011,0.586993295956993,0.272298860776079,0.246559995565637,0.14065993892653,0.0913088782260583,0.0785869430146659,0.0726658565854846,0.225865704768899,0.0859903958430186,0.111588176841664,0.166468539534552,0.122334706580653,0.0634378319346372,0.142585845674736,0.0545278881453935,0.0817708994167563,0.118319874672308,0.963303106650633,2.1433513024836,1.03601201731162,0.170894129351414,0.119825241197181,0.0558579904546078,0.151545317287366,0.124616946362259,0.117450943029269,0.20040364652901,0.0498074600896054,0.113134263970957,0.184853704825888,0.147813484856927,0.0314998649672977,0.083766993548077,0.160954348801567,0.145531607458027,0.285394287741851,0.0533923288510451,0.0297159598972591,0.259572808735834,0.00421896604552614,0.186191041590384,0.174259155750955,0.110260917387033,0.114069948668256,0.873140839102904,2.95874608778646,3.04380193634157,2.54863950474583,0.422110458986423,0.049898162197436
"39",0.214579624879557,0.302479196724286,0.21782622363469,0.228309132880176,0.3716382363391,0.408179681228434,0.328368133408006,0.248919580113775,0.378917787953211,0.401224325412805,0.464357473124651,0.449955858728677,0.380441509936962,0.426922251527216,0.399256674327603,0.953277047782647,1.95939573799101,1.04071821316619,1.54145260055142,1.37864746728157,0.647876703165386,0.447765369413911,0.451651219472273,0.556388021657966,0.546910089135183,0.422294367154331,0.563289033842905,0.394940432681016,0.52580306323147,0.482844398164659,0.32349973706363,0.2439073718456,0.525075982761908,0.566487257775648,0.532927358613441,0.597386344229936,0.606199063045176,1.31460229238524,1.98880373553666,1.75821476629341,1.2490364503269,1.90309642578188,1.68792386558898,1.57075351082612,1.58616180450366,1.41316402733198,0.468171960363203,0.434440102719346,0.402342853240476,0.370473985559968,0.317869008351842,0.312205079934634,0.254783325837857,0.35610214144826,0.261352750931565,0.26110559865519,0.486398122307526,0.325036186804004,0.493947267667627,0.613357618827505,0.570087186842515,0.594653313726197,0.624104317509391,0.859294506643619,0.71396552395026,0.614453894576966,0.424239241106432,0.354215978282008,0.382913522827782,0.45803591437825,0.377415889135459,0.496706543775462,0.478098571738495,0.582404090845361,0.548776872809778,0.575633624672162,0.533988712838033,0.626011624506439,0.30715078825291,0.33739823002328,0.298898665447118,0.387990986863741,0.360501615801331,0.269181895843222,0.224646715661235,0.186769344689282,0.329652495504731,0.327340762568222,0.392722193464026,0.483098539324143,0.467630615182206,0.471476208195742,0.521381684550754,0.514540068133251,0.486093837621393,0.353616446786835,0.243445944425512,0.527479897116266,0.427122273857763,0.370780640827413,0.546111968142713,0.396091720451246,0.683156286290745,0.388419974009155,0.526214007516782,0.518862243456155,0.479340520221538,0.719756761294701,0.477367403482017,0.623152853027188,0.505593885578719,0.624582414229942,0.53765196122216,0.667234793610629,0.849324429929878,1.46660563401816,0.507183289244613,0.460306008413459,0.310608621686602,0.634450875873199,0.408942565023645,0.615488654432418,0.61981655949537,0.515560298788662,0.55744462368853,0.436795271314217,0.361169657620685,0.421310202146501,0.487911348123733,0.383971495894003,0.462407562785769,0.758479513439788,1.04395748792977,1.20202820520718,1.25487366191348,1.07433329648098,0.822449273116297,0.732593703219305,0.622302437965951,0.762153737359124,0.75212039701695,0.726036719893578,0.570132625115486,0.268782938227461,0.44788054431979,0.360250214404564,0.22258197095653,0.289455111328007,0.369824298472863,0.323668814150278,0.333656785892793,0.249131194478734,0.23613683121845,0.203871466694078,0.217774703959954,0.220754725173026,0.35544391382028,0.249022912828379,0.951348604488299,0.705221840209284,0.51732309238684,0.662541408459798,0.682365400566662,0.39815921082429,0.505787738754953,0.500744355634388,0.550666899737704,0.612064891388576,0.512988298603723,0.518132183088487,0.637811036864106,0.983866152964046,2.32801465685123,0.414142973236413,0.353943787950688,0.46526655070225,0.75413616965658,0.433569418769875,0.511072027748808,0.629806214145978,1.01781871463463,1.16818510752677,1.38777525818011,1.25497443203692,1.14976822249663,0.705482530099468,0.838155563867037,0.749453745967044,0.702714703281818,0.871787310127924,0.599921948855652,0.370271342822762,0.401330006018974,0.410617528494462,0.322191404034242,0.422801937523762,0.374291339751597,0.172448908160873,0.289169781598516,0.29045454688049,0.254166090971946,0.326538861359319,0.33458199400323,0.337908340348886,0.358419467278436,0.451250211324429,0.316871792335561,0.573454342624675,0.53384763787899,0.497169697084472,0.581044974014599,0.464487869663015,0.46355790728339,0.466364723481915,0.453939276302752,0.490196580277803,0.667525005695218,0.517818014773607,0.292429699152763,0.523557535182695,0.37712715404421,0.275750930559543,0.299048853012359,0.33858865690196,0.249491292203792,0.309410082639396,0.323318843270563,0.387465095867232,0.371228283098998,0.360544207712395,0.406025526518253,0.360914125919559,0.377221580844919,0.598816282996368,1.37716822347182,1.5464523655037,1.3641253788402,1.34229098651426,0.314394392386048,0.3450562402258,0.300738305857869,0.229557748157895,0.370179264871283,0.269295230639202,0.236933882119819,0.294675774146385,0.319748347549224,0.419361614573341,0.576885666481356,0.454855011704171,0.385561235856835,0.448791326716735,0.454920220952692,0.582905845058149,0.423084651486678,0.662776458248771,0.575269868670135,1.02597160973146,0.543106682602491,0.43910333690856,0.613102368692183,0.489752937593725,0.518454254812176,0.422476458181885,0.509340890067633,0.441217394509624,0.552217366624493,0.483775867985048,0.801633430250196,0.514102604767142,0.584057893357177,0.461114639541683,0.423421774950438,0.539418860212576,1.09332609121457,0.974614170025268,0.941600082000688,1.1564539786456,2.37947521473012,1.23648317867066,0.422241595570117,1.34760514409082,1.81381399825864,1.88640748624147,0.712114415352523,1.1797060025009,0.7598780200983,0.393281743869612,0.450397137074305,0.172800940509791,0.357055618053498,0.335372937328096,0.212945823485259,0.276725508449132,0.270629712186397,0.252794998899195,0.230574808120826,0.371740564916718,0.332721031866145,0.477904839859754,0.478383052796857,0.550076103654751,0.380778725175352,0.400489819867004,0.660849950832015,0.583170399600141,0.471577056289407,0.573336566207232,0.562900352440705,0.540125787004583,0.640050507841799,0.462962530084976,0.544015254353063,0.488795522649968,0.648504990777423,0.424564740176582,0.433694772013564,0.432598124036486,0.491928946658954,1.37778952808858,1.42649487344235,1.4316288705354,1.26388447833325,1.2820316244971,0.598575241253561,0.57187320317264,0.387641530480105,0.558607609481026,1.07791893984972,1.12042939349855,0.656977835860649,0.717560942511313,1.17901320171219,0.613549945861856,0.436169612128354,0.454841234073001,0.362578886564239,0.449977432422166,0.45845629711698,0.42597133328117,0.368364231616262,0.558208126997348,0.647773654315221,0.430958780708448,0.672459429947312,0.669912824063771,0.665277962188408,0.540211238738599,0.570092354170777,0.494615604826218,0.472029878759154,0.733911317086356,0.670011822033713,1.06202766944772,0.887104611023418,0.598303599345363,0.646185044350814,0.573969195183118,0.66832013717959,0.504324135466154,0.670437550857708,0.571552687921156,0.52588442422615,0.639984294099568,0.629110971830052,0.415769929773671,0.535704973064903,0.306140485410224,0.177196020095375,0.250053595285639,0.360464156888694,0.515527206155183,0.546319720474938,0.494616873132149,0.427814553041441,0.371077476242558,1.08836905994079,1.15347687845853,2.05243313525887,0.557585173517378,0.736990125792265,1.98127094538357,1.31326987962761,0.901563827695635,0.928916478047379,0.459923006447367,0.454964205305026,0.297968045467314,0.410218510029926,0.236214250028214,0.166984357806819,0.342164109868167,0.279044543948374,0.312626765091904,0.347232924978768,0.432179400243576,0.226434026489388,0.352212767220559,0.314614476899596,1.19312814821712,0.46372069246682,0.687913616195985,1.54520892885968,0.621384012972186,0.687855782560148,0.659184611805553,0.576375629099354,0.530719393140575,0.532219321675902,0.582978780610166,0.472052734531066,0.499304806745,0.629646548749121,0.43477878481896,0.388794689952399,0.625781236699321,0.453146832806821,0.371313117145451,0.610137440598617,0.517087029983712,0.509893053125755,1.72956552171397,1.27091549112251,0.874354556350927,0.950306604001059,0.979486061489133,1.58851924042108,1.81031279559722,1.52462426613963,1.32976016928472,1.46361520265396,1.0131123016301,1.30149139264402,1.05526033315659,0.542306381930144,0.482705175358628,0.476152409734562,0.306526057168433,0.394668675217486,0.41341467717267,0.302118198796618,0.399731478364424,0.34094241286293,0.39001434985614,0.275888346455743,0.252437750029624,0.355392523540612,0.542944896484562,0.538934977425536,0.447596349566641,0.650898207785517,0.517380036930628,0.57463154464965,0.64307845612794,0.697580397642081,1.18169758321324,1.46789035164573,1.20158205607877,0.509907915796353,0.633288062754049,0.570427431145971,0.586220505654141,0.686433262563976,0.508202570962045,0.593198535546262,0.653199551913781,0.480737074631483,0.608513242078943,0.729804284509435,0.614477121555625,0.730408754680139,0.70662738767315,0.779711312447841,1.5593994239143,0.863216106560398,0.745149203401367,0.814904280086358,0.736692979789712,0.790465947246037,0.66611565547335,0.547590520733145,0.388317218005836,0.41572279563826,0.270529637220607,0.232243392345179,0.349199516221744,0.209722739688069,0.3580379392997,0.38004756149001,0.273397180947912,0.28872527740086,0.452334875831529,0.401631098477632,0.262370915444104,0.423580298268985,0.34948284298416,1.26474344594915,2.77351521723591,0.97070662086647,0.592826338035324,0.807017577613414,0.487652290160965,0.318021823418033,0.439780884674357,0.283067305003647,0.391238572706806,0.371046571543107,0.395998075948946,0.370765395810144,0.341623531249221,0.297335182612351,0.379838842838159,0.35199346649943,0.411524772373548,0.45184332872938,0.474157337863473,0.447255169750552,0.51500636660153,0.640749261618514,0.472137516779943,0.647939399068791,1.09219493434364,1.76330395008061,1.67783710796534,1.44512462532288,1.39623722097603,0.34511386813025,0.309123888051092,0.188339606311949,0.38876534755058,0.221090948177662,0.273202323650959,0.325645412487966,0.32355103338526,0.32726565265404,0.310406119110876,0.282999343195903,0.288019437353495,0.379873073080184,0.422458344271819,0.289906828762154,0.379800597335278,0.261286898767781,0.377374338242089,0.30308076281141,0.550892335395085,0.598641486772542,0.665779292013302,0.56140917893763,0.591864341107634,0.721006739223344,0.628073970743794,0.778652516103728,0.6394397307373,0.428532131496769,0.51009283388852,0.392234465083079,0.270698847083014,0.216053638849094,0.507054172564027,0.449597863940411,0.522901178025666,0.496441411646824,0.54457857215425,0.478846493031468,0.317535052336704,0.335430830258665,0.318685935270563,0.358889655050545,1.4687158856001,1.2890491947312,0.675772800729574,0.671435754167182,0.9958536396488,0.718637554060689,0.892079619270587,0.288290267190226,0.338416956009551,0.395788094773427,0.322602840491923,0.321006444207052,0.394414170531711,0.400883749438948,0.30900876352762,0.430351906136157,0.38844899142444,0.409498586710916,0.904004692396415,1.11945644667594,0.618921254095484,0.319190075843044,0.464366706727903,0.357506553109103,0.537721506483034,1.00832517336811,0.764038922325335,0.518444451839303,0.512523747011353,0.497089819741673,0.704829916336133,0.61916220980481,0.553657253144671,0.47747858752628,0.509462111677628,0.425330752710578,0.451375365018325,0.517271924468049,0.373061764113102,0.609957659094019,0.504109301470632,1.17340252229456,2.06143783610975,1.28151585817696,0.56989847202885,0.552561472445997,1.45363659494927,1.01006792554116,0.688995555373694,0.884270918142736,0.814537260214269,0.71351361739845,0.484499314402999,0.474908879711573,0.423313470019924,0.515496024966345,0.505966923116951,0.528964681513271,0.478554665571841,0.391313085291337,0.256643040247571,0.247768043747167,0.373138705853258,0.384543263076133,0.277459634906444,0.306563602124782,0.255470217154371,0.278225459612935,0.291150813051851,0.291552237488319,0.281376223804956,0.448556683127285,0.406956355880058,0.703031134171045,0.605540973841434,0.402544993839973,0.300251769034868,0.305026661066627,0.425708718459492,0.246200521251854,0.317234320629495,0.330740159734048,0.363500857489019,0.328550053282476,0.319994143787056,0.339387238496666,0.358981112491818,0.413649740470594,0.1655270134737,0.320203886088205,0.366414899110412,0.281848536165984,0.359231371234124,0.555133561507975,0.556820227976388,0.463305334028173,0.622768943084593,0.883362257397659,0.728680545229362,2.13240644882158,1.72700175946058,1.59183806670939,1.43983222432099
"40",0.143016242701907,0.191687251668261,0.0638716920354904,0.10144049886996,0.0622315695734271,0.128182133671166,0.093700686369279,0.093697349324852,0.052734827528098,0.210567894668315,0.582784176240898,0.697276252045925,0.746398921493145,0.712128263383504,0.969724956230123,0.764485770506161,0.343155120613915,0.612480326924871,0.399909232466155,0.35241767585338,0.182989004309091,0.27296637954947,0.177800535656676,0.407347659301905,0.272714499274336,0.496457272052451,0.309136884322809,0.346399355191738,0.479893050208519,0.205801665619848,0.244855081133599,0.342311408618838,0.383614296104629,0.329389531416497,0.871669220009937,1.25753583467314,1.73697328514141,1.68205935547129,1.4885593997075,0.762985629657623,0.219525347700397,0.373473988999494,0.307958357862213,0.208639722307722,0.1905321393319,0.116031851269897,0.155507718912,0.0886133673888482,0.131653508395982,0.114013903276114,0,0.110457228211267,0.0987273488723788,0.122294572013464,0.14775163730109,0.042952520570917,0.30325859842523,0.322231058832611,0.745658565706826,0.631072645238741,0.675275801144545,0.712357802515932,0.936603189012936,1.45984863505281,1.24096572676063,0.595272465388449,0.284031947407736,0.188883189921935,0.0515795512469406,0.0960296657358905,0.16695215338522,0.238007737782988,0.121186819747074,0.183545149856641,0.0693772322537924,0.0595216629059192,0.0237743906534299,0.076615552563286,0.0448814737990667,0.0746831071782956,0.0995182393271873,0.112098973165722,0.639359547694559,0.688100226112992,0.670656667301336,0.715823521919587,0.629321692557416,0.445407932473901,0,0,0.161823388800555,0.156493185837283,0.060190998249991,0.0997500802624944,0.0447962887490315,0.0946759659652751,0.0972927727394246,0,0.00987888078448,0.131149586479788,0.0784381960472916,0.0763384818315812,0.061686550850376,0.0404629769712638,0.089116820825007,0.242636733691379,0.788158120710695,0.656628489369069,0.644326474283306,0.695958918201916,0.700817528461109,0.704229407931032,0.209368987633234,0.160129541798158,0.694828529048786,0.198788369559886,0.172230314272036,0.125721454057085,0.27218410967707,0.161500869205909,0.227817847724152,0.595955967661274,0.468738698840216,0.166459828043183,0.206314435323079,0.170577625306014,0.279605687798771,0.101420171919433,0.346703208157026,0.27068070496701,0.712462889743841,0.920531342719636,1.05430325390187,0.998036728601683,0.871265675923628,0.530210848222733,0.271130075061213,0.250273774592922,0.301503013772323,0.114763277052567,0.185356603896467,0.115410617204497,0.00910293676324908,0.108604239876575,0.0986341639720954,0.103906838831352,0.173406952641158,0.0981227736586262,0.0712941773026639,0.0653708341335946,0.171410755382865,0.121354498174161,0.100288592411081,0.336390045223557,0.7004498672049,0.642865614820681,0.6454217314434,0.769352131257256,0.7209193591165,0.800381914214902,0.399305254994827,0.242819637084493,0.978576790105652,0.503292539045169,0.498222497264797,0.822670932818376,0.940944576956479,0.854232952586858,0.744488862185977,0.719149545260505,0.954706659091661,0.748650965470836,0.782428127160942,0.689619958223911,0.490909312918899,0.163307835224412,0.362576177458367,1.11443039186006,1.63946362802355,1.61278314087974,1.25460929218853,0.861412646654375,0.837274244645866,0.611558943595543,0.315211109538175,0.258365786932449,0.298536667695255,0.0448394943633363,0.0157848166605161,0.149652545400991,0.102603690035901,0.026002397029621,0.0604202938825964,0.13690940931891,0.0625098760341357,0,0.079593017256557,0,0.115878330887793,0.0486011863884026,0.155042268278853,0.384044375771184,0.618654831419091,0.695059760625673,0.67967249742115,0.693579346946128,1.00570635833873,0.670287596447399,0.135817738351628,0.260398604491412,0.155046163339584,0.0629338834539554,0.436843378032841,0.526669693891946,0.237001544501797,0.212048171947483,0.234747553694535,0.0785006440023018,0.048511833520724,0.161254389474346,0.119530265421356,0.975991700635745,0.343609143500574,0.738035234964241,0.103497081684532,0.269340618018565,0.654132149025188,1.14203409238629,1.48660219530741,1.1681436625486,1.19444878849531,0.647305246918343,0.466463475276047,0.29261995799358,0.323994119425228,0.196591015396247,0.206708225887688,0.154723941289158,0.110291004137128,0.0780990964602479,0.0102927910689238,0.0480339536679676,0.0217474001926865,0.180697390849648,0.113119727180146,0.034156263534123,0.038420558466729,0.0873964361417278,0.0509519697700466,0.234138162727549,0.703733317104196,0.720511438914074,0.636464944902837,0.683081316999398,0.955118718684219,0.612126107794494,0.239046278950618,0.231546636422121,0.289449707716829,0.318289131409953,0.100808727102244,0.277071057847222,0.186301452091403,0.242103553792483,0.0343239983447169,0.165211058417069,0.125630180360948,0.12580334924309,0.0575609739960229,0.244793926714285,0.202896905428747,0.654227390886421,1.1254011778423,0.738625958950937,0.84909741012129,0.98261825667898,0.874386102774416,0.808864065714932,1.27534426045276,0.666447740973203,0.367880876478807,0.148924841799703,0.129975855653634,0.272073638962749,0.153856322758573,0.117181235533099,0.135980378374296,0,0.0807221018256792,0.0574715573313887,0.161066086924954,0.136194203959702,0.0767191021800192,0,0.045265933048862,0.0542425916474149,0.0642828722903458,0.307812931595436,0.673853730460222,0.692126913551389,0.802819233133292,0.637482651535221,0.979370857283492,0.84459836048036,0.289200488217606,0.24182682008803,0.385764345504992,0.765776821983174,1.07441838176768,0.930408495283825,0.806398036374407,1.03199832625545,0.682740111990548,0.217962327215296,0.212634104831608,0.0608188753432793,0.100229032485091,0.0585029253043061,0.0932403381036815,0.117208776347682,0.340982822503501,1.01302495818375,1.67194275444087,0.875744138737945,1.02659059952721,1.28520622999786,0.898506604506223,0.582768280501043,0.306914260136708,0.321760967142858,0.307736459232472,0.147395272245723,0.223614664579161,0.135640864270583,0.265867372675056,0.0554811195716766,0.131605605754812,0.12859088981718,0.0274094209365103,0,0.144971646288822,0.069434215558101,0.136015821487043,0.087078166444062,0.155351738293528,0.303249766865115,0.664803046727918,0.81324928385335,0.740468595831128,0.745141868347373,0.898987065371717,1.36400065988684,1.17409707003256,0.257329013450482,0.142987490130371,0.156257239433527,0.888855871796479,0.499538085679818,0,0.0774646415692494,0.0802630441424002,0.152963384702989,0.336215043328777,0.35141355358105,1.03844432279794,1.14801256803537,1.07384484079392,1.2796038151829,1.87553547508579,1.20630496521291,1.10226424118478,1.57410072729146,1.76998130648072,1.23641103497866,0.966810258931652,0.583274044402579,0.273574929002997,0.268110549284601,0.214849181919694,0.188150706142948,0.23922740051487,0.228835270941673,0.138418907631662,0.13522133663583,0.0629870334212253,0.112404654057462,0.0628305003241814,0.0380874107848387,0.163163995567189,0.0779372672236493,0.00354089351810023,0.123479616818868,0.083606984631424,0.328440756096057,0.719802549031503,0.751512630273455,0.67826422593124,0.691655331977891,0.832030350278223,1.0103112708404,0.261744803910464,0.623805658260297,0.155763629800684,0.173728952336892,0.149761809829607,0.108517562719167,0.259410361229743,0.200516867867486,0.155458968999148,0.382198548724777,0.248472377125719,0.159307776853762,0.141162221985036,0.186096467781037,0.187294946576771,1.29124483995768,0.533996578109051,0.17397887265122,0.644250251319809,0.910122294831493,1.9627595615475,1.69132402197861,1.11776655598814,0.583944617602991,0.23442865299715,0.2584209102949,0.321558969671692,0.115777644541566,0.200486670870976,0.18728664828431,0.0849509931584404,0.158292111601649,0.127324508658745,0.0950496140753794,0.120323834793258,0.164903264597375,0.0950072516434821,0.0760121875866399,0.0810420215360272,0,0.0639897573764699,0.297717705727887,0.594880870351435,0.736919427700241,0.683652308152037,0.599106174308602,0.67035241629429,0.725925560114437,0.475060761733876,0.818264507027086,0.327101988169197,0.221871451909564,0.307916661596112,0.461591885631194,0.719110152855672,0.292793971597827,0.288966384889213,0.306137828657886,1.0529677216535,0.734470571913,0.342205450118518,0.209570761603253,0.241420961651198,0.090378828249719,0.146343305778229,0.730101015701065,1.32694591063041,1.57889957787079,1.47114195912653,0.822564005442031,0.736997675468273,0.579572065973759,0.409244162861262,0.39748985519246,0.312035369077137,0.287838520612472,0.159796099734711,0.154833493624717,0.223921123785442,0.089664991149054,0.123930624920182,0.0664359505183395,0.0469909940600347,0.0644736122917615,0.0373206121780781,0.0707009801346522,0.16704527150833,0.0418064764584265,0.119870268314178,0.244185320994267,0.690101187461683,0.690488264736845,0.625417289473508,0.60118697328949,0.631025613676073,0.67707214813248,0.34902461280694,0.392448788426285,0.887335813104396,0.832739454148799,0.181555285021977,0.266118664156417,0.0741260867114891,0.0431874735740697,0.109850283396605,0.0835785095221816,0.173566929196744,0.195218452452619,0.131673667631167,0.00665301799793069,0.181799969896685,0.705028989541831,0.921540965528454,0.614769023862888,0.698482602497102,0.793910007554857,0.959837357464374,0.747879991990262,0.611958049502583,0.463839908191901,0.127617238692557,0.261409513151035,0.135251804661023,0.0832077169883704,0.147208586621887,0.115736317560224,0.0647218031639915,0.0220638461205784,0.203564036013389,0.147594003199003,0.0881721485524975,0.0578271625652739,0.016608444372338,0.0807034206664218,0.116192483553867,0.0647355212515877,0.0860046414189811,0.208318571024417,0.608599083631067,0.71032143611511,0.629118335434338,0.756325582839841,0.689264625711118,0.663535522724033,0.396833656851337,0.523198989864962,0.153207974062381,0.146275145791855,0.0391781062452036,0.115814926673799,0.112312102066409,0.129385937308149,0.198265764671138,0.108115305237329,0.311100037922041,0.0671685984220183,0.087001738178247,0.116833191642723,0.217543154922272,0.165391357908323,0.389145883556616,0.188700094336008,0.810157870115122,0.821686010825555,0.876590361298097,1.01896566889834,1.1011955429443,0.634040019827552,0.253428357054207,0.22184250018905,0.198306732092711,0.151503780691085,0.230439548816226,0.104689452858714,0.168991988090857,0.0928297548806508,0.0815601576240566,0.111200376725466,0.131197761568599,0.0419885590891461,0.171780655732842,0.139838778518861,0.112811899519482,0.123749958755699,0.0595487950237635,0.286601958581018,0.726425733348806,0.855339997113435,0.678503352897077,0.78458501665397,0.835716839982493,0.751081048937268,0.964079583718533,0.412851171885806,0.310221311384085,0.144666421374342,0.0510097671095724,0.114627922274439,0.125883545601401,0.136886704136492,0.10563474235386,0.436989989966678,0.0392862321732321,0.240363624178156,0.249941127785975,0.28897516250828,0.426462648639721,0.172216589994749,0.387157266532328,0.343952312073717,1.24699343964008,1.93061882439428,2.07448911280869,1.47544346172391,0.856978103880862,1.24116028464303,0.527194663994103,0.734499072889084,0.324434445612195,0.326249908679902,0.14948674460861,0.125042728903858,0.102593608212276,0.277388311958548,0.215127798979794,0.309375217827346,0.257830831059176,0.292423754717912,0.229237829625199,0.335223537914275,0.170338705421519,0.216509778492999,0.121969536104165,0.290235513406092,0.768026019147911,0.664931473636951,0.690066915763974,0.729925010202762,0.969511835914154,0.813220724847946,0.455430916874143,0.918452721824886,1.05303932339363,0.177357130522644,0.985548717867563,0.365425199170411,0.192280699717563,0.21745959851849,0.313394514813783,0.282891826999004,0.462092408785242,0.21313013386194,0.221552928602236,0.406833034811814,0.131718235190249,0.199894718780107,0.330173040815767,0.334438198339537,0.768373688377274,0.893816058718065,0.947174469056636,0.83615451332319,0.866853392615303,0.548665970266216,0.194424077392417,0.276336604182296,0.178609751574061,0.141646978725388,0.111329345744893,0.160957621950518,0.145747407819343,0
"41",0.154139124931452,0.0430897845301064,0.0407115667311321,0.182540604790169,0.0738789298025101,0.089417938177896,0.105348399002875,0.0985527577547693,0.116075134977788,0.0860338259911498,0.239292369133302,0.136573385431549,0.130725310155981,0.0641896593147261,0.129062423693031,0.0711096466826905,0.122672969073733,0.142357926684091,0.341629480715721,0.393311599050889,0.240127094500931,0.246222735038151,0.0929859658534965,0.300848133834316,1.25117924892872,1.17302665383591,0.736123773284659,0.998381838562069,1.63330340332259,3.18793452376951,1.81714075724866,0.608690079241635,0.659990017581083,1.84021756131659,1.69132678090845,0.654777381154702,1.69529578022747,0.230232110775226,1.53775216232545,0.335727698969744,1.10601373151642,0.321468998705526,1.20325112424377,0.148417305152826,0.179909884940294,0.176383190969304,0.143796775595228,0.099047326519756,0.0808039078716854,0.0159687267732869,0.157661374584056,0.171808956211425,0.108648112145297,0.0879789576200866,0.0631663239515391,0.103616904337091,0.0576642091527973,0,0.0848713996885929,0.116893877715811,0.181828588678532,0.0506401852809869,0.367597934365425,0.167343071389326,0.288238483200004,0.168429199903887,0.167041286110103,0.780795041656817,0.5322833219236,0.286955180622324,0.209741974802593,0.274529478901911,0.0696530615732991,0.0202185148610945,0.0844816448795117,0.0846962412045858,0.194468012511251,0.260059840677279,0.338607370162895,0.152283002661788,0.315781051139819,1.2369053523232,1.09237601160776,1.7579207914599,1.26423223236888,0.280671552556068,1.23841724842399,0.707904040897398,0.648482061493244,1.99377231608549,0.780705871469866,1.11343539965056,1.08411333535216,1.25540291788078,1.07110990168667,0.948595987844162,0.148087922539787,0.112435266236858,0.192331968445912,0.103141412663796,0.0389327417565214,0.15377089816891,0.0644952508858875,0.162853499593215,0.127258048399992,0.115099010649496,0.15709978398909,0,0.0416218258883455,0.2847385080919,0.144401566046893,0.145697973571641,0.163011972425061,0.205980879303341,0.127812117624777,0.199649387012008,0.148610540690412,0.369184390247886,0.339156907347459,0.203866504295031,0.91892676774627,0.671742459606354,0.902351384533684,0.260131661711957,0.147012602490072,0.269555461892257,0.262983289836561,0.316086490827722,0.155524479774381,0.377357446826532,1.79750791454689,0.680730527692822,0.525937939171937,2.37735260719458,0.756796593211138,0.918621047223514,0.13329160275773,0.150894620717754,0.237947086596544,0.123687026036243,0.0671158778577039,0.190477565514676,0.121388396564796,0.128678024568806,0.0320631206928443,0.051536930257635,0.0618395797790774,0.212031567267117,0.0190064330255408,0.140244748602958,0.0378384655856467,0.0636791051691152,0.164068939216148,0.0546939740252308,0.024582949895623,0.106530273655815,0.0472875090327297,0.0756529349762648,0.133913513393091,0.110146061795886,0.0272206261857121,0.263110425164752,0.177602244454116,0.187341368073031,0.280942285190364,0.17401530254478,0.170365637099834,0.31957703396092,0.194282058997729,0.231857499006373,0.155372233360566,0.0954699156577637,0.374732515427552,0.292574840455221,0.171330261703524,0.234190362980175,0.276784240995343,1.89469494242643,2.49518737143323,0.824583195067575,0.662092345272796,0.269325360691369,0.717147691927178,0.210009634856779,0.303535045132406,0.306328223522173,0.143028702536557,0.0785172375645474,0.184362951887183,0.145582216545294,0.204320076994201,0.188285930661271,0.0894316297406401,0.0800621322949274,0.151762586808225,0.170232934349614,0.0773876183363884,0.0321311509450438,0.178574055738,0.100274345203025,0.145124572097162,0.16068057472452,0.0635512628477585,0.0853855243306217,0.0912414211485692,0.112531799018534,0.15709801747005,0.176987553642115,0.306612676132391,0,0.0900636641911594,0,0.0842170391507295,0,0.0524716744142297,0,0.0746827311365328,0.113939935814712,0.371526268092966,0.26168875670673,0.171935601373624,0.137079104566251,0.416181980478124,0.17531229659553,0.208696050314947,0.310934293777665,0.105826275432811,0.0927044912114865,0.0850568705348125,0.148495845085934,0.120107409132479,0.113724682814958,0.107147986826049,0.0868123890617153,2.06595625959621,2.21275166907748,0.314227689827834,0.102359667093546,0.185298832930399,0.10593661297461,0,0.169488618413137,0.0702135574639617,0.123200025642366,0.101457616497047,0.0807437919707503,0.140592712733662,0.138726299538929,0.124797768778603,0.108528051388316,0.150226844083467,0.0437058907483501,0.137006987709171,0.101299030247669,0.17535907243424,0.171082492264174,0.127988966688473,0.313443101713603,0.357670865231295,0.260462125155674,0.255367064526973,0.658928357750902,0.798163769445286,0.0539522616930992,0.172594830255414,0.219830058440025,0.156346879850268,0.812797078107235,1.57943055125332,0.327719065461166,1.24977758333957,2.27041093355424,1.58137411529928,0.401142199217111,0.805211071253362,1.60935148581771,1.05664100684477,0.304289442574557,0.327772423214014,0.21762737565656,0.157744225840754,0.16651812128526,0.109494754687975,0.173277985177802,0.155553506552237,0.106186444046918,0.044386664846374,0.0434994222347047,0.129142889317516,0.0404682954115085,0.041046704586808,0.134216740428588,0.087366122715791,0.10888632189801,0.233168053861818,0.115815149444946,0.0646628782848737,0.040530049046098,0.111612735965995,0.21025239883762,0.132132572552185,0.200632997703882,0.0643493524231401,0.016459199219284,0.0797748077625553,0.291943767799569,0.0690235894576104,0.245158955942745,0.245911420683745,0.0988892556471456,0.308970646483851,0.135670386482169,0.0701621178085232,0.0640563528469483,0.040389311775593,0.0672885342266844,0.120000102952772,0.092699029015682,0.18407866516558,0.21890340656155,0.206496943299985,0.165115257701857,0.244444916150959,1.98573146430289,2.26720120378416,1.11014977910791,1.04587401875841,1.30077871762808,0.824806298666978,1.18530483996988,0.731833281468274,0.255281223417688,0.18997713167717,0.179880188046534,0.193112955886819,0.0580792992067711,0.150884507032284,0.120492447595609,0.119882063723292,0.13262464271659,0.115889714521108,0.083942869193176,0.0836500210512196,0.173394881531096,0.0268859981763651,0.178336045492502,0.117019763898286,0.0884609260165907,0.145799812974509,0.116905290645589,0.0806321061053961,0.182419880360587,0.154092651556141,0.118307128611202,0.171350238532944,0.14205987800897,0.337920496471537,0.131780513835452,0.161407746955504,0.121007201554974,0.0980241294840927,0.0489755172095842,0.234486763732412,0.0711855229083323,0.146213369412658,0.111849942732031,0.0980595592761996,0,0.102814959580827,0.113884632643053,0.479807264510495,0.563333262234508,0.213014324645109,0.317243806936122,0.316205225550244,0.257036374651928,0.539286123498084,1.14271902718632,0.840812058758095,0.117794840209251,0.16751254122482,0.158984361448875,0.196415388272248,0.129054659597138,0.0802349370064718,0.258865034336989,0.072523600101888,0.0839206583187501,0.122976219286336,0.0941810467109015,0.132905578902585,0.148750588272742,0.184226159250065,0.00769279567606397,0.0886616630453317,0.159103914229808,0.106160985687837,0.0876981240050209,0.441717816361522,0.258761057411135,0.108938196873341,0.267577254941012,0.105494047412096,0.201176721936589,0.241697701765235,0.197155796959562,0,0.0150467836869312,0.0564161251488386,0.0916381795075443,0.0886325457282687,0.370336247255047,0.204513991241857,0.224349574031383,0.164751815789154,0.29449102394672,0.252053580980565,1.099985175959,1.10853683490858,0.925575512871236,1.14879423536637,0.476180483245932,0.229690593713425,0.201430284686125,0.302961981155144,0.226439530636483,0.120313492907126,0.112286605206316,0.0772132737238525,0.087533219082932,0.0579220777987688,0.12444207728597,0.0994597475892288,0.0681677257909679,0.122769487854967,0.159353070489433,0.130386721581746,0.144360552394399,0.118869741250542,0.0282745395576768,0.105691056866678,0.063768067017739,0.159399487118375,0.0960140777503674,0.109019243820743,0.226650007008312,0.192888188368198,0.141211565268858,0.223306789602191,0.218930011133907,0.114737593112558,0.192050574106263,0.264412224080865,0.276440169883223,0.292530851013847,0.191394786855398,0.183670934148923,0.277388807009138,0.369208801129229,0.271343566160054,0.277124847420709,0.322770263787025,0.187776134561275,0.327917296977099,0.502937992622996,2.47398903617185,1.00039329127612,0.25150853943278,1.16588010463855,0.245181554742988,1.32936716415016,0.469947021644366,0.536392821332404,2.22401740710578,0.199444724291852,0.237481257463784,0.286668021243177,0.15498564508079,0.137741237242882,0.129268741382144,0.113222649879558,0.132963416504167,0.0849640446249344,0.0723350281967507,0.0839441327079103,0.139533936371611,0.0843968604799127,0.181950753252209,0.0614247433454159,0.150005268656402,0.102031656934621,0.130375708573021,0.188039007202397,0.11708739167388,0.125019906805814,0.0927898138637182,0.109946769988395,0.292432302019632,0.150552622428177,0.0837770717542855,0.108611729249605,0.171873894171516,0.164885453536409,0.112389959569911,0.139574898223199,0.0879133462007516,0.164007445216866,0.212350628367307,0.249735372247489,0.189730349503421,0.130304055330821,0.143486874452107,0.339066347654375,0.0614140705384969,0.986584360975132,2.41888414881445,2.15367804210229,0.471866401380449,0.860226134244562,0.383774659361479,0.310540340927808,0.355966032419354,0.72745788083082,0.849074895224475,0.952362021904827,0.442233846109831,0.479736341544458,0.152613139664934,0.137794191763667,0.244163850762831,0.0616683444709484,0.0650791977600686,0.121821043714014,0.12575639292045,0.093980872750907,0.108140673456729,0.0519726554114161,0.0470925133570204,0.115353754053837,0.0702721110972701,0.355620409353091,0.152727478372906,0.0731045993856441,0.144985362504622,0.233037768561404,0.160565731875147,0.114060425347125,0.268809944219117,0.196866212445975,0.187078220624323,0.121505940266199,0.265838401464044,0.131607012606839,0.232918447522365,0.13812572084642,0.121781534507608,0.0678839724808835,0.227063102155866,0.213680034428158,0.124527526609157,0.24359706503749,0.218678208959627,0.222362415053567,0.184186975594855,0.103561112522921,0.195325713355604,0.172213187568484,0.0641065242215807,0.204752565766337,0.181329059229739,0.0847136369752093,0.0199561613986683,0.162392856750576,0.197225165262628,0.202660560941674,0.181466615016484,0,0.065275959371089,0.148961331165935,0.217323998912634,0.108300041752318,0.123837347757488,0.172287271682248,0.218124898433138,0.00200403657348638,0.115136597178463,0.177328007476608,0.171069266061447,0.104206713994016,0.0989804213888612,0.169546721235126,0.136574323345235,0.075897157303814,0.160301199732761,0.210206971748109,0.537411583709293,0.201178876113641,0.320386400565677,0.194377257132773,0.343086038637428,0.166879824360407,0.27793942192796,0.208706900264617,0.0877496918109873,0.160876123010566,0.519787144377853,0.384415805505492,0.224854311906764,0.114396066019263,0.166879225694716,0.186127870063931,0.23716383957026,1.87821358911516,2.62832317493026,0.536276234224886,0.188934850003821,0.331976779391343,0.374135833675149,0.232056202880372,0.123945573661211,0.168604454292229,0.182514802446766,0.080280518775397,0,0.122524597112988,0.130070980496581,0.0531024628762035,0.0639322022329134,0.0978459883547783,0.120502635446031,0.127367826240608,0.114256308886675,0.0132688499711149,0.088508333561231,0.102279186927207,0.148507683108599,0.179420264780451,0.163551127759078,0.0826667821693207,0.154228712201401,0.235514427332815,0.227178492667185,0.25801074220489,0.255964279076686,0.28753659797327,0.48182369077435,0.17236189683717,0.255127412700904,0.371492560294514,0.150004413262072,0.191256556697668,0.222883404729782,0.631956138005799,0.214037316368993,0.103302843622949,0.20603117033066,0.161311836259165,0.315325587806203,0.181167142322938,0.225871997769772,0.0996134563107678,0.00997218527025236,0.186725015865841,0.0731672912230998,0.168167891791336,0.107783944747796,0.093846490126662,1.10942504589015,2.18784173486065,0.455164228579825,0.269692887482,0.229991028538328
"42",0.139016182996474,0.0137610319964133,0.173008833337816,0.0931257454203985,0.161710457779255,0.233200625586751,0.193727394641887,0.14876146217091,0.064072254209935,0.336321943093887,0.231685502303754,0.427739513182835,0.15016703305957,0.520075704856397,0.6777172959407,0.387727903927249,0.232136904863629,0.18365728576956,0.154583766791722,0.0693820514457467,0.121687326031115,0.127271977535718,0.16613150055286,0.0999430366253929,0.15756617245503,0.244907625774845,0.146361198652544,0.0857095273738702,0.104902936443677,0.140151064823017,0.0756254794602583,0.120986771561466,0.130633377392796,0.420358909865938,0.239665724107433,0.477741293229794,0.588205396157846,0.535298157483685,0.292090559487519,0.34322591722635,0.157910686559544,0.507200151834699,0.354191684481444,0.387514111152407,0.320037672297885,0.516220668717511,0.252778438371106,0.274814568719019,0.0601078286112142,0.137801774684845,0.167778491424427,0.243345629911041,0.114269373062442,0.200938461726967,0.0628865619919804,0.0986347873589424,0.162151480750296,0.222736803933659,0.197820283910599,0.117906740402423,0.40519166762246,0.0836113093578706,0.146561105995164,0.198970755605158,0.195476245887228,0.178265411017285,0.0127183319940872,0.300238821300473,0.815226606719026,0.930123097526411,0.52194463699896,0.924202278457767,0.377482996903747,0.842866059875205,0.406528425770128,0.0913238653263997,0.2312216948821,0.190074136755042,0.0584734278828027,0.326888098882938,0.368039667333711,0.528366211424017,0.484501181645562,0.932676810144353,1.84487269793774,1.47020110219972,1.21535414683786,0.921283622648186,0.977784295596522,0.349172809600203,0.409854143101242,0.55982827476424,0.438340547723163,0.335473817398371,0.412338922893502,0.296839456821678,0.310043862498066,0.163141464850758,0.182483728431144,0.18676605042018,0.159962471862213,0.00795091049003895,0.150767888162415,0.162650234599727,0.132774447804327,0.0681721442307274,0.0427641560280918,0.197840995467883,0.22935284273332,0.31773830221016,0.24991407851934,0.255201436707001,0.156192991000942,0.366289317706851,0.437085195307618,0.365513783484389,0.403983852790632,0.403807174948946,0.970638408584923,0.737187343798264,0.900335463915677,0.700574306733144,0.386244098496909,0.598754186993076,0.621331943469547,0.607826233602135,0.385352259584636,0.514490970989101,1.06689608774403,0.429742127500343,0.973746603428685,0.364425359446489,0.354720891791167,0.298532556039361,0.262347051333493,0.608325111305662,0.817564242082165,0.758259940450714,0.819674761272334,0.367702676265161,0.472639900029227,0.335427126996139,0.489780165148566,0.190875088492945,0.153292370316764,0.158075745616269,0.120052991735187,0.0805157839863069,0.148268297648744,0.331391913438566,0.258099216202035,0.220474897696782,0.0724011985158442,0.134392760501794,0.121718115420659,0,0.108905927637671,0.292840148812184,0.447385265126215,0.111245720538389,0.0637045832374301,0.353058215611241,0.313106997208882,0.306681350581656,0.454330085092365,0.303594523922513,0.256288776452588,0.192024706695302,0.260623745877575,0.259712297263727,0.354445419789312,0.175068224381843,0.0766713160059967,0.123306318107716,0.160822680242301,0.0569356638912114,0.0864908117490056,0.238464790497118,0.351295743295267,0.514617196967427,0.374444989828546,0.946410665521188,0.551183355947893,0.254461065728367,0.290722049437394,0.401243240260723,0.21871021795404,0.348165195405728,0.327791860018969,0.341616251438437,0.137830777367843,0.197301804633272,0.207852001303578,0.257732762605974,0.226246714765228,0.243846079889624,0.0986854251263056,0.235158961625181,0.0346838180710232,0.121302238479346,0.0657198390139704,0.140106529959579,0.116075233883922,0.0744041888891011,0.131289351358041,1.48132190315206,1.21007928648424,0.431473856486185,0.608243121845413,0.270578722088466,0.454443883719576,0.245777100383774,0.0847336108856603,0.0843146053623754,0.118259821823075,0.124676036209762,0.134870391427844,0.185272919498581,0.192111405701612,0.204695593352813,0.213087879610607,0.287140809876705,0.411988275944904,0.155586859686571,0.0817227976116111,0.185462621379191,0.183878268492761,0.674679482935634,1.07197291828937,0.925449069985599,0.42468659721726,0.470457585791935,1.05083627635465,1.00451643403043,0.526548066326339,1.11733076122909,0.584240351653599,0.414255637703823,0.241115233551323,0.138832372484982,0.1824497398358,0.112846505799618,0.168797716896451,0.142035784169216,0.127181955118334,0.166844990714331,0.205002632267883,0.271506385646763,0.114273725359009,0.0702369035688088,0.239438584125911,0.170904220960163,0.432313036364405,1.33072779127753,1.69291437531161,1.01477050677457,0.948811189371273,1.12699548569589,0.374521635363416,0.26898671290594,0.167153482247536,0.213779107388341,0.400706789015343,0.0997594619121417,0.243435710469292,0.201225076525098,0.290021950050653,0.227120065182784,0.119277241624506,0.198012029753046,0.175001363168968,0.246428168856858,0.173543029310004,0.241130485882681,0.217523631591157,0.0973126969883599,0.182299643165313,0.132469004033129,0.200224462100444,0.12148245804223,0.140062367396386,0.149854879630193,0.0803630721601032,0.273471862612703,0.173216875932937,0.214876924111087,0.377868724108524,0.334201508631724,0.333686839944016,0.323120705358568,0.236210585556199,0.262837489436593,0.170625965806308,0.0507052808180655,0.181808330917352,0.0785290822111989,0.175702262930783,0.0635633977119279,0.135773838240188,0.601578061019502,0.44618918536036,0.379621190631279,0.326433035521056,0.151580372802174,0.0836948069033267,0.0662430096004368,0.0896647224559603,0.193360481829605,0.107897360141988,0.0644837195082463,0.255160234024767,0.145028408985718,0.190707034320518,0.107018029311625,0.21094473412675,0.270784617415435,0.172355008726079,0.185990035023922,0.0633292283835484,0.104637305988936,0.0918047114248801,0.289647839412675,1.25008324021234,0.882836084604632,1.68386760890057,1.34511586936509,0.463363125481299,0.307732349320209,0.388237530846743,0.332092552924235,0.667110290025358,0.346433713496503,0.365981864444697,0.503656539992937,0.66598010490252,0.403252420702721,0.305834546358495,0.23404374134501,0.219433750987523,0.127852599482144,0.122527878299122,0.0797667480705184,0.177015267587913,0.134440509451404,0.120772895534361,0.256660859647067,0.238840995273697,0.303700295714418,0.476879096370767,1.18477558557097,1.22461727846782,0.287590273276166,0.258637791056487,0.0928272056148719,0.131491380779507,0.0642249163658898,0.05281118830602,0.262174460574004,0.159230934640383,0.925430735656084,0.86108093068042,0.374787758512207,0.628932391070228,0.125666393616815,0.224770521735655,0.078388908473973,0.116662144341769,0.0999192275074802,0.232239950249933,0.320317384951993,0.451182714066135,0.445040725127919,0.264975081318589,0.594042006822987,0.530048896862127,0.273354645755293,0.291990366925724,0.221332498866713,0.354633560845251,0.274587655971399,0.340765086309487,0.271396258930517,0.225769316384969,0.29699075333206,0.105312314677711,0.129999742372144,0.0824707808766579,0.191355921253296,0.117359190331625,0.110618450507325,0.271916832108281,0.1702905594766,0.0238552649517157,0.249678940151435,0.169958710243143,0.121670695280405,0.200923933974591,0.237360174105256,0.350155317263303,0.447179838943266,0.372311235555075,0.274400712413451,0.105770248794267,0.249861530345575,0.387815731374013,0.176808963733119,0.505279051976242,0.600999480114832,0.206995991504625,0.173301627206397,0.0829776481712385,0.0573035193707403,0.21254329845361,0.631943455532288,0.0925131348794493,0.161196182553599,0.229313584006254,0.144256891882584,0.264144449884252,0.0762506846343609,0.401613578435505,0.420453472268082,0.39231136802489,0.356850838372246,0.274377366638535,0.264858995604194,0.185865029787132,0.295530096223577,0.0685527401447225,0.102084589356714,0.183725791399274,0.184070734177988,0.167908583065306,0.304552037304694,0.156770077823378,0.114059353556816,0.0580632343842944,0.117936260086142,0.223594977185841,0.160366501976942,0.2579654913303,0.203822774424742,0.201660067500931,0.22459348261894,0.338847664481236,0.248773199663416,0.278441137386073,0.146927012443252,0.220165535949923,0.109770008598337,0.15183189708306,0.196692595525751,0.394888293127097,0.615315627957916,0.301274539912684,0.320881808277156,0.318530342386916,0.190144924955589,0.448709512464433,0.874208140418436,0.511817410463815,0.33926267673715,0.188218884667605,0.225335028563942,0.208802383000237,0.271583620176218,0.631228225592075,0.520691768561555,0.462113860118492,0.409957679598472,0.322185087772452,0.382852153616767,0.239799503082857,0.255240240693737,0.332670105598587,0.351065709307094,0.522477483528248,0.216625635742192,0.325818492958585,0.407662292382783,0.350060792958189,0.13810383393416,0.0931083555193219,0.181940805351915,0.0284322901660459,0.0270271417664259,0.18026570064365,0.192417024033804,0.0915189828834558,0.116262633930596,0.482858584895137,0.239489409830038,0.33221807993598,0.607590024123036,0.858161906516123,0.389687828702627,0.342355750751266,0.305970932240937,0.287136100784295,0.386481275121805,0.442327758663834,0.586603586845912,0.269177992771557,0.409255274422144,0.275862298411986,0.13911922472627,0.205786893039671,0.152908875857999,0.194524060107282,0.468050934095201,0.239113810834992,0.196016186661653,0.18993868816012,0.247238757906108,0.472027158938842,0.407315713004139,0.467955073256729,0.30343531240357,0.348332646343715,0.573003821703263,0.417787226218254,0.533982593749699,0.670749769978842,0.575419442819725,0.451741247125593,0.323843848148137,0.118096149744887,0.128581573501547,0.0553437928503063,0.129290518067116,0.139654632218088,0.146505756450023,0.17599634597809,0.14811795644226,0.138056108404031,0.0712460600443927,0.104875415171159,0.200755439461821,0.216288398044412,0.217620672991831,0.049128549325884,0.55084694885107,0.619749355255495,0.183595675691961,0.168735359311565,0.383120897650539,0.348261219119731,0.415324383798774,0.346876340491262,0.342149353881112,0.468077594514871,0.244106664757638,0.233901203833685,0.609966687294694,0.340973636708566,0.166709712249107,0.0960074017638316,0.191834625637852,0.0554188141255661,0.156331975476279,0.173534474576873,0.165134375582996,0.20137267082276,0.266466749950579,0.540351901300414,1.08144898109315,1.1344798633504,0.754351615019263,0.9498538476586,0.542479722464898,0.946233350690551,0.283124677492812,0.532910077622138,0.540321300077181,0.410738952157301,0.316607703349809,0.223671622544267,0.426615778836465,0.297581403364712,0.254526564464497,0.220445201844283,0.246234398829499,0.132713249606386,0.165827623907494,0.195378616029339,0.0575680449494695,0.0893455535041832,0.202256600630205,0.195823428944221,0.482324910232554,0.811233150913979,0.0650351286316164,0.146900580294243,0.112267193796701,0.3461103819242,0.148124414566613,0.0342406396677505,0.279620274355662,0.164356806842704,0.195694961426226,0.129908337933471,0.148180834421959,0.317401016534769,0.306099577564218,0.378037385381407,0.392593798640523,0.352571015958001,0.296489933745766,0.282963417577528,0.292634786535163,0.197656297603166,0.334462419032792,0.832526854462793,1.16629409672991,1.05073573055152,0.510437162027682,0.630298067208789,0.228604196092327,0.460728854971148,0.695931669143906,0.298841869230024,0.236627547640729,0.116718096576487,0.290658352237263,0.325662562617048,0.20064574105474,0.178829093753988,0.22381235567878,0.139175716873383,0.191492693341063,0.214371975623141,0.157328398219363,0.120145757305209,0.125423956270634,0.179264861081915,0.420719355279191,0.386895936896866,0.921203649067501,1.61163204037269,0.994462086605845,0.755064602249163,0.309515296845719,0.284116224962947,0.167311428309617,0.230314692126631,0.201189345709915,0.142814180691574,0.128823593868484,0,0.0855262086262451,0.124128664665235,0.118013352416889,0.240659942689754,0.158823711605984,0.432674377130106,0.19159237871915,0.239575403232411,0.047295063695405,0.0625766975925073,0.486533259929358,0.417982831744882,0.287862436140085,0.354340769644855,0.432172738610855,0.446967767543834,0.568631659048704,0.527617747727324,0.528163375608322,0.590993201633375,0.534154852255523,0.405116984469287,0.505091304354673
"43",0.795914099573275,0.049921241539752,0.0917010172352043,0.148346586127786,0.570650453991983,0.207232228660334,0.103488246285451,0.00661437004072245,0,0.0224693012043231,0.0530987394772997,0.144090623209159,0.683004611640062,0.172059424738083,0.0105511064712598,0.0701895587724248,0.0421232853995485,0.00342002235970712,0.105964094889703,0.0797922500808588,0.261189194044413,1.10627790036059,1.23747239048349,0.70235771672143,0.108791583481547,0.088919859416826,0,0.856317351029368,2.26104977717735,2.04735892665227,0.814772351400537,2.22543925736144,1.05938329508902,0.496843733556979,0.412932150411318,1.74503006792728,0.700214627289727,0.890457417856641,1.09525002982667,1.13236313405723,1.19203217231945,0.815431369681305,1.21327877085307,2.15323893970583,0.629588564578382,0.820225343861238,1.37291323149284,0.213731741453582,0.0421276622114303,0.0305943117909984,0.655762327828142,0.0331745025489244,0.0274311032246717,0.0526882377677271,0.0943687065349118,0.0192159609467685,0.675775737641364,0.157288001115374,0.144080942460831,0.0879596965503376,0.151361441956817,0.0670665955283609,0.0334115927758628,0.111801298003932,0.718424908213509,0,0.108959531508821,0.104962201533198,0.0382796667386184,0.0592237594273624,0.197602753962023,0.857344990981521,1.38488006447745,2.42503463487669,0.734209196314173,0.135483069780605,0.911461875661432,0.125861562012826,0.427261044693188,0.391259741859476,0.692077708062637,0.214029497924364,0.152996418239177,2.59346961005994,2.59868166231431,1.78408254790538,1.90394110928597,1.27236542218767,0.90340105109589,1.73355993435518,1.10197694472585,0.950044394652902,1.1272796647129,0.922322269597823,1.65244852368434,0.957903335071865,0.536812819572737,0.0840337028998209,0.0353787836023978,0.0131043206516678,0.620187021830446,0.00664535256154443,0.0681082104584061,0.0874906679695567,0.0168836216227101,0.0834566173722425,0.119688876284744,0.721830761472337,0.0915926073799996,0.0155961412665594,0.0460998188974986,0.0273648682817143,0.0420224375830841,0.0198607142135766,0.401774555250773,1.48563354415031,2.05240314848727,0.161240358066025,0.732278000907497,0.185260493979079,0.193780288862835,1.51285243857614,0.57439697955455,1.01464519218528,0.607123956956661,0.241743802704472,0.158916909534048,0.836272633801929,0.242127959659796,0.108124028400006,0.0439650471775424,1.03120946753505,0.790670310484966,1.35735666695466,1.16941515278262,0.890616708008511,0.483723207260265,1.11623361045072,0.511596663178616,1.02629610278332,0.423730069286606,0.0990296001015169,0.0911328264731569,0.0896658548318564,0.756656049379342,0.0267518225194559,0.0538570700451703,0,0.00308988061662728,0.0732232179121972,0.081301351289103,0.0667665614275267,0.57650335880963,0.0733874602598257,0.0269256449452701,0.02405500306895,0.0486124904936823,0.129827911047729,0.117694854380257,0.0814772924243294,0.0950830347088507,0.0413074078763842,0.510400949359976,0.191322621655439,0.0244720345049335,0.10525453394239,0.0711879980324421,0.123554205394875,1.43579733993617,2.42091821636702,0.232525354953666,1.23653201579939,1.59691146675391,1.13386379532057,0.114441416627711,0.615765537673439,0.337842993171068,0.33369155147906,0.759111600958556,0.760913345953998,1.64976609699506,0.651687638004044,1.31729976181818,0.560491958591572,0.71585395867796,0.708698106854806,0.611042659583232,1.19594142134914,1.37485538539285,0.38943357116652,0.851612459441933,0.511725227815929,0.123531901590123,0.0720302809414103,0,0.662978981652321,0.0265582361902938,0.103800678346425,0.129257303487411,0.123912903496097,0.0189298294916098,0.146563431935948,0.0804725439026232,0.654372335764702,0.145743017896069,0.0481095175262358,0.0594023668576488,0.0799558320460308,0.0935037210693537,0.0707619989485565,0.255382673878565,1.11375730802327,1.20822290270018,0.542406820034939,1.29720458502699,1.89236061380439,0.0245357535867203,0.788548482864659,0.0997185095107258,0.0680350590519931,0,0.670444349417216,0.140985122839566,0.0162107724574375,0.0091825387435613,0.103903578128385,0.985250392104326,0.245072142742885,0.153961980687491,0.263035090204558,0.267922254669169,1.54573500088799,1.68897126742857,1.39843595575695,0.599445656202758,0.808761920033933,0.455205488272247,1.17115936231041,0.344841157273269,0.0767885188245491,0.0795287408833923,0.149350418864415,0.134946049010254,0.68695647398238,0.0927711059764377,0.0277254452902982,0.112182137437297,0.000996837793545452,0.000707075054515394,0.138973231661731,0.0615136308600347,0.630526874990747,0.0600749989424401,0.0308348626805766,0.0802169450742601,0.079057278362425,0,0.0579581689075378,0.046422882698693,0.014652935969691,0.0471571474265338,1.24716855520561,0.579729652464913,0.307997758528727,0.112993193082552,0.983164082510176,0.170782701381545,0.483222284150695,0.683611321991361,0.0811200795971586,0.310110738407598,1.59759826792551,0.928672118248944,0.711661708692787,0.148669145063067,0.802987668742513,1.66426323187413,1.1157171315997,0.561658862612609,0.789573687729395,1.09691084212664,0.621060323756933,0.447026417706146,1.22488467689945,1.0074454221782,0.692668541552141,0.266653044353889,0.102066313777865,0.0696025458377806,0.665475989083247,0,0.0084541230474703,0.171813482380022,0,0.0731320824846668,0.0148626806087431,0.690951971448422,0.057668072669902,0.0268095640982932,0.0260609440904802,0.00948798056995982,0.0474932670948678,0.159189591143045,0.0277238054462264,0.0626627170562844,1.1217384830022,0.693545796037601,0.76949424561479,0.106563564870191,0,0.112829242560125,0.821142477827435,0.0680808563093534,0.108735588227735,1.4659745956932,2.23294703169597,0.0785720845782307,0.302829459686168,0.433770134670591,0.170620224234812,0.900814088797194,0.026123359044916,0.114329017650554,1.23966510863629,1.98214074423345,0.955184531209556,1.43532285342795,0.440879903664529,1.33000885683146,0.761632386661419,1.78217333670835,0.9599497914782,0.45948054904239,0.735266360912722,1.13094336590581,0.0499176253809545,0.0234407404441691,0.0951516250203132,0.0544482345300659,0.0623426054571848,0.556466035726771,0.1057281015096,0.0575281893661604,0.0678061211523915,0.011128497561623,0,0.0958196299804874,0.0930989642684462,0.670638336363424,0.0356436647472306,0.13780957692901,0.0527632946181821,0.0935016027179464,0.087306829492706,0.0374704201817094,0.757458467632093,0.0743644932870096,0.0674569887343767,0.0382842879737647,0.0257925880816409,0.863315263875669,0.346989316020928,0.734260156619079,0.0601392887914511,0.131192537907154,0.215906698795056,0.177651226291675,0.6525387930191,0.144701925493122,0.211388442829666,0.238408263352449,1.01446427958974,1.62355847575408,0.757339520624233,0.472940397346552,1.40273325461875,0.497005554387532,0.602814641530603,0.646970369336383,0.624731467570205,0.430427067494519,0.705131713688961,0.283332576379221,0.107579729508657,0.0964694636365532,0.102905785412483,0,0.0744698568097973,0.636722346533746,0.0194905560810666,0.0525757034779147,0.0131407399267498,0,0.0643126956119019,0.0178675613705715,0.0569944580930662,0.0113402239713503,0.699412803940943,0,0.0959495389585832,0.040458289834522,0.0164877887465756,0.0616134135206646,0.435354559427704,0.517710104039038,1.86074286342831,0.609215040513891,0.238278729141406,0.404461966356415,0.594581465302534,0.270270955689901,0.0532470913094431,0.147998058504008,0.766681420517001,0.125702070556299,0.0117323847812685,0.0768109519500033,0.0504976205979686,0.365070169362815,0.770893686144468,0.0424743688221463,0.821635320984824,0.311003595774308,0.0707904140651904,2.00081201293803,3.50306351861352,1.63447700139365,0.683810058543003,0.684358280060892,0.213962338459857,1.06603138366768,1.28880838956101,0.631166068221233,0.102232730223078,0.11501965779968,0.740344130328436,0.110295519694106,0.0944593939153264,0,0.0690823857512347,0.167349739796319,0.0724368064114772,0.57603102318742,0,0.0803154120694706,0.00710685982878156,0.0546447158329205,0.11507328166651,0.171376360384135,0.100443586252682,0.0440060071556901,0.00720497249825978,0.624471711989934,0.00817020185503502,0.128653981067794,0.0908701268360476,0.107516084310027,0.112830451301372,0.645214463361184,0.077746350705749,0.127206564399217,0.0223072177355812,0.304268969770629,0.089592887963043,0.10475760299691,0.63975617041013,0.509658786961067,0.639030028423033,0.169255536308635,0.193484774903696,0.210626933580355,0.673383257077752,0.107959926390161,0.230268694380206,1.87139992864726,1.2868547364755,0.322178173994282,0.0678606007784816,0.769540074748741,0.0761823104174231,0.0658545309498232,0.0547223339387326,0.0712019085592751,0.197590206775569,0.0270691234275639,0.0783959011334099,0.123995881955039,0.798532167581901,0.0658310028648322,0.197886914101812,0.227647808780897,0,0.180222202877178,0.00740357571767895,0.0120873038271287,0.0763102882584621,0.0897945476631236,0.0926690173986811,0.136172804440515,0.690560072208557,0.0385933671282072,0.0231297425744781,1.79510812850461,1.49244547022213,0.810249938299907,0.987043796876243,0.0220472108305056,0.217795525355465,0.669544388773668,1.37515387356697,0.770633160702737,0.657311042839253,0.558804565590193,0.615864641296634,1.42998710159357,1.02496746324559,1.00210048439251,1.62440895278941,0.986130574649833,2.24751741144129,2.00627958123232,1.05373478896062,1.61324833503814,0.712464953414789,0.692801400294497,1.07144143361442,0.538790340694782,0.568271909181714,0.541350301479407,0.158567632510375,0.618209685666554,0.0424602064787444,0.0302652719523428,0,0.121657439323892,0.0911370856387125,0.145487150900015,0.701560908354502,0.0369540997786409,0.0300504898108958,0.0316750592091735,0.0200478233828743,0.121198601663062,0.0445414072239877,0.0772928689637655,0.0489878011151652,0.123488300916696,0.747375937438823,0.148785557487263,0.675954919092227,0.501778259582692,0.852296496972032,1.14527702081242,1.62643239531759,0.041360036379306,0.639339082220536,0.0768990707840578,0.055766948901453,0.0980343508690064,0.77203463815786,0,0.0469371865693177,0.497458437177042,3.28912364414011,1.15672153202469,1.53928796232483,0.796962448644478,1.18878923823372,0.801784399520375,0.822707182880112,0.354039152009243,0.457782972173906,0.406693668098395,0.423436562431935,0.758840636768058,0.615157220553347,0.113587255987107,0.186783961657378,0.0361909423403559,0.156438838577572,0.0091625960022436,0.127470160788866,0.628701278342338,0.0616667763064692,0,0.0366356618382315,0.129438769644902,0.086914003085878,0.16895580666456,0,0.0574223580813852,0.623090862338105,0.0934742304248098,0.0861357465614757,0.00961285438619494,0.140286461246837,0.183671199729585,0.705391428354318,0.105116662432096,0.0716352815542137,0.117484402620558,0.145932112346512,0.945726704654588,0.908675643288706,0.323741291851112,0.0914817108772917,0.751945253296898,0.192805628540391,0.148209175479325,0.290419275483484,0.886314173675015,0.16562257036959,0.134626267735799,0.171690061532441,0.727081268017835,1.74496914028225,3.44923808435793,1.24571357177963,0.804197079097741,1.93062154403618,1.13998418423666,0.705800253981969,0.813969967722697,1.26814573456035,0.14675311767958,0.117574066219736,0.0269461593430062,0.0733982150372985,0.729065281750862,0.0699010167769783,0.0495501707526074,0.0989513201378281,0.114369857273122,0.109764731068644,0.124083155275745,0.590540969517665,0.175237689054579,0,0.0176190738292602,0.0622026401079269,0.0685005832768663,0.0144685670020421,0.0786012721386419,0,0.860777129447967,0.163178337648954,0.7834522070251,0.142281259199312,0.61080493666025,1.69103515517927,0.335973907579096,0.0925818890249369,0.333937392495357,0.723868666368007,0.123383950841233,0.074064265107272,0.122432130991044,0.227433920634603,0.793741662467279,0.114510354843443,0.566636780677615,1.52159424791345,2.69746572547055,2.64253026811204,0.939171311802475,1.58431019874502,1.04309992740589,1.04159283641061,1.49271549296309,1.52085787279343,0.90716966248029,0.579056259968316
"44",0.0256147987904852,0.0591892174072654,0.0224641943476614,0.0590547227721919,0.0666459369126346,0.0551964765368581,0.0737867930924584,0.0121463906026588,0.0981017761934456,8.04774363466487e-05,0.0599603153124046,0.0572385673939206,0.0200967153230128,0.0848901859794592,0.00571380319669882,0.0593158736322732,0.0299621876271289,0.1022329527407,0.10674556190212,0.087252358168915,0.0847097970727875,0.0693410126082625,0,0.100118847579582,0.116059851476326,0.0480541619287149,0.0114048064323478,0.0624408435309108,0.0927266774036264,0,0,0,0.0498448688966385,0,0,0.0482694153507581,0.0421080147922632,0.0478755544757348,0.0733416969233784,0.0888147460706866,0.0296896006580315,0.129802513155239,0,0.0362763172343891,0.0627158872090771,0.0199602540755942,0.0698513786515745,0.0352923176018361,0.0737940037729976,0.0582404625214915,0.0685144036035459,0.0568027708479818,0.0338204231705934,0.0434517723690591,0.18723324067668,0.0278866877944145,0,0.0437844334871813,0.0180875915575683,0.0211918312949915,0.0625614039914533,0,0,0.0723329544425631,0,0,0,0.0555635830586669,0,0.0786900436463718,0.0175390226266358,0.0317343626184957,0.112258721065641,0.212775851743267,0.0272400999267152,0,0.0468406364397974,0,0.0361352566319273,0.0307154286646558,0.0111060257654687,0,0.0490290754369339,0.0987274816789398,0.0959583314370808,0.00980856943222914,0.0646955154396692,0.0488094482531261,0.133856987571723,0.00311476463387887,0.000431634673983862,0,0.0140685103167687,0.126593117317735,0,0.134231310610027,0.0388646605177077,0.0341893743921421,0.0164546035260372,0,0.0729460310717592,0,0,0.116264724483046,0.0888269059753019,0,0.0395206656823533,0.0432941592994011,0.102034085554164,0,0.0364530084828165,0,0.208337769028612,0.102621190221339,0,0.00533464975432912,0.0842144859521654,0.0959622780512765,0.115617896854411,0.0818050028730174,0.0704740933535409,0,0.0146699328731057,0.0660223492053779,0.0188605632485329,0.0595002777974637,0.062653730233837,0,0,0,0.0607786526003999,0.0530601882036461,0,0.136577224694451,0,0,0,0.0203751908198893,0,0.0248292582794919,0.0164719024244946,0.130953894247836,0.0554153784712808,0.0322075102686895,0,0,0.0458972347816833,0.0966660124461338,0.0456187469740331,0.000988157312442568,0,0,0.0445755335016176,0.0147383931547608,0.00949175252458345,0.0825361449772443,0,0,0.0604789976291318,0.0544742812242185,0,0.0275995600692376,0.00171620552076738,0.108867760688162,0.0403203231527572,0.0250690579884348,0.0083883084893634,0.156037558212681,0.0352777407586773,0.029644444271448,0.110567501608583,0.0873693207905042,0.0502811262937424,0,0,0,0.0775089597816571,0.0998213158486968,0.0488077404534352,0.0182858782828352,0.0390818620122567,0.0965923254560233,0.0274190356764056,0,0.0635947650698362,0.131503997260072,0.0219904272031686,0.0634729617200218,0.0188038267470075,0.101600340244426,0.0286164009382319,0.0814245230007492,0,0.0491378757998275,0.100003839513117,0,0,0.0879873160487167,0.0714761639565022,0,0.128438111975105,0.0512476044387212,0.0851243001435765,0,0.0597648642167721,0.0571998796117051,0.0909978212974929,0.0505555135489461,0.0584008333313982,0.0527286674375264,0.0935615836127715,0.0377279381625367,0.0218422042407358,0.0583597357326313,0.0908517937272443,0.0831921183574563,0.147250833272418,0.0278577899620428,0.0119528869999489,0.0721800630062679,0.0881527562338774,0.0219875599346005,0.0484921288042224,0.000747261150364976,0.0777389487287452,0,0.0534406005522089,0.064018455927947,0.0149299807429664,0.0480510958614383,0.0185374270962383,0.0190695805601859,0.089121240276085,0.0297462794726135,0.116914558846991,0.100236066445871,0.0781827407824062,0.0357478982985542,0,0.0745684011962813,0.0725588106672643,0.0350485159774812,0,0.0113185116993532,0.0974948469610388,0.0286164211863984,0.0371650915674241,0.102085525394844,0.0559252805586958,0.0251876776324794,0.0640544157474146,0,0.137777836211939,0.0526705695240329,0.0013484576254408,0.0316074575054384,0.0606599893948839,0,0.0143467740950416,0,0.0634515976006277,0.0141718814745162,0,0,0.0484556604041714,0,0.0413499556039234,0,0.135219315046672,0.0359241699446993,0.0974551909107221,0,0.10488782188155,0.0650521750169116,0.0943184222960334,0.0137671414549623,0.0642484069780218,0,0.0400175265637266,0.0608580245474827,0.0183848290008029,0.0305829118956506,0.0426734118414322,0.0763944913793245,0.0462034665851715,0.010617024682418,0.0816360177331764,0.00772580671527826,0.0123921691975474,0.0636675497867473,0.0436648260918879,0,0,0.0351880388064952,0,0.084541960122739,0.0669264736553204,0.0711604820010699,0,0.0486037719213792,0,0,0,0.0591228542625168,0.114478731471069,0.0506046106001115,0.0295629708438723,0.0420345860457156,0.0876952738948062,0,0.0122863432668041,0.0630950255635732,0.0346087751851699,0.0241757912734442,0.106671098163203,0.0964974286572895,0.020803931372505,0,0,0.0524150096859368,0.0333316623014456,0.0742270608488722,0.0324682490308649,0.00274408140102295,0.0500184157093245,0.0584031426981515,0.0862613204139277,0.0525845502433193,0.0290786063391374,0.0885760012806255,0.0853069301233014,0.0385665881624479,0,0.00100461782496097,0.0977054080463575,0.107631091690056,0.0749975940223341,0.0567554264959936,0.052531894450314,0.0226196450556673,0,0.0451635813651928,0,0,0.0352357216232674,0,0.0313432987925232,0.0736828249996345,0.092830077848436,0.0615185249128829,0,0,0.0881267731479939,0.0849783328687465,0.0919418733992802,0.0252350796960951,0.0749184489165328,0.0524421764562988,0,0,0.0145417905463126,0.0738788381578207,0,0.0553232843521769,0.0385353193938296,0.0373323106951649,0.103613595736696,0.0793949178871942,0.0973467654515823,0.114123410283209,0.092164861266318,0.00321165520012966,0.12379464937151,0,0.0549639148987104,0,0.0541496132372458,0.0406255985764746,0.0811939885634446,0,0.127405341671065,0.0645721869318693,0.0561103128563443,0.0356505700272368,0.0236858539210627,0,0.0959797913889565,0.0166309660966094,0.0476474409884239,0.0744860411976999,0.00373439637931321,0.0570488171912136,0.0268618008219361,0.0479496548820745,0.0177423386319917,0.019649499521292,0.100263590561166,0.032006363431746,0.0200557376236307,0.113733840111569,0.0657124340819303,0.0924552176910139,0,0.0184637755724672,0.0193621998107848,0.0502749395647065,0.0774473106305119,0.0399310690869372,0.155097930374595,0.0673087184038133,0.0772015918562819,0.116986426670776,0.0622977226398396,0.0475941623124702,0.087623412446817,0,0.0466240675533557,0,0.0442816934596782,0,0.022364163391739,0.00647281233342201,0.0465618322275264,0,0.0240886996863572,0.0332849987956333,0.0243905964283534,0.0432946582293936,0.0168006538307245,0.024822182828499,0,0,0.12010586296766,0.0612549043483831,0.0111891760660345,0.0335206876103623,0.0198855372039651,0,0.0384819265872348,0.0534585686266923,0.0925689804348175,0.0301961087228061,0.0415440558342868,0.0111578098952813,0.100626872858867,0,0.0291088158478424,0.0598528132714501,0.0315888608397537,0.0404801418837053,0.0788932988623622,0,0.00957948204963031,0,0.00240888228167289,0,0.108788637172486,0.10350767773045,0.00271498896788802,0.0806200058846225,0.031791796018879,0.0171018203822614,0.0126582666763533,0,0,0.0518950835050616,0.0777132098496089,0,0.0412603041472566,0.0781107181445255,0.080015668770252,0.0458408032318376,0.0626335847195146,0,0.0471216365092017,0,0,0.0502388669758473,0,0.112091466371043,0.0667107981186186,0,0,0.0483707218958879,0.0703106330873823,0.107648761330088,0,0.0137362665573891,0.0181384351284433,0.0700209058608469,0.0934996927077342,0.0615326056651247,0.0865113402318506,0,0.028024814223501,0,0,0.0229164691959885,0.0345348485413158,0.0626357590963612,0,0.0626017503497684,0.000382630110554255,0.0850233109244,0.0450649354579412,0.0838512612579086,0.0462850957562957,0,0.112900249996341,0,0.0476833240592145,0.00747688512903691,0.166760964802723,0,0.0410540676131617,0.029215082490454,0.0318272353519511,0.00136585166852062,0.0507223773470002,0,0,0.0349557696601809,0.0639814441510153,0.0229495016069005,0,0.0821933482275599,0.0717636799297116,0,0.107481287098498,0.00397228191794992,0,0.117196212421209,0.0104522410455068,0,0.0634542693446618,0,0.049320424760598,0.158823633072554,0.0269069691794286,0.0804634037776417,0.0466964900429231,0,0.046213106916212,0.0301522704987183,0.0951361557342734,0.0213990195975477,0.161903406677549,0.0854272536032402,0.0215139686286996,0.0116106020791727,0.0647064919704191,0.0632389549923922,0,0,0.0154015707220937,0.0773489835789423,0.000394681012030197,0,0.0313737205791338,0,0,0.107080569373803,0.0653479792824616,0,0,0.0479133550513004,0.0902106485861029,0,0.121055124363049,0.0661256886204511,0.159775891322375,0.0561797293065612,0.0874300603640557,0,0.0949669775245163,0.0279224584325776,0.0927778110820769,0.00462983463675122,0.133715809450472,0.202639694984933,0.138363715754826,0.0173548530428761,0.0911567676702528,0.0566513048925776,0.0644167486693497,0,0.0240899485986743,0,0.0223016730781213,0.0596545029977271,0.0140087939212602,0,0.0189552553973342,0.0217829875248457,0.0366938740325031,0.0840109702750585,0.0334611652382149,0.0255745469872995,0.0207684975442558,0.0628734211929672,0.0350729410581142,0.0934503756250494,0,0.12840350772033,0.0414059263655352,0.0767961644103413,0.0643293058601421,0.462922267406485,0.178725386486269,0.0404472035480668,0.164181059173863,0.0909575064553893,0.0916774154189795,0.112534732556177,0.0287231589554788,0,0.0717495464417145,0.00860196876184308,0,0.0808154523597378,0.0580774169427088,0.076305624168734,0.11206884991685,0.0219213801697429,0.0661944796971022,0.0198053095883354,0.0618804870746174,0.0398046136052217,0.0930698528084318,0.0185463462229794,0.0527310104380616,0.00189178377218051,0.0696761129192916,0.0787731108994368,0.0207595336464075,0.0247917723975986,0.0495736183020609,0,0.0942684477639729,0.0350700532505124,0,0.0555318142245878,0,0.0396956612971236,0,0.117052692303616,0.0169219023518834,0.117142597298681,0,0.116304554931492,0.110574344409929,0.128698155062565,0.97323041766413,0.433610715856904,0.157922972118845,0.686329463421241,0,0,1.00736556598857,1.51516473334027,1.01340154768399,1.864032892752,2.19474811976421,1.67124414643634,1.41269658166751,1.62326570395082,1.21059827310552,1.41550381142185,1.36921045651329,1.33358462804114,1.37121070074043,1.30736333285432
"45",0.808309073521252,0.748607199605231,0.49992254203956,0.245368872707985,0.169631038909625,0.172732644909005,0.06471992845581,0.111258275520754,0.0707099415426426,0.122488828000925,0.141137562088974,0.0887394722154471,0.179946150105775,0.289773232083469,0.301290211352297,0.612768180938883,0.905626477547198,0.742886950409047,0.819466363180395,1.00139933968908,0.965220390526467,0.589684276253215,0.242152256271391,0.0522458133683844,0.144468028812484,0.147746874212774,0.439462206911954,0.459533542127049,0.373820617992719,0.292198569370327,0.0727162668666398,0.32527980036273,0.361682753783044,0.402840278016177,0.331957461690542,1.3814680100122,1.88742933370883,2.01759564285817,1.02248375313504,0.97963955916958,1.04994512186492,0.880817746002894,0.525019644097142,0.0759224004401855,0.170656031655959,0.577583579430113,0.549569294366391,0.55776591923474,0.551795465013013,0.620328121661306,0.304625003580716,0.15413115492997,0.227190785013001,0.127910543958791,0.152918820218991,0.154731483683185,0.192758271185115,0.163608930714804,0.165241469902624,0.235546841096099,0.389402014902107,0.225247966232279,0.119979336492067,0.558684611936422,0.989265034672377,0.947758102631782,0.810365205726308,0.55265756304968,0.611875747086881,0.404544506564385,0.000845305934484369,0.18138292936653,0.145403763778859,0.0454739814828201,0.0931009397228719,0.115396591956187,0.161007300576387,0.222551054346967,0.35802222119997,0.333512100533598,0.459357312678039,0.164716518536517,0.258314519038014,0.457572689219997,0.45240766443106,0.946486799414443,1.49169087035726,1.40711985302891,1.10235231826469,1.27731738922891,1.13024943035493,0.999370619711533,1.14844116387435,1.51361090337691,1.09745431209727,0.587648002408984,0.621026935169597,0.704449915176295,0.292518321755581,0.155428812523693,0.133445466902401,0.216129544302898,0.175916744951673,0.168339179364255,0.140781046384759,0.0340450544299963,0.161192808615132,0.196644317867919,0.209756735581036,0.183133516364012,0.234444536785659,0.850843140117789,1.44455527424482,0.823446530131212,0.596333721023661,0.662670506032205,0.822421778290724,0.586681915219286,0.203190479979264,0.357199997932024,0.127164767463285,0.17616213878255,0.196582808118276,0.242281947979433,0.249105947417232,0.0632767946473787,0.586665386271752,0.662106345218021,0.233346899118119,0.349811956711318,0.685983325006564,0.558043414692321,1.87510636851575,1.08173993040504,0.888336246518538,0.989468231545347,0.986646219848334,1.06501772733088,0.930196369631181,0.493829251535365,0.420048079984892,0.624624562818009,0.697150090958389,0.728841537089173,0.576668776367177,0.6754726128972,0.26527732329989,0.0171175719714684,0.0998449135975393,0.0908596050039008,0.159937744170106,0.218971361759157,0.182777517134166,0.154236627629943,0.275608846855775,0.368148008713205,0.301939272447815,0.184862861443352,0.177722550915779,0.660469460999312,0.748239564195172,0.720079252568981,0.723419819879327,0.885789089695329,1.15805172535204,0.507314516187332,0.288758168247464,0.481507291508407,0.486643516069104,0.329089520508555,0.0701809540828723,0.281757495585673,1.24379962942997,1.28399592216972,0.710841860796817,0.32765829861225,0.473734594980544,0.561226162250993,0.99360418817096,0.917539827067899,2.28831666633174,0.815356003186884,0.397418229992961,0.425991215073422,0.627184676192791,0.446438486549169,0.584596202355121,0.433023132925445,0.66896175547979,0.89615355130072,0.850834652660727,0.731561017051634,0.678509329325239,0.677580343701348,0.28823516168058,0.0696488056572046,0.140167685392586,0.100897964153496,0.132929683911208,0.11253476546514,0.128446398821133,0.137850661129758,0.199704588665547,0.242380165164098,0.308270533770179,0.38322942498581,0.278942381720988,0.881311146173213,0.98922852019686,0.61089008889063,0.631733806190967,0.625903161263702,0.643259183523023,0.441889143561939,0.117121397803993,0.117949553650218,0.19986215263434,0.100370974246601,0.302173207140667,0.360643028286538,0.391965383961842,0.173149588631412,0.0825280800772904,0.410300852278213,1.08098219772712,1.02876209433718,0.611436576908557,0.889936076124964,1.55627581389736,0.70585840972737,0.438189393126463,0.414080439146507,0.355876185115207,0.455581752464644,0.448521046207471,0.470741532677445,0.47412109814003,0.692704501696161,0.777007627570685,0.560386641166193,0.632602359831857,0.665431142393768,0.220308895874139,0.179233110020938,0.145231164968694,0.0502736584127783,0.105665103477401,0.141189371063537,0.124397997714812,0.177251646489499,0.116031299737978,0.148685238262351,0.0425851497610602,0.171534773313215,0.337699090339039,0.686070914214645,1.03981006537911,0.883470873506012,0.712100074293892,0.573939818765095,0.606102979657051,0.498524371628903,0.304624492658208,0.149940859342503,0.15141035692729,0.379282854406984,0.414183268927637,1.13721957696747,0.918954201219077,0.32570310218835,0.423461647382681,0.389824631567142,0.330978362044961,0.356413576669688,0.395024745531073,0.422634507371138,1.05977216564331,1.14970644348286,0.487318665670224,0.368190875142565,0.362232766645401,0.555459796968262,0.464338916952335,0.610896770345907,0.566370745458829,0.853875174380825,0.944033218058471,0.724158690070002,0.738290705521161,0.516554867184583,0.379315221326064,0.143030334052562,0.16742959532342,0.144691113024802,0.115273655373808,0.215585630097117,0.154678569445196,0.0471168435777429,0.0533723048089521,0.0282052529805654,0.242300915431928,0.328138753471276,0.437939626433989,0.734129614989595,1.20275522590922,0.718546560610978,0.659274641873944,0.693049851018154,0.581524873516054,0.49866452646217,0.0517567994072162,0.210445881007392,0.00404478509791276,0.0890224516032712,0.485266911857282,0.324154803739786,0.557114893457355,0.321996655578172,0.354003314511453,0.117992941215605,0.141450484651746,0.204676968890594,0.35382200352333,0.38980668310901,1.69120163433018,1.34454586339995,1.11629138766225,1.04558625100194,1.05002718110937,1.03653775776771,1.06562105142379,0.920391959540254,0.742618764549608,0.739936370286509,0.688169293392185,0.812197914167514,0.774667678591608,0.698084413009836,0.425465244195984,0.143804135143192,0.162456097163004,0.266385495755818,0.298250791123429,0.189508811385183,0.268614347780838,0.18233802328604,0.336401243197624,0.270625399117731,0.349087702192659,0.225165729108371,0.535441924836201,0.631290734106228,0.766054808636773,0.762557469847094,0.791049282037459,0.836648562151493,0.705969869910668,0.617384380640996,0.335553701713889,0.411030312375113,0.165165012396355,0.0781782632860908,0.0918369055559374,0.165154683347214,0.126194857946759,0.0699186747039497,0.0803857404175712,0.0892308357922707,0.322984346919523,0.505999602230594,0.573213341292771,0.581328071394079,1.29736407132754,1.21555142259388,1.12960378413388,0.885601151472226,0.960142825190233,0.919909325775763,1.07540460415242,1.0121134134223,1.11318266549198,0.962842142983632,0.62869222663662,0.703428311514924,0.559744968797265,0.569427195340547,0.349628943790933,0.0928620584056067,0.0521499312991148,0.127710728321702,0.111700025011264,0.10114245294812,0.111632747731339,0.155967689241462,0.0980962894017096,0.0875053021283173,0.140429243690157,0.114974483102087,0.236501509155793,0.531935565995355,0.749860680680184,0.72792300518671,0.974904229289111,0.563920691294092,0.704856940372086,0.531329398220401,0.195721329025152,0.225532185830156,0.12714731717571,0.194224122458402,0.293351903594128,0.231878283544125,0.505069247038165,0.720979939922435,0.182139868590463,0.301330316698126,0.351008062130749,0.0442333360779987,0.286925224264001,1.00842106842303,1.16124046752618,1.06849835460258,2.00992713132651,1.5178994898305,0.476182196503858,0.436099533049689,0.502099819086158,0.463913089293101,0.606382170775511,1.04118979354222,0.673427179616289,0.71185598594236,0.639994088660742,0.618642297815515,0.26797147434545,0.0878921173798105,0.0774850095091428,0.172488274155371,0.192175222266314,0.30548019731776,0.139589238995527,0.247990031625659,0.189539839612336,0.0357892195681017,0.0819857013349813,0.0892393621317627,0.122100928904006,0.803174769917793,0.749791312004572,0.756130328915996,0.7727340743812,1.07730770067044,0.621529397711792,0.535266951607358,0.308350534905811,0.202402928174839,0.0897915791330248,0.145492713843724,0.270028975259825,0.237006848303303,0.140314827662025,0.272817734899144,0.316538214159681,0.23148319779529,0.1078540762032,0.434050076164247,0.765057807413768,1.22259920656579,1.04805618191382,0.935858183529209,1.00470814258907,0.858183913744469,0.947103253345043,0.987404012881454,0.724807568181017,0.807735148464078,0.810278450008325,0.795351049290001,0.582363228178354,0.554811487993488,0.567561033828623,0.65654398668477,0.305053346421437,0.201927920332096,0.120062857084846,0.174532417387224,0.163015849113379,0.210489905011343,0.330823967142878,0.221662475752968,0.251208260666316,0.138686842382673,0.215171629608603,0.404924220262937,0.310609075603349,0.636303911460979,0.797887246136221,0.779118863507197,0.758004253775753,1.056698413636,0.864264746740345,0.561451627139657,0.309779185156207,0.566268937922104,0.608875681948979,0.600652069942464,0.127403197241202,0.106103696223157,0.258555592072917,1.08848363002271,0.444758042997464,0.359917446850206,0.503928388139015,1.93308114553744,1.28568809347581,0.964102194556614,0.788946691092102,1.5153347961211,1.19585808538114,0.471326264191645,0.504993345187762,0.422008105316641,0.524321311277248,0.472337008079916,0.569408506215939,0.762077788065463,0.516075366681599,0.64954977016777,0.666574331677744,0.615265977063102,0.230827971225127,0.207908814657859,0.108161949740021,0.0861586464433741,0.102520033183413,0.130281653296385,0.115455266256674,0.322382855252371,0.245063011932374,0.111619271569086,0.196686475322837,0.0998169737940682,0.245178936455876,0.695517173187881,0.794200944314883,0.788534788307907,0.813235014165155,0.751062969009308,0.932376919201608,0.674322025153272,0.402581626592479,0.633310482422839,0.416439112258423,0.070143006278262,0.230132379132913,0.456060682886544,0.422120853368246,0.333230736684211,0.311603198238565,0.313987698628278,0.33352157814663,0.533080505568977,0.424523017547695,0.379763932608971,1.64793460118939,0.389515394915244,0.462318507010866,0.58799459880219,0.464419901585061,0.46718337337754,0.384538294654822,0.401640888752743,0.463846368408307,0.595335478188632,0.694153903490114,0.580161014166004,0.58944860774461,0.638859400867719,0.138873974984458,0.0923020165665153,0.131739759626527,0.146894459767851,0.0932942208544559,0.050527569897802,0.0538301418501459,0.0856192433148431,0.112952212264876,0.0381305813072424,0.0427613856602233,0.240087903064657,0.297486355673093,0.606667508069348,0.820532796105424,0.846708294012766,1.06026277293209,0.709636958239556,0.750975180933977,0.591253548334094,0.20795671311263,0.108865982807265,0.10052462232992,0.326863208030501,0.352068129002936,0.12746421449103,0.161773431082056,0.194108072490562,0.332666707981507,0.314503962973806,0.228760372190444,0.196905743039412,0.393686866040318,1.72636844467246,2.45307007951178,1.45879595221563,1.40422896094505,1.50794826806962,1.15594554955873,0.977175184272415,1.05159235393488,0.967620610925997,0.651299735513373,0.708140032183117,0.618022778202603,0.646624594322561,0.674828491453613,0.714833125631918,0.353592787765358,0.104297804518686,0.10137370040459,0,0.0843979984845325,0.326129664750098,0.186918498733237,0.290773823740599,0.164148129620203,0.232276482137876,0.27007525557432,0.221437905441966,0.439720501478975,0.640267376440233,0.887884155511935,0.769072263012473,0.671427811919182,0.821940913515853,0.728976591804542,0.569050329456174,0.139857185407354,0.338827915061326,0.288469861418813,0.516181689564629,0.29395532461508,0.237570145440926,0.225936656173385,0.504010798870844,0.379100894999417,0.672533435034698,0.243867354671281,0.16735782873915,0.393558172051763,0.347845491577203,1.09281271032187,0.478530388425036,0.356361899058394,0.65094896714108,0.727259472836138,0.472249866691725,0.490608806753236,0.391766840442382,0.505352274578177,0.788838919971562,0.617675978454402,0.840678610975406
"46",0.674899480419872,0.8519467135916,0.799554550686462,0.883215109953375,0.881624863866898,0.0758734984047282,0.0725456241122491,0.134037019225469,0.182391644671476,0.193678680662125,0.0351657026646667,0.0990785982248965,0.157185470796691,0.143192693021298,0.213177614270068,0.056356963108928,0.107074256521405,0.506995462270402,0.770408174091689,0.599076603018021,0.683839457182106,0.606190051729676,0.710240344230373,0.62084867568745,0.618573960514183,0.173498722963218,0.770213438691207,0.848369411170402,0.659820575640406,0.478255658930376,0.505291169954129,0.615576273560143,0.590445262590623,0.509165400513191,0.591031366347345,0.459134235253767,0.728540902103961,0.487004540332728,0.268456066820444,0.15393314529889,0.0513966735140228,0.0974359622273821,0.0754017525969941,0.17991495706662,0,0.100327218894255,0.112225038530989,1.76470925692566,4.06008282260131,4.97277223314644,5.00806627649542,5.02976447144326,5.02955260662582,0.645489710173631,0.102453976289132,0.0877054858877431,0.165856113556414,0.160723667189566,0.131867269006763,0.103792217278757,0.129382857240663,0.238672508153452,0.215943604944271,0.0810873536054388,0.170001775040326,0.506397198031355,0.496631810788248,0.611129302619988,0.513874101339981,0.636633095545168,0.694814982991722,0.776493159607466,0.625900222707823,0.233323212897952,0.91498438385211,0.932792606865484,0.700118596407771,0.583259618567759,0.744641017773778,1.08792031145946,0.513699401873149,0.613698912386935,0.712183250439006,0.691398071157719,0.961760703669672,0.31465236047606,0.263213742292826,0.29498512973212,0.311035580172009,1.05639278421634,3.904017571716,4.03258692910991,4.19750621557005,4.13659147135942,4.02791040902168,4.23065701563678,4.20996090899996,4.81128907488684,4.96772278177586,4.87220335080419,3.88922139265499,0.306680285362513,0.112644497425954,0.144385826273728,0.10429751233815,0.115895193326924,0.123329892684778,0.258569634279216,0.0538132480496422,0,0.0738414644941018,0.210210544859795,0.121938406090716,0.506949627583739,0.659558874372379,0.601647298924697,0.63276960071183,0.566885084299129,0.694721033410934,0.579450085941847,0.672478911157415,0.599405022060206,1.7419160278057,0.77978540718858,0.290315695706928,0.543695223270671,0.598905792442422,0.669500096868221,0.655350880958442,0.578480277340648,0.838418026573905,0.834599843314451,1.05934571501817,0.606119885158095,0.578524358525936,0.636405111138684,0.701557373488644,0.403542574044564,0.49153233159546,0.27601668265244,0.275596266084179,0.268791992902917,0.147553534650574,0.192518926303886,0.591706310287981,1.69953718066944,1.10530365959177,1.63325813547938,1.23640535736795,0.162043789552478,0.212093232977089,0.0716015770592804,0.0718278216130778,0.133780849164893,0.1211950200696,0.142355596478134,0.0969769286769997,0.15756665443818,0.143781481861787,0.451735555654712,0.453750791749143,0.595322393638047,0.550497712396508,0.592908056542953,0.633137996638908,0.621721569404979,0.794748503454674,0.811922178834087,0.649429118007567,0.457294855935558,0.910265856482168,0.855626154654978,0.371059940583703,0.326065300087383,0.29388927970669,0.199920926476296,0.21559421448159,0.265522999024376,0.0850403770108072,0.70210571065607,0.838394806011725,0.813903640390252,0.864577603638525,0.627152384079433,0.850333961400669,0.916812499086131,0.990792707745676,0.981587017776405,0.893123490310267,0.888674036970218,0.803983664779984,0.242019638608668,0.41132334060491,0.860041492535234,0.844401408150262,0.865901060517588,0.714567149318653,0.100087153621635,0.0734360140270861,0.201148122348208,0.105489102650854,0.171100211516632,0.133208938930649,0.188186990200391,0.186938530985249,0.0887204140292235,0.0736558932998909,0.140915733904678,0.21001626198196,0.742384820174088,0.554918298492455,0.606004492576262,0.554823645646269,0.659281907145614,0.841721083659488,0.574726492875299,0.62896053726745,0.106376670934646,0.274861768138033,0.170402497529342,0.113795997791288,0.244216035822463,0.14774056936213,0.249385762354424,0.266439241675536,0.277585151176461,0.218675039720465,0.362235561345238,0.416281367560508,0.390414695680689,0.484360235162039,0.626452630474323,0.419899780797055,0.434360686980423,2.90173476010852,4.27731961930828,4.66377469284096,4.8243721101872,4.75128923100159,4.87289088654618,4.67501033718251,0.335270520423896,0.166412285403103,0.0869127802922489,0.06773840553519,0.151073836708609,0.0683782848530984,0.165039341864447,0.0945510010349689,0.0556765818927805,0.106468331126915,0.233127805832731,0.301594159728995,0.202730757701545,0.225040676782865,0.264666735378347,0.128456150639835,0.931760705800921,1.29775677181536,0.910777789211035,0.521403301019115,0.614139815066732,0.695726523277394,0.58709918913527,0.607003152707763,0.148383318495508,0.0609301494384444,0.130630824007371,0.104894947760706,0.861243284999599,1.40783894376026,0.948971841313334,0.568437602994091,0.682723434277453,0.630933321946908,0.524945899763682,0.575054291053617,0.101479769782667,0.150338149967341,0.276026591330004,0.271696571544454,0.154981007457526,0.420185386265516,0.372480136047732,0.288183514875175,0.414334633114809,0.300641310793472,0.200529122272746,0.155283040031213,0.125284133924613,0.125178731940781,0.154907871360737,0.162284546165333,0.202100489540393,0.122959089779347,0.152201792742877,0.0994855528917303,0.174399527540758,0.145749931616216,0.184373118187754,0.0749084555467112,0.114746076900811,0.176288525362165,0.143445713696765,0.105546491158685,0.104988870431671,0.000269956075814731,0.990626019208889,1.38627544573522,1.0058877507604,0.860870138523426,0.671602150120088,0.610769782041958,0.728421825629812,0.609745796576917,0.0563990849019063,0.215412587597127,0.121333617696606,0.147398075233169,0.0381292468651109,0.100326017222842,0,0.0242648902968294,0.0783659682657878,0.151461919177321,0.0919104504203596,0.190243059116937,0.102876241616102,0.12196579830657,0.0567458760985671,0.362102506957416,0.703381239247409,0.424672560817862,0.517931805750804,0.394973618917532,0.350078582090037,0.355812718063562,0.352166656142899,0.482888495589987,0.139636703286301,0.0554352813699976,0.111911758872733,0.0798936416695529,0.0576773939541743,0.0829298491068661,0.193720444688771,0.104385904234672,0.147358695924759,0.105344559816158,0.135847687306978,0.169833513239924,0.0129111237513124,0.100366255606589,1.2279476828222,1.48475445668551,0.999386567182288,0.705845880565989,0.613824960134287,0.703089572290294,0.638721425728906,0.586383760117504,0.106957247027802,0.12365787420955,0.171404227929468,0.0802895849697379,0.909420491702193,1.29820857694043,0.966669535394958,0.635757808825086,0.533701873137965,0.586991933502068,0.669420122763608,0.594430550341283,0.403265754670269,3.0615426556996,4.08324464489762,4.2170514485268,4.39431648366733,4.36901794393783,4.39587978980589,4.46168990111316,4.41761521656124,1.85377552193662,0.165847217723763,0.113854527108844,0.126523907246256,0.211359194868589,0.156762223290036,0.114359839911565,0.0437886015113343,0.088787945017063,0.147550444339036,0.204700659403328,0.161041045781874,0.0581803890787172,0.118783682241338,0.177020660771045,0.151099203160674,0.16538594902931,0.0494688067334362,0.215889537115345,0.965667334611444,1.31487857207075,0.925302477454547,0.615319947132703,0.704360060984082,1.37819524960493,1.21793840389521,0.595702069952754,0.314355085184872,0.143330724702846,0.167907776250534,0.110250285547467,0.874224578692839,1.31854194778414,0.990199146423907,0.545094829232847,0.649202199602117,0.734239861843678,1.5426642185857,1.77455828732802,1.15797558291966,1.73691986405545,1.50974049619469,4.49883009941237,4.97956755597745,5.15599572952807,4.63975361364729,4.3310693490247,4.60897174090067,5.3079235767678,4.82160917526174,4.84642654326984,3.32798864959999,0.382417288415498,0.445404328064945,0.28811482123191,0.246152845375775,0.151115940962738,0.165756750765971,0.222293180872152,0.241463639200606,0.247195219088779,0.196132195090572,0.137480896349824,0.149759218801463,0.265446817222876,0.122944796787899,0.0143277181271492,1.04981623862949,1.32381872616896,1.02138762925902,0.877443175450478,0.672967987328176,1.01197343348939,1.01520528079776,0.640504650119627,0.206078772628385,0.168549761800942,0.119642097938616,0.160534934153222,0.956854620598277,1.3460440468689,1.23669169570095,0.830168235937394,0.714758535147284,0.812426490780162,0.624362212774905,0.901460948103457,0.347211233582681,0.349167550219715,0.318381918532239,0.557890222510223,0.230646497767283,0.363287639526717,0.457986301953737,0.441323371530626,0.485744978231237,0.729407466118361,0.405264461497907,0.394440014539552,0.785680350443232,0.218493935227545,0.344561718287857,0.0713552357211076,0.208651144522501,0,0.141331807623722,0.11994525582281,0.146632881963563,0.169214306946281,0.200606108524643,0.105393498872773,0.0622347697939136,0.106828731965386,0.999368286255228,1.36585116274241,1.05899618494513,0.511126126760993,0.628092262882328,0.685427784711846,0.771780287667398,0.802390441401743,0.71973936784682,0.655003412999993,0.123631065177651,0.0400159272347106,0.170780652098179,0.157002160611633,0.411831402781105,0.262250328094333,0.264252260691559,0.185787450691508,0.121912034855014,0.228428324208633,1.21454346094206,1.70527837047163,1.05810808831714,1.35760695761131,0.504584481617922,0.589670963161175,0.588045511945924,0.651050363979266,0.525365491042427,0.56021089206741,0.590546384768874,0.607595651082912,0.207258518441134,0.117343869155033,0.0814371295492883,0.151687928364055,0.161286443460069,0.168630156015175,0.141770134661937,0.0718459860029415,0.0595644243346325,0,0.0702083893239967,0.095182050585854,0.125533352256958,0.0501009118182131,0.0642497727768547,0.182021650299123,0.222402567821613,0.141230962973857,0.911513910954123,1.37116274716707,0.910176046984891,0.650068622951342,0.564569124857568,0.81028899955134,0.519446864531122,0.61712778670402,0.0619583538200585,0.141302674841683,0.0565928454065135,0.120953138103461,0.0772448301375271,0.204876440330949,0.110500944322426,0.111699110589261,0.0168015334988794,0.161392065440696,0.116739005250948,0.170982184441699,0.0355392950270866,0.123527915303748,0.0424379189144963,0.112818910543881,0.108986033758796,0.163156758065532,0.11654880273504,0.121855202313935,0.239600207654617,0.0859516059783954,0.190274472057252,0.12763435550587,0.101439594347317,0.0771201719630138,0.168177498992276,0.106916551197223,0.0846588583245514,0.0793462755470869,0.0875193187778627,0.188966003156638,0.0945691564380828,0.086448068891686,0.0965184958943345,0.235645784870918,0.118384857836934,0.0491558820023937,0.117586865717702,0.0910904613190519,0.928076474873868,1.35288799824931,1.02241037425981,0.545862112315303,0.570421740197958,0.772407283304258,0.642824732966412,0.625276435574234,0.100282877791442,0.168514450309088,0.189213548057558,0.164528912022788,0.946825581943058,1.39727257397591,1.00087138384822,0.571539990556906,0.63187241746323,0.571374474539864,0.563916205813442,0.513096150994469,0.141144644364006,0.108875160700763,0.132286177379123,0.0753192588413261,0.132874539585513,0.118956279983276,0.0296846932069813,0.0996973735134314,0.026020002863935,0.11910289818099,0.0985779568338561,0.170527885983775,0.102849903350243,0.182446271398202,0.248955728399204,0.0712676432527545,0.0996552498708566,0.124637520063899,0.216361761693408,0.0923647477956762,0.138053677144917,0.0656257431505433,0.0460900148212597,0.0773927802225021,0.147922874975939,0.157334078595009,0.148546289028027,0.194881505112128,0.174089624837285,0.0937569255170015,0.855745318591336,1.35453260195366,0.952066541904946,0.68793476224157,0.640102296943486,0.546881266727014,0.555867896910448,0.539417707600118,0.14003299337168,0.129939184503458,0.0501056938215342,0.0749100309704792,0.112417333775057,0.171818344414228,0.11737497177179,0.11152499959459,0.0901061473996942,0.14187557192325,0.0726289721471498,0.169216755106721,0.106767856358513,0.0760506806347142,0.0329257660984717,0.0694441503469957,0.0901987230344189,0.152465542249051,0.10193557057986,0.0329504669784545,0.147005004561509
"47",0.0709950408245243,0.0333045379466672,0.185554229376927,0.10639901875917,0.127991481038276,0.106377114840537,0.100767562367051,0.117904733496132,0.0297866445674737,0.157585878365327,0.121097147460996,0.00890718634856635,0,0.212373007145029,0.0657572398476363,0.0680566276527157,0.193173811815471,0.209241278748002,0.176801002499804,0.108079559493097,0.11120839321381,0.267093524717018,0.087728832872098,0.039424547434008,0.0495398185072973,0.0568539198784696,0.150048961546729,0.165828317907348,0,0.230275885192233,0.266661658292656,0.190632508944373,0.0483494717932066,0.180007969537481,0.443919947988585,0.534864903182625,0.938745316643662,1.56560922352012,0.597814312591847,0.556128509887079,0.643971111241237,0.687074097820703,0.590930920121096,0.43789150383601,0.374180270349434,0.171104672740742,0.0240357107167765,0.0757649037152726,0.0647124169410774,0.148248832588026,0.0487659788765559,0.111993246631084,0.0602485137962327,0.051673316294481,0.0758356024003239,0.056227079598396,0.0308621335147411,0.0320810521658016,0,0.0504698869290516,0.0723883513780547,0.222343450483714,0.356513419636973,0.399102715756387,0.29659612814255,0.298445954566839,0.214790414625351,0.179814917469564,0.0893866865444093,0.0473343056932688,0.00351872918246619,0,0.195586715126039,0.254090905559704,0.297704382392454,0.356606567776914,0.198185476322073,0.143568474295087,0.135630594924463,0.159593534953897,0.12160899284871,0.32618136540983,0.252004459666395,0.994721008471556,0.737725297517624,0.70463866203517,0.223765769639008,0.228975795975122,0.197885052877681,0.296125248960423,0.340262279338096,0.205447407193383,0.089529430039343,0.0823223570644814,0.068929283177473,0.0240740785714517,0.0218880056070561,0.166897992968687,0.0599165815332159,0.0117247204744755,0.0545335337850819,0.124755958827377,0.016985375396458,0.118853241775674,0.0914448446171565,0.108552842250005,0.188640190051316,0.207441631512995,0.197760610596467,0.188412720509205,0.144606866774508,0.266650088383444,0.14205323245175,0.125876021245008,0.0822862063119175,0.173279003763351,0.220330668477829,0.119750969937136,0.185906519022821,0.0207660955510171,0,0.0654208497342902,0.0688303119856987,0.309910010343543,0.1836728595643,0.253467917625124,0.574591610407792,0.196048589587854,0.16886906096126,0.155582941907895,0.190555746079659,0.123570672030156,0.58943251837273,1.11681795373112,0.241607979463416,0.29498990655902,0.193952278181976,0.293141336208489,0.329928031544698,0.45751599210119,0.32416982961559,0.319334071185455,0.106857110184507,0.0546615340464125,0.0646300015318565,0.0682670008984462,0.040093688098192,0.106954224995887,0.0670778829566806,0,0.0640380385204522,0.119266505071853,0.0323201361796525,0.0606313559730015,0.00538318099954881,0.0928581675575781,0.116750224666052,0.0529460864324568,0.296850441775094,0.114014208211349,0.0742847611629899,0.0735822422475446,0.382307635228296,0.365680289366855,0.0425495651650854,0.124108252387759,0.11515543825501,0.0450268537782587,0.0755994235394659,0.320414880671843,0.137193430792036,0.159586336306738,0.0932015935666936,0.208490507096488,0.265378444004299,0.214133406156272,0.218829794648008,0.0767794614311497,0.234296019533082,0.225310567741556,0.273029513054355,0.526480938592623,0.453278490853853,0.533876352740892,0.442690217102817,0.326641316426865,0.327128727087071,0.295826451144355,0.320381368209431,0.311141752293928,0.132162131882885,0.137740552608164,0.147235329537218,0.234395913626582,0.145380770454619,0.153251826408837,0.137554076727659,0.0626091956130345,0.0516207111769821,0.119892688418955,0.0441569029176969,0.0102246914929755,0.0579002366355922,0,0.079568123427739,0.146332119310744,0.0873725935664522,0.198405808091719,0.132041611285591,0.145199616839831,0.173777951732091,0.064098551987787,0.0689915844217776,0.0936124485332998,0.474211734690017,0.621634373642953,0.459344982632942,0.136391564150473,0.1132992087189,0.084949430268399,0.130027138737528,0.114277936177953,0.104655100317793,0.100161312107619,0.128283151081032,0.403599396589684,0.398734398752462,1.11751823933721,1.09207352812011,1.34683928378591,1.18694940590596,0.330212595867904,0.349422798382187,0.179072544924481,0.209158249538475,0.252081849816343,0.252642525665295,0.127670934114476,0.12283991687897,0.0980720518116704,0.1980564172092,0.0250004064484727,0.0895313661552739,0.0206092396433703,0.11541979248199,0,0.137890344390939,0.0161186462456793,0.0914640499136763,0.0893442351677523,0.0345773643620796,0.0475220293076655,0.0779822072417595,0.0860013850127164,0.0467444729580208,0.248664348264047,0.136291478057072,0.100948764996481,0.188396843786213,0.204467331699672,0.0718303906404122,0.179937644184422,0.227318888313781,0.236374216721073,0.332100323042556,0.283639530655962,0.294627562713639,0.244123248588896,0.150608721246219,0.151009763576501,0.181246746507884,0.304601319011113,0.0609270385424733,0.317569205912558,0.510547926920049,0.245091707890604,0.355599692014605,0.40177407078664,0.396783824057639,0.293587647406643,0.280536023694529,0.203634023136176,0.33488687068955,0.315516686915824,0.406230284430552,0.288535755158637,0.188941257036131,0.147952574229743,0.049587495584812,0.0339481872183056,0.174864037717748,0.0705892251504352,0.0740850257105362,0.0384205407357538,0.0513928324509608,0.0346831288830109,0.0223844791158576,0.0404665544854706,0.0284660647521003,0.0889389975805013,0.0853438740708678,0.134380544068327,0.0909147435358932,0.290546461271274,0.074690678530596,0.218952230758303,0.263021171401655,0.432537366295407,0.512673788216943,0.44114064795128,0.117850902322004,0.0309861607255465,0.00436947615004626,0.179234121332596,0.0621331524415564,0.078381673606017,0.151953119434994,0.287263786009161,0.208836428862208,0.0642067441004119,0.153968048433355,0.14769843509618,0.298929997384554,1.27051408396825,1.29866997781343,1.21045826953757,1.12945608722972,1.09366263432298,0.46913169326791,0.427062066665734,0.451135128134413,0.444412926291268,0.422572477171657,0.354629182771788,0.280607707469892,0.224659498826744,0.197114359775552,0.162488100131701,0.0766423570819237,0.0914315666719068,0,0.0121692108869393,0.111104345242165,0.0648527105284923,0.129869141383628,0.123910170274094,0.0549383399500147,0.0153499765978423,0.113450828454018,0.0330554796144895,0,0.303771998413292,0.146431828690561,0.0965150532152272,0,0.167437315894723,0.0723921083709901,0.144762599684763,0.0508950249002086,0.137490375548437,0.177615216572427,0.302129365283514,0.356130697910324,0.213503206836887,0.188018232960926,0.13718611580282,0.158144826721095,0.268804830840063,0.10821363801986,0.901512371938402,0.780577583351263,0.366833403798109,1.04423717665421,0.74798722499747,0.431501001753588,0.299571080745929,0.384708003014854,0.244483886440231,0.224180021013323,0.321142725125264,0.327650201763465,0.263211291863178,0.377792058383582,0.344102929981398,0.186933286047336,0.154872291414697,0.0416325431437491,0.0978905532939429,0.00625543297746921,0.0379497892811138,0.0666104918782328,0.17454329346656,0.0054913518190543,0.186972475558981,0.151349619948318,0.0661494278615828,0.0886022226235468,0.250076608237143,0.0901334508847224,0.108714687214238,0.183614820860854,0.134699059013217,0.213536761375855,0.204396653338373,0.316704846053736,0.284291253260142,0.125155238228101,0.0255130143651024,0.0626941786003012,0.211243810231944,0.0834106843134763,0.020832727035646,0.141707048228178,0.0402058885802414,0.151487798826996,0.565184985307137,0.216847581172752,0.168513054004817,0.0905034419447255,0.153576152784764,0.296047334947281,0.352431520307681,0.258084216983608,0.307019876045272,0.273229553957888,0.227065907466224,0.266979541248789,0.307727706693586,0.103859454117992,0.069819850307655,0.123926547889475,0,0.153743970186811,0.144396296323718,0.131142524488156,0.0222637851181954,0.0555307302571525,0.148005078147819,0.189638818803896,0.10326940887418,0.0309243639193904,0.015663426011033,0.0698948103770106,0.0784512027918937,0.00983344347116494,0,0.296767184379079,0.141783775324313,0.150300581936481,0.129081474183495,0.972863340147866,0.102888142665608,0.104755950565637,0.105299718946014,0.142165837982038,0.0712001807729786,0.0564167912488211,0.0651809872972684,0,0.0935177636418203,0.0310379948467948,0.0117618759242874,0.211123875547862,0.0471491070332204,0.0278080177111989,0.0477223290042167,0.0473568266584238,0.0445248919662106,0.0614888120403679,0,0.0530127231385293,0.0259690789963977,0.0110756411797732,0.0477004569878208,0.00918546222020752,0.104271024431818,0.123987276466101,0.0151011093536253,0.0892699534955428,0.0182241576261054,0.0498597393196494,0.131474913442095,0.132841363074495,0,0.10989072895045,0.0953910362385892,0.104837399897215,0.0823096398196345,0,0.0642998171830511,0.207172039501352,0.064358859308298,0.0646869396903428,0.113232831078395,0.119046388228231,0.137732042665109,0.0992573778110332,0.11551949477381,0.0480490700661907,0,0.0849467940557998,0.0499375875384437,0.0374768994890067,0.0402811114813867,0.135918867313126,0.163955716217926,0.00673884063948425,0.118548667780785,0.00333898003661308,0.0436461820475983,0.10679301982947,0.063872329625634,0.0928309375330883,0.131482486232241,0.0172891983196155,0.025096867550938,0.0630948818363271,0.0737167254232517,0,0.105256767602967,0.0836470419350672,0.0223323980061499,0.0886670842892276,0.0412341543085764,0.0412621441008855,0.0235849386969085,0.0461876461777245,0.188177826610987,0.13479115165822,0.0352528447047782,0.0717934992854399,0.0845580644730777,0.0440929918290065,0.160830996379012,0.136501225480823,0.0455759863487143,0.0346317825793308,0.162717436785688,0.147710793243493,0.0239986446113827,0.085055484755198,0.0850309337463727,0.0770808202719337,0.0908410962200162,0.062807245190494,0.0951905687904019,0.0691381310376121,0,0,0.0867632263888266,0.0263976055803256,0.0805679997887111,0.0265793071212714,0.0955006595322686,0.012582379970785,0.0146053795315953,0.0536417919056326,0.0346082090844229,0.0348592298701044,0,0.034286013557717,0.0942381412580437,0.0732096539411665,0,0,0.0855542590478574,0.0708728881217043,0.076803782572524,0.18286371861595,0.0405627150815806,0,0.108046227374568,0.0620560098874356,0.0500902509728029,0.122190234139585,0.0203539564124463,0.102011670333751,0.0418820651503691,0.0434220782906813,0.0982875488840653,0.0186472680675144,0.1415251092238,0.146715150674061,0.067091656153697,0.000304508310180113,0.0462110833323187,0.101512974157296,0.0373437084095643,0.0014394387169164,0,0.0814586249261002,0.0113074292477268,0.0175204408242887,0.0551464431487454,0.068056641030679,0.0565112760430046,0.152886664641963,0,0,0.0231434672377734,0.213769600734079,0.0595443927222612,0.0184949262163286,0.0323560701063606,0.106288289306895,0.151125222125125,0.0161708863749241,0.119752012007201,0.00865294114580131,0.00310014231123582,0.089012857812233,0.064911502325528,0,0.0416965964013992,0.0911896409139905,0.110946700207455,0.103187456611402,0.137341973073515,0.106761265392271,0.0690082985979302,0.0898237576025393,0.018288429724044,0.019220519372282,0.120675217863209,0.081312506632952,0.00301843901512999,0.120155359752987,0.128794526765026,0.0790489477589453,0.0270389419742417,0.0103472503276226,0.00539880507278357,0.0741263071854285,0.122236329408022,0.0714254712392175,0.155175513069018,0.0501051857841076,0.0244861324364927,0.0733298679505704,0,0.137482050042308,0.16922557958464,0.147879383776507,0.107503307640105,0.0417765490284586,0.0977953038038736,0.0606858951059832,0,0.0230294683606552,0,0.0930665555552972,0.0896633772159708,0.044210597822943,0.0970035999861881,0.0619352209791645,0.070882583830839,0.0546882362052047,0.144067881945083,0.0328741861153293,0.0304625829711864,0.0859806750053499,0,0.0616559911320917,0.0758738400283881,0.0612122898514517,0.0539050189707267,0.0617786416975184,0.0722375671883401,0.113980406167426,0.126243998586956,0.112137910702035,0
"48",0.537769782858037,0.286437924224365,0.202609485892311,0.163073042319344,0.153802302497056,0.16521687909926,0.163902300898513,0.247018223023437,0.0664329454419421,0.18204444762036,0.13660124877408,0.15711292509097,0.132113447288619,0.0887022175507809,0.0944278627369665,0.162941411944877,0.272893678476897,1.06430902652511,0.995145062756536,0.210139889346079,0.179117521002815,0.255970994865547,0.0813156713810604,0.423031962231232,0.092586281028758,0.0918470995371986,0.0949519768339357,0.0976098956644127,0.146064966677176,0.107931605373568,0.176142422369385,0.182324425323995,0.184965536712604,0.124797348589646,0.576909546924262,2.26728124853752,1.29503120947882,1.11025119796834,0.797145952540644,0.69731001278238,0.743667652544515,0.502779688793506,0.343089199016039,0.6963763306315,0.692212366490208,0.709230107987087,0.769280099330435,0.874484357805622,0.785915324329003,0.583810759147462,0.394640248610136,0.40886035031304,0.322346009099428,0.537119665178015,0.39215022742612,0.373890994327547,0.344969629336493,0.177121213808449,0.339629044902131,0.275226293224611,0.291739202448312,0.31938033209749,0.365305552807545,0.492146136048567,0.384609562029939,0.730728119287198,0.170316579002549,0.286099274803069,0.448235752508768,0.280388438123191,0.128540499133697,0.101535004081164,0.41824686863043,0.337788423026743,0.235754794057872,0.371937991407517,0.214590359012859,0.564687961638097,0.357770943267855,0.408963073182476,0.283698171869846,0.372100590033517,0.444098576468975,2.38503034189551,1.19855979762706,1.0011574360238,0.639841406272822,1.02263176288961,0.80109466023976,0.703936801162376,0.274048056494543,0.839289178085964,0.79592555648339,0.734214408046036,0.715929561692506,0.75083057848458,0.755835916948688,0.555331075731519,0.371232251441771,0.342755736815141,0.252250268879347,0.394562092167052,0.362524030203306,0.417111369206564,0.312140536676857,0.292474645226936,0.349965247647227,0.329863178573606,0.349697226149182,0.458832933784665,0.43663141615957,0.449128209269158,0.328327462008231,0.439517537814998,0.285580481198617,0.69666216196986,0.193920681741417,0.151390337908342,0.239153744499585,0.283881516343244,0.0625667450590159,0.0457806781148234,0.184455298985488,0.131565821110544,0.154044444898181,0.142512734722066,0.272233669110992,0.253674992666917,0.187218512862807,0.196083017515419,0.213963830731358,0.176851237356921,2.23613856377018,1.21663426690844,0.450619400180352,0.593076153468512,0.501890528039198,0.465937987260116,0.354553899718438,0.599981105787365,0.641146646729212,0.72508464358298,0.698986050926603,0.678191807518037,0.690719559615673,0.476297459172352,0.262085281038284,0.35215941576366,0.354657618824427,0.279130530138304,0.340851517343972,0.184215188187417,0.246823813837273,0.273826745874495,0.207266010444905,0.148606698989462,0.355500751302197,0.670199062106084,0.296928301466772,0.492504953703839,0.155119799117815,0.0737066099292825,0.252801848354428,0.183103448083145,0.281960880214222,0.103156193733221,0.121510351820698,0.140432905584866,0.0641727794916352,0.110524255162759,0.221098933640546,0.0706939090520349,0.118802987145525,0.149929121216397,0.27185817323266,0.20039191713794,0.75701397795884,0.402868434870364,0.371006161223858,0.483381222028684,0.480217209951882,1.03832885778204,0.349666700163028,0.339736720119318,0.521098685833216,0.382921082418494,0.483250885663974,0.673607931944799,0.883353983959176,0.735047208821288,0.657874870077221,0.787404605506075,0.742941003991847,0.496941260179865,0.332156434124741,0.356439070624428,0.269616754759896,0.329314367384195,0.325705946274318,0.217603154748986,0.280410741087136,0.219329882009917,0.206477923044559,0.272581545663698,0.266680937102149,0.580857988155074,0.691455285940675,1.16272776398304,0.743062692127727,0.252195598970092,0.289852774039283,0.296742330230721,0.305083060260634,0.321171556281329,0.33940574529211,0.267437549204075,0.179936847003932,0.248400921700142,0.0821595065521568,0.0853869069727647,0.0914089583646651,0.223378574530964,0.0699811941744132,0.0781804928140295,0.169908234449431,0.245052923497392,0.353940890203303,0.881504075279406,1.18063346543235,0.894828387945293,0.50208293894826,0.400825738480371,0.38732836187346,0.354665261570284,0.394704760769353,0.745425376702891,0.639808027779815,0.803626189043232,0.742787112023233,0.60192008550355,0.750085505039655,0.532559579069113,0.411492665041112,0.257205438833122,0.217147767185383,0.35913163986633,0.166678083183669,0.394705571831061,0.308108703688152,0.22549913181789,0.287823913173917,0.307626115131502,0.446238898475693,0.724341517700176,0.494693484148223,0.297821722977694,0.202456899160896,0.400712041232253,0.148740357320973,0.228583217682695,0.623412981953606,0.186614100663015,0.129993247322688,0.0867272811337596,0.139443039915777,0.204146676108844,0.0121039323948986,0.105525249109605,0.259625126013997,0.104519155645446,0.155989639815494,0.0793863987270382,0.113578730886093,0.0744003501292692,0.0671418742955594,0.118987073868883,0.0622911470958263,0.0946142372553329,0.117456526721265,0.105696507341115,0.0912933118045921,0.0835234024553817,0,0.416800532540876,0.429894291761147,0.38424039406767,0.430900126491908,0.538199400348706,0.809165470518197,0.356114506520575,0.268170214679678,0.0934947802214561,0.178642688794226,0.157482589171115,0.107314973565566,0.171819463728057,0.122966417805183,0.116459349793257,0.119485127901349,0.134774528609315,0.190560348396854,0.192375042562623,0.111368235739374,0.118365305709774,0.321678275709374,0.190105298277389,0.110134963702045,0.174541292485912,0.290961962350444,0.168347793462876,0.168655465186115,0.190097613642499,0.446425881214244,0.189511051191392,0.182399331569673,0.2307709253541,0.237873970862838,0.15819808129846,0.155135664553159,0.0596377566875331,0.117008895091516,0.0988534174552573,0.172444221378262,0.0615855009422018,0.166765826894174,0.128560885795183,0.0681097874376436,0.0495656346310325,0.057220783603303,0.209342187493532,0.09793963021372,0.338032059688543,0.596803482574107,0.466565361690149,0.535294928078674,1.00411318442282,0.961441171291297,0.825829900852567,0.18917461371921,0.0820591034315969,0.0142799665788509,0.127510702681579,0.00921268633837077,0.148758002730483,0.17925670484981,0.168688845940627,0.0137959456717606,0.0730454415496376,0.123342576001927,0.141214187067841,0.0677828470834603,0.0879410883228949,0.154100381174889,0.0673143945711941,0.0516406955227011,0,1.06768063465072,0.384111540071922,0.119932144882921,0.196707330671313,0.465496539066595,0.252437193995429,0.0692325157746154,0.113565791499613,0.323055393416074,0.118000991129328,0.168499618055763,0.255987943383181,0.294136539097432,0.185092131207485,0.197484320580128,0.495054874471787,0.306758245793653,1.38474126666039,0.801873893352151,0.856579212361576,0.740058966134552,0.676723021054718,0.652715617867614,1.0087777463828,1.15465481955483,1.15602470447343,0.978401398128901,0.673039347282699,0.701518100135866,0.556701297153793,0.270659522581389,0.349180430351924,0.186757697143158,0.279258092319834,0.28599240170915,0.279785951594436,0.278524358849363,0.270774740247539,0.374146422920875,0.283573894110047,0.332382714683794,0.392770318738898,0.178329420458255,0.575577806486958,0.364611351955254,0.591114055469941,1.07350144885764,0.694843022892615,0.427589987101835,0.465787124550104,0.0887241698228634,0.18605301306242,0.100474142385736,0.284689094029979,0.107933064246719,0.301045171772631,0.126951047145341,0.201994866365306,0.284815022334394,0.181887047893095,0.156110726457108,0.180781147726379,0.242644276966381,0.205256945892174,1.31112916665943,0.925183751238343,0.705993734925467,0.291969740264341,0.478307030118769,0.384105133365375,0.380842516749156,0.778942622052658,0.688919476019723,0.896341626837115,0.858840004821862,1.02708269051544,0.911795729218498,0.67428332875111,0.454429236818601,0.543307030696556,0.619943370398181,0.618177173931193,0.388121389780145,0.301617103093856,0.194183515078127,0.374537229710648,0.229190487065501,0.163448947659489,0.341924411545825,0.300206310321594,0.379544740649059,0.221603657244774,0.715660441717097,0.6652318863185,0.56060705418834,0.337263271093184,0.430711700657064,0.348492837529946,0.244337894191274,0.126614626509465,0.267259887528231,0.421612441763138,0.700521936373083,0.379049109878548,0.545513185931525,0.315871744714903,0.207652552699027,0.28951924512201,0.230442647180531,0.241689914447654,0.136122121233678,0.274155303721042,0.256971527750593,1.55143154228424,0.832855331089868,0.290304744600668,0.25154758501167,0.321951101666277,0.335132636357086,0.788616791403508,1.39802130989505,1.24492486894456,0.845882421369084,0.745490439227697,0.624358167577901,0.631859416386755,0.18386179576757,0.289627438032804,0.222084824701974,0.392095703044377,0.323654926461421,0.317900786574532,0.264329814888244,0.235874313725881,0.23962319186947,0.232993653748273,0.317597546014719,0.292699596706084,0.221821065639973,0.632589682022514,0.221161249241137,0.137709012921719,0.2435517612753,0.182648580702924,0.274837262156504,0.239945180540881,0.219228057391336,0.169014901651694,0.187072546593553,0.111332930099734,0,0.0800262488166983,0,0.0414446306863759,0.0920881665559979,0.242176863554574,0.0957633415953629,0.154118303736768,0.311953092201207,0.578232589188563,1.21019240971439,0.536425058560363,0.419940545457575,0.423715817804257,0.408244097195478,0.362568812242949,0.449823097282153,0.616666215969183,0.780596569712217,0.680534999336568,0.609286336822323,0.595685172653814,0.600904931147726,0.479923968017731,0.240598205149851,0.239440029686965,0.202328244123039,0.188024580004222,0.202680283357233,0.208765657827911,0.270810872944143,0.272614028975265,0.269824184100189,0.321376366053982,0.183339771789022,0.361736292413512,0.662008293029547,0.344931676493896,0.294495640768543,0.0951776520403699,0.0823767347769059,0.0446551813288102,0.0489920280235949,0,0.107071742630976,0.118697580676286,0.185851501693926,0.0917192722220512,0.0406514844105208,0.21140594660006,0.143435029113111,0.19686791285994,0.169865996291174,0.242630127945253,0.293455553323484,0.2516160521905,0.241421327280013,0.464758542662884,0.33080503263387,0.377990082512772,0.341657290327174,0.625938550959709,0.382495578903074,0.338320334935987,0.344904141185576,0.691567444330399,1.36024283137275,1.21916920521029,0.663226079548994,0.637253401081343,0.688674180981128,0.454291474671688,0.25457534149123,0.280392715288425,0.236024091214955,0.371836281170146,0.202430264324264,0.245466630641742,0.332069213117812,0.215949481990486,0.285363686151901,0.229198445954051,0.350302444733003,0.35938095385043,0.409429795314721,0.463757281662483,0.330425467578213,0.373426222905178,0.315166228448329,0.254476754779764,0.158178017426888,0.385351383786766,0.148115428231713,0,0.135927182575029,0.163562116232764,0.0287278082527603,0.107155640637348,0.251070298483063,0.124854365959402,0.0982801965047788,0.367414725476398,0.234467811933733,0.19918374778145,0.235528149615462,0.520194326291194,0.895158258133733,0.460997385824104,0.282461513471246,0.552191928106165,0.322258382182231,0.266894973410985,0.348655909513902,0.652189733074318,0.668231420013032,0.556842682276179,0.701433193941771,0.695785601805306,0.611084401051974,0.56734284622461,0.192501216680598,0.247759302939267,0.231725513328879,0.295347861441636,0.275654261122298,0.202409602271534,0.317418243405888,0.196881962139043,0.285500514557568,0.326609742804947,0.416379932993448,0.540163723452262,0.417415373519404,0.290037603706662,0.374757928128606,0.348415383809704,0.196781932363635,0.150961438274015,0.119083005786801,0.177407415841504,0.16644724376744,0.2528689890263,0.270431861126783,0.042980040993396,0.161860344670386,0.353544999174459,0.30648034691102,0.0617808872708143,0.104470124947317,0.073420519919148,0.0229677556707219,0.110155582876575,0.0898745792424634,0.0124946900395071,0.190289779298199,0.116379147748176,0.154737445729091,0.0928436223884844,0.0403098541383253,0.114685958811319,0.0231671386697377,0.470607483393786,0.680984886471985,0.423236546510821,0.583153128914767,0.479019954661101
"49",0.119450991373563,0.223349696298757,0.0780611660050693,0.0624163661646579,0.0896829299456862,0.140057583099691,0.142434657751012,0.207616575336764,0.11432795472475,0.159410720119833,0.0919512985107099,0.189580499971477,0.105328806424873,0.448740107737822,0.323773298690386,0.241650066510548,0.268664030752472,0.378850534186603,0.466561717302149,0.264464036189024,0.504630536408815,0.294384383962441,0.113062128345958,0.0408267281220279,0.0785177338252225,0.261594679871874,0.486795781324703,0.293549499152476,0.186291478949924,0.0362681822632799,0.207636263874158,0.153062909840493,0.220961113967637,0.313119373437622,0.460819313165453,0.828662976357998,0.53975605904513,0.551698814378804,0.451401412269422,0.296246149187568,0.26512175612151,0.293703825886184,0.249799112637308,0.311262062642588,0.211297511997601,0.619347559132049,0.187626028610396,0.119467786820826,0.0265637092463568,0.0266517377423475,0.145331690048645,0.0849717435871048,0.108574387107604,0.133949024635022,0.093747243777784,0.0997210282465689,0.154323081058588,0.0455030954549185,0.117424699594078,0.065508148592937,0.123154508111953,0.370589088903529,0.348840141350736,0.208186123740155,0.470775986274902,0.224420911135874,0.262249971188028,0.755017788159337,0.711381013818609,0.309335036230882,0.228866505865929,0.207252291570188,0.342535448710353,0.220809939138726,0.263131448677069,0.475547515481847,0.17757391555196,0.254696512717166,0.304789709217423,0.153092276733167,0.0823411185224952,0.210117855900584,0.264109009623652,0.364370547932477,0.572536424311686,0.394634354781753,0.325344618562733,0.356466234601118,0.264551222834102,0.286319904458671,0.229585587389931,0.346894275890543,0.390021064618004,0.123909556610135,0.107715214256657,0.0369710825753794,0.0390160197322467,0.116928506160009,0.21185208252588,0.0157494983133514,0.0855920825765964,0.184934416698543,0.0835461650592816,0.0871795860342115,0.207511230676938,0.134113906825321,0.12731431998244,0.0140838959001069,0.0840471873245355,0.0900841338770021,0.372090634114548,0.155903648455985,0.374461217990019,0.45401703431185,0.326623178186251,0.397204617937383,0.579580160996067,0.41617780970741,0.277801906084676,0.109249093813474,0.304241726540447,0.393576370703605,0.280565603234712,0.493496431609203,0.271619651103356,0.0526796498655791,0.189196435300143,0.0203417442231336,0.075772475186736,0.179581769492591,0.297669081536421,0.302460094934372,0.604491776602423,0.361327459622461,0.424994743455595,0.350753964850189,0.399410117224245,0.630719284065866,0.597063032246965,0.246159390003627,0.293307305107369,0.199475382684475,0.271127123814216,0.139355177987918,0.0190230214746864,0.0372734261167606,0.158379728679488,0.170325703598393,0.165070843734358,0.0793950196188896,0.140902021275095,0.113099979112595,0.101199458932353,0.0977787221582133,0.15490524342972,0.147996488978688,0.109716137526827,0.428914164619538,0.153616908042873,0.257336296048194,0.283721976640559,0.318727592468432,0.700177264205405,0.363282323886575,0.336147013927221,0.181566250239099,0.0901086689106208,0.173575148866367,0.137129339094105,0.338618258728343,0.449633644136772,0.177671579465412,0.293752304609247,0.321610240531298,0.188624091565927,0.397360614166445,0.146087336330808,0.255559975255684,0.194612327384051,0.362847528140457,0.535553739821791,0.491871230882003,0.533677461796593,0.28914272940749,0.332381722176846,0.317721728918692,0.23036270708049,0.254952070217807,0.323738213221512,0.15417638265645,0.240416764626971,0.197822338732401,0.10275017636084,0.151536500371696,0.0915104538516042,0.0840953286100777,0.173015022393415,0.0672317296934445,0.168534640789242,0.195061328082818,0.165649796076922,0.14296340440263,0.0128699368859209,0.091873754136979,0.221037839329007,0.127649853837488,0.180248550873461,0.257805564229782,0.552701855308721,0.684829633685098,0.563153649526532,0.641537644635068,0.718953020962226,0.243192088574733,0.194269018517729,0.131103884439116,0.264337729996569,0.1934934466737,0.256266202504194,0.25722872319313,0.249536005922924,0.193917808251577,0.176727995370852,0.0879430783595229,0.359088381304312,0.204113922878598,0.319507453550471,0.573074730510781,0.46673980715115,0.435954446436034,0.73693939872731,0.463545364321403,0.628191565176759,0.383981346843994,0.327540493576415,0.274640838679653,0.256146620418168,0.267889476073344,0.0942032101335516,0.0673050138302553,0.1428234043593,0.167342588461158,0.111987120615022,0.156162516545749,0.0300039536009316,0.200365563589943,0.109809425110176,0,0.0999744222541702,0.0731657237866205,0.212371823563602,0.139802595496694,0.0823495135978655,0.371928797353411,0.25897295728305,0.258943324009996,0.195577697562974,0.64928108791152,0.408409424933572,0.315832814064889,0.498290951869859,1.00627832121498,0.467432311175815,0.365890909640119,0.320042926917149,0.253383331306225,0.472791979703546,0.272377973929857,0.280111523477141,0.188802832842279,0.182189862147926,0.204227475955328,0.176237113170607,0.556946700741591,0.669527261781973,0.466937739610629,0.308237578970799,0.507562714687421,0.329428620479862,0.356673005888801,0.460751789648747,0.262807285817599,0.211407502521239,0.145857752426898,0.301816306852992,0.301075480211512,0.151966805405089,0.110586183775323,0.112391399963827,0.0990907405537176,0.0432940377485495,0.0396787801409798,0.213222737772744,0.234144652100366,0.106609146006892,0.11192505428126,0.144947339489373,0.0918694141730498,0.208389822401698,0.119077664953695,0.057399343583115,0.364914316291723,0.244146587670686,0.295873347154963,0.489771776000734,0.429491784726674,0.277391278609309,0.658948163769786,0.551767192400765,0.231268275608559,0.279759942640856,0.114536011504771,0.144018499774789,0.18663873291308,0.186167179520096,0.234111706701153,0.563713193260939,0.528346484758408,0.418629829854153,0.184947306227813,0.362305292005542,0.18070497155812,0.345735969105215,0.369025183357419,0.249254263361217,0.146639829815626,0.241190725606913,0.167682182637362,0.329370601640725,0.640831813522055,0.382789566298146,0.296642686413759,0.410060317142665,0.251749624356652,0.195934247380718,0.250431821013555,0.237582620309255,0.108316044279479,0.262644342844479,0.186391874536209,0.161236862537565,0.167177764226683,0.1797226924575,0.0992264868784907,0.133805775618015,0.11368745143606,0.136974326910817,0.163986534097824,0.136474284750792,0.187066218293595,0.2285668953737,0.402602139315364,0.908271585705301,0.581767573287365,0.445152520767355,0.431557046935491,0.589529334489341,0.259706530258282,0.422657805502304,0.324016889785309,0.567723127210821,0.581781747326445,0.766408807389233,0.584381153301433,0.640054773629499,0.616962629143222,0.446832324347572,0.462157004488535,0.2303137800581,0.20253142111468,0.355744622001831,0.269531680688301,0.452727409163988,0.475455112944906,0.510323694627208,0.441244163295311,0.827048255343434,0.772632563578802,0.499626518139318,0.376898626531561,0.436102827575966,0.602149798707972,0.355603789562111,0.251598268963888,0.135627528796272,0.220924312393974,0.0416156636711316,0.206021486501494,0.185471036909438,0.0430512874599868,0.148149846315612,0.0478785599827652,0.0767495194235191,0.108086307254091,0.144280960458233,0.0962634588026435,0.220229705337824,0.18062513280067,0.455836368152425,0.23324950956232,0.156428109224364,0.281068864240951,0.491739814694504,0.474762118719499,0.447511897119529,0.229557510817443,0.109074981277163,0.168920943120118,0.253880498842449,0.17913097570262,0.380033011568841,0.213782351534759,0.315832789507607,0.63097666756926,0.252905657529994,0.296643156024883,0.288616734699241,0.324289943001606,0.351791750871652,0.557840537497875,0.419058622795215,0.450026039173182,0.500452145492962,0.358677158646285,0.347380635573117,0.298137524817877,0.356180983888759,0.333689424601466,0.165804592113922,0.148996882217783,0.176437406560067,0.0463936830083021,0.150170477433786,0.0535658695104614,0.107900019337157,0.00743478795127704,0.143829816445052,0.133547247886101,0.0749270350354554,0.136846402692041,0.129954378559505,0.0437053292117229,0.0811291213918562,0.141967635164267,0.114339878301306,0.0551682096519987,0.315563780811697,0.164802451270281,0.265472581080603,0.294758958855423,0.561956138669269,0.20692595399356,0.909356244043922,0.52748671806055,0.479494475364418,0.292882077824907,0.185041449064088,0.415651679250811,0.483590377894853,0.401619250071484,0.445799787341626,0.574658501964329,0.410127076445188,0.399631498609431,0.222236458454371,0.374252130766869,0.30960053418243,0.342809509298218,0.46393758543589,0.399435613399849,0.350293625606576,0.258609867332207,0.436435825901172,0.356797652869459,0.402268310667265,0.202049584625812,0.319410218091708,0.110547890000902,0.117716597283769,0.197817566955618,0.0701173948741086,0.196560231393348,0.127471896511738,0.0940843450488435,0.17995721122061,0.103535961644471,0.0848664209909286,0.126285408717545,0.00359731488559967,0.149059699742427,0.21577292386002,0.00691002768094161,0.119141780755929,0.444999672037124,0.198043839792578,0.471878973531498,0.595607076493859,0.47682353298358,0.574527380323687,0.385159277943381,0.379504188758051,0.483797355888913,0.283969993587334,0.325526963418229,0.496342399989952,0.769028757371741,0.422217611361114,0.538979631429105,0.473482494133914,0.378085307414251,0.252547064087428,0,0.236137560994423,0.225983638451716,0.19558160524577,0.299089476210894,0.209594305565153,0.432763514380962,0.50100857888824,0.305996252406251,0.315662768727404,0.428520728481369,0.359842489282984,0.21927330690878,0.180892903642846,0.130325307138943,0.0732545666258427,0.0566524558634317,0.163186773034122,0.201795555673601,0.11308039090259,0.182100717007772,0.16574834646721,0.163264063323256,0.130221669668331,0.109870806480597,0.072632591391404,0.24008480131993,0.0360021718267216,0.0848717732996147,0.168476383370038,0.384862410763898,0.309768456233432,0.591215847362823,0.66356592473127,0.413714190954417,0.43125418464654,0.277966988175519,0.176253618339996,0.109610630699036,0.165792495332414,0.236019356351901,0.317923299986077,0.441296629902646,0.205070427765731,0.121143326774823,0.135197869221488,0.0989018400276861,0.380612198563334,0.393972033211499,0.426220444732172,0.284965430280708,0.363550788258902,0.347727506013861,0.577504035710845,0.342460495187518,0.506259494009658,0.210233369313569,0.41121086883874,0.674871418040654,0.370184232799109,0.29034064901557,0.212494499472133,0.109351721295118,0.272126176646631,0.0933479758927396,0.198899001616664,0.100200558287353,0.0692064830694668,0.134505203579015,0.0945027311771175,0.169612383722124,0.154609533662616,0.157437104598045,0.143995897912359,0.10102864533042,0.124462731017557,0.00529615857498211,0.0207843529464494,0.531014993550364,0.472309952469901,0.481027878305973,0.74360514235377,0.551395410036119,0.18367876072382,0.313170623688987,0.344482169456297,0.250276343535085,0.0774427323897444,0.248741595887048,0.304717753207367,0.236810320397295,0.293148022446471,0.253871106022061,0.166159635772682,0.0607461687168797,0.0404462567933431,0.209637973694161,0.206547489907723,0.133462854270175,0.369252639271725,0.406567397423466,0.265662170104664,0.341394537445526,0.568653183387758,0.434731643596449,0.276630913661064,0.256787032210943,0.368029189434817,0.361694118078938,0.22542984894959,0.0468750676103466,0.0993788292712948,0.0537361925132521,0.198721257827098,0.0388247771310007,0.0412175671305087,0.155794839463949,0.0523240270387958,0.0599283211711258,0.142753099703102,0.0723879260338671,0.052598404096948,0.102370229445961,0.138402628433224,0.104390668674363,0.000658192961471557,0.296368768978198,0.324036735835049,0.73765529315545,0.43925301876876,0.53217454318257,0.253468055023615,0.466820761141447,0.635138911787278,0.226055674803992,0.183765935987085,0.158984300752425,0.161560326696427,0.26204497717027,0.183918836163897,0.428815189098206,0.167428856779533,0.296787098787487,0.19954712790066,0.296260437468546,0.317425326158014,0.324422517454983,0.239088995329587,0.436065441648458,0.186253242284485,0.24497581532473,0.190024949759731,0.169742310389612,0.705802891761924,0.245566506761488,0.148280271803987,0.264958846622153,0.536511489440266,0.181489459851068,0.132566209944092,0
"50",0.109447996598072,0.121930171423607,0.0401431592543091,0.221800015040173,0.170136174509443,0.144070045587782,0.114978419965803,0.153310469750561,0.247273433772936,0.13351953256342,0.170151940845343,0.292865235408331,0.335369714638718,0.304084993259734,2.87587357961026,1.87452423317338,0.689581857940334,0.334222815632007,0.070465162541425,0.236313447047413,0.265054660626245,0.45315115533065,0.771199608302856,0.683317903176908,0.2890048254262,0.269530925736712,0.270405343120007,0.19511650853948,0.631481918796928,0.288284234009463,0.17887387308075,0.266193538266606,0.0970180448343894,0.327069730610936,0.562137369648117,0.475523165213479,0.580470121412184,0.429883085530467,0.110094910161068,0.157051129124861,0.130330610229523,0.0991005941283251,0.294503594688205,0.35308421584036,0.217741893621018,0.208810945446593,1.97783514220468,0.888599541603306,0.647261791376183,0.182637441339222,0.10448560685661,0.250472248500026,0.0937389908717554,0.0916037005953762,0.142500989939192,0.0871917553925258,0.0533563496018363,0.198221587134372,0.191898708594826,0.154736163120322,0.284675278776521,0.40729908276402,0.212228009111527,0.192465285357005,0.141736314057545,0.274990800077535,0.246436249303136,0.235534134740486,0.728718085239311,0.534941451886244,0.181364241180123,0.169079364054437,0.197295675508549,0.161449207029236,0.271742808828269,0.249161627626358,0.157668936910359,0.789497781450356,0.345831525932284,0.135393124546423,0.127800700223474,0.272062745445683,0.424106960949301,0.689348639373672,0.568710248274742,0.16137166439989,0.267198521671022,0.628100522617543,0.366020236985232,0.921101540745894,0.416341310997782,0.682376126471321,0.963327073356008,0.680540993199905,0.362760796725735,0.173785487252697,0.131575271799891,0.172628457023889,0.144907419423371,0.0549618984644523,0.0655451570010896,0.123338336501825,0.0504026840915883,0.0731560682646185,0.138410736246508,0.119341113111529,0.138372474059942,0.230039469857142,0.406482963198072,0.328569119047713,0.275583246971276,0.154212940150205,0.457042734184456,2.96460771200503,1.85898850516763,0.2461094904763,0.249124118884515,0.111058852206398,0.430768545617202,0.510350018400809,0.280164598773565,0.26212740158985,0.265373478267513,0.272533308470775,0.337290237302167,0.287670928239667,0.321777482733497,0.158684575212003,0.31163858248465,0.342233189733207,0.487865907153178,0.56020018282167,0.393423026669079,0.466222196913099,0.374800513561832,0.443300665023318,0.391263310231685,0.632886421629108,0.204962012835487,0.360146008645043,0.172778625401267,0.194706495400654,0.366000171895879,0.200197765286233,0.0960944131502628,0.159122559110921,0.160459307358609,0.106233301970084,0.204677064276728,0.118933606165221,0.116314980484282,0.132016503924383,0.138986037715648,0.183439547382365,0.122361735679056,0.110963122966526,0.170617822122573,0.298096069996096,0.230802257136647,0.251264994088906,0.221141013041414,0.230667515641858,1.6079574963955,1.20200224579432,0.238658563349298,0.54632695809376,0.193607156045026,0.151341741728621,0.173984745901664,0.273530763952993,0.220923625411821,0.216656129309929,0.237207300598275,0.189425958112852,0.283096112610462,0.328058420444766,0.127840994042828,0.159253242558182,0.157617534290129,0.211943975889203,0.459417637407995,0.290432507240735,0.32327690906734,0.258210988531394,0.284302845707728,0.201944596628119,0.207148178274104,0.157403354659094,0.422447854750598,0.355025879416605,0.196815806905682,0.300335037874901,0.199240819672655,0.195273594679586,0.211960280657775,0.210106003991491,0.135744087960049,0.136601167170212,0.0560106257443732,0.126755969034127,0.136841108193322,0.19728356436809,0.168112105874565,0.195097031368482,0.268902844621236,0.340383391605006,0.347176547381643,0.182278693679431,0.196672420938356,0.136637747178565,0.0713001646327872,0.0715832695184658,0.178230910406404,0.353746271317441,0.375943046593625,0.297244446888896,1.12928617780324,0.19477349979882,0.16705801371895,0.181770552866315,0.205341250885134,0.14502633939203,0.180809818779776,0.183558656818745,0.180289857208265,0.147635085858198,0.18620572690915,0.322623275710746,0.57192144584255,0.560597436322622,0.337157181911917,0.296816583024673,0.319623574459464,0.531797556343376,0.450203599821214,0.401407953358114,0.405489047445652,0.312166920549172,0.382909071009922,0.310996558399358,0.22905883019599,0.227950396014791,0.165060807612353,0.133236636020273,0.143723633699432,0.0963843336330549,0.204241385638922,0.167787395279147,0.188043576001795,0.154013523116863,0.0654637122312467,0.16734048505726,0.721830665556889,0.332561892918093,0.25086945111975,0.170812430148609,0.310191445504208,0.149764050726977,0.178103014056602,0.350542430032884,0.662449502046292,0.754473149240858,0.561096411080169,0.282138989330306,0.249136278169604,0.471812693126111,0.201988097572903,0.184385629677934,0.111424103365659,0.269072672928851,0.344773171069155,0.309350373363424,0.287826611418225,0.117882593986901,0.312850378348279,1.04691089940719,0.704911572699228,0.585525075064803,0.577883691733772,0.524145182516253,0.314487471630523,0.150440327964456,0.22383024492285,0.16836355773986,0.309103703935319,0.335098399263442,0.401166862868519,0.322220405004112,0.28908997044833,0.127603612064818,0.181650626681341,0.152098226596054,0.140373502754748,0.166546097322088,0.134415408149766,0.0676278701915657,0.00661108447103372,0.107587180996296,0.0389531016199514,0.157004055947873,0.0731925015682373,0.22961334912931,0.40910027382907,0.205348133510441,0.30582031035216,0.618186253160327,0.71407665492392,0.553460355902394,0.0815569649475546,0.0990043962636013,0.239743524206829,0.161591217844236,0.116220480995924,0.100604171992121,0.0659905583058929,0.150949891171269,0.241121678115486,0.058824193378844,0.111767332403692,0.196390633520856,0.0365170101492513,0.133852632874916,0.0965553371076665,0.149231582363206,0.0836227900351233,0.0576032333239073,0.191533965696355,0.131671108338056,0.0746398525242147,0.150836453143822,0.128791730839768,0.0928895381427671,0.0403009446885381,0.181746983180899,0.0513736002869872,0.10939736159766,0.140815014759324,0.138775853108975,0.0779831684334083,0.0689071412090003,0.199881020229333,0.19089471005518,0.125818565543302,0.163342895067944,0.204390899384766,0.10072441905635,0.16469032894621,0.070808623362977,0.06588529219186,0.110706999135778,0.105851782786115,0.247312290786209,0.272228919785541,0.124935322303221,0.175503281883601,0.103182643139643,0.228975308645363,0.283888515222754,0.184235856488456,0.142314032803552,0.13193657587046,0.142771845410082,0.16853267972162,0.177296241222001,0.136821414218844,0.2081098826454,0.13777457440114,0.125324136000191,0.161818927027149,1.39990780469942,1.58354763618767,1.03533896719011,0.989291191203463,1.57894126141663,0.562938775982288,0.524779782858335,0.462716562199471,0.517134188888104,0.2737189636634,0.549847585522979,0.569297120564175,0.532056151712182,0.732277998551494,0.214913803901459,0.247664003758558,0.0628509378665369,0.117366998528964,0.148349732422781,0.190641914095056,0.0175609467056671,0.204468493691097,0.0935978959101247,0.189187241421664,0.144083609076201,0.0911694907454341,0.175370104817865,0.0992169266639888,0.321506325609979,0.255405253031879,0.105559700340282,0.130645788825713,0.111159663001668,0.129180740156317,0.169040901328752,0.364199576834944,0.298117017538675,0.290264920906484,0.153139671631225,0.27081313505824,0.203277876006363,0.847153749631692,0.578466912292115,0.367162961222108,0.286295966586828,0.229306420604184,0.134346215913434,0.321431758327619,0.359737764799962,0.98969032044005,1.01196825900448,0.874614257894143,0.667014946155664,0.519784642683239,0.458603342970198,0.615518390617905,0.329667315326324,0.509790898431574,0.532808859531199,0.431851904247245,0.378110342191916,0.669545069368323,0.37671974565294,0.287032534960941,0.170090112095188,0.107708389862352,0.225340589796705,0.144299193836408,0.200118201170993,0.133883356542179,0.148988581646944,0.18295419910852,0.0377155443928111,0.160922563523807,0.0847827746346244,0.023064212179905,0.248201000989493,0.305659283968303,0.0832465027111733,0.216847996280424,0.4366853586702,0.211112418016938,0.144592746627924,0.510742016533717,0.591938053304191,0.926955995871926,0.53082658540047,0.538691000336809,0.439401759758645,0.313047511297323,0.307533517710945,0.366615396113209,0.49693499808796,0.250658377843725,0.377437882828449,0.326059317831151,0.238676056987912,0.338305462926997,0.368593330360984,0.365260395296708,0.461791807808432,0.675169111809838,0.296172993744129,0.34909165667589,0.288025897497061,0.685232082104841,0.511833220244912,0.658780204493463,0.557115029772026,0.243655125241425,0.360777496662125,0.13464846072794,0.2079823831199,0.0916639579621415,0.172433137579955,0.19571880674411,0.199050431665723,0.0875548040249437,0.144620360007745,0.139136173697637,0.148965931977117,0.147314933246323,0.16271849782195,0.348039480911292,0.241219522732593,0.205311623225651,0.255034179729266,0.188295869784921,0.140857388153573,0.418539070844045,0.216905713830129,0.505067331429242,0.311935327350597,0.409735077417595,0.401149020324204,0.541321824092053,0.84770795395403,0.767518895812504,0.698711693460512,0.610115449722173,0.344149984078394,0.348342880444867,0.267249266157135,0.219480505195841,0.141675657222464,0.305838360863931,0.489223716708717,0.445429999803134,0.293259175430698,0.168529278377324,0.525908353760029,0.40795836256685,0.476397420099126,0.459152320664457,0.514815465069086,0.413446133969938,0.470804532985108,0.382630143564063,0.369636720245729,0.197337171129719,0.246207862900717,0.0792972965804996,0.146593930347037,0.0974632847035873,0.166046872809897,0.0770035102171037,0.0324325296774089,0.11739478846832,0.137805632370799,0.0403038343093717,0.154271898779542,0.0902345930279814,0.203548101805167,0.148323548817604,0.274641897613259,0.269495112207282,0.310672745428686,0.628913665921407,1.250708531645,0.389863683755743,0.49147021956486,0.479193283562589,0.250240105305198,0.220337250411716,0.232906660396352,0.159596986904471,0.934935191222495,0.615373760808789,1.14571711117927,0.778532039906244,0.247592536271182,0.22497625086239,0.210789339428515,0.377937567791154,0.45421590111408,0.452773146732239,0.370906599942357,0.96001589483225,0.681893564426165,0.883410543781962,0.743194653930831,0.42088386835773,0.541057056363757,0.463005003686444,0.330175113032855,0.485003352567457,0.264463245064589,0.178934733635699,0.236106219884811,0.169049690843201,0.228375381275806,0.0651490791878942,0.142106394634028,0.111420804800342,0.0825790101665993,0.133134734460347,0.131388223108266,0.175291929342902,0.18981181342639,0.118017452967878,0.329738821086954,0.0600470592189257,0.143179153580417,0.404846398366418,0.127218855433796,0.74467406918118,0.760379941231337,0.743535791802748,0.58588264764921,1.0503326361256,0.294541430931594,0.295601725558469,0.171049212234166,0.484196610501853,0.685118246643575,0.408653327110037,0.271408308284259,0.300395649996132,0.291782780874444,0.398732530065704,0.358840076221956,0.350217873836006,0.297989620725631,0.338722310489517,0.98152710977996,0.625345063324394,0.51528083212847,0.561516100568829,0.540268507340878,0.589488204347296,0.565990841992544,0.360685031150714,0.484596528393299,0.29896403115503,0.251876465342473,0.441013394914813,0.397034907157874,0.152339367027978,0.0545031654257357,0.211905698946992,0.060032412148068,0.151772008065035,0.0996223221869599,0.120023870734304,0.202837009641488,0.104261199953519,0.24600496046772,0.183944652811435,0.252486047710759,0.137913224526698,0.320966326379448,0.481473855067221,0.179239834676445,0.272303245167886,0.492465882326148,0.906487246912348,0.349094714502576,0.333827772314714,0.244531850185368,0.37512933628951,0.186071173050796,0.286340157585166,0.390708098877392,0.28537420651593,0.0735323202365687,0.123249566937665,0.189257263027705,0.266492128168467,0.254321112181792,0.646634382967785,0.429134957956159,0.780274550598485,0.536284031913588,0.451697932071904,0.373223778966227,0.439487831486418,0.452281939290359,0.371901985531544,0.326423593986989,0.435204444449738,0.565277078945036,0.408612720814575,0.273186921842049'

