##' Plot theoretical and estimated variances of Wstar
##' 
##' Given a vector of Wilcoxon statistics generated through permutation, plot
##' theoretical and esitimated variance by cumulative number of permutations 
##' 
##' 
##' @param Wstar the vector of Wilcoxon values generated by permutation
##' @param n1 number of items (SNPs) in regions to be tested.
##' @param n2 number of items (SNPs) in regions the control regions.
##' @return None
##' @author Chris Wallace <chris.wallace at cimr.cam.ac.uk>
##' @export
##' @seealso \code{\link{wilcoxon}}
##' @keywords plot
##' @examples
##' 
##' x<- matrix(exp(-rexp(200000)),nrow=2000)
##' Wstar<-wilcoxon(p=x,snps.in=1:1000)
##' varplot(Wstar=Wstar,1000,1000)
##' 
varplot <- function(Wstar,n1,n2) {
  n.perm <- length(Wstar)
  W.var <- sapply(2:n.perm,function(i) var(Wstar[1:i]))
  var.theoretical <- exp(log(n1) + log(n2) + log(n1+n2+1) - log(12))
  plot(2:n.perm,W.var,ylim=range(c(W.var,var.theoretical)),pch=".",
       xlab="Permutation number",ylab="Var(W*)",main="Estimate of Var(W*) vs number of permutations",
       sub="(dotted line shows theoretical value)")
  abline(h=var.theoretical,lty=3)  
}
