# Copyright 2021 Bedford Freeman & Worth Pub Grp LLC DBA Macmillan Learning.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# wordpiece_tokenize ----------------------------------------------------

#' Tokenize Sequence with Word Pieces
#'
#' Given a sequence of text and a wordpiece vocabulary, tokenizes the text.
#'
#' @inheritParams .wp_tokenize_single_string
#' @param text Character; text to tokenize.
#'
#' @return A list of named integer vectors, giving the tokenization of the input
#'   sequences. The integer values are the token ids, and the names are the
#'   tokens.
#' @export
#'
#' @examples
#' tokens <- wordpiece_tokenize(
#'   text = c(
#'     "I love tacos!",
#'     "I also kinda like apples."
#'   )
#' )
wordpiece_tokenize <- function(text,
                               vocab = wordpiece_vocab(),
                               unk_token = "[UNK]",
                               max_chars = 100) {
  is_cased <- attr(vocab, "is_cased")
  if (!is_cased) {
    text <- tolower(text)
  }

  text <- piecemaker::prepare_and_tokenize(
    text = text,
    prepare = TRUE,
    remove_terminal_hyphens = FALSE
  )

  tokens <- purrr::map(
    text,
    .f = .wp_tokenize_single_string,
    vocab = vocab,
    unk_token = unk_token,
    max_chars = max_chars
  )
  return(tokens)
}


# .wp_tokenize_single_string -------------------------------------------------

#' Tokenize an Input Word-by-word
#'
#' @param words Character; a vector of words (generated by space-tokenizing a
#'   single input).
#' @inheritParams .wp_tokenize_word
#'
#' @return A named integer vector of tokenized words.
#' @keywords internal
.wp_tokenize_single_string <- function(words,
                                       vocab,
                                       unk_token,
                                       max_chars) {
  token_vector <- unlist(
    purrr::map(
      words,
      .f = .wp_tokenize_word,
      vocab = vocab,
      unk_token = unk_token,
      max_chars = max_chars
    )
  )
  ids <- vocab[token_vector]
  names(ids) <- token_vector
  return(ids)
}


# .wp_tokenize_word -----------------------------------------------------------

#' Tokenize a Word
#'
#' Tokenize a single "word" (no whitespace). The word can technically contain
#' punctuation, but in BERT's tokenization, punctuation has been split out by
#' this point.
#'
#' @param word Word to tokenize.
#' @param vocab Named integer vector containing vocabulary words
#' @param unk_token Token to represent unknown words.
#' @param max_chars Maximum length of word recognized.
#'
#' @return Input word as a list of tokens.
#' @keywords internal
.wp_tokenize_word <- function(word,
                              vocab,
                              unk_token = "[UNK]",
                              max_chars = 100) {
  vocab <- names(vocab)
  if (stringi::stri_length(word) > max_chars) {
    return(unk_token)
  }
  if (word %in% vocab) {
    return(word)
  }

  is_bad <- FALSE
  start <- 1
  sub_tokens <- character(0)
  while (start <= stringi::stri_length(word)) {
    end <- stringi::stri_length(word)

    cur_substr <- NA_character_
    while (start <= end) {
      sub_str <- substr(word, start, end) # inclusive on both ends
      if (start > 1) { # means this substring is a suffix, so add '##'
        sub_str <- paste0("##", sub_str)
      }
      if (sub_str %in% vocab) {
        cur_substr <- sub_str
        break
      }
      end <- end - 1
    }
    if (is.na(cur_substr)) {
      is_bad <- TRUE # nocov
      break # nocov
    }

    sub_tokens <- append(sub_tokens, cur_substr)
    start <- end + 1 # pick up where we left off
  }

  if (is_bad) {
    return(unk_token) # nocov
  }
  return(sub_tokens)
}
