% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/workflow_set.R
\name{workflow_set}
\alias{workflow_set}
\title{Generate a set of workflow objects from preprocessing and model objects}
\usage{
workflow_set(preproc, models, cross = TRUE)
}
\arguments{
\item{preproc}{A list (preferably named) with preprocessing objects:
formulas, recipes, or \code{\link[workflows:add_variables]{workflows::workflow_variables()}}.}

\item{models}{A list (preferably named) of \code{parsnip} model specifications.}

\item{cross}{A logical: should all combinations of the preprocessors and
models be used to create the workflows? If \code{FALSE}, the length of \code{preproc}
and \code{models} should be equal.}
}
\value{
A tibble with extra class 'workflow_set'. A new set includes four
columns (but others can be added):
\itemize{
\item \code{wflow_id} contains character strings for the preprocessor/workflow
combination. These can be changed but must be unique.
\item \code{info} is a list column with tibbles containing more specific information,
including any comments added using \code{\link[=comment_add]{comment_add()}}. This tibble also
contains the workflow object (which can be easily retrieved using
\code{\link[=extract_workflow]{extract_workflow()}}).
\item \code{option} is a list column that will include a list of optional arguments
passed to the functions from the \code{tune} package. They can be added
manually via \code{\link[=option_add]{option_add()}} or automatically when options are passed to
\code{\link[=workflow_map]{workflow_map()}}.
\item \code{result} is a list column that will contain any objects produced when
\code{\link[=workflow_map]{workflow_map()}} is used.
}
}
\description{
Generate a set of workflow objects from preprocessing and model objects
}
\details{
The preprocessors that can be combined with the model objects can be one or
more of:
\itemize{
\item A traditional R formula.
\item A recipe definition (un-prepared) via \code{\link[recipes:recipe]{recipes::recipe()}}.
\item A selectors object created by \code{\link[workflows:add_variables]{workflows::workflow_variables()}}.
}

Since \code{preproc} is a named list column, any combination of these can be
used in that argument (i.e., \code{preproc} can be mixed types).
}
\examples{
library(workflowsets)
library(workflows)
library(modeldata)
library(recipes)
library(parsnip)
library(dplyr)
library(rsample)
library(tune)
library(yardstick)

# ------------------------------------------------------------------------------

data(cells)
cells <- cells \%>\% dplyr::select(-case)

set.seed(1)
val_set <- validation_split(cells)

# ------------------------------------------------------------------------------

basic_recipe <-
   recipe(class ~ ., data = cells) \%>\%
   step_YeoJohnson(all_predictors()) \%>\%
   step_normalize(all_predictors())

pca_recipe <-
   basic_recipe \%>\%
   step_pca(all_predictors(), num_comp = tune())

ss_recipe <-
   basic_recipe \%>\%
   step_spatialsign(all_predictors())

# ------------------------------------------------------------------------------

knn_mod <-
   nearest_neighbor(neighbors = tune(), weight_func = tune()) \%>\%
   set_engine("kknn") \%>\%
   set_mode("classification")

lr_mod <-
   logistic_reg() \%>\%
   set_engine("glm")

# ------------------------------------------------------------------------------

preproc <- list(none = basic_recipe, pca = pca_recipe, sp_sign = ss_recipe)
models <- list(knn = knn_mod, logistic = lr_mod)

cell_set <- workflow_set(preproc, models, cross = TRUE)
cell_set

# ------------------------------------------------------------------------------
# Using variables and formulas

# Select predictors by their names
channels <- paste0("ch_", 1:4)
preproc <- purrr::map(channels, ~ workflow_variables(class, c(contains(!!.x))))
names(preproc) <- channels
preproc$everything <- class ~ .
preproc

cell_set_by_group <- workflow_set(preproc, models["logistic"])
cell_set_by_group
}
\seealso{
\code{\link[=workflow_map]{workflow_map()}}, \code{\link[=comment_add]{comment_add()}}, \code{\link[=option_add]{option_add()}},
\code{\link[=as_workflow_set]{as_workflow_set()}}
}
