#' Estimates circadian cycle by cosinor fit of minute-level activity (period = 1440)
#'
#' Fits \eqn{C(t) = m + a \cos\{2\pi (t - \phi)/P\}} with period \eqn{P = 1440} minutes
#' to a rescaled activity series. Returns fitted curve, binary curve after
#' thresholding, and canonicalized parameters (non-negative amplitude).
#' MESOR (mesor)
#' The baseline or “midline” level around which the rhythm oscillates.
#' Units = same as actigraphy your data
#' Roughly the average activity across the cycle.
#'
#' Amplitude (amp)
#' Half the peak-to-trough swing of the fitted rhythm.
#' Units is the same as actigraphy data.
#' Larger amplitude → stronger rhythmicity (bigger day–night contrast).
#' Negative amp will be flipped, and P/2 will be added to phase.
#
#' Acrophase (phase)
#' The timing of the peak of the fitted cosine.
#' Units is the epoch units of the data (e.g. minutes).
#' P=1440, interpret phase as minutes-of-day relative to your x origin.

#'
#' @param clean_df cleaned dataframe with cleaned_df$Activity.
#' @param thr Dichotomization threshold for fitted curve.
#'
#' @return A list with elements:
#' \describe{
#'   \item{fitted}{fitted cosine curve.}
#'   \item{label.cos}{dichotimized cosine curve}
#'   \item{cos_para}{`c(Mesor, Amplitude, Acrophase)`.}
#'   \item{rmse}{Root mean squared error between fitted cosine curve and data.}
#' }
#' @seealso [screen_wear()], [sleep_detection()], [extract_sleep_metrics()]


sleep_cos <- function(clean_df,thr =0.2){
  y <- abs(clean_df$Activity)
  x <- 1:length(y)
  p0 <- c(500,550,227)

  fhnonlin <-  function(p) {
    p[1] +p[2]*cos(2*pi*(x-p[3])/1440)-y
  }
  para <- pracma::lsqnonlin(fhnonlin,p0)$x
  fitted <- fhnonlin(para)+y;
  RMSE <- sqrt(mean((y -fitted)^2))
  fitted_norm = (fitted - min(fitted))/(max(fitted)-min(fitted)) ## normalized cosine curve
  label.cos <-  prob2vect(fitted_norm,thr)

  return(list(fitted= fitted_norm,label.cos=label.cos,cos_para = para,
              RMSE = RMSE))
}

