# HCGenExp.R
#' Half-Cauchy Inverse Gompertz Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Half-Cauchy Inverse Gompertz distribution.
#'
#' The Half-Cauchy Inverse Gompertz distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Half-Cauchy Inverse Gompertz distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \lambda, \theta) =
#' \quad 1 - \frac{2}{\pi }\arctan \left\{ { - \frac{\lambda }{{\alpha \theta }}
#' \left( {1 - {e^{\alpha /x}}} \right)} \right\}  \quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\lambda}, and \eqn{\theta} are the parameters.
#'
#' The implementation includes the following functions:
#' \itemize{
#'   \item \code{dhc.inv.gpz()} — Density function
#'   \item \code{phc.inv.gpz()} — Distribution function
#'   \item \code{qhc.inv.gpz()} — Quantile function
#'   \item \code{rhc.inv.gpz()} — Random generation
#'   \item \code{hhc.inv.gpz()} — Hazard function
#' }
#'
#' @usage
#' dhc.inv.gpz(x, alpha, lambda, theta, log = FALSE)
#' phc.inv.gpz(q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qhc.inv.gpz(p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rhc.inv.gpz(n, alpha, lambda, theta)
#' hhc.inv.gpz(x, alpha, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dhc.inv.gpz}: numeric vector of (log-)densities
#'   \item \code{phc.inv.gpz}: numeric vector of probabilities
#'   \item \code{qhc.inv.gpz}: numeric vector of quantiles
#'   \item \code{rhc.inv.gpz}: numeric vector of random variates
#'   \item \code{hhc.inv.gpz}: numeric vector of hazard values
#' }
#'
#' @references 
#' Chaudhary, A. K., Yadav, R. S., & Kumar, V. (2022).
#' Half-Cauchy Inverse Gompertz distribution: Theory and applications.
#' \emph{International Journal of Statistics and Applied Mathematics},
#' \bold{7}(5), 94--102. \doi{10.22271/maths.2022.v7.i5b.885}
#'
#'
#' @examples
#' x <- seq(1.0, 10, 0.25)
#' dhc.inv.gpz(x, 2.0, 0.5, 2.5)
#' phc.inv.gpz(x, 2.0, 0.5, 2.5)
#' qhc.inv.gpz(0.5, 2.0, 0.5, 2.5)
#' rhc.inv.gpz(10, 2.0, 0.5, 2.5)
#' hhc.inv.gpz(x, 2.0, 0.5, 2.5)
#' 
#' # Data
#' x <- relief
#' # ML estimates
#' params = list(alpha=9.0830, lambda=0.8369, theta=17.9925)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = phc.inv.gpz, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qhc.inv.gpz, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dhc.inv.gpz, pfun=phc.inv.gpz, plot=TRUE)
#' print.gofic(out)
#' 
#' @name HCInvGPZ
#' @aliases dhc.inv.gpz phc.inv.gpz qhc.inv.gpz rhc.inv.gpz hhc.inv.gpz
#' @export
NULL

#' @rdname HCInvGPZ
#' @usage NULL
#' @export   
dhc.inv.gpz <- function (x, alpha, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, alpha, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(alpha, lambda, theta) <= 0)) {
        stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- alpha / x[valid]		
	z <- (lambda/alpha)*(1.0 - exp(u))	
    pdf[valid] <- (2/pi)*(lambda/theta)*exp(u) * (x[valid]^-2) * (1/(1+(-z/theta)^2))   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname HCInvGPZ
#' @usage NULL
#' @export  
phc.inv.gpz <- function (q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda, theta) <= 0)) {
		stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- -(lambda/(theta*alpha))*(1- exp(alpha/q[valid]))	
    cdf[valid] <- 1.0 - (2/pi) * atan(u)  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname HCInvGPZ
#' @usage NULL
#' @export  
qhc.inv.gpz <- function (p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda' and 'theta' must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
    qtl[valid] <-  alpha / log(1.0+ ((alpha*theta)/lambda)*tan((1-p[valid]) * pi *0.5)) 	   
    }    
    return(qtl)  
 } 

#' @rdname HCInvGPZ
#' @usage NULL
#' @export  
rhc.inv.gpz <- function(n, alpha, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda, theta), is.numeric)))
		stop("'alpha', 'lambda', and 'theta' must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda', and theta must be positive.")
  
    u <- runif(n)
    q <- qhc.inv.gpz(u, alpha, lambda, theta)
    return(q)
}

#' @rdname HCInvGPZ
#' @usage NULL
#' @export  
hhc.inv.gpz <- function (x, alpha, lambda, theta)
{
	if (!all(sapply(list(x, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
	nume <- dhc.inv.gpz(x, alpha, lambda, theta) 
	surv <- 1.0 - phc.inv.gpz(x, alpha, lambda, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)   
} 

