# LindleyHC.R
#' Lindley Half-Cauchy(LHC) Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Lindley Half-Cauchy distribution.
#'
#' The Lindley Half-Cauchy distribution is parameterized by the parameters
#' \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Lindley Half-Cauchy distribution has CDF:
#'
#' \deqn{
#' F(x; \lambda, \theta) =
#' 1 - {\left\{ {1 - \frac{2}{\pi }{{\tan }^{ - 1}}\left( {\frac{x}{\lambda }} \right)} \right\}
#' ^\theta }\left\{ {1 - \left( {\frac{\theta }{{1 + \theta }}} \right)
#' \ln \left[ {1 - \frac{2}{\pi }{{\tan }^{ - 1}}\left( {\frac{x}{\lambda }} \right)} \right]}
#' \right\} \; ;\;x > 0.
#' }
#'
#' where\eqn{\lambda} and \eqn{\theta} are the parameters.
#' 
#' @usage
#' dlindley.HC(x, lambda, theta, log = FALSE)
#' plindley.HC(q, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qlindley.HC(p, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rlindley.HC(n, lambda, theta)
#' hlindley.HC(x, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlindley.HC}: numeric vector of (log-)densities
#'   \item \code{plindley.HC}: numeric vector of probabilities
#'   \item \code{qlindley.HC}: numeric vector of quantiles
#'   \item \code{rlindley.HC}: numeric vector of random variates
#'   \item \code{hlindley.HC}: numeric vector of hazard values
#' }
#'
#' @references  
#' Chaudhary, A.K. & Kumar, V. (2020). 
#' Lindley Half Cauchy Distribution: Properties and Applications. 
#' \emph{International Journal For Research in Applied Science & Engineering 
#' Technology (IJRASET)},  \bold{8(9)}, 1232--1242. \doi{10.22214/ijraset.2020.31745}  
#'
#' @examples
#' x <- seq(1, 10, 0.5)
#' dlindley.HC(x, 0.5, 1.5)
#' plindley.HC(x, 0.5, 1.5)
#' qlindley.HC(0.5, 0.5, 1.5)
#' rlindley.HC(10, 0.5, 1.5)
#' hlindley.HC(x, 0.5, 1.5)
#' 
#' # Data
#' x <- reactorpump
#' # ML estimates
#' params = list(lambda=0.5479, theta=1.2766)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plindley.HC, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlindley.HC, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlindley.HC, pfun=plindley.HC, plot=FALSE)
#' print.gofic(out)
#' 
#' @name LindleyHC
#' @aliases dlindley.HC plindley.HC qlindley.HC rlindley.HC hlindley.HC
#' @export
NULL

#' @rdname LindleyHC
#' @usage NULL
#' @export   
dlindley.HC <- function (x, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(lambda, theta) <= 0)) {
        stop("Parameters 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- 1-(2/pi) * atan(x[valid]/lambda)
	w <- (theta * theta/(1 + theta))
	v <- lambda/(lambda * lambda + x[valid] * x[valid])
    pdf[valid] <- (2/pi) * w * v * (u ^ (theta-1)) * (1 - log(u))   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname LindleyHC
#' @usage NULL
#' @export   
plindley.HC <- function (q, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(lambda, theta) <= 0)) {
		stop("Parameters 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- 1-(2/pi)*atan(q[valid]/lambda)
    cdf[valid] <- 1.0 - (u ^ theta) * (1 - (theta/(1+theta)) * log(u))  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
} 

#' @rdname LindleyHC
#' @usage NULL
#' @export   
qlindley.HC <- function (p, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(lambda, theta) <= 0))
		stop("Parameters 'lambda', and 'theta' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	    theta*log(1-(2/pi) * atan(x/lambda))+log(1-(theta/(theta+1))
	      *log(1-(2/pi) * atan(x/lambda))) -log(1-wi)   
	}
	
	for (i in to_solve) {
      wi <- p[i]
      lower <- .Machine$double.eps * 10   
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-8),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
}

#' @rdname LindleyHC
#' @usage NULL
#' @export   
rlindley.HC <- function(n, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(lambda, theta), is.numeric)))
		stop("'lambda', and 'theta' must be numeric.")
	if (any(c(lambda, theta) <= 0))
		stop("Invalid parameters: lambda, and theta must be positive.")

    u <- runif(n)
    q <- qlindley.HC(u, lambda, theta)
    return(q)
}

#' @rdname LindleyHC
#' @usage NULL
#' @export   
hlindley.HC <- function (x, lambda, theta)
{
    if (!all(sapply(list(x, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(lambda, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
      
    nume <- dlindley.HC(x, lambda, theta) 
    surv <- 1.0 - plindley.HC(x, lambda, theta)
    hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)  
} 




