# LindleyInvExp.R
#' Lindley Inverse Exponential Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Lindley Inverse Exponential distribution.
#'
#' The Lindley Inverse Exponential distribution is parameterized by the parameters
#' \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Lindley Inverse Exponential distribution has CDF:
#'
#' \deqn{
#' F(x; \lambda, \theta) =
#' 1-\left(1-e^{-\lambda / x}\right)^\theta\left\{1-\left(\frac{\theta}{1+\theta}\right)
#' \ln \left(1-e^{-\lambda / x}\right)\right\} \quad ;\;x > 0.
#' }
#'
#' where\eqn{\lambda} and \eqn{\theta} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dlindley.inv.exp()} — Density function
#'   \item \code{plindley.inv.exp()} — Distribution function
#'   \item \code{qlindley.inv.exp()} — Quantile function
#'   \item \code{rlindley.inv.exp()} — Random generation
#'   \item \code{hlindley.inv.exp()} — Hazard function
#' }
#'
#' @usage
#' dlindley.inv.exp(x, lambda, theta, log = FALSE)
#' plindley.inv.exp(q, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qlindley.inv.exp(p, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rlindley.inv.exp(n, lambda, theta)
#' hlindley.inv.exp(x, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlindley.inv.exp}: numeric vector of (log-)densities
#'   \item \code{plindley.inv.exp}: numeric vector of probabilities
#'   \item \code{qlindley.inv.exp}: numeric vector of quantiles
#'   \item \code{rlindley.inv.exp}: numeric vector of random variates
#'   \item \code{hlindley.inv.exp}: numeric vector of hazard values
#' }
#'
#' @references 
#' Chaudhary,A.K., & Kumar, V. (2020). 
#' Lindley Inverse Exponential Distribution With Properties and Applications. 
#' \emph{Bulletin of Mathematics and Statistics Research (BOMSR)}, \bold{8(4)}, 1--13.   
#'
#' @examples
#' x <- seq(5, 10, 0.5)
#' dlindley.inv.exp(x, 1.5, 5.0)
#' plindley.inv.exp(x, 1.5, 5.0)
#' qlindley.inv.exp(0.5, 1.5, 5.0)
#' rlindley.inv.exp(10, 1.5, 5.0)
#' hlindley.inv.exp(x, 1.5, 5.0)
#' 
#' # Data
#' x <- conductors
#' # ML estimates
#' params = list(lambda=33.8992, theta=96.0743)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plindley.inv.exp, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlindley.inv.exp, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlindley.inv.exp, pfun=plindley.inv.exp, plot=FALSE)
#' print.gofic(out)
#' 
#' @name LindleyInvExp
#' @aliases dlindley.inv.exp plindley.inv.exp qlindley.inv.exp rlindley.inv.exp hlindley.inv.exp
#' @export
NULL

#' @rdname LindleyInvExp
#' @usage NULL
#' @export   
dlindley.inv.exp <- function (x, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(lambda, theta) <= 0)) {
        stop("Parameters 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- lambda / x[valid]	
	z <- 1 - exp(-u)	
    pdf[valid] <- (theta*theta/(1+theta)) * (x[valid]^-2) * lambda * exp(-u) * (z ^ (theta-1)) * (1 - log(z))   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname LindleyInvExp
#' @usage NULL
#' @export
plindley.inv.exp <- function (q, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(lambda, theta) <= 0)) {
		stop("Parameters 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- lambda / q[valid]	
	z <- 1 - exp(-u)
    cdf[valid] <- 1- (z ^ theta) * (1 - (theta/(1+theta)) * log(z))  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LindleyInvExp
#' @usage NULL
#' @export
qlindley.inv.exp <- function (p, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(lambda, theta) <= 0))
		stop("Parameters 'lambda', and 'theta' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	    ((1-exp(-lambda/x))^theta) *(1-(theta/(theta+1)) 
	         * log((1-exp(-lambda/x)))) -(1-wi)   
	}
	
	for (i in to_solve) {
      wi <- p[i]
      lower <- .Machine$double.eps * 10   
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-8),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
}

#' @rdname LindleyInvExp
#' @usage NULL
#' @export
rlindley.inv.exp <- function(n, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(lambda, theta), is.numeric)))
		stop("'lambda', and 'theta' must be numeric.")
	if (any(c(lambda, theta) <= 0))
		stop("Invalid parameters: lambda, and theta must be positive.")

    u <- runif(n)
    q <- qlindley.inv.exp(u, lambda, theta)
    return(q)
}

#' @rdname LindleyInvExp
#' @usage NULL
#' @export
hlindley.inv.exp <- function (x, lambda, theta)
{
    if (!all(sapply(list(x, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(lambda, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
      
    nume <- dlindley.inv.exp(x, lambda, theta) 
    surv <- 1.0 - plindley.inv.exp(x, lambda, theta)
    hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)  
} 


