# ModInvNHE.R
#' Modified Inverse NHE Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Modified Inverse NHE distribution.
#'
#' The Modified Inverse NHE distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Modified Inverse NHE distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad \exp \left\{1-\left(1+\frac{\lambda}{x} 
#' e^{-\beta x}\right)^\alpha\right\}  \quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dmod.inv.NHE()} — Density function
#'   \item \code{pmod.inv.NHE()} — Distribution function
#'   \item \code{qmod.inv.NHE()} — Quantile function
#'   \item \code{rmod.inv.NHE()} — Random generation
#'   \item \code{hmod.inv.NHE()} — Hazard function
#' }
#'
#' @usage
#' dmod.inv.NHE(x, alpha, beta, lambda, log = FALSE)
#' pmod.inv.NHE(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qmod.inv.NHE(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rmod.inv.NHE(n, alpha, beta, lambda)
#' hmod.inv.NHE(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dmod.inv.NHE}: numeric vector of (log-)densities
#'   \item \code{pmod.inv.NHE}: numeric vector of probabilities
#'   \item \code{qmod.inv.NHE}: numeric vector of quantiles
#'   \item \code{rmod.inv.NHE}: numeric vector of random variates
#'   \item \code{hmod.inv.NHE}: numeric vector of hazard values
#' }
#'
#' @references
#' Chaudhary, A. K., Sapkota, L. P., & Kumar, V. (2022). 
#' Modified Inverse NHE Distribution: Properties and Application. 
#' \emph{Journal of Institute of Science and Technology}, \bold{27(1)}, 
#' 125–-133. \doi{10.3126/jist.v27i1.46695}    
#'
#' @examples
#' x <- seq(0.1, 10, 0.2)
#' dmod.inv.NHE(x, 2.0, 0.5, 0.2)
#' pmod.inv.NHE(x, 2.0, 0.5, 0.2)
#' qmod.inv.NHE(0.5, 2.0, 0.5, 0.2)
#' rmod.inv.NHE(10, 2.0, 0.5, 0.2)
#' hmod.inv.NHE(x, 2.0, 0.5, 0.2)
#' 
#' # Data
#' x <- waiting
#' # ML estimates
#' params = list(alpha=0.4858, beta=0.1099, lambda=37.5129)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pmod.inv.NHE, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qmod.inv.NHE, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dmod.inv.NHE, pfun=pmod.inv.NHE, plot=FALSE)
#' print.gofic(out)
#' 
#' @name ModInvNHE
#' @aliases dmod.inv.NHE pmod.inv.NHE qmod.inv.NHE rmod.inv.NHE hmod.inv.NHE
#' @export
NULL

#' @rdname ModInvNHE
#' @usage NULL
#' @export  
dmod.inv.NHE <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	u <- exp(-beta*x[valid])	
	z <- 1+(lambda/x[valid]) * u 	
    pdf[valid] <- (alpha*lambda/(x[valid]*x[valid]))*(1+(beta*x[valid])) * u * (z ^ (alpha-1)) * exp(1-(z^ alpha))  
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname ModInvNHE
#' @usage NULL
#' @export
pmod.inv.NHE <- function (q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- (1+(lambda / q[valid]) * exp(-beta*q[valid])) ^ alpha
    cdf[valid] <- exp(1.0 - u) 
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname ModInvNHE
#' @usage NULL
#' @export
qmod.inv.NHE <- function (p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Parameters 'alpha', 'beta', and 'lambda' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	(lambda/ x) * exp(-beta * x) + 1 - ((1-log(wi)) ^(1/alpha))   
	}
	for (i in to_solve) {
      wi <- p[i]
      lower <- .Machine$double.eps * 10   
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-8),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
} 

#' @rdname ModInvNHE
#' @usage NULL
#' @export
rmod.inv.NHE <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qmod.inv.NHE(u, alpha, beta, lambda)
    return(q)
}

#' @rdname ModInvNHE
#' @usage NULL
#' @export
hmod.inv.NHE <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
     
    nume <- dmod.inv.NHE(x, alpha, beta, lambda)
	surv <- 1.0 - pmod.inv.NHE(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 

