% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/U03-weights.R
\name{estimate_weights}
\alias{estimate_weights}
\title{Estimate Propensity Score Weights}
\usage{
estimate_weights(
  ps_result,
  data,
  treatment_var,
  estimand = "ATE",
  att_group = NULL,
  trim = NULL,
  delta = NULL,
  alpha = NULL
)
}
\arguments{
\item{ps_result}{A list returned by \code{estimate_ps()}, containing the
fitted propensity score model and estimated propensity scores.}

\item{data}{A data.frame containing the treatment variable (same data used
in \code{estimate_ps()}).}

\item{treatment_var}{A character string specifying the name of the treatment
variable in \code{data}.}

\item{estimand}{Character string specifying the target population. One of:
\itemize{
  \item "ATE": Average Treatment Effect (default). Uses IPW method.
  \item "ATT": Average Treatment Effect on the Treated. Uses IPW method.
  \item "overlap": Overlap population (Li & Li, 2019). Uses overlap weighting.
}}

\item{att_group}{For ATT estimation, specifies which treatment group to target.
This is MANDATORY when estimand = "ATT". Ignored for other estimands.}

\item{trim}{Character string specifying the trimming method, or NULL for no
trimming (default). Options: "symmetric" (Crump extension) or "asymmetric"
(Sturmer extension). Trimming is NOT supported with overlap estimand.}

\item{delta}{Trimming threshold for symmetric trimming in (0, 1/J], where J is
the number of treatment levels. If NULL (default), uses recommended values
from Yoshida et al. (2019). Ignored unless trim = "symmetric".}

\item{alpha}{Percentile threshold for asymmetric trimming in (0, 0.5). If NULL
(default), uses recommended values from Yoshida et al. (2019). Ignored unless
trim = "asymmetric".}
}
\value{
A list containing:
  \item{weights}{Numeric vector of weights (length = nrow(data)).}
  \item{trim_summary}{Data frame with trimming summary by treatment group.}
  \item{ess}{Named numeric vector of effective sample size by treatment group.}
  \item{method}{Character string: "IPW" for ATE/ATT, "overlap" for overlap.}
  \item{estimand}{Character string of estimand used.}
  \item{att_group}{Target group for ATT (NULL if not applicable).}
  \item{trim_method}{Character string of trimming method (NULL if no trimming).}
  \item{delta}{Numeric trimming threshold used for symmetric trimming (NULL if not applicable).}
  \item{alpha}{Numeric percentile threshold used for asymmetric trimming (NULL if not applicable).}
  \item{n_levels}{Number of treatment levels.}
  \item{treatment_levels}{Vector of treatment level values.}
  \item{ps_result}{PS result object (refitted after trimming if trimming was applied).}
}
\description{
Calculates propensity score weights for causal inference with optional trimming.
Supports ATE, ATT, and overlap population estimands for binary and multiple
treatment groups.
}
\details{
\strong{Trimming Workflow:}
When trimming is requested, the function: (1) identifies observations to trim
using PS from full data, (2) re-estimates PS on trimmed data, (3) calculates
weights from re-estimated PS. This ensures trimming uses the original covariate
distribution while weights reflect the overlapping population.

Overlap weights do not support trimming (already bounded in [0,1]).
}
\examples{
\donttest{
# Example 1: Overlap weighting for binary treatment
data(simdata_bin)
ps_bin <- estimate_ps(
  data = simdata_bin,
  treatment_var = "Z",
  ps_formula = Z ~ X1 + X2 + X3 + B1 + B2
)
weights_ow <- estimate_weights(
  ps_result = ps_bin,
  data = simdata_bin,
  treatment_var = "Z",
  estimand = "overlap"
)
summary(weights_ow$weights)

# Example 2: ATT with multiple treatments
data(simdata_multi)
ps_multi <- estimate_ps(
  data = simdata_multi,
  treatment_var = "Z",
  ps_formula = Z ~ X1 + X2 + X3 + B1 + B2
)
weights_att <- estimate_weights(
  ps_result = ps_multi,
  data = simdata_multi,
  treatment_var = "Z",
  estimand = "ATT",
  att_group = "C"
)
summary(weights_att$weights)
}

}
\references{
Li, F., & Li, F. (2019). Propensity score weighting for causal inference with
multiple treatments. \emph{The Annals of Applied Statistics}, 13(4), 2389-2415.

Yoshida, K., et al. (2019). Multinomial extension of propensity score trimming
methods: A simulation study. \emph{American Journal of Epidemiology}, 188(3),
609-616.

Crump, R. K., et al. (2009). Dealing with limited overlap in estimation of
average treatment effects. \emph{Biometrika}, 96(1), 187-199.
}
